from __future__ import absolute_import, division, print_function
import time
import mmtbx.model
import iotbx.pdb
from mmtbx.hydrogens import reduce_hydrogen
from libtbx.utils import null_out

# ------------------------------------------------------------------------------

def run():
  pdb_inp = iotbx.pdb.input(lines=pdb_str.split("\n"), source_info=None)
  model = mmtbx.model.manager(model_input = pdb_inp, log = null_out())
  #print(model.get_number_of_atoms())
  model.expand_with_BIOMT_records()
  #print(model.get_number_of_atoms())

  hydro_obj = reduce_hydrogen.place_hydrogens(model = model, print_time=True)
  hydro_obj.run()
  #
  model_h_added = hydro_obj.get_model()
  #print(model_h_added.get_number_of_atoms())
  t = hydro_obj.get_times()
  assert (t.time_rebox_model < 0.06)
  assert (t.time_add_missing_H < 2.70)
  assert (t.time_terminal_propeller < 0.01)
  assert (t.time_make_grm < 160)
  assert (t.time_remove_isolated < 0.27)
  assert (t.time_riding_manager < 6.5)
  assert (t.time_remove_H_nopara < 11.5)
  assert (t.time_reset_idealize < 5)
  assert (t.time_remove_H_on_links < 2)

pdb_str = '''
REMARK 350 MOLECULE CAN BE GENERATED BY APPLYING BIOMT TRANSFORMATIONS
REMARK 350   BIOMT1   1  1.000000  0.000000  0.000000        0.00000
REMARK 350   BIOMT2   1  0.000000  1.000000  0.000000        0.00000
REMARK 350   BIOMT3   1  0.000000  0.000000  1.000000        0.00000
REMARK 350   BIOMT1   2  0.000000  1.000000  0.000000        0.00000
REMARK 350   BIOMT2   2 -1.000000  0.000000  0.000000      540.00000
REMARK 350   BIOMT3   2  0.000000  0.000000  1.000000     -174.60000
REMARK 350   BIOMT1   3 -0.939693  0.342020  0.000000      431.37157
REMARK 350   BIOMT2   3 -0.342020 -0.939693  0.000000      616.06245
REMARK 350   BIOMT3   3  0.000000  0.000000  1.000000     -155.20000
REMARK 350   BIOMT1   4 -0.642788 -0.766044  0.000000      650.38465
REMARK 350   BIOMT2   4  0.766044 -0.642788  0.000000      236.72065
REMARK 350   BIOMT3   4  0.000000  0.000000  1.000000     -135.80000
REMARK 350   BIOMT1   5  0.500000 -0.866025  0.000000      368.82686
REMARK 350   BIOMT2   5  0.866025  0.500000  0.000000      -98.82686
REMARK 350   BIOMT3   5  0.000000  0.000000  1.000000     -116.40000
REMARK 350   BIOMT1   6  0.984808  0.173648  0.000000      -42.78310
REMARK 350   BIOMT2   6 -0.173648  0.984808  0.000000       50.98691
REMARK 350   BIOMT3   6  0.000000  0.000000  1.000000      -97.00000
REMARK 350   BIOMT1   7  0.173648  0.984808  0.000000      -42.78310
REMARK 350   BIOMT2   7 -0.984808  0.173648  0.000000      489.01309
REMARK 350   BIOMT3   7  0.000000  0.000000  1.000000      -77.60000
REMARK 350   BIOMT1   8 -0.866025  0.500000  0.000000      368.82686
REMARK 350   BIOMT2   8 -0.500000 -0.866025  0.000000      638.82686
REMARK 350   BIOMT3   8  0.000000  0.000000  1.000000      -58.20000
REMARK 350   BIOMT1   9 -0.766044 -0.642788  0.000000      650.38465
REMARK 350   BIOMT2   9  0.642788 -0.766044  0.000000      303.27935
REMARK 350   BIOMT3   9  0.000000  0.000000  1.000000      -38.80000
REMARK 350   BIOMT1  10  0.342020 -0.939693  0.000000      431.37157
REMARK 350   BIOMT2  10  0.939693  0.342020  0.000000      -76.06245
REMARK 350   BIOMT3  10  0.000000  0.000000  1.000000      -19.40000
REMARK 350   BIOMT1  11  0.342020  0.939693  0.000000      -76.06245
REMARK 350   BIOMT2  11 -0.939693  0.342020  0.000000      431.37157
REMARK 350   BIOMT3  11  0.000000  0.000000  1.000000       19.40000
REMARK 350   BIOMT1  12 -0.766044  0.642788  0.000000      303.27935
REMARK 350   BIOMT2  12 -0.642788 -0.766044  0.000000      650.38465
REMARK 350   BIOMT3  12  0.000000  0.000000  1.000000       38.80000
REMARK 350   BIOMT1  13 -0.866025 -0.500000  0.000000      638.82686
REMARK 350   BIOMT2  13  0.500000 -0.866025  0.000000      368.82686
REMARK 350   BIOMT3  13  0.000000  0.000000  1.000000       58.20000
REMARK 350   BIOMT1  14  0.173648 -0.984808  0.000000      489.01309
REMARK 350   BIOMT2  14  0.984808  0.173648  0.000000      -42.78310
REMARK 350   BIOMT3  14  0.000000  0.000000  1.000000       77.60000
REMARK 350   BIOMT1  15  0.984808 -0.173648  0.000000       50.98691
REMARK 350   BIOMT2  15  0.173648  0.984808  0.000000      -42.78310
REMARK 350   BIOMT3  15  0.000000  0.000000  1.000000       97.00000
REMARK 350   BIOMT1  16  0.500000  0.866025  0.000000      -98.82686
REMARK 350   BIOMT2  16 -0.866025  0.500000  0.000000      368.82686
REMARK 350   BIOMT3  16  0.000000  0.000000  1.000000      116.40000
REMARK 350   BIOMT1  17 -0.642788  0.766044  0.000000      236.72065
REMARK 350   BIOMT2  17 -0.766044 -0.642788  0.000000      650.38465
REMARK 350   BIOMT3  17  0.000000  0.000000  1.000000      135.80000
REMARK 350   BIOMT1  18 -0.939693 -0.342020  0.000000      616.06245
REMARK 350   BIOMT2  18  0.342020 -0.939693  0.000000      431.37157
REMARK 350   BIOMT3  18  0.000000  0.000000  1.000000      155.20000
REMARK 350   BIOMT1  19  0.000000 -1.000000  0.000000      540.00000
REMARK 350   BIOMT2  19  1.000000  0.000000  0.000000        0.00000
REMARK 350   BIOMT3  19  0.000000  0.000000  1.000000      174.60000
REMARK 350   BIOMT1  20  0.939693 -0.342020  0.000000      108.62843
REMARK 350   BIOMT2  20  0.342020  0.939693  0.000000      -76.06245
REMARK 350   BIOMT3  20  0.000000  0.000000  1.000000      194.00000
ATOM      1  N   SER A   5     320.275 318.898 273.938  1.00 77.65           N
ATOM      2  CA  SER A   5     320.746 319.843 274.942  1.00 77.65           C
ATOM      3  C   SER A   5     319.646 320.173 275.945  1.00 77.65           C
ATOM      4  O   SER A   5     318.778 320.999 275.670  1.00 77.65           O
ATOM      5  CB  SER A   5     321.251 321.125 274.278  1.00 77.65           C
ATOM      6  OG  SER A   5     322.385 320.865 273.471  1.00 77.65           O
ATOM      7  N   PRO A   6     319.678 319.524 277.104  1.00 78.17           N
ATOM      8  CA  PRO A   6     318.681 319.803 278.139  1.00 78.17           C
ATOM      9  C   PRO A   6     318.851 321.200 278.718  1.00 78.17           C
ATOM     10  O   PRO A   6     319.940 321.772 278.737  1.00 78.17           O
ATOM     11  CB  PRO A   6     318.962 318.732 279.198  1.00 78.17           C
ATOM     12  CG  PRO A   6     320.410 318.430 279.029  1.00 78.17           C
ATOM     13  CD  PRO A   6     320.674 318.536 277.552  1.00 78.17           C
ATOM     14  N   THR A   7     317.734 321.746 279.191  1.00 82.30           N
ATOM     15  CA  THR A   7     317.740 323.065 279.805  1.00 82.30           C
ATOM     16  C   THR A   7     318.419 323.016 281.167  1.00 82.30           C
ATOM     17  O   THR A   7     318.330 322.019 281.889  1.00 82.30           O
ATOM     18  CB  THR A   7     316.314 323.602 279.945  1.00 82.30           C
ATOM     19  OG1 THR A   7     316.354 324.928 280.487  1.00 82.30           O
ATOM     20  CG2 THR A   7     315.479 322.706 280.849  1.00 82.30           C
ATOM     21  N   ILE A   8     319.116 324.092 281.506  1.00 79.33           N
ATOM     22  CA  ILE A   8     319.819 324.194 282.777  1.00 79.33           C
ATOM     23  C   ILE A   8     319.148 325.262 283.625  1.00 79.33           C
ATOM     24  O   ILE A   8     318.757 326.328 283.133  1.00 79.33           O
ATOM     25  CB  ILE A   8     321.318 324.488 282.571  1.00 79.33           C
ATOM     26  CG1 ILE A   8     321.517 325.780 281.787  1.00 79.33           C
ATOM     27  CG2 ILE A   8     321.983 323.343 281.841  1.00 79.33           C
ATOM     28  CD1 ILE A   8     322.962 326.162 281.640  1.00 79.33           C
ATOM     29  N   ASN A   9     319.002 324.969 284.916  1.00 84.33           N
ATOM     30  CA  ASN A   9     318.256 325.826 285.821  1.00 84.33           C
ATOM     31  C   ASN A   9     319.105 326.498 286.887  1.00 84.33           C
ATOM     32  O   ASN A   9     318.591 327.370 287.596  1.00 84.33           O
ATOM     33  CB  ASN A   9     317.146 325.028 286.520  1.00 84.33           C
ATOM     34  CG  ASN A   9     316.134 324.466 285.550  1.00 84.33           C
ATOM     35  OD1 ASN A   9     315.775 325.111 284.566  1.00 84.33           O
ATOM     36  ND2 ASN A   9     315.661 323.257 285.825  1.00 84.33           N
ATOM     37  N   PHE A  10     320.376 326.128 287.029  1.00 80.50           N
ATOM     38  CA  PHE A  10     321.162 326.626 288.147  1.00 80.50           C
ATOM     39  C   PHE A  10     322.641 326.450 287.849  1.00 80.50           C
ATOM     40  O   PHE A  10     323.081 325.354 287.498  1.00 80.50           O
ATOM     41  CB  PHE A  10     320.784 325.891 289.437  1.00 80.50           C
ATOM     42  CG  PHE A  10     321.452 326.429 290.663  1.00 80.50           C
ATOM     43  CD1 PHE A  10     320.900 327.490 291.357  1.00 80.50           C
ATOM     44  CD2 PHE A  10     322.620 325.862 291.133  1.00 80.50           C
ATOM     45  CE1 PHE A  10     321.511 327.983 292.487  1.00 80.50           C
ATOM     46  CE2 PHE A  10     323.231 326.349 292.261  1.00 80.50           C
ATOM     47  CZ  PHE A  10     322.679 327.411 292.939  1.00 80.50           C
ATOM     48  N   ILE A  11     323.398 327.535 287.988  1.00 73.41           N
ATOM     49  CA  ILE A  11     324.848 327.511 287.858  1.00 73.41           C
ATOM     50  C   ILE A  11     325.440 328.358 288.972  1.00 73.41           C
ATOM     51  O   ILE A  11     324.923 329.436 289.283  1.00 73.41           O
ATOM     52  CB  ILE A  11     325.320 328.042 286.488  1.00 73.41           C
ATOM     53  CG1 ILE A  11     324.650 327.289 285.342  1.00 73.41           C
ATOM     54  CG2 ILE A  11     326.824 327.915 286.362  1.00 73.41           C
ATOM     55  CD1 ILE A  11     324.978 327.871 283.991  1.00 73.41           C
ATOM     56  N   ASN A  12     326.519 327.871 289.580  1.00 73.41           N
ATOM     57  CA  ASN A  12     327.281 328.686 290.515  1.00 73.41           C
ATOM     58  C   ASN A  12     328.668 328.089 290.692  1.00 73.41           C
ATOM     59  O   ASN A  12     328.870 326.886 290.519  1.00 73.41           O
ATOM     60  CB  ASN A  12     326.584 328.813 291.875  1.00 73.41           C
ATOM     61  CG  ASN A  12     326.292 327.474 292.513  1.00 73.41           C
ATOM     62  OD1 ASN A  12     326.540 326.422 291.926  1.00 73.41           O
ATOM     63  ND2 ASN A  12     325.778 327.506 293.734  1.00 73.41           N
ATOM     64  N   PHE A  13     329.617 328.949 291.033  1.00 73.41           N
ATOM     65  CA  PHE A  13     330.958 328.532 291.394  1.00 73.41           C
ATOM     66  C   PHE A  13     330.976 328.016 292.829  1.00 73.41           C
ATOM     67  O   PHE A  13     329.999 328.136 293.568  1.00 73.41           O
ATOM     68  CB  PHE A  13     331.938 329.694 291.248  1.00 73.41           C
ATOM     69  CG  PHE A  13     332.245 330.051 289.828  1.00 73.41           C
ATOM     70  CD1 PHE A  13     333.449 329.687 289.256  1.00 73.41           C
ATOM     71  CD2 PHE A  13     331.324 330.735 289.057  1.00 73.41           C
ATOM     72  CE1 PHE A  13     333.736 330.013 287.950  1.00 73.41           C
ATOM     73  CE2 PHE A  13     331.604 331.060 287.751  1.00 73.41           C
ATOM     74  CZ  PHE A  13     332.811 330.696 287.195  1.00 73.41           C
ATOM     75  N   ASN A  14     332.104 327.425 293.215  1.00 78.03           N
ATOM     76  CA  ASN A  14     332.301 327.105 294.617  1.00 78.03           C
ATOM     77  C   ASN A  14     332.773 328.353 295.359  1.00 78.03           C
ATOM     78  O   ASN A  14     332.940 329.430 294.780  1.00 78.03           O
ATOM     79  CB  ASN A  14     333.284 325.948 294.781  1.00 78.03           C
ATOM     80  CG  ASN A  14     334.697 326.307 294.362  1.00 78.03           C
ATOM     81  OD1 ASN A  14     334.950 327.382 293.825  1.00 78.03           O
ATOM     82  ND2 ASN A  14     335.631 325.402 294.622  1.00 78.03           N
ATOM     83  N   GLN A  15     333.007 328.202 296.660  1.00 80.13           N
ATOM     84  CA  GLN A  15     333.372 329.351 297.478  1.00 80.13           C
ATOM     85  C   GLN A  15     334.763 329.879 297.156  1.00 80.13           C
ATOM     86  O   GLN A  15     335.116 330.971 297.610  1.00 80.13           O
ATOM     87  CB  GLN A  15     333.264 328.986 298.955  1.00 80.13           C
ATOM     88  CG  GLN A  15     331.831 328.810 299.434  1.00 80.13           C
ATOM     89  CD  GLN A  15     331.204 327.523 298.941  1.00 80.13           C
ATOM     90  OE1 GLN A  15     331.901 326.614 298.498  1.00 80.13           O
ATOM     91  NE2 GLN A  15     329.881 327.444 299.006  1.00 80.13           N
ATOM     92  N   THR A  16     335.558 329.136 296.386  1.00 82.41           N
ATOM     93  CA  THR A  16     336.863 329.606 295.941  1.00 82.41           C
ATOM     94  C   THR A  16     336.902 329.971 294.464  1.00 82.41           C
ATOM     95  O   THR A  16     337.871 330.595 294.025  1.00 82.41           O
ATOM     96  CB  THR A  16     337.941 328.553 296.221  1.00 82.41           C
ATOM     97  OG1 THR A  16     337.526 327.287 295.695  1.00 82.41           O
ATOM     98  CG2 THR A  16     338.197 328.432 297.710  1.00 82.41           C
ATOM     99  N   GLY A  17     335.882 329.603 293.692  1.00 78.20           N
ATOM    100  CA  GLY A  17     335.844 329.949 292.287  1.00 78.20           C
ATOM    101  C   GLY A  17     336.651 329.055 291.375  1.00 78.20           C
ATOM    102  O   GLY A  17     336.877 329.426 290.217  1.00 78.20           O
ATOM    103  N   THR A  18     337.101 327.895 291.854  1.00 77.58           N
ATOM    104  CA  THR A  18     337.873 326.977 291.028  1.00 77.58           C
ATOM    105  C   THR A  18     337.075 325.768 290.562  1.00 77.58           C
ATOM    106  O   THR A  18     337.664 324.832 290.014  1.00 77.58           O
ATOM    107  CB  THR A  18     339.113 326.490 291.780  1.00 77.58           C
ATOM    108  OG1 THR A  18     338.709 325.713 292.916  1.00 77.58           O
ATOM    109  CG2 THR A  18     339.949 327.669 292.249  1.00 77.58           C
ATOM    110  N   CYS A  19     335.763 325.754 290.777  1.00 73.41           N
ATOM    111  CA  CYS A  19     334.913 324.658 290.340  1.00 73.41           C
ATOM    112  C   CYS A  19     333.603 325.213 289.806  1.00 73.41           C
ATOM    113  O   CYS A  19     333.312 326.404 289.935  1.00 73.41           O
ATOM    114  CB  CYS A  19     334.638 323.667 291.477  1.00 73.41           C
ATOM    115  SG  CYS A  19     336.118 322.907 292.184  1.00 73.41           S
ATOM    116  N   ILE A  20     332.818 324.339 289.185  1.00 73.41           N
ATOM    117  CA  ILE A  20     331.510 324.689 288.645  1.00 73.41           C
ATOM    118  C   ILE A  20     330.502 323.662 289.130  1.00 73.41           C
ATOM    119  O   ILE A  20     330.750 322.454 289.051  1.00 73.41           O
ATOM    120  CB  ILE A  20     331.515 324.742 287.107  1.00 73.41           C
ATOM    121  CG1 ILE A  20     332.546 325.744 286.598  1.00 73.41           C
ATOM    122  CG2 ILE A  20     330.134 325.106 286.589  1.00 73.41           C
ATOM    123  CD1 ILE A  20     332.216 327.158 286.938  1.00 73.41           C
ATOM    124  N   SER A  21     329.372 324.139 289.631  1.00 75.58           N
ATOM    125  CA  SER A  21     328.207 323.303 289.867  1.00 75.58           C
ATOM    126  C   SER A  21     327.123 323.706 288.882  1.00 75.58           C
ATOM    127  O   SER A  21     327.029 324.874 288.495  1.00 75.58           O
ATOM    128  CB  SER A  21     327.699 323.441 291.302  1.00 75.58           C
ATOM    129  OG  SER A  21     328.660 322.968 292.229  1.00 75.58           O
ATOM    130  N   LEU A  22     326.317 322.739 288.462  1.00 76.98           N
ATOM    131  CA  LEU A  22     325.313 323.013 287.450  1.00 76.98           C
ATOM    132  C   LEU A  22     324.155 322.045 287.617  1.00 76.98           C
ATOM    133  O   LEU A  22     324.357 320.884 287.978  1.00 76.98           O
ATOM    134  CB  LEU A  22     325.919 322.909 286.046  1.00 76.98           C
ATOM    135  CG  LEU A  22     325.045 323.302 284.858  1.00 76.98           C
ATOM    136  CD1 LEU A  22     325.881 324.015 283.821  1.00 76.98           C
ATOM    137  CD2 LEU A  22     324.405 322.074 284.254  1.00 76.98           C
ATOM    138  N   GLY A  23     322.946 322.532 287.354  1.00 82.19           N
ATOM    139  CA  GLY A  23     321.763 321.704 287.452  1.00 82.19           C
ATOM    140  C   GLY A  23     320.893 321.770 286.216  1.00 82.19           C
ATOM    141  O   GLY A  23     320.467 322.854 285.808  1.00 82.19           O
ATOM    142  N   THR A  24     320.621 320.619 285.611  1.00 85.29           N
ATOM    143  CA  THR A  24     319.785 320.518 284.428  1.00 85.29           C
ATOM    144  C   THR A  24     318.381 320.086 284.832  1.00 85.29           C
ATOM    145  O   THR A  24     318.065 319.943 286.019  1.00 85.29           O
ATOM    146  CB  THR A  24     320.393 319.537 283.426  1.00 85.29           C
ATOM    147  OG1 THR A  24     320.373 318.215 283.981  1.00 85.29           O
ATOM    148  CG2 THR A  24     321.824 319.916 283.109  1.00 85.29           C
ATOM    149  N   SER A  25     317.523 319.877 283.833  1.00 88.54           N
ATOM    150  CA  SER A  25     316.219 319.286 284.096  1.00 88.54           C
ATOM    151  C   SER A  25     316.288 317.768 284.162  1.00 88.54           C
ATOM    152  O   SER A  25     315.292 317.128 284.515  1.00 88.54           O
ATOM    153  CB  SER A  25     315.215 319.716 283.027  1.00 88.54           C
ATOM    154  OG  SER A  25     313.950 319.124 283.256  1.00 88.54           O
ATOM    155  N   LYS A  26     317.437 317.182 283.826  1.00 89.87           N
ATOM    156  CA  LYS A  26     317.637 315.744 283.897  1.00 89.87           C
ATOM    157  C   LYS A  26     318.655 315.318 284.942  1.00 89.87           C
ATOM    158  O   LYS A  26     318.723 314.125 285.259  1.00 89.87           O
ATOM    159  CB  LYS A  26     318.051 315.198 282.518  1.00 89.87           C
ATOM    160  CG  LYS A  26     319.385 315.720 281.981  1.00 89.87           C
ATOM    161  CD  LYS A  26     320.571 314.850 282.373  1.00 89.87           C
ATOM    162  CE  LYS A  26     321.852 315.352 281.726  1.00 89.87           C
ATOM    163  NZ  LYS A  26     323.038 314.538 282.110  1.00 89.87           N
ATOM    164  N   GLY A  27     319.442 316.242 285.484  1.00 87.83           N
ATOM    165  CA  GLY A  27     320.435 315.880 286.477  1.00 87.83           C
ATOM    166  C   GLY A  27     321.346 317.053 286.775  1.00 87.83           C
ATOM    167  O   GLY A  27     321.048 318.192 286.416  1.00 87.83           O
ATOM    168  N   PHE A  28     322.462 316.751 287.433  1.00 83.20           N
ATOM    169  CA  PHE A  28     323.434 317.762 287.813  1.00 83.20           C
ATOM    170  C   PHE A  28     324.817 317.341 287.342  1.00 83.20           C
ATOM    171  O   PHE A  28     325.087 316.161 287.111  1.00 83.20           O
ATOM    172  CB  PHE A  28     323.436 318.003 289.330  1.00 83.20           C
ATOM    173  CG  PHE A  28     323.987 316.858 290.132  1.00 83.20           C
ATOM    174  CD1 PHE A  28     325.306 316.862 290.555  1.00 83.20           C
ATOM    175  CD2 PHE A  28     323.184 315.783 290.470  1.00 83.20           C
ATOM    176  CE1 PHE A  28     325.814 315.813 291.294  1.00 83.20           C
ATOM    177  CE2 PHE A  28     323.687 314.732 291.206  1.00 83.20           C
ATOM    178  CZ  PHE A  28     325.002 314.747 291.621  1.00 83.20           C
ATOM    179  N   LYS A  29     325.691 318.331 287.191  1.00 77.31           N
ATOM    180  CA  LYS A  29     327.056 318.104 286.743  1.00 77.31           C
ATOM    181  C   LYS A  29     327.999 318.942 287.589  1.00 77.31           C
ATOM    182  O   LYS A  29     327.604 319.964 288.153  1.00 77.31           O
ATOM    183  CB  LYS A  29     327.228 318.459 285.263  1.00 77.31           C
ATOM    184  CG  LYS A  29     326.346 317.665 284.319  1.00 77.31           C
ATOM    185  CD  LYS A  29     326.542 318.110 282.881  1.00 77.31           C
ATOM    186  CE  LYS A  29     325.669 317.303 281.939  1.00 77.31           C
ATOM    187  NZ  LYS A  29     324.228 317.630 282.118  1.00 77.31           N
ATOM    188  N   ILE A  30     329.251 318.500 287.678  1.00 73.41           N
ATOM    189  CA  ILE A  30     330.285 319.218 288.414  1.00 73.41           C
ATOM    190  C   ILE A  30     331.514 319.285 287.518  1.00 73.41           C
ATOM    191  O   ILE A  30     331.969 318.256 287.012  1.00 73.41           O
ATOM    192  CB  ILE A  30     330.630 318.552 289.751  1.00 73.41           C
ATOM    193  CG1 ILE A  30     329.438 318.619 290.702  1.00 73.41           C
ATOM    194  CG2 ILE A  30     331.835 319.225 290.383  1.00 73.41           C
ATOM    195  CD1 ILE A  30     329.716 318.013 292.045  1.00 73.41           C
ATOM    196  N   PHE A  31     332.047 320.482 287.320  1.00 73.41           N
ATOM    197  CA  PHE A  31     333.179 320.681 286.437  1.00 73.41           C
ATOM    198  C   PHE A  31     334.390 321.165 287.227  1.00 73.41           C
ATOM    199  O   PHE A  31     334.380 321.218 288.459  1.00 73.41           O
ATOM    200  CB  PHE A  31     332.811 321.658 285.317  1.00 73.41           C
ATOM    201  CG  PHE A  31     331.777 321.125 284.367  1.00 73.41           C
ATOM    202  CD1 PHE A  31     330.448 321.046 284.738  1.00 73.41           C
ATOM    203  CD2 PHE A  31     332.142 320.687 283.108  1.00 73.41           C
ATOM    204  CE1 PHE A  31     329.501 320.554 283.867  1.00 73.41           C
ATOM    205  CE2 PHE A  31     331.199 320.190 282.233  1.00 73.41           C
ATOM    206  CZ  PHE A  31     329.877 320.121 282.615  1.00 73.41           C
ATOM    207  N   ASN A  32     335.455 321.505 286.509  1.00 73.41           N
ATOM    208  CA  ASN A  32     336.642 322.118 287.084  1.00 73.41           C
ATOM    209  C   ASN A  32     337.062 323.247 286.151  1.00 73.41           C
ATOM    210  O   ASN A  32     336.565 323.359 285.028  1.00 73.41           O
ATOM    211  CB  ASN A  32     337.743 321.075 287.297  1.00 73.41           C
ATOM    212  CG  ASN A  32     338.918 321.615 288.083  1.00 73.41           C
ATOM    213  OD1 ASN A  32     340.064 321.253 287.834  1.00 73.41           O
ATOM    214  ND2 ASN A  32     338.635 322.486 289.040  1.00 73.41           N
ATOM    215  N   CYS A  33     337.978 324.100 286.607  1.00 74.73           N
ATOM    216  CA  CYS A  33     338.216 325.337 285.872  1.00 74.73           C
ATOM    217  C   CYS A  33     339.553 325.376 285.142  1.00 74.73           C
ATOM    218  O   CYS A  33     339.580 325.559 283.921  1.00 74.73           O
ATOM    219  CB  CYS A  33     338.114 326.534 286.819  1.00 74.73           C
ATOM    220  SG  CYS A  33     336.431 326.899 287.362  1.00 74.73           S
ATOM    221  N   GLU A  34     340.666 325.213 285.854  1.00 78.81           N
ATOM    222  CA  GLU A  34     341.959 325.358 285.190  1.00 78.81           C
ATOM    223  C   GLU A  34     342.330 324.130 284.360  1.00 78.81           C
ATOM    224  O   GLU A  34     342.532 324.288 283.143  1.00 78.81           O
ATOM    225  CB  GLU A  34     343.041 325.719 286.217  1.00 78.81           C
ATOM    226  CG  GLU A  34     344.298 326.317 285.612  1.00 78.81           C
ATOM    227  CD  GLU A  34     345.255 325.264 285.097  1.00 78.81           C
ATOM    228  OE1 GLU A  34     345.273 324.151 285.662  1.00 78.81           O
ATOM    229  OE2 GLU A  34     345.986 325.548 284.126  1.00 78.81           O
ATOM    230  N   PRO A  35     342.448 322.907 284.918  1.00 75.47           N
ATOM    231  CA  PRO A  35     342.512 321.728 284.050  1.00 75.47           C
ATOM    232  C   PRO A  35     341.116 321.205 283.766  1.00 75.47           C
ATOM    233  O   PRO A  35     340.675 320.246 284.405  1.00 75.47           O
ATOM    234  CB  PRO A  35     343.346 320.733 284.865  1.00 75.47           C
ATOM    235  CG  PRO A  35     343.258 321.216 286.302  1.00 75.47           C
ATOM    236  CD  PRO A  35     342.481 322.506 286.333  1.00 75.47           C
ATOM    237  N   PHE A  36     340.419 321.836 282.819  1.00 75.47           N
ATOM    238  CA  PHE A  36     338.981 321.645 282.669  1.00 75.47           C
ATOM    239  C   PHE A  36     338.626 320.171 282.533  1.00 75.47           C
ATOM    240  O   PHE A  36     339.291 319.416 281.821  1.00 75.47           O
ATOM    241  CB  PHE A  36     338.474 322.418 281.451  1.00 74.25           C
ATOM    242  CG  PHE A  36     336.995 322.292 281.230  1.00 74.25           C
ATOM    243  CD1 PHE A  36     336.113 323.144 281.868  1.00 74.25           C
ATOM    244  CD2 PHE A  36     336.485 321.319 280.387  1.00 74.25           C
ATOM    245  CE1 PHE A  36     334.751 323.031 281.668  1.00 75.62           C
ATOM    246  CE2 PHE A  36     335.123 321.200 280.186  1.00 75.62           C
ATOM    247  CZ  PHE A  36     334.256 322.058 280.825  1.00 75.62           C
ATOM    248  N   GLY A  37     337.571 319.767 283.230  1.00 75.62           N
ATOM    249  CA  GLY A  37     337.133 318.387 283.207  1.00 75.62           C
ATOM    250  C   GLY A  37     335.795 318.244 283.892  1.00 75.62           C
ATOM    251  O   GLY A  37     335.298 319.179 284.529  1.00 75.62           O
ATOM    252  N   LYS A  38     335.213 317.054 283.748  1.00 75.62           N
ATOM    253  CA  LYS A  38     333.912 316.746 284.338  1.00 75.62           C
ATOM    254  C   LYS A  38     334.115 315.754 285.480  1.00 77.01           C
ATOM    255  O   LYS A  38     334.259 314.546 285.284  1.00 77.01           O
ATOM    256  CB  LYS A  38     332.964 316.204 283.278  1.00 77.01           C
ATOM    257  CG  LYS A  38     331.527 316.094 283.725  1.00 77.01           C
ATOM    258  CD  LYS A  38     330.648 315.631 282.580  1.00 77.01           C
ATOM    259  CE  LYS A  38     329.184 315.652 282.968  1.00 77.01           C
ATOM    260  NZ  LYS A  38     328.304 315.146 281.879  1.00 77.01           N
ATOM    261  N   PHE A  39     334.135 316.289 286.704  1.00 77.01           N
ATOM    262  CA  PHE A  39     334.393 315.452 287.871  1.00 77.01           C
ATOM    263  C   PHE A  39     333.257 314.481 288.158  1.00 77.01           C
ATOM    264  O   PHE A  39     333.507 313.315 288.481  1.00 77.01           O
ATOM    265  CB  PHE A  39     334.655 316.321 289.100  1.00 83.28           C
ATOM    266  CG  PHE A  39     336.002 316.976 289.100  1.00 83.28           C
ATOM    267  CD1 PHE A  39     336.980 316.573 288.209  1.00 83.28           C
ATOM    268  CD2 PHE A  39     336.291 317.992 289.992  1.00 83.28           C
ATOM    269  CE1 PHE A  39     338.223 317.172 288.206  1.00 83.28           C
ATOM    270  CE2 PHE A  39     337.532 318.596 289.994  1.00 83.28           C
ATOM    271  CZ  PHE A  39     338.500 318.186 289.101  1.00 83.28           C
ATOM    272  N   TYR A  40     332.009 314.929 288.053  1.00 83.28           N
ATOM    273  CA  TYR A  40     330.903 314.078 288.465  1.00 83.28           C
ATOM    274  C   TYR A  40     329.652 314.452 287.688  1.00 83.28           C
ATOM    275  O   TYR A  40     329.549 315.545 287.129  1.00 83.28           O
ATOM    276  CB  TYR A  40     330.647 314.195 289.969  1.00 83.28           C
ATOM    277  CG  TYR A  40     329.765 313.106 290.518  1.00 82.37           C
ATOM    278  CD1 TYR A  40     330.169 311.780 290.477  1.00 82.37           C
ATOM    279  CD2 TYR A  40     328.525 313.395 291.066  1.00 82.37           C
ATOM    280  CE1 TYR A  40     329.372 310.775 290.976  1.00 82.37           C
ATOM    281  CE2 TYR A  40     327.718 312.395 291.570  1.00 82.37           C
ATOM    282  CZ  TYR A  40     328.147 311.086 291.521  1.00 82.37           C
ATOM    283  OH  TYR A  40     327.356 310.077 292.015  1.00 86.06           O
ATOM    284  N   SER A  41     328.697 313.523 287.668  1.00 86.06           N
ATOM    285  CA  SER A  41     327.417 313.737 287.010  1.00 86.06           C
ATOM    286  C   SER A  41     326.470 312.601 287.366  1.00 86.06           C
ATOM    287  O   SER A  41     326.891 311.455 287.530  1.00 86.06           O
ATOM    288  CB  SER A  41     327.570 313.826 285.490  1.00 86.06           C
ATOM    289  OG  SER A  41     326.312 314.014 284.863  1.00 86.06           O
ATOM    290  N   GLU A  42     325.185 312.935 287.471  1.00 86.06           N
ATOM    291  CA  GLU A  42     324.132 311.942 287.649  1.00 86.06           C
ATOM    292  C   GLU A  42     322.897 312.400 286.891  1.00 92.66           C
ATOM    293  O   GLU A  42     322.518 313.572 286.975  1.00 92.66           O
ATOM    294  CB  GLU A  42     323.782 311.732 289.126  1.00 92.66           C
ATOM    295  CG  GLU A  42     324.870 311.093 289.968  1.00 92.66           C
ATOM    296  CD  GLU A  42     324.455 310.933 291.416  1.00 92.66           C
ATOM    297  OE1 GLU A  42     323.370 311.434 291.782  1.00 92.66           O
ATOM    298  OE2 GLU A  42     325.205 310.303 292.189  1.00 92.66           O
ATOM    299  N   ASP A  43     322.280 311.482 286.154  1.00 92.66           N
ATOM    300  CA  ASP A  43     321.022 311.762 285.466  1.00 94.16           C
ATOM    301  C   ASP A  43     319.856 311.240 286.307  1.00 94.16           C
ATOM    302  O   ASP A  43     319.140 310.310 285.938  1.00 94.16           O
ATOM    303  CB  ASP A  43     321.032 311.148 284.070  1.00 94.16           C
ATOM    304  CG  ASP A  43     322.140 311.704 283.198  1.00 94.16           C
ATOM    305  OD1 ASP A  43     322.985 312.463 283.719  1.00 94.16           O
ATOM    306  OD2 ASP A  43     322.165 311.386 281.991  1.00 94.52           O
ATOM    307  N   SER A  44     319.679 311.873 287.465  1.00 94.52           N
ATOM    308  CA  SER A  44     318.652 311.457 288.415  1.00 94.52           C
ATOM    309  C   SER A  44     317.312 312.131 288.126  1.00 94.52           C
ATOM    310  O   SER A  44     316.301 311.453 287.916  1.00 98.97           O
ATOM    311  CB  SER A  44     319.110 311.759 289.846  1.00 98.97           C
ATOM    312  OG  SER A  44     320.379 311.184 290.108  1.00 98.97           O
ATOM    313  N   GLY A  45     317.294 313.456 288.114  1.00 98.97           N
ATOM    314  CA  GLY A  45     316.063 314.174 287.860  1.00 96.48           C
ATOM    315  C   GLY A  45     316.311 315.662 287.783  1.00 96.48           C
ATOM    316  O   GLY A  45     317.451 316.117 287.684  1.00 96.48           O
ATOM    317  N   GLY A  46     315.219 316.419 287.818  1.00 96.48           N
ATOM    318  CA  GLY A  46     315.333 317.866 287.811  1.00 96.48           C
ATOM    319  C   GLY A  46     315.928 318.379 289.110  1.00 96.48           C
ATOM    320  O   GLY A  46     315.501 318.011 290.204  1.00 96.48           O
ATOM    321  N   TYR A  47     316.933 319.242 288.982  1.00 96.48           N
ATOM    322  CA  TYR A  47     317.589 319.852 290.126  1.00 96.48           C
ATOM    323  C   TYR A  47     317.575 321.365 289.982  1.00 96.48           C
ATOM    324  O   TYR A  47     317.949 321.902 288.936  1.00 96.48           O
ATOM    325  CB  TYR A  47     319.029 319.356 290.276  1.00 96.48           C
ATOM    326  CG  TYR A  47     319.124 317.940 290.787  1.00 93.36           C
ATOM    327  CD1 TYR A  47     318.986 317.661 292.136  1.00 93.36           C
ATOM    328  CD2 TYR A  47     319.359 316.884 289.921  1.00 93.36           C
ATOM    329  CE1 TYR A  47     319.071 316.369 292.610  1.00 93.36           C
ATOM    330  CE2 TYR A  47     319.448 315.588 290.385  1.00 93.36           C
ATOM    331  CZ  TYR A  47     319.303 315.336 291.730  1.00 88.81           C
ATOM    332  OH  TYR A  47     319.391 314.047 292.199  1.00 88.81           O
ATOM    333  N   ALA A  48     317.142 322.047 291.041  1.00 88.81           N
ATOM    334  CA  ALA A  48     317.070 323.500 291.053  1.00 88.81           C
ATOM    335  C   ALA A  48     318.177 324.151 291.864  1.00 88.81           C
ATOM    336  O   ALA A  48     318.545 325.293 291.578  1.00 88.81           O
ATOM    337  CB  ALA A  48     315.714 323.958 291.599  1.00 88.81           C
ATOM    338  N   ILE A  49     318.707 323.460 292.871  1.00 88.81           N
ATOM    339  CA  ILE A  49     319.781 324.004 293.694  1.00 85.51           C
ATOM    340  C   ILE A  49     320.879 322.962 293.846  1.00 85.51           C
ATOM    341  O   ILE A  49     320.673 321.911 294.463  1.00 85.51           O
ATOM    342  CB  ILE A  49     319.263 324.454 295.073  1.00 85.51           C
ATOM    343  CG1 ILE A  49     318.222 325.561 294.921  1.00 85.51           C
ATOM    344  CG2 ILE A  49     320.411 324.933 295.942  1.00 85.51           C
ATOM    345  CD1 ILE A  49     317.422 325.808 296.169  1.00 85.51           C
ATOM    346  N   VAL A  50     322.052 323.244 293.285  1.00 85.29           N
ATOM    347  CA  VAL A  50     323.225 322.390 293.421  1.00 85.29           C
ATOM    348  C   VAL A  50     324.363 323.256 293.939  1.00 85.29           C
ATOM    349  O   VAL A  50     325.005 323.978 293.165  1.00 85.29           O
ATOM    350  CB  VAL A  50     323.600 321.713 292.094  1.00 85.29           C
ATOM    351  CG1 VAL A  50     324.916 320.965 292.230  1.00 85.29           C
ATOM    352  CG2 VAL A  50     322.494 320.771 291.653  1.00 85.29           C
ATOM    353  N   GLU A  51     324.612 323.189 295.244  1.00 85.29           N
ATOM    354  CA  GLU A  51     325.624 324.011 295.891  1.00 85.29           C
ATOM    355  C   GLU A  51     326.688 323.104 296.487  1.00 85.08           C
ATOM    356  O   GLU A  51     326.368 322.059 297.061  1.00 85.08           O
ATOM    357  CB  GLU A  51     325.006 324.899 296.976  1.00 85.08           C
ATOM    358  CG  GLU A  51     326.021 325.685 297.790  1.00 85.08           C
ATOM    359  CD  GLU A  51     326.411 326.991 297.131  1.00 85.08           C
ATOM    360  OE1 GLU A  51     325.645 327.477 296.273  1.00 85.08           O
ATOM    361  OE2 GLU A  51     327.486 327.531 297.467  1.00 85.08           O
ATOM    362  N   MET A  52     327.949 323.509 296.354  1.00 85.08           N
ATOM    363  CA  MET A  52     329.083 322.707 296.784  1.00 80.47           C
ATOM    364  C   MET A  52     329.929 323.502 297.764  1.00 80.47           C
ATOM    365  O   MET A  52     329.919 324.737 297.739  1.00 80.47           O
ATOM    366  CB  MET A  52     329.936 322.276 295.589  1.00 80.47           C
ATOM    367  CG  MET A  52     330.605 323.428 294.862  1.00 80.47           C
ATOM    368  SD  MET A  52     331.611 322.870 293.477  1.00 80.47           S
ATOM    369  CE  MET A  52     332.985 322.125 294.347  1.00 80.47           C
ATOM    370  N   LEU A  53     330.648 322.792 298.627  1.00 80.47           N
ATOM    371  CA  LEU A  53     331.587 323.398 299.554  1.00 82.05           C
ATOM    372  C   LEU A  53     332.958 323.524 298.880  1.00 82.05           C
ATOM    373  O   LEU A  53     333.093 323.364 297.665  1.00 82.05           O
ATOM    374  CB  LEU A  53     331.643 322.572 300.842  1.00 82.05           C
ATOM    375  CG  LEU A  53     332.316 323.087 302.117  1.00 82.05           C
ATOM    376  CD1 LEU A  53     331.703 324.405 302.541  1.00 82.05           C
ATOM    377  CD2 LEU A  53     332.194 322.058 303.225  1.00 82.05           C
ATOM    378  N   PHE A  54     333.984 323.809 299.676  1.00 82.05           N
ATOM    379  CA  PHE A  54     335.331 323.923 299.134  1.00 82.05           C
ATOM    380  C   PHE A  54     335.822 322.547 298.711  1.00 82.05           C
ATOM    381  O   PHE A  54     336.570 321.900 299.448  1.00 82.05           O
ATOM    382  CB  PHE A  54     336.285 324.495 300.182  1.00 84.50           C
ATOM    383  CG  PHE A  54     335.860 325.810 300.759  1.00 84.50           C
ATOM    384  CD1 PHE A  54     336.411 326.988 300.296  1.00 84.50           C
ATOM    385  CD2 PHE A  54     334.925 325.868 301.776  1.00 84.50           C
ATOM    386  CE1 PHE A  54     336.037 328.200 300.830  1.00 84.50           C
ATOM    387  CE2 PHE A  54     334.544 327.077 302.313  1.00 84.50           C
ATOM    388  CZ  PHE A  54     335.101 328.244 301.839  1.00 84.50           C
ATOM    389  N   SER A  55     335.428 322.119 297.513  1.00 90.97           N
ATOM    390  CA  SER A  55     335.637 320.736 297.059  1.00 90.97           C
ATOM    391  C   SER A  55     335.046 319.826 298.137  1.00 90.97           C
ATOM    392  O   SER A  55     333.833 319.887 298.380  1.00 90.97           O
ATOM    393  CB  SER A  55     337.110 320.518 296.744  1.00 90.97           C
ATOM    394  OG  SER A  55     337.553 321.407 295.735  1.00 90.97           O
ATOM    395  N   THR A  56     335.838 318.977 298.785  1.00 89.53           N
ATOM    396  CA  THR A  56     335.478 318.157 299.942  1.00 89.53           C
ATOM    397  C   THR A  56     334.308 317.214 299.680  1.00 89.53           C
ATOM    398  O   THR A  56     333.794 316.632 300.642  1.00 89.53           O
ATOM    399  CB  THR A  56     335.145 318.987 301.193  1.00 89.53           C
ATOM    400  OG1 THR A  56     334.104 319.922 300.893  1.00 89.53           O
ATOM    401  CG2 THR A  56     336.373 319.731 301.686  1.00 89.53           C
ATOM    402  N   SER A  57     333.866 317.047 298.434  1.00 89.53           N
ATOM    403  CA  SER A  57     332.877 316.026 298.081  1.00 91.53           C
ATOM    404  C   SER A  57     331.586 316.166 298.884  1.00 91.53           C
ATOM    405  O   SER A  57     331.000 315.174 299.317  1.00 91.53           O
ATOM    406  CB  SER A  57     333.465 314.624 298.257  1.00 91.53           C
ATOM    407  OG  SER A  57     332.518 313.626 297.928  1.00 91.53           O
ATOM    408  N   LEU A  58     331.131 317.399 299.089  1.00 91.53           N
ATOM    409  CA  LEU A  58     329.891 317.659 299.809  1.00 91.53           C
ATOM    410  C   LEU A  58     328.986 318.521 298.942  1.00 91.53           C
ATOM    411  O   LEU A  58     329.430 319.537 298.400  1.00 91.11           O
ATOM    412  CB  LEU A  58     330.163 318.347 301.148  1.00 91.11           C
ATOM    413  CG  LEU A  58     328.984 318.389 302.119  1.00 91.11           C
ATOM    414  CD1 LEU A  58     328.591 316.981 302.521  1.00 91.11           C
ATOM    415  CD2 LEU A  58     329.326 319.222 303.341  1.00 91.11           C
ATOM    416  N   LEU A  59     327.722 318.119 298.817  1.00 94.71           N
ATOM    417  CA  LEU A  59     326.777 318.779 297.930  1.00 94.71           C
ATOM    418  C   LEU A  59     325.432 318.961 298.617  1.00 94.71           C
ATOM    419  O   LEU A  59     325.062 318.191 299.506  1.00 94.71           O
ATOM    420  CB  LEU A  59     326.567 317.981 296.639  1.00 94.71           C
ATOM    421  CG  LEU A  59     327.775 317.729 295.744  1.00 94.71           C
ATOM    422  CD1 LEU A  59     327.349 316.926 294.532  1.00 94.71           C
ATOM    423  CD2 LEU A  59     328.406 319.039 295.325  1.00 94.71           C
ATOM    424  N   ALA A  60     324.700 319.981 298.179  1.00 97.69           N
ATOM    425  CA  ALA A  60     323.311 320.191 298.562  1.00 97.69           C
ATOM    426  C   ALA A  60     322.446 320.121 297.313  1.00 97.69           C
ATOM    427  O   ALA A  60     322.635 320.908 296.380  1.00 97.69           O
ATOM    428  CB  ALA A  60     323.119 321.538 299.255  1.00 97.69           C
ATOM    429  N   LEU A  61     321.497 319.189 297.300  1.00 97.69           N
ATOM    430  CA  LEU A  61     320.692 318.914 296.117  1.00 97.69           C
ATOM    431  C   LEU A  61     319.227 319.146 296.445  1.00103.43           C
ATOM    432  O   LEU A  61     318.701 318.556 297.395  1.00103.43           O
ATOM    433  CB  LEU A  61     320.915 317.481 295.625  1.00103.43           C
ATOM    434  CG  LEU A  61     321.972 317.265 294.540  1.00103.43           C
ATOM    435  CD1 LEU A  61     323.316 317.853 294.934  1.00104.40           C
ATOM    436  CD2 LEU A  61     322.112 315.785 294.240  1.00104.40           C
ATOM    437  N   VAL A  62     318.572 319.993 295.658  1.00104.40           N
ATOM    438  CA  VAL A  62     317.155 320.297 295.815  1.00104.40           C
ATOM    439  C   VAL A  62     316.450 319.904 294.527  1.00104.40           C
ATOM    440  O   VAL A  62     316.854 320.333 293.438  1.00104.40           O
ATOM    441  CB  VAL A  62     316.918 321.778 296.135  1.00104.40           C
ATOM    442  CG1 VAL A  62     315.431 322.075 296.183  1.00104.40           C
ATOM    443  CG2 VAL A  62     317.580 322.149 297.450  1.00109.20           C
ATOM    444  N   GLY A  63     315.400 319.096 294.648  1.00109.20           N
ATOM    445  CA  GLY A  63     314.654 318.685 293.477  1.00109.20           C
ATOM    446  C   GLY A  63     313.918 319.841 292.830  1.00109.20           C
ATOM    447  O   GLY A  63     313.628 320.862 293.452  1.00108.50           O
ATOM    448  N   ILE A  64     313.618 319.666 291.541  1.00108.50           N
ATOM    449  CA  ILE A  64     312.927 320.714 290.793  1.00108.50           C
ATOM    450  C   ILE A  64     311.503 320.887 291.303  1.00108.50           C
ATOM    451  O   ILE A  64     310.929 321.982 291.227  1.00108.50           O
ATOM    452  CB  ILE A  64     312.966 320.408 289.283  1.00108.50           C
ATOM    453  CG1 ILE A  64     312.654 321.669 288.473  1.00108.50           C
ATOM    454  CG2 ILE A  64     312.010 319.278 288.924  1.00108.50           C
ATOM    455  CD1 ILE A  64     313.538 322.846 288.809  1.00105.42           C
ATOM    456  N   LEU A  75     315.462 318.936 300.825  1.00101.27           N
ATOM    457  CA  LEU A  75     316.888 319.172 300.655  1.00101.27           C
ATOM    458  C   LEU A  75     317.632 317.897 301.020  1.00101.27           C
ATOM    459  O   LEU A  75     317.363 317.294 302.064  1.00101.27           O
ATOM    460  CB  LEU A  75     317.373 320.337 301.518  1.00 99.06           C
ATOM    461  CG  LEU A  75     318.886 320.487 301.700  1.00 99.06           C
ATOM    462  CD1 LEU A  75     319.544 321.184 300.517  1.00 99.06           C
ATOM    463  CD2 LEU A  75     319.205 321.216 302.994  1.00 99.06           C
ATOM    464  N   ARG A  76     318.555 317.485 300.159  1.00 99.06           N
ATOM    465  CA  ARG A  76     319.318 316.258 300.346  1.00 99.06           C
ATOM    466  C   ARG A  76     320.801 316.581 300.262  1.00 99.06           C
ATOM    467  O   ARG A  76     321.231 317.294 299.349  1.00 99.06           O
ATOM    468  CB  ARG A  76     318.929 315.213 299.302  1.00 95.74           C
ATOM    469  CG  ARG A  76     319.651 313.896 299.448  1.00 95.74           C
ATOM    470  CD  ARG A  76     319.237 312.939 298.353  1.00 95.74           C
ATOM    471  NE  ARG A  76     317.830 312.572 298.452  1.00 95.74           N
ATOM    472  CZ  ARG A  76     317.186 311.836 297.558  1.00 95.74           C
ATOM    473  NH1 ARG A  76     317.792 311.373 296.478  1.00 95.74           N
ATOM    474  NH2 ARG A  76     315.901 311.559 297.753  1.00 95.74           N
ATOM    475  N   ILE A  77     321.575 316.067 301.213  1.00 95.74           N
ATOM    476  CA  ILE A  77     323.018 316.270 301.257  1.00 94.80           C
ATOM    477  C   ILE A  77     323.686 314.969 300.833  1.00 94.80           C
ATOM    478  O   ILE A  77     323.475 313.924 301.457  1.00 94.80           O
ATOM    479  CB  ILE A  77     323.486 316.694 302.657  1.00 94.80           C
ATOM    480  CG1 ILE A  77     323.063 318.132 302.957  1.00 94.80           C
ATOM    481  CG2 ILE A  77     324.993 316.571 302.766  1.00 94.80           C
ATOM    482  CD1 ILE A  77     323.921 319.173 302.293  1.00 94.80           C
ATOM    483  N   ILE A  78     324.495 315.028 299.777  1.00 94.80           N
ATOM    484  CA  ILE A  78     325.097 313.836 299.194  1.00 92.92           C
ATOM    485  C   ILE A  78     326.607 314.013 299.119  1.00 92.92           C
ATOM    486  O   ILE A  78     327.114 315.070 298.732  1.00 92.92           O
ATOM    487  CB  ILE A  78     324.514 313.535 297.798  1.00 92.92           C
ATOM    488  CG1 ILE A  78     323.028 313.191 297.903  1.00 92.92           C
ATOM    489  CG2 ILE A  78     325.268 312.398 297.130  1.00 92.92           C
ATOM    490  CD1 ILE A  78     322.357 313.018 296.563  1.00 92.92           C
ATOM    491  N   ASN A  79     327.325 312.959 299.502  1.00 95.33           N
ATOM    492  CA  ASN A  79     328.772 312.878 299.348  1.00 95.33           C
ATOM    493  C   ASN A  79     329.054 312.171 298.029  1.00 95.33           C
ATOM    494  O   ASN A  79     328.761 310.978 297.888  1.00 95.33           O
ATOM    495  CB  ASN A  79     329.404 312.131 300.523  1.00 95.33           C
ATOM    496  CG  ASN A  79     330.923 312.169 300.501  1.00 95.33           C
ATOM    497  OD1 ASN A  79     331.542 312.389 299.463  1.00 95.33           O
ATOM    498  ND2 ASN A  79     331.532 311.961 301.663  1.00 95.33           N
ATOM    499  N   THR A  80     329.617 312.905 297.068  1.00 95.33           N
ATOM    500  CA  THR A  80     329.800 312.350 295.733  1.00 95.63           C
ATOM    501  C   THR A  80     330.888 311.285 295.684  1.00 95.63           C
ATOM    502  O   THR A  80     330.775 310.333 294.904  1.00 95.63           O
ATOM    503  CB  THR A  80     330.102 313.465 294.726  1.00 95.63           C
ATOM    504  OG1 THR A  80     330.327 312.891 293.434  1.00 95.63           O
ATOM    505  CG2 THR A  80     331.320 314.273 295.137  1.00 95.63           C
ATOM    506  N   LYS A  81     331.943 311.417 296.492  1.00 95.63           N
ATOM    507  CA  LYS A  81     332.986 310.397 296.501  1.00 95.63           C
ATOM    508  C   LYS A  81     332.477 309.063 297.028  1.00 95.63           C
ATOM    509  O   LYS A  81     332.771 308.018 296.441  1.00 94.21           O
ATOM    510  CB  LYS A  81     334.180 310.865 297.335  1.00 94.21           C
ATOM    511  CG  LYS A  81     335.185 309.766 297.633  1.00 94.21           C
ATOM    512  CD  LYS A  81     336.602 310.304 297.680  1.00 94.21           C
ATOM    513  CE  LYS A  81     336.894 310.986 299.005  1.00 94.21           C
ATOM    514  NZ  LYS A  81     338.327 310.856 299.383  1.00 94.21           N
ATOM    515  N   LYS A  82     331.710 309.079 298.115  1.00 94.21           N
ATOM    516  CA  LYS A  82     331.178 307.859 298.704  1.00 94.21           C
ATOM    517  C   LYS A  82     329.886 307.392 298.050  1.00 94.21           C
ATOM    518  O   LYS A  82     329.465 306.257 298.296  1.00 94.21           O
ATOM    519  CB  LYS A  82     330.933 308.063 300.203  1.00 97.25           C
ATOM    520  CG  LYS A  82     332.167 308.462 300.991  1.00 97.25           C
ATOM    521  CD  LYS A  82     333.163 307.321 301.080  1.00 97.25           C
ATOM    522  CE  LYS A  82     334.385 307.728 301.883  1.00 97.25           C
ATOM    523  NZ  LYS A  82     334.047 307.952 303.316  1.00 97.25           N
ATOM    524  N   HIS A  83     329.254 308.237 297.233  1.00 97.25           N
ATOM    525  CA  HIS A  83     327.953 307.947 296.628  1.00 99.43           C
ATOM    526  C   HIS A  83     326.898 307.618 297.679  1.00 99.43           C
ATOM    527  O   HIS A  83     325.978 306.835 297.424  1.00 99.43           O
ATOM    528  CB  HIS A  83     328.053 306.818 295.597  1.00 99.43           C
ATOM    529  CG  HIS A  83     328.843 307.181 294.379  1.00 99.43           C
ATOM    530  ND1 HIS A  83     330.154 307.600 294.439  1.00 99.43           N
ATOM    531  CD2 HIS A  83     328.506 307.190 293.067  1.00 99.43           C
ATOM    532  CE1 HIS A  83     330.592 307.849 293.218  1.00 99.43           C
ATOM    533  NE2 HIS A  83     329.611 307.609 292.367  1.00100.96           N
ATOM    534  N   SER A  84     327.021 308.211 298.861  1.00100.96           N
ATOM    535  CA  SER A  84     326.107 307.976 299.969  1.00100.96           C
ATOM    536  C   SER A  84     325.309 309.237 300.283  1.00100.96           C
ATOM    537  O   SER A  84     325.511 310.301 299.693  1.00100.96           O
ATOM    538  CB  SER A  84     326.876 307.505 301.207  1.00100.96           C
ATOM    539  OG  SER A  84     327.534 308.589 301.840  1.00100.96           O
ATOM    540  N   ILE A  85     324.390 309.102 301.232  1.00100.96           N
ATOM    541  CA  ILE A  85     323.518 310.189 301.661  1.00102.82           C
ATOM    542  C   ILE A  85     323.852 310.499 303.112  1.00102.82           C
ATOM    543  O   ILE A  85     323.680 309.648 303.994  1.00102.82           O
ATOM    544  CB  ILE A  85     322.037 309.824 301.502  1.00102.82           C
ATOM    545  CG1 ILE A  85     321.712 309.528 300.037  1.00102.82           C
ATOM    546  CG2 ILE A  85     321.155 310.940 302.037  1.00102.82           C
ATOM    547  CD1 ILE A  85     320.231 309.404 299.752  1.00105.26           C
ATOM    548  N   ILE A  86     324.330 311.719 303.370  1.00105.26           N
ATOM    549  CA  ILE A  86     324.584 312.139 304.744  1.00105.26           C
ATOM    550  C   ILE A  86     323.272 312.307 305.500  1.00105.26           C
ATOM    551  O   ILE A  86     323.130 311.853 306.640  1.00105.26           O
ATOM    552  CB  ILE A  86     325.419 313.432 304.779  1.00105.26           C
ATOM    553  CG1 ILE A  86     326.820 313.193 304.216  1.00105.26           C
ATOM    554  CG2 ILE A  86     325.507 313.973 306.197  1.00105.26           C
ATOM    555  CD1 ILE A  86     326.930 313.438 302.738  1.00105.26           C
ATOM    556  N   CYS A  87     322.291 312.954 304.876  1.00101.96           N
ATOM    557  CA  CYS A  87     321.039 313.245 305.559  1.00101.96           C
ATOM    558  C   CYS A  87     319.983 313.624 304.535  1.00101.96           C
ATOM    559  O   CYS A  87     320.262 313.775 303.342  1.00101.96           O
ATOM    560  CB  CYS A  87     321.215 314.369 306.585  1.00101.96           C
ATOM    561  SG  CYS A  87     321.586 315.975 305.852  1.00101.96           S
ATOM    562  N   GLU A  88     318.756 313.775 305.030  1.00101.96           N
ATOM    563  CA  GLU A  88     317.624 314.223 304.233  1.00 96.37           C
ATOM    564  C   GLU A  88     316.703 315.031 305.131  1.00 96.37           C
ATOM    565  O   GLU A  88     316.273 314.547 306.179  1.00 96.37           O
ATOM    566  CB  GLU A  88     316.870 313.038 303.621  1.00 96.37           C
ATOM    567  CG  GLU A  88     315.720 313.435 302.718  1.00 96.37           C
ATOM    568  CD  GLU A  88     315.032 312.232 302.101  1.00 96.37           C
ATOM    569  OE1 GLU A  88     315.444 311.093 302.396  1.00 96.37           O
ATOM    570  OE2 GLU A  88     314.074 312.424 301.328  1.00 85.90           O
ATOM    571  N   VAL A  89     316.397 316.257 304.708  1.00 85.90           N
ATOM    572  CA  VAL A  89     315.577 317.172 305.489  1.00 85.90           C
ATOM    573  C   VAL A  89     314.521 317.791 304.587  1.00 85.90           C
ATOM    574  O   VAL A  89     314.786 318.140 303.433  1.00 85.90           O
ATOM    575  CB  VAL A  89     316.423 318.271 306.174  1.00 85.90           C
ATOM    576  CG1 VAL A  89     317.335 317.668 307.230  1.00 85.90           C
ATOM    577  CG2 VAL A  89     317.243 319.029 305.150  1.00 85.90           C
ATOM    578  N   THR A  90     313.308 317.911 305.120  1.00 85.90           N
ATOM    579  CA  THR A  90     312.201 318.532 304.415  1.00 85.90           C
ATOM    580  C   THR A  90     311.870 319.873 305.055  1.00 85.90           C
ATOM    581  O   THR A  90     312.239 320.141 306.202  1.00 84.52           O
ATOM    582  CB  THR A  90     310.960 317.629 304.411  1.00 84.52           C
ATOM    583  OG1 THR A  90     310.031 318.097 303.427  1.00 84.52           O
ATOM    584  CG2 THR A  90     310.288 317.635 305.775  1.00 84.52           C
ATOM    585  N   PHE A  91     311.191 320.718 304.295  1.00 84.52           N
ATOM    586  CA  PHE A  91     310.758 322.035 304.731  1.00 84.52           C
ATOM    587  C   PHE A  91     309.306 322.238 304.330  1.00 84.52           C
ATOM    588  O   PHE A  91     308.834 321.641 303.357  1.00 83.03           O
ATOM    589  CB  PHE A  91     311.641 323.134 304.121  1.00 83.03           C
ATOM    590  CG  PHE A  91     313.077 323.066 304.559  1.00 83.03           C
ATOM    591  CD1 PHE A  91     313.507 323.771 305.667  1.00 83.03           C
ATOM    592  CD2 PHE A  91     313.990 322.277 303.882  1.00 83.03           C
ATOM    593  CE1 PHE A  91     314.822 323.709 306.080  1.00 83.03           C
ATOM    594  CE2 PHE A  91     315.305 322.210 304.293  1.00 83.03           C
ATOM    595  CZ  PHE A  91     315.721 322.925 305.394  1.00 82.15           C
ATOM    596  N   PRO A  92     308.561 323.057 305.077  1.00 82.15           N
ATOM    597  CA  PRO A  92     307.146 323.263 304.730  1.00 82.15           C
ATOM    598  C   PRO A  92     306.937 323.898 303.364  1.00 82.15           C
ATOM    599  O   PRO A  92     305.906 323.639 302.731  1.00 82.15           O
ATOM    600  CB  PRO A  92     306.632 324.165 305.861  1.00 82.15           C
ATOM    601  CG  PRO A  92     307.850 324.825 306.410  1.00 77.29           C
ATOM    602  CD  PRO A  92     308.960 323.833 306.263  1.00 77.29           C
ATOM    603  N   THR A  93     307.872 324.718 302.890  1.00 77.29           N
ATOM    604  CA  THR A  93     307.739 325.400 301.612  1.00 77.29           C
ATOM    605  C   THR A  93     308.980 325.158 300.762  1.00 77.29           C
ATOM    606  O   THR A  93     309.950 324.532 301.196  1.00 77.29           O
ATOM    607  CB  THR A  93     307.511 326.907 301.797  1.00 77.29           C
ATOM    608  OG1 THR A  93     308.696 327.512 302.329  1.00 77.29           O
ATOM    609  CG2 THR A  93     306.347 327.163 302.743  1.00 73.41           C
ATOM    610  N   SER A  94     308.936 325.664 299.530  1.00 73.41           N
ATOM    611  CA  SER A  94     310.037 325.468 298.597  1.00 73.41           C
ATOM    612  C   SER A  94     311.272 326.243 299.041  1.00 73.41           C
ATOM    613  O   SER A  94     311.175 327.359 299.556  1.00 73.41           O
ATOM    614  CB  SER A  94     309.622 325.906 297.193  1.00 73.41           C
ATOM    615  OG  SER A  94     309.837 327.294 297.014  1.00 73.41           O
ATOM    616  N   ILE A  95     312.438 325.643 298.831  1.00 73.41           N
ATOM    617  CA  ILE A  95     313.705 326.276 299.186  1.00 73.41           C
ATOM    618  C   ILE A  95     314.064 327.293 298.115  1.00 73.41           C
ATOM    619  O   ILE A  95     313.918 327.028 296.917  1.00 73.41           O
ATOM    620  CB  ILE A  95     314.813 325.224 299.341  1.00 73.41           C
ATOM    621  CG1 ILE A  95     314.393 324.150 300.339  1.00 73.41           C
ATOM    622  CG2 ILE A  95     316.108 325.879 299.789  1.00 73.41           C
ATOM    623  CD1 ILE A  95     315.265 322.931 300.293  1.00 73.41           C
ATOM    624  N   LEU A  96     314.531 328.464 298.539  1.00 73.41           N
ATOM    625  CA  LEU A  96     314.906 329.509 297.597  1.00 73.41           C
ATOM    626  C   LEU A  96     316.408 329.698 297.462  1.00 73.41           C
ATOM    627  O   LEU A  96     316.867 330.149 296.408  1.00 73.41           O
ATOM    628  CB  LEU A  96     314.267 330.843 297.995  1.00 73.41           C
ATOM    629  CG  LEU A  96     312.916 331.166 297.352  1.00 73.41           C
ATOM    630  CD1 LEU A  96     311.881 330.089 297.641  1.00 73.41           C
ATOM    631  CD2 LEU A  96     312.425 332.523 297.815  1.00 73.41           C
ATOM    632  N   SER A  97     317.187 329.375 298.491  1.00 73.41           N
ATOM    633  CA  SER A  97     318.633 329.517 298.402  1.00 73.41           C
ATOM    634  C   SER A  97     319.287 328.658 299.472  1.00 73.41           C
ATOM    635  O   SER A  97     318.711 328.410 300.533  1.00 73.41           O
ATOM    636  CB  SER A  97     319.064 330.978 298.552  1.00 73.41           C
ATOM    637  OG  SER A  97     320.473 331.082 298.640  1.00 73.41           O
ATOM    638  N   VAL A  98     320.497 328.198 299.168  1.00 73.41           N
ATOM    639  CA  VAL A  98     321.309 327.412 300.090  1.00 74.01           C
ATOM    640  C   VAL A  98     322.751 327.879 299.958  1.00 74.01           C
ATOM    641  O   VAL A  98     323.251 328.039 298.841  1.00 74.01           O
ATOM    642  CB  VAL A  98     321.208 325.899 299.808  1.00 74.01           C
ATOM    643  CG1 VAL A  98     322.220 325.133 300.639  1.00 74.01           C
ATOM    644  CG2 VAL A  98     319.801 325.390 300.072  1.00 74.01           C
ATOM    645  N   LYS A  99     323.417 328.104 301.085  1.00 74.01           N
ATOM    646  CA  LYS A  99     324.820 328.489 301.076  1.00 74.01           C
ATOM    647  C   LYS A  99     325.531 327.807 302.232  1.00 75.02           C
ATOM    648  O   LYS A  99     324.912 327.419 303.225  1.00 75.02           O
ATOM    649  CB  LYS A  99     324.991 330.010 301.163  1.00 75.02           C
ATOM    650  CG  LYS A  99     324.682 330.734 299.869  1.00 75.02           C
ATOM    651  CD  LYS A  99     325.078 332.195 299.939  1.00 75.02           C
ATOM    652  CE  LYS A  99     325.026 332.840 298.566  1.00 75.02           C
ATOM    653  NZ  LYS A  99     323.630 332.958 298.063  1.00 75.02           N
ATOM    654  N   MET A 100     326.847 327.651 302.092  1.00 75.02           N
ATOM    655  CA  MET A 100     327.630 326.988 303.122  1.00 76.98           C
ATOM    656  C   MET A 100     328.973 327.674 303.303  1.00 76.98           C
ATOM    657  O   MET A 100     329.498 328.314 302.389  1.00 76.98           O
ATOM    658  CB  MET A 100     327.893 325.514 302.805  1.00 76.98           C
ATOM    659  CG  MET A 100     326.675 324.651 302.620  1.00 76.98           C
ATOM    660  SD  MET A 100     327.087 323.234 301.591  1.00 76.98           S
ATOM    661  CE  MET A 100     325.531 322.369 301.598  1.00 76.98           C
ATOM    662  N   ASN A 101     329.508 327.531 304.509  1.00 76.98           N
ATOM    663  CA  ASN A 101     330.930 327.630 304.793  1.00 76.98           C
ATOM    664  C   ASN A 101     331.341 326.327 305.464  1.00 73.53           C
ATOM    665  O   ASN A 101     330.532 325.407 305.614  1.00 73.53           O
ATOM    666  CB  ASN A 101     331.251 328.853 305.664  1.00 73.53           C
ATOM    667  CG  ASN A 101     330.415 328.910 306.928  1.00 73.53           C
ATOM    668  OD1 ASN A 101     329.731 327.952 307.284  1.00 73.53           O
ATOM    669  ND2 ASN A 101     330.468 330.044 307.616  1.00 73.53           N
ATOM    670  N   LYS A 102     332.599 326.235 305.879  1.00 73.41           N
ATOM    671  CA  LYS A 102     333.053 325.014 306.534  1.00 73.41           C
ATOM    672  C   LYS A 102     332.589 324.910 307.982  1.00 73.41           C
ATOM    673  O   LYS A 102     333.079 324.043 308.711  1.00 73.41           O
ATOM    674  CB  LYS A 102     334.579 324.905 306.459  1.00 73.41           C
ATOM    675  CG  LYS A 102     335.324 326.086 307.038  1.00 73.41           C
ATOM    676  CD  LYS A 102     336.823 325.844 307.031  1.00 73.41           C
ATOM    677  CE  LYS A 102     337.361 325.753 305.616  1.00 73.41           C
ATOM    678  NZ  LYS A 102     338.824 325.480 305.601  1.00 73.41           N
ATOM    679  N   SER A 103     331.658 325.762 308.410  1.00 73.41           N
ATOM    680  CA  SER A 103     331.129 325.737 309.766  1.00 73.41           C
ATOM    681  C   SER A 103     329.611 325.665 309.842  1.00 73.41           C
ATOM    682  O   SER A 103     329.090 324.981 310.724  1.00 73.41           O
ATOM    683  CB  SER A 103     331.595 326.975 310.544  1.00 73.41           C
ATOM    684  OG  SER A 103     331.040 326.996 311.847  1.00 73.41           O
ATOM    685  N   ARG A 104     328.890 326.337 308.947  1.00 73.41           N
ATOM    686  CA  ARG A 104     327.443 326.450 309.058  1.00 73.41           C
ATOM    687  C   ARG A 104     326.785 326.169 307.716  1.00 73.41           C
ATOM    688  O   ARG A 104     327.394 326.333 306.657  1.00 73.41           O
ATOM    689  CB  ARG A 104     327.026 327.842 309.545  1.00 73.41           C
ATOM    690  CG  ARG A 104     327.461 328.164 310.960  1.00 73.41           C
ATOM    691  CD  ARG A 104     326.647 327.395 311.976  1.00 73.41           C
ATOM    692  NE  ARG A 104     326.868 327.892 313.327  1.00 73.41           N
ATOM    693  CZ  ARG A 104     327.785 327.418 314.159  1.00 73.41           C
ATOM    694  NH1 ARG A 104     328.588 326.428 313.810  1.00 73.41           N
ATOM    695  NH2 ARG A 104     327.900 327.952 315.370  1.00 73.41           N
ATOM    696  N   LEU A 105     325.525 325.744 307.779  1.00 73.41           N
ATOM    697  CA  LEU A 105     324.694 325.514 306.606  1.00 73.41           C
ATOM    698  C   LEU A 105     323.409 326.316 306.747  1.00 73.41           C
ATOM    699  O   LEU A 105     322.737 326.244 307.780  1.00 73.41           O
ATOM    700  CB  LEU A 105     324.376 324.023 306.449  1.00 73.41           C
ATOM    701  CG  LEU A 105     323.587 323.528 305.236  1.00 73.41           C
ATOM    702  CD1 LEU A 105     324.018 322.116 304.907  1.00 73.41           C
ATOM    703  CD2 LEU A 105     322.094 323.558 305.493  1.00 73.41           C
ATOM    704  N   VAL A 106     323.061 327.067 305.705  1.00 73.41           N
ATOM    705  CA  VAL A 106     321.909 327.962 305.730  1.00 73.41           C
ATOM    706  C   VAL A 106     320.929 327.560 304.640  1.00 73.41           C
ATOM    707  O   VAL A 106     321.315 327.403 303.478  1.00 73.41           O
ATOM    708  CB  VAL A 106     322.324 329.433 305.560  1.00 73.41           C
ATOM    709  CG1 VAL A 106     321.099 330.304 305.476  1.00 73.41           C
ATOM    710  CG2 VAL A 106     323.154 329.864 306.744  1.00 73.41           C
ATOM    711  N   VAL A 107     319.665 327.401 305.019  1.00 73.41           N
ATOM    712  CA  VAL A 107     318.572 327.150 304.090  1.00 73.41           C
ATOM    713  C   VAL A 107     317.556 328.268 304.244  1.00 73.41           C
ATOM    714  O   VAL A 107     317.091 328.538 305.358  1.00 73.41           O
ATOM    715  CB  VAL A 107     317.916 325.783 304.341  1.00 73.41           C
ATOM    716  CG1 VAL A 107     316.696 325.622 303.457  1.00 73.41           C
ATOM    717  CG2 VAL A 107     318.907 324.671 304.098  1.00 73.41           C
ATOM    718  N   LEU A 108     317.204 328.909 303.137  1.00 73.41           N
ATOM    719  CA  LEU A 108     316.272 330.029 303.141  1.00 74.21           C
ATOM    720  C   LEU A 108     314.984 329.645 302.430  1.00 74.21           C
ATOM    721  O   LEU A 108     315.011 329.195 301.280  1.00 74.21           O
ATOM    722  CB  LEU A 108     316.894 331.263 302.490  1.00 74.21           C
ATOM    723  CG  LEU A 108     316.018 332.516 302.487  1.00 74.21           C
ATOM    724  CD1 LEU A 108     316.817 333.756 302.752  1.00 74.21           C
ATOM    725  CD2 LEU A 108     315.360 332.649 301.128  1.00 74.21           C
ATOM    726  N   LEU A 109     313.864 329.834 303.115  1.00 74.21           N
ATOM    727  CA  LEU A 109     312.548 329.841 302.501  1.00 74.21           C
ATOM    728  C   LEU A 109     312.091 331.285 302.343  1.00 74.00           C
ATOM    729  O   LEU A 109     312.787 332.221 302.738  1.00 74.00           O
ATOM    730  CB  LEU A 109     311.540 329.043 303.333  1.00 74.00           C
ATOM    731  CG  LEU A 109     312.029 327.682 303.829  1.00 74.00           C
ATOM    732  CD1 LEU A 109     310.974 327.007 304.685  1.00 74.00           C
ATOM    733  CD2 LEU A 109     312.415 326.799 302.660  1.00 74.00           C
ATOM    734  N   GLN A 110     310.899 331.457 301.769  1.00 74.00           N
ATOM    735  CA  GLN A 110     310.445 332.793 301.398  1.00 74.00           C
ATOM    736  C   GLN A 110     310.335 333.711 302.610  1.00 74.00           C
ATOM    737  O   GLN A 110     310.543 334.925 302.499  1.00 73.41           O
ATOM    738  CB  GLN A 110     309.102 332.723 300.662  1.00 73.41           C
ATOM    739  CG  GLN A 110     307.978 331.981 301.388  1.00 73.41           C
ATOM    740  CD  GLN A 110     308.228 330.495 301.536  1.00 73.41           C
ATOM    741  OE1 GLN A 110     308.547 329.805 300.568  1.00 73.41           O
ATOM    742  NE2 GLN A 110     308.095 329.994 302.759  1.00 73.41           N
ATOM    743  N   GLU A 111     310.013 333.152 303.780  1.00 73.41           N
ATOM    744  CA  GLU A 111     309.837 333.949 304.988  1.00 73.41           C
ATOM    745  C   GLU A 111     310.643 333.436 306.176  1.00 73.41           C
ATOM    746  O   GLU A 111     310.584 334.040 307.252  1.00 73.41           O
ATOM    747  CB  GLU A 111     308.355 334.019 305.376  1.00 73.41           C
ATOM    748  CG  GLU A 111     307.440 334.508 304.270  1.00 73.41           C
ATOM    749  CD  GLU A 111     305.986 334.538 304.694  1.00 73.41           C
ATOM    750  OE1 GLU A 111     305.651 333.898 305.713  1.00 73.41           O
ATOM    751  OE2 GLU A 111     305.179 335.201 304.009  1.00 73.41           O
ATOM    752  N   GLN A 112     311.393 332.347 306.018  1.00 73.41           N
ATOM    753  CA  GLN A 112     312.104 331.751 307.139  1.00 73.41           C
ATOM    754  C   GLN A 112     313.540 331.459 306.737  1.00 73.41           C
ATOM    755  O   GLN A 112     313.827 331.156 305.577  1.00 73.41           O
ATOM    756  CB  GLN A 112     311.422 330.462 307.611  1.00 73.41           C
ATOM    757  CG  GLN A 112     310.006 330.660 308.117  1.00 73.41           C
ATOM    758  CD  GLN A 112     309.163 329.411 307.978  1.00 73.41           C
ATOM    759  OE1 GLN A 112     309.364 328.612 307.064  1.00 73.41           O
ATOM    760  NE2 GLN A 112     308.210 329.235 308.885  1.00 73.41           N
ATOM    761  N   ILE A 113     314.440 331.550 307.712  1.00 73.41           N
ATOM    762  CA  ILE A 113     315.846 331.208 307.539  1.00 73.41           C
ATOM    763  C   ILE A 113     316.191 330.119 308.544  1.00 73.41           C
ATOM    764  O   ILE A 113     315.946 330.277 309.744  1.00 73.41           O
ATOM    765  CB  ILE A 113     316.761 332.431 307.725  1.00 73.41           C
ATOM    766  CG1 ILE A 113     316.657 333.363 306.522  1.00 73.41           C
ATOM    767  CG2 ILE A 113     318.200 331.998 307.938  1.00 73.41           C
ATOM    768  CD1 ILE A 113     317.221 334.737 306.772  1.00 73.41           C
ATOM    769  N   TYR A 114     316.752 329.021 308.055  1.00 73.41           N
ATOM    770  CA  TYR A 114     317.192 327.917 308.893  1.00 73.41           C
ATOM    771  C   TYR A 114     318.709 327.838 308.844  1.00 73.41           C
ATOM    772  O   TYR A 114     319.299 327.879 307.761  1.00 73.41           O
ATOM    773  CB  TYR A 114     316.582 326.594 308.428  1.00 73.41           C
ATOM    774  CG  TYR A 114     315.099 326.464 308.685  1.00 73.41           C
ATOM    775  CD1 TYR A 114     314.178 327.190 307.945  1.00 73.41           C
ATOM    776  CD2 TYR A 114     314.620 325.597 309.654  1.00 73.41           C
ATOM    777  CE1 TYR A 114     312.822 327.065 308.173  1.00 73.41           C
ATOM    778  CE2 TYR A 114     313.268 325.464 309.890  1.00 73.41           C
ATOM    779  CZ  TYR A 114     312.373 326.201 309.148  1.00 73.41           C
ATOM    780  OH  TYR A 114     311.025 326.072 309.381  1.00 73.41           O
ATOM    781  N   ILE A 115     319.337 327.725 310.009  1.00 73.41           N
ATOM    782  CA  ILE A 115     320.788 327.637 310.107  1.00 73.41           C
ATOM    783  C   ILE A 115     321.139 326.299 310.736  1.00 73.41           C
ATOM    784  O   ILE A 115     320.623 325.959 311.809  1.00 73.41           O
ATOM    785  CB  ILE A 115     321.385 328.793 310.924  1.00 73.41           C
ATOM    786  CG1 ILE A 115     321.062 330.129 310.261  1.00 75.06           C
ATOM    787  CG2 ILE A 115     322.883 328.623 311.057  1.00 75.06           C
ATOM    788  CD1 ILE A 115     321.355 331.321 311.134  1.00 75.06           C
ATOM    789  N   TYR A 116     322.007 325.544 310.073  1.00 75.06           N
ATOM    790  CA  TYR A 116     322.435 324.235 310.535  1.00 75.06           C
ATOM    791  C   TYR A 116     323.945 324.221 310.721  1.00 75.06           C
ATOM    792  O   TYR A 116     324.687 324.915 310.020  1.00 75.06           O
ATOM    793  CB  TYR A 116     322.034 323.136 309.544  1.00 75.06           C
ATOM    794  CG  TYR A 116     320.560 322.816 309.523  1.00 76.98           C
ATOM    795  CD1 TYR A 116     319.644 323.698 308.972  1.00 76.98           C
ATOM    796  CD2 TYR A 116     320.087 321.620 310.037  1.00 76.98           C
ATOM    797  CE1 TYR A 116     318.298 323.403 308.948  1.00 76.98           C
ATOM    798  CE2 TYR A 116     318.743 321.316 310.016  1.00 76.98           C
ATOM    799  CZ  TYR A 116     317.853 322.210 309.472  1.00 76.98           C
ATOM    800  OH  TYR A 116     316.511 321.907 309.450  1.00 76.98           O
ATOM    801  N   ASP A 117     324.396 323.425 311.687  1.00 76.98           N
ATOM    802  CA  ASP A 117     325.821 323.173 311.863  1.00 79.30           C
ATOM    803  C   ASP A 117     326.241 322.173 310.795  1.00 79.30           C
ATOM    804  O   ASP A 117     325.593 321.139 310.619  1.00 79.30           O
ATOM    805  CB  ASP A 117     326.108 322.644 313.265  1.00 79.30           C
ATOM    806  CG  ASP A 117     327.581 322.700 313.622  1.00 79.30           C
ATOM    807  OD1 ASP A 117     328.426 322.703 312.702  1.00 79.30           O
ATOM    808  OD2 ASP A 117     327.898 322.738 314.829  1.00 79.30           O
ATOM    809  N   ILE A 118     327.317 322.478 310.067  1.00 79.30           N
ATOM    810  CA  ILE A 118     327.696 321.620 308.950  1.00 76.57           C
ATOM    811  C   ILE A 118     328.131 320.243 309.441  1.00 76.57           C
ATOM    812  O   ILE A 118     327.877 319.233 308.775  1.00 76.57           O
ATOM    813  CB  ILE A 118     328.783 322.297 308.091  1.00 76.57           C
ATOM    814  CG1 ILE A 118     328.971 321.531 306.784  1.00 76.57           C
ATOM    815  CG2 ILE A 118     330.099 322.396 308.842  1.00 76.57           C
ATOM    816  CD1 ILE A 118     327.731 321.493 305.924  1.00 76.57           C
ATOM    817  N   ASN A 119     328.775 320.171 310.605  1.00 76.27           N
ATOM    818  CA  ASN A 119     329.294 318.907 311.111  1.00 76.27           C
ATOM    819  C   ASN A 119     328.195 318.057 311.738  1.00 76.27           C
ATOM    820  O   ASN A 119     327.913 316.951 311.267  1.00 76.27           O
ATOM    821  CB  ASN A 119     330.403 319.162 312.132  1.00 76.27           C
ATOM    822  CG  ASN A 119     331.528 320.007 311.570  1.00 76.27           C
ATOM    823  OD1 ASN A 119     331.790 321.110 312.048  1.00 76.27           O
ATOM    824  ND2 ASN A 119     332.195 319.496 310.544  1.00 76.27           N
ATOM    825  N   THR A 120     327.565 318.569 312.797  1.00 74.64           N
ATOM    826  CA  THR A 120     326.595 317.770 313.538  1.00 74.64           C
ATOM    827  C   THR A 120     325.271 317.658 312.790  1.00 74.64           C
ATOM    828  O   THR A 120     324.535 316.680 312.964  1.00 74.64           O
ATOM    829  CB  THR A 120     326.380 318.368 314.928  1.00 74.64           C
ATOM    830  OG1 THR A 120     325.677 319.610 314.811  1.00 74.64           O
ATOM    831  CG2 THR A 120     327.715 318.612 315.611  1.00 74.64           C
ATOM    832  N   MET A 121     324.943 318.660 311.974  1.00 74.64           N
ATOM    833  CA  MET A 121     323.755 318.736 311.119  1.00 74.64           C
ATOM    834  C   MET A 121     322.497 318.969 311.951  1.00 74.64           C
ATOM    835  O   MET A 121     321.385 319.006 311.403  1.00 74.64           O
ATOM    836  CB  MET A 121     323.628 317.492 310.225  1.00 73.41           C
ATOM    837  CG  MET A 121     322.504 317.506 309.194  1.00 73.41           C
ATOM    838  SD  MET A 121     322.725 318.816 307.973  1.00 73.41           S
ATOM    839  CE  MET A 121     324.094 318.174 307.014  1.00 73.41           C
ATOM    840  N   ARG A 122     322.628 319.161 313.260  1.00 73.41           N
ATOM    841  CA  ARG A 122     321.476 319.531 314.068  1.00 73.41           C
ATOM    842  C   ARG A 122     321.132 321.003 313.863  1.00 73.41           C
ATOM    843  O   ARG A 122     322.000 321.837 313.595  1.00 73.41           O
ATOM    844  CB  ARG A 122     321.739 319.257 315.550  1.00 73.41           C
ATOM    845  CG  ARG A 122     322.826 320.124 316.164  1.00 73.41           C
ATOM    846  CD  ARG A 122     322.978 319.847 317.651  1.00 73.41           C
ATOM    847  NE  ARG A 122     321.827 320.327 318.408  1.00 73.41           N
ATOM    848  CZ  ARG A 122     321.738 321.529 318.957  1.00 73.41           C
ATOM    849  NH1 ARG A 122     322.726 322.405 318.868  1.00 73.41           N
ATOM    850  NH2 ARG A 122     320.630 321.863 319.610  1.00 73.41           N
ATOM    851  N   LEU A 123     319.844 321.315 313.989  1.00 73.41           N
ATOM    852  CA  LEU A 123     319.389 322.690 313.826  1.00 73.41           C
ATOM    853  C   LEU A 123     319.907 323.557 314.967  1.00 73.41           C
ATOM    854  O   LEU A 123     319.884 323.153 316.134  1.00 73.41           O
ATOM    855  CB  LEU A 123     317.862 322.738 313.775  1.00 73.41           C
ATOM    856  CG  LEU A 123     317.189 324.035 313.317  1.00 73.41           C
ATOM    857  CD1 LEU A 123     315.878 323.723 312.618  1.00 73.41           C
ATOM    858  CD2 LEU A 123     316.950 324.981 314.483  1.00 73.41           C
ATOM    859  N   LEU A 124     320.373 324.755 314.629  1.00 73.41           N
ATOM    860  CA  LEU A 124     320.902 325.687 315.618  1.00 73.41           C
ATOM    861  C   LEU A 124     320.016 326.896 315.857  1.00 73.41           C
ATOM    862  O   LEU A 124     319.867 327.324 317.001  1.00 73.41           O
ATOM    863  CB  LEU A 124     322.294 326.172 315.204  1.00 73.41           C
ATOM    864  CG  LEU A 124     323.350 325.100 314.961  1.00 73.41           C
ATOM    865  CD1 LEU A 124     324.648 325.744 314.525  1.00 73.41           C
ATOM    866  CD2 LEU A 124     323.555 324.274 316.213  1.00 73.41           C
ATOM    867  N   HIS A 125     319.424 327.462 314.809  1.00 73.41           N
ATOM    868  CA  HIS A 125     318.673 328.700 314.958  1.00 73.41           C
ATOM    869  C   HIS A 125     317.637 328.790 313.850  1.00 73.41           C
ATOM    870  O   HIS A 125     317.725 328.101 312.832  1.00 73.41           O
ATOM    871  CB  HIS A 125     319.602 329.917 314.931  1.00 73.41           C
ATOM    872  CG  HIS A 125     318.976 331.169 315.458  1.00 73.41           C
ATOM    873  ND1 HIS A 125     318.475 332.154 314.635  1.00 73.41           N
ATOM    874  CD2 HIS A 125     318.771 331.597 316.725  1.00 73.41           C
ATOM    875  CE1 HIS A 125     317.988 333.135 315.374  1.00 73.41           C
ATOM    876  NE2 HIS A 125     318.155 332.823 316.645  1.00 73.41           N
ATOM    877  N   THR A 126     316.646 329.649 314.071  1.00 73.56           N
ATOM    878  CA  THR A 126     315.624 329.949 313.080  1.00 73.56           C
ATOM    879  C   THR A 126     315.320 331.433 313.157  1.00 73.56           C
ATOM    880  O   THR A 126     315.184 331.980 314.253  1.00 73.56           O
ATOM    881  CB  THR A 126     314.352 329.128 313.321  1.00 73.56           C
ATOM    882  OG1 THR A 126     314.631 327.740 313.111  1.00 73.56           O
ATOM    883  CG2 THR A 126     313.249 329.559 312.369  1.00 73.56           C
ATOM    884  N   ILE A 127     315.227 332.087 312.004  1.00 73.56           N
ATOM    885  CA  ILE A 127     314.946 333.515 311.928  1.00 73.56           C
ATOM    886  C   ILE A 127     313.656 333.714 311.148  1.00 73.41           C
ATOM    887  O   ILE A 127     313.502 333.185 310.041  1.00 73.41           O
ATOM    888  CB  ILE A 127     316.105 334.287 311.277  1.00 73.41           C
ATOM    889  CG1 ILE A 127     317.406 334.028 312.034  1.00 73.41           C
ATOM    890  CG2 ILE A 127     315.802 335.772 311.239  1.00 73.41           C
ATOM    891  CD1 ILE A 127     318.626 334.534 311.314  1.00 73.41           C
ATOM    892  N   GLU A 128     312.727 334.469 311.729  1.00 73.41           N
ATOM    893  CA  GLU A 128     311.485 334.836 311.060  1.00 73.41           C
ATOM    894  C   GLU A 128     311.669 336.214 310.439  1.00 73.41           C
ATOM    895  O   GLU A 128     311.717 337.223 311.153  1.00 73.41           O
ATOM    896  CB  GLU A 128     310.318 334.823 312.040  1.00 73.41           C
ATOM    897  CG  GLU A 128     309.991 333.454 312.602  1.00 73.41           C
ATOM    898  CD  GLU A 128     309.203 332.604 311.630  1.00 73.41           C
ATOM    899  OE1 GLU A 128     308.421 333.174 310.842  1.00 73.41           O
ATOM    900  OE2 GLU A 128     309.366 331.367 311.654  1.00 73.41           O
ATOM    901  N   THR A 129     311.773 336.259 309.119  1.00 73.41           N
ATOM    902  CA  THR A 129     312.003 337.499 308.397  1.00 73.41           C
ATOM    903  C   THR A 129     310.796 337.854 307.539  1.00 73.41           C
ATOM    904  O   THR A 129     309.942 337.016 307.241  1.00 73.41           O
ATOM    905  CB  THR A 129     313.251 337.394 307.516  1.00 73.41           C
ATOM    906  OG1 THR A 129     313.529 338.668 306.922  1.00 73.41           O
ATOM    907  CG2 THR A 129     313.043 336.362 306.415  1.00 73.41           C
ATOM    908  N   ASN A 130     310.735 339.122 307.149  1.00 73.41           N
ATOM    909  CA  ASN A 130     309.721 339.560 306.211  1.00 73.41           C
ATOM    910  C   ASN A 130     309.952 338.880 304.862  1.00 73.41           C
ATOM    911  O   ASN A 130     311.090 338.546 304.518  1.00 73.41           O
ATOM    912  CB  ASN A 130     309.763 341.075 306.056  1.00 73.41           C
ATOM    913  CG  ASN A 130     309.264 341.796 307.291  1.00 73.41           C
ATOM    914  OD1 ASN A 130     308.280 341.388 307.905  1.00 73.41           O
ATOM    915  ND2 ASN A 130     309.947 342.870 307.665  1.00 73.41           N
ATOM    916  N   PRO A 131     308.886 338.650 304.084  1.00 73.41           N
ATOM    917  CA  PRO A 131     309.009 337.807 302.886  1.00 73.41           C
ATOM    918  C   PRO A 131     310.083 338.257 301.908  1.00 73.41           C
ATOM    919  O   PRO A 131     310.221 339.448 301.614  1.00 73.41           O
ATOM    920  CB  PRO A 131     307.614 337.905 302.257  1.00 73.41           C
ATOM    921  CG  PRO A 131     306.711 338.138 303.413  1.00 73.41           C
ATOM    922  CD  PRO A 131     307.486 339.019 304.356  1.00 73.41           C
ATOM    923  N   ASN A 132     310.854 337.293 301.407  1.00 73.41           N
ATOM    924  CA  ASN A 132     311.867 337.517 300.376  1.00 73.41           C
ATOM    925  C   ASN A 132     311.590 336.575 299.216  1.00 73.41           C
ATOM    926  O   ASN A 132     312.119 335.459 299.161  1.00 73.41           O
ATOM    927  CB  ASN A 132     313.281 337.306 300.918  1.00 73.41           C
ATOM    928  CG  ASN A 132     313.836 338.537 301.595  1.00 73.41           C
ATOM    929  OD1 ASN A 132     313.626 339.658 301.136  1.00 73.41           O
ATOM    930  ND2 ASN A 132     314.560 338.334 302.688  1.00 73.41           N
ATOM    931  N   PRO A 133     310.745 336.991 298.269  1.00 73.41           N
ATOM    932  CA  PRO A 133     310.474 336.141 297.100  1.00 73.41           C
ATOM    933  C   PRO A 133     311.683 335.931 296.207  1.00 73.41           C
ATOM    934  O   PRO A 133     311.688 334.974 295.422  1.00 73.41           O
ATOM    935  CB  PRO A 133     309.369 336.905 296.367  1.00 73.41           C
ATOM    936  CG  PRO A 133     309.614 338.333 296.734  1.00 73.41           C
ATOM    937  CD  PRO A 133     310.097 338.308 298.158  1.00 73.41           C
ATOM    938  N   ARG A 134     312.698 336.785 296.300  1.00 73.41           N
ATOM    939  CA  ARG A 134     313.876 336.681 295.450  1.00 73.41           C
ATOM    940  C   ARG A 134     314.948 335.755 296.011  1.00 73.41           C
ATOM    941  O   ARG A 134     315.911 335.456 295.298  1.00 73.41           O
ATOM    942  CB  ARG A 134     314.467 338.074 295.211  1.00 73.41           C
ATOM    943  CG  ARG A 134     314.240 338.597 293.806  1.00 73.41           C
ATOM    944  CD  ARG A 134     314.688 340.040 293.664  1.00 73.41           C
ATOM    945  NE  ARG A 134     313.733 340.968 294.258  1.00 73.41           N
ATOM    946  CZ  ARG A 134     313.937 342.271 294.386  1.00 73.41           C
ATOM    947  NH1 ARG A 134     315.059 342.837 293.975  1.00 73.41           N
ATOM    948  NH2 ARG A 134     312.992 343.024 294.939  1.00 73.41           N
ATOM    949  N   GLY A 135     314.809 335.291 297.251  1.00 73.41           N
ATOM    950  CA  GLY A 135     315.755 334.348 297.818  1.00 73.41           C
ATOM    951  C   GLY A 135     317.176 334.870 297.891  1.00 73.41           C
ATOM    952  O   GLY A 135     318.115 334.188 297.472  1.00 73.41           O
ATOM    953  N   LEU A 136     317.348 336.076 298.421  1.00 73.41           N
ATOM    954  CA  LEU A 136     318.636 336.756 298.423  1.00 73.41           C
ATOM    955  C   LEU A 136     319.320 336.592 299.774  1.00 73.41           C
ATOM    956  O   LEU A 136     318.697 336.804 300.818  1.00 73.41           O
ATOM    957  CB  LEU A 136     318.450 338.240 298.116  1.00 73.41           C
ATOM    958  CG  LEU A 136     317.880 338.579 296.742  1.00 73.41           C
ATOM    959  CD1 LEU A 136     317.580 340.059 296.660  1.00 73.41           C
ATOM    960  CD2 LEU A 136     318.844 338.166 295.650  1.00 73.41           C
ATOM    961  N   MET A 137     320.599 336.222 299.748  1.00 73.41           N
ATOM    962  CA  MET A 137     321.437 336.213 300.942  1.00 73.41           C
ATOM    963  C   MET A 137     322.863 336.547 300.541  1.00 73.41           C
ATOM    964  O   MET A 137     323.144 336.900 299.394  1.00 73.41           O
ATOM    965  CB  MET A 137     321.433 334.866 301.672  1.00 73.41           C
ATOM    966  CG  MET A 137     320.135 334.458 302.298  1.00 73.41           C
ATOM    967  SD  MET A 137     320.297 332.896 303.180  1.00 73.41           S
ATOM    968  CE  MET A 137     320.386 331.739 301.824  1.00 73.41           C
ATOM    969  N   ALA A 138     323.754 336.439 301.519  1.00 73.41           N
ATOM    970  CA  ALA A 138     325.190 336.472 301.306  1.00 73.41           C
ATOM    971  C   ALA A 138     325.856 335.832 302.512  1.00 73.41           C
ATOM    972  O   ALA A 138     325.559 336.191 303.652  1.00 73.41           O
ATOM    973  CB  ALA A 138     325.701 337.897 301.096  1.00 78.14           C
ATOM    974  N   MET A 139     326.746 334.882 302.254  1.00 78.14           N
ATOM    975  CA  MET A 139     327.389 334.099 303.298  1.00 78.14           C
ATOM    976  C   MET A 139     328.895 334.291 303.238  1.00 78.14           C
ATOM    977  O   MET A 139     329.506 334.117 302.180  1.00 78.14           O
ATOM    978  CB  MET A 139     327.071 332.614 303.147  1.00 78.14           C
ATOM    979  CG  MET A 139     327.879 331.732 304.069  1.00 78.14           C
ATOM    980  SD  MET A 139     327.013 331.326 305.583  1.00 77.59           S
ATOM    981  CE  MET A 139     326.138 329.867 305.047  1.00 77.59           C
ATOM    982  N   SER A 140     329.489 334.641 304.365  1.00 77.59           N
ATOM    983  CA  SER A 140     330.940 334.705 304.431  1.00 77.59           C
ATOM    984  C   SER A 140     331.501 333.293 304.534  1.00 77.59           C
ATOM    985  O   SER A 140     331.098 332.536 305.424  1.00 77.59           O
ATOM    986  CB  SER A 140     331.396 335.541 305.618  1.00 75.47           C
ATOM    987  OG  SER A 140     332.751 335.270 305.923  1.00 75.47           O
ATOM    988  N   PRO A 141     332.424 332.898 303.653  1.00 75.47           N
ATOM    989  CA  PRO A 141     332.910 331.514 303.647  1.00 75.47           C
ATOM    990  C   PRO A 141     333.994 331.212 304.671  1.00 75.47           C
ATOM    991  O   PRO A 141     334.361 330.041 304.823  1.00 75.47           O
ATOM    992  CB  PRO A 141     333.445 331.360 302.219  1.00 75.47           C
ATOM    993  CG  PRO A 141     333.913 332.724 301.857  1.00 78.40           C
ATOM    994  CD  PRO A 141     333.012 333.701 302.569  1.00 78.40           C
ATOM    995  N   SER A 142     334.513 332.210 305.378  1.00 78.40           N
ATOM    996  CA  SER A 142     335.568 332.009 306.363  1.00 78.40           C
ATOM    997  C   SER A 142     334.961 331.920 307.755  1.00 78.40           C
ATOM    998  O   SER A 142     334.030 332.664 308.078  1.00 74.52           O
ATOM    999  CB  SER A 142     336.592 333.143 306.302  1.00 74.52           C
ATOM   1000  OG  SER A 142     335.961 334.404 306.431  1.00 74.52           O
ATOM   1001  N   VAL A 143     335.488 331.008 308.575  1.00 74.52           N
ATOM   1002  CA  VAL A 143     334.945 330.812 309.917  1.00 74.52           C
ATOM   1003  C   VAL A 143     335.379 331.929 310.858  1.00 74.52           C
ATOM   1004  O   VAL A 143     334.746 332.151 311.897  1.00 74.52           O
ATOM   1005  CB  VAL A 143     335.360 329.437 310.465  1.00 74.52           C
ATOM   1006  CG1 VAL A 143     334.701 328.341 309.668  1.00 73.41           C
ATOM   1007  CG2 VAL A 143     336.872 329.290 310.427  1.00 73.41           C
ATOM   1008  N   ALA A 144     336.459 332.640 310.525  1.00 73.41           N
ATOM   1009  CA  ALA A 144     336.914 333.727 311.384  1.00 73.41           C
ATOM   1010  C   ALA A 144     335.887 334.848 311.445  1.00 73.41           C
ATOM   1011  O   ALA A 144     335.676 335.452 312.503  1.00 73.41           O
ATOM   1012  CB  ALA A 144     338.261 334.258 310.892  1.00 73.41           C
ATOM   1013  N   ASN A 145     335.237 335.141 310.320  1.00 73.41           N
ATOM   1014  CA  ASN A 145     334.266 336.225 310.257  1.00 73.41           C
ATOM   1015  C   ASN A 145     332.942 335.733 309.690  1.00 73.41           C
ATOM   1016  O   ASN A 145     332.385 336.349 308.777  1.00 73.41           O
ATOM   1017  CB  ASN A 145     334.817 337.376 309.418  1.00 73.41           C
ATOM   1018  CG  ASN A 145     336.059 337.991 310.024  1.00 73.41           C
ATOM   1019  OD1 ASN A 145     336.037 338.477 311.155  1.00 73.41           O
ATOM   1020  ND2 ASN A 145     337.157 337.965 309.279  1.00 73.41           N
ATOM   1021  N   SER A 146     332.435 334.621 310.219  1.00 73.41           N
ATOM   1022  CA  SER A 146     331.221 333.987 309.709  1.00 73.41           C
ATOM   1023  C   SER A 146     330.036 334.915 309.938  1.00 73.41           C
ATOM   1024  O   SER A 146     329.545 335.069 311.055  1.00 73.41           O
ATOM   1025  CB  SER A 146     331.003 332.640 310.385  1.00 73.41           C
ATOM   1026  OG  SER A 146     329.808 332.035 309.933  1.00 73.41           O
ATOM   1027  N   TYR A 147     329.565 335.537 308.858  1.00 73.41           N
ATOM   1028  CA  TYR A 147     328.436 336.453 308.900  1.00 73.41           C
ATOM   1029  C   TYR A 147     327.413 336.080 307.840  1.00 73.41           C
ATOM   1030  O   TYR A 147     327.768 335.614 306.755  1.00 73.41           O
ATOM   1031  CB  TYR A 147     328.886 337.899 308.685  1.00 73.41           C
ATOM   1032  CG  TYR A 147     329.641 338.485 309.851  1.00 73.41           C
ATOM   1033  CD1 TYR A 147     329.046 338.604 311.098  1.00 73.41           C
ATOM   1034  CD2 TYR A 147     330.946 338.928 309.705  1.00 73.41           C
ATOM   1035  CE1 TYR A 147     329.731 339.139 312.167  1.00 73.41           C
ATOM   1036  CE2 TYR A 147     331.637 339.466 310.768  1.00 73.41           C
ATOM   1037  CZ  TYR A 147     331.026 339.571 311.995  1.00 73.41           C
ATOM   1038  OH  TYR A 147     331.715 340.108 313.055  1.00 73.41           O
ATOM   1039  N   LEU A 148     326.142 336.288 308.164  1.00 73.41           N
ATOM   1040  CA  LEU A 148     325.040 336.137 307.224  1.00 73.41           C
ATOM   1041  C   LEU A 148     324.264 337.443 307.177  1.00 73.41           C
ATOM   1042  O   LEU A 148     323.991 338.039 308.222  1.00 73.41           O
ATOM   1043  CB  LEU A 148     324.120 334.989 307.636  1.00 73.41           C
ATOM   1044  CG  LEU A 148     322.820 334.847 306.846  1.00 73.41           C
ATOM   1045  CD1 LEU A 148     323.107 334.521 305.396  1.00 73.41           C
ATOM   1046  CD2 LEU A 148     321.939 333.782 307.469  1.00 73.41           C
ATOM   1047  N   VAL A 149     323.916 337.893 305.974  1.00 73.41           N
ATOM   1048  CA  VAL A 149     323.136 339.110 305.797  1.00 73.41           C
ATOM   1049  C   VAL A 149     321.928 338.797 304.926  1.00 73.41           C
ATOM   1050  O   VAL A 149     322.046 338.121 303.900  1.00 73.41           O
ATOM   1051  CB  VAL A 149     323.978 340.255 305.194  1.00 73.41           C
ATOM   1052  CG1 VAL A 149     325.053 340.683 306.170  1.00 73.41           C
ATOM   1053  CG2 VAL A 149     324.610 339.833 303.894  1.00 73.41           C
ATOM   1054  N   TYR A 150     320.760 339.269 305.354  1.00 73.41           N
ATOM   1055  CA  TYR A 150     319.512 339.047 304.640  1.00 73.41           C
ATOM   1056  C   TYR A 150     318.691 340.324 304.689  1.00 73.41           C
ATOM   1057  O   TYR A 150     318.763 341.063 305.678  1.00 73.41           O
ATOM   1058  CB  TYR A 150     318.723 337.874 305.245  1.00 73.41           C
ATOM   1059  CG  TYR A 150     318.286 338.096 306.675  1.00 73.41           C
ATOM   1060  CD1 TYR A 150     317.007 338.543 306.970  1.00 73.41           C
ATOM   1061  CD2 TYR A 150     319.156 337.861 307.729  1.00 73.41           C
ATOM   1062  CE1 TYR A 150     316.606 338.747 308.275  1.00 73.41           C
ATOM   1063  CE2 TYR A 150     318.764 338.062 309.036  1.00 73.41           C
ATOM   1064  CZ  TYR A 150     317.488 338.506 309.304  1.00 73.41           C
ATOM   1065  OH  TYR A 150     317.095 338.708 310.605  1.00 73.41           O
ATOM   1066  N   PRO A 151     317.908 340.615 303.653  1.00 73.41           N
ATOM   1067  CA  PRO A 151     317.151 341.870 303.634  1.00 73.41           C
ATOM   1068  C   PRO A 151     315.933 341.817 304.543  1.00 73.41           C
ATOM   1069  O   PRO A 151     315.339 340.759 304.769  1.00 73.41           O
ATOM   1070  CB  PRO A 151     316.738 342.012 302.166  1.00 73.41           C
ATOM   1071  CG  PRO A 151     316.691 340.623 301.655  1.00 73.41           C
ATOM   1072  CD  PRO A 151     317.662 339.799 302.450  1.00 73.41           C
ATOM   1073  N   SER A 152     315.560 342.985 305.061  1.00 73.41           N
ATOM   1074  CA  SER A 152     314.405 343.125 305.949  1.00 73.41           C
ATOM   1075  C   SER A 152     313.527 344.266 305.461  1.00 73.41           C
ATOM   1076  O   SER A 152     313.665 345.413 305.903  1.00 73.41           O
ATOM   1077  CB  SER A 152     314.847 343.355 307.396  1.00 73.41           C
ATOM   1078  OG  SER A 152     313.749 343.241 308.282  1.00 73.41           O
ATOM   1079  N   PRO A 153     312.615 343.990 304.536  1.00 73.41           N
ATOM   1080  CA  PRO A 153     311.701 345.026 304.061  1.00 73.41           C
ATOM   1081  C   PRO A 153     310.547 345.214 305.029  1.00 73.41           C
ATOM   1082  O   PRO A 153     310.175 344.274 305.749  1.00 73.41           O
ATOM   1083  CB  PRO A 153     311.217 344.467 302.719  1.00 73.41           C
ATOM   1084  CG  PRO A 153     311.217 342.990 302.941  1.00 73.41           C
ATOM   1085  CD  PRO A 153     312.376 342.697 303.868  1.00 73.41           C
ATOM   1086  N   PRO A 154     309.958 346.405 305.086  1.00 73.41           N
ATOM   1087  CA  PRO A 154     308.764 346.597 305.914  1.00 73.41           C
ATOM   1088  C   PRO A 154     307.612 345.730 305.442  1.00 73.41           C
ATOM   1089  O   PRO A 154     307.421 345.495 304.247  1.00 73.41           O
ATOM   1090  CB  PRO A 154     308.439 348.086 305.742  1.00 73.41           C
ATOM   1091  CG  PRO A 154     309.167 348.525 304.536  1.00 73.41           C
ATOM   1092  CD  PRO A 154     310.346 347.624 304.358  1.00 73.41           C
ATOM   1093  N   LYS A 155     306.838 345.248 306.409  1.00 73.41           N
ATOM   1094  CA  LYS A 155     305.699 344.388 306.114  1.00 75.33           C
ATOM   1095  C   LYS A 155     304.459 345.242 305.882  1.00 75.33           C
ATOM   1096  O   LYS A 155     304.144 346.123 306.691  1.00 75.33           O
ATOM   1097  CB  LYS A 155     305.467 343.396 307.251  1.00 75.33           C
ATOM   1098  CG  LYS A 155     304.382 342.378 306.961  1.00 75.33           C
ATOM   1099  CD  LYS A 155     304.298 341.332 308.057  1.00 75.33           C
ATOM   1100  CE  LYS A 155     303.204 340.320 307.763  1.00 75.33           C
ATOM   1101  NZ  LYS A 155     303.117 339.270 308.813  1.00 75.33           N
ATOM   1102  N   VAL A 156     303.761 344.984 304.783  1.00 73.41           N
ATOM   1103  CA  VAL A 156     302.559 345.734 304.445  1.00 73.41           C
ATOM   1104  C   VAL A 156     301.318 344.884 304.692  1.00 73.41           C
ATOM   1105  O   VAL A 156     301.402 343.660 304.794  1.00 73.41           O
ATOM   1106  CB  VAL A 156     302.602 346.223 302.988  1.00 73.41           C
ATOM   1107  CG1 VAL A 156     302.619 345.039 302.034  1.00 73.41           C
ATOM   1108  CG2 VAL A 156     301.415 347.127 302.702  1.00 73.41           C
ATOM   1109  N   ILE A 222     312.280 356.102 302.625  1.00 73.41           N
ATOM   1110  CA  ILE A 222     311.156 355.180 302.531  1.00 73.41           C
ATOM   1111  C   ILE A 222     311.483 353.918 303.323  1.00 73.41           C
ATOM   1112  O   ILE A 222     310.593 353.259 303.862  1.00 73.41           O
ATOM   1113  CB  ILE A 222     310.825 354.855 301.065  1.00 73.41           C
ATOM   1114  CG1 ILE A 222     310.680 356.145 300.257  1.00 73.41           C
ATOM   1115  CG2 ILE A 222     309.535 354.059 300.973  1.00 73.41           C
ATOM   1116  CD1 ILE A 222     310.573 355.922 298.766  1.00 73.41           C
ATOM   1117  N   LYS A 223     312.776 353.600 303.382  1.00 73.41           N
ATOM   1118  CA  LYS A 223     313.296 352.491 304.187  1.00 73.41           C
ATOM   1119  C   LYS A 223     312.741 351.143 303.729  1.00 73.41           C
ATOM   1120  O   LYS A 223     312.125 350.409 304.503  1.00 73.41           O
ATOM   1121  CB  LYS A 223     313.030 352.709 305.681  1.00 73.41           C
ATOM   1122  CG  LYS A 223     313.561 354.026 306.278  1.00 73.41           C
ATOM   1123  CD  LYS A 223     314.876 354.544 305.672  1.00 73.41           C
ATOM   1124  CE  LYS A 223     316.032 353.543 305.753  1.00 73.41           C
ATOM   1125  NZ  LYS A 223     317.220 354.021 304.997  1.00 73.41           N
ATOM   1126  N   ASN A 224     312.966 350.815 302.459  1.00 73.41           N
ATOM   1127  CA  ASN A 224     312.699 349.483 301.933  1.00 73.41           C
ATOM   1128  C   ASN A 224     313.952 348.624 301.865  1.00 73.41           C
ATOM   1129  O   ASN A 224     313.871 347.398 301.984  1.00 73.41           O
ATOM   1130  CB  ASN A 224     312.063 349.575 300.539  1.00 73.41           C
ATOM   1131  CG  ASN A 224     310.609 349.984 300.587  1.00 73.41           C
ATOM   1132  OD1 ASN A 224     310.241 351.067 300.132  1.00 73.41           O
ATOM   1133  ND2 ASN A 224     309.767 349.115 301.132  1.00 73.41           N
ATOM   1134  N   GLY A 225     315.115 349.241 301.671  1.00 73.41           N
ATOM   1135  CA  GLY A 225     316.352 348.500 301.543  1.00 73.41           C
ATOM   1136  C   GLY A 225     317.102 348.307 302.844  1.00 73.41           C
ATOM   1137  O   GLY A 225     318.296 348.607 302.924  1.00 73.41           O
ATOM   1138  N   ASP A 226     316.416 347.821 303.872  1.00 73.41           N
ATOM   1139  CA  ASP A 226     317.078 347.483 305.120  1.00 73.41           C
ATOM   1140  C   ASP A 226     317.542 346.034 305.100  1.00 73.41           C
ATOM   1141  O   ASP A 226     316.803 345.125 304.712  1.00 73.41           O
ATOM   1142  CB  ASP A 226     316.149 347.716 306.312  1.00 73.41           C
ATOM   1143  CG  ASP A 226     316.069 349.175 306.708  1.00 73.41           C
ATOM   1144  OD1 ASP A 226     316.867 349.978 306.178  1.00 73.41           O
ATOM   1145  OD2 ASP A 226     315.214 349.517 307.552  1.00 73.41           O
ATOM   1146  N   VAL A 227     318.784 345.824 305.529  1.00 73.41           N
ATOM   1147  CA  VAL A 227     319.402 344.506 305.538  1.00 73.41           C
ATOM   1148  C   VAL A 227     319.905 344.220 306.947  1.00 73.41           C
ATOM   1149  O   VAL A 227     320.371 345.125 307.647  1.00 73.41           O
ATOM   1150  CB  VAL A 227     320.542 344.410 304.498  1.00 73.41           C
ATOM   1151  CG1 VAL A 227     321.725 345.277 304.895  1.00 73.41           C
ATOM   1152  CG2 VAL A 227     320.962 342.968 304.293  1.00 73.41           C
ATOM   1153  N   ILE A 228     319.779 342.966 307.373  1.00 73.41           N
ATOM   1154  CA  ILE A 228     320.111 342.546 308.729  1.00 73.41           C
ATOM   1155  C   ILE A 228     321.444 341.817 308.705  1.00 73.41           C
ATOM   1156  O   ILE A 228     321.592 340.807 308.008  1.00 73.41           O
ATOM   1157  CB  ILE A 228     319.015 341.640 309.316  1.00 73.41           C
ATOM   1158  CG1 ILE A 228     317.698 342.399 309.457  1.00 73.41           C
ATOM   1159  CG2 ILE A 228     319.447 341.092 310.659  1.00 73.41           C
ATOM   1160  CD1 ILE A 228     317.837 343.731 310.136  1.00 73.41           C
ATOM   1161  N   VAL A 229     322.407 342.310 309.476  1.00 73.41           N
ATOM   1162  CA  VAL A 229     323.715 341.679 309.609  1.00 73.41           C
ATOM   1163  C   VAL A 229     323.639 340.694 310.767  1.00 73.41           C
ATOM   1164  O   VAL A 229     323.512 341.099 311.928  1.00 73.41           O
ATOM   1165  CB  VAL A 229     324.827 342.713 309.838  1.00 73.41           C
ATOM   1166  CG1 VAL A 229     326.152 342.015 310.052  1.00 73.41           C
ATOM   1167  CG2 VAL A 229     324.909 343.670 308.665  1.00 73.41           C
ATOM   1168  N   PHE A 230     323.722 339.407 310.454  1.00 73.41           N
ATOM   1169  CA  PHE A 230     323.538 338.338 311.426  1.00 73.41           C
ATOM   1170  C   PHE A 230     324.843 337.577 311.587  1.00 73.41           C
ATOM   1171  O   PHE A 230     325.468 337.195 310.593  1.00 73.41           O
ATOM   1172  CB  PHE A 230     322.418 337.395 310.983  1.00 73.41           C
ATOM   1173  CG  PHE A 230     322.009 336.401 312.025  1.00 73.41           C
ATOM   1174  CD1 PHE A 230     321.010 336.701 312.929  1.00 73.41           C
ATOM   1175  CD2 PHE A 230     322.611 335.159 312.089  1.00 73.41           C
ATOM   1176  CE1 PHE A 230     320.626 335.786 313.889  1.00 73.41           C
ATOM   1177  CE2 PHE A 230     322.230 334.239 313.042  1.00 73.41           C
ATOM   1178  CZ  PHE A 230     321.235 334.551 313.944  1.00 73.41           C
ATOM   1179  N   ASN A 231     325.253 337.359 312.833  1.00 73.41           N
ATOM   1180  CA  ASN A 231     326.479 336.631 313.129  1.00 73.41           C
ATOM   1181  C   ASN A 231     326.139 335.164 313.362  1.00 73.41           C
ATOM   1182  O   ASN A 231     325.331 334.840 314.237  1.00 73.41           O
ATOM   1183  CB  ASN A 231     327.183 337.223 314.345  1.00 73.41           C
ATOM   1184  CG  ASN A 231     328.604 336.721 314.498  1.00 73.41           C
ATOM   1185  OD1 ASN A 231     328.940 335.622 314.059  1.00 73.41           O
ATOM   1186  ND2 ASN A 231     329.450 337.527 315.125  1.00 73.41           N
ATOM   1187  N   LEU A 232     326.753 334.283 312.573  1.00 73.41           N
ATOM   1188  CA  LEU A 232     326.457 332.858 312.675  1.00 73.41           C
ATOM   1189  C   LEU A 232     327.144 332.215 313.870  1.00 73.41           C
ATOM   1190  O   LEU A 232     326.637 331.230 314.417  1.00 73.41           O
ATOM   1191  CB  LEU A 232     326.873 332.151 311.387  1.00 73.41           C
ATOM   1192  CG  LEU A 232     326.005 332.473 310.175  1.00 73.41           C
ATOM   1193  CD1 LEU A 232     326.822 332.359 308.915  1.00 73.41           C
ATOM   1194  CD2 LEU A 232     324.813 331.545 310.123  1.00 73.41           C
ATOM   1195  N   GLU A 233     328.298 332.744 314.278  1.00 73.41           N
ATOM   1196  CA  GLU A 233     329.057 332.131 315.363  1.00 73.41           C
ATOM   1197  C   GLU A 233     328.277 332.158 316.671  1.00 73.41           C
ATOM   1198  O   GLU A 233     328.107 331.124 317.326  1.00 73.41           O
ATOM   1199  CB  GLU A 233     330.398 332.844 315.521  1.00 73.41           C
ATOM   1200  CG  GLU A 233     331.379 332.135 316.426  1.00 73.41           C
ATOM   1201  CD  GLU A 233     332.678 332.897 316.567  1.00 73.41           C
ATOM   1202  OE1 GLU A 233     332.783 334.003 315.997  1.00 73.41           O
ATOM   1203  OE2 GLU A 233     333.596 332.392 317.246  1.00 73.41           O
ATOM   1204  N   THR A 234     327.790 333.332 317.064  1.00 73.41           N
ATOM   1205  CA  THR A 234     327.047 333.493 318.305  1.00 73.41           C
ATOM   1206  C   THR A 234     325.538 333.426 318.113  1.00 73.41           C
ATOM   1207  O   THR A 234     324.804 333.443 319.105  1.00 73.41           O
ATOM   1208  CB  THR A 234     327.416 334.821 318.968  1.00 73.41           C
ATOM   1209  OG1 THR A 234     326.968 335.905 318.146  1.00 73.41           O
ATOM   1210  CG2 THR A 234     328.919 334.923 319.135  1.00 73.41           C
ATOM   1211  N   LEU A 235     325.066 333.345 316.871  1.00 73.41           N
ATOM   1212  CA  LEU A 235     323.641 333.290 316.543  1.00 73.41           C
ATOM   1213  C   LEU A 235     322.884 334.479 317.144  1.00 73.41           C
ATOM   1214  O   LEU A 235     321.840 334.340 317.779  1.00 73.41           O
ATOM   1215  CB  LEU A 235     323.025 331.964 316.995  1.00 73.41           C
ATOM   1216  CG  LEU A 235     323.522 330.712 316.274  1.00 73.41           C
ATOM   1217  CD1 LEU A 235     322.869 329.470 316.850  1.00 73.41           C
ATOM   1218  CD2 LEU A 235     323.254 330.817 314.789  1.00 73.41           C
ATOM   1219  N   GLN A 236     323.448 335.668 316.916  1.00 73.41           N
ATOM   1220  CA  GLN A 236     322.776 336.895 317.320  1.00 73.41           C
ATOM   1221  C   GLN A 236     323.060 337.964 316.282  1.00 73.41           C
ATOM   1222  O   GLN A 236     324.192 338.038 315.784  1.00 73.41           O
ATOM   1223  CB  GLN A 236     323.227 337.361 318.711  1.00 73.41           C
ATOM   1224  CG  GLN A 236     324.532 338.141 318.746  1.00 73.41           C
ATOM   1225  CD  GLN A 236     324.935 338.546 320.146  1.00 73.41           C
ATOM   1226  OE1 GLN A 236     325.530 337.764 320.883  1.00 73.41           O
ATOM   1227  NE2 GLN A 236     324.614 339.777 320.520  1.00 73.41           N
ATOM   1228  N   PRO A 237     322.071 338.770 315.900  1.00 73.41           N
ATOM   1229  CA  PRO A 237     322.335 339.861 314.956  1.00 73.41           C
ATOM   1230  C   PRO A 237     323.265 340.900 315.559  1.00 73.41           C
ATOM   1231  O   PRO A 237     323.276 341.125 316.772  1.00 73.41           O
ATOM   1232  CB  PRO A 237     320.944 340.445 314.685  1.00 73.41           C
ATOM   1233  CG  PRO A 237     320.104 339.996 315.822  1.00 73.41           C
ATOM   1234  CD  PRO A 237     320.650 338.677 316.262  1.00 73.41           C
ATOM   1235  N   THR A 238     324.056 341.530 314.699  1.00 73.41           N
ATOM   1236  CA  THR A 238     325.022 342.528 315.138  1.00 73.41           C
ATOM   1237  C   THR A 238     324.558 343.957 314.908  1.00 73.41           C
ATOM   1238  O   THR A 238     324.649 344.778 315.826  1.00 73.41           O
ATOM   1239  CB  THR A 238     326.363 342.319 314.428  1.00 73.41           C
ATOM   1240  OG1 THR A 238     326.863 341.012 314.728  1.00 73.41           O
ATOM   1241  CG2 THR A 238     327.377 343.354 314.888  1.00 73.41           C
ATOM   1242  N   MET A 239     324.058 344.278 313.718  1.00 73.41           N
ATOM   1243  CA  MET A 239     323.613 345.634 313.425  1.00 73.41           C
ATOM   1244  C   MET A 239     322.665 345.588 312.235  1.00 73.41           C
ATOM   1245  O   MET A 239     322.566 344.578 311.534  1.00 73.41           O
ATOM   1246  CB  MET A 239     324.794 346.563 313.147  1.00 73.41           C
ATOM   1247  CG  MET A 239     325.629 346.162 311.953  1.00 73.41           C
ATOM   1248  SD  MET A 239     327.376 346.496 312.224  1.00 73.41           S
ATOM   1249  CE  MET A 239     327.383 348.281 312.144  1.00 73.41           C
ATOM   1250  N   VAL A 240     321.967 346.701 312.023  1.00 76.84           N
ATOM   1251  CA  VAL A 240     321.039 346.866 310.912  1.00 76.84           C
ATOM   1252  C   VAL A 240     321.583 347.964 310.013  1.00 76.84           C
ATOM   1253  O   VAL A 240     321.881 349.068 310.487  1.00 76.84           O
ATOM   1254  CB  VAL A 240     319.622 347.206 311.392  1.00 76.84           C
ATOM   1255  CG1 VAL A 240     318.669 347.261 310.218  1.00 76.84           C
ATOM   1256  CG2 VAL A 240     319.154 346.184 312.407  1.00 76.84           C
ATOM   1257  N   ILE A 241     321.717 347.664 308.725  1.00 76.84           N
ATOM   1258  CA  ILE A 241     322.300 348.584 307.756  1.00 76.84           C
ATOM   1259  C   ILE A 241     321.191 349.134 306.875  1.00 73.41           C
ATOM   1260  O   ILE A 241     320.451 348.371 306.241  1.00 73.41           O
ATOM   1261  CB  ILE A 241     323.381 347.894 306.909  1.00 73.41           C
ATOM   1262  CG1 ILE A 241     324.499 347.350 307.795  1.00 73.41           C
ATOM   1263  CG2 ILE A 241     323.963 348.869 305.908  1.00 73.41           C
ATOM   1264  CD1 ILE A 241     325.217 348.418 308.586  1.00 73.41           C
ATOM   1265  N   GLU A 242     321.071 350.461 306.839  1.00 73.41           N
ATOM   1266  CA  GLU A 242     320.146 351.140 305.935  1.00 73.41           C
ATOM   1267  C   GLU A 242     320.836 351.256 304.580  1.00 73.41           C
ATOM   1268  O   GLU A 242     321.432 352.277 304.227  1.00 73.41           O
ATOM   1269  CB  GLU A 242     319.748 352.499 306.490  1.00 73.41           C
ATOM   1270  CG  GLU A 242     318.994 352.428 307.807  1.00 73.41           C
ATOM   1271  CD  GLU A 242     318.677 353.797 308.374  1.00 73.41           C
ATOM   1272  OE1 GLU A 242     319.024 354.805 307.726  1.00 73.41           O
ATOM   1273  OE2 GLU A 242     318.077 353.861 309.469  1.00 73.41           O
ATOM   1274  N   ALA A 243     320.747 350.172 303.808  1.00 73.41           N
ATOM   1275  CA  ALA A 243     321.537 350.071 302.586  1.00 73.41           C
ATOM   1276  C   ALA A 243     321.028 351.019 301.508  1.00 73.41           C
ATOM   1277  O   ALA A 243     321.748 351.927 301.078  1.00 73.41           O
ATOM   1278  CB  ALA A 243     321.529 348.628 302.082  1.00 73.41           C
ATOM   1279  N   HIS A 244     319.787 350.833 301.066  1.00 73.41           N
ATOM   1280  CA  HIS A 244     319.268 351.574 299.927  1.00 73.41           C
ATOM   1281  C   HIS A 244     317.837 352.010 300.194  1.00 73.41           C
ATOM   1282  O   HIS A 244     317.181 351.538 301.126  1.00 73.41           O
ATOM   1283  CB  HIS A 244     319.323 350.743 298.638  1.00 73.41           C
ATOM   1284  CG  HIS A 244     320.691 350.643 298.041  1.00 73.41           C
ATOM   1285  ND1 HIS A 244     321.462 349.505 298.136  1.00 73.41           N
ATOM   1286  CD2 HIS A 244     321.432 351.543 297.353  1.00 73.41           C
ATOM   1287  CE1 HIS A 244     322.616 349.705 297.524  1.00 73.41           C
ATOM   1288  NE2 HIS A 244     322.622 350.935 297.041  1.00 73.41           N
ATOM   1289  N   LYS A 245     317.363 352.935 299.357  1.00 73.41           N
ATOM   1290  CA  LYS A 245     315.978 353.377 299.410  1.00 73.41           C
ATOM   1291  C   LYS A 245     315.013 352.359 298.820  1.00 73.41           C
ATOM   1292  O   LYS A 245     313.906 352.197 299.344  1.00 73.41           O
ATOM   1293  CB  LYS A 245     315.829 354.711 298.677  1.00 73.41           C
ATOM   1294  CG  LYS A 245     314.469 355.359 298.827  1.00 73.41           C
ATOM   1295  CD  LYS A 245     314.361 356.609 297.973  1.00 73.41           C
ATOM   1296  CE  LYS A 245     315.278 357.705 298.489  1.00 73.41           C
ATOM   1297  NZ  LYS A 245     315.507 358.757 297.463  1.00 73.41           N
ATOM   1298  N   GLY A 246     315.408 351.663 297.749  1.00 73.41           N
ATOM   1299  CA  GLY A 246     314.580 350.654 297.132  1.00 73.41           C
ATOM   1300  C   GLY A 246     314.981 349.260 297.588  1.00 73.41           C
ATOM   1301  O   GLY A 246     315.945 349.076 298.338  1.00 73.41           O
ATOM   1302  N   GLU A 247     314.224 348.272 297.115  1.00 73.41           N
ATOM   1303  CA  GLU A 247     314.454 346.892 297.510  1.00 73.41           C
ATOM   1304  C   GLU A 247     315.833 346.419 297.069  1.00 73.41           C
ATOM   1305  O   GLU A 247     316.344 346.827 296.025  1.00 73.41           O
ATOM   1306  CB  GLU A 247     313.377 345.982 296.915  1.00 73.41           C
ATOM   1307  CG  GLU A 247     312.055 346.012 297.657  1.00 73.41           C
ATOM   1308  CD  GLU A 247     311.103 344.928 297.192  1.00 73.41           C
ATOM   1309  OE1 GLU A 247     311.479 344.150 296.292  1.00 73.41           O
ATOM   1310  OE2 GLU A 247     309.977 344.853 297.725  1.00 73.41           O
ATOM   1311  N   ILE A 248     316.438 345.559 297.888  1.00 73.41           N
ATOM   1312  CA  ILE A 248     317.736 344.984 297.559  1.00 73.41           C
ATOM   1313  C   ILE A 248     317.566 343.917 296.487  1.00 73.41           C
ATOM   1314  O   ILE A 248     316.623 343.115 296.527  1.00 73.41           O
ATOM   1315  CB  ILE A 248     318.398 344.412 298.824  1.00 73.41           C
ATOM   1316  CG1 ILE A 248     318.625 345.523 299.847  1.00 73.41           C
ATOM   1317  CG2 ILE A 248     319.710 343.739 298.490  1.00 73.41           C
ATOM   1318  CD1 ILE A 248     319.508 346.637 299.342  1.00 73.41           C
ATOM   1319  N   ALA A 249     318.474 343.910 295.509  1.00 73.41           N
ATOM   1320  CA  ALA A 249     318.410 342.960 294.407  1.00 73.41           C
ATOM   1321  C   ALA A 249     319.543 341.945 294.388  1.00 73.41           C
ATOM   1322  O   ALA A 249     319.366 340.867 293.811  1.00 73.41           O
ATOM   1323  CB  ALA A 249     318.392 343.704 293.063  1.00 73.41           C
ATOM   1324  N   ALA A 250     320.690 342.250 294.990  1.00 73.41           N
ATOM   1325  CA  ALA A 250     321.803 341.311 295.003  1.00 73.41           C
ATOM   1326  C   ALA A 250     322.743 341.653 296.150  1.00 73.41           C
ATOM   1327  O   ALA A 250     322.961 342.827 296.461  1.00 73.41           O
ATOM   1328  CB  ALA A 250     322.559 341.320 293.672  1.00 73.41           C
ATOM   1329  N   MET A 251     323.305 340.614 296.766  1.00 73.41           N
ATOM   1330  CA  MET A 251     324.207 340.775 297.896  1.00 73.41           C
ATOM   1331  C   MET A 251     325.407 339.855 297.723  1.00 73.41           C
ATOM   1332  O   MET A 251     325.282 338.744 297.204  1.00 73.41           O
ATOM   1333  CB  MET A 251     323.504 340.471 299.223  1.00 73.41           C
ATOM   1334  CG  MET A 251     322.837 341.671 299.862  1.00 73.41           C
ATOM   1335  SD  MET A 251     321.694 341.198 301.166  1.00 75.35           S
ATOM   1336  CE  MET A 251     320.713 339.976 300.304  1.00 75.35           C
ATOM   1337  N   ALA A 252     326.569 340.326 298.174  1.00 75.35           N
ATOM   1338  CA  ALA A 252     327.800 339.557 298.078  1.00 75.35           C
ATOM   1339  C   ALA A 252     328.733 339.924 299.223  1.00 75.35           C
ATOM   1340  O   ALA A 252     328.883 341.100 299.566  1.00 75.35           O
ATOM   1341  CB  ALA A 252     328.507 339.795 296.742  1.00 82.15           C
ATOM   1342  N   ILE A 253     329.371 338.905 299.798  1.00 82.15           N
ATOM   1343  CA  ILE A 253     330.326 339.066 300.889  1.00 82.15           C
ATOM   1344  C   ILE A 253     331.680 338.541 300.435  1.00 82.15           C
ATOM   1345  O   ILE A 253     331.763 337.472 299.820  1.00 82.15           O
ATOM   1346  CB  ILE A 253     329.872 338.336 302.169  1.00 82.15           C
ATOM   1347  CG1 ILE A 253     328.618 338.980 302.742  1.00 82.15           C
ATOM   1348  CG2 ILE A 253     330.968 338.343 303.214  1.00 82.15           C
ATOM   1349  CD1 ILE A 253     328.047 338.227 303.906  1.00 82.15           C
ATOM   1350  N   SER A 254     332.736 339.289 300.745  1.00 82.15           N
ATOM   1351  CA  SER A 254     334.080 338.889 300.362  1.00 82.15           C
ATOM   1352  C   SER A 254     334.463 337.570 301.026  1.00 79.38           C
ATOM   1353  O   SER A 254     333.762 337.048 301.897  1.00 79.38           O
ATOM   1354  CB  SER A 254     335.092 339.971 300.732  1.00 79.38           C
ATOM   1355  OG  SER A 254     334.843 340.468 302.032  1.00 79.38           O
ATOM   1356  N   PHE A 255     335.606 337.030 300.595  1.00 79.38           N
ATOM   1357  CA  PHE A 255     336.040 335.721 301.072  1.00 79.38           C
ATOM   1358  C   PHE A 255     336.312 335.726 302.571  1.00 79.38           C
ATOM   1359  O   PHE A 255     335.973 334.762 303.265  1.00 79.38           O
ATOM   1360  CB  PHE A 255     337.285 335.279 300.302  1.00 76.22           C
ATOM   1361  CG  PHE A 255     337.632 333.832 300.490  1.00 76.22           C
ATOM   1362  CD1 PHE A 255     336.896 332.844 299.862  1.00 76.22           C
ATOM   1363  CD2 PHE A 255     338.700 333.461 301.286  1.00 76.22           C
ATOM   1364  CE1 PHE A 255     337.213 331.513 300.030  1.00 77.24           C
ATOM   1365  CE2 PHE A 255     339.019 332.130 301.457  1.00 77.24           C
ATOM   1366  CZ  PHE A 255     338.274 331.156 300.827  1.00 77.24           C
ATOM   1367  N   ASP A 256     336.913 336.794 303.087  1.00 77.24           N
ATOM   1368  CA  ASP A 256     337.268 336.882 304.495  1.00 77.24           C
ATOM   1369  C   ASP A 256     336.127 337.388 305.369  1.00 77.24           C
ATOM   1370  O   ASP A 256     336.295 337.483 306.588  1.00 77.24           O
ATOM   1371  CB  ASP A 256     338.488 337.788 304.672  1.00 73.81           C
ATOM   1372  CG  ASP A 256     338.308 339.139 304.016  1.00 73.81           C
ATOM   1373  OD1 ASP A 256     337.327 339.311 303.263  1.00 73.81           O
ATOM   1374  OD2 ASP A 256     339.145 340.034 304.253  1.00 73.81           O
ATOM   1375  N   GLY A 257     334.978 337.714 304.782  1.00 73.81           N
ATOM   1376  CA  GLY A 257     333.865 338.224 305.557  1.00 73.81           C
ATOM   1377  C   GLY A 257     334.018 339.649 306.034  1.00 73.81           C
ATOM   1378  O   GLY A 257     333.361 340.042 306.999  1.00 73.81           O
ATOM   1379  N   THR A 258     334.870 340.438 305.385  1.00 73.41           N
ATOM   1380  CA  THR A 258     335.133 341.808 305.805  1.00 73.41           C
ATOM   1381  C   THR A 258     334.126 342.796 305.222  1.00 73.41           C
ATOM   1382  O   THR A 258     333.651 343.690 305.928  1.00 73.41           O
ATOM   1383  CB  THR A 258     336.555 342.213 305.406  1.00 73.41           C
ATOM   1384  OG1 THR A 258     337.499 341.415 306.128  1.00 73.41           O
ATOM   1385  CG2 THR A 258     336.806 343.680 305.707  1.00 73.41           C
ATOM   1386  N   LEU A 259     333.788 342.649 303.944  1.00 73.41           N
ATOM   1387  CA  LEU A 259     332.956 343.608 303.237  1.00 73.41           C
ATOM   1388  C   LEU A 259     331.663 342.954 302.767  1.00 73.41           C
ATOM   1389  O   LEU A 259     331.569 341.731 302.638  1.00 73.41           O
ATOM   1390  CB  LEU A 259     333.703 344.200 302.038  1.00 73.41           C
ATOM   1391  CG  LEU A 259     335.112 344.694 302.353  1.00 73.41           C
ATOM   1392  CD1 LEU A 259     335.902 344.921 301.083  1.00 73.41           C
ATOM   1393  CD2 LEU A 259     335.046 345.965 303.173  1.00 73.41           C
ATOM   1394  N   MET A 260     330.659 343.792 302.525  1.00 73.41           N
ATOM   1395  CA  MET A 260     329.394 343.372 301.943  1.00 73.41           C
ATOM   1396  C   MET A 260     328.984 344.375 300.877  1.00 73.41           C
ATOM   1397  O   MET A 260     329.081 345.586 301.090  1.00 73.41           O
ATOM   1398  CB  MET A 260     328.293 343.265 303.001  1.00 73.41           C
ATOM   1399  CG  MET A 260     326.904 343.034 302.428  1.00 73.41           C
ATOM   1400  SD  MET A 260     325.588 343.458 303.581  1.00 73.41           S
ATOM   1401  CE  MET A 260     326.170 345.039 304.185  1.00 73.41           C
ATOM   1402  N   ALA A 261     328.521 343.872 299.735  1.00 73.41           N
ATOM   1403  CA  ALA A 261     328.089 344.710 298.626  1.00 73.41           C
ATOM   1404  C   ALA A 261     326.618 344.452 298.329  1.00 73.41           C
ATOM   1405  O   ALA A 261     326.164 343.306 298.352  1.00 73.41           O
ATOM   1406  CB  ALA A 261     328.926 344.450 297.376  1.00 73.41           C
ATOM   1407  N   THR A 262     325.881 345.526 298.040  1.00 73.41           N
ATOM   1408  CA  THR A 262     324.442 345.452 297.842  1.00 73.41           C
ATOM   1409  C   THR A 262     324.065 346.167 296.554  1.00 73.41           C
ATOM   1410  O   THR A 262     324.766 347.068 296.090  1.00 73.41           O
ATOM   1411  CB  THR A 262     323.673 346.072 299.018  1.00 73.41           C
ATOM   1412  OG1 THR A 262     324.294 347.304 299.401  1.00 73.41           O
ATOM   1413  CG2 THR A 262     323.653 345.131 300.204  1.00 73.41           C
ATOM   1414  N   ALA A 263     322.934 345.756 295.984  1.00 73.41           N
ATOM   1415  CA  ALA A 263     322.444 346.342 294.746  1.00 73.41           C
ATOM   1416  C   ALA A 263     320.936 346.505 294.825  1.00 73.41           C
ATOM   1417  O   ALA A 263     320.212 345.560 295.147  1.00 73.41           O
ATOM   1418  CB  ALA A 263     322.813 345.484 293.534  1.00 73.41           C
ATOM   1419  N   SER A 264     320.470 347.710 294.512  1.00 73.41           N
ATOM   1420  CA  SER A 264     319.051 348.023 294.503  1.00 73.41           C
ATOM   1421  C   SER A 264     318.394 347.389 293.280  1.00 73.41           C
ATOM   1422  O   SER A 264     319.060 346.876 292.380  1.00 73.41           O
ATOM   1423  CB  SER A 264     318.839 349.537 294.524  1.00 73.41           C
ATOM   1424  OG  SER A 264     319.187 350.115 293.279  1.00 73.41           O
ATOM   1425  N   ASP A 265     317.061 347.423 293.251  1.00 73.41           N
ATOM   1426  CA  ASP A 265     316.331 346.820 292.142  1.00 73.41           C
ATOM   1427  C   ASP A 265     316.517 347.568 290.832  1.00 73.41           C
ATOM   1428  O   ASP A 265     316.258 346.991 289.769  1.00 73.41           O
ATOM   1429  CB  ASP A 265     314.842 346.729 292.480  1.00 73.41           C
ATOM   1430  CG  ASP A 265     314.234 348.077 292.800  1.00 73.41           C
ATOM   1431  OD1 ASP A 265     314.997 349.054 292.953  1.00 73.41           O
ATOM   1432  OD2 ASP A 265     312.993 348.158 292.908  1.00 73.41           O
ATOM   1433  N   LYS A 266     316.959 348.822 290.873  1.00 73.41           N
ATOM   1434  CA  LYS A 266     317.232 349.584 289.664  1.00 73.41           C
ATOM   1435  C   LYS A 266     318.637 349.362 289.125  1.00 73.41           C
ATOM   1436  O   LYS A 266     318.903 349.708 287.972  1.00 73.41           O
ATOM   1437  CB  LYS A 266     317.013 351.076 289.923  1.00 73.41           C
ATOM   1438  CG  LYS A 266     315.555 351.455 290.033  1.00 73.41           C
ATOM   1439  CD  LYS A 266     315.389 352.859 290.565  1.00 73.41           C
ATOM   1440  CE  LYS A 266     313.937 353.295 290.500  1.00 73.41           C
ATOM   1441  NZ  LYS A 266     313.032 352.283 291.111  1.00 73.41           N
ATOM   1442  N   GLY A 267     319.537 348.800 289.927  1.00 73.41           N
ATOM   1443  CA  GLY A 267     320.862 348.450 289.461  1.00 73.41           C
ATOM   1444  C   GLY A 267     321.719 349.619 289.027  1.00 73.41           C
ATOM   1445  O   GLY A 267     322.730 349.426 288.348  1.00 73.41           O
ATOM   1446  N   THR A 268     321.340 350.835 289.409  1.00 73.41           N
ATOM   1447  CA  THR A 268     322.119 352.003 289.025  1.00 73.41           C
ATOM   1448  C   THR A 268     323.248 352.309 289.995  1.00 73.41           C
ATOM   1449  O   THR A 268     324.305 352.789 289.569  1.00 73.41           O
ATOM   1450  CB  THR A 268     321.217 353.231 288.905  1.00 73.41           C
ATOM   1451  OG1 THR A 268     320.412 353.349 290.084  1.00 73.41           O
ATOM   1452  CG2 THR A 268     320.317 353.104 287.699  1.00 73.41           C
ATOM   1453  N   ILE A 269     323.057 352.042 291.282  1.00 73.41           N
ATOM   1454  CA  ILE A 269     324.057 352.333 292.302  1.00 73.41           C
ATOM   1455  C   ILE A 269     324.451 351.033 292.982  1.00 73.41           C
ATOM   1456  O   ILE A 269     323.586 350.283 293.450  1.00 73.41           O
ATOM   1457  CB  ILE A 269     323.544 353.348 293.339  1.00 73.41           C
ATOM   1458  CG1 ILE A 269     323.507 354.752 292.746  1.00 73.41           C
ATOM   1459  CG2 ILE A 269     324.420 353.333 294.573  1.00 73.41           C
ATOM   1460  CD1 ILE A 269     322.623 355.696 293.514  1.00 73.41           C
ATOM   1461  N   ILE A 270     325.753 350.769 293.032  1.00 73.41           N
ATOM   1462  CA  ILE A 270     326.313 349.637 293.760  1.00 73.41           C
ATOM   1463  C   ILE A 270     326.986 350.177 295.010  1.00 73.41           C
ATOM   1464  O   ILE A 270     327.852 351.057 294.926  1.00 73.41           O
ATOM   1465  CB  ILE A 270     327.307 348.843 292.898  1.00 73.56           C
ATOM   1466  CG1 ILE A 270     326.593 348.229 291.697  1.00 73.56           C
ATOM   1467  CG2 ILE A 270     327.980 347.766 293.725  1.00 73.56           C
ATOM   1468  CD1 ILE A 270     325.822 346.982 292.027  1.00 73.56           C
ATOM   1469  N   ARG A 271     326.583 349.663 296.165  1.00 73.56           N
ATOM   1470  CA  ARG A 271     327.106 350.108 297.444  1.00 73.56           C
ATOM   1471  C   ARG A 271     327.744 348.940 298.176  1.00 73.56           C
ATOM   1472  O   ARG A 271     327.181 347.844 298.241  1.00 77.36           O
ATOM   1473  CB  ARG A 271     326.002 350.728 298.307  1.00 77.36           C
ATOM   1474  CG  ARG A 271     325.632 352.144 297.914  1.00 77.36           C
ATOM   1475  CD  ARG A 271     324.654 352.752 298.904  1.00 77.36           C
ATOM   1476  NE  ARG A 271     324.191 354.065 298.474  1.00 77.36           N
ATOM   1477  CZ  ARG A 271     323.013 354.583 298.787  1.00 77.36           C
ATOM   1478  NH1 ARG A 271     322.138 353.918 299.524  1.00 77.36           N
ATOM   1479  NH2 ARG A 271     322.703 355.801 298.349  1.00 77.36           N
ATOM   1480  N   VAL A 272     328.930 349.184 298.722  1.00 77.36           N
ATOM   1481  CA  VAL A 272     329.660 348.193 299.500  1.00 77.36           C
ATOM   1482  C   VAL A 272     329.850 348.753 300.904  1.00 77.36           C
ATOM   1483  O   VAL A 272     330.226 349.920 301.067  1.00 82.18           O
ATOM   1484  CB  VAL A 272     331.009 347.825 298.851  1.00 82.18           C
ATOM   1485  CG1 VAL A 272     331.943 349.024 298.786  1.00 82.18           C
ATOM   1486  CG2 VAL A 272     331.666 346.672 299.588  1.00 82.18           C
ATOM   1487  N   PHE A 273     329.543 347.943 301.912  1.00 82.18           N
ATOM   1488  CA  PHE A 273     329.670 348.354 303.298  1.00 82.18           C
ATOM   1489  C   PHE A 273     330.699 347.489 304.011  1.00 82.18           C
ATOM   1490  O   PHE A 273     330.878 346.311 303.698  1.00 82.18           O
ATOM   1491  CB  PHE A 273     328.334 348.267 304.048  1.00 78.36           C
ATOM   1492  CG  PHE A 273     327.203 348.978 303.369  1.00 78.36           C
ATOM   1493  CD1 PHE A 273     326.328 348.292 302.548  1.00 78.36           C
ATOM   1494  CD2 PHE A 273     327.015 350.337 303.554  1.00 78.36           C
ATOM   1495  CE1 PHE A 273     325.286 348.947 301.927  1.00 78.36           C
ATOM   1496  CE2 PHE A 273     325.978 350.996 302.933  1.00 78.36           C
ATOM   1497  CZ  PHE A 273     325.112 350.301 302.119  1.00 78.36           C
ATOM   1498  N   ASP A 274     331.374 348.099 304.981  1.00 78.36           N
ATOM   1499  CA  ASP A 274     332.293 347.391 305.859  1.00 83.51           C
ATOM   1500  C   ASP A 274     331.486 346.799 307.008  1.00 83.51           C
ATOM   1501  O   ASP A 274     330.876 347.539 307.787  1.00 83.51           O
ATOM   1502  CB  ASP A 274     333.377 348.340 306.366  1.00 83.51           C
ATOM   1503  CG  ASP A 274     334.455 347.631 307.152  1.00 83.51           C
ATOM   1504  OD1 ASP A 274     335.265 346.910 306.537  1.00 83.51           O
ATOM   1505  OD2 ASP A 274     334.498 347.808 308.384  1.00 83.51           O
ATOM   1506  N   ILE A 275     331.476 345.468 307.110  1.00 83.51           N
ATOM   1507  CA  ILE A 275     330.467 344.793 307.924  1.00 83.51           C
ATOM   1508  C   ILE A 275     330.704 345.032 309.413  1.00 86.46           C
ATOM   1509  O   ILE A 275     329.752 345.083 310.200  1.00 86.46           O
ATOM   1510  CB  ILE A 275     330.422 343.292 307.582  1.00 86.46           C
ATOM   1511  CG1 ILE A 275     329.121 342.672 308.083  1.00 86.46           C
ATOM   1512  CG2 ILE A 275     331.603 342.558 308.180  1.00 86.46           C
ATOM   1513  CD1 ILE A 275     328.812 341.342 307.447  1.00 86.46           C
ATOM   1514  N   GLU A 276     331.964 345.170 309.830  1.00 86.46           N
ATOM   1515  CA  GLU A 276     332.240 345.316 311.257  1.00 88.10           C
ATOM   1516  C   GLU A 276     331.918 346.724 311.746  1.00 88.10           C
ATOM   1517  O   GLU A 276     331.405 346.906 312.856  1.00 88.10           O
ATOM   1518  CB  GLU A 276     333.694 344.946 311.559  1.00 88.10           C
ATOM   1519  CG  GLU A 276     334.739 345.769 310.830  1.00 89.98           C
ATOM   1520  CD  GLU A 276     335.189 345.123 309.538  1.00 89.98           C
ATOM   1521  OE1 GLU A 276     334.460 344.247 309.027  1.00 89.98           O
ATOM   1522  OE2 GLU A 276     336.270 345.491 309.033  1.00 89.98           O
ATOM   1523  N   THR A 277     332.204 347.738 310.929  1.00 89.98           N
ATOM   1524  CA  THR A 277     331.941 349.113 311.342  1.00 89.98           C
ATOM   1525  C   THR A 277     330.570 349.580 310.871  1.00 89.98           C
ATOM   1526  O   THR A 277     329.888 350.333 311.573  1.00 89.98           O
ATOM   1527  CB  THR A 277     333.030 350.045 310.810  1.00 81.18           C
ATOM   1528  OG1 THR A 277     332.971 350.083 309.379  1.00 81.18           O
ATOM   1529  CG2 THR A 277     334.404 349.559 311.239  1.00 81.18           C
ATOM   1530  N   GLY A 278     330.155 349.148 309.682  1.00 81.18           N
ATOM   1531  CA  GLY A 278     328.881 349.537 309.118  1.00 81.18           C
ATOM   1532  C   GLY A 278     328.929 350.722 308.181  1.00 81.18           C
ATOM   1533  O   GLY A 278     327.896 351.060 307.590  1.00 81.18           O
ATOM   1534  N   ASP A 279     330.084 351.358 308.020  1.00 81.18           N
ATOM   1535  CA  ASP A 279     330.176 352.536 307.170  1.00 81.18           C
ATOM   1536  C   ASP A 279     330.277 352.135 305.705  1.00 82.21           C
ATOM   1537  O   ASP A 279     330.972 351.176 305.359  1.00 82.21           O
ATOM   1538  CB  ASP A 279     331.383 353.383 307.569  1.00 82.21           C
ATOM   1539  CG  ASP A 279     331.204 354.052 308.917  1.00 82.21           C
ATOM   1540  OD1 ASP A 279     330.220 353.729 309.615  1.00 82.21           O
ATOM   1541  OD2 ASP A 279     332.047 354.901 309.278  1.00 82.21           O
ATOM   1542  N   LYS A 280     329.581 352.873 304.844  1.00 82.21           N
ATOM   1543  CA  LYS A 280     329.701 352.672 303.408  1.00 82.21           C
ATOM   1544  C   LYS A 280     331.061 353.173 302.943  1.00 77.83           C
ATOM   1545  O   LYS A 280     331.395 354.347 303.132  1.00 77.83           O
ATOM   1546  CB  LYS A 280     328.581 353.407 302.678  1.00 77.83           C
ATOM   1547  CG  LYS A 280     328.749 353.449 301.173  1.00 77.83           C
ATOM   1548  CD  LYS A 280     327.650 354.266 300.519  1.00 77.83           C
ATOM   1549  CE  LYS A 280     327.788 355.739 300.855  1.00 77.83           C
ATOM   1550  NZ  LYS A 280     327.119 356.599 299.846  1.00 77.83           N
ATOM   1551  N   ILE A 281     331.848 352.289 302.333  1.00 77.83           N
ATOM   1552  CA  ILE A 281     333.223 352.619 301.980  1.00 77.83           C
ATOM   1553  C   ILE A 281     333.286 353.130 300.547  1.00 77.83           C
ATOM   1554  O   ILE A 281     333.858 354.192 300.283  1.00 77.83           O
ATOM   1555  CB  ILE A 281     334.151 351.408 302.171  1.00 77.83           C
ATOM   1556  CG1 ILE A 281     333.969 350.812 303.565  1.00 75.34           C
ATOM   1557  CG2 ILE A 281     335.595 351.825 301.983  1.00 75.34           C
ATOM   1558  CD1 ILE A 281     334.113 351.822 304.681  1.00 75.34           C
ATOM   1559  N   TYR A 282     332.700 352.381 299.613  1.00 75.34           N
ATOM   1560  CA  TYR A 282     332.712 352.757 298.208  1.00 75.34           C
ATOM   1561  C   TYR A 282     331.296 352.735 297.656  1.00 75.34           C
ATOM   1562  O   TYR A 282     330.443 351.973 298.116  1.00 75.34           O
ATOM   1563  CB  TYR A 282     333.592 351.829 297.363  1.00 75.34           C
ATOM   1564  CG  TYR A 282     334.985 351.613 297.900  1.00 75.34           C
ATOM   1565  CD1 TYR A 282     335.959 352.591 297.770  1.00 73.41           C
ATOM   1566  CD2 TYR A 282     335.331 350.421 298.521  1.00 73.41           C
ATOM   1567  CE1 TYR A 282     337.235 352.394 298.256  1.00 73.41           C
ATOM   1568  CE2 TYR A 282     336.605 350.214 299.006  1.00 73.41           C
ATOM   1569  CZ  TYR A 282     337.553 351.204 298.872  1.00 73.41           C
ATOM   1570  OH  TYR A 282     338.823 351.002 299.356  1.00 73.41           O
ATOM   1571  N   GLN A 283     331.062 353.581 296.655  1.00 73.41           N
ATOM   1572  CA  GLN A 283     329.785 353.644 295.954  1.00 73.41           C
ATOM   1573  C   GLN A 283     330.070 353.693 294.463  1.00 73.41           C
ATOM   1574  O   GLN A 283     330.809 354.570 294.002  1.00 73.41           O
ATOM   1575  CB  GLN A 283     328.976 354.868 296.390  1.00 73.41           C
ATOM   1576  CG  GLN A 283     327.537 354.873 295.915  1.00 73.41           C
ATOM   1577  CD  GLN A 283     326.708 355.929 296.613  1.00 73.41           C
ATOM   1578  OE1 GLN A 283     327.241 356.790 297.312  1.00 73.41           O
ATOM   1579  NE2 GLN A 283     325.396 355.867 296.433  1.00 73.41           N
ATOM   1580  N   PHE A 284     329.497 352.758 293.711  1.00 73.41           N
ATOM   1581  CA  PHE A 284     329.745 352.652 292.284  1.00 73.41           C
ATOM   1582  C   PHE A 284     328.484 352.945 291.485  1.00 73.41           C
ATOM   1583  O   PHE A 284     327.383 352.521 291.848  1.00 73.41           O
ATOM   1584  CB  PHE A 284     330.266 351.261 291.909  1.00 73.41           C
ATOM   1585  CG  PHE A 284     331.399 350.776 292.767  1.00 73.41           C
ATOM   1586  CD1 PHE A 284     332.638 351.387 292.710  1.00 73.41           C
ATOM   1587  CD2 PHE A 284     331.232 349.690 293.604  1.00 73.41           C
ATOM   1588  CE1 PHE A 284     333.678 350.939 293.487  1.00 73.41           C
ATOM   1589  CE2 PHE A 284     332.270 349.238 294.382  1.00 73.41           C
ATOM   1590  CZ  PHE A 284     333.494 349.864 294.323  1.00 73.41           C
ATOM   1591  N   ARG A 285     328.662 353.668 290.384  1.00 73.41           N
ATOM   1592  CA  ARG A 285     327.617 353.872 289.392  1.00 73.41           C
ATOM   1593  C   ARG A 285     327.757 352.816 288.307  1.00 73.41           C
ATOM   1594  O   ARG A 285     328.859 352.565 287.811  1.00 73.41           O
ATOM   1595  CB  ARG A 285     327.706 355.266 288.767  1.00 73.41           C
ATOM   1596  CG  ARG A 285     327.488 356.410 289.731  1.00 73.41           C
ATOM   1597  CD  ARG A 285     327.468 357.731 288.982  1.00 73.41           C
ATOM   1598  NE  ARG A 285     327.255 358.869 289.864  1.00 73.41           N
ATOM   1599  CZ  ARG A 285     327.249 360.133 289.462  1.00 73.41           C
ATOM   1600  NH1 ARG A 285     327.437 360.454 288.193  1.00 73.41           N
ATOM   1601  NH2 ARG A 285     327.047 361.097 290.354  1.00 73.41           N
ATOM   1602  N   ARG A 286     326.641 352.197 287.940  1.00 73.41           N
ATOM   1603  CA  ARG A 286     326.645 351.081 287.006  1.00 73.41           C
ATOM   1604  C   ARG A 286     325.885 351.438 285.740  1.00 73.41           C
ATOM   1605  O   ARG A 286     324.681 351.699 285.786  1.00 73.41           O
ATOM   1606  CB  ARG A 286     326.033 349.840 287.650  1.00 73.41           C
ATOM   1607  CG  ARG A 286     325.950 348.659 286.721  1.00 73.41           C
ATOM   1608  CD  ARG A 286     325.163 347.546 287.358  1.00 73.41           C
ATOM   1609  NE  ARG A 286     324.714 346.579 286.370  1.00 73.41           N
ATOM   1610  CZ  ARG A 286     323.464 346.483 285.947  1.00 73.41           C
ATOM   1611  NH1 ARG A 286     322.513 347.268 286.420  1.00 73.41           N
ATOM   1612  NH2 ARG A 286     323.159 345.575 285.027  1.00 73.41           N
ATOM   1613  N   GLY A 287     326.586 351.418 284.610  1.00 73.41           N
ATOM   1614  CA  GLY A 287     325.962 351.645 283.320  1.00 73.41           C
ATOM   1615  C   GLY A 287     325.487 353.062 283.084  1.00 73.41           C
ATOM   1616  O   GLY A 287     326.071 354.019 283.604  1.00 73.41           O
ATOM   1617  N   THR A 288     324.445 353.211 282.265  1.00 73.41           N
ATOM   1618  CA  THR A 288     323.832 354.503 281.984  1.00 73.41           C
ATOM   1619  C   THR A 288     322.384 354.557 282.452  1.00 73.41           C
ATOM   1620  O   THR A 288     322.003 355.485 283.166  1.00 73.41           O
ATOM   1621  CB  THR A 288     323.924 354.821 280.487  1.00 73.41           C
ATOM   1622  OG1 THR A 288     323.105 353.905 279.753  1.00 73.41           O
ATOM   1623  CG2 THR A 288     325.361 354.711 280.007  1.00 73.41           C
ATOM   1624  N   TYR A 289     321.563 353.583 282.072  1.00 73.41           N
ATOM   1625  CA  TYR A 289     320.177 353.534 282.511  1.00 73.41           C
ATOM   1626  C   TYR A 289     319.939 352.343 283.432  1.00 73.41           C
ATOM   1627  O   TYR A 289     320.752 351.419 283.512  1.00 73.41           O
ATOM   1628  CB  TYR A 289     319.228 353.468 281.315  1.00 73.41           C
ATOM   1629  CG  TYR A 289     319.566 352.396 280.308  1.00 73.41           C
ATOM   1630  CD1 TYR A 289     319.079 351.106 280.449  1.00 73.41           C
ATOM   1631  CD2 TYR A 289     320.359 352.678 279.207  1.00 73.41           C
ATOM   1632  CE1 TYR A 289     319.381 350.128 279.530  1.00 73.41           C
ATOM   1633  CE2 TYR A 289     320.666 351.705 278.283  1.00 73.41           C
ATOM   1634  CZ  TYR A 289     320.175 350.432 278.448  1.00 73.41           C
ATOM   1635  OH  TYR A 289     320.477 349.456 277.529  1.00 73.41           O
ATOM   1636  N   ALA A 290     318.813 352.382 284.139  1.00 73.41           N
ATOM   1637  CA  ALA A 290     318.488 351.332 285.096  1.00 73.41           C
ATOM   1638  C   ALA A 290     318.191 350.020 284.386  1.00 73.41           C
ATOM   1639  O   ALA A 290     317.479 349.992 283.379  1.00 73.41           O
ATOM   1640  CB  ALA A 290     317.291 351.751 285.945  1.00 73.41           C
ATOM   1641  N   THR A 291     318.739 348.931 284.916  1.00 73.41           N
ATOM   1642  CA  THR A 291     318.475 347.595 284.406  1.00 73.41           C
ATOM   1643  C   THR A 291     318.766 346.568 285.492  1.00 73.41           C
ATOM   1644  O   THR A 291     319.478 346.840 286.462  1.00 73.41           O
ATOM   1645  CB  THR A 291     319.299 347.296 283.147  1.00 73.41           C
ATOM   1646  OG1 THR A 291     318.854 346.067 282.562  1.00 73.41           O
ATOM   1647  CG2 THR A 291     320.774 347.197 283.479  1.00 73.41           C
ATOM   1648  N   ARG A 292     318.207 345.374 285.315  1.00 73.41           N
ATOM   1649  CA  ARG A 292     318.266 344.346 286.344  1.00 73.41           C
ATOM   1650  C   ARG A 292     319.693 343.844 286.536  1.00 73.41           C
ATOM   1651  O   ARG A 292     320.515 343.867 285.616  1.00 73.41           O
ATOM   1652  CB  ARG A 292     317.337 343.188 285.975  1.00 73.41           C
ATOM   1653  CG  ARG A 292     317.178 342.123 287.045  1.00 73.41           C
ATOM   1654  CD  ARG A 292     316.046 341.172 286.697  1.00 73.41           C
ATOM   1655  NE  ARG A 292     315.850 340.149 287.719  1.00 73.41           N
ATOM   1656  CZ  ARG A 292     316.396 338.943 287.687  1.00 73.41           C
ATOM   1657  NH1 ARG A 292     317.167 338.560 286.681  1.00 73.41           N
ATOM   1658  NH2 ARG A 292     316.158 338.096 288.683  1.00 73.41           N
ATOM   1659  N   ILE A 293     319.991 343.402 287.756  1.00 73.41           N
ATOM   1660  CA  ILE A 293     321.288 342.838 288.115  1.00 73.41           C
ATOM   1661  C   ILE A 293     321.085 341.378 288.488  1.00 73.41           C
ATOM   1662  O   ILE A 293     320.179 341.050 289.262  1.00 73.41           O
ATOM   1663  CB  ILE A 293     321.943 343.608 289.274  1.00 73.41           C
ATOM   1664  CG1 ILE A 293     322.429 344.971 288.799  1.00 73.41           C
ATOM   1665  CG2 ILE A 293     323.106 342.824 289.840  1.00 73.41           C
ATOM   1666  CD1 ILE A 293     323.231 345.706 289.833  1.00 73.41           C
ATOM   1667  N   TYR A 294     321.924 340.503 287.937  1.00 73.41           N
ATOM   1668  CA  TYR A 294     321.784 339.074 288.189  1.00 73.41           C
ATOM   1669  C   TYR A 294     322.528 338.620 289.437  1.00 73.41           C
ATOM   1670  O   TYR A 294     321.941 337.948 290.291  1.00 73.41           O
ATOM   1671  CB  TYR A 294     322.274 338.273 286.985  1.00 73.41           C
ATOM   1672  CG  TYR A 294     321.491 338.532 285.725  1.00 73.41           C
ATOM   1673  CD1 TYR A 294     320.175 338.121 285.603  1.00 73.41           C
ATOM   1674  CD2 TYR A 294     322.078 339.172 284.646  1.00 73.41           C
ATOM   1675  CE1 TYR A 294     319.461 338.354 284.449  1.00 73.41           C
ATOM   1676  CE2 TYR A 294     321.374 339.398 283.493  1.00 73.41           C
ATOM   1677  CZ  TYR A 294     320.068 338.996 283.397  1.00 73.41           C
ATOM   1678  OH  TYR A 294     319.377 339.235 282.235  1.00 73.41           O
ATOM   1679  N   SER A 295     323.810 338.961 289.561  1.00 73.41           N
ATOM   1680  CA  SER A 295     324.598 338.512 290.698  1.00 73.41           C
ATOM   1681  C   SER A 295     325.816 339.405 290.877  1.00 73.41           C
ATOM   1682  O   SER A 295     326.252 340.083 289.944  1.00 73.41           O
ATOM   1683  CB  SER A 295     325.046 337.059 290.525  1.00 73.41           C
ATOM   1684  OG  SER A 295     326.108 336.970 289.592  1.00 73.41           O
ATOM   1685  N   ILE A 296     326.361 339.391 292.091  1.00 73.41           N
ATOM   1686  CA  ILE A 296     327.583 340.103 292.434  1.00 73.41           C
ATOM   1687  C   ILE A 296     328.539 339.114 293.085  1.00 73.41           C
ATOM   1688  O   ILE A 296     328.103 338.180 293.768  1.00 73.41           O
ATOM   1689  CB  ILE A 296     327.314 341.296 293.378  1.00 73.41           C
ATOM   1690  CG1 ILE A 296     326.065 342.056 292.946  1.00 73.41           C
ATOM   1691  CG2 ILE A 296     328.488 342.246 293.396  1.00 73.41           C
ATOM   1692  CD1 ILE A 296     325.723 343.201 293.858  1.00 73.41           C
ATOM   1693  N   SER A 297     329.838 339.307 292.868  1.00 73.41           N
ATOM   1694  CA  SER A 297     330.837 338.424 293.453  1.00 73.41           C
ATOM   1695  C   SER A 297     332.152 339.173 293.605  1.00 79.14           C
ATOM   1696  O   SER A 297     332.522 339.966 292.735  1.00 79.14           O
ATOM   1697  CB  SER A 297     331.037 337.174 292.594  1.00 79.14           C
ATOM   1698  OG  SER A 297     331.643 337.501 291.357  1.00 79.14           O
ATOM   1699  N   PHE A 298     332.848 338.913 294.708  1.00 79.14           N
ATOM   1700  CA  PHE A 298     334.138 339.535 294.969  1.00 79.14           C
ATOM   1701  C   PHE A 298     335.266 338.703 294.380  1.00 85.05           C
ATOM   1702  O   PHE A 298     335.077 337.542 294.010  1.00 85.05           O
ATOM   1703  CB  PHE A 298     334.362 339.692 296.471  1.00 85.05           C
ATOM   1704  CG  PHE A 298     333.610 340.827 297.089  1.00 85.05           C
ATOM   1705  CD1 PHE A 298     334.188 342.078 297.198  1.00 85.05           C
ATOM   1706  CD2 PHE A 298     332.330 340.642 297.575  1.00 85.05           C
ATOM   1707  CE1 PHE A 298     333.503 343.121 297.768  1.00 85.05           C
ATOM   1708  CE2 PHE A 298     331.640 341.680 298.149  1.00 85.05           C
ATOM   1709  CZ  PHE A 298     332.228 342.923 298.246  1.00 85.05           C
ATOM   1710  N   SER A 299     336.447 339.307 294.295  1.00 86.05           N
ATOM   1711  CA  SER A 299     337.649 338.539 294.020  1.00 86.05           C
ATOM   1712  C   SER A 299     338.116 337.835 295.291  1.00 86.05           C
ATOM   1713  O   SER A 299     337.645 338.114 296.396  1.00 86.05           O
ATOM   1714  CB  SER A 299     338.755 339.441 293.483  1.00 86.05           C
ATOM   1715  OG  SER A 299     339.241 340.304 294.496  1.00 86.05           O
ATOM   1716  N   GLU A 300     339.053 336.900 295.121  1.00 86.05           N
ATOM   1717  CA  GLU A 300     339.599 336.194 296.276  1.00 86.05           C
ATOM   1718  C   GLU A 300     340.342 337.145 297.203  1.00 82.95           C
ATOM   1719  O   GLU A 300     340.256 337.022 298.430  1.00 82.95           O
ATOM   1720  CB  GLU A 300     340.515 335.063 295.816  1.00 82.95           C
ATOM   1721  CG  GLU A 300     339.787 333.940 295.107  1.00 82.95           C
ATOM   1722  CD  GLU A 300     338.719 333.305 295.975  1.00 82.95           C
ATOM   1723  OE1 GLU A 300     338.990 333.063 297.170  1.00 82.95           O
ATOM   1724  OE2 GLU A 300     337.608 333.052 295.465  1.00 80.50           O
ATOM   1725  N   ASP A 301     341.073 338.100 296.637  1.00 80.50           N
ATOM   1726  CA  ASP A 301     341.834 339.069 297.409  1.00 80.50           C
ATOM   1727  C   ASP A 301     341.081 340.376 297.620  1.00 80.50           C
ATOM   1728  O   ASP A 301     341.691 341.368 298.035  1.00 80.50           O
ATOM   1729  CB  ASP A 301     343.176 339.343 296.727  1.00 80.50           C
ATOM   1730  CG  ASP A 301     343.030 339.623 295.243  1.00 80.50           C
ATOM   1731  OD1 ASP A 301     341.929 339.394 294.699  1.00 80.50           O
ATOM   1732  OD2 ASP A 301     344.018 340.062 294.620  1.00 80.50           O
ATOM   1733  N   SER A 302     339.780 340.402 297.336  1.00 75.70           N
ATOM   1734  CA  SER A 302     338.927 341.579 297.479  1.00 75.70           C
ATOM   1735  C   SER A 302     339.435 342.770 296.682  1.00 75.70           C
ATOM   1736  O   SER A 302     339.212 343.919 297.083  1.00 75.70           O
ATOM   1737  CB  SER A 302     338.749 341.973 298.950  1.00 75.70           C
ATOM   1738  OG  SER A 302     338.078 340.955 299.673  1.00 75.70           O
ATOM   1739  N   GLN A 303     340.131 342.527 295.571  1.00 75.70           N
ATOM   1740  CA  GLN A 303     340.651 343.627 294.764  1.00 75.70           C
ATOM   1741  C   GLN A 303     339.596 344.149 293.795  1.00 75.70           C
ATOM   1742  O   GLN A 303     339.516 345.354 293.539  1.00 75.70           O
ATOM   1743  CB  GLN A 303     341.899 343.170 294.009  1.00 75.70           C
ATOM   1744  CG  GLN A 303     342.757 344.300 293.472  1.00 75.70           C
ATOM   1745  CD  GLN A 303     343.512 345.023 294.563  1.00 73.41           C
ATOM   1746  OE1 GLN A 303     343.105 346.093 295.008  1.00 73.41           O
ATOM   1747  NE2 GLN A 303     344.622 344.442 294.998  1.00 73.41           N
ATOM   1748  N   TYR A 304     338.776 343.256 293.247  1.00 73.41           N
ATOM   1749  CA  TYR A 304     337.810 343.618 292.222  1.00 73.41           C
ATOM   1750  C   TYR A 304     336.439 343.060 292.573  1.00 73.41           C
ATOM   1751  O   TYR A 304     336.311 342.085 293.318  1.00 73.41           O
ATOM   1752  CB  TYR A 304     338.237 343.103 290.840  1.00 73.41           C
ATOM   1753  CG  TYR A 304     339.676 343.393 290.488  1.00 73.41           C
ATOM   1754  CD1 TYR A 304     340.038 344.591 289.894  1.00 73.41           C
ATOM   1755  CD2 TYR A 304     340.672 342.465 290.748  1.00 73.41           C
ATOM   1756  CE1 TYR A 304     341.352 344.858 289.571  1.00 73.41           C
ATOM   1757  CE2 TYR A 304     341.987 342.724 290.428  1.00 73.41           C
ATOM   1758  CZ  TYR A 304     342.323 343.921 289.840  1.00 73.41           C
ATOM   1759  OH  TYR A 304     343.633 344.182 289.522  1.00 73.41           O
ATOM   1760  N   LEU A 305     335.409 343.698 292.019  1.00 73.41           N
ATOM   1761  CA  LEU A 305     334.021 343.299 292.221  1.00 73.41           C
ATOM   1762  C   LEU A 305     333.347 343.169 290.864  1.00 73.41           C
ATOM   1763  O   LEU A 305     333.463 344.066 290.024  1.00 73.41           O
ATOM   1764  CB  LEU A 305     333.277 344.317 293.090  1.00 73.41           C
ATOM   1765  CG  LEU A 305     331.841 343.961 293.466  1.00 73.41           C
ATOM   1766  CD1 LEU A 305     331.842 342.711 294.310  1.00 73.41           C
ATOM   1767  CD2 LEU A 305     331.174 345.104 294.207  1.00 73.41           C
ATOM   1768  N   ALA A 306     332.634 342.065 290.657  1.00 73.41           N
ATOM   1769  CA  ALA A 306     332.042 341.754 289.364  1.00 73.41           C
ATOM   1770  C   ALA A 306     330.521 341.751 289.448  1.00 73.41           C
ATOM   1771  O   ALA A 306     329.940 341.084 290.309  1.00 73.41           O
ATOM   1772  CB  ALA A 306     332.528 340.396 288.860  1.00 73.41           C
ATOM   1773  N   VAL A 307     329.883 342.488 288.542  1.00 73.41           N
ATOM   1774  CA  VAL A 307     328.427 342.534 288.447  1.00 73.41           C
ATOM   1775  C   VAL A 307     328.030 342.198 287.016  1.00 73.41           C
ATOM   1776  O   VAL A 307     328.734 342.564 286.068  1.00 73.41           O
ATOM   1777  CB  VAL A 307     327.852 343.904 288.860  1.00 73.41           C
ATOM   1778  CG1 VAL A 307     328.333 344.291 290.244  1.00 73.41           C
ATOM   1779  CG2 VAL A 307     328.212 344.971 287.850  1.00 73.41           C
ATOM   1780  N   THR A 308     326.924 341.479 286.861  1.00 73.41           N
ATOM   1781  CA  THR A 308     326.394 341.113 285.557  1.00 73.41           C
ATOM   1782  C   THR A 308     324.926 341.497 285.483  1.00 73.41           C
ATOM   1783  O   THR A 308     324.168 341.261 286.427  1.00 73.41           O
ATOM   1784  CB  THR A 308     326.544 339.614 285.293  1.00 73.41           C
ATOM   1785  OG1 THR A 308     325.475 338.911 285.936  1.00 73.41           O
ATOM   1786  CG2 THR A 308     327.867 339.098 285.826  1.00 73.41           C
ATOM   1787  N   GLY A 309     324.522 342.072 284.352  1.00 73.41           N
ATOM   1788  CA  GLY A 309     323.161 342.553 284.214  1.00 73.41           C
ATOM   1789  C   GLY A 309     322.510 342.216 282.892  1.00 73.41           C
ATOM   1790  O   GLY A 309     323.118 341.555 282.047  1.00 73.41           O
ATOM   1791  N   SER A 310     321.271 342.664 282.702  1.00 73.41           N
ATOM   1792  CA  SER A 310     320.478 342.292 281.537  1.00 73.41           C
ATOM   1793  C   SER A 310     320.943 342.979 280.263  1.00 73.41           C
ATOM   1794  O   SER A 310     320.430 342.661 279.184  1.00 73.41           O
ATOM   1795  CB  SER A 310     319.000 342.606 281.780  1.00 73.41           C
ATOM   1796  OG  SER A 310     318.719 343.969 281.515  1.00 73.41           O
ATOM   1797  N   SER A 311     321.894 343.902 280.358  1.00 73.41           N
ATOM   1798  CA  SER A 311     322.465 344.553 279.189  1.00 73.41           C
ATOM   1799  C   SER A 311     323.484 343.683 278.471  1.00 73.41           C
ATOM   1800  O   SER A 311     324.177 344.179 277.578  1.00 73.41           O
ATOM   1801  CB  SER A 311     323.117 345.881 279.591  1.00 73.41           C
ATOM   1802  OG  SER A 311     324.076 345.683 280.614  1.00 73.41           O
ATOM   1803  N   LYS A 312     323.600 342.408 278.849  1.00 73.41           N
ATOM   1804  CA  LYS A 312     324.548 341.470 278.252  1.00 73.41           C
ATOM   1805  C   LYS A 312     325.987 341.952 278.399  1.00 73.41           C
ATOM   1806  O   LYS A 312     326.860 341.590 277.603  1.00 73.41           O
ATOM   1807  CB  LYS A 312     324.197 341.209 276.780  1.00 73.41           C
ATOM   1808  CG  LYS A 312     324.775 339.941 276.176  1.00 73.41           C
ATOM   1809  CD  LYS A 312     324.255 339.739 274.762  1.00 73.41           C
ATOM   1810  CE  LYS A 312     324.717 340.852 273.838  1.00 73.41           C
ATOM   1811  NZ  LYS A 312     326.192 340.871 273.677  1.00 73.41           N
ATOM   1812  N   THR A 313     326.251 342.776 279.408  1.00 73.41           N
ATOM   1813  CA  THR A 313     327.576 343.310 279.676  1.00 73.41           C
ATOM   1814  C   THR A 313     327.924 343.094 281.142  1.00 73.41           C
ATOM   1815  O   THR A 313     327.078 343.274 282.023  1.00 73.41           O
ATOM   1816  CB  THR A 313     327.649 344.800 279.331  1.00 73.41           C
ATOM   1817  OG1 THR A 313     327.301 344.989 277.955  1.00 73.41           O
ATOM   1818  CG2 THR A 313     329.048 345.345 279.569  1.00 73.41           C
ATOM   1819  N   VAL A 314     329.171 342.709 281.395  1.00 73.41           N
ATOM   1820  CA  VAL A 314     329.670 342.450 282.739  1.00 73.41           C
ATOM   1821  C   VAL A 314     330.671 343.541 283.082  1.00 73.41           C
ATOM   1822  O   VAL A 314     331.584 343.821 282.295  1.00 73.41           O
ATOM   1823  CB  VAL A 314     330.313 341.059 282.850  1.00 73.41           C
ATOM   1824  CG1 VAL A 314     330.899 340.857 284.233  1.00 73.41           C
ATOM   1825  CG2 VAL A 314     329.292 339.983 282.538  1.00 73.41           C
ATOM   1826  N   HIS A 315     330.500 344.158 284.246  1.00 73.41           N
ATOM   1827  CA  HIS A 315     331.357 345.243 284.697  1.00 73.41           C
ATOM   1828  C   HIS A 315     332.252 344.766 285.832  1.00 73.41           C
ATOM   1829  O   HIS A 315     331.864 343.898 286.620  1.00 73.41           O
ATOM   1830  CB  HIS A 315     330.531 346.441 285.166  1.00 73.41           C
ATOM   1831  CG  HIS A 315     329.749 347.101 284.077  1.00 73.41           C
ATOM   1832  ND1 HIS A 315     330.289 348.061 283.251  1.00 73.41           N
ATOM   1833  CD2 HIS A 315     328.464 346.944 283.680  1.00 73.41           C
ATOM   1834  CE1 HIS A 315     329.373 348.466 282.390  1.00 73.41           C
ATOM   1835  NE2 HIS A 315     328.256 347.803 282.628  1.00 73.41           N
ATOM   1836  N   ILE A 316     333.450 345.335 285.908  1.00 73.41           N
ATOM   1837  CA  ILE A 316     334.388 345.066 286.989  1.00 73.41           C
ATOM   1838  C   ILE A 316     334.698 346.379 287.690  1.00 73.41           C
ATOM   1839  O   ILE A 316     335.138 347.342 287.049  1.00 73.41           O
ATOM   1840  CB  ILE A 316     335.677 344.410 286.476  1.00 76.75           C
ATOM   1841  CG1 ILE A 316     335.350 343.130 285.711  1.00 76.75           C
ATOM   1842  CG2 ILE A 316     336.601 344.111 287.637  1.00 76.75           C
ATOM   1843  CD1 ILE A 316     334.665 342.091 286.551  1.00 76.75           C
ATOM   1844  N   PHE A 317     334.481 346.413 289.000  1.00 76.75           N
ATOM   1845  CA  PHE A 317     334.782 347.574 289.821  1.00 76.75           C
ATOM   1846  C   PHE A 317     336.002 347.282 290.682  1.00 76.75           C
ATOM   1847  O   PHE A 317     336.068 346.243 291.346  1.00 76.75           O
ATOM   1848  CB  PHE A 317     333.594 347.944 290.710  1.00 76.75           C
ATOM   1849  CG  PHE A 317     332.318 348.175 289.955  1.00 77.82           C
ATOM   1850  CD1 PHE A 317     332.068 349.390 289.343  1.00 77.82           C
ATOM   1851  CD2 PHE A 317     331.359 347.184 289.873  1.00 77.82           C
ATOM   1852  CE1 PHE A 317     330.891 349.604 288.656  1.00 77.82           C
ATOM   1853  CE2 PHE A 317     330.183 347.396 289.192  1.00 77.82           C
ATOM   1854  CZ  PHE A 317     329.949 348.606 288.580  1.00 77.82           C
ATOM   1855  N   LYS A 318     336.962 348.199 290.673  1.00 77.82           N
ATOM   1856  CA  LYS A 318     338.214 348.041 291.401  1.00 77.82           C
ATOM   1857  C   LYS A 318     338.029 348.561 292.819  1.00 85.53           C
ATOM   1858  O   LYS A 318     337.698 349.735 293.015  1.00 85.53           O
ATOM   1859  CB  LYS A 318     339.350 348.785 290.700  1.00 85.53           C
ATOM   1860  CG  LYS A 318     340.619 348.892 291.519  1.00 85.53           C
ATOM   1861  CD  LYS A 318     341.741 349.522 290.716  1.00 73.41           C
ATOM   1862  CE  LYS A 318     342.616 348.472 290.065  1.00 73.41           C
ATOM   1863  NZ  LYS A 318     343.769 348.118 290.933  1.00 73.41           N
ATOM   1864  N   LEU A 319     338.247 347.689 293.804  1.00 73.41           N
ATOM   1865  CA  LEU A 319     338.142 348.047 295.212  1.00 73.41           C
ATOM   1866  C   LEU A 319     339.499 348.362 295.828  1.00 73.41           C
ATOM   1867  O   LEU A 319     339.643 348.334 297.055  1.00 73.41           O
ATOM   1868  CB  LEU A 319     337.452 346.931 295.996  1.00 73.41           C
ATOM   1869  CG  LEU A 319     335.986 346.655 295.656  1.00 73.41           C
ATOM   1870  CD1 LEU A 319     335.652 345.189 295.833  1.00 73.41           C
ATOM   1871  CD2 LEU A 319     335.082 347.506 296.521  1.00 73.41           C
ATOM   1872  N   GLY A 320     340.503 348.653 295.002  1.00 73.41           N
ATOM   1873  CA  GLY A 320     341.817 348.965 295.538  1.00 73.41           C
ATOM   1874  C   GLY A 320     341.911 350.376 296.084  1.00 73.41           C
ATOM   1875  O   GLY A 320     342.917 350.750 296.693  1.00 73.41           O
ATOM   1876  N   HIS A 321     340.872 351.178 295.870  1.00 73.41           N
ATOM   1877  CA  HIS A 321     340.868 352.572 296.299  1.00 73.41           C
ATOM   1878  C   HIS A 321     340.981 352.688 297.817  1.00 73.41           C
ATOM   1879  O   HIS A 321     341.492 353.679 298.336  1.00 73.41           O
ATOM   1880  CB  HIS A 321     339.599 353.276 295.818  1.00 73.41           C
ATOM   1881  CG  HIS A 321     339.414 353.244 294.334  1.00 73.41           C
ATOM   1882  ND1 HIS A 321     340.149 354.030 293.473  1.00 73.41           N
ATOM   1883  CD2 HIS A 321     338.570 352.525 293.557  1.00 73.41           C
ATOM   1884  CE1 HIS A 321     339.771 353.793 292.231  1.00 73.41           C
ATOM   1885  NE2 HIS A 321     338.812 352.884 292.254  1.00 73.41           N
ATOM   1886  N   SER A 409     332.578 354.434 291.857  1.00 73.41           N
ATOM   1887  CA  SER A 409     332.497 355.341 290.720  1.00 73.41           C
ATOM   1888  C   SER A 409     332.209 354.580 289.431  1.00 77.82           C
ATOM   1889  O   SER A 409     331.394 353.660 289.411  1.00 77.82           O
ATOM   1890  CB  SER A 409     333.793 356.140 290.580  1.00 77.82           C
ATOM   1891  OG  SER A 409     334.890 355.288 290.300  1.00 77.82           O
ATOM   1892  N   ARG A 410     332.880 354.979 288.356  1.00 77.82           N
ATOM   1893  CA  ARG A 410     332.702 354.312 287.076  1.00 77.82           C
ATOM   1894  C   ARG A 410     333.345 352.932 287.101  1.00 77.82           C
ATOM   1895  O   ARG A 410     334.330 352.702 287.807  1.00 77.82           O
ATOM   1896  CB  ARG A 410     333.307 355.155 285.952  1.00 77.82           C
ATOM   1897  CG  ARG A 410     332.544 356.434 285.659  1.00 77.82           C
ATOM   1898  CD  ARG A 410     331.271 356.148 284.887  1.00 77.82           C
ATOM   1899  NE  ARG A 410     331.551 355.610 283.561  1.00 74.50           N
ATOM   1900  CZ  ARG A 410     330.638 355.074 282.764  1.00 74.50           C
ATOM   1901  NH1 ARG A 410     329.369 354.984 283.128  1.00 74.50           N
ATOM   1902  NH2 ARG A 410     331.007 354.616 281.571  1.00 74.50           N
ATOM   1903  N   HIS A 411     332.776 352.008 286.327  1.00 74.50           N
ATOM   1904  CA  HIS A 411     333.319 350.659 286.251  1.00 73.41           C
ATOM   1905  C   HIS A 411     334.742 350.687 285.709  1.00 73.41           C
ATOM   1906  O   HIS A 411     335.052 351.394 284.747  1.00 73.41           O
ATOM   1907  CB  HIS A 411     332.435 349.777 285.367  1.00 73.41           C
ATOM   1908  CG  HIS A 411     332.230 350.318 283.986  1.00 73.41           C
ATOM   1909  ND1 HIS A 411     333.117 350.090 282.958  1.00 73.41           N
ATOM   1910  CD2 HIS A 411     331.237 351.076 283.465  1.00 73.41           C
ATOM   1911  CE1 HIS A 411     332.682 350.686 281.862  1.00 73.41           C
ATOM   1912  NE2 HIS A 411     331.541 351.291 282.143  1.00 73.41           N
ATOM   1913  N   PHE A 412     335.616 349.909 286.347  1.00 73.41           N
ATOM   1914  CA  PHE A 412     337.015 349.876 285.936  1.00 73.41           C
ATOM   1915  C   PHE A 412     337.203 349.215 284.579  1.00 73.41           C
ATOM   1916  O   PHE A 412     337.884 349.775 283.712  1.00 73.41           O
ATOM   1917  CB  PHE A 412     337.852 349.151 286.988  1.00 73.41           C
ATOM   1918  CG  PHE A 412     339.313 349.096 286.665  1.00 73.41           C
ATOM   1919  CD1 PHE A 412     339.973 350.219 286.199  1.00 73.41           C
ATOM   1920  CD2 PHE A 412     340.027 347.924 286.830  1.00 73.41           C
ATOM   1921  CE1 PHE A 412     341.315 350.173 285.903  1.00 73.41           C
ATOM   1922  CE2 PHE A 412     341.370 347.873 286.536  1.00 73.41           C
ATOM   1923  CZ  PHE A 412     342.014 348.999 286.072  1.00 73.41           C
ATOM   1924  N   ALA A 413     336.611 348.043 284.372  1.00 73.41           N
ATOM   1925  CA  ALA A 413     336.750 347.322 283.116  1.00 73.41           C
ATOM   1926  C   ALA A 413     335.423 346.655 282.795  1.00 73.41           C
ATOM   1927  O   ALA A 413     334.660 346.287 283.691  1.00 73.41           O
ATOM   1928  CB  ALA A 413     337.883 346.295 283.184  1.00 73.41           C
ATOM   1929  N   SER A 414     335.163 346.490 281.502  1.00 73.41           N
ATOM   1930  CA  SER A 414     333.884 345.970 281.048  1.00 73.41           C
ATOM   1931  C   SER A 414     334.119 344.743 280.183  1.00 73.41           C
ATOM   1932  O   SER A 414     335.067 344.691 279.398  1.00 73.41           O
ATOM   1933  CB  SER A 414     333.097 347.022 280.261  1.00 73.41           C
ATOM   1934  OG  SER A 414     333.644 347.194 278.967  1.00 73.41           O
ATOM   1935  N   LEU A 415     333.237 343.758 280.334  1.00 73.41           N
ATOM   1936  CA  LEU A 415     333.307 342.511 279.585  1.00 73.41           C
ATOM   1937  C   LEU A 415     332.163 342.483 278.586  1.00 73.41           C
ATOM   1938  O   LEU A 415     330.993 342.463 278.979  1.00 73.41           O
ATOM   1939  CB  LEU A 415     333.227 341.301 280.518  1.00 73.41           C
ATOM   1940  CG  LEU A 415     334.125 341.277 281.754  1.00 73.41           C
ATOM   1941  CD1 LEU A 415     334.131 339.889 282.367  1.00 73.41           C
ATOM   1942  CD2 LEU A 415     335.536 341.711 281.411  1.00 73.41           C
ATOM   1943  N   LYS A 416     332.496 342.482 277.301  1.00 73.41           N
ATOM   1944  CA  LYS A 416     331.503 342.434 276.239  1.00 73.41           C
ATOM   1945  C   LYS A 416     331.506 341.042 275.624  1.00 73.41           C
ATOM   1946  O   LYS A 416     332.500 340.626 275.021  1.00 73.41           O
ATOM   1947  CB  LYS A 416     331.785 343.496 275.178  1.00 73.41           C
ATOM   1948  CG  LYS A 416     332.181 344.842 275.754  1.00 73.41           C
ATOM   1949  CD  LYS A 416     331.850 345.974 274.803  1.00 73.41           C
ATOM   1950  CE  LYS A 416     332.108 347.323 275.451  1.00 73.41           C
ATOM   1951  NZ  LYS A 416     330.843 347.999 275.847  1.00 73.41           N
ATOM   1952  N   LEU A 417     330.395 340.328 275.779  1.00 73.41           N
ATOM   1953  CA  LEU A 417     330.273 339.003 275.206  1.00 73.41           C
ATOM   1954  C   LEU A 417     330.170 339.099 273.684  1.00 73.41           C
ATOM   1955  O   LEU A 417     329.864 340.163 273.141  1.00 73.41           O
ATOM   1956  CB  LEU A 417     329.053 338.291 275.781  1.00 73.41           C
ATOM   1957  CG  LEU A 417     329.068 338.042 277.289  1.00 73.41           C
ATOM   1958  CD1 LEU A 417     328.050 336.983 277.665  1.00 73.41           C
ATOM   1959  CD2 LEU A 417     330.454 337.637 277.749  1.00 73.41           C
ATOM   1960  N   PRO A 418     330.446 338.006 272.972  1.00 73.41           N
ATOM   1961  CA  PRO A 418     330.297 338.024 271.515  1.00 73.41           C
ATOM   1962  C   PRO A 418     328.867 338.344 271.102  1.00 73.41           C
ATOM   1963  O   PRO A 418     327.907 338.010 271.797  1.00 73.41           O
ATOM   1964  CB  PRO A 418     330.702 336.604 271.097  1.00 73.41           C
ATOM   1965  CG  PRO A 418     330.685 335.800 272.355  1.00 73.41           C
ATOM   1966  CD  PRO A 418     331.056 336.752 273.435  1.00 73.41           C
ATOM   1967  N   VAL A 419     328.743 339.003 269.947  1.00 73.41           N
ATOM   1968  CA  VAL A 419     327.462 339.564 269.522  1.00 73.41           C
ATOM   1969  C   VAL A 419     326.430 338.463 269.307  1.00 73.41           C
ATOM   1970  O   VAL A 419     325.238 338.649 269.577  1.00 73.41           O
ATOM   1971  CB  VAL A 419     327.654 340.414 268.253  1.00 73.41           C
ATOM   1972  CG1 VAL A 419     326.325 340.948 267.757  1.00 73.41           C
ATOM   1973  CG2 VAL A 419     328.622 341.552 268.525  1.00 73.41           C
ATOM   1974  N   GLU A 420     326.870 337.297 268.833  1.00 73.41           N
ATOM   1975  CA  GLU A 420     325.928 336.235 268.490  1.00 73.41           C
ATOM   1976  C   GLU A 420     325.220 335.677 269.718  1.00 73.41           C
ATOM   1977  O   GLU A 420     324.242 334.932 269.583  1.00 73.41           O
ATOM   1978  CB  GLU A 420     326.651 335.111 267.744  1.00 73.41           C
ATOM   1979  CG  GLU A 420     327.591 335.582 266.643  1.00 73.41           C
ATOM   1980  CD  GLU A 420     328.977 335.922 267.156  1.00 73.41           C
ATOM   1981  OE1 GLU A 420     329.827 336.348 266.346  1.00 73.41           O
ATOM   1982  OE2 GLU A 420     329.218 335.760 268.369  1.00 73.41           O
ATOM   1983  N   THR A 421     325.696 336.011 270.915  1.00 73.41           N
ATOM   1984  CA  THR A 421     325.092 335.497 272.137  1.00 73.41           C
ATOM   1985  C   THR A 421     323.679 336.038 272.314  1.00 73.41           C
ATOM   1986  O   THR A 421     323.405 337.206 272.018  1.00 73.41           O
ATOM   1987  CB  THR A 421     325.947 335.871 273.346  1.00 73.41           C
ATOM   1988  OG1 THR A 421     326.142 337.290 273.373  1.00 73.41           O
ATOM   1989  CG2 THR A 421     327.293 335.185 273.272  1.00 73.41           C
ATOM   1990  N   ASN A 422     322.784 335.183 272.800  1.00 73.41           N
ATOM   1991  CA  ASN A 422     321.422 335.604 273.090  1.00 73.41           C
ATOM   1992  C   ASN A 422     321.428 336.652 274.195  1.00 73.41           C
ATOM   1993  O   ASN A 422     322.249 336.592 275.115  1.00 73.41           O
ATOM   1994  CB  ASN A 422     320.576 334.401 273.503  1.00 73.41           C
ATOM   1995  CG  ASN A 422     319.093 334.690 273.453  1.00 73.41           C
ATOM   1996  OD1 ASN A 422     318.678 335.826 273.226  1.00 73.41           O
ATOM   1997  ND2 ASN A 422     318.282 333.662 273.668  1.00 73.41           N
ATOM   1998  N   SER A 423     320.521 337.625 274.097  1.00 73.41           N
ATOM   1999  CA  SER A 423     320.455 338.676 275.105  1.00 73.41           C
ATOM   2000  C   SER A 423     319.771 338.220 276.389  1.00 73.41           C
ATOM   2001  O   SER A 423     320.067 338.763 277.457  1.00 73.41           O
ATOM   2002  CB  SER A 423     319.738 339.900 274.538  1.00 73.41           C
ATOM   2003  OG  SER A 423     320.492 340.493 273.497  1.00 73.41           O
ATOM   2004  N   HIS A 424     318.871 337.240 276.310  1.00 73.41           N
ATOM   2005  CA  HIS A 424     318.100 336.792 277.471  1.00 73.41           C
ATOM   2006  C   HIS A 424     318.806 335.610 278.137  1.00 73.41           C
ATOM   2007  O   HIS A 424     318.317 334.479 278.167  1.00 73.41           O
ATOM   2008  CB  HIS A 424     316.685 336.418 277.052  1.00 73.41           C
ATOM   2009  CG  HIS A 424     315.924 337.537 276.414  1.00 73.41           C
ATOM   2010  ND1 HIS A 424     315.381 338.577 277.136  1.00 73.41           N
ATOM   2011  CD2 HIS A 424     315.610 337.775 275.119  1.00 73.41           C
ATOM   2012  CE1 HIS A 424     314.767 339.407 276.314  1.00 73.41           C
ATOM   2013  NE2 HIS A 424     314.892 338.945 275.083  1.00 73.41           N
ATOM   2014  N   VAL A 425     319.983 335.890 278.692  1.00 73.41           N
ATOM   2015  CA  VAL A 425     320.808 334.861 279.317  1.00 73.41           C
ATOM   2016  C   VAL A 425     321.187 335.310 280.721  1.00 73.41           C
ATOM   2017  O   VAL A 425     321.672 336.433 280.911  1.00 73.41           O
ATOM   2018  CB  VAL A 425     322.065 334.540 278.489  1.00 73.41           C
ATOM   2019  CG1 VAL A 425     321.686 333.804 277.215  1.00 73.41           C
ATOM   2020  CG2 VAL A 425     322.839 335.808 278.157  1.00 73.41           C
ATOM   2021  N   MET A 426     320.954 334.444 281.703  1.00 73.41           N
ATOM   2022  CA  MET A 426     321.497 334.654 283.034  1.00 73.41           C
ATOM   2023  C   MET A 426     323.007 334.457 282.997  1.00 73.41           C
ATOM   2024  O   MET A 426     323.505 333.426 282.543  1.00 73.41           O
ATOM   2025  CB  MET A 426     320.878 333.670 284.029  1.00 73.41           C
ATOM   2026  CG  MET A 426     319.465 333.209 283.684  1.00 73.41           C
ATOM   2027  SD  MET A 426     318.205 334.494 283.717  1.00 73.41           S
ATOM   2028  CE  MET A 426     318.482 335.164 285.350  1.00 73.41           C
ATOM   2029  N   THR A 427     323.744 335.453 283.481  1.00 73.41           N
ATOM   2030  CA  THR A 427     325.201 335.393 283.418  1.00 73.41           C
ATOM   2031  C   THR A 427     325.775 335.603 284.807  1.00 73.41           C
ATOM   2032  O   THR A 427     325.507 336.627 285.440  1.00 73.41           O
ATOM   2033  CB  THR A 427     325.764 336.436 282.450  1.00 73.41           C
ATOM   2034  OG1 THR A 427     325.386 337.746 282.881  1.00 73.41           O
ATOM   2035  CG2 THR A 427     325.239 336.201 281.045  1.00 73.41           C
ATOM   2036  N   ILE A 428     326.560 334.640 285.271  1.00 73.41           N
ATOM   2037  CA  ILE A 428     327.316 334.758 286.509  1.00 73.41           C
ATOM   2038  C   ILE A 428     328.793 334.714 286.164  1.00 73.41           C
ATOM   2039  O   ILE A 428     329.198 334.122 285.158  1.00 76.91           O
ATOM   2040  CB  ILE A 428     326.942 333.644 287.505  1.00 76.91           C
ATOM   2041  CG1 ILE A 428     327.280 332.284 286.913  1.00 76.91           C
ATOM   2042  CG2 ILE A 428     325.465 333.706 287.839  1.00 76.91           C
ATOM   2043  CD1 ILE A 428     327.193 331.166 287.898  1.00 81.59           C
ATOM   2044  N   SER A 429     329.613 335.350 286.994  1.00 81.59           N
ATOM   2045  CA  SER A 429     331.035 335.455 286.710  1.00 81.59           C
ATOM   2046  C   SER A 429     331.836 335.155 287.966  1.00 81.59           C
ATOM   2047  O   SER A 429     331.285 334.935 289.046  1.00 81.59           O
ATOM   2048  CB  SER A 429     331.404 336.842 286.179  1.00 81.59           C
ATOM   2049  OG  SER A 429     331.423 337.795 287.223  1.00 85.64           O
ATOM   2050  N   SER A 430     333.155 335.140 287.801  1.00 85.64           N
ATOM   2051  CA  SER A 430     334.080 334.960 288.909  1.00 85.64           C
ATOM   2052  C   SER A 430     335.427 335.529 288.500  1.00 85.64           C
ATOM   2053  O   SER A 430     335.751 335.569 287.313  1.00 85.64           O
ATOM   2054  CB  SER A 430     334.224 333.487 289.292  1.00 85.64           C
ATOM   2055  OG  SER A 430     334.869 332.771 288.257  1.00 85.64           O
ATOM   2056  N   ILE A 431     336.203 335.968 289.482  1.00 90.60           N
ATOM   2057  CA  ILE A 431     337.523 336.543 289.240  1.00 90.60           C
ATOM   2058  C   ILE A 431     338.560 335.675 289.937  1.00 90.60           C
ATOM   2059  O   ILE A 431     338.428 335.369 291.129  1.00 90.60           O
ATOM   2060  CB  ILE A 431     337.595 337.998 289.723  1.00 90.60           C
ATOM   2061  CG1 ILE A 431     336.563 338.849 288.987  1.00 90.60           C
ATOM   2062  CG2 ILE A 431     338.986 338.559 289.522  1.00 90.60           C
ATOM   2063  CD1 ILE A 431     336.481 340.265 289.486  1.00 90.60           C
ATOM   2064  N   GLY A 432     339.592 335.275 289.197  1.00 97.31           N
ATOM   2065  CA  GLY A 432     340.649 334.449 289.734  1.00 97.31           C
ATOM   2066  C   GLY A 432     341.929 335.232 289.988  1.00 97.31           C
ATOM   2067  O   GLY A 432     342.064 336.396 289.626  1.00 97.31           O
ATOM   2068  N   SER A 433     342.875 334.556 290.630  1.00 97.31           N
ATOM   2069  CA  SER A 433     344.151 335.176 290.941  1.00 97.31           C
ATOM   2070  C   SER A 433     344.954 335.408 289.662  1.00 97.31           C
ATOM   2071  O   SER A 433     344.844 334.640 288.703  1.00 97.31           O
ATOM   2072  CB  SER A 433     344.948 334.303 291.905  1.00 96.71           C
ATOM   2073  OG  SER A 433     344.515 334.496 293.238  1.00 96.71           O
ATOM   2074  N   PRO A 434     345.765 336.465 289.621  1.00 96.71           N
ATOM   2075  CA  PRO A 434     346.547 336.748 288.412  1.00 96.71           C
ATOM   2076  C   PRO A 434     347.501 335.612 288.077  1.00 96.71           C
ATOM   2077  O   PRO A 434     348.083 334.981 288.961  1.00 96.71           O
ATOM   2078  CB  PRO A 434     347.304 338.033 288.771  1.00 96.71           C
ATOM   2079  CG  PRO A 434     347.280 338.092 290.263  1.00 96.71           C
ATOM   2080  CD  PRO A 434     345.978 337.480 290.665  1.00 99.41           C
ATOM   2081  N   ILE A 435     347.651 335.355 286.777  1.00 99.41           N
ATOM   2082  CA  ILE A 435     348.493 334.281 286.270  1.00 99.41           C
ATOM   2083  C   ILE A 435     349.352 334.825 285.137  1.00 99.41           C
ATOM   2084  O   ILE A 435     349.089 335.894 284.581  1.00 99.41           O
ATOM   2085  CB  ILE A 435     347.671 333.073 285.777  1.00 99.41           C
ATOM   2086  CG1 ILE A 435     346.769 333.488 284.615  1.00 99.41           C
ATOM   2087  CG2 ILE A 435     346.855 332.477 286.914  1.00 99.41           C
ATOM   2088  CD1 ILE A 435     346.111 332.328 283.917  1.00 99.58           C
ATOM   2089  N   ASP A 436     350.388 334.066 284.795  1.00 99.58           N
ATOM   2090  CA  ASP A 436     351.306 334.419 283.720  1.00 99.58           C
ATOM   2091  C   ASP A 436     350.776 333.883 282.399  1.00 99.58           C
ATOM   2092  O   ASP A 436     350.446 332.698 282.293  1.00 99.58           O
ATOM   2093  CB  ASP A 436     352.705 333.864 283.990  1.00 99.58           C
ATOM   2094  CG  ASP A 436     353.406 334.577 285.125  1.00 99.58           C
ATOM   2095  OD1 ASP A 436     353.133 335.776 285.333  1.00 95.91           O
ATOM   2096  OD2 ASP A 436     354.234 333.939 285.811  1.00 95.91           O
ATOM   2097  N   ILE A 437     350.702 334.755 281.396  1.00 95.91           N
ATOM   2098  CA  ILE A 437     350.227 334.393 280.068  1.00 95.91           C
ATOM   2099  C   ILE A 437     351.249 334.879 279.047  1.00 95.91           C
ATOM   2100  O   ILE A 437     351.909 335.904 279.256  1.00 95.91           O
ATOM   2101  CB  ILE A 437     348.827 334.979 279.785  1.00 89.79           C
ATOM   2102  CG1 ILE A 437     348.229 334.375 278.514  1.00 89.79           C
ATOM   2103  CG2 ILE A 437     348.881 336.490 279.679  1.00 89.79           C
ATOM   2104  CD1 ILE A 437     346.759 334.662 278.345  1.00 89.79           C
ATOM   2105  N   ASP A 438     351.396 334.132 277.956  1.00 89.79           N
ATOM   2106  CA  ASP A 438     352.415 334.429 276.957  1.00 89.79           C
ATOM   2107  C   ASP A 438     351.875 335.430 275.944  1.00 89.79           C
ATOM   2108  O   ASP A 438     350.850 335.181 275.302  1.00 89.79           O
ATOM   2109  CB  ASP A 438     352.865 333.148 276.261  1.00 89.79           C
ATOM   2110  CG  ASP A 438     354.282 333.237 275.732  1.00 87.18           C
ATOM   2111  OD1 ASP A 438     354.626 334.272 275.123  1.00 87.18           O
ATOM   2112  OD2 ASP A 438     355.051 332.274 275.927  1.00 87.18           O
ATOM   2113  N   THR A 439     352.573 336.559 275.792  1.00 87.18           N
ATOM   2114  CA  THR A 439     352.138 337.572 274.836  1.00 87.18           C
ATOM   2115  C   THR A 439     352.554 337.211 273.416  1.00 87.18           C
ATOM   2116  O   THR A 439     352.052 337.794 272.448  1.00 87.18           O
ATOM   2117  CB  THR A 439     352.698 338.940 275.222  1.00 87.18           C
ATOM   2118  OG1 THR A 439     354.128 338.911 275.144  1.00 87.18           O
ATOM   2119  CG2 THR A 439     352.280 339.303 276.634  1.00 87.18           C
ATOM   2120  N   SER A 440     353.485 336.266 273.269  1.00 87.18           N
ATOM   2121  CA  SER A 440     353.876 335.831 271.932  1.00 87.18           C
ATOM   2122  C   SER A 440     352.707 335.178 271.208  1.00 89.37           C
ATOM   2123  O   SER A 440     352.496 335.415 270.012  1.00 89.37           O
ATOM   2124  CB  SER A 440     355.062 334.871 272.014  1.00 89.37           C
ATOM   2125  OG  SER A 440     354.624 333.544 272.238  1.00 89.37           O
ATOM   2126  N   GLU A 441     351.934 334.353 271.916  1.00 89.37           N
ATOM   2127  CA  GLU A 441     350.761 333.735 271.309  1.00 89.37           C
ATOM   2128  C   GLU A 441     349.633 334.742 271.127  1.00 89.37           C
ATOM   2129  O   GLU A 441     348.963 334.753 270.087  1.00 91.36           O
ATOM   2130  CB  GLU A 441     350.294 332.554 272.160  1.00 91.36           C
ATOM   2131  CG  GLU A 441     348.901 332.038 271.823  1.00 91.36           C
ATOM   2132  CD  GLU A 441     348.785 331.507 270.408  1.00 91.36           C
ATOM   2133  OE1 GLU A 441     349.818 331.113 269.826  1.00 91.36           O
ATOM   2134  OE2 GLU A 441     347.655 331.475 269.879  1.00 91.36           O
ATOM   2135  N   TYR A 442     349.409 335.597 272.122  1.00 91.36           N
ATOM   2136  CA  TYR A 442     348.300 336.531 272.107  1.00 91.36           C
ATOM   2137  C   TYR A 442     348.804 337.949 271.873  1.00 91.36           C
ATOM   2138  O   TYR A 442     349.445 338.525 272.762  1.00111.57           O
ATOM   2139  CB  TYR A 442     347.519 336.473 273.429  1.00111.57           C
ATOM   2140  CG  TYR A 442     346.723 335.204 273.631  1.00111.57           C
ATOM   2141  CD1 TYR A 442     345.542 334.986 272.938  1.00111.57           C
ATOM   2142  CD2 TYR A 442     347.143 334.231 274.529  1.00111.57           C
ATOM   2143  CE1 TYR A 442     344.807 333.832 273.123  1.00111.57           C
ATOM   2144  CE2 TYR A 442     346.413 333.075 274.721  1.00111.57           C
ATOM   2145  CZ  TYR A 442     345.245 332.881 274.015  1.00107.07           C
ATOM   2146  OH  TYR A 442     344.510 331.733 274.197  1.00107.07           O
ATOM   2147  N   PRO A 443     348.529 338.550 270.715  1.00107.07           N
ATOM   2148  CA  PRO A 443     348.929 339.946 270.494  1.00107.07           C
ATOM   2149  C   PRO A 443     348.040 340.919 271.254  1.00107.07           C
ATOM   2150  O   PRO A 443     347.060 341.434 270.708  1.00107.07           O
ATOM   2151  CB  PRO A 443     348.790 340.115 268.977  1.00107.07           C
ATOM   2152  CG  PRO A 443     347.751 339.123 268.588  1.00 99.96           C
ATOM   2153  CD  PRO A 443     347.901 337.954 269.523  1.00 99.96           C
ATOM   2154  N   GLU A 444     348.373 341.158 272.525  1.00 99.96           N
ATOM   2155  CA  GLU A 444     347.536 342.006 273.369  1.00 99.96           C
ATOM   2156  C   GLU A 444     347.460 343.430 272.836  1.00 99.96           C
ATOM   2157  O   GLU A 444     346.386 344.042 272.836  1.00 99.96           O
ATOM   2158  CB  GLU A 444     348.068 342.004 274.802  1.00 99.96           C
ATOM   2159  CG  GLU A 444     348.131 340.630 275.442  1.00 99.96           C
ATOM   2160  CD  GLU A 444     346.765 340.012 275.628  1.00 93.82           C
ATOM   2161  OE1 GLU A 444     345.792 340.762 275.846  1.00 93.82           O
ATOM   2162  OE2 GLU A 444     346.662 338.770 275.557  1.00 93.82           O
ATOM   2163  N   LEU A 445     348.589 343.973 272.384  1.00 93.82           N
ATOM   2164  CA  LEU A 445     348.672 345.350 271.894  1.00 93.82           C
ATOM   2165  C   LEU A 445     348.207 346.355 272.946  1.00 93.82           C
ATOM   2166  O   LEU A 445     348.297 346.100 274.147  1.00 93.82           O
ATOM   2167  CB  LEU A 445     347.855 345.526 270.612  1.00 93.82           C
ATOM   2168  CG  LEU A 445     348.327 344.753 269.380  1.00 93.82           C
ATOM   2169  CD1 LEU A 445     347.372 344.972 268.218  1.00 85.24           C
ATOM   2170  CD2 LEU A 445     349.743 345.154 269.004  1.00 85.24           C
ATOM   2171  N   HIS A 458     356.624 344.592 273.962  1.00 85.24           N
ATOM   2172  CA  HIS A 458     355.872 343.501 274.571  1.00 85.24           C
ATOM   2173  C   HIS A 458     356.794 342.430 275.139  1.00 85.24           C
ATOM   2174  O   HIS A 458     357.490 341.738 274.397  1.00 85.24           O
ATOM   2175  CB  HIS A 458     354.917 342.875 273.554  1.00 85.24           C
ATOM   2176  CG  HIS A 458     353.686 343.686 273.299  1.00 85.24           C
ATOM   2177  ND1 HIS A 458     353.663 344.749 272.423  1.00 76.05           N
ATOM   2178  CD2 HIS A 458     352.435 343.589 273.807  1.00 76.05           C
ATOM   2179  CE1 HIS A 458     352.450 345.272 272.402  1.00 76.05           C
ATOM   2180  NE2 HIS A 458     351.686 344.587 273.232  1.00 76.05           N
ATOM   2181  N   GLU A 459     356.796 342.303 276.463  1.00 76.05           N
ATOM   2182  CA  GLU A 459     357.567 341.267 277.136  1.00 76.05           C
ATOM   2183  C   GLU A 459     356.938 339.904 276.860  1.00 76.05           C
ATOM   2184  O   GLU A 459     355.741 339.828 276.560  1.00 73.41           O
ATOM   2185  CB  GLU A 459     357.640 341.547 278.637  1.00 73.41           C
ATOM   2186  CG  GLU A 459     358.438 342.791 278.987  1.00 73.41           C
ATOM   2187  CD  GLU A 459     358.479 343.059 280.477  1.00 73.41           C
ATOM   2188  OE1 GLU A 459     357.716 342.407 281.221  1.00 73.41           O
ATOM   2189  OE2 GLU A 459     359.274 343.921 280.905  1.00 73.41           O
ATOM   2190  N   PRO A 460     357.710 338.814 276.932  1.00 73.41           N
ATOM   2191  CA  PRO A 460     357.136 337.496 276.605  1.00 73.41           C
ATOM   2192  C   PRO A 460     355.955 337.101 277.474  1.00 73.41           C
ATOM   2193  O   PRO A 460     355.019 336.464 276.978  1.00 73.41           O
ATOM   2194  CB  PRO A 460     358.328 336.544 276.796  1.00 73.41           C
ATOM   2195  CG  PRO A 460     359.277 337.289 277.684  1.00 73.41           C
ATOM   2196  CD  PRO A 460     359.138 338.719 277.275  1.00 73.41           C
ATOM   2197  N   VAL A 461     355.968 337.454 278.757  1.00 73.41           N
ATOM   2198  CA  VAL A 461     354.901 337.090 279.680  1.00 73.41           C
ATOM   2199  C   VAL A 461     354.531 338.298 280.527  1.00 73.41           C
ATOM   2200  O   VAL A 461     355.400 339.084 280.923  1.00 73.41           O
ATOM   2201  CB  VAL A 461     355.296 335.898 280.578  1.00 73.41           C
ATOM   2202  CG1 VAL A 461     355.330 334.613 279.768  1.00 73.41           C
ATOM   2203  CG2 VAL A 461     356.642 336.149 281.238  1.00 73.41           C
ATOM   2204  N   MET A 462     353.236 338.454 280.790  1.00 73.41           N
ATOM   2205  CA  MET A 462     352.715 339.496 281.663  1.00 73.41           C
ATOM   2206  C   MET A 462     351.527 338.939 282.438  1.00 73.41           C
ATOM   2207  O   MET A 462     350.751 338.142 281.901  1.00 73.41           O
ATOM   2208  CB  MET A 462     352.322 340.742 280.861  1.00 73.41           C
ATOM   2209  CG  MET A 462     350.924 340.718 280.266  1.00 73.41           C
ATOM   2210  SD  MET A 462     350.691 341.978 278.996  1.00 73.41           S
ATOM   2211  CE  MET A 462     351.289 343.433 279.849  1.00 73.41           C
ATOM   2212  N   LYS A 463     351.411 339.329 283.706  1.00 73.41           N
ATOM   2213  CA  LYS A 463     350.313 338.848 284.532  1.00 73.41           C
ATOM   2214  C   LYS A 463     349.000 339.489 284.107  1.00 73.41           C
ATOM   2215  O   LYS A 463     348.899 340.713 283.979  1.00 73.41           O
ATOM   2216  CB  LYS A 463     350.583 339.144 286.007  1.00 73.41           C
ATOM   2217  CG  LYS A 463     351.685 338.306 286.620  1.00 73.41           C
ATOM   2218  CD  LYS A 463     351.848 338.611 288.099  1.00 73.41           C
ATOM   2219  CE  LYS A 463     352.897 337.716 288.735  1.00 73.41           C
ATOM   2220  NZ  LYS A 463     353.053 337.999 290.188  1.00 73.41           N
ATOM   2221  N   MET A 464     347.984 338.654 283.891  1.00 73.41           N
ATOM   2222  CA  MET A 464     346.655 339.128 283.511  1.00 73.41           C
ATOM   2223  C   MET A 464     345.625 338.352 284.319  1.00 73.41           C
ATOM   2224  O   MET A 464     345.686 337.121 284.387  1.00 73.41           O
ATOM   2225  CB  MET A 464     346.406 338.970 282.010  1.00 73.41           C
ATOM   2226  CG  MET A 464     347.526 339.503 281.135  1.00 73.41           C
ATOM   2227  SD  MET A 464     347.253 341.195 280.583  1.00 73.41           S
ATOM   2228  CE  MET A 464     345.759 341.002 279.620  1.00 73.41           C
ATOM   2229  N   VAL A 465     344.686 339.071 284.922  1.00 73.41           N
ATOM   2230  CA  VAL A 465     343.660 338.478 285.772  1.00 73.41           C
ATOM   2231  C   VAL A 465     342.674 337.697 284.911  1.00 73.41           C
ATOM   2232  O   VAL A 465     342.038 338.278 284.023  1.00 73.41           O
ATOM   2233  CB  VAL A 465     342.932 339.557 286.588  1.00 73.41           C
ATOM   2234  CG1 VAL A 465     341.985 338.919 287.579  1.00 73.41           C
ATOM   2235  CG2 VAL A 465     343.936 340.453 287.290  1.00 73.41           C
ATOM   2236  N   PRO A 466     342.510 336.398 285.134  1.00 73.41           N
ATOM   2237  CA  PRO A 466     341.551 335.630 284.338  1.00 73.41           C
ATOM   2238  C   PRO A 466     340.149 335.711 284.917  1.00 73.41           C
ATOM   2239  O   PRO A 466     339.949 335.633 286.132  1.00 73.41           O
ATOM   2240  CB  PRO A 466     342.100 334.202 284.420  1.00 73.41           C
ATOM   2241  CG  PRO A 466     342.752 334.154 285.757  1.00 73.41           C
ATOM   2242  CD  PRO A 466     343.297 335.537 286.032  1.00 73.41           C
ATOM   2243  N   ILE A 467     339.173 335.864 284.030  1.00 73.41           N
ATOM   2244  CA  ILE A 467     337.770 335.972 284.406  1.00 73.41           C
ATOM   2245  C   ILE A 467     336.995 334.890 283.677  1.00 73.41           C
ATOM   2246  O   ILE A 467     337.147 334.723 282.461  1.00 73.41           O
ATOM   2247  CB  ILE A 467     337.189 337.358 284.075  1.00 73.41           C
ATOM   2248  CG1 ILE A 467     338.123 338.465 284.556  1.00 73.41           C
ATOM   2249  CG2 ILE A 467     335.812 337.515 284.685  1.00 73.41           C
ATOM   2250  CD1 ILE A 467     337.699 339.833 284.105  1.00 73.41           C
ATOM   2251  N   ARG A 468     336.170 334.158 284.411  1.00 73.41           N
ATOM   2252  CA  ARG A 468     335.336 333.108 283.854  1.00 73.41           C
ATOM   2253  C   ARG A 468     333.880 333.527 283.954  1.00 73.41           C
ATOM   2254  O   ARG A 468     333.386 333.819 285.046  1.00 73.41           O
ATOM   2255  CB  ARG A 468     335.557 331.793 284.599  1.00 73.41           C
ATOM   2256  CG  ARG A 468     336.983 331.292 284.525  1.00 73.41           C
ATOM   2257  CD  ARG A 468     337.197 330.075 285.400  1.00 73.41           C
ATOM   2258  NE  ARG A 468     338.591 329.650 285.374  1.00 73.41           N
ATOM   2259  CZ  ARG A 468     339.531 330.133 286.175  1.00 73.41           C
ATOM   2260  NH1 ARG A 468     339.259 331.059 287.079  1.00 73.41           N
ATOM   2261  NH2 ARG A 468     340.776 329.680 286.062  1.00 73.41           N
ATOM   2262  N   VAL A 469     333.194 333.560 282.817  1.00 73.41           N
ATOM   2263  CA  VAL A 469     331.804 333.994 282.758  1.00 73.41           C
ATOM   2264  C   VAL A 469     330.990 332.883 282.121  1.00 73.41           C
ATOM   2265  O   VAL A 469     331.076 332.666 280.909  1.00 73.41           O
ATOM   2266  CB  VAL A 469     331.643 335.300 281.965  1.00 73.41           C
ATOM   2267  CG1 VAL A 469     330.181 335.680 281.873  1.00 73.41           C
ATOM   2268  CG2 VAL A 469     332.446 336.413 282.610  1.00 73.41           C
ATOM   2269  N   VAL A 470     330.198 332.188 282.928  1.00 73.41           N
ATOM   2270  CA  VAL A 470     329.327 331.127 282.443  1.00 73.41           C
ATOM   2271  C   VAL A 470     327.907 331.668 282.380  1.00 73.41           C
ATOM   2272  O   VAL A 470     327.445 332.336 283.310  1.00 73.41           O
ATOM   2273  CB  VAL A 470     329.406 329.862 283.322  1.00 73.41           C
ATOM   2274  CG1 VAL A 470     330.778 329.240 283.218  1.00 73.41           C
ATOM   2275  CG2 VAL A 470     329.121 330.195 284.761  1.00 73.41           C
ATOM   2276  N   SER A 471     327.227 331.400 281.274  1.00 73.41           N
ATOM   2277  CA  SER A 471     325.893 331.925 281.040  1.00 73.41           C
ATOM   2278  C   SER A 471     324.869 330.799 281.030  1.00 73.41           C
ATOM   2279  O   SER A 471     325.209 329.618 280.943  1.00 73.41           O
ATOM   2280  CB  SER A 471     325.840 332.703 279.725  1.00 73.41           C
ATOM   2281  OG  SER A 471     326.254 331.892 278.644  1.00 73.41           O
ATOM   2282  N   SER A 472     323.596 331.188 281.121  1.00 73.41           N
ATOM   2283  CA  SER A 472     322.527 330.203 281.257  1.00 73.41           C
ATOM   2284  C   SER A 472     322.280 329.451 279.956  1.00 73.41           C
ATOM   2285  O   SER A 472     321.574 328.437 279.947  1.00 73.41           O
ATOM   2286  CB  SER A 472     321.244 330.883 281.725  1.00 73.41           C
ATOM   2287  OG  SER A 472     320.774 331.807 280.760  1.00 73.41           O
ATOM   2288  N   ASP A 473     322.826 329.940 278.844  1.00 73.41           N
ATOM   2289  CA  ASP A 473     322.709 329.194 277.598  1.00 73.41           C
ATOM   2290  C   ASP A 473     323.621 327.978 277.563  1.00 73.41           C
ATOM   2291  O   ASP A 473     323.398 327.079 276.746  1.00 73.41           O
ATOM   2292  CB  ASP A 473     322.990 330.113 276.407  1.00 73.41           C
ATOM   2293  CG  ASP A 473     324.392 330.679 276.424  1.00 73.41           C
ATOM   2294  OD1 ASP A 473     325.081 330.534 277.454  1.00 73.41           O
ATOM   2295  OD2 ASP A 473     324.803 331.282 275.411  1.00 73.41           O
ATOM   2296  N   GLY A 474     324.628 327.924 278.428  1.00 73.41           N
ATOM   2297  CA  GLY A 474     325.494 326.776 278.537  1.00 73.41           C
ATOM   2298  C   GLY A 474     326.884 326.941 277.973  1.00 73.41           C
ATOM   2299  O   GLY A 474     327.428 325.975 277.430  1.00 73.41           O
ATOM   2300  N   TYR A 475     327.478 328.125 278.077  1.00 73.41           N
ATOM   2301  CA  TYR A 475     328.811 328.376 277.551  1.00 73.41           C
ATOM   2302  C   TYR A 475     329.669 329.065 278.597  1.00 73.41           C
ATOM   2303  O   TYR A 475     329.167 329.803 279.447  1.00 73.41           O
ATOM   2304  CB  TYR A 475     328.769 329.237 276.284  1.00 73.41           C
ATOM   2305  CG  TYR A 475     328.057 328.598 275.113  1.00 73.41           C
ATOM   2306  CD1 TYR A 475     326.672 328.544 275.060  1.00 79.97           C
ATOM   2307  CD2 TYR A 475     328.771 328.052 274.057  1.00 79.97           C
ATOM   2308  CE1 TYR A 475     326.019 327.964 273.995  1.00 79.97           C
ATOM   2309  CE2 TYR A 475     328.128 327.469 272.989  1.00 79.97           C
ATOM   2310  CZ  TYR A 475     326.752 327.429 272.963  1.00 79.97           C
ATOM   2311  OH  TYR A 475     326.099 326.850 271.901  1.00 79.97           O
ATOM   2312  N   LEU A 476     330.973 328.821 278.524  1.00 79.97           N
ATOM   2313  CA  LEU A 476     331.949 329.458 279.391  1.00 79.97           C
ATOM   2314  C   LEU A 476     332.826 330.377 278.557  1.00 84.71           C
ATOM   2315  O   LEU A 476     333.253 330.012 277.460  1.00 84.71           O
ATOM   2316  CB  LEU A 476     332.805 328.412 280.113  1.00 84.71           C
ATOM   2317  CG  LEU A 476     333.781 328.866 281.202  1.00 84.71           C
ATOM   2318  CD1 LEU A 476     334.001 327.739 282.193  1.00 84.71           C
ATOM   2319  CD2 LEU A 476     335.111 329.315 280.629  1.00 84.71           C
ATOM   2320  N   TYR A 477     333.097 331.566 279.080  1.00 84.71           N
ATOM   2321  CA  TYR A 477     333.909 332.559 278.396  1.00 84.71           C
ATOM   2322  C   TYR A 477     335.065 332.968 279.295  1.00 83.84           C
ATOM   2323  O   TYR A 477     334.861 333.298 280.466  1.00 83.84           O
ATOM   2324  CB  TYR A 477     333.085 333.799 278.012  1.00 83.84           C
ATOM   2325  CG  TYR A 477     331.805 333.509 277.258  1.00 83.84           C
ATOM   2326  CD1 TYR A 477     330.701 332.979 277.905  1.00 83.84           C
ATOM   2327  CD2 TYR A 477     331.698 333.773 275.905  1.00 83.84           C
ATOM   2328  CE1 TYR A 477     329.534 332.715 277.231  1.00 83.84           C
ATOM   2329  CE2 TYR A 477     330.533 333.511 275.219  1.00 86.13           C
ATOM   2330  CZ  TYR A 477     329.454 332.981 275.889  1.00 86.13           C
ATOM   2331  OH  TYR A 477     328.285 332.713 275.222  1.00 86.13           O
ATOM   2332  N   ASN A 478     336.274 332.952 278.746  1.00 86.13           N
ATOM   2333  CA  ASN A 478     337.478 333.294 279.490  1.00 86.13           C
ATOM   2334  C   ASN A 478     337.987 334.646 279.015  1.00 86.13           C
ATOM   2335  O   ASN A 478     338.205 334.842 277.817  1.00 86.13           O
ATOM   2336  CB  ASN A 478     338.554 332.225 279.310  1.00 86.13           C
ATOM   2337  CG  ASN A 478     338.195 330.918 279.984  1.00 86.13           C
ATOM   2338  OD1 ASN A 478     338.052 330.852 281.202  1.00 84.91           O
ATOM   2339  ND2 ASN A 478     338.046 329.867 279.191  1.00 84.91           N
ATOM   2340  N   PHE A 479     338.179 335.569 279.951  1.00 84.91           N
ATOM   2341  CA  PHE A 479     338.689 336.898 279.657  1.00 84.91           C
ATOM   2342  C   PHE A 479     339.974 337.145 280.432  1.00 84.91           C
ATOM   2343  O   PHE A 479     340.356 336.370 281.311  1.00 84.91           O
ATOM   2344  CB  PHE A 479     337.666 337.982 280.008  1.00 84.91           C
ATOM   2345  CG  PHE A 479     336.415 337.946 279.174  1.00 84.91           C
ATOM   2346  CD1 PHE A 479     335.428 337.009 279.417  1.00 84.91           C
ATOM   2347  CD2 PHE A 479     336.221 338.864 278.160  1.00 84.91           C
ATOM   2348  CE1 PHE A 479     334.279 336.983 278.656  1.00 84.91           C
ATOM   2349  CE2 PHE A 479     335.073 338.838 277.397  1.00 79.49           C
ATOM   2350  CZ  PHE A 479     334.102 337.897 277.646  1.00 79.49           C
ATOM   2351  N   VAL A 480     340.644 338.241 280.089  1.00 79.49           N
ATOM   2352  CA  VAL A 480     341.812 338.715 280.820  1.00 79.49           C
ATOM   2353  C   VAL A 480     341.716 340.225 280.977  1.00 73.71           C
ATOM   2354  O   VAL A 480     341.303 340.935 280.054  1.00 73.71           O
ATOM   2355  CB  VAL A 480     343.135 338.326 280.125  1.00 73.71           C
ATOM   2356  CG1 VAL A 480     343.370 336.840 280.236  1.00 73.71           C
ATOM   2357  CG2 VAL A 480     343.108 338.739 278.665  1.00 73.41           C
ATOM   2358  N   MET A 481     342.088 340.713 282.155  1.00 73.41           N
ATOM   2359  CA  MET A 481     342.104 342.136 282.453  1.00 73.41           C
ATOM   2360  C   MET A 481     343.510 342.563 282.841  1.00 73.41           C
ATOM   2361  O   MET A 481     344.181 341.884 283.623  1.00 73.41           O
ATOM   2362  CB  MET A 481     341.142 342.473 283.592  1.00 73.41           C
ATOM   2363  CG  MET A 481     341.091 343.956 283.933  1.00 73.41           C
ATOM   2364  SD  MET A 481     340.075 344.322 285.374  1.00 73.41           S
ATOM   2365  CE  MET A 481     339.025 342.879 285.404  1.00 73.41           C
ATOM   2366  N   ASP A 482     343.951 343.692 282.300  1.00 73.41           N
ATOM   2367  CA  ASP A 482     345.225 344.264 282.704  1.00 73.41           C
ATOM   2368  C   ASP A 482     344.991 345.178 283.898  1.00 73.41           C
ATOM   2369  O   ASP A 482     344.290 346.189 283.760  1.00 73.41           O
ATOM   2370  CB  ASP A 482     345.854 345.039 281.558  1.00 73.41           C
ATOM   2371  CG  ASP A 482     347.346 345.239 281.740  1.00 73.41           C
ATOM   2372  OD1 ASP A 482     347.903 344.727 282.733  1.00 73.41           O
ATOM   2373  OD2 ASP A 482     347.962 345.907 280.885  1.00 73.41           O
ATOM   2374  N   PRO A 483     345.544 344.869 285.074  1.00 73.41           N
ATOM   2375  CA  PRO A 483     345.205 345.647 286.278  1.00 73.41           C
ATOM   2376  C   PRO A 483     345.521 347.128 286.173  1.00 73.41           C
ATOM   2377  O   PRO A 483     344.831 347.937 286.805  1.00 73.41           O
ATOM   2378  CB  PRO A 483     346.038 344.975 287.375  1.00 73.41           C
ATOM   2379  CG  PRO A 483     346.268 343.592 286.874  1.00 73.41           C
ATOM   2380  CD  PRO A 483     346.395 343.711 285.386  1.00 73.41           C
ATOM   2381  N   GLU A 484     346.539 347.516 285.410  1.00 73.41           N
ATOM   2382  CA  GLU A 484     346.842 348.927 285.210  1.00 73.41           C
ATOM   2383  C   GLU A 484     346.011 349.542 284.094  1.00 73.41           C
ATOM   2384  O   GLU A 484     345.392 350.593 284.287  1.00 73.41           O
ATOM   2385  CB  GLU A 484     348.335 349.115 284.913  1.00 73.41           C
ATOM   2386  CG  GLU A 484     349.259 348.545 285.976  1.00 73.41           C
ATOM   2387  CD  GLU A 484     349.941 347.267 285.535  1.00 73.41           C
ATOM   2388  OE1 GLU A 484     349.667 346.805 284.410  1.00 73.41           O
ATOM   2389  OE2 GLU A 484     350.755 346.728 286.314  1.00 73.41           O
ATOM   2390  N   ARG A 485     345.982 348.906 282.923  1.00 73.41           N
ATOM   2391  CA  ARG A 485     345.222 349.452 281.803  1.00 73.41           C
ATOM   2392  C   ARG A 485     343.721 349.331 282.035  1.00 73.41           C
ATOM   2393  O   ARG A 485     342.971 350.281 281.781  1.00 73.41           O
ATOM   2394  CB  ARG A 485     345.623 348.750 280.505  1.00 73.41           C
ATOM   2395  CG  ARG A 485     346.807 349.382 279.787  1.00 73.41           C
ATOM   2396  CD  ARG A 485     348.109 349.180 280.547  1.00 73.41           C
ATOM   2397  NE  ARG A 485     349.259 349.657 279.787  1.00 73.41           N
ATOM   2398  CZ  ARG A 485     350.496 349.722 280.260  1.00 73.41           C
ATOM   2399  NH1 ARG A 485     350.785 349.344 281.494  1.00 73.41           N
ATOM   2400  NH2 ARG A 485     351.468 350.180 279.476  1.00 73.41           N
ATOM   2401  N   GLY A 486     343.267 348.186 282.531  1.00 73.41           N
ATOM   2402  CA  GLY A 486     341.837 347.958 282.634  1.00 73.41           C
ATOM   2403  C   GLY A 486     341.198 347.989 281.261  1.00 73.41           C
ATOM   2404  O   GLY A 486     341.688 347.372 280.309  1.00 73.41           O
ATOM   2405  N   GLY A 487     340.089 348.713 281.146  1.00 73.41           N
ATOM   2406  CA  GLY A 487     339.432 348.890 279.867  1.00 73.41           C
ATOM   2407  C   GLY A 487     338.733 347.635 279.387  1.00 73.41           C
ATOM   2408  O   GLY A 487     338.544 346.659 280.114  1.00 73.41           O
ATOM   2409  N   ASP A 488     338.338 347.672 278.115  1.00 73.41           N
ATOM   2410  CA  ASP A 488     337.677 346.531 277.497  1.00 73.41           C
ATOM   2411  C   ASP A 488     338.607 345.326 277.490  1.00 73.41           C
ATOM   2412  O   ASP A 488     339.792 345.441 277.165  1.00 73.41           O
ATOM   2413  CB  ASP A 488     337.247 346.887 276.072  1.00 73.41           C
ATOM   2414  CG  ASP A 488     336.250 345.899 275.495  1.00 73.41           C
ATOM   2415  OD1 ASP A 488     336.157 344.762 276.004  1.00 73.41           O
ATOM   2416  OD2 ASP A 488     335.554 346.263 274.524  1.00 73.41           O
ATOM   2417  N   CYS A 489     338.063 344.167 277.846  1.00 73.41           N
ATOM   2418  CA  CYS A 489     338.865 342.969 278.033  1.00 73.41           C
ATOM   2419  C   CYS A 489     338.793 342.051 276.820  1.00 73.41           C
ATOM   2420  O   CYS A 489     337.736 341.868 276.211  1.00 73.41           O
ATOM   2421  CB  CYS A 489     338.413 342.212 279.280  1.00 73.41           C
ATOM   2422  SG  CYS A 489     338.825 343.036 280.829  1.00 73.41           S
ATOM   2423  N   LEU A 490     339.940 341.475 276.478  1.00 73.41           N
ATOM   2424  CA  LEU A 490     340.039 340.504 275.400  1.00 73.41           C
ATOM   2425  C   LEU A 490     339.515 339.146 275.852  1.00 73.41           C
ATOM   2426  O   LEU A 490     339.586 338.791 277.032  1.00 73.41           O
ATOM   2427  CB  LEU A 490     341.493 340.391 274.937  1.00 73.41           C
ATOM   2428  CG  LEU A 490     341.857 339.571 273.699  1.00 73.41           C
ATOM   2429  CD1 LEU A 490     343.020 340.217 272.973  1.00 73.41           C
ATOM   2430  CD2 LEU A 490     342.226 338.161 274.101  1.00 73.41           C
ATOM   2431  N   ILE A 491     338.985 338.384 274.901  1.00 73.41           N
ATOM   2432  CA  ILE A 491     338.445 337.053 275.159  1.00 73.41           C
ATOM   2433  C   ILE A 491     339.406 336.046 274.541  1.00 73.41           C
ATOM   2434  O   ILE A 491     339.823 336.188 273.387  1.00 73.41           O
ATOM   2435  CB  ILE A 491     337.020 336.892 274.616  1.00 73.41           C
ATOM   2436  CG1 ILE A 491     336.527 335.463 274.842  1.00 73.41           C
ATOM   2437  CG2 ILE A 491     336.941 337.289 273.151  1.00 73.41           C
ATOM   2438  CD1 ILE A 491     335.051 335.284 274.617  1.00 73.41           C
ATOM   2439  N   LEU A 492     339.777 335.026 275.318  1.00 73.41           N
ATOM   2440  CA  LEU A 492     340.737 334.042 274.828  1.00 73.41           C
ATOM   2441  C   LEU A 492     340.039 332.843 274.201  1.00 73.41           C
ATOM   2442  O   LEU A 492     340.308 332.496 273.046  1.00 73.41           O
ATOM   2443  CB  LEU A 492     341.653 333.591 275.962  1.00 73.41           C
ATOM   2444  CG  LEU A 492     342.237 334.695 276.837  1.00 73.41           C
ATOM   2445  CD1 LEU A 492     342.749 334.106 278.124  1.00 73.41           C
ATOM   2446  CD2 LEU A 492     343.348 335.419 276.110  1.00 73.41           C
ATOM   2447  N   SER A 493     339.139 332.200 274.936  1.00 73.41           N
ATOM   2448  CA  SER A 493     338.574 330.945 274.468  1.00 73.41           C
ATOM   2449  C   SER A 493     337.139 330.812 274.947  1.00 73.41           C
ATOM   2450  O   SER A 493     336.686 331.539 275.832  1.00 73.41           O
ATOM   2451  CB  SER A 493     339.395 329.754 274.953  1.00 73.41           C
ATOM   2452  OG  SER A 493     339.309 329.634 276.360  1.00 73.41           O
ATOM   2453  N   GLN A 494     336.434 329.861 274.343  1.00 73.41           N
ATOM   2454  CA  GLN A 494     335.053 329.550 274.694  1.00 73.41           C
ATOM   2455  C   GLN A 494     334.861 328.044 274.636  1.00 73.41           C
ATOM   2456  O   GLN A 494     335.083 327.432 273.588  1.00 73.41           O
ATOM   2457  CB  GLN A 494     334.072 330.237 273.752  1.00 73.41           C
ATOM   2458  CG  GLN A 494     332.633 329.970 274.118  1.00 76.73           C
ATOM   2459  CD  GLN A 494     331.663 330.682 273.213  1.00 76.73           C
ATOM   2460  OE1 GLN A 494     330.450 330.592 273.397  1.00 76.73           O
ATOM   2461  NE2 GLN A 494     332.186 331.401 272.228  1.00 76.73           N
ATOM   2462  N   TYR A 495     334.440 327.448 275.742  1.00 76.73           N
ATOM   2463  CA  TYR A 495     334.015 326.059 275.744  1.00 76.73           C
ATOM   2464  C   TYR A 495     332.502 325.995 275.607  1.00 76.73           C
ATOM   2465  O   TYR A 495     331.833 326.991 275.326  1.00 81.87           O
ATOM   2466  CB  TYR A 495     334.469 325.340 277.017  1.00 81.87           C
ATOM   2467  CG  TYR A 495     335.826 325.770 277.507  1.00 81.87           C
ATOM   2468  CD1 TYR A 495     336.949 325.620 276.715  1.00 81.87           C
ATOM   2469  CD2 TYR A 495     335.985 326.302 278.775  1.00 81.87           C
ATOM   2470  CE1 TYR A 495     338.191 326.009 277.163  1.00 81.87           C
ATOM   2471  CE2 TYR A 495     337.221 326.692 279.231  1.00 81.87           C
ATOM   2472  CZ  TYR A 495     338.322 326.542 278.422  1.00 81.87           C
ATOM   2473  OH  TYR A 495     339.559 326.928 278.876  1.00 81.87           O
ATOM   2474  N   SER A 496     331.962 324.801 275.803  1.00 81.87           N
ATOM   2475  CA  SER A 496     330.524 324.585 275.796  1.00 81.87           C
ATOM   2476  C   SER A 496     330.184 323.552 276.854  1.00 81.87           C
ATOM   2477  O   SER A 496     330.419 322.355 276.654  1.00 81.87           O
ATOM   2478  CB  SER A 496     330.032 324.128 274.424  1.00 81.87           C
ATOM   2479  OG  SER A 496     328.639 323.875 274.452  1.00 81.87           O
ATOM   2480  N   ILE A 497     329.642 324.014 277.977  1.00 86.58           N
ATOM   2481  CA  ILE A 497     329.156 323.093 278.987  1.00 86.58           C
ATOM   2482  C   ILE A 497     328.003 322.277 278.406  1.00 86.58           C
ATOM   2483  O   ILE A 497     327.293 322.717 277.494  1.00 86.58           O
ATOM   2484  CB  ILE A 497     328.734 323.872 280.243  1.00 86.58           C
ATOM   2485  CG1 ILE A 497     329.920 324.670 280.781  1.00 86.58           C
ATOM   2486  CG2 ILE A 497     328.253 322.943 281.332  1.00 86.58           C
ATOM   2487  CD1 ILE A 497     331.090 323.821 281.188  1.00 86.58           C
ATOM   2488  N   LEU A 498     327.838 321.060 278.925  1.00 88.68           N
ATOM   2489  CA  LEU A 498     326.927 320.012 278.458  1.00 88.68           C
ATOM   2490  C   LEU A 498     327.483 319.322 277.217  1.00 88.68           C
ATOM   2491  O   LEU A 498     326.729 318.621 276.531  1.00 88.68           O
ATOM   2492  CB  LEU A 498     325.503 320.508 278.144  1.00 88.68           C
ATOM   2493  CG  LEU A 498     324.523 320.811 279.279  1.00 88.68           C
ATOM   2494  CD1 LEU A 498     324.975 322.008 280.095  1.00 88.68           C
ATOM   2495  CD2 LEU A 498     323.128 321.041 278.723  1.00 88.68           C
ATOM   2496  N   MET A 499     328.763 319.495 276.903  1.00 88.68           N
ATOM   2497  CA  MET A 499     329.357 318.871 275.729  1.00 88.68           C
ATOM   2498  C   MET A 499     330.857 318.679 275.921  1.00 88.68           C
ATOM   2499  O   MET A 499     331.595 319.640 276.137  1.00 78.11           O
ATOM   2500  CB  MET A 499     329.080 319.713 274.480  1.00 78.11           C
ATOM   2501  CG  MET A 499     329.830 319.265 273.235  1.00 78.11           C
ATOM   2502  SD  MET A 499     331.376 320.157 272.973  1.00 78.11           S
ATOM   2503  CE  MET A 499     330.823 321.490 271.914  1.00 78.11           C
TER
END
'''

if (__name__ == "__main__"):
  t0 = time.time()
  run()
  print("OK. Time: %8.3f"%(time.time()-t0))
