/*
// $Id: //open/dev/farrago/src/net/sf/farrago/namespace/FarragoMedLocalDataServer.java#23 $
// Farrago is an extensible data management system.
// Copyright (C) 2005-2009 The Eigenbase Project
// Copyright (C) 2005-2009 SQLstream, Inc.
// Copyright (C) 2005-2009 LucidEra, Inc.
// Portions Copyright (C) 2003-2009 John V. Sichi
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the Free
// Software Foundation; either version 2 of the License, or (at your option)
// any later version approved by The Eigenbase Project.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
package net.sf.farrago.namespace;

import java.sql.*;

import net.sf.farrago.fem.med.*;
import net.sf.farrago.fennel.*;

import org.eigenbase.rel.*;
import org.eigenbase.relopt.*;


/**
 * FarragoMedLocalDataServer represents a {@link FarragoMedDataServer} instance
 * originating from a {@link FarragoMedDataWrapper} managing local data. It
 * defines extra methods not relevant in the context of foreign data.
 *
 * @author John V. Sichi
 * @version $Id: //open/dev/farrago/src/net/sf/farrago/namespace/FarragoMedLocalDataServer.java#23 $
 */
public interface FarragoMedLocalDataServer
    extends FarragoMedDataServer
{
    //~ Methods ----------------------------------------------------------------

    /**
     * Sets the Fennel database handle to use for accessing local storage.
     *
     * @param fennelDbHandle the handle to use
     */
    public void setFennelDbHandle(FennelDbHandle fennelDbHandle);

    /**
     * Validates the definition of a table being created. May have side-effects
     * such as creating additional system-maintained indexes.
     *
     * @param table definition of the table being created
     * @param generatedPrimaryKeyIndex if non-null, a reference to an index
     * generated by the system to enforce the primary key
     */
    public void validateTableDefinition(
        FemLocalTable table,
        FemLocalIndex generatedPrimaryKeyIndex)
        throws SQLException;

    /**
     * Validates the definition of a table being created. May have side-effects
     * such as creating additional system-maintained indexes.
     *
     * @param table definition of the table being created
     * @param generatedPrimaryKeyIndex if non-null, a reference to an index
     * generated by the system to enforce the primary key
     * @param creation true if the table being validated is being newly created
     */
    public void validateTableDefinition(
        FemLocalTable table,
        FemLocalIndex generatedPrimaryKeyIndex,
        boolean creation)
        throws SQLException;

    /**
     * Creates an index.
     *
     * @param index definition of the index to create
     * @param txnContext Fennel txn context
     *
     * @return root PageId of index
     */
    public long createIndex(FemLocalIndex index, FennelTxnContext txnContext)
        throws SQLException;

    /**
     * Drops or truncates an index.
     *
     * @param index definition of the index to drop
     * @param rootPageId root PageId of index
     * @param truncate if true, only truncate storage; if false, drop storage
     * entirely
     * @param txnContext Fennel txn context
     */
    public void dropIndex(
        FemLocalIndex index,
        long rootPageId,
        boolean truncate,
        FennelTxnContext txnContext)
        throws SQLException;

    /**
     * Verifies an index and returns its page count.
     *
     * @param index definition of the index to verify
     * @param rootPageId root PageId of index
     * @param estimate whether to estimate statistics for a quicker result (and
     * at the cost of no unique key count)
     * @param txnContext Fennel txn context
     *
     * @return index statistics for the specified index
     */
    public FarragoMedLocalIndexStats computeIndexStats(
        FemLocalIndex index,
        long rootPageId,
        boolean estimate,
        FennelTxnContext txnContext)
        throws SQLException;

    /**
     * Creates a plan for loading existing rows of a table into an index.
     * Typically, the topmost node of this plan will be a {@link
     * net.sf.farrago.query.FarragoIndexBuilderRel}, and this plugin will supply
     * a rule which transforms that into a corresponding physical rel for
     * writing to the index. Where possible, the returned plan should consist of
     * logical rels, increasing optimization potential.
     *
     * @param table the optimizer representation for the indexed table
     * @param index the index to be loaded
     * @param cluster container for newly created relational expressions
     *
     * @return plan which will be used as input to optimization
     */
    public RelNode constructIndexBuildPlan(
        RelOptTable table,
        FemLocalIndex index,
        RelOptCluster cluster);

    /**
     * Versions the root page of an index
     *
     * @param oldRoot original root page
     * @param newRoot new versioned, root page
     * @param txnContext Fennel txn context
     */
    public void versionIndexRoot(
        Long oldRoot,
        Long newRoot,
        FennelTxnContext txnContext)
        throws SQLException;

    /**
     * Checks whether ALTER TABLE ADD COLUMN is implemented for tables stored by
     * this local data server.
     *
     * @return whether ALTER TABLE ADD COLUMN can be executed
     */
    public boolean supportsAlterTableAddColumn();
}

// End FarragoMedLocalDataServer.java
