#
# Wireshark tests
#
# Copyright (c) 2018 Peter Wu <peter@lekensteyn.nl>
#
# SPDX-License-Identifier: GPL-2.0-or-later
#
'''Follow Stream tests'''

import subprocess
import pytest


class TestFollowTcp:
    def test_follow_tcp_bad_conditions(self, cmd_tshark, capture_file, base_env):
        '''Checks whether Follow TCP correctly handles lots of edge cases.'''
        # Edge cases include:
        # 1. two sequential segments
        # 2. out-of-order (swapped two sequential segments)
        # 3. Bad overlap (second overlap with different data should be ignored)
        # 4. Ignore bad retransmitted data, but extend with remaining data.
        # 5. Check handling of overlapping data while fragments are incomplete
        #    (out-of-order - cannot add fragments to stream)
        # 6. lost but acked segments
        # 7. lost 3/5 fragments, but acked
        # Not checked: lost and not acked (currently truncated, is that OK?)
        proc_stdout = subprocess.check_output((cmd_tshark,
                                '-r', capture_file('tcp-badsegments.pcap'),
                                '-qz', 'follow,tcp,hex,0',
                                ), encoding='utf-8', env=base_env)

        assert """\
===================================================================
Follow: tcp,hex
Filter: tcp.stream eq 0
Node 0: 10.0.0.1:32323
Node 1: 10.0.0.2:80
00000000  47 45 54 20 2f 20 48 54  54 50 2f 31 2e 31 0d 0a  GET / HT TP/1.1..
00000010  48 6f 73 74 3a 6c 6f 63  61 6c 68 6f 73 74 0d 0a  Host:loc alhost..
00000020  58 2d 53 77 61 70 70 65  64 3a 20 31 73 74 0d 0a  X-Swappe d: 1st..
00000030  58 2d 53 77 61 70 70 65  64 3a 20 32 6e 64 0d 0a  X-Swappe d: 2nd..
00000040  58 2d 4f 76 65 72 6c 61  70 2d 50 61 63 6b 65 74  X-Overla p-Packet
00000050  3a 20 65 78 74 72 61 20  64 61 74 61 2d 2d 0d 0a  : extra  data--..
00000060  58 2d 4f 6f 4f 2d 4f 76  65 72 6c 61 70 3a 20 74  X-OoO-Ov erlap: t
00000070  68 69 73 20 69 73 20 64  65 6c 61 79 65 64 0d 0a  his is d elayed..
00000080  58 2d 4f 6f 4f 2d 4f 76  65 72 6c 61 70 32 3a 20  X-OoO-Ov erlap2:
00000090  73 65 63 6f 6e 64 20 64  65 6c 61 79 65 64 0d 0a  second d elayed..
000000A0  58 2d 4f 6f 4f 2d 4f 76  65 72 6c 61 70 33 3a 65  X-OoO-Ov erlap3:e
000000B0  78 74 65 6e 64 20 66 72  61 67 6d 65 6e 74 0d 0a  xtend fr agment..
000000C0  5b 33 32 20 62 79 74 65  73 20 6d 69 73 73 69 6e  [32 byte s missin
000000D0  67 20 69 6e 20 63 61 70  74 75 72 65 20 66 69 6c  g in cap ture fil
000000E0  65 5d 00                                          e].
000000E3  58 2d 4d 69 73 73 69 6e  67 2d 42 75 74 2d 41 63  X-Missin g-But-Ac
000000F3  6b 65 64 2d 50 72 65 76  69 6f 75 73 3a 31 0d 0a  ked-Prev ious:1..
00000103  5b 31 36 20 62 79 74 65  73 20 6d 69 73 73 69 6e  [16 byte s missin
00000113  67 20 69 6e 20 63 61 70  74 75 72 65 20 66 69 6c  g in cap ture fil
00000123  65 5d 00                                          e].
00000126  3a                                                :
00000127  5b 31 33 20 62 79 74 65  73 20 6d 69 73 73 69 6e  [13 byte s missin
00000137  67 20 69 6e 20 63 61 70  74 75 72 65 20 66 69 6c  g in cap ture fil
00000147  65 5d 00                                          e].
0000014A  0d                                                .
0000014B  5b 31 20 62 79 74 65 73  20 6d 69 73 73 69 6e 67  [1 bytes  missing
0000015B  20 69 6e 20 63 61 70 74  75 72 65 20 66 69 6c 65   in capt ure file
0000016B  5d 00                                             ].
0000016D  58 2d 4d 69 73 73 69 6e  67 2d 33 2d 4f 75 74 2d  X-Missin g-3-Out-
0000017D  4f 66 2d 35 2d 42 75 74  2d 41 43 4b 3a 59 0d 0a  Of-5-But -ACK:Y..
0000018D  0d 0a                                             ..
===================================================================
""".replace("\r\n", "\n") in proc_stdout

    def test_follow_websocket(self, cmd_tshark, capture_file, base_env):
        '''Checks whether Follow Websocket correctly handles masked data.'''
        proc_stdout = subprocess.check_output((cmd_tshark,
                        '-r', capture_file('websocket.pcapng.gz'),
                        '-qz', 'follow,websocket,hex,0',
                        ), encoding='utf-8', env=base_env)

        assert """\
===================================================================
Follow: websocket,hex
Filter: tcp.stream eq 0
Node 0: 127.0.0.1:44380
Node 1: 127.0.0.1:8080
00000000  7b 22 74 79 70 65 22 3a  22 72 65 71 75 65 73 74  {"type": "request
00000010  22 2c 22 64 61 74 61 22  3a 22 31 32 36 35 33 30  ","data" :"126530
00000020  37 34 34 37 34 30 39 34  39 39 30 31 38 22 7d 0a  74474094 99018"}.
	00000000  7b 22 74 79 70 65 22 3a  22 72 65 73 70 6f 6e 73  {"type": "respons
	00000010  65 22 2c 22 64 61 74 61  22 3a 22 31 32 36 35 33  e","data ":"12653
	00000020  30 37 34 34 37 34 30 39  34 39 39 30 31 38 22 7d  07447409 499018"}
	00000030  0a                                                .
00000030  7b 22 74 79 70 65 22 3a  22 72 65 71 75 65 73 74  {"type": "request
00000040  22 2c 22 64 61 74 61 22  3a 22 37 38 32 30 36 32  ","data" :"782062
00000050  33 35 30 33 34 38 39 32  35 35 35 31 31 22 7d 0a  35034892 55511"}.
	00000031  7b 22 74 79 70 65 22 3a  22 72 65 73 70 6f 6e 73  {"type": "respons
	00000041  65 22 2c 22 64 61 74 61  22 3a 22 37 38 32 30 36  e","data ":"78206
	00000051  32 33 35 30 33 34 38 39  32 35 35 35 31 31 22 7d  23503489 255511"}
	00000061  0a                                                .
00000060  7b 22 74 79 70 65 22 3a  22 72 65 71 75 65 73 74  {"type": "request
00000070  22 2c 22 64 61 74 61 22  3a 22 31 34 32 36 39 33  ","data" :"142693
00000080  39 37 32 39 33 36 33 37  39 36 30 39 39 22 7d 0a  97293637 96099"}.
	00000062  7b 22 74 79 70 65 22 3a  22 72 65 73 70 6f 6e 73  {"type": "respons
	00000072  65 22 2c 22 64 61 74 61  22 3a 22 31 34 32 36 39  e","data ":"14269
	00000082  33 39 37 32 39 33 36 33  37 39 36 30 39 39 22 7d  39729363 796099"}
	00000092  0a                                                .
00000090  7b 22 74 79 70 65 22 3a  22 72 65 71 75 65 73 74  {"type": "request
000000A0  22 2c 22 64 61 74 61 22  3a 22 31 30 33 30 31 37  ","data" :"103017
000000B0  37 39 31 35 36 35 32 31  34 34 31 31 36 22 7d 0a  79156521 44116"}.
	00000093  7b 22 74 79 70 65 22 3a  22 72 65 73 70 6f 6e 73  {"type": "respons
	000000A3  65 22 2c 22 64 61 74 61  22 3a 22 31 30 33 30 31  e","data ":"10301
	000000B3  37 37 39 31 35 36 35 32  31 34 34 31 31 36 22 7d  77915652 144116"}
	000000C3  0a                                                .
===================================================================
""".replace("\r\n", "\n") in proc_stdout

    def test_follow_websocket_fragmented(self, cmd_tshark, capture_file, base_env):
        '''Checks whether Follow Websocket correctly handles fragmented data.'''
        proc_stdout = subprocess.check_output((cmd_tshark,
                        '-r', capture_file('websocket-fragmented.pcapng.gz'),
                        '-qz', 'follow,websocket,hex,0',
                        ), encoding='utf-8', env=base_env)

        assert """\
===================================================================
Follow: websocket,hex
Filter: tcp.stream eq 0
Node 0: 127.0.0.1:59700
Node 1: 127.0.0.1:8080
00000000  7b 22 74 79 70 65 22 3a  22 72 65 71 75 65 73 74  {"type": "request
00000010  22 2c 22 64 61 74 61 22  3a 22 37 34 35 33 36 33  ","data" :"745363
00000020  31 36 30 32 33 33 38 32  37 33 33 37 32 36 32 32  16023382 73372622
00000030  38 33 38 32 34 39 31 36  39 38 32 35 39 39 39 38  83824916 98259998
00000040  37 30 30 36 32 36 34 37  36 33 37 35 34 39 30 35  70062647 63754905
00000050  31 36 34 32 34 32 32 33  33 37 39 37 30 31 37 38  16424223 37970178
00000060  37 32 36 30 38 33 38 32  31 36 35 33 32 37 33 35  72608382 16532735
00000070  31 39 32 38 39 33 32 39  31 36 36 33 35 38 37 36  19289329 16635876
00000080  36 34 37 35 30 30 31 39  32 36 32 30 33 38 31 37  64750019 26203817
00000090  33 30 37 34 31 36 31 32  33 36 36 38 35 38 31 30  30741612 36685810
000000A0  33 32 36 34 32 36 35 31  38 37 37 31 34 33 31 38  32642651 87714318
000000B0  31 22 7d 0a                                       1"}.
	00000000  7b 22 74 79 70 65 22 3a  22 72 65 73 70 6f 6e 73  {"type": "respons
	00000010  65 22 2c 22 64 61 74 61  22 3a 22 37 34 35 33 36  e","data ":"74536
	00000020  33 31 36 30 32 33 33 38  32 37 33 33 37 32 36 32  31602338 27337262
	00000030  32 38 33 38 32 34 39 31  36 39 38 32 35 39 39 39  28382491 69825999
	00000040  38 37 30 30 36 32 36 34  37 36 33 37 35 34 39 30  87006264 76375490
	00000050  35 31 36 34 32 34 32 32  33 33 37 39 37 30 31 37  51642422 33797017
	00000060  38 37 32 36 30 38 33 38  32 31 36 35 33 32 37 33  87260838 21653273
	00000070  35 31 39 32 38 39 33 32  39 31 36 36 33 35 38 37  51928932 91663587
	00000080  36 36 34 37 35 30 30 31  39 32 36 32 30 33 38 31  66475001 92620381
	00000090  37 33 30 37 34 31 36 31  32 33 36 36 38 35 38 31  73074161 23668581
	000000A0  30 33 32 36 34 32 36 35  31 38 37 37 31 34 33 31  03264265 18771431
	000000B0  38 31 22 7d 0a                                    81"}.
000000B4  7b 22 74 79 70 65 22 3a  22 72 65 71 75 65 73 74  {"type": "request
000000C4  22 2c 22 64 61 74 61 22  3a 22 35 30 30 30 37 39  ","data" :"500079
000000D4  37 37 35 32 38 33 33 37  32 35 36 31 33 38 30 30  77528337 25613800
000000E4  36 36 30 36 30 39 30 35  37 30 30 37 38 35 35 38  66060905 70078558
000000F4  35 30 39 38 31 35 35 33  31 36 36 39 30 34 38 34  50981553 16690484
00000104  30 34 39 39 31 36 31 38  34 38 37 37 34 34 39 33  04991618 48774493
00000114  32 35 34 34 35 36 38 31  32 35 30 36 37 37 30 39  25445681 25067709
00000124  35 38 32 38 35 34 30 36  39 36 36 34 34 36 32 30  58285406 96644620
00000134  30 34 38 34 38 34 32 39  35 39 34 34 34 35 34 33  04848429 59444543
00000144  31 39 33 39 39 39 32 35  39 32 39 38 37 32 31 38  19399925 92987218
00000154  39 34 36 38 31 38 33 33  30 35 33 38 33 38 36 30  94681833 05383860
00000164  32 38 22 7d 0a                                    28"}.
	000000B5  7b 22 74 79 70 65 22 3a  22 72 65 73 70 6f 6e 73  {"type": "respons
	000000C5  65 22 2c 22 64 61 74 61  22 3a 22 35 30 30 30 37  e","data ":"50007
	000000D5  39 37 37 35 32 38 33 33  37 32 35 36 31 33 38 30  97752833 72561380
	000000E5  30 36 36 30 36 30 39 30  35 37 30 30 37 38 35 35  06606090 57007855
	000000F5  38 35 30 39 38 31 35 35  33 31 36 36 39 30 34 38  85098155 31669048
	00000105  34 30 34 39 39 31 36 31  38 34 38 37 37 34 34 39  40499161 84877449
	00000115  33 32 35 34 34 35 36 38  31 32 35 30 36 37 37 30  32544568 12506770
	00000125  39 35 38 32 38 35 34 30  36 39 36 36 34 34 36 32  95828540 69664462
	00000135  30 30 34 38 34 38 34 32  39 35 39 34 34 34 35 34  00484842 95944454
	00000145  33 31 39 33 39 39 39 32  35 39 32 39 38 37 32 31  31939992 59298721
	00000155  38 39 34 36 38 31 38 33  33 30 35 33 38 33 38 36  89468183 30538386
	00000165  30 32 38 22 7d 0a                                 028"}.
===================================================================
""".replace("\r\n", "\n") in proc_stdout

    def test_follow_websocket_compressed(self, cmd_tshark, capture_file, base_env):
        '''Checks whether Follow Websocket correctly handles compressed data.'''
        proc_stdout = subprocess.check_output((cmd_tshark,
                        '-r', capture_file('websocket-compressed.pcapng.gz'),
                        '-qz', 'follow,websocket,hex,0',
                        ), encoding='utf-8', env=base_env)

        assert """\
===================================================================
Follow: websocket,hex
Filter: tcp.stream eq 0
Node 0: 127.0.0.1:55256
Node 1: 127.0.0.1:8080
00000000  7b 22 74 79 70 65 22 3a  22 72 65 71 75 65 73 74  {"type": "request
00000010  22 2c 22 64 61 74 61 22  3a 22 33 38 35 37 34 34  ","data" :"385744
00000020  35 34 33 35 38 33 37 32  39 37 33 30 37 33 38 37  54358372 97307387
00000030  31 35 33 32 32 35 32 38  37 39 34 30 34 34 32 38  15322528 79404428
00000040  36 30 33 38 36 31 34 30  38 33 30 33 36 33 38 36  60386140 83036386
00000050  36 34 38 37 30 39 36 33  31 39 36 30 32 36 33 34  64870963 19602634
00000060  39 36 38 30 37 36 34 31  33 30 33 35 35 39 30 30  96807641 30355900
00000070  37 30 37 36 36 30 33 33  31 35 32 33 34 36 32 33  70766033 15234623
00000080  34 30 33 39 35 39 34 37  35 30 36 39 31 35 39 34  40395947 50691594
00000090  38 34 34 35 37 36 37 39  38 38 34 30 30 33 34 34  84457679 88400344
000000A0  36 39 32 35 38 36 32 36  33 30 39 34 34 32 35 36  69258626 30944256
000000B0  32 22 7d 0a                                       2"}.
	00000000  7b 22 74 79 70 65 22 3a  22 72 65 73 70 6f 6e 73  {"type": "respons
	00000010  65 22 2c 22 64 61 74 61  22 3a 22 33 38 35 37 34  e","data ":"38574
	00000020  34 35 34 33 35 38 33 37  32 39 37 33 30 37 33 38  45435837 29730738
	00000030  37 31 35 33 32 32 35 32  38 37 39 34 30 34 34 32  71532252 87940442
	00000040  38 36 30 33 38 36 31 34  30 38 33 30 33 36 33 38  86038614 08303638
	00000050  36 36 34 38 37 30 39 36  33 31 39 36 30 32 36 33  66487096 31960263
	00000060  34 39 36 38 30 37 36 34  31 33 30 33 35 35 39 30  49680764 13035590
	00000070  30 37 30 37 36 36 30 33  33 31 35 32 33 34 36 32  07076603 31523462
	00000080  33 34 30 33 39 35 39 34  37 35 30 36 39 31 35 39  34039594 75069159
	00000090  34 38 34 34 35 37 36 37  39 38 38 34 30 30 33 34  48445767 98840034
	000000A0  34 36 39 32 35 38 36 32  36 33 30 39 34 34 32 35  46925862 63094425
	000000B0  36 32 22 7d 0a                                    62"}.
000000B4  7b 22 74 79 70 65 22 3a  22 72 65 71 75 65 73 74  {"type": "request
000000C4  22 2c 22 64 61 74 61 22  3a 22 36 31 37 30 30 39  ","data" :"617009
000000D4  33 36 30 38 33 33 38 33  34 38 39 30 37 31 39 38  36083383 48907198
000000E4  34 39 33 35 35 38 38 38  33 31 30 35 30 36 32 31  49355888 31050621
000000F4  33 38 31 33 35 39 31 38  30 38 34 37 33 39 36 32  38135918 08473962
00000104  35 31 36 37 35 37 33 34  30 31 31 36 36 31 34 37  51675734 01166147
00000114  38 34 34 34 33 31 33 30  39 31 35 30 33 33 35 37  84443130 91503357
00000124  31 31 36 35 38 37 39 34  32 36 34 32 30 34 36 34  11658794 26420464
00000134  37 39 38 39 30 33 39 30  37 31 39 30 34 35 35 39  79890390 71904559
00000144  33 31 37 35 37 30 31 34  31 33 33 31 35 33 33 34  31757014 13315334
00000154  31 34 30 35 39 32 38 36  34 37 33 30 35 32 31 38  14059286 47305218
00000164  39 32 22 7d 0a                                    92"}.
	000000B5  7b 22 74 79 70 65 22 3a  22 72 65 73 70 6f 6e 73  {"type": "respons
	000000C5  65 22 2c 22 64 61 74 61  22 3a 22 36 31 37 30 30  e","data ":"61700
	000000D5  39 33 36 30 38 33 33 38  33 34 38 39 30 37 31 39  93608338 34890719
	000000E5  38 34 39 33 35 35 38 38  38 33 31 30 35 30 36 32  84935588 83105062
	000000F5  31 33 38 31 33 35 39 31  38 30 38 34 37 33 39 36  13813591 80847396
	00000105  32 35 31 36 37 35 37 33  34 30 31 31 36 36 31 34  25167573 40116614
	00000115  37 38 34 34 34 33 31 33  30 39 31 35 30 33 33 35  78444313 09150335
	00000125  37 31 31 36 35 38 37 39  34 32 36 34 32 30 34 36  71165879 42642046
	00000135  34 37 39 38 39 30 33 39  30 37 31 39 30 34 35 35  47989039 07190455
	00000145  39 33 31 37 35 37 30 31  34 31 33 33 31 35 33 33  93175701 41331533
	00000155  34 31 34 30 35 39 32 38  36 34 37 33 30 35 32 31  41405928 64730521
	00000165  38 39 32 22 7d 0a                                 892"}.
===================================================================
""".replace("\r\n", "\n") in proc_stdout

    def test_follow_websocket_compressed_fragmented(self, cmd_tshark, capture_file, base_env):
        '''Checks whether Follow Websocket correctly handles compressed and fragmented data.'''
        proc_stdout = subprocess.check_output((cmd_tshark,
                        '-r', capture_file('websocket-compressed-fragmented.pcapng.gz'),
                        '-qz', 'follow,websocket,hex,0',
                        ), encoding='utf-8', env=base_env)

        assert """\
===================================================================
Follow: websocket,hex
Filter: tcp.stream eq 0
Node 0: 127.0.0.1:39964
Node 1: 127.0.0.1:8080
00000000  7b 22 74 79 70 65 22 3a  22 72 65 71 75 65 73 74  {"type": "request
00000010  22 2c 22 64 61 74 61 22  3a 22 36 30 36 32 39 36  ","data" :"606296
00000020  32 34 35 35 32 37 33 39  33 32 32 35 33 36 35 33  24552739 32253653
00000030  38 37 33 33 35 37 37 37  30 39 37 32 36 31 38 36  87335777 09726186
00000040  36 34 38 38 36 32 39 32  30 34 31 32 30 38 36 37  64886292 04120867
00000050  34 32 30 33 38 33 39 38  33 33 35 32 34 35 31 35  42038398 33524515
00000060  35 32 38 34 33 39 36 36  35 31 36 36 39 35 31 37  52843966 51669517
00000070  36 34 37 35 35 30 35 39  39 37 36 34 30 32 38 33  64755059 97640283
00000080  35 31 39 34 33 35 35 34  39 34 30 38 36 33 38 37  51943554 94086387
00000090  34 34 37 31 35 33 35 33  31 37 33 34 33 33 38 34  44715353 17343384
000000A0  31 35 31 30 37 35 37 30  35 39 37 33 37 37 30 39  15107570 59737709
000000B0  39 39 30 30 34 31 36 38  38 38 38 38 35 32 38 30  99004168 88885280
000000C0  32 36 34 35 38 32 36 35  35 32 39 36 38 33 35 38  26458265 52968358
000000D0  38 39 30 37 33 35 31 31  37 36 36 31 36 33 31 38  89073511 76616318
000000E0  39 33 31 31 33 33 39 38  33 39 34 30 30 37 39 36  93113398 39400796
000000F0  32 33 38 38 38 39 33 34  31 37 35 35 36 34 36 35  23888934 17556465
00000100  37 35 30 38 33 37 39 32  36 39 35 31 36 31 33 37  75083792 69516137
00000110  32 34 38 38 38 30 33 33  39 33 32 31 35 37 32 34  24888033 93215724
00000120  38 34 38 36 31 36 39 39  30 30 33 38 31 36 31 35  84861699 00381615
00000130  34 38 35 34 34 38 33 39  34 30 34 37 37 35 34 35  48544839 40477545
00000140  33 32 36 33 36 35 39 33  22 7d 0a                 32636593 "}.
	00000000  7b 22 74 79 70 65 22 3a  22 72 65 73 70 6f 6e 73  {"type": "respons
	00000010  65 22 2c 22 64 61 74 61  22 3a 22 36 30 36 32 39  e","data ":"60629
	00000020  36 32 34 35 35 32 37 33  39 33 32 32 35 33 36 35  62455273 93225365
	00000030  33 38 37 33 33 35 37 37  37 30 39 37 32 36 31 38  38733577 70972618
	00000040  36 36 34 38 38 36 32 39  32 30 34 31 32 30 38 36  66488629 20412086
	00000050  37 34 32 30 33 38 33 39  38 33 33 35 32 34 35 31  74203839 83352451
	00000060  35 35 32 38 34 33 39 36  36 35 31 36 36 39 35 31  55284396 65166951
	00000070  37 36 34 37 35 35 30 35  39 39 37 36 34 30 32 38  76475505 99764028
	00000080  33 35 31 39 34 33 35 35  34 39 34 30 38 36 33 38  35194355 49408638
	00000090  37 34 34 37 31 35 33 35  33 31 37 33 34 33 33 38  74471535 31734338
	000000A0  34 31 35 31 30 37 35 37  30 35 39 37 33 37 37 30  41510757 05973770
	000000B0  39 39 39 30 30 34 31 36  38 38 38 38 38 35 32 38  99900416 88888528
	000000C0  30 32 36 34 35 38 32 36  35 35 32 39 36 38 33 35  02645826 55296835
	000000D0  38 38 39 30 37 33 35 31  31 37 36 36 31 36 33 31  88907351 17661631
	000000E0  38 39 33 31 31 33 33 39  38 33 39 34 30 30 37 39  89311339 83940079
	000000F0  36 32 33 38 38 38 39 33  34 31 37 35 35 36 34 36  62388893 41755646
	00000100  35 37 35 30 38 33 37 39  32 36 39 35 31 36 31 33  57508379 26951613
	00000110  37 32 34 38 38 38 30 33  33 39 33 32 31 35 37 32  72488803 39321572
	00000120  34 38 34 38 36 31 36 39  39 30 30 33 38 31 36 31  48486169 90038161
	00000130  35 34 38 35 34 34 38 33  39 34 30 34 37 37 35 34  54854483 94047754
	00000140  35 33 32 36 33 36 35 39  33 22 7d 0a              53263659 3"}.
0000014B  7b 22 74 79 70 65 22 3a  22 72 65 71 75 65 73 74  {"type": "request
0000015B  22 2c 22 64 61 74 61 22  3a 22 34 31 30 35 32 30  ","data" :"410520
0000016B  34 31 33 30 36 36 36 36  32 32 30 38 32 35 33 31  41306666 22082531
0000017B  36 36 33 34 37 39 32 39  32 31 31 39 39 37 32 35  66347929 21199725
0000018B  35 38 35 38 37 39 32 32  30 39 31 36 31 38 37 39  58587922 09161879
0000019B  36 38 36 37 32 36 33 32  34 35 39 35 34 35 39 35  68672632 45954595
000001AB  34 31 31 32 38 31 36 32  39 38 37 33 31 31 32 32  41128162 98731122
000001BB  31 36 39 38 31 38 35 37  32 33 38 34 36 37 39 34  16981857 23846794
000001CB  37 36 36 33 30 38 38 36  34 39 33 36 36 37 34 35  76630886 49366745
000001DB  32 37 37 39 39 33 31 37  34 37 32 33 39 37 37 39  27799317 47239779
000001EB  38 38 39 30 31 32 33 39  33 33 37 37 35 30 35 36  88901239 33775056
000001FB  31 34 30 34 37 38 31 32  39 32 30 36 31 30 34 32  14047812 92061042
0000020B  35 30 34 38 35 39 34 35  34 37 35 33 30 39 34 35  50485945 47530945
0000021B  37 32 35 30 35 33 34 33  39 32 31 36 34 35 37 35  72505343 92164575
0000022B  37 34 39 30 34 33 30 36  35 38 32 31 39 36 39 35  74904306 58219695
0000023B  37 37 37 34 32 39 35 39  39 32 34 32 37 38 32 32  77742959 92427822
0000024B  37 38 38 31 37 33 36 38  38 38 39 31 37 37 32 31  78817368 88917721
0000025B  36 39 38 30 35 31 38 38  35 39 32 32 32 32 34 37  69805188 59222247
0000026B  32 36 31 37 39 38 35 36  32 31 34 33 30 38 33 35  26179856 21430835
0000027B  38 38 38 38 39 36 38 33  39 38 33 38 33 34 31 33  88889683 98383413
0000028B  31 35 34 35 35 33 22 7d  0a                       154553"} .
	0000014C  7b 22 74 79 70 65 22 3a  22 72 65 73 70 6f 6e 73  {"type": "respons
	0000015C  65 22 2c 22 64 61 74 61  22 3a 22 34 31 30 35 32  e","data ":"41052
	0000016C  30 34 31 33 30 36 36 36  36 32 32 30 38 32 35 33  04130666 62208253
	0000017C  31 36 36 33 34 37 39 32  39 32 31 31 39 39 37 32  16634792 92119972
	0000018C  35 35 38 35 38 37 39 32  32 30 39 31 36 31 38 37  55858792 20916187
	0000019C  39 36 38 36 37 32 36 33  32 34 35 39 35 34 35 39  96867263 24595459
	000001AC  35 34 31 31 32 38 31 36  32 39 38 37 33 31 31 32  54112816 29873112
	000001BC  32 31 36 39 38 31 38 35  37 32 33 38 34 36 37 39  21698185 72384679
	000001CC  34 37 36 36 33 30 38 38  36 34 39 33 36 36 37 34  47663088 64936674
	000001DC  35 32 37 37 39 39 33 31  37 34 37 32 33 39 37 37  52779931 74723977
	000001EC  39 38 38 39 30 31 32 33  39 33 33 37 37 35 30 35  98890123 93377505
	000001FC  36 31 34 30 34 37 38 31  32 39 32 30 36 31 30 34  61404781 29206104
	0000020C  32 35 30 34 38 35 39 34  35 34 37 35 33 30 39 34  25048594 54753094
	0000021C  35 37 32 35 30 35 33 34  33 39 32 31 36 34 35 37  57250534 39216457
	0000022C  35 37 34 39 30 34 33 30  36 35 38 32 31 39 36 39  57490430 65821969
	0000023C  35 37 37 37 34 32 39 35  39 39 32 34 32 37 38 32  57774295 99242782
	0000024C  32 37 38 38 31 37 33 36  38 38 38 39 31 37 37 32  27881736 88891772
	0000025C  31 36 39 38 30 35 31 38  38 35 39 32 32 32 32 34  16980518 85922224
	0000026C  37 32 36 31 37 39 38 35  36 32 31 34 33 30 38 33  72617985 62143083
	0000027C  35 38 38 38 38 39 36 38  33 39 38 33 38 33 34 31  58888968 39838341
	0000028C  33 31 35 34 35 35 33 22  7d 0a                    3154553" }.
===================================================================
""".replace("\r\n", "\n") in proc_stdout

    def test_follow_websocket_compressed_server_only(self, cmd_tshark, capture_file, base_env):
        '''Checks whether Follow Websocket correctly handles only server compressed data.'''
        proc_stdout = subprocess.check_output((cmd_tshark,
                        '-r', capture_file('websocket-compressed-server-only.pcap'),
                        '-qz', 'follow,websocket,hex,0',
                        ), encoding='utf-8', env=base_env)

        assert """\
===================================================================
Follow: websocket,hex
Filter: tcp.stream eq 0
Node 0: 127.0.0.1:44920
Node 1: 127.0.0.1:8080
00000000  66 72 6f 6d 20 63 6c 69  65 6e 74 2e 2e 20 30     from cli ent.. 0
0000000F  66 72 6f 6d 20 63 6c 69  65 6e 74 2e 2e 20 31     from cli ent.. 1
0000001E  66 72 6f 6d 20 63 6c 69  65 6e 74 2e 2e 20 32     from cli ent.. 2
0000002D  66 72 6f 6d 20 63 6c 69  65 6e 74 2e 2e 20 33     from cli ent.. 3
0000003C  66 72 6f 6d 20 63 6c 69  65 6e 74 2e 2e 20 34     from cli ent.. 4
0000004B  66 72 6f 6d 20 63 6c 69  65 6e 74 2e 2e 20 35     from cli ent.. 5
0000005A  66 72 6f 6d 20 63 6c 69  65 6e 74 2e 2e 20 36     from cli ent.. 6
00000069  66 72 6f 6d 20 63 6c 69  65 6e 74 2e 2e 20 37     from cli ent.. 7
00000078  66 72 6f 6d 20 63 6c 69  65 6e 74 2e 2e 20 38     from cli ent.. 8
00000087  66 72 6f 6d 20 63 6c 69  65 6e 74 2e 2e 20 39     from cli ent.. 9
	00000000  70 6f 6e 67 3a 20 66 72  6f 6d 20 63 6c 69 65 6e  pong: fr om clien
	00000010  74 2e 2e 20 30                                    t.. 0
	00000015  70 6f 6e 67 3a 20 66 72  6f 6d 20 63 6c 69 65 6e  pong: fr om clien
	00000025  74 2e 2e 20 31                                    t.. 1
	0000002A  70 6f 6e 67 3a 20 66 72  6f 6d 20 63 6c 69 65 6e  pong: fr om clien
	0000003A  74 2e 2e 20 32                                    t.. 2
	0000003F  70 6f 6e 67 3a 20 66 72  6f 6d 20 63 6c 69 65 6e  pong: fr om clien
	0000004F  74 2e 2e 20 33                                    t.. 3
	00000054  70 6f 6e 67 3a 20 66 72  6f 6d 20 63 6c 69 65 6e  pong: fr om clien
	00000064  74 2e 2e 20 34                                    t.. 4
	00000069  70 6f 6e 67 3a 20 66 72  6f 6d 20 63 6c 69 65 6e  pong: fr om clien
	00000079  74 2e 2e 20 35                                    t.. 5
	0000007E  70 6f 6e 67 3a 20 66 72  6f 6d 20 63 6c 69 65 6e  pong: fr om clien
	0000008E  74 2e 2e 20 36                                    t.. 6
	00000093  70 6f 6e 67 3a 20 66 72  6f 6d 20 63 6c 69 65 6e  pong: fr om clien
	000000A3  74 2e 2e 20 37                                    t.. 7
	000000A8  70 6f 6e 67 3a 20 66 72  6f 6d 20 63 6c 69 65 6e  pong: fr om clien
	000000B8  74 2e 2e 20 38                                    t.. 8
	000000BD  70 6f 6e 67 3a 20 66 72  6f 6d 20 63 6c 69 65 6e  pong: fr om clien
	000000CD  74 2e 2e 20 39                                    t.. 9
===================================================================
""".replace("\r\n", "\n") in proc_stdout
