/*
 * Copyright (C) 2022 Michael Brown <mbrown@fensystems.co.uk>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 * You can also choose to distribute this program under the terms of
 * the Unmodified Binary Distribution Licence (as given in the file
 * COPYING.UBDL), provided that you have satisfied its requirements.
 */

FILE_LICENCE ( GPL2_OR_LATER_OR_UBDL );

/** @file
 *
 * Galois/Counter Mode (GCM) tests
 *
 * These test vectors are provided by NIST as part of the GCM proposed
 * specification document (which, unlike the final published
 * specification document, includes test vectors with intermediate
 * values):
 *
 * https://csrc.nist.rip/groups/ST/toolkit/BCM/documents/proposedmodes/gcm/gcm-spec.pdf
 *
 */

/* Forcibly enable assertions */
#undef NDEBUG

#include <string.h>
#include <ipxe/gcm.h>
#include <ipxe/aes.h>
#include <ipxe/test.h>
#include "cipher_test.h"

/** 128-bit zero key */
#define GCM_KEY_128_ZERO						\
	KEY ( 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,	\
	      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 )

/** 128-bit key */
#define GCM_KEY_128							\
	KEY ( 0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c, 0x6d,	\
	      0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08 )

/** 192-bit zero key */
#define GCM_KEY_192_ZERO						\
	KEY ( 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,	\
	      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,	\
	      0x00, 0x00, 0x00, 0x00, 0x00, 0x00 )

/** 192-bit key */
#define GCM_KEY_192							\
	KEY ( 0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c, 0x6d,	\
	      0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08, 0xfe, 0xff,	\
	      0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c )

/** 256-bit zero key */
#define GCM_KEY_256_ZERO						\
	KEY ( 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,	\
	      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,	\
	      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,	\
	      0x00, 0x00, 0x00, 0x00, 0x00 )

/** 256-bit key */
#define GCM_KEY_256							\
	KEY ( 0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c, 0x6d,	\
	      0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08, 0xfe, 0xff,	\
	      0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c, 0x6d, 0x6a, 0x8f,	\
	      0x94, 0x67, 0x30, 0x83, 0x08 )

/** 64-bit IV */
#define GCM_IV_64							\
	IV ( 0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce, 0xdb, 0xad )

/** 96-bit zero IV */
#define GCM_IV_96_ZERO							\
	IV ( 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,	\
	     0x00, 0x00, 0x00 )

/** 96-bit IV */
#define GCM_IV_96							\
	IV ( 0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce, 0xdb, 0xad, 0xde,	\
	     0xca, 0xf8, 0x88 )

/** 480-bit IV */
#define GCM_IV_480							\
	IV ( 0x93, 0x13, 0x22, 0x5d, 0xf8, 0x84, 0x06, 0xe5, 0x55,	\
	     0x90, 0x9c, 0x5a, 0xff, 0x52, 0x69, 0xaa, 0x6a, 0x7a,	\
	     0x95, 0x38, 0x53, 0x4f, 0x7d, 0xa1, 0xe4, 0xc3, 0x03,	\
	     0xd2, 0xa3, 0x18, 0xa7, 0x28, 0xc3, 0xc0, 0xc9, 0x51,	\
	     0x56, 0x80, 0x95, 0x39, 0xfc, 0xf0, 0xe2, 0x42, 0x9a,	\
	     0x6b, 0x52, 0x54, 0x16, 0xae, 0xdb, 0xf5, 0xa0, 0xde,	\
	     0x6a, 0x57, 0xa6, 0x37, 0xb3, 0x9b )

/** Empty additional data */
#define GCM_ADDITIONAL_EMPTY ADDITIONAL()

/** 160-bit additional data */
#define GCM_ADDITIONAL_160						\
	ADDITIONAL ( 0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,	\
		     0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,	\
		     0xab, 0xad, 0xda, 0xd2 )

/** Empty plaintext */
#define GCM_PLAINTEXT_EMPTY PLAINTEXT()

/** 128-bit zero plaintext */
#define GCM_PLAINTEXT_128_ZERO						\
	PLAINTEXT ( 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,	\
		    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 )

/** 512-bit plaintext */
#define GCM_PLAINTEXT_512						\
	PLAINTEXT ( 0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,	\
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,	\
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,	\
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,	\
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,	\
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,	\
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,	\
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55 )

/** 480-bit plaintext */
#define GCM_PLAINTEXT_480						\
	PLAINTEXT ( 0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,	\
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,	\
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,	\
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,	\
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,	\
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,	\
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,	\
		    0xba, 0x63, 0x7b, 0x39 )

/** Test 1 */
CIPHER_TEST ( gcm_test_1, &aes_gcm_algorithm, GCM_KEY_128_ZERO,
	      GCM_IV_96_ZERO, GCM_ADDITIONAL_EMPTY, GCM_PLAINTEXT_EMPTY,
	      CIPHERTEXT(),
	      AUTH ( 0x58, 0xe2, 0xfc, 0xce, 0xfa, 0x7e, 0x30, 0x61, 0x36,
		     0x7f, 0x1d, 0x57, 0xa4, 0xe7, 0x45, 0x5a ) );

/** Test 2 */
CIPHER_TEST ( gcm_test_2, &aes_gcm_algorithm, GCM_KEY_128_ZERO,
	      GCM_IV_96_ZERO, GCM_ADDITIONAL_EMPTY, GCM_PLAINTEXT_128_ZERO,
	      CIPHERTEXT ( 0x03, 0x88, 0xda, 0xce, 0x60, 0xb6, 0xa3, 0x92,
			   0xf3, 0x28, 0xc2, 0xb9, 0x71, 0xb2, 0xfe, 0x78 ),
	      AUTH ( 0xab, 0x6e, 0x47, 0xd4, 0x2c, 0xec, 0x13, 0xbd, 0xf5,
		     0x3a, 0x67, 0xb2, 0x12, 0x57, 0xbd, 0xdf ) );

/** Test 3 */
CIPHER_TEST ( gcm_test_3, &aes_gcm_algorithm, GCM_KEY_128,
	      GCM_IV_96, GCM_ADDITIONAL_EMPTY, GCM_PLAINTEXT_512,
	      CIPHERTEXT ( 0x42, 0x83, 0x1e, 0xc2, 0x21, 0x77, 0x74, 0x24,
			   0x4b, 0x72, 0x21, 0xb7, 0x84, 0xd0, 0xd4, 0x9c,
			   0xe3, 0xaa, 0x21, 0x2f, 0x2c, 0x02, 0xa4, 0xe0,
			   0x35, 0xc1, 0x7e, 0x23, 0x29, 0xac, 0xa1, 0x2e,
			   0x21, 0xd5, 0x14, 0xb2, 0x54, 0x66, 0x93, 0x1c,
			   0x7d, 0x8f, 0x6a, 0x5a, 0xac, 0x84, 0xaa, 0x05,
			   0x1b, 0xa3, 0x0b, 0x39, 0x6a, 0x0a, 0xac, 0x97,
			   0x3d, 0x58, 0xe0, 0x91, 0x47, 0x3f, 0x59, 0x85 ),
	      AUTH ( 0x4d, 0x5c, 0x2a, 0xf3, 0x27, 0xcd, 0x64, 0xa6, 0x2c,
		     0xf3, 0x5a, 0xbd, 0x2b, 0xa6, 0xfa, 0xb4 ) );

/** Test 4 */
CIPHER_TEST ( gcm_test_4, &aes_gcm_algorithm, GCM_KEY_128,
	      GCM_IV_96, GCM_ADDITIONAL_160, GCM_PLAINTEXT_480,
	      CIPHERTEXT ( 0x42, 0x83, 0x1e, 0xc2, 0x21, 0x77, 0x74, 0x24,
			   0x4b, 0x72, 0x21, 0xb7, 0x84, 0xd0, 0xd4, 0x9c,
			   0xe3, 0xaa, 0x21, 0x2f, 0x2c, 0x02, 0xa4, 0xe0,
			   0x35, 0xc1, 0x7e, 0x23, 0x29, 0xac, 0xa1, 0x2e,
			   0x21, 0xd5, 0x14, 0xb2, 0x54, 0x66, 0x93, 0x1c,
			   0x7d, 0x8f, 0x6a, 0x5a, 0xac, 0x84, 0xaa, 0x05,
			   0x1b, 0xa3, 0x0b, 0x39, 0x6a, 0x0a, 0xac, 0x97,
			   0x3d, 0x58, 0xe0, 0x91 ),
	      AUTH ( 0x5b, 0xc9, 0x4f, 0xbc, 0x32, 0x21, 0xa5, 0xdb, 0x94,
		     0xfa, 0xe9, 0x5a, 0xe7, 0x12, 0x1a, 0x47 ) );

/** Test 5 */
CIPHER_TEST ( gcm_test_5, &aes_gcm_algorithm, GCM_KEY_128,
	      GCM_IV_64, GCM_ADDITIONAL_160, GCM_PLAINTEXT_480,
	      CIPHERTEXT ( 0x61, 0x35, 0x3b, 0x4c, 0x28, 0x06, 0x93, 0x4a,
			   0x77, 0x7f, 0xf5, 0x1f, 0xa2, 0x2a, 0x47, 0x55,
			   0x69, 0x9b, 0x2a, 0x71, 0x4f, 0xcd, 0xc6, 0xf8,
			   0x37, 0x66, 0xe5, 0xf9, 0x7b, 0x6c, 0x74, 0x23,
			   0x73, 0x80, 0x69, 0x00, 0xe4, 0x9f, 0x24, 0xb2,
			   0x2b, 0x09, 0x75, 0x44, 0xd4, 0x89, 0x6b, 0x42,
			   0x49, 0x89, 0xb5, 0xe1, 0xeb, 0xac, 0x0f, 0x07,
			   0xc2, 0x3f, 0x45, 0x98 ),
	      AUTH ( 0x36, 0x12, 0xd2, 0xe7, 0x9e, 0x3b, 0x07, 0x85, 0x56,
		     0x1b, 0xe1, 0x4a, 0xac, 0xa2, 0xfc, 0xcb ) );

/** Test 6 */
CIPHER_TEST ( gcm_test_6, &aes_gcm_algorithm, GCM_KEY_128,
	      GCM_IV_480, GCM_ADDITIONAL_160, GCM_PLAINTEXT_480,
	      CIPHERTEXT ( 0x8c, 0xe2, 0x49, 0x98, 0x62, 0x56, 0x15, 0xb6,
			   0x03, 0xa0, 0x33, 0xac, 0xa1, 0x3f, 0xb8, 0x94,
			   0xbe, 0x91, 0x12, 0xa5, 0xc3, 0xa2, 0x11, 0xa8,
			   0xba, 0x26, 0x2a, 0x3c, 0xca, 0x7e, 0x2c, 0xa7,
			   0x01, 0xe4, 0xa9, 0xa4, 0xfb, 0xa4, 0x3c, 0x90,
			   0xcc, 0xdc, 0xb2, 0x81, 0xd4, 0x8c, 0x7c, 0x6f,
			   0xd6, 0x28, 0x75, 0xd2, 0xac, 0xa4, 0x17, 0x03,
			   0x4c, 0x34, 0xae, 0xe5 ),
	      AUTH ( 0x61, 0x9c, 0xc5, 0xae, 0xff, 0xfe, 0x0b, 0xfa, 0x46,
		     0x2a, 0xf4, 0x3c, 0x16, 0x99, 0xd0, 0x50 ) );

/** Test 7 */
CIPHER_TEST ( gcm_test_7, &aes_gcm_algorithm, GCM_KEY_192_ZERO,
	      GCM_IV_96_ZERO, GCM_ADDITIONAL_EMPTY, GCM_PLAINTEXT_EMPTY,
	      CIPHERTEXT(),
	      AUTH ( 0xcd, 0x33, 0xb2, 0x8a, 0xc7, 0x73, 0xf7, 0x4b, 0xa0,
		     0x0e, 0xd1, 0xf3, 0x12, 0x57, 0x24, 0x35 ) );

/** Test 8 */
CIPHER_TEST ( gcm_test_8, &aes_gcm_algorithm, GCM_KEY_192_ZERO,
	      GCM_IV_96_ZERO, GCM_ADDITIONAL_EMPTY, GCM_PLAINTEXT_128_ZERO,
	      CIPHERTEXT ( 0x98, 0xe7, 0x24, 0x7c, 0x07, 0xf0, 0xfe, 0x41,
			   0x1c, 0x26, 0x7e, 0x43, 0x84, 0xb0, 0xf6, 0x00 ),
	      AUTH ( 0x2f, 0xf5, 0x8d, 0x80, 0x03, 0x39, 0x27, 0xab, 0x8e,
		     0xf4, 0xd4, 0x58, 0x75, 0x14, 0xf0, 0xfb ) );

/** Test 9 */
CIPHER_TEST ( gcm_test_9, &aes_gcm_algorithm, GCM_KEY_192,
	      GCM_IV_96, GCM_ADDITIONAL_EMPTY, GCM_PLAINTEXT_512,
	      CIPHERTEXT ( 0x39, 0x80, 0xca, 0x0b, 0x3c, 0x00, 0xe8, 0x41,
			   0xeb, 0x06, 0xfa, 0xc4, 0x87, 0x2a, 0x27, 0x57,
			   0x85, 0x9e, 0x1c, 0xea, 0xa6, 0xef, 0xd9, 0x84,
			   0x62, 0x85, 0x93, 0xb4, 0x0c, 0xa1, 0xe1, 0x9c,
			   0x7d, 0x77, 0x3d, 0x00, 0xc1, 0x44, 0xc5, 0x25,
			   0xac, 0x61, 0x9d, 0x18, 0xc8, 0x4a, 0x3f, 0x47,
			   0x18, 0xe2, 0x44, 0x8b, 0x2f, 0xe3, 0x24, 0xd9,
			   0xcc, 0xda, 0x27, 0x10, 0xac, 0xad, 0xe2, 0x56 ),
	      AUTH ( 0x99, 0x24, 0xa7, 0xc8, 0x58, 0x73, 0x36, 0xbf, 0xb1,
		     0x18, 0x02, 0x4d, 0xb8, 0x67, 0x4a, 0x14 ) );

/** Test 10 */
CIPHER_TEST ( gcm_test_10, &aes_gcm_algorithm, GCM_KEY_192,
	      GCM_IV_96, GCM_ADDITIONAL_160, GCM_PLAINTEXT_480,
	      CIPHERTEXT ( 0x39, 0x80, 0xca, 0x0b, 0x3c, 0x00, 0xe8, 0x41,
			   0xeb, 0x06, 0xfa, 0xc4, 0x87, 0x2a, 0x27, 0x57,
			   0x85, 0x9e, 0x1c, 0xea, 0xa6, 0xef, 0xd9, 0x84,
			   0x62, 0x85, 0x93, 0xb4, 0x0c, 0xa1, 0xe1, 0x9c,
			   0x7d, 0x77, 0x3d, 0x00, 0xc1, 0x44, 0xc5, 0x25,
			   0xac, 0x61, 0x9d, 0x18, 0xc8, 0x4a, 0x3f, 0x47,
			   0x18, 0xe2, 0x44, 0x8b, 0x2f, 0xe3, 0x24, 0xd9,
			   0xcc, 0xda, 0x27, 0x10 ),
	      AUTH ( 0x25, 0x19, 0x49, 0x8e, 0x80, 0xf1, 0x47, 0x8f, 0x37,
		     0xba, 0x55, 0xbd, 0x6d, 0x27, 0x61, 0x8c ) );

/** Test 11 */
CIPHER_TEST ( gcm_test_11, &aes_gcm_algorithm, GCM_KEY_192,
	      GCM_IV_64, GCM_ADDITIONAL_160, GCM_PLAINTEXT_480,
	      CIPHERTEXT ( 0x0f, 0x10, 0xf5, 0x99, 0xae, 0x14, 0xa1, 0x54,
			   0xed, 0x24, 0xb3, 0x6e, 0x25, 0x32, 0x4d, 0xb8,
			   0xc5, 0x66, 0x63, 0x2e, 0xf2, 0xbb, 0xb3, 0x4f,
			   0x83, 0x47, 0x28, 0x0f, 0xc4, 0x50, 0x70, 0x57,
			   0xfd, 0xdc, 0x29, 0xdf, 0x9a, 0x47, 0x1f, 0x75,
			   0xc6, 0x65, 0x41, 0xd4, 0xd4, 0xda, 0xd1, 0xc9,
			   0xe9, 0x3a, 0x19, 0xa5, 0x8e, 0x8b, 0x47, 0x3f,
			   0xa0, 0xf0, 0x62, 0xf7 ),
	      AUTH ( 0x65, 0xdc, 0xc5, 0x7f, 0xcf, 0x62, 0x3a, 0x24, 0x09,
		     0x4f, 0xcc, 0xa4, 0x0d, 0x35, 0x33, 0xf8 ) );

/** Test 12 */
CIPHER_TEST ( gcm_test_12, &aes_gcm_algorithm, GCM_KEY_192,
	      GCM_IV_480, GCM_ADDITIONAL_160, GCM_PLAINTEXT_480,
	      CIPHERTEXT ( 0xd2, 0x7e, 0x88, 0x68, 0x1c, 0xe3, 0x24, 0x3c,
			   0x48, 0x30, 0x16, 0x5a, 0x8f, 0xdc, 0xf9, 0xff,
			   0x1d, 0xe9, 0xa1, 0xd8, 0xe6, 0xb4, 0x47, 0xef,
			   0x6e, 0xf7, 0xb7, 0x98, 0x28, 0x66, 0x6e, 0x45,
			   0x81, 0xe7, 0x90, 0x12, 0xaf, 0x34, 0xdd, 0xd9,
			   0xe2, 0xf0, 0x37, 0x58, 0x9b, 0x29, 0x2d, 0xb3,
			   0xe6, 0x7c, 0x03, 0x67, 0x45, 0xfa, 0x22, 0xe7,
			   0xe9, 0xb7, 0x37, 0x3b ),
	      AUTH ( 0xdc, 0xf5, 0x66, 0xff, 0x29, 0x1c, 0x25, 0xbb, 0xb8,
		     0x56, 0x8f, 0xc3, 0xd3, 0x76, 0xa6, 0xd9 ) );

/** Test 13 */
CIPHER_TEST ( gcm_test_13, &aes_gcm_algorithm, GCM_KEY_256_ZERO,
	      GCM_IV_96_ZERO, GCM_ADDITIONAL_EMPTY, GCM_PLAINTEXT_EMPTY,
	      CIPHERTEXT(),
	      AUTH ( 0x53, 0x0f, 0x8a, 0xfb, 0xc7, 0x45, 0x36, 0xb9, 0xa9,
		     0x63, 0xb4, 0xf1, 0xc4, 0xcb, 0x73, 0x8b ) );

/** Test 14 */
CIPHER_TEST ( gcm_test_14, &aes_gcm_algorithm, GCM_KEY_256_ZERO,
	      GCM_IV_96_ZERO, GCM_ADDITIONAL_EMPTY, GCM_PLAINTEXT_128_ZERO,
	      CIPHERTEXT ( 0xce, 0xa7, 0x40, 0x3d, 0x4d, 0x60, 0x6b, 0x6e,
			   0x07, 0x4e, 0xc5, 0xd3, 0xba, 0xf3, 0x9d, 0x18 ),
	      AUTH ( 0xd0, 0xd1, 0xc8, 0xa7, 0x99, 0x99, 0x6b, 0xf0, 0x26,
		     0x5b, 0x98, 0xb5, 0xd4, 0x8a, 0xb9, 0x19 ) );

/** Test 15 */
CIPHER_TEST ( gcm_test_15, &aes_gcm_algorithm, GCM_KEY_256,
	      GCM_IV_96, GCM_ADDITIONAL_EMPTY, GCM_PLAINTEXT_512,
	      CIPHERTEXT ( 0x52, 0x2d, 0xc1, 0xf0, 0x99, 0x56, 0x7d, 0x07,
			   0xf4, 0x7f, 0x37, 0xa3, 0x2a, 0x84, 0x42, 0x7d,
			   0x64, 0x3a, 0x8c, 0xdc, 0xbf, 0xe5, 0xc0, 0xc9,
			   0x75, 0x98, 0xa2, 0xbd, 0x25, 0x55, 0xd1, 0xaa,
			   0x8c, 0xb0, 0x8e, 0x48, 0x59, 0x0d, 0xbb, 0x3d,
			   0xa7, 0xb0, 0x8b, 0x10, 0x56, 0x82, 0x88, 0x38,
			   0xc5, 0xf6, 0x1e, 0x63, 0x93, 0xba, 0x7a, 0x0a,
			   0xbc, 0xc9, 0xf6, 0x62, 0x89, 0x80, 0x15, 0xad ),
	      AUTH ( 0xb0, 0x94, 0xda, 0xc5, 0xd9, 0x34, 0x71, 0xbd, 0xec,
		     0x1a, 0x50, 0x22, 0x70, 0xe3, 0xcc, 0x6c ) );

/** Test 16 */
CIPHER_TEST ( gcm_test_16, &aes_gcm_algorithm, GCM_KEY_256,
	      GCM_IV_96, GCM_ADDITIONAL_160, GCM_PLAINTEXT_480,
	      CIPHERTEXT ( 0x52, 0x2d, 0xc1, 0xf0, 0x99, 0x56, 0x7d, 0x07,
			   0xf4, 0x7f, 0x37, 0xa3, 0x2a, 0x84, 0x42, 0x7d,
			   0x64, 0x3a, 0x8c, 0xdc, 0xbf, 0xe5, 0xc0, 0xc9,
			   0x75, 0x98, 0xa2, 0xbd, 0x25, 0x55, 0xd1, 0xaa,
			   0x8c, 0xb0, 0x8e, 0x48, 0x59, 0x0d, 0xbb, 0x3d,
			   0xa7, 0xb0, 0x8b, 0x10, 0x56, 0x82, 0x88, 0x38,
			   0xc5, 0xf6, 0x1e, 0x63, 0x93, 0xba, 0x7a, 0x0a,
			   0xbc, 0xc9, 0xf6, 0x62 ),
	      AUTH ( 0x76, 0xfc, 0x6e, 0xce, 0x0f, 0x4e, 0x17, 0x68, 0xcd,
		     0xdf, 0x88, 0x53, 0xbb, 0x2d, 0x55, 0x1b ) );

/** Test 17 */
CIPHER_TEST ( gcm_test_17, &aes_gcm_algorithm, GCM_KEY_256,
	      GCM_IV_64, GCM_ADDITIONAL_160, GCM_PLAINTEXT_480,
	      CIPHERTEXT ( 0xc3, 0x76, 0x2d, 0xf1, 0xca, 0x78, 0x7d, 0x32,
			   0xae, 0x47, 0xc1, 0x3b, 0xf1, 0x98, 0x44, 0xcb,
			   0xaf, 0x1a, 0xe1, 0x4d, 0x0b, 0x97, 0x6a, 0xfa,
			   0xc5, 0x2f, 0xf7, 0xd7, 0x9b, 0xba, 0x9d, 0xe0,
			   0xfe, 0xb5, 0x82, 0xd3, 0x39, 0x34, 0xa4, 0xf0,
			   0x95, 0x4c, 0xc2, 0x36, 0x3b, 0xc7, 0x3f, 0x78,
			   0x62, 0xac, 0x43, 0x0e, 0x64, 0xab, 0xe4, 0x99,
			   0xf4, 0x7c, 0x9b, 0x1f ),
	      AUTH ( 0x3a, 0x33, 0x7d, 0xbf, 0x46, 0xa7, 0x92, 0xc4, 0x5e,
		     0x45, 0x49, 0x13, 0xfe, 0x2e, 0xa8, 0xf2 ) );

/** Test 18 */
CIPHER_TEST ( gcm_test_18, &aes_gcm_algorithm, GCM_KEY_256,
	      GCM_IV_480, GCM_ADDITIONAL_160, GCM_PLAINTEXT_480,
	      CIPHERTEXT ( 0x5a, 0x8d, 0xef, 0x2f, 0x0c, 0x9e, 0x53, 0xf1,
			   0xf7, 0x5d, 0x78, 0x53, 0x65, 0x9e, 0x2a, 0x20,
			   0xee, 0xb2, 0xb2, 0x2a, 0xaf, 0xde, 0x64, 0x19,
			   0xa0, 0x58, 0xab, 0x4f, 0x6f, 0x74, 0x6b, 0xf4,
			   0x0f, 0xc0, 0xc3, 0xb7, 0x80, 0xf2, 0x44, 0x45,
			   0x2d, 0xa3, 0xeb, 0xf1, 0xc5, 0xd8, 0x2c, 0xde,
			   0xa2, 0x41, 0x89, 0x97, 0x20, 0x0e, 0xf8, 0x2e,
			   0x44, 0xae, 0x7e, 0x3f ),
	      AUTH ( 0xa4, 0x4a, 0x82, 0x66, 0xee, 0x1c, 0x8e, 0xb0, 0xc8,
		     0xb5, 0xd4, 0xcf, 0x5a, 0xe9, 0xf1, 0x9a ) );

/**
 * Perform Galois/Counter Mode self-test
 *
 */
static void gcm_test_exec ( void ) {
	struct cipher_algorithm *gcm = &aes_gcm_algorithm;
	unsigned int keylen;

	/* Correctness tests */
	cipher_ok ( &gcm_test_1 );
	cipher_ok ( &gcm_test_2 );
	cipher_ok ( &gcm_test_3 );
	cipher_ok ( &gcm_test_4 );
	cipher_ok ( &gcm_test_5 );
	cipher_ok ( &gcm_test_6 );
	cipher_ok ( &gcm_test_7 );
	cipher_ok ( &gcm_test_8 );
	cipher_ok ( &gcm_test_9 );
	cipher_ok ( &gcm_test_10 );
	cipher_ok ( &gcm_test_11 );
	cipher_ok ( &gcm_test_12 );
	cipher_ok ( &gcm_test_13 );
	cipher_ok ( &gcm_test_14 );
	cipher_ok ( &gcm_test_15 );
	cipher_ok ( &gcm_test_16 );
	cipher_ok ( &gcm_test_17 );
	cipher_ok ( &gcm_test_18 );

	/* Speed tests */
	for ( keylen = 128 ; keylen <= 256 ; keylen += 64 ) {
		DBG ( "AES-%d-GCM encryption required %ld cycles per byte\n",
		      keylen, cipher_cost_encrypt ( gcm, ( keylen / 8 ) ) );
		DBG ( "AES-%d-GCM decryption required %ld cycles per byte\n",
		      keylen, cipher_cost_decrypt ( gcm, ( keylen / 8 ) ) );
	}
}

/** Galois/Counter Mode self-test */
struct self_test gcm_test __self_test = {
	.name = "gcm",
	.exec = gcm_test_exec,
};
