/*
 * $Id: glue-gui-recorder.c,v 1.3 2009-10-15 11:47:09 potyra Exp $
 *
 * Copyright (C) 2003-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <assert.h>
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <limits.h>

#include "rd_types.h"
#include "glue-main.h"
#include "glue-png.h"

#include "glue-gui-recorder.h"

struct cpssp {
	unsigned int monitor_width;
	unsigned int monitor_height;
	unsigned int true_width;
	unsigned int true_height;
	unsigned int recording;

	uint8_t *screen_data;

	FILE *fp;
};

void *
gui_recorder_create
(
	unsigned int monitor_width,
	unsigned int monitor_height
)
{
	struct cpssp *cpssp;
	unsigned int i;
	uint8_t *val;

	cpssp = malloc(sizeof(*cpssp));
	assert(cpssp);

	cpssp->monitor_width = monitor_width;
	cpssp->monitor_height = monitor_height;

	cpssp->true_width = monitor_width;
	cpssp->true_height = monitor_height;

	cpssp->recording = 0;

	cpssp->fp = NULL;

	/* black and completely opaque */
	cpssp->screen_data = malloc(monitor_width * monitor_height * 4);
	assert(cpssp->screen_data);
	val = cpssp->screen_data;
	for (i = 0; i < monitor_width * monitor_height; i++) {
		*val++ = 0;
		*val++ = 0;
		*val++ = 0;
		*val++ = 0xff;
	}

	return cpssp;
}

void
gui_recorder_destroy(void *_cpssp)
{
	struct cpssp *cpssp = (struct cpssp *)_cpssp;

	assert(cpssp);
	assert(cpssp->screen_data);

	free(cpssp->screen_data);
	free(cpssp);
}

void
gui_recorder_screenshot(void *_cpssp)
{
	struct cpssp *cpssp = (struct cpssp *)_cpssp;
	char path[PATH_MAX];
	int ret;
	int nr;
	static int auto_nr = 0;

	/* don't overwrite screenshot, search for free entry */

	nr = auto_nr; /* So we can recognize if we're looping because
			 there are no more free filenames */
	do {
		auto_nr = (auto_nr + 1) % 1000;
		ret = snprintf(path, sizeof(path), 
			"%s/screenshot-%03d.png", basedir, auto_nr);
		assert(ret < sizeof(path));

		if (access(path, F_OK) != 0) {
			break;
		}
	} while (nr != auto_nr);
	nr = auto_nr;

	ret = snprintf(path, sizeof(path), 
			"%s/screenshot-%03d.png", basedir, nr);
	assert(ret < sizeof(path));

	png_write(
		(uint32_t *) cpssp->screen_data,
		cpssp->monitor_width, cpssp->monitor_height,
		0, 0, cpssp->true_width, cpssp->true_height,
		path);
}

void
gui_recorder_pixel_set(
	void *_cpssp,
	unsigned int x,
	unsigned int y,
	uint8_t r,
	uint8_t g,
	uint8_t b
)
{
	struct cpssp *cpssp = (struct cpssp *)_cpssp;
	unsigned int index;

	if (cpssp->monitor_width <= x
			|| cpssp->monitor_height <= y) return;

	index = (y * cpssp->monitor_width + x) * 4;

	cpssp->screen_data[index + 0] = r;
	cpssp->screen_data[index + 1] = g;
	cpssp->screen_data[index + 2] = b;

	if (cpssp->fp) {
		size_t ret;
		static const unsigned char id = RD_MON_UPD_ID;
		struct rd_mon_upd_data data;

		data.t = time_virt();
		data.x = x;
		data.y = y;
		data.r = r;
		data.b = b;
		data.g = g;

		ret = fwrite((const void *) &id, sizeof(id), 1, cpssp->fp);
		assert(ret == 1);
		ret = fwrite((const void *) &data, sizeof(data), 1, cpssp->fp);
		assert(ret == 1);

	}
}

static void
rec_monitor_set(struct cpssp *cpssp)
{
	unsigned char id = RD_MON_SET_ID;
	struct rd_mon_set_data data;

	if (cpssp->fp) {
		size_t ret;

		data.x = cpssp->true_width;
		data.y = cpssp->true_height;

		ret = fwrite((void*) &id, sizeof(id), 1, cpssp->fp);
		assert(ret == 1);
		ret = fwrite((void*) &data, sizeof(data), 1, cpssp->fp);
		assert(ret == 1);
	}
}

void
gui_recorder_size_set
(
	void *_cpssp,
	unsigned int width,
	unsigned int height
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (cpssp->monitor_width < width 
			|| cpssp->monitor_height < height) return;

	cpssp->true_width = width;
	cpssp->true_height = height;

	rec_monitor_set(cpssp);
}

void
gui_recorder_rec_set(void *_cpssp, unsigned int on)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (on && !cpssp->recording) {
		static const uint16_t id = RD_MAGIC_ID;
		static const unsigned char all_id = RD_MON_ALL_ID;
		static int auto_nr = 0;
		struct rd_mon_init_data init_data;
		char path[PATH_MAX];
		unsigned int nr;
		int ret;
		size_t sz;

		assert(!cpssp->fp);

		/* Don't overwrite records, search for free entry. */

		nr = auto_nr; /* So we can recognize if we're looping because
				 there are no more free filenames */
		do {
			auto_nr = (auto_nr + 1) % 1000;
			ret = snprintf(path, sizeof(path), 
					"%s/record-%03d", basedir, auto_nr);
			assert(ret < sizeof(path));
			if (access(path, F_OK) != 0) {
				break;
			}
		} while (nr != auto_nr);
		nr = auto_nr;

		ret = snprintf(path, sizeof(path), 
				"%s/record-%03d", basedir, nr);
		assert(ret < sizeof(path));

		cpssp->fp = fopen(path, "w");
		assert(cpssp->fp);

		sz = fwrite(&id, sizeof(id), 1, cpssp->fp);
		assert(sz == 1);

		rec_monitor_set(cpssp);

		init_data.t = time_virt();
		init_data.time_hz = TIME_HZ;
		init_data.max_width = cpssp->monitor_width;
		init_data.max_height = cpssp->monitor_height;
		sz = fwrite((const void *) &all_id, 
				sizeof(all_id), 
				1, 
				cpssp->fp);
		assert(sz == 1);
		sz = fwrite((void *) &init_data, 
				sizeof(init_data), 
				1, 
				cpssp->fp);
		assert(sz == 1);
		sz = fwrite((const void *) cpssp->screen_data,
				cpssp->monitor_width * cpssp->monitor_height * 4,
				1, cpssp->fp);
		assert(sz == 1);

		cpssp->recording = 1;

	} else if (!on && cpssp->recording) {
		assert(cpssp->fp);

		assert(!ferror(cpssp->fp));
		fclose(cpssp->fp);
		cpssp->fp = NULL;

		cpssp->recording = 0;
	}
}
