/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file monitor.c
 * \brief Call monitor handling
 */

#ifndef _WIN32_WINNT
#define _WIN32_WINNT 0x0501
#endif

#include <sys/types.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>

#include <ffgtk.h>

#ifndef G_OS_WIN32
#include <sys/socket.h>
#include <netdb.h>
#include <resolv.h>
#endif

#ifdef HAVE_LIBNOTIFY
#include <libnotify/notify.h>
#endif

#ifdef HAVE_APPINDICATOR
#include <libappindicator/app-indicator.h>
extern AppIndicator *psIndicator;
#endif

#define BUFFER_LENGTH	1024
#define IMAGE_SIZE  100

//#define DEBUG_MONITOR

/** connection structure */
struct sConnection {
	int nId;
	int nType;
	gchar *pnLocal;
	gchar *pnNumber;
	gchar *pnName;
	GtkWidget *psDialWindow;
	GtkWidget *psPopup;
};

/** Global connect list pointer */
static GList *psConnectionList = NULL;
static guint nPopupTimeout = 0;

/** Ignore the following number */
static gchar *pnIgnore = NULL;

/**
 * \brief Add connection structure to connect list
 * \param psList list to add entry
 * \param nId connection id
 * \return new list pointer
 */
static GList *connectionAddCall( GList *psList, int nId, int nType, const gchar *pnLocal, const gchar *pnNumber, gchar *pnName ) {
	struct sConnection *psConnection = g_malloc0( sizeof( struct sConnection ) );

	if ( psConnection != NULL ) {
		psConnection -> nId = nId;
		psConnection -> nType = nType;
		psConnection -> pnLocal = g_strdup( pnLocal );
		psConnection -> pnNumber = g_strdup( pnNumber );
		if ( pnName != NULL ) {
			psConnection -> pnName = g_strdup( pnName );
		} else {
			psConnection -> pnName = NULL;
		}
		psConnection -> psDialWindow = NULL;

		psList = g_list_append( psList, psConnection );
	}

	return psList;
}

/**
 * \brief Find connection entry by id
 * \param nId connection id
 * \return connection pointer or NULL on error
 */
static struct sConnection *connectionFindById( int nId ) {
	GList *psList = psConnectionList;
	struct sConnection *psConnection;

	while ( psList ) {
		psConnection = psList -> data;

		if ( psConnection != NULL && psConnection -> nId == nId ) {
			return psConnection;
		}

		psList = psList -> next;
	}

	return NULL;
}

/**
 * \brief Find connection entry by number
 * \param pnNumber connection number
 * \return connection pointer or NULL on error
 */
struct sConnection *connectionFindByNumber( const gchar *pnNumber ) {
	GList *psList = psConnectionList;
	struct sConnection *psConnection;

	while ( psList ) {
		psConnection = psList -> data;

		Debug( KERN_DEBUG, "%s <-> %s\n", psConnection ? psConnection -> pnNumber : "?", pnNumber );
		if ( psConnection != NULL && !strcmp( psConnection -> pnNumber, pnNumber ) ) {
			return psConnection;
		}

		psList = psList -> next;
	}

	return NULL;
}

/**
 * \brief Add dialing window to connection information structure
 * \param psConnection connection structure
 * \param psWidget dialing window
 */
void connectionAddDialWindow( struct sConnection *psConnection, GtkWidget *psWidget ) {
	if ( psConnection != NULL ) {
		psConnection -> psDialWindow = psWidget;
	}
}

/**
 * \brief Remove connection from connection list
 * \param psList glist pointer
 * \param psConnection connection pointer
 * \return new psList pointer
 */
static GList *connectionRemove( GList *psList, struct sConnection *psConnection ) {
	if ( psList != NULL && psConnection != NULL ) {
		psList = g_list_remove( psList, psConnection );
		g_free( psConnection -> pnLocal );
		g_free( psConnection -> pnNumber );
		if ( psConnection -> pnName ) {
			g_free( psConnection -> pnName );
		}
		g_free( psConnection );
	}

	return psList;
}

/**
 * \brief Ignore next monitor messages
 * \param psConnection connection structure
 * */
void connectionIgnore( struct sConnection *psConnection ) {
	if ( psConnection == NULL ) {
		Debug( KERN_DEBUG, "Warning, connection == NULL\n" );
		return;
	}

	if ( pnIgnore != NULL ) {
		g_free( pnIgnore );
		pnIgnore = NULL;
	}
	pnIgnore = g_strdup( psConnection -> pnNumber );  

	psConnectionList = connectionRemove( psConnectionList, psConnection );
	Debug( KERN_DEBUG, "Connection removed!\n" );
}

/**
 * \brief Close notification window
 * \param pData popup window
 * \return FALSE
 */
static gboolean closeNotification( gpointer pData ) {
	struct sConnection *psConnection = pData;

	if ( nPopupTimeout != 0 ) {
		g_source_remove( nPopupTimeout );
		nPopupTimeout = 0;
	}

	if ( psConnection -> psPopup != NULL ) {
		gtk_widget_destroy( GTK_WIDGET( psConnection -> psPopup ) );
		psConnection -> psPopup = NULL;
	}
	return FALSE;
}

#ifdef HAVE_FAXOPHONE
/**
 * \brief Accept incoming call
 * \param psWidget button widget
 * \param pData popup widget pointer
 */
void acceptClicked( GtkWidget *psWidget, gpointer pData ) {
	gchar *pnNumber = g_object_get_data( G_OBJECT( pData ), "number" );

	/* Set dialog information */
	dialSetNumber( pnNumber );
	dialingDialog( pnNumber, _( "Softphone" ) );

	/* Remove popup timeout */
	if ( nPopupTimeout != 0 ) {
		g_source_remove( nPopupTimeout );
		nPopupTimeout = 0;
	}

	/* Destroy popup widget */
	gtk_widget_destroy( GTK_WIDGET( pData ) );
	/* Stop ring tone */
	StopRingtone();

	/* Ignore next monitor messages */
	connectionIgnore( connectionFindByNumber( pnNumber ) );

	/* Pickup call */
	softphonePickup();
}

/**
 * \brief Deny incoming call
 * \param psWidget button widget
 * \param pData popup widget pointer
 */
void denyClicked( GtkWidget *psWidget, gpointer pData ) {
	softphoneHangup();
	g_source_remove( nPopupTimeout );
	gtk_widget_destroy( GTK_WIDGET( pData ) );
}
#endif

/**
 * \brief Show popup call notification (FFGTK)
 * \param nType call type
 * \param psPerson person structure
 * \param pnNumber remote number
 * \param pnLocal local number
 * \return popup window
 */
GtkWidget *showNotificationFfgtk( struct sConnection *psConnection, eCallType nType, struct sPerson *psPerson, gchar *pnNumber, gchar *pnLocal ) {
	GtkBuilder *psBuilder = NULL;
	GError *psError = NULL;
	GtkWidget *psDialog;
	GtkWidget *psFrame;
	gint nWidth, nHeight;
	gint nPosition = callMonitorGetPopupPosition( getActiveProfile() );
	GtkWidget *psTypeLabel;
	GtkWidget *psLocalLabel;
	GtkWidget *psDirectionLabel;
	GtkWidget *psNameLabel;
	GtkWidget *psCompanyLabel;
	GtkWidget *psStreetLabel;
	GtkWidget *psCityLabel;
	GtkWidget *psImage;
	GtkWidget *psAcceptButton;
	GtkWidget *psDenyButton;
	gchar *pnUiFile = getUiFile( "popup.ui" );
	GdkPixbuf *psScaled;
	gchar *pnTmp = NULL;
	gint nMenuBarOffset = 0;

	psBuilder = gtk_builder_new();
	if ( gtk_builder_add_from_file( psBuilder, pnUiFile, &psError ) == 0 ) {
		Debug( KERN_WARNING, "Error: %s\n", psError -> message );
		g_error_free( psError );
		g_free( pnUiFile );
		return NULL;
	}
	g_free( pnUiFile );

	psDialog = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psPopup" ) );
	psFrame = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psFrame" ) );
	psTypeLabel = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psTypeLabel" ) );
	psLocalLabel = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psLocalLabel" ) );
	psDirectionLabel = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psDirectionLabel" ) );
	psNameLabel = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psNameLabel" ) );
	psCompanyLabel = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psCompanyLabel" ) );
	psStreetLabel = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psStreetLabel" ) );
	psCityLabel = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psCityLabel" ) );
	psImage = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psImage" ) );
	psAcceptButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psAcceptButton" ) );
	psDenyButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psDenyButton" ) );

#ifdef HAVE_FAXOPHONE
	g_signal_connect( G_OBJECT( psAcceptButton ), "clicked", G_CALLBACK( acceptClicked ), psDialog );
	g_signal_connect( G_OBJECT( psDenyButton ), "clicked", G_CALLBACK( denyClicked ), psDialog );
#else
	gtk_widget_set_sensitive( GTK_WIDGET( psAcceptButton ), FALSE );
	gtk_widget_set_sensitive( GTK_WIDGET( psDenyButton ), FALSE );
#endif

	g_object_set_data( G_OBJECT( psDialog ), "type_label", psTypeLabel );
	g_object_set_data( G_OBJECT( psDialog ), "direction_label", psDirectionLabel );
	g_object_set_data( G_OBJECT( psDialog ), "name_label", psNameLabel );
	g_object_set_data( G_OBJECT( psDialog ), "street_label", psStreetLabel );
	g_object_set_data( G_OBJECT( psDialog ), "city_label", psCityLabel );
	g_object_set_data( G_OBJECT( psDialog ), "number" , pnNumber ? strdup( pnNumber ) : strdup( "" ) );
	g_object_set_data( G_OBJECT( psDialog ), "person" , psPerson );

	if ( nType == CALL_TYPE_INCOMING ) {
		gtk_label_set_markup( GTK_LABEL( psTypeLabel ), _( "<b>Incoming call</b>" ) );
		if ( pnLocal != NULL && strlen( pnLocal ) > 0 ) {
			pnTmp = g_strdup_printf( _( "(on %s)" ), pnLocal );
		} else {
			pnTmp = g_strdup_printf( _( "(on ?)" ) );
		}
		gtk_label_set_text( GTK_LABEL( psLocalLabel ), pnTmp );
		g_free( pnTmp );
		gtk_label_set_markup( GTK_LABEL( psDirectionLabel ), _( "<i>From:</i>" ) );

		gtk_widget_show( GTK_WIDGET( psAcceptButton ) );
		gtk_widget_show( GTK_WIDGET( psDenyButton ) );
	} else {
		gtk_label_set_markup( GTK_LABEL( psTypeLabel ), _( "<b>Outgoing call</b>" ) );
		gtk_label_set_markup( GTK_LABEL( psDirectionLabel ), _( "<i>To:</i>" ) );
	}

	gtk_label_set_text( GTK_LABEL( psStreetLabel ), _( "Unknown" ) );
	gtk_label_set_text( GTK_LABEL( psCityLabel ), _( "Unknown" ) );

	if ( psPerson == NULL ) {
		gtk_label_set_text( GTK_LABEL( psNameLabel ), pnNumber );
	} else {
		gtk_label_set_text( GTK_LABEL( psNameLabel ), psPerson -> pnDisplayName );
		gtk_label_set_text( GTK_LABEL( psCompanyLabel ), psPerson -> pnCompany );
		gtk_label_set_text( GTK_LABEL( psStreetLabel ), psPerson -> pnPrivateStreet );
		gchar *pnTmp = g_strdup_printf( "%s %s", psPerson -> pnPrivateZipCode, psPerson -> pnPrivateCity );
		gtk_label_set_text( GTK_LABEL( psCityLabel ), pnTmp );
		g_free( pnTmp );

		psScaled = getScaledImage( psPerson, 2 );
		if ( psScaled != NULL ) {
			gtk_image_set_from_pixbuf( GTK_IMAGE( psImage ), psScaled );
		}
	}

	gtk_widget_show_all( GTK_WIDGET( psFrame ) );

	gtk_window_get_size( GTK_WINDOW( psDialog ), &nWidth, &nHeight );

#ifdef __MACOSX__
	nMenuBarOffset = 24;
#endif

	switch ( nPosition ) {
		case 0:
			/* Top Left */
			gtk_window_move( GTK_WINDOW( psDialog ), 0, nMenuBarOffset );
			break;
		case 1:
			/* Top Right */
			gtk_window_move( GTK_WINDOW( psDialog ), gdk_screen_width() - nWidth, nMenuBarOffset );
			break;
		case 2:
			/* Bottom Right */
			gtk_window_move( GTK_WINDOW( psDialog ), gdk_screen_width() - nWidth, gdk_screen_height() - nHeight );
			break;
		case 3:
			/* Bottom Left */
			gtk_window_move( GTK_WINDOW( psDialog ), 0, gdk_screen_height() - nHeight );
			break;
	}

	gtk_window_stick( GTK_WINDOW( psDialog ) );
	gtk_window_set_keep_above( GTK_WINDOW( psDialog ), TRUE );

	gtk_builder_connect_signals( psBuilder, NULL );

	g_object_unref( G_OBJECT( psBuilder ) );

	/* Timeout sec * 1000 */
	psConnection -> psPopup = psDialog;
	if ( nType == CALL_TYPE_OUTGOING ) {
		nPopupTimeout = g_timeout_add( callMonitorGetDisplayTime( getActiveProfile() ) * 1000, closeNotification, psConnection );
	}

	gtk_widget_show_all( GTK_WIDGET( psDialog ) );

	return psDialog;
}

#ifdef HAVE_LIBNOTIFY
/**
 * \brief Show popup call notification (System)
 * \param nType call type
 * \param psPerson person structure
 * \param pnNumber remote number
 * \param pnLocal local number
 * \return popup window
 */
NotifyNotification *showNotificationSystem( eCallType nType, struct sPerson *psPerson, gchar *pnNumber, gchar *pnLocal ) {
#ifndef HAVE_APPINDICATOR
	GdkScreen *psScreen = NULL;
	GdkRectangle sArea;
#endif
	gchar *pnText = NULL;
	NotifyNotification *psTooltip = NULL;

#if NOTIFY_VERSION_MINOR >= 7
	notify_init( "ffgtk" );
#endif

	Debug( KERN_DEBUG, "System...\n" );

	if ( nType == CALL_TYPE_INCOMING ) {
		pnText = g_markup_printf_escaped( _( "<i>From:</i> %s" ), psPerson ? psPerson -> pnDisplayName : pnNumber );
	} else {
		pnText = g_markup_printf_escaped( _( "<i>To:</i> %s" ), psPerson ? psPerson -> pnDisplayName : pnNumber );
	}

	if ( psPerson != NULL && psPerson -> psImage != NULL ) {
		psTooltip = notify_notification_new( nType == CALL_TYPE_INCOMING ? _( "Incoming call" ) : _( "Outgoing call" ), pnText, NULL
#if NOTIFY_VERSION_MINOR < 7
	, NULL
#endif
	);
		notify_notification_set_icon_from_pixbuf( psTooltip, psPerson -> psImage );
	} else {
		psTooltip = notify_notification_new( nType == CALL_TYPE_INCOMING ? _( "Incoming call" ) : _( "Outgoing call" ), pnText,
			nType == CALL_TYPE_INCOMING ? /*SHAREDIR*/ "notification-message-ffgtk-in.svg" : /*SHAREDIR*/ "notification-message-ffgtk-out.svg"
#if NOTIFY_VERSION_MINOR < 7
			, NULL
#endif
		);
	}

	notify_notification_set_category( psTooltip, "information" );
	notify_notification_set_timeout( psTooltip, callMonitorGetDisplayTime( getActiveProfile() ) * 1000 );
	notify_notification_set_urgency( psTooltip, NOTIFY_URGENCY_NORMAL );
#if NOTIFY_VERSION_MINOR < 7
	notify_notification_attach_to_status_icon( psTooltip, getTrayIcon() );
#endif

#ifndef HAVE_APPINDICATOR
	gtk_status_icon_get_geometry( getTrayIcon(), &psScreen, &sArea, NULL );
#if NOTIFY_VERSION_MINOR < 7
	notify_notification_set_geometry_hints( psTooltip, psScreen, sArea.x, sArea.y );
#endif
#endif
	notify_notification_show( psTooltip, NULL );

	g_free( pnText );

	return psTooltip;
}
#endif

/**
 * \brief Show popup call notification
 * \param nType call type
 * \param psPerson person structure
 * \param pnNumber remote number
 * \param pnLocal local number
 * \return popup window
 */
void *showNotification( struct sConnection *psConnection, eCallType nType, struct sPerson *psPerson, gchar *pnNumber, gchar *pnLocal ) {
#ifdef HAVE_LIBNOTIFY
	Debug( KERN_DEBUG, "System: %d, Ffgtk: %d\n", callMonitorGetSystemNotification( getActiveProfile() ),
	      callMonitorGetFfgtkNotification( getActiveProfile() ) );
	if ( callMonitorGetSystemNotification( getActiveProfile() ) == TRUE ) {
		return showNotificationSystem( nType, psPerson, pnNumber, pnLocal );
	}
#endif

	return showNotificationFfgtk( psConnection, nType, psPerson, pnNumber, pnLocal );
}

#ifdef HAVE_LIBNOTIFY
/**
 * \brief Update person information in popup window (System)
 * \param psPopup popup window
 * \param nType call type
 * \param pnName remote name
 * \param pnStreet remote street
 * \param pnCity remote city
 */
static void updateNotificationSystem( NotifyNotification *psTooltip, eCallType nType, gchar *pnName, gchar *pnStreet, gchar *pnCity ) {
	gchar *pnText = NULL;

	pnText = g_markup_printf_escaped( nType == CALL_TYPE_INCOMING ? _( "<i>From:</i> %s" ) : _( "<i>To:</i> %s" ), pnName );
	notify_notification_update( psTooltip, nType == CALL_TYPE_INCOMING ? _( "Incoming call" ) : _( "Outgoing call" ), pnText, "gtk-dialog-info" );

	notify_notification_show( psTooltip, NULL );
	g_free( pnText );
}
#endif

/**
 * \brief Update person information in popup window (FFGTK)
 * \param psPopup popup window
 * \param nType call type
 * \param pnName remote name
 * \param pnStreet remote street
 * \param pnCity remote city
 */
static void updateNotificationFfgtk( GtkWidget *psPopup, eCallType nType, gchar *pnName, gchar *pnStreet, gchar *pnCity ) {
	GtkWidget *psTypeLabel = g_object_get_data( G_OBJECT( psPopup ), "type_label" );
	GtkWidget *psDirectionLabel = g_object_get_data( G_OBJECT( psPopup ), "direction_label" );
	GtkWidget *psNameLabel = g_object_get_data( G_OBJECT( psPopup ), "name_label" );
	GtkWidget *psStreetLabel = g_object_get_data( G_OBJECT( psPopup ), "street_label" );
	GtkWidget *psCityLabel = g_object_get_data( G_OBJECT( psPopup ), "city_label" );
	gint nWidth, nHeight;
	gint nPosition = callMonitorGetPopupPosition( getActiveProfile() );

	gtk_widget_hide( GTK_WIDGET( psPopup ) );

	gtk_label_set_text( GTK_LABEL( psTypeLabel ), nType == CALL_TYPE_INCOMING ? _( "Incoming call" ) : _( "Outgoing call" ) );
	gtk_label_set_text( GTK_LABEL( psDirectionLabel ), nType == CALL_TYPE_INCOMING ? _( "From:" ) : _( "To:" ) );
	gtk_label_set_text( GTK_LABEL( psNameLabel ), pnName );
	if ( pnStreet != NULL ) {
		gtk_label_set_text( GTK_LABEL( psStreetLabel ), pnStreet );
	}
	if ( pnCity != NULL ) {
		gtk_label_set_text( GTK_LABEL( psCityLabel ), pnCity );
	}
	g_object_set_data( G_OBJECT( psPopup ), "name" , pnName ? strdup( pnName ) : strdup( "" ) );

	gtk_window_get_size( GTK_WINDOW( psPopup ), &nWidth, &nHeight );

	switch ( nPosition ) {
		case 0:
			gtk_window_move( GTK_WINDOW( psPopup ), 0, 0 );
			break;
		case 1:
			gtk_window_move( GTK_WINDOW( psPopup ), gdk_screen_width() - nWidth, 0 );
			break;
		case 2:
			gtk_window_move( GTK_WINDOW( psPopup ), gdk_screen_width() - nWidth, gdk_screen_height() - nHeight );
			break;
		case 3:
			gtk_window_move( GTK_WINDOW( psPopup ), 0, gdk_screen_height() - nHeight );
			break;
	}

	gtk_widget_show_all( GTK_WIDGET( psPopup ) );
}

/**
 * \brief Update person information in popup window
 * \param psPopup popup window
 * \param nType call type
 * \param pnName remote name
 * \param pnStreet remote street
 * \param pnCity remote city
 */
static void updateNotification( void *psPopup, eCallType nType, gchar *pnName, gchar *pnStreet, gchar *pnCity ) {
#ifdef HAVE_LIBNOTIFY
	if ( callMonitorGetSystemNotification( getActiveProfile() ) == TRUE ) {
		return updateNotificationSystem( psPopup, nType, pnName, pnStreet, pnCity );
	}
#endif

	return updateNotificationFfgtk( psPopup, nType, pnName, pnStreet, pnCity );
}

/**
 * \brief Handle call
 * \param nType call type
 * \param nConnection current connection id
 * \param pnRemote remote number
 * \param pnLocal local number
 * \param pnMedium transfer medium
 */
static void HandleCall( eCallType nType, int nConnection, char *pnNumber, char *pnLocal, char *pnMedium ) {
	gchar *pnFirstName;
	gchar *pnLastName;
	gchar *pnStreet;
	gchar *pnZipCode;
	gchar *pnCity;
	struct sConnection *psConnection = NULL;
	struct sPerson *psPerson = NULL;
	struct sProfile *psProfile = getActiveProfile();
	const gchar *pnLineAccessCode = routerGetLineAccessCode( psProfile );
	gint nDisplayTime = callMonitorGetDisplayTime( psProfile );

	Debug( KERN_DEBUG, "nType %d, nConnection: %d\n", nType, nConnection );
	/* Skip line access code */
	if ( pnLineAccessCode != NULL && strlen( pnLineAccessCode ) > 0 ) {
		pnNumber = pnNumber + strlen( pnLineAccessCode );
	}

	/* In case we want to fast dial, remove last # */
	if ( pnNumber[ strlen( pnNumber ) - 1 ] == '#' ) {
		pnNumber[ strlen( pnNumber ) - 1 ] = '\0';
	}

	if ( nType == CALL_TYPE_INCOMING || nType == CALL_TYPE_OUTGOING ) {
		if ( nType == CALL_TYPE_INCOMING && pnIgnore != NULL && !strcmp( pnIgnore, pnNumber ) ) {
			Debug( KERN_DEBUG, "Ignoring monitor event\n" );
			lastCallAddNumber( 0, pnIgnore );

			g_free( pnIgnore );
			pnIgnore = NULL;

			goto updateList;
		}

		Debug( KERN_DEBUG, "Find person by number\n" );
		psPerson = findPersonByNumber( pnNumber );

		/* Add call to connection list */
		psConnectionList = connectionAddCall( psConnectionList, nConnection, nType, pnLocal, pnNumber, psPerson ? psPerson -> pnDisplayName : NULL );

		Debug( KERN_DEBUG, "Checking preferences state for '%s'\n", pnLocal );
		if ( ( callMonitorGetIncomingState( psProfile, pnLocal ) && ( nType == CALL_TYPE_INCOMING ) ) || ( callMonitorGetOutgoingState( psProfile, pnLocal ) && ( nType == CALL_TYPE_OUTGOING ) ) ) {
			PlayRingtone( nType );

			if ( nDisplayTime != 0 ) {
				Debug( KERN_DEBUG, "Show popup\n" );
				psConnection = connectionFindById( nConnection );

				psConnection -> psPopup = showNotification( psConnection, nType, psPerson, pnNumber, pnLocal );

				if ( psConnection -> psPopup != NULL && psPerson == NULL &&
					( ( nType == CALL_TYPE_INCOMING && callMonitorGetReverseLookupIn( psProfile ) ) ||
					 ( nType == CALL_TYPE_OUTGOING && callMonitorGetReverseLookupOut( psProfile ) ) ) ) {
					ReverseLookup( pnNumber, &pnFirstName, &pnLastName, &pnStreet, &pnZipCode, &pnCity );
					if ( pnLastName != NULL ) {
						/* TODO: add code to handle new fields */
						updateNotification( psConnection -> psPopup, nType, pnLastName, pnStreet, pnCity );
						if ( pnStreet != NULL ) {
							g_free( pnStreet );
						}
						if ( pnCity != NULL ) {
							g_free( pnCity );
						}
					}
				}
			}
		}

		if ( nType == CALL_TYPE_OUTGOING && dialGetNumber() != NULL && !strcmp( dialGetNumber(), pnNumber ) ) {
			Debug( KERN_DEBUG, "Add connection to dial window\n" );
			connectionAddDialWindow( connectionFindById( nConnection ), dialGetDialog() );
		}
	}

	/* execute actions */
	Debug( KERN_DEBUG, "Find connection by id\n" );
	psConnection = connectionFindById( nConnection );
	if ( psConnection != NULL ) {
		Debug( KERN_DEBUG, "Found connection, if needed execute actions\n" );
		executeActions( psConnection -> nType, nType, psConnection -> pnLocal, psConnection -> pnNumber, psConnection -> pnName );
	}

	if ( nType == CALL_TYPE_CONNECT || nType == CALL_TYPE_DISCONNECT ) {
		if ( psConnection != NULL ) {
			closeNotification( psConnection );

			/* In case we used the dial out function, update status label of dialing dialog */
			if ( ( psConnection -> nType & ~CALL_TYPE_INCOMING ) && dialGetNumber() != NULL && !strcmp( dialGetNumber(), psConnection -> pnNumber ) ) {
				if ( nType == CALL_TYPE_DISCONNECT || nType == CALL_TYPE_CONNECT ) {
					dialWindowSetStatus( psConnection -> psDialWindow, nType );
				}
			}

			if ( nType == CALL_TYPE_DISCONNECT ) {
				Debug( KERN_DEBUG, "Type: %x\n", psConnection -> nType );

				if ( psConnection -> nType == CALL_TYPE_INCOMING ) {
					/* if we missed that call, set icon to missed */
#ifndef HAVE_APPINDICATOR
					gtk_status_icon_set_from_pixbuf( GTK_STATUS_ICON( getTrayIcon() ), getTypeIcon( 2 ) );
#else
					app_indicator_set_icon( psIndicator, "callinfailed" );
#endif
					lastCallAddNumber( 2, psConnection -> pnNumber );
				} else if ( psConnection -> nType & CALL_TYPE_OUTGOING ) {
					lastCallAddNumber( 1, psConnection -> pnNumber );
				} else if ( psConnection -> nType & CALL_TYPE_CONNECT ) {
					lastCallAddNumber( 0, psConnection -> pnNumber );
				}

				psConnectionList = connectionRemove( psConnectionList, psConnection );

				updateList:
#ifdef HAVE_APPINDICATOR
				recreateAppMenu(/* psConnection -> nType == CALL_TYPE_INCOMING */);
#endif

#ifndef DEBUG_MONITOR
				//g_add_full( G_PRIORITY_DEFAULT, ( GSourceFunc ) updateCallList, NULL, NULL );
				CREATE_THREAD( "calllist", updateCallList, NULL );
#endif
				psConnectionList = psConnectionList;
			} else {
				psConnection -> nType |= nType;
				Debug( KERN_DEBUG, "Type set to %x\n", psConnection -> nType );
			}
		}
	}
}

/**
 * \brief Convert fritzbox call line to a readable output
 * \param pnText input line
 * \return error code
 */
int convert( char *pnText ) {
	int anPos[ 8 ];
	int nIndex;
	char *pnDate;
	char *pnType;
	char *pnConnection;
	char *pnPartA;
	char *pnPartB;
	char *pnPartC;
	char *pnPartD;
	int nConnection;

	memset( anPos, -1, sizeof( anPos ) );

	for ( nIndex = 1; nIndex < 8; nIndex++ ) {
		anPos[ nIndex ] = findString( pnText, anPos[ nIndex - 1 ] + 1, ";" );
		if ( anPos[ nIndex ] == -1 ) {
			break;
		}
	}
	anPos[ 0 ] = 0;

	pnDate = getSubString( pnText, anPos[ 0 ], anPos[ 1 ] - anPos[ 0 ] );
	if ( pnDate == NULL ) {
		Debug( KERN_DEBUG, "Could not get date string\n" );
		goto error1;
	}

	pnType = getSubString( pnText, anPos[ 1 ] + 1 , anPos[ 2 ] - anPos[ 1 ] - 1 );
	pnConnection = getSubString( pnText, anPos[ 2 ] + 1, anPos[ 3 ] - anPos[ 2 ] - 1 );
	nConnection = atoi( pnConnection );

	pnPartA = anPos[ 4 ] == -1 ? NULL : getSubString( pnText, anPos[ 3 ] + 1, anPos[ 4 ] - anPos[ 3 ] - 1 );
	pnPartB = anPos[ 5 ] == -1 ? NULL : getSubString( pnText, anPos[ 4 ] + 1, anPos[ 5 ] - anPos[ 4 ] - 1 );
	pnPartC = anPos[ 6 ] == -1 ? NULL : getSubString( pnText, anPos[ 5 ] + 1, anPos[ 6 ] - anPos[ 5 ] - 1 );
	pnPartD = anPos[ 7 ] == -1 ? NULL : getSubString( pnText, anPos[ 6 ] + 1, anPos[ 7 ] - anPos[ 6 ] - 1 );

	if ( !strcmp( pnType, "CALL" ) ) {
		/**
		 * pnPartA : box port
		 * pnPartB : caller id
		 * pnPartC : called id
		 * pnPartD : POTS, SIP, ISDN, ...
		 */
		 HandleCall( CALL_TYPE_OUTGOING, nConnection, pnPartC, pnPartB, pnPartD );
	} else if ( !strcmp( pnType, "RING" ) ) {
		/**
		 * pnPartA : caller id
		 * pnPartB : called id
		 * pnPartC : medium
		 */
		HandleCall( CALL_TYPE_INCOMING, nConnection, pnPartA, pnPartB, pnPartC );
	} else if ( !strcmp( pnType, "CONNECT" ) ) {
		/**
		 * pnPartA : caller id
		 * pnPartB : id
		 */
		HandleCall( CALL_TYPE_CONNECT, nConnection, pnPartA, pnPartB, pnPartC );
	} else if ( !strcmp( pnType, "DISCONNECT" ) ) {
		/**
		 * pnPartA : caller id
		 * pnPartB : id
		 */
		HandleCall( CALL_TYPE_DISCONNECT, nConnection, pnPartA, pnPartB, pnPartC );
	}

	if ( pnPartD ) {
		g_free( pnPartD );
	}
	if ( pnPartC ) {
		g_free( pnPartC );
	}
	if ( pnPartB ) {
		g_free( pnPartB );
	}
	if ( pnPartA ) {
		g_free( pnPartA );
	}
	g_free( pnConnection );
	g_free( pnType );
	g_free( pnDate );

error1:

	return 0;
}

/**
 * \brief Initialize server and wait for incoming transmissions
 * \param data data pointer
 * \return error code
 */
gpointer telnetMonitor( gpointer data ) {
	struct addrinfo *psAi;
	int nSock, nError;
	char anBuffer[ BUFFER_LENGTH ];
	ssize_t nReadResult;
	fd_set sFds;
	struct timeval sTimeout;
	int nResult;
	const gchar *pnHostName = NULL;

	while ( 1 ) {
		nSock = -1;

		if ( getActiveProfile() == NULL ) {
			g_usleep( 2 * G_USEC_PER_SEC );
			continue;
		}

		pnHostName = routerGetHost( getActiveProfile() );

		psAi = NULL;
		nError = getaddrinfo( pnHostName, NULL, NULL, &psAi );
		if ( nError != 0 ) {
			Debug( KERN_WARNING, _( "Error on hostname %s: %s\n" ), pnHostName, gai_strerror( nError ) );
			g_usleep( 5 * G_USEC_PER_SEC );
#ifndef G_OS_WIN32
			res_init();
#endif
			goto end;
		}

		nSock = socket( psAi -> ai_family, psAi -> ai_socktype, psAi -> ai_protocol );
		if ( nSock < 0 ) {
			Debug( KERN_WARNING, _( "could not open socket, abort\n" ) );
			g_usleep( 30 * G_USEC_PER_SEC );
			goto end;
		}

		( ( struct sockaddr_in * )( psAi -> ai_addr ) ) -> sin_port = htons( 1012 );

		if ( ( connect( nSock, psAi -> ai_addr, psAi -> ai_addrlen ) ) < 0 ) {
			Debug( KERN_WARNING, _( "could not bind socket, abort\n" ) );
			g_usleep( 30 * G_USEC_PER_SEC );
			goto end;
		}
		Debug( KERN_DEBUG, "Connected to '%s' on port 1012\n", pnHostName );

		while ( 1 ) {
			FD_ZERO( &sFds );
			FD_SET( nSock, &sFds );

			sTimeout.tv_sec = 2;
			sTimeout.tv_usec = 0;

			nResult = select( nSock + 1, &sFds, NULL, NULL, &sTimeout );
			if ( nResult < 0 ) {
				break;
			}
			if ( nResult == 0 ) {
				if ( getActiveProfile() != NULL && !strcmp( routerGetHost( getActiveProfile() ), pnHostName ) ) {
					continue;
				}
				Debug( KERN_DEBUG, "Reconnect needed!\n" );
				close( nSock );
				nSock = -1;
				break;
			}

			memset( anBuffer, 0, BUFFER_LENGTH );
			nReadResult = read( nSock, anBuffer, BUFFER_LENGTH );
			if ( nReadResult > 0 || ( nReadResult == -1 && errno == EINTR ) ) {
				if ( nReadResult > 0 ) {
					gdk_threads_enter();

					gchar **ppnLines = g_strsplit( anBuffer, "\n", -1 );
					if ( ppnLines != NULL ) {
						gint nCount = 0;
						while ( ppnLines[ nCount ] != NULL ) {
							Debug( KERN_DEBUG, "Convert line: %s\n", ppnLines[ nCount ] );
							convert( ppnLines[ nCount ] );
							nCount++;
						}
						g_strfreev( ppnLines );
					}
					gdk_threads_leave();
				} else {
					g_usleep( 1 * G_USEC_PER_SEC );
				}
			} else {
				break;
			}
		}

end:
		if ( nSock >= 0 ) {
			close( nSock );
			g_usleep( 30 * G_USEC_PER_SEC );
		}
	}

	return NULL;
}

/**
 * \brief Starts call monitor thread
 */
void InitMonitor( void ) {
	CREATE_THREAD( "telnet", telnetMonitor, NULL );
}
