/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file router.c
 * \brief General router functions, which sends commands to the real router
 * interface.
 */

#include <ffgtk.h>

#ifdef HAVE_APPINDICATOR
#include <libappindicator/app-indicator.h>
extern AppIndicator *psIndicator;
#endif

/** Global diversity list */
GList *psDiversityList = NULL;

/**
 * \brief Get router used within the given profile
 * \param psProfile profile containing the router information
 * \return router structure
 */
static struct sRouter *getRouter( struct sProfile *psProfile ) {
	struct sRouter *psRouter = NULL;
	int nType;

	if ( psProfile == NULL ) {
		return NULL;
	}

	nType = prefsGetInt( psProfile, "/ffgtk/router/type" );

	switch ( nType ) {
		case 0:
			psRouter = &sFritzBox;
			break;
		case 1:
			psRouter = &sSpeedport;
			break;
		case 2:
			psRouter = &sAliceBox;
			break;
		default:
			Debug( KERN_WARNING, "Unknown router!\n" );
			break;
	}

	return psRouter;
}

/**
 * \brief Show login error message
 * \param bThread in thread flag
 * \param nSleep seconds to sleep
 */
void RouterLoginError( gboolean bThread, gint nSleep ) {
	gchar anMessage[ 64 ];

	if ( nSleep > 0 ) {
		snprintf( anMessage, sizeof( anMessage ), _( "Wrong password! Wait for %d seconds" ), nSleep );
		ShowError( bThread, _( "Login failed" ), anMessage );
		g_usleep( nSleep * G_USEC_PER_SEC );
	}
}

/**
 * \brief Toggle router lock state
 * \param psProfile profile structure
 * \param bLock lock flag
 */
void routerLock( struct sProfile *psProfile, gboolean bLock ) {
	psProfile -> bLocked = bLock;
}

/**
 * \brief Initialize router structure
 * \param psProfile profile containing the router information
 * \return error code (0=success, else error)
 */
int routerInit( struct sProfile *psProfile ) {
	struct sRouter *psRouter = getRouter( psProfile );

	/* If we have a init function, call it */
	if ( psRouter != NULL && psRouter -> Init != NULL ) {
		return psRouter -> Init( psProfile );
	}

	return -1;

}

/**
 * \brief Login to router
 * \param psProfile profile containing the router information
 * \return error code (0=success, else error)
 */
int routerLogin( struct sProfile *psProfile ) {
	struct sRouter *psRouter = getRouter( psProfile );

	if ( psProfile -> bLocked == TRUE ) {
		Debug( KERN_DEBUG, "Loggin LOCKED!! Abort\n" );
		return 33485;
	}

	/* If we have a login function, call it */
	if ( psRouter != NULL && psRouter -> Login != NULL ) {
		return psRouter -> Login( psProfile );
	}
	Debug( KERN_DEBUG, "Failed!!\n" );

	return -1;
}

/**
 * \brief Logout from router
 * \param psProfile profile containing the router information
 * \return error code (0=success, else error)
 */
int routerLogout( struct sProfile *psProfile ) {
	struct sRouter *psRouter = getRouter( psProfile );

	/* If we have a logout function, call it */
	if ( psRouter != NULL && psRouter -> Logout != NULL ) {
		return psRouter -> Logout( psProfile );
	}

	return -1;
}

/**
 * \brief Detect router firmware
 * \param psProfile profile containing the router information
 * \return error code (0=success, else error)
 */
int routerFirmwareDetect( struct sProfile *psProfile ) {
	struct sRouter *psRouter = getRouter( psProfile );

	/* If we have a detect firmware function, call it */
	if ( psRouter != NULL && psRouter -> DetectFirmware != NULL ) {
		return psRouter -> DetectFirmware( psProfile );
	}

	return -1;
}

/**
 * \brief Get call list
 * \param psProfile profile containing the router information
 * \return error code (0=success, else error)
 */
int routerGetCallList( struct sProfile *psProfile ) {
	struct sRouter *psRouter = getRouter( psProfile );

	/* If we have a get call list function, call it */
	if ( psRouter != NULL && psRouter -> GetCallList != NULL ) {
		return psRouter -> GetCallList( psProfile, getCallerList( psProfile ) );
	}

	return -1;
}

/**
 * \brief Clear call list
 * \param psProfile profile containing the router information
 * \return error code (0=success, else error)
 */
int routerCallListClear( struct sProfile *psProfile ) {
	struct sRouter *psRouter = getRouter( psProfile );

	/* If we have a clear calllist function, call it */
	if ( psRouter != NULL && psRouter -> ClearCallList != NULL ) {
		return psRouter -> ClearCallList( psProfile );
	}

	return -1;
}

/**
 * \brief Get Phone Settings
 * \param psProfile profile containing the router information
 * \return error code (0=success, else error)
 */
int routerGetPhoneSettings( struct sProfile *psProfile ) {
	struct sRouter *psRouter = getRouter( psProfile );

	if ( psRouter != NULL && psRouter -> GetPhoneSettings != NULL ) {
		return psRouter -> GetPhoneSettings( psProfile );
	}

	return -1;
}

/**
 * \brief Get active ports
 * \param psProfile profile containing the router information
 * \return error code (0=success, else error)
 */
int routerGetActivePorts( struct sProfile *psProfile ) {
	struct sRouter *psRouter = getRouter( psProfile );

	/* if we have a get ports function, call it */
	if ( psRouter != NULL && psRouter -> GetPorts != NULL ) {
		return psRouter -> GetPorts( psProfile );
	}

	return -1;

}

/**
 * \brief Call number
 * \param psProfile profile containing the router information
 * \param pnNumber number we want to call
 * \param pnPort port name we want to use
 * \return error code (0=success, else error)
 */
int routerCallNumber( struct sProfile *psProfile, const gchar *pnNumber, const gchar *pnPort ) {
	struct sRouter *psRouter = getRouter( psProfile );

	/* if we have a call number function, call it */
	if ( psRouter != NULL && pnNumber != NULL && psRouter -> CallNumber != NULL ) {
		return psRouter -> CallNumber( psProfile, pnNumber, routerGetPortType( psProfile, pnPort ) );
	}

	return -1;
}

/**
 * \brief Check for dial function
 * \param psProfile profile containing the router information
 * \return true or false
 */
gboolean routerHasDial( struct sProfile *psProfile ) {
	struct sRouter *psRouter = getRouter( psProfile );

	/* if we have a call number function, call it */
	if ( psRouter != NULL && psRouter -> CallNumber != NULL ) {
		return true;
	}

	return false;
}

/**
 * \brief Hang up
 * \param psProfile profile containing the router information
 * \param pnPort port name we want to hangup
 * \return error code (0=success, else error)
 */
int routerHangup( struct sProfile *psProfile, const gchar *pnPort ) {
	struct sRouter *psRouter = getRouter( psProfile );

	/* if we have a hangup function, call it */
	if ( psRouter != NULL && pnPort != NULL && psRouter -> Hangup != NULL ) {
		return psRouter -> Hangup( psProfile, routerGetPortType( psProfile, pnPort ) );
	}

	return -1;

}

/**
 * \brief Reconnect internet connection
 * \param psProfile profile containing the router information
 * \return error code (0=success, else error)
 */
int routerReconnect( struct sProfile *psProfile ) {
	struct sRouter *psRouter = getRouter( psProfile );

	/* if we have a reconnect function, call it */
	if ( psRouter != NULL && psRouter -> Reconnect != NULL ) {
		return psRouter -> Reconnect( psProfile );
	}

	return -1;
}

/**
 * \brief Get external ip address
 * \param psProfile profile containing the router information
 * \return ip address or NULL on error
 */
gchar *routerGetIp( struct sProfile *psProfile ) {
	struct sRouter *psRouter = getRouter( psProfile );

	/* if we have a get ip function, call it */
	if ( psRouter != NULL && psRouter -> GetIp != NULL ) {
		return psRouter -> GetIp( psProfile );
	}

	return NULL;
}

/**
 * \brief Get max. up/down speed
 * \param psProfile profile containing the router information
 * \param pnMaxUp maximal up speed pointer
 * \param pnMaxDown maximual down speed pointer
 */
void routerGetSpeed( struct sProfile *psProfile, gint *pnMaxUp, gint *pnMaxDown ) {
	struct sRouter *psRouter = getRouter( psProfile );

	/* if we have a get speed function, call it */
	if ( psRouter != NULL && psRouter -> GetSpeed != NULL ) {
		psRouter -> GetSpeed( psProfile, pnMaxUp, pnMaxDown );
	}
}

/**
 * \brief Get router name
 * \param psProfile profile containing the router information
 * \return router name or NULL on error
 */
const gchar *routerGetName( struct sProfile *psProfile ) {
	struct sRouter *psRouter = getRouter( psProfile );

	/* if we have a get name function, call it */
	if ( psRouter != NULL && psRouter -> GetName != NULL ) {
		return psRouter -> GetName( psProfile );
	}

	return NULL;
}

/**
 * \brief Get router version (firmware)
 * \param psProfile profile containing the router information
 * \return router version or NULL on error
 */
const gchar *routerGetVersion( struct sProfile *psProfile ) {
	struct sRouter *psRouter = getRouter( psProfile );

	/* if we have a get version function, call it */
	if ( psRouter != NULL && psRouter -> GetVersion != NULL ) {
		return psRouter -> GetVersion( psProfile );
	}

	return NULL;
}

/**
 * \brief Read diversity
 * \param psProfile profile containing the router information
 * \return error code
 */
gint routerReadDiversity( struct sProfile *psProfile ) {
	struct sRouter *psRouter = getRouter( psProfile );

	/* if we have a read diversity function, call it */
	if ( psRouter != NULL && psRouter -> ReadDiversity != NULL ) {
		return psRouter -> ReadDiversity( psProfile );
	}

	return 0;
}

/**
 * \brief Set diversity
 * \param psProfile profile containing the router information
 * \param nIndex diversity index
 * \param bSet set flag
 * \return error code
 */
gint routerSetDiversity( struct sProfile *psProfile, gint nIndex, gboolean bSet ) {
	struct sRouter *psRouter = getRouter( psProfile );

	/* if we have a set diversity function, call it */
	if ( psRouter != NULL && psRouter -> SetDiversity != NULL ) {
		return psRouter -> SetDiversity( psProfile, nIndex, bSet );
	}

	return -1;
}

/**
 * \brief Load diversity thread
 * \param pData UNUSED
 * \return FALSE
 */
static gpointer loadDiversity( gpointer pData ) {
	gint nError = -1;

	/* Try to read as long as there is data available */
	while ( 1 ) {
		nError = routerReadDiversity( getActiveProfile () );
		if ( nError == 0 ) {
#ifdef HAVE_APPINDICATOR
			recreateAppMenu();
#endif
			break;
		}

		if ( nError < 0 ) {
			RouterLoginError( TRUE, -nError );
		} else if ( nError != 33485 ) {
			Debug( KERN_DEBUG, "%s: %s\n", _( "Network error" ), curl_easy_strerror( nError ) );
		}
		g_usleep( 5 * G_USEC_PER_SEC );
	}

	return NULL;
}

/**
 * \brief Starts load diversity thread, called by main()
 */
void DiversityLoad( void ) {
	CREATE_THREAD( "diversity", loadDiversity, NULL );
	//g_idle_add_full( G_PRIORITY_LOW, loadDiversity, NULL, NULL );
}

/**
 * \brief Get global diversity list
 * \return list pointer
 */
GList *routerGetDiversity( void ) {
	return psDiversityList;
}

/**
 * \brief Load voice box from router
 * \param psProfile profile structure
 * \return error code
 */
int routerLoadVoiceBox( struct sProfile *psProfile ) {
	struct sRouter *psRouter = getRouter( psProfile );

	/* if we have a load voice box function, call it */
	if ( psRouter != NULL && psRouter -> LoadVoiceBox != NULL ) {
		return psRouter -> LoadVoiceBox( psProfile );
	}

	return 0;
}

/**
 * \brief Load a single voice box file from router
 * \param psProfile profile structure
 * \param pnFile file name we want to load
 * \return error code
 */
int routerLoadVoiceFile( struct sProfile *psProfile, const gchar *pnFile ) {
	struct sRouter *psRouter = getRouter( psProfile );

	/* if we have a load voice file function, call it */
	if ( psRouter != NULL && psRouter -> LoadVoiceFile != NULL ) {
		return psRouter -> LoadVoiceFile( psProfile, pnFile );
	}

	return 0;
}

/**
 * \brief Delete voice box file from router
 * \param psProfile profile structure
 * \param pnFile voice box file we want to delete
 * \return error code
 */
int routerDeleteVoiceFile( struct sProfile *psProfile, const gchar *pnFile ) {
	struct sRouter *psRouter = getRouter( psProfile );

	/* if we have a delete voice file function, call it */
	if ( psRouter != NULL && psRouter -> DeleteVoiceFile != NULL ) {
		return psRouter -> DeleteVoiceFile( psProfile, pnFile );
	}

	return 0;
}

/**
 * \brief Load fax box from router
 * \param psProfile profile structure
 * \return error code
 */
int routerLoadFaxBox( struct sProfile *psProfile ) {
	struct sRouter *psRouter = getRouter( psProfile );

	/* if we have a load fax box function, call it */
	if ( psRouter != NULL && psRouter -> LoadFaxBox != NULL ) {
		return psRouter -> LoadFaxBox( psProfile );
	}

	return 0;
}

/**
 * \brief Load a single fax box file from router
 * \param psProfile profile structure
 * \param pnFile file name we want to load
 * \return error code
 */
int routerLoadFaxFile( struct sProfile *psProfile, const gchar *pnFile ) {
	struct sRouter *psRouter = getRouter( psProfile );

	/* if we have a load fax file function, call it */
	if ( psRouter != NULL && psRouter -> LoadFaxFile != NULL ) {
		return psRouter -> LoadFaxFile( psProfile, pnFile );
	}

	return 0;
}

/**
 * \brief Delete fax box file from router
 * \param psProfile profile structure
 * \param pnFile fax box file we want to delete
 * \return error code
 */
int routerDeleteFaxFile( struct sProfile *psProfile, const gchar *pnFile ) {
	struct sRouter *psRouter = getRouter( psProfile );

	/* if we have a delete fax file function, call it */
	if ( psRouter != NULL && psRouter -> DeleteFaxFile != NULL ) {
		return psRouter -> DeleteFaxFile( psProfile, pnFile );
	}

	return 0;
}
