/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file trayicon.c
 * \brief Trayicon related functions
 */

#include <ffgtk.h>

#ifdef HAVE_APPINDICATOR
#include <libappindicator/app-indicator.h>
AppIndicator *psIndicator = NULL;
#endif

/** Widget containing trayicon */
static GtkStatusIcon *psTrayIcon = NULL;
/** trayicon open state */
static bool bOpen = FALSE;
/** main gtk loop */
extern GMainLoop *psLoop;
static GtkWidget *psMenu = NULL;

/**
 * \brief Set try icon state
 * \param bSet set state
 */
void setTrayOpen( bool bSet ) {
	bOpen = bSet;
}

/**
 * \brief Destroy callback
 * \param psWidget widget
 * \param pData optional data pointer
 */
static void destroy( GtkWidget *psWidget, gpointer pData ) {
	g_main_quit( psLoop );
}

/**
 * \brief Called on tray icon activated, opens or close CallMonitor
 */
static void trayIconActivated( void ) {
	static GtkWidget *psMonitor = NULL;

	if ( bOpen == false ) {
		psMonitor = CallMonitor();
		if ( psMonitor != NULL ) {
			bOpen = TRUE;
		}
	} else {
		if ( psMonitor != NULL ) {
			gtk_widget_hide( GTK_WIDGET( psMonitor ) );
		}
		bOpen = FALSE;
	}
	
#ifndef HAVE_APPINDICATOR
	gtk_status_icon_set_from_pixbuf( GTK_STATUS_ICON( getTrayIcon() ), getTypeIcon( 3 ) );
#else
	app_indicator_set_icon( psIndicator, "ffgtk" );
#endif
}

/**
 * \brief Open addressbook - callback
 */
static void trayIconAddressBook( void ) {
	AddressBook( 0 );
}

/**
 * \brief Select active profile callback
 * \param psItem menu item pressed
 * \param pUserData name of profile
 */
static void trayIconProfile( GtkMenuItem *psItem, gpointer pUserData ) {
	if ( gtk_check_menu_item_get_active( GTK_CHECK_MENU_ITEM( psItem ) ) == TRUE ) {
		const gchar *pnName = pUserData;
		struct sProfile *psProfile = findProfile( pnName );
		setActiveProfile( psProfile );
		saveProfiles();
	}
}

/**
 * \brief Dial number - callback
 * \param psWidget menu item widget
 * \param pnNumber dial number
 */
static void trayIconDialNumber( GtkWidget *psWidget, gchar *pnNumber ) {
	struct sPerson *psPerson = NULL;

	if ( pnNumber != NULL ) {
		psPerson = findPersonByNumber( pnNumber );
	}

	dialNumberDialog( psPerson, pnNumber );
}

/**
 * \brief Call router reconnect - callback
 */
static void trayIconReconnect( void ) {
	routerReconnect( getActiveProfile() );
}

/**
 * \brief Copy external ip to clipboard - callback
 */
void trayIconIpToClipboard( void ) {
	gchar *pnIp = routerGetIp( getActiveProfile() );

	if ( pnIp != NULL ) {
		gtk_clipboard_set_text( gtk_clipboard_get( GDK_NONE ), pnIp, -1 );
		g_free( pnIp );
	}
}

/**
 * \brief Mute sounds - callback
 * \param psItem menu item widget
 * \param pUserData UNUSED
 */
static void trayIconMute( GtkCheckMenuItem *psItem, gpointer pUserData ) {
	gboolean bMute = callMonitorGetMute( getActiveProfile() );

	gtk_check_menu_item_set_active( psItem, !bMute );
	callMonitorSetMute( getActiveProfile(), !bMute );
}

/**
 * \brief Create separator item with 'text'
 * \param pnText separator text
 * \return new menuitem widget
 */
GtkWidget *createSeparatorItem( gchar *pnText ) {
/*#ifdef HAVE_APPINDICATOR
	GtkWidget *psItem = NULL;
	psItem = gtk_menu_item_new_with_label( pnText );
#else*/
	GtkWidget *psLabel = NULL;
	GtkWidget *psBox = gtk_hbox_new( FALSE, 0 );
	GtkWidget *psItem = NULL;

	psItem = gtk_menu_item_new();

	psLabel = gtk_label_new( pnText );

	gtk_box_pack_start( GTK_BOX( psBox ), gtk_hseparator_new(), TRUE, TRUE, 0 );
	gtk_box_pack_start( GTK_BOX( psBox ), psLabel, FALSE, FALSE, 0 );
	gtk_box_pack_start( GTK_BOX( psBox ), gtk_hseparator_new(), TRUE, TRUE, 0 );

	g_object_set( G_OBJECT( psItem ), "child", psBox, "sensitive", FALSE, NULL );
//#endif

	return psItem;
}

/**
 * \brief Create call menu item for type nType
 * \param pnNumber call number
 * \param nType call type
 * \return new menuitem widget
 */
GtkWidget *createCallItem( char *pnNumber, int nType ) {
/*#ifdef HAVE_APPINDICATOR
	GtkWidget *psItem = NULL;

	psItem = gtk_menu_item_new_with_label( pnNumber );
#else*/
	GtkWidget *psLabel = NULL;
	GtkWidget *psNameLabel = NULL;
	GtkWidget *psBox = gtk_vbox_new( FALSE, 0 );
	GtkWidget *psItem = NULL;
	GtkWidget *psImage = NULL;
	GdkPixbuf *psPix = NULL;
	struct sPerson *psPerson = findPersonByNumber( pnNumber );
	gchar *pnName;
	gchar *pnNumberText;

	psItem = gtk_image_menu_item_new();

	psLabel = gtk_label_new( "" );
	pnNumberText = g_markup_printf_escaped( "<small>%s</small>", pnNumber );
	gtk_label_set_markup( GTK_LABEL( psLabel ), pnNumberText );
	g_free( pnNumberText );

	psNameLabel = gtk_label_new( "" );
	pnName = g_markup_printf_escaped( "<b><small>%s</small></b>", psPerson != NULL ? psPerson -> pnDisplayName : _( "unknown" ) );
	gtk_label_set_markup( GTK_LABEL( psNameLabel ), pnName );
	g_free( pnName );

	gtk_box_pack_start( GTK_BOX( psBox ), psNameLabel, TRUE, TRUE, 0 );
	gtk_box_pack_start( GTK_BOX( psBox ), psLabel, FALSE, FALSE, 0 );
	g_object_set( G_OBJECT( psItem ), "child", psBox, NULL );

	gtk_misc_set_alignment( GTK_MISC( psNameLabel ), 0.0, 0.5 );
	gtk_misc_set_alignment( GTK_MISC( psLabel ), 0.0, 0.5 );

	psPix = getTypeIcon( nType );
	psImage = GTK_WIDGET( gtk_image_new_from_pixbuf( psPix ) );
	gtk_image_menu_item_set_image( GTK_IMAGE_MENU_ITEM( psItem ), psImage );
	gtk_image_menu_item_set_always_show_image( GTK_IMAGE_MENU_ITEM( psItem ), TRUE );
//#endif

	return psItem;
}

/**
 * \brief Add last call to submenu - callback
 * \param psMenu connect each entry to this menu
 */
void addLastcalls( GtkWidget *psMenu ) {
	GList *psListIn = NULL;
	GList *psListOut = NULL;
	GList *psListMissed = NULL;
	GtkWidget *psItem = NULL;
	GtkWidget *psSubMenu = NULL;

	psItem = gtk_image_menu_item_new_with_label( _( "Last calls" ) );
	psSubMenu = gtk_menu_new();
	GtkWidget *psLastCalls = GTK_WIDGET( gtk_image_new_from_stock( GTK_STOCK_JUMP_TO, GTK_ICON_SIZE_MENU ) );
	gtk_image_menu_item_set_image( GTK_IMAGE_MENU_ITEM( psItem ), psLastCalls );
	gtk_menu_item_set_submenu( GTK_MENU_ITEM( psItem ), psSubMenu );
	gtk_menu_shell_append( GTK_MENU_SHELL( psMenu ), psItem );

	psListIn = getLastcalls( 0 );
	psListOut = getLastcalls( 1 );
	psListMissed = getLastcalls( 2 );

	if ( psListIn == NULL && psListOut == NULL && psListMissed == NULL ) {
		gtk_widget_set_sensitive( GTK_WIDGET( psItem ), FALSE );
		return;
	}

	if ( psListIn != NULL ) {
		psItem = createSeparatorItem( _( "Incoming" ) );
		gtk_menu_shell_append( GTK_MENU_SHELL( psSubMenu ), psItem );

		while ( psListIn != NULL ) {
			psItem = createCallItem( psListIn -> data, 1 );
			gtk_menu_shell_append( GTK_MENU_SHELL( psSubMenu ), psItem );
			g_signal_connect( G_OBJECT( psItem ), "activate", G_CALLBACK( trayIconDialNumber ), psListIn -> data );
			gtk_widget_set_sensitive( GTK_WIDGET( psItem ), routerHasDial( getActiveProfile() ) );
			psListIn = psListIn -> next;
		}
	}

	if ( psListOut != NULL ) {
		psItem = createSeparatorItem( _( "Outgoing" ) );
		gtk_menu_shell_append( GTK_MENU_SHELL( psSubMenu ), psItem );

		while ( psListOut != NULL ) {
			psItem = createCallItem( psListOut -> data, 3 );
			gtk_menu_shell_append( GTK_MENU_SHELL( psSubMenu ), psItem );
			g_signal_connect( G_OBJECT( psItem ), "activate", G_CALLBACK( trayIconDialNumber ), psListOut -> data );
			gtk_widget_set_sensitive( GTK_WIDGET( psItem ), routerHasDial( getActiveProfile() ) );
			psListOut = psListOut -> next;
		}
	}

	if ( psListMissed != NULL ) {
		psItem = createSeparatorItem( _( "Missed" ) );
		gtk_menu_shell_append( GTK_MENU_SHELL( psSubMenu ), psItem );

		while ( psListMissed != NULL ) {
			psItem = createCallItem( psListMissed -> data, 2 );

			gtk_menu_shell_append( GTK_MENU_SHELL( psSubMenu ), psItem );
			g_signal_connect( G_OBJECT( psItem ), "activate", G_CALLBACK( trayIconDialNumber ), psListMissed -> data );
			gtk_widget_set_sensitive( GTK_WIDGET( psItem ), routerHasDial( getActiveProfile() ) );
			psListMissed = psListMissed -> next;
		}
	}
}

/**
 * \brief Set diversity state - callback
 * \param psItem menu item widget
 * \param psDiversity diversity structure
 */
static void trayIconSetDiversity( GtkMenuItem *psItem, struct sDiversity *psDiversity ) {
	psDiversity -> bSet = !psDiversity -> bSet;

	gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM( psItem ), psDiversity -> bSet );
	routerSetDiversity( getActiveProfile(), psDiversity -> nIndex, psDiversity -> bSet );
}

#ifndef G_OS_WIN32
/**
 * \brief Send SMS - callback
 * \param psItem menu item widget
 * \param pUserData UNUSED
 */
static void trayIconSendSms( GtkMenuItem *psItem, gpointer pUserData ) {
	sendSmsDialog();
}
#endif

/**
 * \brief Open help - callback
 * \param psItem menu item widget
 * \param psDiversity diversity structure
 */
static void trayIconHelp( GtkMenuItem *psItem, struct sDiversity *psDiversity ) {
	osExecute( "http://www.tabos.org/forum" );
}

/**
 * \brief Add diversity information to submenu
 * \param psSubMenu add all information to this submenu
 */
static void addDiversity( GtkWidget *psSubMenu ) {
	struct sDiversity *psDiversity = NULL;
	GList *psList = NULL;
	GtkWidget *psItem = NULL;

	psItem = gtk_image_menu_item_new_with_label( _( "Diversity" ) );
	GtkWidget *psSubMenu3 = gtk_menu_new();
	GtkWidget *psDiversityImage = GTK_WIDGET( gtk_image_new_from_stock( GTK_STOCK_NETWORK, GTK_ICON_SIZE_MENU ) );
	gtk_image_menu_item_set_image( GTK_IMAGE_MENU_ITEM( psItem ), psDiversityImage );
	gtk_menu_item_set_submenu( GTK_MENU_ITEM( psItem ), psSubMenu3 );
	gtk_menu_shell_append( GTK_MENU_SHELL( psSubMenu ), psItem );

	psList = routerGetDiversity();
	if ( psList == NULL || g_list_length( psList ) == 0 ) {
		gtk_widget_set_sensitive( psItem, FALSE );
		return;
	}

	for ( psList = routerGetDiversity(); psList != NULL && psList -> data != NULL; psList = psList -> next ) {
		psDiversity = psList -> data;

		psItem = gtk_check_menu_item_new_with_label( psDiversity -> pnName );
		gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM( psItem ), psDiversity -> bSet );
		gtk_menu_shell_append( GTK_MENU_SHELL( psSubMenu3 ), psItem );
		g_signal_connect( G_OBJECT( psItem ), "activate", G_CALLBACK( trayIconSetDiversity ), psDiversity );
	}
}

static GtkWidget *createMenu( void ) {
	GtkWidget *psItem;
	GtkWidget *psMenu;

	psMenu = gtk_menu_new();

	/* Callmonitor */
	psItem = gtk_image_menu_item_new_with_label( _( "Call Monitor" ) );
	GtkWidget *psMonitor = GTK_WIDGET( gtk_image_new_from_stock( GTK_STOCK_ADD, GTK_ICON_SIZE_MENU ) );
	gtk_image_menu_item_set_image( GTK_IMAGE_MENU_ITEM( psItem ), psMonitor );
	gtk_menu_shell_append( GTK_MENU_SHELL( psMenu ), psItem );
	g_signal_connect( G_OBJECT( psItem ), "activate", G_CALLBACK( trayIconActivated ), NULL );

	/* Addressbook */
	psItem = gtk_image_menu_item_new_with_label( _( "Address Book" ) );
	GtkWidget *psAddress = GTK_WIDGET( gtk_image_new_from_stock( GTK_STOCK_ADD, GTK_ICON_SIZE_MENU ) );
	gtk_image_menu_item_set_image( GTK_IMAGE_MENU_ITEM( psItem ), psAddress );
	gtk_menu_shell_append( GTK_MENU_SHELL( psMenu ), psItem );
	g_signal_connect( G_OBJECT( psItem ), "activate", G_CALLBACK( trayIconAddressBook ), NULL );

	/* Last calls */
	addLastcalls( psMenu );

	/* Functions */
	GtkWidget *psFunctions = gtk_image_menu_item_new_with_label( _( "Functions" ) );
	GtkWidget *psSubMenu1 = gtk_menu_new();
	GtkWidget *psFunctionsImage = GTK_WIDGET( gtk_image_new_from_stock( GTK_STOCK_EXECUTE, GTK_ICON_SIZE_MENU ) );
	gtk_image_menu_item_set_image( GTK_IMAGE_MENU_ITEM( psFunctions ), psFunctionsImage );

	/* Functions - Call number */
	psItem = gtk_image_menu_item_new_with_label( _( "Call number" ) );
	GtkWidget *psCallMonitor = GTK_WIDGET( gtk_image_new_from_pixbuf( getTypeIcon( 3 ) ) );
	gtk_image_menu_item_set_image( GTK_IMAGE_MENU_ITEM( psItem ), psCallMonitor );
	gtk_menu_shell_append( GTK_MENU_SHELL( psSubMenu1 ), psItem );
	g_signal_connect( G_OBJECT( psItem ), "activate", G_CALLBACK( trayIconDialNumber ), NULL );
	gtk_widget_set_sensitive( GTK_WIDGET( psItem ), routerHasDial( getActiveProfile() ) );

	/* Functions - Reconnect */
	psItem = gtk_image_menu_item_new_with_label( _( "Reconnect" ) );
	GtkWidget *psReconnect = GTK_WIDGET( gtk_image_new_from_stock( GTK_STOCK_REFRESH, GTK_ICON_SIZE_MENU ) );
	gtk_image_menu_item_set_image( GTK_IMAGE_MENU_ITEM( psItem ), psReconnect );
	gtk_menu_shell_append( GTK_MENU_SHELL( psSubMenu1 ), psItem );
	g_signal_connect( G_OBJECT( psItem ), "activate", G_CALLBACK( trayIconReconnect ), NULL );

	/* Functions - Copy IP */
	psItem = gtk_image_menu_item_new_with_label( _( "Copy IP" ) );
	GtkWidget *psClipboard = GTK_WIDGET( gtk_image_new_from_stock( GTK_STOCK_COPY, GTK_ICON_SIZE_MENU ) );
	gtk_image_menu_item_set_image( GTK_IMAGE_MENU_ITEM( psItem ), psClipboard );
	gtk_menu_shell_append( GTK_MENU_SHELL( psSubMenu1 ), psItem );
	g_signal_connect( G_OBJECT( psItem ), "activate", G_CALLBACK( trayIconIpToClipboard ), NULL );

#ifndef G_OS_WIN32
	/* Functions - Send SMS */
	psItem = gtk_image_menu_item_new_with_label( _( "Send SMS" ) );
	GtkWidget *psSendSms = GTK_WIDGET( gtk_image_new_from_stock( GTK_STOCK_EDIT, GTK_ICON_SIZE_MENU ) );
	gtk_image_menu_item_set_image( GTK_IMAGE_MENU_ITEM( psItem ), psSendSms );
	gtk_menu_shell_append( GTK_MENU_SHELL( psSubMenu1 ), psItem );
	g_signal_connect( G_OBJECT( psItem ), "activate", G_CALLBACK( trayIconSendSms ), NULL );
	gtk_widget_set_sensitive( GTK_WIDGET( psItem ), routerHasDial( getActiveProfile() ) );
#endif

	/* Functions - Diversity */
	addDiversity( psSubMenu1 );

	gtk_menu_item_set_submenu( GTK_MENU_ITEM( psFunctions ), psSubMenu1 );
	gtk_menu_shell_append( GTK_MENU_SHELL( psMenu ), psFunctions );

	/* Line */
	psItem = gtk_separator_menu_item_new();
	gtk_menu_shell_append( GTK_MENU_SHELL( psMenu ), psItem );

	/* Preferences */
	psItem = gtk_image_menu_item_new_with_label( _( "Preferences" ) );
	GtkWidget *psPref = GTK_WIDGET( gtk_image_new_from_stock( GTK_STOCK_PREFERENCES, GTK_ICON_SIZE_MENU ) );
	gtk_image_menu_item_set_image( GTK_IMAGE_MENU_ITEM( psItem ), psPref );
	gtk_menu_shell_append( GTK_MENU_SHELL( psMenu ), psItem );
	g_signal_connect( G_OBJECT( psItem ), "activate", G_CALLBACK( preferences ), NULL );

	/* Profiles */
	GtkWidget *psProfilesMenu = gtk_image_menu_item_new_with_label( _( "Profiles" ) );
	GtkWidget *psSubMenu = gtk_menu_new();
	GtkWidget *psCall = GTK_WIDGET( gtk_image_new_from_stock( GTK_STOCK_INDEX, GTK_ICON_SIZE_MENU ) );
	gtk_image_menu_item_set_image( GTK_IMAGE_MENU_ITEM( psProfilesMenu ), psCall );

	/* Profiles - Manage */
	psItem = gtk_image_menu_item_new_with_label( _( "Manage" ) );
	GtkWidget *psManage = GTK_WIDGET( gtk_image_new_from_stock( GTK_STOCK_INDEX, GTK_ICON_SIZE_MENU ) );
	gtk_image_menu_item_set_image( GTK_IMAGE_MENU_ITEM( psItem ), psManage );
	gtk_menu_shell_append( GTK_MENU_SHELL( psSubMenu ), psItem );
	g_signal_connect( G_OBJECT( psItem ), "activate", G_CALLBACK( ProfilesManage ), NULL );

	/* Profiles - Line */
	psItem = gtk_separator_menu_item_new();
	gtk_menu_shell_append( GTK_MENU_SHELL( psSubMenu ), psItem );

	/* Profiles - Profile list */
	GList *psList = getProfiles();
	struct sProfile *psProfile = NULL;
	psItem = NULL;
	GSList *psGroup = NULL;

	while ( psList != NULL ) {
		psProfile = psList -> data;
		if ( psProfile != NULL ) {
			psItem = gtk_radio_menu_item_new_with_label( psGroup, psProfile -> pnName );
			psGroup = gtk_radio_menu_item_get_group( GTK_RADIO_MENU_ITEM( psItem ) );
			if ( psProfile == getActiveProfile() ) {
				gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM( psItem ), TRUE );
			}
			gtk_menu_shell_append( GTK_MENU_SHELL( psSubMenu ), psItem );
			g_signal_connect( G_OBJECT( psItem ), "activate", G_CALLBACK( trayIconProfile ), psProfile -> pnName );
		}
		psList = psList -> next;
	}

	gtk_menu_item_set_submenu( GTK_MENU_ITEM( psProfilesMenu ), psSubMenu );
	gtk_menu_shell_append( GTK_MENU_SHELL( psMenu ), psProfilesMenu );

	/* Line */
	psItem = gtk_separator_menu_item_new();
	gtk_menu_shell_append( GTK_MENU_SHELL( psMenu ), psItem );

	/* Mute */
	psItem = gtk_check_menu_item_new_with_label( _( "Mute" ) );
	gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM( psItem ), callMonitorGetMute( getActiveProfile() ) );
	gtk_menu_shell_append( GTK_MENU_SHELL( psMenu ), psItem );
	g_signal_connect( G_OBJECT( psItem ), "activate", G_CALLBACK( trayIconMute ), NULL );

	/* Line */
	psItem = gtk_separator_menu_item_new();
	gtk_menu_shell_append( GTK_MENU_SHELL( psMenu ), psItem );

	/* Help */
	psItem = gtk_image_menu_item_new_with_label( _( "Help" ) );
	GtkWidget *psHelp = GTK_WIDGET( gtk_image_new_from_stock( GTK_STOCK_HELP, GTK_ICON_SIZE_MENU ) );
	gtk_image_menu_item_set_image( GTK_IMAGE_MENU_ITEM( psItem ), psHelp );
	gtk_menu_shell_append( GTK_MENU_SHELL( psMenu ), psItem );
	g_signal_connect( G_OBJECT( psItem ), "activate", G_CALLBACK( trayIconHelp ), NULL );

	/* About */
	psItem = gtk_image_menu_item_new_with_label( _( "About" ) );
	GtkWidget *psAbout = GTK_WIDGET( gtk_image_new_from_stock( GTK_STOCK_ABOUT, GTK_ICON_SIZE_MENU ) );
	gtk_image_menu_item_set_image( GTK_IMAGE_MENU_ITEM( psItem ), psAbout );
	gtk_menu_shell_append( GTK_MENU_SHELL( psMenu ), psItem );
	g_signal_connect( G_OBJECT( psItem ), "activate", G_CALLBACK( aboutDialog ), NULL );

	/* Line */
	psItem = gtk_separator_menu_item_new();
	gtk_menu_shell_append( GTK_MENU_SHELL( psMenu ), psItem );

	/* Quit */
	psItem = gtk_image_menu_item_new_with_label( _( "Quit" ) );
	GtkWidget *psQuit = GTK_WIDGET( gtk_image_new_from_stock( GTK_STOCK_QUIT, GTK_ICON_SIZE_MENU ) );
	gtk_image_menu_item_set_image( GTK_IMAGE_MENU_ITEM( psItem ), psQuit );
	gtk_menu_shell_append( GTK_MENU_SHELL( psMenu ), psItem );
	g_signal_connect( G_OBJECT( psItem ), "activate", G_CALLBACK( destroy ), NULL );

	return psMenu;
}

#ifdef G_OS_WIN32
/* This is a workaround for a bug in windows GTK+. Clicking outside of the
 * menu does not get rid of it, so instead we get rid of it as soon as the
 * pointer leaves the menu.
 */

static gboolean hide_docklet_menu( gpointer data ) {
	if (data != NULL) {
		gtk_menu_popdown( GTK_MENU( data ) );
	}

	return FALSE;
}

gboolean trayIconLeaveEnter( GtkWidget *psMenu, GdkEventCrossing *psEvent, void *pData ) {
	static guint nHideTimer = 0;

	if ( psEvent -> type == GDK_LEAVE_NOTIFY && ( psEvent -> detail == GDK_NOTIFY_ANCESTOR || psEvent -> detail == GDK_NOTIFY_UNKNOWN ) ) {
		if ( nHideTimer == 0 ) {
			nHideTimer = g_timeout_add( 500, hide_docklet_menu, psMenu );
		}
	} else if ( psEvent -> type == GDK_ENTER_NOTIFY && psEvent -> detail == GDK_NOTIFY_ANCESTOR ) {
		if ( nHideTimer != 0 ) {
			g_source_remove( nHideTimer );
			nHideTimer = 0;
		}
	}

	return FALSE;
}
#endif

/**
 * \brief Handle tray icon menu
 * \param psStatusIcon status icon pointer
 * \param nButton which button
 * \param nActivationTime activation time
 * \param pUserData user data pointer
 */
static void trayIconOnMenu( GtkStatusIcon *psStatusIcon, guint nButton, guint nActivateTime, gpointer pUserData) {
	if ( psMenu != NULL ) {
		gtk_widget_destroy( psMenu );
		psMenu = NULL;
	}
	psMenu = createMenu();

#ifdef G_OS_WIN32
	/* Workaround for several Windows bugs */
	g_signal_connect( psMenu, "leave-notify-event", G_CALLBACK( trayIconLeaveEnter ), NULL );
	g_signal_connect( psMenu, "enter-notify-event", G_CALLBACK( trayIconLeaveEnter ), NULL );
#endif

	gtk_widget_show_all( psMenu );

#ifdef G_OS_WIN32
	gtk_menu_popup( GTK_MENU( psMenu ), NULL, NULL, gtk_status_icon_position_menu, psStatusIcon, 0, nActivateTime );
#else
	gtk_menu_popup( GTK_MENU( psMenu ), NULL, NULL, gtk_status_icon_position_menu, psStatusIcon, nButton, nActivateTime );
#endif
}

/**
 * \brief Create tray icon and set menu/tooltip
 */
void createTrayIcon( void ) {
	psTrayIcon = gtk_status_icon_new();
	g_signal_connect( G_OBJECT( psTrayIcon ), "popup-menu", G_CALLBACK( trayIconOnMenu ), NULL );
	g_signal_connect( G_OBJECT( psTrayIcon ), "activate", G_CALLBACK( trayIconActivated ), NULL );
	gtk_status_icon_set_from_pixbuf( psTrayIcon, getTypeIcon( 3 ) );
	gtk_status_icon_set_tooltip_text( psTrayIcon,  "ffgtk" );
	gtk_status_icon_set_visible( psTrayIcon, TRUE );
}

/**
 * \brief Periodically update tooltip information
 * \param pUserData pointer to tray icon
 * \return error code
 */
gpointer updateTooltip( gpointer pUserData ) {
	gchar *pnIp = NULL;
	gchar *pnText = NULL;
	gint nMaxUp = 0;
	gint nMaxDown = 0;

	while ( 1 ) {
		pnIp = routerGetIp( getActiveProfile() );

		if ( pnIp != NULL ) {
			routerGetSpeed( getActiveProfile(), &nMaxUp, &nMaxDown );
			pnText = g_strdup_printf( _( "IP: %s\nMax. Up: %d kbit/s\nMax. Down: %d kbit/s" ), pnIp, nMaxUp >> 10, nMaxDown >> 10 );
			g_free( pnIp );
			pnIp = NULL;
		} else {
			pnText = g_strdup_printf( _( "Not connected" ) );
		}

#ifndef HAVE_APPINDICATOR
		GtkStatusIcon *psIcon = ( GtkStatusIcon * ) pUserData;
		ffgtkLock();
		gtk_status_icon_set_tooltip_text( psIcon, pnText );
		ffgtkUnlock();
#endif

		g_free( pnText );
		pnText = NULL;

		g_usleep( 15 * G_USEC_PER_SEC );
	}

	return NULL;
}

/**
 * \brief Get tray icon widget
 * \return trayicon widget
 */
GtkStatusIcon *getTrayIcon( void ) {
	return psTrayIcon;
}

#ifdef HAVE_APPINDICATOR
/** menu update id */
static gint nMenuUpdateId = 0;

/**
 * \brief Update menu signal callback
 * \return FALSE
 */
static gboolean updateMenu( void ) {
	if ( psMenu != NULL ) {
		gtk_widget_destroy( psMenu );
	}
	psMenu = createMenu();
	gtk_widget_show_all( GTK_WIDGET( psMenu ) );
	app_indicator_set_menu( psIndicator, GTK_MENU( psMenu ) );
	nMenuUpdateId = 0;

	return FALSE;
}
#endif

/**
 * \brief Recreate app menu
 */
void recreateAppMenu( void ) {
#ifdef HAVE_APPINDICATOR
	if ( !nMenuUpdateId ) {
		nMenuUpdateId = g_idle_add_full( G_PRIORITY_LOW, ( GSourceFunc ) updateMenu, NULL, NULL );
	}
#endif
}

/**
 * \brief Initialize tray icon structure and start update tooltip thread
 */
void TrayIconInit( void ) {
#ifdef HAVE_APPINDICATOR
	gchar *pnDir = getDirectory( PKGDATADIR );

	psIndicator = app_indicator_new_with_path( "ffgtk", "ffgtk", APP_INDICATOR_CATEGORY_APPLICATION_STATUS, pnDir );
	g_free( pnDir );
	app_indicator_set_status( psIndicator, APP_INDICATOR_STATUS_ACTIVE );

	recreateAppMenu();
#else
	createTrayIcon();
	CREATE_THREAD("tooltip", updateTooltip, psTrayIcon );
#endif
}
