/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \page lcmaps_poolaccount.mod poolaccount plugin

    \section poolaccountsection1 SYNOPSIS
    \b lcmaps_poolaccount.mod
        [-gridmapfile|-GRIDMAPFILE|-gridmap|-GRIDMAP \<location gridmapfile\>]
        [-gridmapdir|-GRIDMAPDIR \<location gridmapdir\>]
        [-override_inconsistency|-OVERRIDE_INCONSISTENCY]
        [-max_mappings_per_credential \<max nr of mappings\>]
 
    \section poolaccountsection2 DESCRIPTION
 
    This plugin is a Acquisition Plugin and will provide the LCMAPS system with Pool Account
    information.
    To do this it needs to look up the Distinguished Name (DN) from a user's certificate in the gridmapfile.
    If this DN is found in the gridmapfile the plugin now knows to which pool of local system accounts
    the user wil be mapped.
    The poolname (starting with a dot ('.') instead of an alphanumeric character)
    will be converted into the an account from a list of local accounts.
    This list is located in the \e gridmapdir and is made out of filenames.
    These filenames correspond to the system poolaccount names.
    (E.g. if a DN corresponds to \c .test in the gridmapfile, it will be mapped to 
    \c test001, \c test002, etc., which names can be found in the gridmapdir)

    If there is no pool account assigned to the user yet, the plugin
    will get a directory listing of the gridmapdir.
    This list will contain usernames corresponding to system accounts specially designated for pool accounting.
    If the plugin resolved the mapping of a certain pool name, let's say '.test',
    the plugin will look into the directory list and will find the first available file in the list
    corresponding with 'test' (e.g. 'test001') by checking the number of links to its i-node.
    If this number is 1, this account is still available.
    To lease this account a second hard link is created, named after the URL-encoded, decapitalized DN.

    When a user returns to this site the plugin will look for the DN of
    the user (URL encoded) in this directory. If found, the corresponding poolaccount will be assigned
    to the user.

    The plugin will resolve the UID, GID and all the secondary GIDs belonging to the poolaccount.
    When this has been done and there weren't any problems detected,
    the plugin will add this information to a datastructure in the Plugin Manager.
    The plugin will finish its run with a LCMAPS_MOD_SUCCESS.
    This result will be reported to the Plugin Manager which started this plugin and
    it will forward this result to the Evaluation Manager, which will take appropriate actions
    for the next plugin to run.
    Normally this plugin would be followed by an Enforcement plugin that can apply these gathered
    credentials in a way that is appropriate to a system administration's needs.
 
    \section poolaccountoptions OPTIONS
    \subsection poolaccountoptie1 -GRIDMAPFILE \<gridmapfile\>
        See \ref poolaccountoptie4 "-gridmap"
 
    \subsection poolaccountoptie2 -gridmapfile \<gridmapfile\>
        See \ref poolaccountoptie4 "-gridmap"
 
    \subsection poolaccountoptie3 -GRIDMAP \<gridmapfile\>
        See \ref poolaccountoptie4 "-gridmap"
 
    \subsection poolaccountoptie4 -gridmap \<gridmapfile\>
        If this option is set, it will override the default path of the gridmapfile.
        It is advised to use an absolute path to the gridmapfile to avoid usage of the wrong file(path).

    \subsection poolaccountoptie5 -GRIDMAPDIR \<gridmapdir\>
        See \ref poolaccountoptie6 "-gridmapdir"
 
    \subsection poolaccountoptie6 -gridmapdir \<gridmapdir\>
        If this option is set, it will override the default path to the gridmapdir.
        It is advised to use an absolute path to the gridmapdir to avoid usage of the wrong path.

    \subsection poolaccountoptie7 -OVERRIDE_INCONSISTENCY
        See \ref poolaccountoptie8 "-override_inconsistency"

    \subsection poolaccountoptie8 -override_inconsistency
        Moving a user from one pool to another (because of a VO change)
        should only be done by changing the gridmapfile indicating the new pool for this user.
        If a user has already been mapped previously to a poolaccount, there is a link present
        between this poolaccount and his DN.
        In the good old days prior to LCMAPS, a 'pool change' would still result in a mapping to
        the old pool account, neglecting the administrative changes in the gridmapfile.
        LCMAPS corrects this behaviour:
        By default the poolaccount plugin will \e fail if the pool designated by the gridmapfile
        doesn't match the previously mapped poolaccount leasename.
        If the site doesn't want a failure on this inconsistency it can turn on this parameter.
        When the inconsistency is detected the plugin will automatically unlink the previous mapping
        and will proceed by making a \e new lease from the new pool.

    \subsection poolaccountoptie9 -max_mappings_per_credential \<max nr of mappings\>
        This value indicates the maximum number of accounts a user, or more specifically
        a set of credentials (=DN + FQANS), can be mapped to. Normally this number is 1.
        But if each job should run under its own account the number should be increased.
        The leasename (or poolindex) in this case looks like:
                url_encoded(<DN>):mapcount=<mapnumber>)

    \subsection poolaccountoptie10 -no_wildcard
        If this option is set, wildcards cannot be used in the grid-mapfile (on by default)

    \subsection poolaccountoptie11 -strict_poolprefix_match [yes|no]. Default is 'yes'.
        If this is set to 'yes', a line in the gridmapfile like
        <DN> .pool
        will result in accounts matching the regexp 'pool[0-9]+'.
        Otherwise it will be allowed to match 'pool.*' (legacy behaviour).

\section poolaccountReturnvalue RETURN VALUES
        \li LCMAPS_MOD_SUCCESS : Success
        \li LCMAPS_MOD_FAIL    : Failure
 
 
\section poolaccountErrors ERRORS
        See bugzilla for known errors (http://marianne.in2p3.fr/datagrid/bugzilla/)
 
\section poolaccountSeeAlso SEE ALSO
        \ref lcmaps_localaccount.mod "lcmaps_localaccount.mod",
        \ref lcmaps_posix_enf.mod "lcmaps_posix_enf.mod",
        \ref lcmaps_ldap_enf.mod "lcmaps_ldap_enf.mod",
        \ref lcmaps_voms.mod "lcmaps_voms.mod",
        \ref lcmaps_voms_poolaccount.mod "lcmaps_voms_poolaccount.mod",
        \ref lcmaps_voms_poolgroup.mod "lcmaps_voms_poolgroup.mod",
        \ref lcmaps_voms_localgroup.mod "lcmaps_voms_localgroup.mod"
*/

/*!
    \file   lcmaps_poolaccount.c
    \brief  Interface to the LCMAPS plugins
    \author Martijn Steenbakkers for the EU DataGrid.

    This file contains the code of the poolaccount plugin
    -# plugin_initialize()
    -# plugin_run()
    -# plugin_terminate()
    -# plugin_introspect()
*/

/*****************************************************************************
                            Include header files
******************************************************************************/
/* Try to provide RTLD_DEFAULT */
#define _GNU_SOURCE
#include <dlfcn.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "lcmaps_plugins_basic_config.h"
#include <lcmaps/lcmaps_modules.h>
#include <lcmaps/lcmaps_arguments.h>
#include <lcmaps/lcmaps_cred_data.h>
#include "lcmaps_gridlist.h"

/******************************************************************************
                                Definitions
******************************************************************************/
#define PLUGIN_RUN      0
#define PLUGIN_VERIFY   1
#define LCMAPS_ABSOLUTE_MAPPING_MAX 9999
#define LCMAPS_DEFAULT_MAPPING_MAX 1
#define LCMAPS_INVALID_MAPPING_MAX -1
#define LCMAPS_MAPPING_MIN 1

/******************************************************************************
                          Module specific prototypes
******************************************************************************/
static int plugin_run_or_verify(int, lcmaps_argument_t *, int);
static char * add_mapcount_to_leasename(int, int, const char *);

/******************************************************************************
                       Define module specific variables
******************************************************************************/

static char *gridmapfile            = NULL;
static char *gridmapdir             = NULL;
static int  override_inconsistency  = 0;
static int  use_wildcard            = 1;
static int  mapping_max             = LCMAPS_DEFAULT_MAPPING_MAX;
static int  strict_poolprefix_match = 1; /* By default strict matching */

/******************************************************************************
Function:   plugin_initialize
Description:
    Initialize plugin
Parameters:
    argc, argv
    argv[0]: the name of the plugin
Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
    LCMAPS_MOD_NOFILE  : db file not found (will halt LCMAPS initialization)
******************************************************************************/
int plugin_initialize(
        int argc,
        char ** argv
)
{
    char * logstr = "lcmaps_plugin_poolaccount-plugin_initialize()";
    int i = 0, j = 0;
    struct stat s;

    lcmaps_log_debug(5,"%s: passed arguments:\n", logstr);
    for (i=0; i < argc; i++)
    {
       lcmaps_log_debug(5,"%s: arg %d is %s\n", logstr, i, argv[i]);
    }

    /*
     * the first will be the thing to edit/select (gridmap(file))
     * the second will be the path && filename of the gridmapfile
     */

    /*
     * Parse arguments, argv[0] = name of plugin, so start with i = 1
     */
    for (i = 1; i < argc; i++)
    {
        if ( ((strcmp(argv[i], "-gridmap") == 0) ||
              (strcmp(argv[i], "-GRIDMAP") == 0) ||
              (strcmp(argv[i], "-gridmapfile") == 0) ||
              (strcmp(argv[i], "-GRIDMAPFILE") == 0))
             && (i + 1 < argc))
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                if (stat (argv[i + 1], &s) < 0)
                {
                    lcmaps_log(LOG_ERR,"%s: Error: The gridmapfile \"%s\" doesn't exist\n", logstr, argv[i + 1]);
                    return LCMAPS_MOD_FAIL;
                }
                gridmapfile = strdup(argv[i + 1]);
            }
            i++;
        }
        else if ( ((strcmp(argv[i], "-gridmapdir") == 0) ||
              (strcmp(argv[i], "-GRIDMAPDIR") == 0))
             && (i + 1 < argc))
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                if (stat (argv[i + 1], &s) < 0)
                {
                    lcmaps_log(LOG_ERR,"%s: Error: The gridmapdir \"%s\" doesn't exist\n", logstr, argv[i + 1]);
                    return LCMAPS_MOD_FAIL;
                }
                gridmapdir = strdup(argv[i + 1]);
            }
            i++;
        }
        else if ( (strcmp(argv[i], "-max_mappings_per_credential") == 0)
             && (i + 1 < argc) )
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                lcmaps_log_debug(2,"%s: Checking if argument behind \"-max_mappings_per_credential\" is a number\n", logstr);
                for (j = 0; j < strlen(argv[i + 1]); j++)
                { 
                    if (!isdigit((argv[i + 1])[j]))
                    {
                        lcmaps_log(LOG_ERR,"%s: Error: maybe found some digits, but there is at least one char corrupting this parameter: %s\n",  logstr, argv[i + 1]);
                        return LCMAPS_MOD_FAIL;
                    }
                }
                mapping_max = atoi(argv[i + 1]);
                if ((mapping_max < LCMAPS_MAPPING_MIN) || (mapping_max > LCMAPS_ABSOLUTE_MAPPING_MAX))
                {
                    lcmaps_log(LOG_ERR,"%s: Illegal value for \"-max_mappings_per_credential\" (%d): Should be in the range: %d-%d (failure)\n", logstr, mapping_max, LCMAPS_MAPPING_MIN, LCMAPS_ABSOLUTE_MAPPING_MAX);
                    return LCMAPS_MOD_FAIL;
                }
            }
            else
            {
                lcmaps_log(LOG_ERR,"%s: no argument found for %s (failure)\n", logstr, argv[i]);
                return LCMAPS_MOD_FAIL;
            }
            i++;
        }
        else if ( (strcmp(argv[i], "-override_inconsistency") == 0) ||
                  (strcmp(argv[i], "-OVERRIDE_INCONSISTENCY") == 0))
        {
            override_inconsistency = 1;
        }
        else if (strcmp(argv[i], "-no_wildcard") == 0)
        {
            use_wildcard = 0;
        }
        else if  ( (strcmp(argv[i], "-strict_poolprefix_match") == 0)
             && (i + 1 < argc) )
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                 if (strcmp(argv[i+1],"yes") == 0)
                 {
                     strict_poolprefix_match = 1;
                 }
                 else if (strcmp(argv[i+1],"no") == 0)
                 {
                     strict_poolprefix_match = 0;
                 }
                 else
                 {
                     lcmaps_log(LOG_ERR,"%s: use \"yes\" or \"no\" for option %s\n", logstr, argv[i]);
                     return LCMAPS_MOD_FAIL;
                 }
            }
            else
            {
                lcmaps_log(LOG_ERR,"%s: no argument found for %s (failure)\n", logstr, argv[i]);
                return LCMAPS_MOD_FAIL;
            }
            i++;
        }
        else
        {
            lcmaps_log(LOG_ERR,"%s: Error in initialization parameter: %s (failure)\n", logstr, argv[i]);
            return LCMAPS_MOD_FAIL;
        }
    }

    return LCMAPS_MOD_SUCCESS;
}

/******************************************************************************
Function:   plugin_introspect
Description:
    return list of required arguments
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_introspect(
        int * argc,
        lcmaps_argument_t ** argv
)
{
    char * logstr = "lcmaps_plugin_poolaccount-plugin_introspect()";
    static lcmaps_argument_t argList[] = {
        {"user_dn"          ,   "char *"    ,   0   ,   NULL},
        {"mapcounter"       ,   "int"       ,   1   ,   NULL},
        {"requested_uid"    ,   "uid_t"     ,   0   ,   NULL},
        {NULL               ,   NULL        ,   -1  ,   NULL},
        {NULL               ,   NULL        ,   -1  ,   NULL}
    };

    /* Get the version of LCMAPS being used: we need at least 1.5.8 to be able
     * to demand "requested_username" in the argList */
    int major=0,minor=0,patch=0;
    /* Most UNIX now support RTLD_DEFAULT (POSIX reserved) */
#ifdef RTLD_DEFAULT
    char *errorstr;
    int (*lcmaps_major)(void),(*lcmaps_minor)(void),(*lcmaps_patch)(void);
    dlerror();
    lcmaps_major=dlsym(RTLD_DEFAULT,"lcmaps_get_major_version");
    lcmaps_minor=dlsym(RTLD_DEFAULT,"lcmaps_get_minor_version");
    lcmaps_patch=dlsym(RTLD_DEFAULT,"lcmaps_get_patch_version");
    if (dlerror()==NULL)    {
	major=lcmaps_major();
	minor=lcmaps_minor();
	patch=lcmaps_patch();
    }
#else
#warning old LCMAPS versions do not have lcmaps_get_major_version() functions
    /* No RTLD_DEFAULT, just hope the symbol exists in LCMAPS */
    major=lcmaps_get_major_version();
    minor=lcmaps_get_minor_version();
    patch=lcmaps_get_patch_version();
#endif

    /* Too old when older than 1.5.8 */
    if (major<1 || (major==1 && (minor<5 || (minor==5 && patch<8))))	{
	lcmaps_log_debug(4,
	    "%s: Old LCMAPS found (%d.%d.%d), not using requested_username\n",
	    logstr,major,minor,patch);
    } else {
	lcmaps_log_debug(4,
	    "%s LCMAPS (%d.%d.%d) supports using requested_username\n",
	    logstr,major,minor,patch);
	argList[3].argName="requested_username";
	argList[3].argType="char *";
	argList[3].argInOut=1;
	argList[3].value=NULL;
    }
    
    lcmaps_log_debug(4,"%s: introspecting\n", logstr);

    *argv = argList;
    *argc = lcmaps_cntArgs(argList);
    lcmaps_log_debug(5,"%s: address first argument: 0x%x\n", logstr,argList);

    return LCMAPS_MOD_SUCCESS;
}


/******************************************************************************
Function:   plugin_run
Description:
    Gather credentials for LCMAPS
Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_run(
        int argc,
        lcmaps_argument_t * argv
)
{
    return plugin_run_or_verify(argc, argv, PLUGIN_RUN);
}

/******************************************************************************
Function:   plugin_verify
Description:
    Verify if user is entitled to use local credentials based on his grid
    credentials. This means that the site should already have been set up
    by, e.g., LCMAPS in a previous run. This method will not try to setup
    account leases, modify (distributed) passwd/group files, etc. etc.
    The outcome should be identical to that of plugin_run().

    Policy: This method will not fail if the uid found is not the
            uid requested for the user, but it will issue a warning.
            The full check is deferred to the stage after which all plugins have run.

Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_verify(
        int argc,
        lcmaps_argument_t * argv
)
{
    return plugin_run_or_verify(argc, argv, PLUGIN_VERIFY);
}

/******************************************************************************
Function: add_mapcount_to_leasename

Description:
    Add the mapcount to the leasename.

   Check the mapcounter against maximum number of allowed mappings
   The following scenarios may happen:
    1. mapping_max is not set (==LCMAPS_INVALID_MAPPING_MAX), mapcounter is set (>0):
       Enforce only one mapping with mapcounter is 1, reflected in the poolindex.
       If mapcounter>1: error.
    2. mapping_max is not set (==LCMAPS_INVALID_MAPPING_MAX), mapcounter is not set (==-1, 0)
       Legacy: one mapping is allowed, not reflected in the poolindex
    3. mapping_max is set (>LCMAPS_INVALID_MAPPING_MAX), mapcounter is not set
       If mapping_max==0: error
    4. mapping_max is set (>LCMAPS_INVALID_MAPPING_MAX), mapcounter is set (>0)
       Mapping is allowed, if mapcounter <= mapping_max
       If mapping_max==0: error

    NOTE: to separate the pure DN from attached strings we insert the character '\001'
          after the DN. The url encoding (in lcmaps_gridlist()) will be done up to this
          point.
          The '\001' character may already be there if a gidstring was inserted first,
          so this is checked for. In that case a ':' is inserted.

Parameters:
    mapping_max:  maximum number of account mappings for this user+FQANS combination
    mapcounter:   account mapping number
    leasename:    the leasename after which the mapcount string should be appended
Returns:
    the new leasename (should be freed)
    NULL in case of error
******************************************************************************/
static char * add_mapcount_to_leasename(
        int mapping_max,
        int mapcounter,
        const char * leasename
)
{
    char * newleasename = NULL;
    int    newleasenamelen = 0;
    char * logstr = "lcmaps_plugin_poolaccount-add_mapcount_to_leasename()";

    if (mapping_max == LCMAPS_INVALID_MAPPING_MAX)
    {
        if (mapcounter < LCMAPS_MAPPING_MIN)
        {
            /* legacy mode, nothing is set */
            /* do nothing, leasename unchanged */
            lcmaps_log_debug(4,"%s: mapcounter not used, leasename unchanged\n", logstr);
            if (leasename) newleasename = strdup(leasename);
        }
        else if (mapcounter == LCMAPS_MAPPING_MIN) /* OK, create a leasename with a ':mapcount=1' at the end */
        {
            char mapcount_string[14];

            snprintf(mapcount_string, 14, "mapcount=%04d", mapcounter);

            if (leasename) newleasename = strdup(leasename);
            newleasenamelen = strlen(newleasename) + 15;
            newleasename = realloc(newleasename, newleasenamelen*sizeof(char));
            lcmaps_log_debug(4,"%s: leasename before adding mapcount: %s\n", logstr, newleasename);
            if (strstr(leasename, "\001"))
                strncat(newleasename, ":", 2);
            else
                strncat(newleasename, "\001", 2);
            strncat(newleasename, mapcount_string, 14);
            lcmaps_log_debug(4,"%s: leasename after adding mapcount: %s\n", logstr, newleasename);
        }
        else
        {
            lcmaps_log(1,"%s: mapping_max is not set, so only 1 mapping is allowed. The request (%d), exceeds this (failure)\n", logstr, mapcounter);
            return NULL;
        }
    }
    else if (mapping_max >= LCMAPS_MAPPING_MIN)
    {
        if (mapcounter < LCMAPS_MAPPING_MIN)
        {
            /* legacy mode, nothing is set */
            /* do nothing, leasename unchanged */
            lcmaps_log_debug(4,"%s: mapcounter not used, leasename unchanged\n", logstr);
            if (leasename) newleasename = strdup(leasename);
        }
        else if (mapcounter <= mapping_max) /* Create a leasename */
        {
            char mapcount_string[14];

            snprintf(mapcount_string, 14, "mapcount=%04d", mapcounter);

            if (leasename) newleasename = strdup(leasename);
            newleasenamelen = strlen(newleasename) + 15;
            newleasename = realloc(newleasename, newleasenamelen*sizeof(char));
            lcmaps_log_debug(4,"%s: leasename before adding mapcount: %s\n", logstr, newleasename);
            if (strstr(leasename, "\001"))
                strncat(newleasename, ":", 2);
            else
                strncat(newleasename, "\001", 2);
            strncat(newleasename, mapcount_string, 14);
            lcmaps_log_debug(4,"%s: leasename after adding mapcount: %s\n", logstr, newleasename);
        }
        else
        {
            lcmaps_log(LOG_ERR,"%s: the request for mapping nr. %d, exceeds the maximum of %d\n", logstr, mapcounter, mapping_max);
            return NULL;
        }
    }
    else if (mapping_max == 0) /* This means no mapping is allowed at all! */
    {
        lcmaps_log(LOG_ERR,"%s: The maximum number of account mappings per credential set is zero, so nothing to map (failure)! (You may want to adjust the \"-max_mappings_per_credential\" configuration parameter!)\n", logstr);
        return NULL;
    }
    else
    {
        lcmaps_log(LOG_ERR,"%s: Illegal value for \"-max_mappings_per_credential\" (%d). Should at least be %d (failure)\n", logstr, mapping_max, LCMAPS_MAPPING_MIN);
        return NULL;
    }
    return newleasename;
}

static int plugin_run_or_verify(
        int argc,
        lcmaps_argument_t * argv,
        int lcmaps_mode
)
{
    char *              logstr          = "lcmaps_plugin_poolaccount-plugin_run()";
    char *              dn              = NULL; 
    int                 dn_cnt          = 0;
    char *              username        = NULL;
    struct passwd       *user_info      = NULL;
    int                 i               = 0;
    int                 cnt_sec_gid     = 0;
    gid_t *             sec_gid         = NULL;
    unsigned short      matching_type   = ((unsigned short)0x0000);
    int                 rc              = 0;
    char *              leasename       = NULL;
    char *              newleasename    = NULL;
    int                 mapcounter      = -1;
    int *               pmapcounter     = NULL;
    char *              requested_username = NULL;
    uid_t               requested_uid   = -1;
    struct passwd       *requested_user_info = NULL;
    void *              value = NULL;

    /*
     * The beginning
     */
    if (lcmaps_mode == PLUGIN_RUN)
        logstr = "lcmaps_plugin_poolaccount-plugin_run()";
    else if (lcmaps_mode == PLUGIN_VERIFY)
        logstr = "lcmaps_plugin_poolaccount-plugin_verify()";
    else
    {
        lcmaps_log(LOG_ERR, "lcmaps_plugin_poolaccount-plugin_run_or_verify(): attempt to run plugin in invalid mode: %d\n", lcmaps_mode);
        goto fail_poolaccount;
    }

    /*
     * Try to get the ordered values:
     */
    if ( ( value = lcmaps_getArgValue("user_dn", "char *", argc, argv) ) )
    {
	dn = *(char **) value;
        lcmaps_log_debug(5,"%s: found dn: %s\n", logstr, dn);

        /* Check if we don't have a DN already registered, if not, add it to the internal registry */
        getCredentialData (DN, &dn_cnt);
        if (dn_cnt == 0)
        {
            lcmaps_log_debug (5, "%s: Adding DN: %s\n", logstr, dn);
            addCredentialData(DN, &dn);
        }
    }
    else
    {
        lcmaps_log_debug(LOG_ERR,"%s: could not get value of dn ! This is a requirement for the poolaccount plugin.\n", logstr);
        goto fail_poolaccount;
    }

    if ( ( value = lcmaps_getArgValue("mapcounter", "int", argc, argv) ) )
    {
	pmapcounter = (int *) value;
        lcmaps_log_debug(5,"%s: mapcounter: %d\n", logstr, *pmapcounter);
        mapcounter = *pmapcounter;
    }
    else
    {
        /* Don't quit because of backwards compatibility */
        lcmaps_log_debug(5,"%s: could not get value of mapcounter !\n", logstr);
    }

    /* 
     * Try to fetch uid the invocator of LCMAPS wants to be
     * verified. (only in PLUGIN_VERIFY mode).
     */
    if (lcmaps_mode == PLUGIN_VERIFY)
    {
        if ( (value = lcmaps_getArgValue("requested_uid", "uid_t", argc, argv)) > 0 )
        {
            struct passwd * requested_user = NULL;
	    requested_uid = *(uid_t *) value;
            if ((requested_user = getpwuid(requested_uid)) != NULL)
            {
                requested_username = strdup(requested_user->pw_name);
		if (requested_username)
		    lcmaps_log_debug(4,"%s: the requested user is %s(%d)\n", logstr, requested_username, (int) requested_uid);
		else	{
		    lcmaps_log(LOG_ERR,"%s: out of memory\n",logstr);
		    goto fail_poolaccount;
		}
            }
            else
            {
                lcmaps_log(LOG_ERR,"%s: the requested uid is not found on this system.\n", logstr, (int) requested_uid);
                goto fail_poolaccount;
            }
        }
        else if (requested_uid == 0)
        {
            lcmaps_log(LOG_ERR,"%s: illegal request for uid == 0 (failure)\n", logstr);
            goto fail_poolaccount;
        }
        else
        {
            lcmaps_log(LOG_ERR,"%s: illegal request for uid < 0 (failure)\n", logstr);
            goto fail_poolaccount;
        }
    } else {
	/*
	 * See if we have a requested username: make sure not to get value
	 * immediately or we'll get a segfault
	 */
	requested_username=NULL;
	if ( (value = lcmaps_getArgValue("requested_username", "char *", argc, argv))!=NULL &&
	     (*(char **)value)!=NULL &&
	     (requested_username = strdup(*(char **)value))==NULL)
	{
	    lcmaps_log(LOG_ERR,"%s: out of memory\n",logstr);
	    goto fail_poolaccount;
	}
	lcmaps_log_debug(1,"%s: requested username is %s\n", logstr, 
		requested_username ? requested_username : "unset");
    }

    /*
     * Check the gridmapfile
     */
    if ((gridmapfile != NULL) && (strlen(gridmapfile) > 0))
        lcmaps_log_debug(3,"%s: gridmapfile is: %s\n", logstr, gridmapfile);
    else
    {
        if (gridmapfile) free(gridmapfile);
        gridmapfile = NULL;
        lcmaps_log_debug(5,"%s: No gridmapfile assigned, so function must find out for it self\n", logstr);
    }

    /*
     * Check gridmapdir
     */
    if (gridmapdir == NULL) /* try if GRIDMAPDIR is already set */
    {
        char * tmpptr=NULL;
        if ((tmpptr = getenv("GRIDMAPDIR")) == NULL)
        {
            lcmaps_log(LOG_ERR,"%s: GRIDMAPDIR unknown !\n", logstr);
            lcmaps_log(1,"%s:  specify as option or set GRIDMAPDIR\n", logstr);
            goto fail_poolaccount;
        }
        else
        {
            gridmapdir = strdup(tmpptr);
        }
    }
    if (strlen(gridmapdir) == 0)
    {
        lcmaps_log(1,"%s: cannot set MAPDIR (strlen(gridmapdir) == 0)\n", logstr);
        goto fail_poolaccount;
    }
    lcmaps_log_debug(5,"%s: setting MAPDIR to %s\n", logstr, gridmapdir);
    if (setenv("MAPDIR", gridmapdir, 1))
    {
        lcmaps_log(1,"%s: cannot set MAPDIR\n", logstr);
        goto fail_poolaccount;
    }

    /* Create leasename from dn */
    leasename = strdup(dn);

    /*
     * Try to find the dn in the gridmapfile
     */
    lcmaps_log_debug(4,"%s: using leasename: %s\n", logstr,leasename);
    /* check if we should use wildcards ... */
    if (use_wildcard)
    {
        matching_type = MATCH_INCLUDE|MATCH_WILD_CHARS;
    }
    else
    {
        matching_type = MATCH_INCLUDE|MATCH_NO_WILD_CHARS;
    }

    /* if override_consistency is set add this to the matching_type so it will take effect */
    if (override_inconsistency)
        matching_type = matching_type|OVERRIDE_INCONSISTANCY;

    /* Do not create new leases in verification mode */
    if (lcmaps_mode == PLUGIN_VERIFY)
        matching_type = matching_type|ONLY_USE_EXISTING_LEASE;

    /* if strict_poolprefix_match is set add this to the matchin_type so it will take effect */
    if (strict_poolprefix_match)
        matching_type = matching_type|MATCH_STRICT_PREFIX_NUM;

    if ((lcmaps_mode == PLUGIN_VERIFY) && (mapcounter < LCMAPS_MAPPING_MIN)) /* loop through all possible mapcounts */
    {
        int imap = 0;
        int found_mapping = 0;

        for (imap = 1; imap <= mapping_max; imap++)
        {
            /* clean username and newleasename before each call to lcmaps_gridlist */
            if (username) free(username);
            username = NULL;
            if (newleasename) free(newleasename);
            newleasename = NULL;

            if ((newleasename = add_mapcount_to_leasename(mapping_max, imap, leasename)) == NULL)
            {
                lcmaps_log(LOG_ERR, "%s: Error creating leasename including mapcounter (%d)\n", logstr, imap);
                goto fail_poolaccount;
            }

            if ( (rc = lcmaps_gridlist(dn, &username, gridmapfile, matching_type, ".", newleasename)) == 0)
            {
		/* in verify mode, requested_username!=NULL */
                if (strcmp(username, requested_username) == 0)
                {
                    lcmaps_log_debug(3,"%s: found requested username: %s\n", logstr, username);
                    found_mapping = 1;
                    break;
                }
		lcmaps_log_debug(3,"%s: found username %s but does not match requested username %s\n", logstr, username, requested_username);
            }
            else if (rc == LCMAPS_MOD_NOFILE)
            {
                lcmaps_log(LOG_ERR, "%s: Could not find the gridmapfile %s\n", logstr, gridmapfile);
                goto fail_poolaccount;
            }
            else
            {
                lcmaps_log(LOG_ERR, "%s: No poolaccount lease available for %s, mapcount=%d\n", logstr, dn, imap);
            }
        }
        if (found_mapping == 0) /* Let's try the legacy behaviour (use the (invalid) mapcounter) */
        {
            /* clean username and newleasename before each call to lcmaps_gridlist */
            if (username) free(username);
            username = NULL;
            if (newleasename) free(newleasename);
            newleasename = NULL;

            if ((newleasename = add_mapcount_to_leasename(mapping_max, mapcounter, leasename)) == NULL)
            {
                lcmaps_log(LOG_ERR, "%s: Error creating leasename including mapcounter (%d)\n", logstr, mapcounter);
                goto fail_poolaccount;
            }
            if ( (rc = lcmaps_gridlist(dn, &username, gridmapfile, matching_type, ".", newleasename)) == 0)
            {
		/* in verify mode, requested_username!=NULL */
                if (strcmp(username, requested_username) == 0)
                {
                    lcmaps_log_debug(3,"%s: found requested username: %s\n", logstr, username);
                    found_mapping = 1;
                }
		else
		    lcmaps_log_debug(3,"%s: found username %s but does not match requested username: %s\n", logstr, username, requested_username);

            }
            else if (rc == LCMAPS_MOD_NOFILE)
            {
                lcmaps_log(LOG_ERR, "%s: Could not find the gridmapfile %s\n", logstr, gridmapfile);
                goto fail_poolaccount;
            }
            else
            {
                lcmaps_log(LOG_NOTICE, "%s: No poolaccount lease available for %s, mapcount=%d\n", logstr, dn, imap);
            }
        }
        if (found_mapping == 0)
        {
            lcmaps_log(LOG_ERR, "%s: The requested username was not found for %s in %s\n", logstr, requested_username, dn, gridmapfile);
            goto fail_poolaccount;
        }
    }
    else
    {
        /* clean username and newleasename before each call to lcmaps_gridlist */
        if (username) free(username);
        username = NULL;
        if (newleasename) free(newleasename);
        newleasename = NULL;

        if ((newleasename = add_mapcount_to_leasename(mapping_max, mapcounter, leasename)) == NULL)
        {
            lcmaps_log(LOG_ERR, "%s: Error creating leasename including mapcounter (%d)\n", logstr, mapcounter);
            goto fail_poolaccount;
        }
        if ( (rc = lcmaps_gridlist(dn, &username, gridmapfile, matching_type, ".", newleasename)) == 0)
        {
            if (lcmaps_mode == PLUGIN_VERIFY)
            {
                lcmaps_log_debug(5,"%s: found a username: %s\n", logstr, username);
                /* in verify mode, requested_username!=NULL */
		if (strcmp(username, requested_username) == 0)
                {
                    lcmaps_log_debug(5,"%s: found requested username: %s\n", logstr, username);
                }
                else
                {
                    lcmaps_log(LOG_ERR,"%s: The requested username \"%s\" was not found for %s in %s, mapcount=%d\n", logstr, requested_username, dn, gridmapfile, mapcounter);
                    goto fail_poolaccount;
                }
            }
            else
	    {
		/* requested_username can be NULL */
		if (requested_username==NULL)   {
		    lcmaps_log_debug(5,"%s: found username: %s\n", logstr, username);
		}
		else if (strcmp(username, requested_username) == 0)
                {
                    lcmaps_log_debug(5,"%s: found requested username: %s\n", logstr, username);
                }
                else
                {
                    lcmaps_log(LOG_ERR,"%s: The requested username \"%s\" does not match the poolaccount \"%s\" for %s in %s, mapcount=%d\n",
			    logstr, requested_username, username, dn, gridmapfile, mapcounter);
                    goto fail_poolaccount;
                }
	    }

        }
        else if (rc == LCMAPS_MOD_NOFILE)
        {
            lcmaps_log(LOG_ERR, "%s: Could not find the gridmapfile %s\n", logstr, gridmapfile);
            goto fail_poolaccount;
        }
        else
        {
            lcmaps_log(LOG_NOTICE, "%s: No lease found for %s, mapcount=%d\n", logstr, dn, mapcounter);
            goto fail_poolaccount;
        }
    }

    /*
     * Get userid to pwd_t structure
     */
    if (username && (strlen(username) > 0))
    {

        if ( ( user_info = getpwnam(username) ) )
        {
            char *  encoded_dn = NULL;

            lcmaps_log_debug(4,"%s: username : %s\n", logstr, user_info->pw_name);
            lcmaps_log_debug(4,"%s: user_id  : %d\n", logstr, user_info->pw_uid);
            lcmaps_log_debug(4,"%s: group_id : %d\n", logstr, user_info->pw_gid);
            lcmaps_log_debug(4,"%s: home dir : %s\n", logstr, user_info->pw_dir);

            /* 
             * Add this credential data to the credential data repository in the plugin manager
             */
            addCredentialData(UID, &(user_info->pw_uid));
            addCredentialData(PRI_GID, &(user_info->pw_gid));

            /*
             * Retrieve secondary group id's
             */
            if (lcmaps_get_gidlist(username, &cnt_sec_gid, &sec_gid)==0)
            {
                for (i = 0; i < cnt_sec_gid; i++)
                {
                    addCredentialData(SEC_GID, &(sec_gid[i]));
                }
                free(sec_gid);
            }

            /* Added for the POOL_INDEX request for the DAS */
            encoded_dn = gridmapdir_urlencode(newleasename);

            addCredentialData(POOL_INDEX, &encoded_dn);
            if (encoded_dn)
            {
                free(encoded_dn);
                encoded_dn = NULL;
            }
        }
        else
        {
            lcmaps_log(LOG_ERR,"%s: no valid user account found with the name \"%s\"\n", logstr, username);
            goto fail_poolaccount;
        }
    }
    else
    {   // error (msg is already given)
        goto fail_poolaccount;
    }

    /* succes */
 success_poolaccount:
    if (requested_username) free(requested_username);
    if (username) free(username);
    if (leasename) free(leasename);
    if (newleasename) free(newleasename);
    lcmaps_log(LOG_INFO,"%s: poolaccount plugin succeeded\n", logstr);
    return LCMAPS_MOD_SUCCESS;

 fail_poolaccount:
    if (requested_username) free(requested_username);
    if (username) free(username);
    if (leasename) free(leasename);
    if (newleasename) free(newleasename);
    lcmaps_log(LOG_INFO,"%s: poolaccount plugin failed\n", logstr);
    return LCMAPS_MOD_FAIL;
}

/******************************************************************************
Function:   plugin_terminate
Description:
    Terminate plugin
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_terminate()
{
    char *              logstr = "lcmaps_plugin_poolaccount-plugin_terminate()";
    lcmaps_log_debug(4,"%s: terminating\n", logstr);

    if (gridmapfile) free(gridmapfile);
    if (gridmapdir) free(gridmapdir);

    return LCMAPS_MOD_SUCCESS;
}

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps-plugins-basic/src/poolaccount/lcmaps_poolaccount.c,v $
    $Date: 2010-02-19 06:08:47 $
    $Revision: 1.20 $
    $Author: okoeroo $
******************************************************************************/
