// Boost.Geometry
// Unit Test

// Copyright (c) 2017-2022, Oracle and/or its affiliates.
// Contributed and/or modified by Vissarion Fysikopoulos, on behalf of Oracle
// Contributed and/or modified by Adam Wulkiewicz, on behalf of Oracle

// Use, modification and distribution is subject to the Boost Software License,
// Version 1.0. (See accompanying file LICENSE_1_0.txt or copy at
// http://www.boost.org/LICENSE_1_0.txt)

// This file contains test cases converted from PROJ4, http://trac.osgeo.org/proj
// PROJ4 is originally written by Gerald Evenden (then of the USGS)
// PROJ4 is maintained by Frank Warmerdam
// Test cases was converted to Boost.Geometry by Adam Wulkiewicz

// Original copyright notice:

// Copyright (c) 1995, Gerald Evenden

// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included
// in all copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#ifndef BOOST_GEOMETRY_TEST_SRS_PROJECTION_SELFTEST_CASES_HPP
#define BOOST_GEOMETRY_TEST_SRS_PROJECTION_SELFTEST_CASES_HPP


#include <geometry_test_common.hpp>

#include <boost/geometry.hpp>
#include <boost/geometry/geometries/register/point.hpp>


struct XY { double x, y; };
struct LL { double lon, lat; };

BOOST_GEOMETRY_REGISTER_POINT_2D(XY, double, bg::cs::cartesian, x, y)
BOOST_GEOMETRY_REGISTER_POINT_2D(LL, double, bg::cs::geographic<bg::degree>, lon, lat)

struct projection_case
{
    std::string id;
    std::string args;
    LL fwd_in[4];
    XY fwd_expect[4];
    XY inv_in[4];
    LL inv_expect[4];
};

static const projection_case projection_cases[] = {
    {
        "aea_e",
        "+proj=aea   +ellps=GRS80  +lat_1=0 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {222571.60875710563,  110653.32674302977},
            {222706.30650839131,  -110484.26714439997},
            {-222571.60875710563,  110653.32674302977},
            {-222706.30650839131,  -110484.26714439997}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0017966310597749514,  0.00090436885862202158},
            {0.0017966300767030448,  -0.00090437009538581453},
            {-0.0017966310597749514,  0.00090436885862202158},
            {-0.0017966300767030448,  -0.00090437009538581453}
        }
    },{
        "aea_s",
        "+proj=aea   +R=6400000    +lat_1=0 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {223334.08517088494,  111780.43188447191},
            {223470.15499168713,  -111610.33943099028},
            {-223334.08517088494,  111780.43188447191},
            {-223470.15499168713,  -111610.33943099028}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0017904935979658752,  0.00089524594491375306},
            {0.0017904926216016812,  -0.00089524716502493225},
            {-0.0017904935979658752,  0.00089524594491375306},
            {-0.0017904926216016812,  -0.00089524716502493225}
        }
    },{
        "leac_e",
        "+proj=leac   +ellps=GRS80  +lat_1=0 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {220685.14054297868,  112983.50088939646},
            {224553.31227982609,  -108128.63674487274},
            {-220685.14054297868,  112983.50088939646},
            {-224553.31227982609,  -108128.63674487274}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0017966446840328458,  0.00090435171340223211},
            {0.0017966164523713021,  -0.00090438724081843625},
            {-0.0017966446840328458,  0.00090435171340223211},
            {-0.0017966164523713021,  -0.00090438724081843625}
        }
    },{
        "leac_s",
        "+proj=leac   +R=6400000    +lat_1=0 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {221432.86859285168,  114119.45452653214},
            {225331.72412711097,  -109245.82943505641},
            {-221432.86859285168,  114119.45452653214},
            {-225331.72412711097,  -109245.82943505641}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0017905070979748127,  0.00089522906964877795},
            {0.001790479121519977,  -0.00089526404022281043},
            {-0.0017905070979748127,  0.00089522906964877795},
            {-0.001790479121519977,  -0.00089526404022281043}
        }
    },{
        "aeqd_e",
        "+proj=aeqd   +ellps=GRS80  +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 222616.522190051648,  110596.996549550197},
            { 222616.522190051648, -110596.996549550211},
            {-222616.522190051648,  110596.996549550197},
            {-222616.522190051648, -110596.996549550211}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00179663056838724787,  0.000904369476930248902},
            { 0.00179663056838724787, -0.000904369476930248469},
            {-0.00179663056838724787,  0.000904369476930248902},
            {-0.00179663056838724787, -0.000904369476930248469}
        }
    },{
        "aeqd_s",
        "+proj=aeqd   +R=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223379.456047271,  111723.757570854126},
            { 223379.456047271, -111723.757570854126},
            {-223379.456047271,  111723.757570854126},
            {-223379.456047271, -111723.757570854126}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00179049310992953335,  0.000895246554746200623},
            { 0.00179049310992953335, -0.000895246554746200623},
            {-0.00179049310992953335,  0.000895246554746200623},
            {-0.00179049310992953335, -0.000895246554746200623}
        }
    },{
        "airy",
        "+proj=airy   +a=6400000    +lat_1=0 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 189109.88690862127,   94583.752387504152},
            { 189109.88690862127,  -94583.752387504152},
            {-189109.88690862127,   94583.752387504152},
            {-189109.88690862127,  -94583.752387504152}
        },
        {},{{HUGE_VAL, HUGE_VAL}}
    },{
        "aitoff",
        "+proj=aitoff   +R=6400000    +lat_1=0 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {223379.45881169615,  111706.74288385305},
            {223379.45881169615,  -111706.74288385305},
            {-223379.45881169615,  111706.74288385305},
            {-223379.45881169615,  -111706.74288385305}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0017904931100388164,  0.00089524655491012516},
            {0.0017904931100388164,  -0.00089524655491012516},
            {-0.0017904931100388164,  0.00089524655491012516},
            {-0.0017904931100388164,  -0.00089524655491012516}
        }
    },{
        "wintri",
        "+proj=wintri   +a=6400000    +lat_1=0 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {223390.80153348515,  111703.90750574505},
            {223390.80153348515,  -111703.90750574505},
            {-223390.80153348515,  111703.90750574505},
            {-223390.80153348515,  -111703.90750574505}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0017904931099113196,  0.00089524655490101819},
            {0.0017904931099113196,  -0.00089524655490101819},
            {-0.0017904931099113196,  0.00089524655490101819},
            {-0.0017904931099113196,  -0.00089524655490101819}
        }
    },{
        "august",
        "+proj=august   +a=6400000    +lat_1=0 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {223404.97818097242,  111722.34028976287},
            {223404.97818097242,  -111722.34028976287},
            {-223404.97818097242,  111722.34028976287},
            {-223404.97818097242,  -111722.34028976287}
        },
        {},{{HUGE_VAL, HUGE_VAL}}
    },{
        "bacon",
        "+proj=bacon   +a=6400000    +lat_1=0 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {223334.13255596498,  175450.72592266591},
            {223334.13255596498,  -175450.72592266591},
            {-223334.13255596498,  175450.72592266591},
            {-223334.13255596498,  -175450.72592266591}
        },
        {},{{HUGE_VAL, HUGE_VAL}}
    },{
        "apian",
        "+proj=apian   +a=6400000    +lat_1=0 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223374.57735525275,   111701.07212763709},
            { 223374.57735525275,  -111701.07212763709},
            {-223374.57735525275,   111701.07212763709},
            {-223374.57735525275,  -111701.07212763709}
        },
        {},{{HUGE_VAL, HUGE_VAL}}
    },{
        "ortel",
        "+proj=ortel   +a=6400000    +lat_1=0 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223374.57735525275,   111701.07212763709},
            { 223374.57735525275,  -111701.07212763709},
            {-223374.57735525275,   111701.07212763709},
            {-223374.57735525275,  -111701.07212763709}
        },
        {},{{HUGE_VAL, HUGE_VAL}}
    },{
        "bipc_e",
        "+proj=bipc   +ellps=GRS80  +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {2452160.2177257561,  -14548450.759654747},
            {2447915.213725341,  -14763427.21279873},
            {2021695.5229349085,  -14540413.695283702},
            {2018090.5030046992,  -14755620.651414108}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {-73.038700284978702,  17.248118466239116},
            {-73.03730373933017,  17.249414978178777},
            {-73.03589317304332,  17.245536403008771},
            {-73.034496627213585,  17.246832895573739}
        }
    },{
        "bipc_s",
        "+proj=bipc   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {2460565.7409749646,  -14598319.9893308},
            {2456306.1859352002,  -14814033.339502094},
            {2028625.4978190989,  -14590255.375482792},
            {2025008.1205891429,  -14806200.018759441}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {-73.038693104942126,  17.248116270440242},
            {-73.037301330021322,  17.24940835333777},
            {-73.035895582251086,  17.245543027866539},
            {-73.034503807150301,  17.246835091521532}
        }
    },{
        "boggs",
        "+proj=boggs   +a=6400000    +lat_1=0 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 211949.70080818201,   117720.99830541089},
            { 211949.70080818201,  -117720.99830541089},
            {-211949.70080818201,   117720.99830541089},
            {-211949.70080818201,  -117720.99830541089},
        },
        {},{{HUGE_VAL, HUGE_VAL}}
    },{
        "bonne_e",
        "+proj=bonne   +ellps=GRS80  +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 222605.29609715697,   55321.139565494814},
            { 222605.29609923941,  -165827.64779905154},
            {-222605.29609715697,   55321.139565494814},
            {-222605.29609923941,  -165827.64779905154}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0017966987691132891,  0.50090436853737497},
            { 0.0017966982774478867,  0.4990956309655612},
            {-0.0017966987691132891,  0.50090436853737497},
            {-0.0017966982774478867,  0.4990956309655612}
        }
    },{
        "bonne_s",
        "+proj=bonne   +R=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223368.11557252839,   55884.555246393575},
            { 223368.11557463196,  -167517.59936969393},
            {-223368.11557252839,   55884.555246393575},
            {-223368.11557463196,  -167517.59936969393}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0017905615332457991,  0.50089524631087834},
            { 0.0017905610449335603,  0.49910475320072978},
            {-0.0017905615332457991,  0.50089524631087834},
            {-0.0017905610449335603,  0.49910475320072978}
        }
    },
    // NOT IMPLEMENTED IN Boost.Geometry
    /*
    {
        "calcofi_e",
        "+proj=calcofi   +ellps=GRS80  +lat_1=0.5 +lat_2=2 +no_defs",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {508.44487214981905,  -1171.7648604175156},
            {514.99916815188112,  -1145.8219814677668},
            {500.68538412539851,  -1131.4453779204598},
            {507.36971913666355,  -1106.1782014834275}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {-110.36330792469906,  12.032056975840137},
            {-98.455008863288782,  18.698723642506803},
            {-207.4470245036909,  81.314089278595247},
            {-62.486322854481287,  87.980755945261919}
        }
    },{
        "calcofi_s",
        "+proj=calcofi   +R=6400000    +lat_1=0.5 +lat_2=2 +no_defs",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {507.09050748781806,  -1164.7273751978314},
            {513.68613637462886,  -1138.9992682173072},
            {499.33626147591531,  -1124.4351309968195},
            {506.0605703929898,  -1099.3756650673038}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {-110.30519040955151,  12.032056975840137},
            {-98.322360950234085,  18.698723642506803},
            {-207.54490681381429,  81.314089278595247},
            {-62.576950371885275,  87.980755945261919}
        }
    },*/
    // NOTE: cart projection test cases are implemented differently in Proj4
    {
        "cass_e",
        "+proj=cass   +ellps=GRS80  +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 222605.28577699114,   110642.22925399939},
            { 222605.28577699114,  -110642.22925399939},
            {-222605.28577699114,   110642.22925399939},
            {-222605.28577699114,  -110642.22925399939}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0017966305684613522,   0.00090436947663183841},
            { 0.0017966305684613522,  -0.00090436947663183841},
            {-0.0017966305684613522,   0.00090436947663183841},
            {-0.0017966305684613522,  -0.00090436947663183841}
        }
    },{
        "cass_s",
        "+proj=cass   +R=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223368.10520348375,  111769.14504058579},
            { 223368.10520348375, -111769.14504058579},
            {-223368.10520348375,  111769.14504058579},
            {-223368.10520348375, -111769.14504058579}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0017904931100023887,   0.00089524655445477922},
            { 0.0017904931100023887,  -0.00089524655445477922},
            {-0.0017904931100023887,   0.00089524655445477922},
            {-0.0017904931100023887,  -0.00089524655445477922}
        }
    },{
        "cc",
        "+proj=cc   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {223402.14425527418,  111712.41554059254},
            {223402.14425527418,  -111712.41554059254},
            {-223402.14425527418,  111712.41554059254},
            {-223402.14425527418,  -111712.41554059254}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0017904931097838226,  0.00089524655481905597},
            {0.0017904931097838226,  -0.00089524655481905597},
            {-0.0017904931097838226,  0.00089524655481905597},
            {-0.0017904931097838226,  -0.00089524655481905597}
        }
    },{
        "cea_e",
        "+proj=cea   +ellps=GRS80  +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 222638.981586547132,  110568.812396267356},
            { 222638.981586547132, -110568.812396265886},
            {-222638.981586547132,  110568.812396267356},
            {-222638.981586547132, -110568.812396265886}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00179663056823904264,  0.000904369476105564289},
            { 0.00179663056823904264, -0.000904369476105564289},
            {-0.00179663056823904264,  0.000904369476105564289},
            {-0.00179663056823904264, -0.000904369476105564289}
        }
    },{
        "cea_s",
        "+proj=cea   +R=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223402.144255274179,  111695.401198614476},
            { 223402.144255274179, -111695.401198614476},
            {-223402.144255274179,  111695.401198614476},
            {-223402.144255274179, -111695.401198614476}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00179049310978382265,  0.000895246554928338998},
            { 0.00179049310978382265, -0.000895246554928338998},
            {-0.00179049310978382265,  0.000895246554928338998},
            {-0.00179049310978382265, -0.000895246554928338998}
        }
    },{
        "chamb",
        "+proj=chamb   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {-27864.7795868005815,  -223364.324593274243},
            {-251312.283053493476,  -223402.145526208304},
            {-27864.7856491046077,  223364.327328827145},
            {-251312.289116443484,  223402.142197287147}
        },
        {},{{HUGE_VAL, HUGE_VAL}}
    },{
        "col_urban",
        "+proj=col_urban   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {223368.3467527547, 110953.3027055085},
            {223368.3467527547, -110953.3027055085},
            {-223368.3467527547, 110953.3027055085},
            {-223368.3467527547, -110953.3027055085}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.001790493110003862, 0.0009012800661321395},
            {0.001790493110003862, -0.0009012800661321395},
            {-0.001790493110003862, 0.0009012800661321395},
            {-0.001790493110003862, -0.0009012800661321395}
        }
    },{
        "collg",
        "+proj=collg   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {249872.921577929839,  99423.1747884602082},
            {254272.532301245432,  -98559.3077607425657},
            {-249872.921577929839,  99423.1747884602082},
            {-254272.532301245432,  -98559.3077607425657}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.00158679719207879865,  0.00101017310941749921},
            {0.001586769215623956,  -0.00101018201458258111},
            {-0.00158679719207879865,  0.00101017310941749921},
            {-0.001586769215623956,  -0.00101018201458258111}
        }
    },
    // NOT IMPLEMENTED IN Boost.Geometry
    /*{
        "comill",
        "+proj=comill   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {223402.144255274179,  110611.859089458536},
            {223402.144255274179,  -110611.859089458536},
            {-223402.144255274179,  110611.859089458536},
            {-223402.144255274179,  -110611.859089458536}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.00179049310978382265,  0.000904106801510605831},
            {0.00179049310978382265,  -0.000904106801510605831},
            {-0.00179049310978382265,  0.000904106801510605831},
            {-0.00179049310978382265,  -0.000904106801510605831}
        }
    },*/{
        "crast",
        "+proj=crast   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {218280.142056780722,  114306.045604279774},
            {218280.142056780722,  -114306.045604279774},
            {-218280.142056780722,  114306.045604279774},
            {-218280.142056780722,  -114306.045604279774}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.00183225941982580187,  0.00087483943098902331},
            {0.00183225941982580187,  -0.00087483943098902331},
            {-0.00183225941982580187,  0.00087483943098902331},
            {-0.00183225941982580187,  -0.00087483943098902331}
        }
    },{
        "denoy",
        "+proj=denoy   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223377.422876954137,  111701.07212763709},
            { 223377.422876954137, -111701.07212763709},
            {-223377.422876954137,  111701.07212763709},
            {-223377.422876954137, -111701.07212763709}
        },
        {},{{HUGE_VAL, HUGE_VAL}}
    },{
        "eck1",
        "+proj=eck1   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 204680.88820295094,  102912.17842606473},
            { 204680.88820295094, -102912.17842606473},
            {-204680.88820295094,  102912.17842606473},
            {-204680.88820295094, -102912.17842606473}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0019434150820034624,  0.00097170229538813102},
            { 0.0019434150820034624, -0.00097170229538813102},
            {-0.0019434150820034624,  0.00097170229538813102},
            {-0.0019434150820034624, -0.00097170229538813102}
        }
    },{
        "eck2",
        "+proj=eck2   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 204472.87090796008,  121633.73497524235},
            { 204472.87090796008, -121633.73497524235},
            {-204472.87090796008,  121633.73497524235},
            {-204472.87090796008, -121633.73497524235}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0019434150820034624,  0.00082480429919795412},
            { 0.0019434150820034624, -0.00082480429919795412},
            {-0.0019434150820034624,  0.00082480429919795412},
            {-0.0019434150820034624, -0.00082480429919795412}
        }
    },{
        "eck3",
        "+proj=eck3   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 188652.01572153764,  94328.919337031271},
            { 188652.01572153764, -94328.919337031271},
            {-188652.01572153764,  94328.919337031271},
            {-188652.01572153764, -94328.919337031271}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0021202405520236059,  0.0010601202759750307},
            { 0.0021202405520236059, -0.0010601202759750307},
            {-0.0021202405520236059,  0.0010601202759750307},
            {-0.0021202405520236059, -0.0010601202759750307}
        }
    },{
        "eck4",
        "+proj=eck4   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 188646.38935641639,  132268.54017406539},
            { 188646.38935641639, -132268.54017406539},
            {-188646.38935641639,  132268.54017406539},
            {-188646.38935641639, -132268.54017406539}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0021202405520236059, 0.00075601458836610643},
            { 0.0021202405520236059, -0.00075601458836610643},
            {-0.0021202405520236059, 0.00075601458836610643},
            {-0.0021202405520236059, -0.00075601458836610643}
        }
    },{
        "eck5",
        "+proj=eck5   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 197031.39213406085,  98523.198847226551},
            { 197031.39213406085, -98523.198847226551},
            {-197031.39213406085,  98523.198847226551},
            {-197031.39213406085, -98523.198847226551}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.002029978749734037,  0.001014989374787388},
            {0.002029978749734037,  -0.001014989374787388},
            {-0.002029978749734037,  0.001014989374787388},
            {-0.002029978749734037,  -0.001014989374787388}
        }
    },{
        "eqc",
        "+proj=eqc   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223402.144255274179,  111701.07212763709},
            { 223402.144255274179, -111701.07212763709},
            {-223402.144255274179,  111701.07212763709},
            {-223402.144255274179, -111701.07212763709}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00179049310978382265,  0.000895246554891911323},
            { 0.00179049310978382265, -0.000895246554891911323},
            {-0.00179049310978382265,  0.000895246554891911323},
            {-0.00179049310978382265, -0.000895246554891911323}
        }
    },{
        "eqdc_e",
        "+proj=eqdc   +ellps=GRS80  +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 222588.440269285755,  110659.134907347048},
            { 222756.836702042434, -110489.578087220681},
            {-222588.440269285755,  110659.134907347048},
            {-222756.836702042434, -110489.578087220681}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00179635944879094839,  0.000904368858588402644},
            { 0.00179635822020772734, -0.000904370095529954975},
            {-0.00179635944879094839,  0.000904368858588402644},
            {-0.00179635822020772734, -0.000904370095529954975}
        }
    },{
        "eqdc_s",
        "+proj=eqdc   +R=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223351.088175113517,  111786.108747173785},
            { 223521.200266735133, -111615.970741240744},
            {-223351.088175113517,  111786.108747173785},
            {-223521.200266735133, -111615.970741240744}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0017902210900486641,   0.000895245944814909169},
            { 0.00179021986984890255, -0.000895247165333684842},
            {-0.0017902210900486641,   0.000895245944814909169},
            {-0.00179021986984890255, -0.000895247165333684842}
        }
    },{
        "fahey",
        "+proj=fahey   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 182993.34464912376,  101603.19356988439},
            { 182993.34464912376, -101603.19356988439},
            {-182993.34464912376,  101603.19356988439},
            {-182993.34464912376, -101603.19356988439}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0021857886080359551,  0.00098424601668238403},
            {0.0021857886080359551,  -0.00098424601668238403},
            {-0.0021857886080359551,  0.00098424601668238403},
            {-0.0021857886080359551,  -0.00098424601668238403}
        }
    },{
        "fouc_s",
        "+proj=fouc_s   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223402.14425527424,  111695.40119861449},
            { 223402.14425527424, -111695.40119861449},
            {-223402.14425527424,  111695.40119861449},
            {-223402.14425527424, -111695.40119861449}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0017904931097838226,  0.000895246554928339},
            { 0.0017904931097838226, -0.000895246554928339},
            {-0.0017904931097838226,  0.000895246554928339},
            {-0.0017904931097838226, -0.000895246554928339}
        }
    },{
        "gall",
        "+proj=gall   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 157969.17113451968,  95345.249178385886},
            { 157969.17113451968, -95345.249178385886},
            {-157969.17113451968,  95345.249178385886},
            {-157969.17113451968, -95345.249178385886}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0025321396391918614,  0.001048846580346495},
            { 0.0025321396391918614, -0.001048846580346495},
            {-0.0025321396391918614,  0.001048846580346495},
            {-0.0025321396391918614, -0.001048846580346495}
        }
    },{
        "geos_e",
        "+proj=geos   +ellps=GRS80  +lat_1=0.5 +lat_2=2 +h=35785831",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 222527.07036580026,  110551.30341332949},
            { 222527.07036580026, -110551.30341332949},
            {-222527.07036580026,  110551.30341332949},
            {-222527.07036580026, -110551.30341332949}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0017966305689715385,  0.00090436947723267452},
            { 0.0017966305689715385, -0.00090436947723267452},
            {-0.0017966305689715385,  0.00090436947723267452},
            {-0.0017966305689715385, -0.00090436947723267452}
        }
    },{
        "geos_s",
        "+proj=geos   +R=6400000    +lat_1=0.5 +lat_2=2 +h=35785831",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223289.45763579503,  111677.65745653701},
            { 223289.45763579503,  -111677.65745653701},
            {-223289.45763579503,  111677.65745653701},
            {-223289.45763579503,  -111677.65745653701}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0017904931105078943,  0.00089524655504237148},
            { 0.0017904931105078943, -0.00089524655504237148},
            {-0.0017904931105078943,  0.00089524655504237148},
            {-0.0017904931105078943, -0.00089524655504237148}
        }
    },{
        "gins8",
        "+proj=gins8   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 194350.25093959007,  111703.90763533533},
            { 194350.25093959007, -111703.90763533533},
            {-194350.25093959007,  111703.90763533533},
            {-194350.25093959007, -111703.90763533533}
        },
        {},{{HUGE_VAL, HUGE_VAL}}
    },{
        "sinu_e",
        "+proj=sinu   +ellps=GRS80  +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 222605.29953946592,  110574.38855415257},
            { 222605.29953946592, -110574.38855415257},
            {-222605.29953946592,  110574.38855415257},
            {-222605.29953946592, -110574.38855415257}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0017966305684613522,  0.00090436947707945409},
            { 0.0017966305684613522, -0.00090436947707945409},
            {-0.0017966305684613522,  0.00090436947707945409},
            {-0.0017966305684613522, -0.00090436947707945409}
        }
    },{
        "sinu_s",
        "+proj=sinu   +R=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223368.11902663155,  111701.07212763709},
            { 223368.11902663155, -111701.07212763709},
            {-223368.11902663155,  111701.07212763709},
            {-223368.11902663155, -111701.07212763709}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0017904931100023887,  0.00089524655489191132},
            { 0.0017904931100023887, -0.00089524655489191132},
            {-0.0017904931100023887,  0.00089524655489191132},
            {-0.0017904931100023887, -0.00089524655489191132}
        }
    },{
        "eck6",
        "+proj=eck6   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 197021.60562899226,  126640.42073317352},
            { 197021.60562899226, -126640.42073317352},
            {-197021.60562899226,  126640.42073317352},
            {-197021.60562899226, -126640.42073317352}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.002029978749734037,  0.00078963032910382171},
            { 0.002029978749734037, -0.00078963032910382171},
            {-0.002029978749734037,  0.00078963032910382171},
            {-0.002029978749734037, -0.00078963032910382171}
        }
    },{
        "mbtfps",
        "+proj=mbtfps   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 204740.11747857218,  121864.72971934026},
            { 204740.11747857218, -121864.72971934026},
            {-204740.11747857218,  121864.72971934026},
            {-204740.11747857218, -121864.72971934026}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0019534152166442065,  0.00082057965689633387},
            { 0.0019534152166442065, -0.00082057965689633387},
            {-0.0019534152166442065,  0.00082057965689633387},
            {-0.0019534152166442065, -0.00082057965689633387}
        }
    },{
        "gn_sinu",
        "+proj=gn_sinu   +a=6400000    +lat_1=0.5 +lat_2=2 +m=1 +n=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223385.13250469571,  111698.23644718733},
            { 223385.13250469571, -111698.23644718733},
            {-223385.13250469571,  111698.23644718733},
            {-223385.13250469571, -111698.23644718733}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0017904931098931057,  0.00089524655491012516},
            { 0.0017904931098931057, -0.00089524655491012516},
            {-0.0017904931098931057,  0.00089524655491012516},
            {-0.0017904931098931057, -0.00089524655491012516}
        }
    },{
        "gnom",
        "+proj=gnom   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223492.92474718543,  111780.50920659291},
            { 223492.92474718543, -111780.50920659291},
            {-223492.92474718543,  111780.50920659291},
            {-223492.92474718543, -111780.50920659291}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0017904931092009798,  0.00089524655438192376},
            { 0.0017904931092009798, -0.00089524655438192376},
            {-0.0017904931092009798,  0.00089524655438192376},
            {-0.0017904931092009798, -0.00089524655438192376}
        }
    },{
        "goode",
        "+proj=goode   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223368.11902663155,  111701.07212763709},
            { 223368.11902663155, -111701.07212763709},
            {-223368.11902663155,  111701.07212763709},
            {-223368.11902663155, -111701.07212763709}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0017904931100023887,  0.00089524655489191132},
            { 0.0017904931100023887, -0.00089524655489191132},
            {-0.0017904931100023887,  0.00089524655489191132},
            {-0.0017904931100023887, -0.00089524655489191132}
        }
    },{
        "gstmerc",
        "+proj=gstmerc   +R=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223413.46640632182,  111769.14504058557},
            { 223413.46640632182, -111769.14504058668},
            {-223413.46640632302,  111769.14504058557},
            {-223413.46640632302, -111769.14504058668}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0017904931097109673,  0.0008952465544509083},
            { 0.0017904931097109673, -0.0008952465544509083},
            {-0.0017904931097109673,  0.0008952465544509083},
            {-0.0017904931097109673, -0.0008952465544509083}
        }
    },{
        "hammer",
        "+proj=hammer   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223373.78870324057,  111703.90739776699},
            { 223373.78870324057, -111703.90739776699},
            {-223373.78870324057,  111703.90739776699},
            {-223373.78870324057, -111703.90739776699}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.001790493109965961,  0.00089524655487369749},
            { 0.001790493109965961, -0.00089524655487369749},
            {-0.001790493109965961,  0.00089524655487369749},
            {-0.001790493109965961, -0.00089524655487369749}
        }
    },{
        "hatano",
        "+proj=hatano   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 189878.87894652804,  131409.8024406255 },
            { 189881.08195244463, -131409.14227607418 },
            {-189878.87894652804,  131409.8024406255 },
            {-189881.08195244463, -131409.14227607418 }
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0021064624821817597,  0.00076095689425791926 },
            { 0.0021064624821676096, -0.00076095777439265377 },
            {-0.0021064624821817597,  0.00076095689425791926 },
            {-0.0021064624821676096, -0.00076095777439265377 }
        }
    },{
        "healpix_e",
        "+proj=healpix   +ellps=GRS80  +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 222390.10394923863,  130406.58866448226},
            { 222390.10394923863, -130406.58866448054},
            {-222390.10394923863,  130406.58866448226},
            {-222390.10394923863, -130406.58866448054}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0017986411845524453,  0.00076679453057823619},
            { 0.0017986411845524453, -0.00076679453057823619},
            {-0.0017986411845524453,  0.00076679453057823619},
            {-0.0017986411845524453, -0.00076679453057823619}
        }
    },{
        "healpix_s",
        "+proj=healpix   +R=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223402.14425527418,  131588.04444199943},
            { 223402.14425527418, -131588.04444199943},
            {-223402.14425527418,  131588.04444199943},
            {-223402.14425527418, -131588.04444199943}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0017904931097838226,  0.00075990887733981202},
            { 0.0017904931097838226, -0.00075990887733981202},
            {-0.0017904931097838226,  0.00075990887733981202},
            {-0.0017904931097838226, -0.00075990887733981202}
        }
    },{
        "rhealpix_e",
        "+proj=rhealpix   +ellps=GRS80  +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 222390.10394923863,  130406.58866448226},
            { 222390.10394923863, -130406.58866448054},
            {-222390.10394923863,  130406.58866448226},
            {-222390.10394923863, -130406.58866448054}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0017986411845524453,  0.00076679453057823619},
            { 0.0017986411845524453, -0.00076679453057823619},
            {-0.0017986411845524453,  0.00076679453057823619},
            {-0.0017986411845524453, -0.00076679453057823619}
        }
    },{
        "rhealpix_s",
        "+proj=rhealpix   +R=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223402.14425527418,  131588.04444199943},
            { 223402.14425527418, -131588.04444199943},
            {-223402.14425527418,  131588.04444199943},
            {-223402.14425527418, -131588.04444199943}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0017904931097838226,  0.00075990887733981202},
            { 0.0017904931097838226, -0.00075990887733981202},
            {-0.0017904931097838226,  0.00075990887733981202},
            {-0.0017904931097838226, -0.00075990887733981202}
        }
    },
    // helmert projection is not implemented in Boost.Geometry
    // hgridshift projection is not implemented in Boost.Geometry
    // horner projection is not implemented in Boost.Geometry
    {
        "igh",
        "+proj=igh   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{ { 223878.49745627123,  111701.07212763709},
            { 223708.37131305804, -111701.07212763709},
            {-222857.74059699223,  111701.07212763709},
            {-223027.86674020503, -111701.07212763709}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{ { 0.001790489447892545,   0.00089524655489191132},
            { 0.0017904906685957927, -0.00089524655489191132},
            {-0.001790496772112032,   0.00089524655489191132},
            {-0.0017904955514087843, -0.00089524655489191132}
        }
    },{
        "imw_p",
        "+proj=imw_p   +ellps=GRS80  +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 222588.4411393762,   55321.128653809537},
            { 222756.90637768712, -165827.58428832365},
            {-222588.4411393762,   55321.128653809537},
            {-222756.90637768712, -165827.58428832365}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0017966991379592214, 0.50090492361427374},
            { 0.0017966979081574697, 0.49909507588689922},
            {-0.0017966991379592214, 0.50090492361427374},
            {-0.0017966979081574697, 0.49909507588689922}
        }
    },{
        "isea",
        "+proj=isea   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {-1097074.9480224741, 3442909.3090371834},
            {-1097074.9482647954, 3233611.7285857084},
            {-1575486.3536415542, 3442168.3420281881},
            {-1575486.353880283,  3234352.6955947056}
        },
        {},{{HUGE_VAL, HUGE_VAL}}
    },{
        "krovak",
        "+proj=krovak +ellps=GRS80  +no_defs",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {-3196535.2325636409,  -6617878.8675514441},
            {-3260035.4405521089,  -6898873.6148780314},
            {-3756305.3288691747,  -6478142.5615715114},
            {-3831703.6585019818,  -6759107.1701553948}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {24.836218918719162,  59.758403933233858},
            {24.836315484509566,  59.756888425730189},
            {24.830447747947495,  59.758403933233858},
            {24.830351182157091,  59.756888425730189}
        }
    },{
        "labrd",
        "+proj=labrd   +ellps=GRS80  +lon_0=0.5 +lat_0=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 166973.166090228391, -110536.912730266107},
            { 166973.168287157256, -331761.993650884193},
            {-278345.500519976194, -110469.032642031714},
            {-278345.504185269645, -331829.870790275279}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.501797719349373672, 2.00090435742047923},
            {0.501797717380853658, 1.99909564058898681},
            {0.498202280650626328, 2.00090435742047923},
            {0.498202282619146342, 1.99909564058898681}
        }
    },{
        "laea_e",
        "+proj=laea   +ellps=GRS80  +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 222602.471450095181,  110589.82722441027},
            { 222602.471450095181, -110589.827224408786},
            {-222602.471450095181,  110589.82722441027},
            {-222602.471450095181, -110589.827224408786}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00179663056847900867,  0.000904369475966495845},
            { 0.00179663056847900867, -0.000904369475966495845},
            {-0.00179663056847900867,  0.000904369475966495845},
            {-0.00179663056847900867, -0.000904369475966495845}
        }
    },{
        "laea_s",
        "+proj=laea   +R=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223365.281370124663,  111716.668072915665},
            { 223365.281370124663, -111716.668072915665},
            {-223365.281370124663,  111716.668072915665},
            {-223365.281370124663, -111716.668072915665}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00179049311002060264,  0.000895246554791735271},
            { 0.00179049311002060264, -0.000895246554791735271},
            {-0.00179049311002060264,  0.000895246554791735271},
            {-0.00179049311002060264, -0.000895246554791735271}
        }
    },{
        "lagrng",
        "+proj=lagrng   +a=6400000 +W=2   +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 111703.37591722561,   27929.8319080333386},
            { 111699.122088816002, -83784.1780133577704},
            {-111703.37591722561,   27929.8319080333386},
            {-111699.122088816002, -83784.1780133577704}
        },
        {},{{HUGE_VAL, HUGE_VAL}}
    },{
        "larr",
        "+proj=larr   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {223393.637624200899,  111707.215961255497},
            {223393.637624200899,  -111707.215961255497},
            {-223393.637624200899,  111707.215961255497},
            {-223393.637624200899,  -111707.215961255497}
        },
        {},{{HUGE_VAL, HUGE_VAL}}
    },{
        "lask",
        "+proj=lask   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 217928.275907355128,  112144.32922014239},
            { 217928.275907355128, -112144.32922014239},
            {-217928.275907355128,  112144.32922014239},
            {-217928.275907355128, -112144.32922014239}
        },
        {},{{HUGE_VAL, HUGE_VAL}}
    },{
        "lcc",
        "+proj=lcc   +ellps=GRS80  +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 222588.439735968423,  110660.533870799671},
            { 222756.879700278747, -110532.797660827026},
            {-222588.439735968423,  110660.533870799671},
            {-222756.879700278747, -110532.797660827026}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00179635940600536667,  0.000904232207322381741},
            { 0.00179635817735249777, -0.000904233135128348995},
            {-0.00179635940600536667,  0.000904232207322381741},
            {-0.00179635817735249777, -0.000904233135128348995}
        }
    },{
        "lcca",
        "+proj=lcca   +ellps=GRS80  +lat_0=1 +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 222605.285770237417,  67.8060072715846616},
            { 222740.037637936533, -221125.539829601563},
            {-222605.285770237417,  67.8060072715846616},
            {-222740.037637936533, -221125.539829601563}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00179690290525662526, 1.00090436621350798},
            { 0.00179690192174008037, 0.999095632791497268},
            {-0.00179690290525662526, 1.00090436621350798},
            {-0.00179690192174008037, 0.999095632791497268}
        }
    },{
        "loxim",
        "+proj=loxim   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223382.295791338867,  55850.5360638185448},
            { 223393.637462243292, -167551.608191455656},
            {-223382.295791338867,  55850.5360638185448},
            {-223393.637462243292, -167551.608191455656}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00179056141104335601, 0.500895246554891926},
            { 0.00179056116683692576, 0.499104753445108074},
            {-0.00179056141104335601, 0.500895246554891926},
            {-0.00179056116683692576, 0.499104753445108074}
        }
    },{
        "lsat",
        "+proj=lsat +ellps=GRS80 +lat_1=0.5 +lat_2=2 +lsat=1 +path=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {18241950.01455855,    9998256.83982293494},
            {18746856.2533194572, 10215761.669925211},
            {18565503.6836331636,  9085039.14672705345},
            {19019696.9020289108,  9247763.0394328218}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {126.000423834530011,  0.00172378224025701425},
            {126.002213738256714,  0.00188015467480917966},
            {126.000734468914601, -0.00188015467480917966},
            {126.002524372641304, -0.00172378224025701425}
        }
    }, {
        "mbt_fps",
        "+proj=mbt_fps   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 198798.176129849948,  125512.017254530627},
            { 198798.176129849948, -125512.017254530627},
            {-198798.176129849948,  125512.017254530627},
            {-198798.176129849948, -125512.017254530627}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00201197086238270742,  0.000796711850174446003},
            { 0.00201197086238270742, -0.000796711850174446003},
            {-0.00201197086238270742,  0.000796711850174446003},
            {-0.00201197086238270742, -0.000796711850174446003}
        }
    },{
        "mbtfpp",
        "+proj=mbtfpp   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {206804.786929820373,  120649.762565792524},
            {206804.786929820373,  -120649.762565792524},
            {-206804.786929820373,  120649.762565792524},
            {-206804.786929820373,  -120649.762565792524}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.00193395359462902698,  0.00082883725477665357},
            {0.00193395359462902698,  -0.00082883725477665357},
            {-0.00193395359462902698,  0.00082883725477665357},
            {-0.00193395359462902698,  -0.00082883725477665357}
        }
    },{
        "mbtfpq",
        "+proj=mbtfpq   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 209391.854738393013,  119161.040199054827},
            { 209391.854738393013, -119161.040199054827},
            {-209391.854738393013,  119161.040199054827},
            {-209391.854738393013, -119161.040199054827}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00191010555824111571,  0.000839185447792341723},
            { 0.00191010555824111571, -0.000839185447792341723},
            {-0.00191010555824111571,  0.000839185447792341723},
            {-0.00191010555824111571, -0.000839185447792341723}
        }
    },{
        "merc_e",
        "+proj=merc   +ellps=GRS80  +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 222638.981586547132,  110579.965218249708},
            { 222638.981586547132, -110579.965218249112},
            {-222638.981586547132,  110579.965218249708},
            {-222638.981586547132, -110579.965218249112}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00179663056823904264,  0.00090436947522799056},
            { 0.00179663056823904264, -0.00090436947522799056},
            {-0.00179663056823904264,  0.00090436947522799056},
            {-0.00179663056823904264,  -0.00090436947522799056}
        }
    },{
        "merc_s",
        "+proj=merc   +R=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223402.144255274179,  111706.743574944077},
            { 223402.144255274179, -111706.743574944485},
            {-223402.144255274179,  111706.743574944077},
            {-223402.144255274179, -111706.743574944485}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00179049310978382265,  0.000895246554845297135},
            { 0.00179049310978382265, -0.000895246554858019272},
            {-0.00179049310978382265,  0.000895246554845297135},
            {-0.00179049310978382265, -0.000895246554858019272}
        }
    },{
        "mill",
        "+proj=mill   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223402.144255274179,  111704.701754393827},
            { 223402.144255274179, -111704.701754396243},
            {-223402.144255274179,  111704.701754393827},
            {-223402.144255274179, -111704.701754396243}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00179049310978382265,  0.000895246554873922024},
            { 0.00179049310978382265, -0.000895246554873922024},
            {-0.00179049310978382265,  0.000895246554873922024},
            {-0.00179049310978382265, -0.000895246554873922024}
        }
    },
    // the following projections are not implemented in Boost.Geometry
    /*{
        "misrsom_e",
        "+proj=misrsom   +ellps=GRS80  +lat_1=0.5 +lat_2=2 +path=1",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {18556630.3683698252, 9533394.6753112711},
            {19041866.0067297369, 9707182.17532352544},
            {18816810.1301847994, 8647669.64980295487},
            {19252610.7845367305, 8778164.08580140397}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {127.759503987730625,  0.00173515039622462014},
            {127.761295471077958,  0.00187196632421706517},
            {127.759775773557251, -0.00187196632421891525},
            {127.76156725690457,  -0.00173515039622462014}
        }
    },{
        "misrsom_s",
        "+proj=misrsom   +R=6400000    +lat_1=0.5 +lat_2=2 +path=1",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {18641249.2791703865, 9563342.53233416565},
            {19130982.4615812786, 9739539.59350463562},
            {18903483.5150115378, 8675064.50061797537},
            {19343388.3998006098, 8807471.90406848863}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {127.75950514818588,   0.00171623111593511971},
            {127.761290323778738,  0.00185412132880796244},
            {127.759780920856471, -0.00185412132880796244},
            {127.761566096449329, -0.00171623111593511971}
        }
    },{
        "mil_os",
        "+proj=mil_os   +R=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {-1908527.94959420455, -1726237.4730614475},
            {-1916673.02291848511, -1943133.88812552323},
            {-2344429.41208962305, -1706258.05121891224},
            {-2354637.83553299867, -1926468.60513541684}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {20.0020363939492398, 18.0009683469140498},
            {20.0020363715837419, 17.999031631815086},
            {19.9979636060507602, 18.0009683469140498},
            {19.9979636284162581, 17.999031631815086}
        }
    },{
        "lee_os",
        "+proj=lee_os   +R=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {-25564478.9526050538, 154490848.8286255},
            { 30115393.9385746419, 125193997.439701974},
            {-31039340.5921660066,  57678685.0448915437},
            {-3088419.93942357088,  58150091.0991110131}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {-164.997479457813824,  -9.99875886103541411},
            {-164.997479438558884, -10.0012411200022751},
            {-165.002520542186289,  -9.99875886103545142},
            {-165.002520561440946, -10.0012411200022999}
        }
    },{
        "gs48",
        "+proj=gs48 +R=6370997",
        // All latitudes and longitudes within the continental US
        {
            { -119.0, 40.0},
            {  -70.0, 64.0},
            {  -80.0, 25.0},
            {  -95.0, 35.0}
        },{
            { -1923908.446529345820,   355874.658944479190},
            {  1354020.375109298155,  3040846.007866524626},
            {  1625139.160484319553, -1413614.894029108109},
            {    90241.658071457961,  -439595.048485902138}
        },{
            { -1923000.0,   355000.0},
            {  1354000.0,  3040000.0},
            {  1625000.0, -1413000.0},
            {    90000.0,  -439000.0}
        },{
            {-118.987112613284, 39.994449789388},
            { -70.005208999424, 63.993387835525},
            { -80.000346610440, 25.005602546594},
            { -95.002606473071, 35.005424705030}
        }
    },{
        "alsk_e",
        "+proj=alsk +ellps=clrk66",
        {
            {-160.0, 55.0},
            {-160.0, 70.0},
            {-145.0, 70.0},
            {-145.0, 60.0}
        },{
            {-513253.146950842060, -968928.031867943470},
            {-305001.133897637190,  687494.464958650530},
            {266454.305088600490,   683423.477493030950},
            {389141.322439243960,  -423913.251230396680}
        },{
            {-500000.0, -950000.0},
            {-305000.0,  700000.0},
            { 250000.0,  700000.0},
            { 400000.0, -400000.0}
        },{
            {-159.830804302926, 55.183195262220},
            {-160.042203155537, 70.111086864056},
            {-145.381043551466, 70.163900908411},
            {-144.758985461448, 60.202929200739}
        }
    },{
        "alsk_s",
        "+proj=alsk +R=6370997",
        {
            {-160.0, 55.0},
            {-160.0, 70.0},
            {-145.0, 70.0},
            {-145.0, 60.0}
        },{
            {-511510.319410844070, -967150.991676078060},
            {-303744.771290368980,  685439.745941123230},
            {265354.974019662940,   681386.892874573010},
            {387711.995394026630,  -422980.685505462640}
        },{
            {-500000.0, -950000.0},
            {-305000.0,  700000.0},
            { 250000.0,  700000.0},
            { 400000.0, -400000.0}
        },{
            {-159.854014457557, 55.165653849074},
            {-160.082332371601, 70.128307617632},
            {-145.347827407243, 70.181566919011},
            {-144.734239827146, 60.193564732505}
        }
    },{
        "gs50_e",
        "+proj=gs50 +ellps=clrk66",
        {
            {-160.0, 65.0},
            {-130.0, 45.0},
            { -65.0, 45.0},
            { -80.0, 36.0}
        },{
            {-1874628.5377402329,   2660907.942291015300},
            { -771831.51885333552,    48465.166491304852},
            { 4030931.8339815089,   1323687.864777399200},
            { 3450764.2615361013,   -175619.041820732440}
        },{
            {-1800000.0, 2600000.0},
            { -800000.0,  500000.0},
            { 4000000.0, 1300000.0},
            { 3900000.0, -170000.0}
        },{
            {-157.989284999679, 64.851559609698},
            {-131.171390466814, 49.084969745967},
            { -65.491568685301, 44.992837923774},
            { -75.550660091101, 34.191114075743}
        }
    },{
        "gs50_s",
        "+proj=gs50 +R=6370997",
        {
            {-160.0, 65.0},
            {-130.0, 45.0},
            { -65.0, 45.0},
            { -80.0, 36.0}
        },{
            {-1867268.2534600089,   2656506.230401823300},
            { -769572.18967299373,    48324.312440863941},
            { 4019393.068680791200, 1320191.309350289200},
            { 3442685.615172345700, -178760.423489428680}
        },{
            {-1800000.0, 2600000.0},
            { -800000.0,  500000.0},
            { 4000000.0, 1300000.0},
            { 3900000.0, -170000.0}
        },{
            {-158.163295044933, 64.854288364994},
            {-131.206816959506, 49.082915350974},
            { -65.348945220767, 44.957292681774},
            { -75.446820242089, 34.185406225616}
        }
    },*/
    {
        "moll",
        "+proj=moll   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {201113.698641813244,  124066.283433859542},
            {201113.698641813244,  -124066.283433859542},
            {-201113.698641813244,  124066.283433859542},
            {-201113.698641813244,  -124066.283433859542}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.00198873782220854774,  0.000806005080362811612},
            {0.00198873782220854774,  -0.000806005080362811612},
            {-0.00198873782220854774,  0.000806005080362811612},
            {-0.00198873782220854774,  -0.000806005080362811612}
        }
    },{
        "wag4",
        "+proj=wag4   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 192801.218662384286,  129416.216394802992},
            { 192801.218662384286, -129416.216394802992},
            {-192801.218662384286,  129416.216394802992},
            {-192801.218662384286, -129416.216394802992}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00207450259783523421, 0.000772682950537716476},
            { 0.00207450259783523421, -0.000772682950537716476},
            {-0.00207450259783523421,  0.000772682950537716476},
            {-0.00207450259783523421, -0.000772682950537716476}
       }
    },{
        "wag5",
        "+proj=wag5   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 203227.05192532466,  138651.631442713202},
            { 203227.05192532466, -138651.631442713202},
            {-203227.05192532466,  138651.631442713202},
            {-203227.05192532466, -138651.631442713202}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00196807227086416396,  0.00072121615041701424},
            { 0.00196807227086416396, -0.00072121615041701424},
            {-0.00196807227086416396,  0.00072121615041701424},
            {-0.00196807227086416396, -0.00072121615041701424}
        }
    },{
        "natearth",
        "+proj=natearth   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 194507.265257889288,  112508.737358294515},
            { 194507.265257889288, -112508.737358294515},
            {-194507.265257889288,  112508.737358294515},
            {-194507.265257889288, -112508.737358294515}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00205638349586440223,  0.000888823913291242177},
            { 0.00205638349586440223, -0.000888823913291242177},
            {-0.00205638349586440223,  0.000888823913291242177},
            {-0.00205638349586440223, -0.000888823913291242177}
        }
    },
    // the following projection is not implemented in Boost.Geometry
    /*{
        "natearth2",
        "+proj=natearth2   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 189255.172934730799,  113022.495810907014},
            { 189255.172934730799, -113022.495810907014},
            {-189255.172934730799,  113022.495810907014},
            {-189255.172934730799, -113022.495810907014}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00211344929691056112,  0.000884779612080993237},
            { 0.00211344929691056112, -0.000884779612080993237},
            {-0.00211344929691056112,  0.000884779612080993237},
            {-0.00211344929691056112, -0.000884779612080993237}
        }
    },*/
    {
        "nell",
        "+proj=nell   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223385.132504695706,  111698.23644718733},
            { 223385.132504695706, -111698.23644718733},
            {-223385.132504695706,  111698.23644718733},
            {-223385.132504695706, -111698.23644718733}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00179049310989310567,  0.000895246554910125161},
            { 0.00179049310989310567, -0.000895246554910125161},
            {-0.00179049310989310567,  0.000895246554910125161},
            {-0.00179049310989310567, -0.000895246554910125161}
        }
    },{
        "nell_h",
        "+proj=nell_h   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223385.131640952837,  111698.236533561678},
            { 223385.131640952837, -111698.236533561678},
            {-223385.131640952837,  111698.236533561678},
            {-223385.131640952837, -111698.236533561678}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00179049310989310567,  0.000895246554910125378},
            { 0.00179049310989310567, -0.000895246554910125378},
            {-0.00179049310989310567,  0.000895246554910125378},
            {-0.00179049310989310567, -0.000895246554910125378}
        }
    },{
        "nicol",
        "+proj=nicol   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223374.561814139714,  111732.553988545071},
            { 223374.561814139714, -111732.553988545071},
            {-223374.561814139714,  111732.553988545071},
            {-223374.561814139714, -111732.553988545071}
        },
        {},{{HUGE_VAL, HUGE_VAL}}
    },{
        "nsper",
        "+proj=nsper   +a=6400000  +h=1000000",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 222239.816114099842,  111153.763991924759},
            { 222239.816114099842, -111153.763991924759},
            {-222239.816114099842,  111153.763991924759},
            {-222239.816114099842, -111153.763991924759}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00179049311728792437,  0.000895246558425396135},
            { 0.00179049311728792437, -0.000895246558425396135},
            {-0.00179049311728792437,  0.000895246558425396135},
            {-0.00179049311728792437, -0.000895246558425396135}
        }
    },{
        "tpers",
        "+proj=tpers   +a=6400000  +h=1000000 +azi=20",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 170820.288955531199,  180460.865555804776},
            { 246853.941538942483, -28439.8780357754222},
            {-246853.941538942483,  28439.8780357754222},
            {-170820.288955531199, -180460.865555804776}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00198870552603137678,  0.000228871872278689991},
            { 0.00137632081376749859, -0.00145364129728205432},
            {-0.00137632081376749859,  0.00145364129728205432},
            {-0.00198870552603137678, -0.000228871872278689991}
        }
    },{
        "nzmg",
        "+proj=nzmg   +ellps=GRS80  +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {3352675144.74742508,  -7043205391.10024357},
            {3691989502.77930641,  -6729069415.33210468},
            {4099000768.45323849,  -7863208779.66724873},
            {4466166927.36997604,  -7502531736.62860489}
        },{
            { 200000, 100000},
            { 200000,-100000},
            {-200000, 100000},
            {-200000,-100000}
        },{
            {175.48208682711271,  -69.4226921826331846},
            {175.756819472543611, -69.5335710883796168},
            {134.605119233460016, -61.4599957106629091},
            {134.333684315954827, -61.6215536756024349}
        }
    },{
        "ob_tran",
        "+proj=ob_tran +a=6400000 +o_proj=latlon +o_lon_p=20 +o_lat_p=20 +lon_0=180",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {-2.6856872138416592, 1.2374302350496296},
            {-2.6954069748943286, 1.2026833954513816},
            {-2.8993663925401947, 1.2374302350496296},
            {-2.8896466314875244, 1.2026833954513816}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 121.5518748407577,  -2.5361001573966084},
            { 63.261184340201858,  17.585319578673531},
            {-141.10073322351622,  26.091712304855108},
            {-65.862385598848391,  51.830295078417215}
        }
    },{
        "ocea",
        "+proj=ocea   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {19994423.837934087962,  223322.760576727800},
            {20217962.128015257418,  223322.760576729401},
            {19994423.837934091687, -223322.760576726549},
            {20217962.128015264869, -223322.760576724948},
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 179.999104753445,  0.001790493110},
            {-179.999104753445,  0.001790493110},
            { 179.999104753445, -0.001790493110},
            {-179.999104753445, -0.001790493110}
        }
    },{
        "oea",
        "+proj=oea   +a=6400000    +lat_1=0.5 +lat_2=2 +n=1 +m=2 +theta=3",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 228926.872097864107,  99870.4884300760023},
            { 217242.584036940476, -123247.885607474513},
            {-217242.584036940476,  123247.885607474556},
            {-228926.872097864078, -99870.4884300760168}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0017411857167771369,   0.000987726819566195693},
            { 0.00183489288577854998, -0.000800312481495174641},
            {-0.00183489288577854954,  0.000800312481495174966},
            {-0.00174118571677713712, -0.000987726819566195043}
        }
    },{
        "omerc",
        "+proj=omerc   +ellps=GRS80  +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 222650.796885261341,  110642.229314983808},
            { 222650.796885261341, -110642.229314983808},
            {-222650.796885261545,  110642.229314983808},
            {-222650.796885261545, -110642.229314983808}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00179663056816996357,  0.000904369474808157338},
            { 0.00179663056816996357, -0.000904369474820879583},
            {-0.0017966305681604536,   0.000904369474808157338},
            {-0.0017966305681604536,  -0.000904369474820879583}
        }
    },{
        "ortho",
        "+proj=ortho   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223322.76057672748,  111695.401198614476},
            { 223322.76057672748, -111695.401198614476},
            {-223322.76057672748,  111695.401198614476},
            {-223322.76057672748, -111695.401198614476}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0017904931102938101,  0.000895246554928338998},
            { 0.0017904931102938101, -0.000895246554928338998},
            {-0.0017904931102938101,  0.000895246554928338998},
            {-0.0017904931102938101, -0.000895246554928338998}
        }
    },
    // patterson projection is not implemented in Boost.Geometry
    /*{
        "patterson",
        "+proj=patterson   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {223402.144255274179,  113354.250397779804},
            {223402.144255274179,  -113354.250397779804},
            {-223402.144255274179,  113354.250397779804},
            {-223402.144255274179,  -113354.250397779804}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.00179049310978382265,  0.000882190140807953657},
            {0.00179049310978382265,  -0.000882190140807953657},
            {-0.00179049310978382265,  0.000882190140807953657},
            {-0.00179049310978382265,  -0.000882190140807953657}
        }
    },*/
    // pipeline projection is not implemented in Boost.Geometry
    // and tests are implemented differently
    {
        "poly_e",
        "+proj=poly   +ellps=GRS80  +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 222605.285770237475,  110642.194561440483},
            { 222605.285770237475, -110642.194561440483},
            {-222605.285770237475,  110642.194561440483},
            {-222605.285770237475, -110642.194561440483}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00179663056846135222,  0.000904369476631838518},
            { 0.00179663056846135222, -0.000904369476631838518},
            {-0.00179663056846135222,  0.000904369476631838518},
            {-0.00179663056846135222, -0.000904369476631838518}
        }
    },{
        "poly_s",
        "+proj=poly   +R=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223368.105210218986,  111769.110491224754},
            { 223368.105210218986, -111769.110491224754},
            {-223368.105210218986,  111769.110491224754},
            {-223368.105210218986, -111769.110491224754}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0017904931100023887,  0.000895246554454779222},
            { 0.0017904931100023887, -0.000895246554454779222},
            {-0.0017904931100023887,  0.000895246554454779222},
            {-0.0017904931100023887, -0.000895246554454779222}
        }
    },{
        "putp2",
        "+proj=putp2   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 211638.039634339279,  117895.033043379764},
            { 211638.039634339279, -117895.033043379764},
            {-211638.039634339279,  117895.033043379764},
            {-211638.039634339279, -117895.033043379764}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00188980221640386672,  0.000848201580276863377},
            { 0.00188980221640386672, -0.000848201580276863377},
            {-0.00188980221640386672,  0.000848201580276863377},
            {-0.00188980221640386672, -0.000848201580276863377}
        }
    },{
        "putp3",
        "+proj=putp3   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 178227.115507793525,  89124.5607860879827},
            { 178227.115507793525, -89124.5607860879827},
            {-178227.115507793525,  89124.5607860879827},
            {-178227.115507793525, -89124.5607860879827}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00224405032986489889,  0.00112202516475805899},
            { 0.00224405032986489889, -0.00112202516475805899},
            {-0.00224405032986489889,  0.00112202516475805899},
            {-0.00224405032986489889, -0.00112202516475805899}
        }
    },{
        "putp3p",
        "+proj=putp3p   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 178238.118539984745,  89124.5607860879827},
            { 178238.118539984745, -89124.5607860879827},
            {-178238.118539984745,  89124.5607860879827},
            {-178238.118539984745, -89124.5607860879827}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00224405032969050844,  0.00112202516475805899},
            { 0.00224405032969050844, -0.00112202516475805899},
            {-0.00224405032969050844,  0.00112202516475805899},
            {-0.00224405032969050844, -0.00112202516475805899}
        }
    },{
        "putp4p",
        "+proj=putp4p   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 195241.47734938623,  127796.782307926231},
            { 195241.47734938623, -127796.782307926231},
            {-195241.47734938623,  127796.782307926231},
            {-195241.47734938623, -127796.782307926231}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00204852830860296001,  0.000782480174932193733},
            { 0.00204852830860296001, -0.000782480174932193733},
            {-0.00204852830860296001,  0.000782480174932193733},
            {-0.00204852830860296001, -0.000782480174932193733}
        }
    },{
        "weren",
        "+proj=weren   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223378.515757633519,  146214.093042288267},
            { 223378.515757633519, -146214.093042288267},
            {-223378.515757633519,  146214.093042288267},
            {-223378.515757633519, -146214.093042288267}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00179049310987240413,  0.000683917989676492265},
            { 0.00179049310987240413, -0.000683917989676492265},
            {-0.00179049310987240413,  0.000683917989676492265},
            {-0.00179049310987240413, -0.000683917989676492265}
        }
    },{
        "putp5",
        "+proj=putp5   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 226367.21338056153,  113204.56855847509},
            { 226367.21338056153, -113204.56855847509},
            {-226367.21338056153,  113204.56855847509},
            {-226367.21338056153, -113204.56855847509}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00176671315102969553,  0.000883356575387199546},
            { 0.00176671315102969553, -0.000883356575387199546},
            {-0.00176671315102969553,  0.000883356575387199546},
            {-0.00176671315102969553, -0.000883356575387199546}
        }
    },{
        "putp5p",
        "+proj=putp5p   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 226388.175248755841,  113204.56855847509},
            { 226388.175248755841, -113204.56855847509},
            {-226388.175248755841,  113204.56855847509},
            {-226388.175248755841, -113204.56855847509}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00176671315090204742,  0.000883356575387199546},
            { 0.00176671315090204742, -0.000883356575387199546},
            {-0.00176671315090204742,  0.000883356575387199546},
            {-0.00176671315090204742, -0.000883356575387199546}
        }
    },{
        "putp6",
        "+proj=putp6   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 226369.395133402577,  110218.523796520662},
            { 226369.395133402577, -110218.523796520749},
            {-226369.395133402577,  110218.523796520662},
            {-226369.395133402577, -110218.523796520749}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00176671315102969921,  0.000907295534210503544},
            { 0.00176671315102969921, -0.000907295534205924308},
            {-0.00176671315102969921,  0.000907295534210503544},
            {-0.00176671315102969921, -0.000907295534205924308}
        }
    },{
        "putp6p",
        "+proj=putp6p   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 198034.195132195076,  125989.475461323193},
            { 198034.195132195076, -125989.475461323193},
            {-198034.195132195076,  125989.475461323193},
            {-198034.195132195076, -125989.475461323193}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00201955053120177067,  0.000793716441164738612},
            { 0.00201955053120177067, -0.000793716441164738612},
            {-0.00201955053120177067,  0.000793716441164738612},
            {-0.00201955053120177067, -0.000793716441164738612}
        }
    },{
        "qsc_e",
        "+proj=qsc   +ellps=GRS80  +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 304638.450843852363,  164123.870923793991},
            { 304638.450843852363, -164123.870923793991},
            {-304638.450843852363,  164123.870923793962},
            {-304638.450843852421, -164123.870923793904}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00132134098471627126,  0.000610652900922527926},
            { 0.00132134098471627126, -0.000610652900922527926},
            {-0.00132134098471627126,  0.000610652900922527926},
            {-0.00132134098471627126, -0.000610652900922527926}
        }
    },{
        "qsc_s",
        "+proj=qsc   +R=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 305863.792402890511,  165827.722754715243},
            { 305863.792402890511, -165827.722754715243},
            {-305863.792402890511,  165827.722754715243},
            {-305863.792402890569, -165827.722754715156}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.00131682718763827234,  0.000604493198178676161},
            { 0.00131682718763827234, -0.000604493198178676161},
            {-0.00131682718763827234,  0.000604493198178676161},
            {-0.00131682718763827234, -0.000604493198178676161}
        }
    },{
        "robin",
        "+proj=robin   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{ // original expected different because float coefficients are defined
            /*{ 189588.423282507836,  107318.530350702888},
            { 189588.423282507836, -107318.530350702888},
            {-189588.423282507836,  107318.530350702888},
            {-189588.423282507836, -107318.530350702888}*/
            { 189588.4232821252,  107318.5272684303},
            { 189588.4232821252, -107318.5272684303},
            {-189588.4232821252,  107318.5272684303},
            {-189588.4232821252, -107318.5272684303}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.002109689065506131,  0.000931805533547745983},
            { 0.002109689065506131, -0.000931805533547745983},
            {-0.002109689065506131,  0.000931805533547745983},
            {-0.002109689065506131, -0.000931805533547745983}
        }
    },{
        "rpoly",
        "+proj=rpoly   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223368.09830201423,  111769.110486991223},
            { 223368.09830201423, -111769.110486991223},
            {-223368.09830201423,  111769.110486991223},
            {-223368.09830201423, -111769.110486991223}
        },
        {},{{HUGE_VAL, HUGE_VAL}}
    },
    // sch projection not implemented in Boost.Geometry
    // and the tests are not implemented in Proj4 anyway   
    {
        "euler_e",
        "+proj=euler   +ellps=GRS80  +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {222597.63465910763,  111404.24054991946},
            {222767.16563187627,  -111234.6764910177},
            {-222597.63465910763,  111404.24054991946},
            {-222767.16563187627,  -111234.6764910177}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0017962807023075235,  0.0008983146697688839},
            {0.0017962794738334226,  -0.00089831589842987965},
            {-0.0017962807023075235,  0.0008983146697688839},
            {-0.0017962794738334226,  -0.00089831589842987965}
        }
    },{
        "euler_s",
        "+proj=euler   +a=6400000    +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {223360.65559869423,  111786.11238979101},
            {223530.76769031584,  -111615.96709862351},
            {-223360.65559869423,  111786.11238979101},
            {-223530.76769031584,  -111615.96709862351}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0017901444369360026,  0.00089524594522202015},
            {0.001790143216840731,  -0.00089524716533368484},
            {-0.0017901444369360026,  0.00089524594522202015},
            {-0.001790143216840731,  -0.00089524716533368484}
        }
    },{
        "murd1_e",
        "+proj=murd1   +ellps=GRS80  +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {222600.81347355421,  111404.24418054636},
            {222770.3492878644,  -111234.6728566746},
            {-222600.81347355421,  111404.24418054636},
            {-222770.3492878644,  -111234.6728566746}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0017962550410516366,  0.0008983146697688839},
            {0.0017962538125775522,  -0.00089831589842987965},
            {-0.0017962550410516366,  0.0008983146697688839},
            {-0.0017962538125775522,  -0.00089831589842987965}
        }
    },{
        "murd1_s",
        "+proj=murd1   +a=6400000    +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {223363.84530949194,  111786.11603286299},
            {223533.96225925098,  -111615.96345182261},
            {-223363.84530949194,  111786.11603286299},
            {-223533.96225925098,  -111615.96345182261}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0017901188633413715,  0.00089524594522202015},
            {0.0017901176432461162,  -0.00089524716492657387},
            {-0.0017901188633413715,  0.00089524594522202015},
            {-0.0017901176432461162,  -0.00089524716492657387}
        }
    },{
        "murd2_e",
        "+proj=murd2   +ellps=GRS80  +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {222588.09975123021,  111426.14002741246},
            {222757.72626701824,  -111341.43131750476},
            {-222588.09975123021,  111426.14002741246},
            {-222757.72626701824,  -111341.43131750476}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0017963574947305447,  0.00089788747830845382},
            {0.0017963562661689487,  -0.00089788809264252983},
            {-0.0017963574947305447,  0.00089788747830845382},
            {-0.0017963562661689487,  -0.00089788809264252983}
        }
    },{
        "murd2_s",
        "+proj=murd2   +a=6400000    +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {223351.08800702673,  111808.08693438848},
            {223521.2959691704,  -111723.08785967289},
            {-223351.08800702673,  111808.08693438848},
            {-223521.2959691704,  -111723.08785967289}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0017902209670287586,  0.00089482021163422854},
            {0.0017902197468465887,  -0.00089482082161134206},
            {-0.0017902209670287586,  0.00089482021163422854},
            {-0.0017902197468465887,  -0.00089482082161134206}
        }
    },{
        "murd3_e",
        "+proj=murd3   +ellps=GRS80  +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {222600.81407757697,  111404.24660137216},
            {222770.35473389886,  -111234.67043217793},
            {-222600.81407757697,  111404.24660137216},
            {-222770.35473389886,  -111234.67043217793}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0017962550166583809,  0.0008983146697688839},
            {0.0017962537881492445,  -0.00089831589842987965},
            {-0.0017962550166583809,  0.0008983146697688839},
            {-0.0017962537881492445,  -0.00089831589842987965}
        }
    },{
        "murd3_s",
        "+proj=murd3   +a=6400000    +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {223363.84591558515,  111786.11846198692},
            {223533.96772395336,  -111615.96101901523},
            {-223363.84591558515,  111786.11846198692},
            {-223533.96772395336,  -111615.96101901523}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0017901188390313859,  0.00089524594522202015},
            {0.0017901176189013177,  -0.00089524716533368484},
            {-0.0017901188390313859,  0.00089524594522202015},
            {-0.0017901176189013177,  -0.00089524716533368484}
        }
    },{
        "pconic_e",
        "+proj=pconic   +ellps=GRS80  +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {222588.09884161691,  111416.60477006658},
            {222757.71809109033,  -111331.88153107995},
            {-222588.09884161691,  111416.60477006658},
            {-222757.71809109033,  -111331.88153107995}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0017963575313784969,  0.0008979644089172499},
            {0.0017963563027642206,  -0.00089796502355327969},
            {-0.0017963575313784969,  0.0008979644089172499},
            {-0.0017963563027642206,  -0.00089796502355327969}
        }
    },{
        "pconic_s",
        "+proj=pconic   +a=6400000    +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {223351.08709429545,  111798.5189920546},
            {223521.28776521701,  -111713.50533845725},
            {-223351.08709429545,  111798.5189920546},
            {-223521.28776521701,  -111713.50533845725}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0017902210035514285,  0.0008948968793741558},
            {0.0017902197833169374,  -0.00089489748965381963},
            {-0.0017902210035514285,  0.0008948968793741558},
            {-0.0017902197833169374,  -0.00089489748965381963}
        }
    },{
        "tissot_e",
        "+proj=tissot   +ellps=GRS80  +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {222641.07869963095,  54347.828487281469},
            {222810.61451394114,  -168291.08854993948},
            {-222641.07869963095,  54347.828487281469},
            {-222810.61451394114,  -168291.08854993948}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0017962807107425871,  0.51344495513064536},
            {0.0017962794822333915,  0.51164832456244658},
            {-0.0017962807107425871,  0.51344495513064536},
            {-0.0017962794822333915,  0.51164832456244658}
        }
    },{
        "tissot_s",
        "+proj=tissot   +a=6400000    +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {223404.24855684943,  54534.122161157939},
            {223574.36550660848,  -168867.95732352766},
            {-223404.24855684943,  54534.122161157939},
            {-223574.36550660848,  -168867.95732352766}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0017901444453421915,  0.51344188640609856},
            {0.001790143225212064,  0.51165139329554277},
            {-0.0017901444453421915,  0.51344188640609856},
            {-0.001790143225212064,  0.51165139329554277}
        }
    },{
        "vitk1_e",
        "+proj=vitk1   +ellps=GRS80  +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {222607.17121145778,  111404.25144243463},
            {222776.71670959776,  -111234.66558744459},
            {-222607.17121145778,  111404.25144243463},
            {-222776.71670959776,  -111234.66558744459}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0017962037198570686,  0.0008983146697688839},
            {0.0017962024913830157,  -0.00089831589842987965},
            {-0.0017962037198570686,  0.0008983146697688839},
            {-0.0017962024913830157,  -0.00089831589842987965}
        }
    },{
        "vitk1_s",
        "+proj=vitk1   +a=6400000    +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {223370.22484047143,  111786.12331964359},
            {223540.3515072545,  -111615.9561576751},
            {-223370.22484047143,  111786.12331964359},
            {-223540.3515072545,  -111615.9561576751}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0017900677174648159,  0.00089524594522202015},
            {0.0017900664973695916,  -0.00089524716533368484},
            {-0.0017900677174648159,  0.00089524594522202015},
            {-0.0017900664973695916,  -0.00089524716533368484}
        }
    },{
        "somerc_e",
        "+proj=somerc   +ellps=GRS80  +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {222638.98158654713,  110579.96521824898},
            {222638.98158654713,  -110579.96521825089},
            {-222638.98158654713,  110579.96521824898},
            {-222638.98158654713,  -110579.96521825089}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0017966305682390426,  0.00090436947704129484},
            {0.0017966305682390426,  -0.00090436947704377105},
            {-0.0017966305682390426,  0.00090436947704129484},
            {-0.0017966305682390426,  -0.00090436947704377105}
        }
    },{
        "somerc_s",
        "+proj=somerc   +R=6400000    +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {223402.14425527418,  111706.74357494408},
            {223402.14425527418,  -111706.74357494518},
            {-223402.14425527418,  111706.74357494408},
            {-223402.14425527418,  -111706.74357494518}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0017904931097838226,  0.00089524655485801927},
            {0.0017904931097838226,  -0.00089524655484529714},
            {-0.0017904931097838226,  0.00089524655485801927},
            {-0.0017904931097838226,  -0.00089524655484529714}
        }
    },{
        "stere_e",
        "+proj=stere   +ellps=GRS80  +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 222644.8545501172,  110610.8834741739},
            { 222644.8545501172, -110610.8834741739},
            {-222644.8545501172,  110610.8834741739},
            {-222644.8545501172, -110610.8834741739}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0017966305682022392,  0.00090436947502443507},
            { 0.0017966305682022392, -0.00090436947502443507},
            {-0.0017966305682022392,  0.00090436947502443507},
            {-0.0017966305682022392, -0.00090436947502443507}
        }
    },{
        "stere_s",
        "+proj=stere   +R=6400000    +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223407.81025950745,  111737.938996443},
            { 223407.81025950745, -111737.938996443},
            {-223407.81025950745,  111737.938996443},
            {-223407.81025950745, -111737.938996443}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.001790493109747395,  0.00089524655465513144},
            { 0.001790493109747395, -0.00089524655465513144},
            {-0.001790493109747395,  0.00089524655465513144},
            {-0.001790493109747395, -0.00089524655465513144}
        }
    },{
        "ups",
        "+proj=ups   +ellps=GRS80  +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {2433455.5634384668,  -10412543.301512826},
            {2448749.1185681992,  -10850493.419804076},
            {1566544.4365615332,  -10412543.301512826},
            {1551250.8814318008,  -10850493.419804076}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {-44.998567498074834,  64.9182362867341},
            {-44.995702709112308,  64.917020250675748},
            {-45.004297076028529,  64.915804280954518},
            {-45.001432287066002,  64.914588377560719}
        }
    },{
        "sterea_e",
        "+proj=sterea   +ellps=GRS80  +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 222644.89410919772,  110611.09187173686},
            { 222644.89410919772, -110611.09187173827},
            {-222644.89410919772,  110611.09187173686},
            {-222644.89410919772, -110611.09187173827}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0017966305682019911,  0.00090436947683099009},
            { 0.0017966305682019911, -0.00090436947684371233},
            {-0.0017966305682019911,  0.00090436947683099009},
            {-0.0017966305682019911, -0.00090436947684371233}
        }
    },{
        "sterea_s",
        "+proj=sterea   +R=6400000    +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223407.81025950745,  111737.93899644315},
            { 223407.81025950745, -111737.93899644315},
            {-223407.81025950745,  111737.93899644315},
            {-223407.81025950745, -111737.93899644315}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.001790493109747395,  0.00089524655465446378},
            { 0.001790493109747395, -0.00089524655465446378},
            {-0.001790493109747395,  0.00089524655465446378},
            {-0.001790493109747395, -0.00089524655465446378}
        }
    },{
        "fouc_e",
        "+proj=fouc   +ellps=GRS80  +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {222588.12067589167,  111322.31670069379},
            {222588.12067589167,  -111322.31670069379},
            {-222588.12067589167,  111322.31670069379},
            {-222588.12067589167,  -111322.31670069379}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0017966305685702751,  0.00089831528410111959},
            {0.0017966305685702751,  -0.00089831528410111959},
            {-0.0017966305685702751,  0.00089831528410111959},
            {-0.0017966305685702751,  -0.00089831528410111959}
        }
    },{
        "fouc_s",
        "+proj=fouc   +a=6400000    +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {223351.10900341379,  111703.9077217125},
            {223351.10900341379,  -111703.9077217125},
            {-223351.10900341379,  111703.9077217125},
            {-223351.10900341379,  -111703.9077217125}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0017904931101116717,  0.00089524655487369749},
            {0.0017904931101116717,  -0.00089524655487369749},
            {-0.0017904931101116717,  0.00089524655487369749},
            {-0.0017904931101116717,  -0.00089524655487369749}
        }
    },{
        "kav5_e",
        "+proj=kav5   +ellps=GRS80  +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {200360.90530882866,  123685.08247699818},
            {200360.90530882866,  -123685.08247699818},
            {-200360.90530882866,  123685.08247699818},
            {-200360.90530882866,  -123685.08247699818}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0019962591348533314,  0.00080848256185253912},
            {0.0019962591348533314,  -0.00080848256185253912},
            {-0.0019962591348533314,  0.00080848256185253912},
            {-0.0019962591348533314,  -0.00080848256185253912}
        }
    },{
        "kav5_s",
        "+proj=kav5   +a=6400000    +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {201047.7031108776,  124109.05062917093},
            {201047.7031108776,  -124109.05062917093},
            {-201047.7031108776,  124109.05062917093},
            {-201047.7031108776,  -124109.05062917093}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0019894397264987643,  0.00080572070962591153},
            {0.0019894397264987643,  -0.00080572070962591153},
            {-0.0019894397264987643,  0.00080572070962591153},
            {-0.0019894397264987643,  -0.00080572070962591153}
        }
    },{
        "qua_aut_e",
        "+proj=qua_aut   +ellps=GRS80  +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {222613.54903309655,  111318.07788798446},
            {222613.54903309655,  -111318.07788798446},
            {-222613.54903309655,  111318.07788798446},
            {-222613.54903309655,  -111318.07788798446}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0017966305684046586,  0.00089831528412872229},
            {0.0017966305684046586,  -0.00089831528412872229},
            {-0.0017966305684046586,  0.00089831528412872229},
            {-0.0017966305684046586,  -0.00089831528412872229}
        }
    },{
        "qua_aut_s",
        "+proj=qua_aut   +a=6400000    +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {223376.62452402918,  111699.65437918637},
            {223376.62452402918,  -111699.65437918637},
            {-223376.62452402918,  111699.65437918637},
            {-223376.62452402918,  -111699.65437918637}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0017904931099477471,  0.00089524655490101819},
            {0.0017904931099477471,  -0.00089524655490101819},
            {-0.0017904931099477471,  0.00089524655490101819},
            {-0.0017904931099477471,  -0.00089524655490101819}
        }
    },{
        "mbt_s_e",
        "+proj=mbt_s   +ellps=GRS80  +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {204131.51785027285,  121400.33022550763},
            {204131.51785027285,  -121400.33022550763},
            {-204131.51785027285,  121400.33022550763},
            {-204131.51785027285,  -121400.33022550763}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0019593827209883237,  0.00082369854658027549},
            {0.0019593827209883237,  -0.00082369854658027549},
            {-0.0019593827209883237,  0.00082369854658027549},
            {-0.0019593827209883237,  -0.00082369854658027549}
        }
    },{
        "mbt_s_s",
        "+proj=mbt_s   +a=6400000    +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {204831.24057099217,  121816.46669603503},
            {204831.24057099217,  -121816.46669603503},
            {-204831.24057099217,  121816.46669603503},
            {-204831.24057099217,  -121816.46669603503}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0019526892859206603,  0.00082088471512331508},
            {0.0019526892859206603,  -0.00082088471512331508},
            {-0.0019526892859206603,  0.00082088471512331508},
            {-0.0019526892859206603,  -0.00082088471512331508}
        }
    },{
        "tcc",
        "+proj=tcc   +a=6400000    +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {223458.84419245756,  111769.14504058579},
            {223458.84419245756,  -111769.14504058579},
            {-223458.84419245756,  111769.14504058579},
            {-223458.84419245756,  -111769.14504058579}
        },
        {},{{HUGE_VAL, HUGE_VAL}}
    },{
        "tcea",
        "+proj=tcea   +a=6400000    +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223322.76057672748,  111769.14504058579},
            { 223322.76057672748, -111769.14504058579},
            {-223322.76057672748,  111769.14504058579},
            {-223322.76057672748, -111769.14504058579}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0017904931102938101,  0.00089524655445477922},
            { 0.0017904931102938101, -0.00089524655445477922},
            {-0.0017904931102938101,  0.00089524655445477922},
            {-0.0017904931102938101, -0.00089524655445477922}
        }
    },
    // times projection not implemented
    {
        "tmerc_e",
        "+proj=tmerc   +ellps=GRS80  +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 222650.7967975855, 110642.2294119332},
            { 222650.7967975855, -110642.2294119332},
            {-222650.7967975855, 110642.2294119332},
            {-222650.7967975855, -110642.2294119332}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0017966305681649396,  0.00090436947663183841},
            { 0.0017966305681649396, -0.00090436947663183841},
            {-0.0017966305681649396,  0.00090436947663183841},
            {-0.0017966305681649396, -0.00090436947663183841}
        }
    },{
        "tmerc_s",
        "+proj=tmerc   +R=6400000    +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223413.46640632232,  111769.14504059685},
            { 223413.46640632232, -111769.14504059685},
            {-223413.46640632208,  111769.14504059685},
            {-223413.46640632208, -111769.14504059685}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0017904931097048034,  0.00089524670602767842},
            { 0.0017904931097048034, -0.00089524670602767842},
            {-0.001790493109714345,   0.00089524670602767842},
            {-0.001790493109714345,  -0.00089524670602767842}
        }
    },{
        "tpeqd_e",
        "+proj=tpeqd   +ellps=GRS80  +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {-27750.758831679042,  -222599.40369177726},
            {-250434.93702403645,  -222655.93819326628},
            {-27750.758831679042,  222599.40369177726},
            {-250434.93702403645,  222655.93819326628}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {-0.00089855554821257374,  1.2517966304145272},
            {0.0008985555481998515,  1.2517966304145272},
            {-0.00089855431859741167,  1.2482033692781642},
            {0.00089855431859741167,  1.2482033692781642}
        }
    },{
        "tpeqd_s",
        "+proj=tpeqd   +a=6400000    +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {-27845.882978485075,  -223362.43069526015},
            {-251293.37876465076,  -223419.15898590829},
            {-27845.882978485075,  223362.43069526015},
            {-251293.37876465076,  223419.15898590829}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {-0.00089548606640108474,  1.2517904929571837},
            {0.0008954860663883625,  1.2517904929571837},
            {-0.000895484845182587,  1.248209506737604},
            {0.00089548484516986475,  1.248209506737604}
        }
    },{
        "urm5",
        "+proj=urm5   +a=6400000    +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223393.6384339639,  111696.81878511712},
            { 223393.6384339639, -111696.81878511712},
            {-223393.6384339639,  111696.81878511712},
            {-223393.6384339639, -111696.81878511712}
        },
        {},{{HUGE_VAL, HUGE_VAL}}
    },{
        "urmfps",
        "+proj=urmfps   +a=6400000    +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 196001.70813419219,  127306.84332999329},
            { 196001.70813419219, -127306.84332999329},
            {-196001.70813419219,  127306.84332999329},
            {-196001.70813419219, -127306.84332999329}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.002040720839642371,  0.00078547381740438178},
            { 0.002040720839642371, -0.00078547381740438178},
            {-0.002040720839642371,  0.00078547381740438178},
            {-0.002040720839642371, -0.00078547381740438178}
        }
    },{
        "wag1",
        "+proj=wag1   +a=6400000    +lat_1=0.5 +lat_2=2 +n=0.5",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 195986.78156115755,  127310.07506065986},
            { 195986.78156115755, -127310.07506065986},
            {-195986.78156115755,  127310.07506065986},
            {-195986.78156115755, -127310.07506065986}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.002040720839738254,  0.00078547381739207999},
            { 0.002040720839738254, -0.00078547381739207999},
            {-0.002040720839738254,  0.00078547381739207999},
            {-0.002040720839738254, -0.00078547381739207999}
        }
    },{
        "vandg",
        "+proj=vandg   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223395.24954340671,  111704.59663367498},
            { 223395.24954340671, -111704.59663367498},
            {-223395.24954340671,  111704.59663367498},
            {-223395.24954340671, -111704.59663367498}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.001790493715929761,  0.00089524655486993867},
            { 0.001790493715929761, -0.00089524655486993867},
            {-0.001790493715929761,  0.00089524655486993867},
            {-0.001790493715929761, -0.00089524655486993867}
        }
    },{
        "vandg2",
        "+proj=vandg2   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223395.24785043663,  111718.49103722633},
            { 223395.24785043663, -111718.49103722633},
            {-223395.24785043663,  111718.49103722633},
            {-223395.24785043663, -111718.49103722633}
        },
        {},{{HUGE_VAL, HUGE_VAL}}
    },{
        "vandg3",
        "+proj=vandg3   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223395.24955283134,  111704.51990442065},
            { 223395.24955283134, -111704.51990442065},
            {-223395.24955283134,  111704.51990442065},
            {-223395.24955283134, -111704.51990442065}
        },
        {},{{HUGE_VAL, HUGE_VAL}}
    },{
        "vandg4",
        "+proj=vandg4   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223374.57729435508,  111701.19548415358 },
            { 223374.57729435508, -111701.19548415358 },
            {-223374.57729435508,  111701.19548415358 },
            {-223374.57729435508, -111701.19548415358 }
        },
        {},{{HUGE_VAL, HUGE_VAL}}
    },
    // vgridshift projection is not implemented in Boost.Geometry
    // and the Proj4 test is in a different format
    {
        "wag2",
        "+proj=wag2   +a=6400000  +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 206589.88809996162,   120778.04035754716},
            { 206589.88809996162,  -120778.04035754716},
            {-206589.88809996162,   120778.04035754716},
            {-206589.88809996162,  -120778.04035754716}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0019360240367390709,   0.00082795765763814082},
            { 0.0019360240367390709,  -0.00082795765763814082},
            {-0.0019360240367390709,   0.00082795765763814082},
            {-0.0019360240367390709,  -0.00082795765763814082}
        }
    },{
        "wag3",
        "+proj=wag3   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {223387.02171816575,  111701.07212763709},
            {223387.02171816575,  -111701.07212763709},
            {-223387.02171816575,  111701.07212763709},
            {-223387.02171816575,  -111701.07212763709}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.001790493109880963,  0.00089524655489191132},
            {0.001790493109880963,  -0.00089524655489191132},
            {-0.001790493109880963,  0.00089524655489191132},
            {-0.001790493109880963,  -0.00089524655489191132}
        }
    },{
        "wag7",
        "+proj=wag7   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 198601.87695731167,  125637.0457141714},
            { 198601.87695731167, -125637.0457141714},
            {-198601.87695731167,  125637.0457141714},
            {-198601.87695731167, -125637.0457141714}
        },
        {},{{HUGE_VAL, HUGE_VAL}}
    },{
        "wink1",
        "+proj=wink1   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223385.13164095284,  111701.07212763709},
            { 223385.13164095284,  -111701.07212763709},
            {-223385.13164095284,  111701.07212763709},
            {-223385.13164095284,  -111701.07212763709}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            { 0.0017904931098931057,  0.00089524655489191132},
            { 0.0017904931098931057, -0.00089524655489191132},
            {-0.0017904931098931057,  0.00089524655489191132},
            {-0.0017904931098931057, -0.00089524655489191132}
        }
    },{
        "wink2",
        "+proj=wink2   +a=6400000    +lat_1=0.5 +lat_2=2",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            { 223387.39643378611,  124752.03279744535},
            { 223387.39643378611, -124752.03279744535},
            {-223387.39643378611,  124752.03279744535},
            {-223387.39643378611, -124752.03279744535}
        },
        {},{{HUGE_VAL, HUGE_VAL}}
    },{
        "etmerc",
        "+proj=etmerc   +ellps=GRS80  +lat_1=0.5 +lat_2=2 +n=0.5 +zone=30",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {222650.79679758562,   110642.22941193319},
            {222650.79679758562,  -110642.22941193319},
            {-222650.79679758562,  110642.22941193319},
            {-222650.79679758562, -110642.22941193319}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {0.0017966305681649398,   0.00090436947663183873},
            {0.0017966305681649398,  -0.00090436947663183873},
            {-0.0017966305681649398,  0.00090436947663183873},
            {-0.0017966305681649398, -0.00090436947663183873}
        }
    },{
        "utm",
        "+proj=utm   +ellps=GRS80  +lat_1=0.5 +lat_2=2 +n=0.5 +zone=30",
        {
            { 2, 1},
            { 2,-1},
            {-2, 1},
            {-2,-1}
        },{
            {1057002.4054912981,  110955.14117594929},
            {1057002.4054912981, -110955.14117594929},
            {611263.81227890507,  110547.10569680421},
            {611263.81227890507, -110547.10569680421}
        },{
            { 200, 100},
            { 200,-100},
            {-200, 100},
            {-200,-100}
        },{
            {-7.4869520833902357,  0.00090193980983462605},
            {-7.4869520833902357, -0.00090193980983462605},
            {-7.4905356820622613,  0.00090193535121489081},
            {-7.4905356820622613, -0.00090193535121489081}
        }
    }
};

#endif // BOOST_GEOMETRY_TEST_SRS_PROJECTION_SELFTEST_CASES_HPP
