/*
 * Copyright (C) 2022 Tobias Brunner
 *
 * Copyright (C) secunet Security Networks AG
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

#include <crypto/crypto_tester.h>

/**
 * This is an old test vector derived from one in RFC 4868.
 */
kdf_test_vector_t prf_plus_sha256_old = {
	.alg = KDF_PRF_PLUS, .arg.prf = PRF_HMAC_SHA2_256,
	.key = chunk_from_chars(
		0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
		0x0b,0x0b,0x0b,0x0b),
	.salt = chunk_from_chars(
		0x48,0x69,0x20,0x54,0x68,0x65,0x72,0x65),
	.out = chunk_from_chars(
		0xb9,0xbd,0xc0,0x89,0x88,0xb4,0xc2,0xb7,0x5a,0xa9,0x3e,0x59,0x6a,0xc8,0x42,0x05,
		0xfa,0x2d,0xdd,0xe1,0xbf,0x7a,0x25,0x72,0x06,0x7b,0x00,0xe1,0x4b,0x23,0x77,0x32,
		0x83,0x05,0x09,0x98,0x1a,0xd2,0xf9,0x4a,0x8c,0x32,0xa4,0x7d,0xaa,0x22,0x55,0xb6,
		0x60,0xc4,0x36,0x34,0x7a,0xe7,0x56,0xa6,0xed,0xc0,0x23,0x47,0x7d,0x80,0x95,0x90,
		0xe6,0x82,0xf6,0x1d,0x9c,0x04,0xb0,0x6b,0x4a,0xd9,0x71,0xa3,0x4c,0x81,0x47,0xfa,
		0x66,0x79,0x2f,0xf1,0x43,0x4b,0x93,0xc7,0x22,0xb3,0x2e,0x12,0xf4,0x88,0x32,0xeb,
		0xc1,0x5c,0xe2,0x36,0x9c,0xe7,0x1f,0xe9,0xb7,0xb8,0x1e,0x57,0x04,0xc1,0x4d,0x0f,
		0x52,0x80,0xa6,0xec,0x62,0x6e,0x99,0x2d,0x7a,0x9f),
};

/**
 * The following test vectors are from CAVP/SP 800-135 for IKEv2.
 *
 * key = SKEYSEED, salt = Ni | Nr | SPIi | SPIr (one vector with min. and one
 * with max. size for nonces)
 */
kdf_test_vector_t prf_plus_sha256_1 = {
	.alg = KDF_PRF_PLUS, .arg.prf = PRF_HMAC_SHA2_256,
	.key = chunk_from_chars(
		0x37,0xfd,0xe9,0x0b,0x81,0xd6,0x36,0x92,0x62,0x0f,0x67,0x36,0x7b,0x62,0x09,0x2e,
		0x3e,0xfd,0xc6,0xa0,0x35,0x09,0x13,0x7d,0x73,0x10,0x04,0x76,0x7f,0xf3,0x50,0x10),
	.salt = chunk_from_chars(
		0xed,0x80,0xdc,0x79,0x91,0x2c,0x32,0xa9,0x35,0xfb,0x6d,0x1a,0x3f,0xea,0xc0,0x78,
		0x47,0xc1,0x85,0x8e,0xfc,0x93,0x2e,0xa4,0x60,0x6f,0xd0,0x56,0x09,0x62,0x40,0x02),
	.out = chunk_from_chars(
		0x63,0xd0,0x18,0xa0,0x8f,0x7a,0x29,0xda,0xa7,0xd1,0xf9,0x2c,0xb6,0x3f,0x45,0xfa,
		0x39,0xbb,0xaf,0x5e,0x5f,0x1a,0x78,0x13,0x96,0x13,0xbb,0x89,0x1e,0x3e,0x6b,0xf9,
		0xd0,0x44,0x8f,0x7c,0x74,0xe0,0x5b,0x4c,0x1a,0x25,0xc0,0xb6,0x1e,0x62,0xc8,0x75,
		0xe8,0x52,0xf2,0x87,0xe1,0x92,0xd4,0xae,0x53,0x61,0x12,0xdb,0x51,0x97,0x21,0x64,
		0x00,0xbd,0x5e,0x12,0x34,0x70,0xef,0xc1,0x5f,0x53,0x16,0x18,0x65,0xfe,0x8b,0x19,
		0xa1,0x41,0x65,0x18,0xa1,0x57,0xd6,0x51,0xec,0xd2,0xca,0xdd,0xab,0x6a,0x9e,0xae,
		0x6a,0x27,0xa5,0x5a,0xf0,0x88,0x21,0x8d,0x51,0x56,0xed,0xaa,0x97,0x89,0xf9,0x34,
		0xd9,0x83,0x6e,0xd4,0x3e,0xdf,0xfd,0xa5,0x53,0x82,0x22,0x02,0xff,0x9f,0x7d,0x48,
		0x60,0xca,0x72,0xe0),
};

kdf_test_vector_t prf_plus_sha256_2 = {
	.alg = KDF_PRF_PLUS, .arg.prf = PRF_HMAC_SHA2_256,
	.key = chunk_from_chars(
		0x29,0xbd,0x11,0x55,0x68,0xae,0x09,0x88,0x27,0x0f,0xc3,0x86,0xd3,0x95,0xfe,0x37,
		0x07,0xa4,0xd0,0x62,0x89,0xf3,0x52,0xbb,0xa4,0xc0,0x0a,0x9a,0xd8,0x55,0xa0,0x8d),
	.salt = chunk_from_chars(
		0xef,0xa7,0x29,0x13,0x18,0x22,0x78,0xff,0xbd,0x14,0xe7,0x89,0x20,0xc0,0x62,0x51,
		0x9a,0xba,0xb8,0xc1,0x6e,0x5e,0xd7,0x0c,0x08,0x41,0xa4,0x8c,0xdb,0x98,0x23,0x7e,
		0xe9,0x3f,0x73,0x5f,0xb2,0xdf,0x18,0x43,0x58,0xaa,0xdc,0x13,0xd8,0x3f,0x43,0xfe,
		0x8d,0x87,0x05,0x17,0x6c,0x8c,0xa3,0x13,0x82,0x5a,0x1b,0xcd,0xf7,0x79,0x11,0xc5,
		0x98,0x00,0x16,0x71,0xff,0xbf,0x01,0x4e,0x37,0xa8,0xc0,0x4d,0x49,0xa7,0x83,0x9d,
		0xfe,0xa6,0xcd,0xc5,0x87,0x68,0x8d,0x45,0x88,0xfe,0x43,0x23,0x5b,0x71,0x69,0x3f,
		0xfd,0x07,0x29,0x33,0x86,0xb6,0xbf,0x4c,0x19,0x9e,0x33,0x61,0x65,0xb2,0x60,0x78,
		0x77,0x36,0xf5,0x4b,0xe9,0x5d,0xb6,0x91,0x16,0x38,0x8b,0xc2,0xec,0xa2,0xb3,0xb2,
		0x94,0x84,0x71,0x74,0x17,0xbb,0x3c,0x71,0x81,0x4c,0xe1,0x3b,0x84,0x44,0x6d,0xc3,
		0x96,0x4c,0x30,0x29,0x84,0xf9,0x77,0x81,0xf6,0x31,0x66,0x24,0x08,0x90,0x10,0x7c,
		0x2e,0x75,0x1a,0x00,0x43,0x6f,0x7c,0x3c,0x9f,0xf1,0x27,0x60,0xe4,0x9d,0x91,0x56,
		0x3b,0xe6,0x03,0xfd,0x96,0x41,0xa0,0xa6,0x49,0x18,0xa9,0x32,0x91,0xed,0x11,0x3d,
		0xb1,0x2f,0x97,0x07,0x60,0x9d,0x17,0x20,0x96,0xeb,0x58,0xf9,0x15,0x44,0x74,0xda,
		0x40,0xc1,0xf5,0xc0,0x90,0x3e,0x9c,0xa2,0xf9,0x1b,0xa6,0x60,0x07,0x75,0xdf,0x71,
		0x66,0xca,0xf8,0xe3,0x27,0x85,0x9e,0x67,0x62,0x32,0xd3,0x40,0x46,0x04,0x4c,0xee,
		0x43,0xf9,0x01,0x9f,0x04,0x68,0x56,0x12,0x63,0x5b,0x99,0xcb,0xeb,0xcb,0x36,0x3d,
		0x56,0x5e,0xaf,0x0e,0x54,0x7e,0xec,0xb9,0x41,0xc9,0x94,0xdf,0xd4,0x71,0xed,0x56,
		0x43,0xc0,0x87,0x74,0x4f,0x77,0x09,0xcc,0x3e,0x25,0x10,0xf2,0x74,0x26,0xc6,0x2c,
		0x0f,0xf3,0xac,0xb3,0xc2,0x76,0x61,0xd2,0x6a,0x6d,0x83,0xc2,0xa2,0x5e,0x13,0xa6,
		0xd4,0x65,0xbd,0x04,0x7f,0x90,0x55,0x00,0xe5,0xeb,0xbe,0x42,0x66,0x43,0x0d,0x56,
		0x67,0x14,0x0f,0x77,0xe7,0x97,0x71,0x2a,0x8c,0x8f,0x63,0xc5,0x83,0xf4,0xb4,0x64,
		0x9b,0x72,0x89,0x9e,0xa3,0x4a,0xbf,0xdb,0x17,0x61,0x7c,0x46,0x0c,0x35,0xf2,0x50,
		0x64,0x94,0x49,0x4f,0x22,0x3e,0x25,0x1a,0xc7,0x1a,0x5b,0x9b,0x7e,0xea,0x87,0xf4,
		0xf5,0xe3,0x33,0xa3,0xc1,0xbb,0xb4,0xbb,0x09,0x25,0x8b,0x6a,0x4b,0x5f,0x8c,0x9f,
		0xb8,0x2b,0xf4,0x2d,0xa9,0xd4,0xa4,0x65,0x43,0xc6,0xa9,0xeb,0x9a,0xa3,0x0e,0xa9,
		0xda,0x80,0x19,0x15,0xab,0xcc,0x17,0x12,0xd0,0x82,0xf2,0x92,0xa6,0x3f,0xd9,0xaf,
		0x71,0x54,0xa9,0x7c,0xc7,0x38,0x59,0xa4,0xbe,0x3c,0xba,0x35,0x9d,0x32,0x18,0x00,
		0x4e,0x14,0xdf,0x02,0xd0,0x9e,0xdf,0x0a,0xd5,0x79,0x6b,0xb0,0x10,0x99,0x52,0x93,
		0xab,0x5d,0x04,0x2c,0x31,0x05,0x53,0x80,0xcc,0x9c,0xb2,0xe3,0x61,0x79,0x82,0xc4,
		0x5f,0x6b,0xce,0x1f,0xb8,0xa4,0x0e,0xf9,0xea,0xc4,0x8a,0xe1,0x77,0x20,0xdf,0xec,
		0xc7,0x1e,0xc8,0x57,0xea,0x33,0xf3,0x2e,0xb3,0x46,0xba,0x60,0x36,0xe8,0xf9,0xcc,
		0xd7,0xbc,0xad,0xc6,0xc1,0xab,0x92,0xa8,0x0c,0x57,0xe7,0x89,0x59,0xd8,0xb8,0x28,
		0x57,0xe9,0x1d,0xf9,0xc5,0xff,0xb8,0x42,0x4d,0x5e,0xad,0xac,0x0e,0x57,0x0f,0x7c),
	.out = chunk_from_chars(
		0xd5,0x03,0x3d,0x08,0x79,0x34,0xc6,0x15,0x38,0xb2,0x3d,0xff,0x87,0x5b,0x3b,0xa9,
		0x20,0xe0,0x5c,0x1b,0x42,0xac,0x7f,0x97,0x93,0x99,0x5e,0x76,0xba,0x3e,0x46,0x1d,
		0x6e,0x83,0xb2,0xfb,0xe0,0xfa,0x68,0x1b,0xa2,0x85,0x69,0x6d,0x53,0xb1,0x75,0xe0,
		0x70,0xc4,0xc1,0xcb,0xc7,0x40,0x43,0xf7,0xca,0xc5,0x58,0xc4,0x94,0xac,0xd2,0x3a,
		0xab,0xfc,0x7a,0x68,0x5b,0x62,0x74,0x13,0x44,0x08,0xca,0xfc,0x16,0x92,0x85,0x2a,
		0xca,0x66,0x3c,0xef,0xd7,0xdd,0x0c,0x8a,0x87,0x2a,0x36,0xa1,0x41,0x4d,0xb4,0xb7,
		0xf3,0x14,0xad,0x0e,0x49,0xf2,0xc1,0x87,0x16,0x2e,0x28,0x76,0xc6,0x05,0x3d,0xbe,
		0xf5,0xa2,0xea,0x8c,0x33,0xdc,0xd3,0xdd,0x91,0x68,0x76,0x44,0x37,0x7c,0x45,0xf4,
		0xb5,0xc4,0x18,0xe1,0xe1,0x0b,0xec,0x22,0xba,0xac,0x31,0x3b,0x90,0x2b,0xdf,0xfc,
		0xfb,0x69,0xc6,0x8e,0xf4,0xe7,0x1c,0x5a,0xcc,0x39,0xa9,0xf0,0x7a,0xcb,0x1f,0xc3,
		0xba,0x8e,0x92,0xc6,0xdc,0xa8,0x61,0xa6,0x41,0x63,0xb0,0x68,0x9d,0xa4,0xa9,0x7f,
		0x12,0x40,0x26,0x36,0x09,0x90,0xc9,0xf1,0xae,0x4e,0x8d,0x02,0x98,0xf0,0x77,0xfc,
		0xbb,0x0c,0x20,0x3f,0xd2,0xca,0x82,0x1d,0xea,0xb0,0x59,0x1a,0x22,0x1d,0x9d,0x0a,
		0x79,0x8d,0x02,0xf0,0x78,0xd8,0xfb,0x6e,0x93,0x1c,0xa4,0x46,0xa1,0x3f,0x11,0x63,
		0x8c,0x6b,0x6a,0xfb,0x82,0x25,0xa0,0x08,0xf9,0x38,0xa6,0x87,0x8c,0x5b,0x57,0x7e,
		0x03,0xae,0xcf,0xa7,0x31,0x41,0x01,0xed,0xd0,0xfd,0xa3,0xbd,0xbe,0xa4,0x98,0x5c),
};

kdf_test_vector_t prf_plus_sha384_1 = {
	.alg = KDF_PRF_PLUS, .arg.prf = PRF_HMAC_SHA2_384,
	.key = chunk_from_chars(
		0x54,0x43,0x6a,0x9e,0xa9,0x5d,0x6f,0xf7,0x9b,0x96,0x7f,0x4b,0x07,0xf6,0xde,0x97,
		0x6a,0x37,0x6e,0x8e,0xa2,0x6a,0xa9,0x57,0x47,0x09,0xaf,0xc6,0x02,0x43,0xc9,0xc1,
		0x41,0xda,0x4c,0xa0,0xe1,0x58,0xe6,0x27,0xa7,0x5e,0xa8,0x7f,0x6f,0xeb,0x07,0xef),
	.salt = chunk_from_chars(
		0xd6,0x72,0xb0,0xbc,0x85,0x28,0x29,0xb9,0x35,0x09,0xf3,0xb7,0x24,0x70,0x63,0x64,
		0x0f,0x9a,0x01,0x6d,0x7b,0x22,0x22,0x3c,0x85,0xf7,0x10,0xda,0xf8,0xb9,0x82,0xd8),
	.out = chunk_from_chars(
		0x10,0xb0,0x0a,0x65,0x46,0x43,0xb3,0xb9,0x85,0x76,0x4f,0x86,0xe2,0x3e,0xf6,0x24,
		0x31,0x12,0x2f,0xcc,0x96,0xde,0x16,0xe3,0x38,0xa3,0x72,0xbe,0xe4,0x93,0x91,0xdb,
		0x1f,0x83,0xa7,0x35,0x8c,0x08,0x03,0x80,0xab,0xbf,0x4a,0xf4,0x7c,0xdb,0x83,0xb9,
		0x53,0x03,0x43,0x16,0x8f,0x7d,0xbc,0x92,0x21,0x26,0x0e,0x15,0x46,0xc6,0xb3,0x83,
		0xe2,0x23,0x5d,0x27,0x05,0xa7,0x27,0xf0,0xb4,0x1a,0xeb,0xf6,0xa1,0xb4,0xc6,0x1c,
		0xb4,0x60,0x66,0x01,0x52,0xa1,0x7b,0xa7,0x4c,0x20,0xd5,0x19,0x1b,0xa6,0x84,0x17,
		0x94,0x94,0x52,0x70,0xd8,0x29,0x58,0x7b,0x7f,0x32,0x94,0x63,0x9e,0xa7,0xb7,0x71,
		0x49,0xe1,0xd0,0x72,0x30,0xac,0x16,0xe0,0xcd,0x5a,0xb2,0x36,0xfa,0x32,0x12,0xbd,
		0x56,0x9e,0xe7,0x0f),
};

kdf_test_vector_t prf_plus_sha384_2 = {
	.alg = KDF_PRF_PLUS, .arg.prf = PRF_HMAC_SHA2_384,
	.key = chunk_from_chars(
		0xff,0x66,0xe9,0xd0,0x92,0xdc,0x01,0xe0,0xb8,0x1f,0x93,0x9f,0x52,0xf5,0xc0,0x7d,
		0x38,0xd8,0x05,0xb9,0x86,0x28,0xce,0x1a,0xc5,0xfe,0x94,0xc0,0x98,0x57,0x76,0x47,
		0x33,0x9f,0xad,0x68,0x94,0x1f,0xfe,0x21,0xe0,0x1e,0xfb,0x4e,0x70,0x50,0x21,0x3b),
	.salt = chunk_from_chars(
		0x9e,0x2a,0x62,0xf9,0x36,0x28,0x93,0xdd,0xf8,0x47,0x16,0xfe,0xc2,0xf2,0x3f,0x9d,
		0xcb,0xd9,0x01,0x0d,0xf6,0xfe,0x9e,0x0e,0xb4,0x6d,0x03,0xd9,0x14,0xf3,0x04,0xd8,
		0xfe,0x4d,0x3e,0xe2,0xd6,0xa0,0x3b,0x40,0xe5,0x6a,0x32,0x5e,0x82,0x2a,0x17,0x36,
		0x19,0x29,0x18,0x4a,0xde,0x09,0xea,0xa4,0x45,0x27,0x8d,0x38,0x70,0x41,0x7a,0x7c,
		0xf5,0x65,0x58,0x4f,0x57,0x2a,0xd3,0x4f,0xf7,0x2b,0xc7,0x78,0x1a,0x39,0xa4,0x8b,
		0x54,0xb5,0x5d,0x6e,0xb0,0xed,0x68,0x55,0x1b,0x22,0x2c,0x7a,0xfa,0xda,0x0b,0xc8,
		0x22,0x36,0xec,0x31,0xce,0x6c,0x04,0x6b,0x3b,0x2e,0xdb,0x2d,0xef,0x61,0xf4,0xd2,
		0xd8,0x57,0xb2,0xd1,0xcb,0x36,0x96,0xc0,0x23,0xe0,0x8c,0x08,0xd6,0xab,0xd0,0x4d,
		0x4f,0x69,0x09,0x2b,0x14,0x58,0x37,0x35,0xb9,0xe9,0x18,0xae,0xe5,0xa3,0x99,0x9d,
		0xcb,0xf5,0x8f,0xda,0xd6,0xfc,0xd4,0x7c,0x95,0x92,0x98,0x77,0x03,0x0e,0x54,0xb7,
		0x08,0x23,0x5f,0x2a,0x2e,0x11,0xe7,0xc5,0x85,0x84,0x61,0x9a,0xa2,0xfa,0x69,0x31,
		0x53,0x44,0xd3,0x65,0x7b,0x55,0x72,0x0a,0x25,0xeb,0xe8,0x8e,0xa0,0x77,0x69,0x72,
		0xc9,0xe2,0x24,0x69,0xb7,0xed,0x5d,0xa9,0x6b,0x3c,0x76,0x85,0xf7,0xb0,0x56,0x99,
		0x60,0xbd,0x64,0x4f,0x13,0x0b,0x44,0xa0,0xd5,0x51,0xbb,0x0e,0x90,0x2e,0xd6,0x8a,
		0xb3,0x84,0xfd,0xc2,0xfa,0xca,0xf7,0x9b,0xbf,0x8d,0x6d,0x37,0x36,0xdf,0xa9,0x52,
		0xcd,0x70,0xf4,0x74,0x6f,0x1a,0x4a,0xea,0xc2,0xbd,0xbe,0xca,0x97,0xb7,0x8f,0xc1,
		0x77,0x78,0x78,0xc1,0x28,0x59,0x43,0x35,0x52,0xa7,0x7d,0x7f,0x94,0x2e,0x5b,0x60,
		0x47,0x69,0x91,0xa1,0xe3,0xd0,0x42,0x7c,0xd6,0x77,0x34,0x5a,0x1c,0xe2,0x06,0x3a,
		0x2e,0x0e,0xc5,0x47,0xa9,0xd8,0x21,0xda,0x75,0x9f,0x1a,0x91,0xb5,0x88,0x17,0xbd,
		0x0d,0xc4,0xef,0xfc,0x12,0x6f,0x6f,0x4e,0xb0,0xb9,0x11,0xe9,0x04,0xed,0x21,0xdc,
		0x43,0x9d,0x65,0x8a,0x77,0x3f,0x97,0xe1,0x79,0xad,0x20,0xbc,0x3c,0x63,0x60,0x9f,
		0x28,0x74,0x06,0x2e,0x83,0x5f,0x6a,0xe0,0x8d,0x59,0x65,0x4f,0x9c,0x88,0x61,0xe5,
		0x27,0x03,0x9e,0xea,0xc3,0x2b,0x9e,0xed,0x29,0x3b,0xd8,0xb1,0xe0,0xe7,0xc6,0x7b,
		0xf1,0xd7,0x55,0x24,0x6a,0x1b,0x06,0x3f,0xf0,0x06,0x32,0xa4,0x6d,0xd5,0xcf,0x69,
		0x48,0xf0,0xee,0xd9,0xb6,0x5c,0x59,0x39,0xdf,0xd2,0x14,0x80,0xa6,0x3d,0xf0,0xca,
		0xb0,0xa7,0x50,0x9c,0x42,0x45,0xa1,0xbe,0x75,0x47,0xcc,0xc6,0xf7,0x3c,0x72,0x4b,
		0x48,0xb7,0x86,0x70,0x12,0xe0,0xca,0x3e,0x47,0x2e,0x0d,0x55,0x06,0x49,0xa7,0x34,
		0x44,0x0f,0xef,0xfc,0x8e,0x73,0x68,0x2c,0xb6,0x04,0x53,0xe7,0xa4,0x7e,0x72,0xfe,
		0x08,0x74,0xeb,0x40,0xac,0xd0,0xd4,0x8e,0x4e,0x57,0x19,0x74,0x16,0x11,0x2b,0xcd,
		0xc8,0xbb,0x7a,0x58,0xbe,0xa7,0x45,0xfd,0xd6,0x4c,0x16,0xf6,0x66,0xc8,0x8d,0x9e,
		0x3b,0xd2,0x35,0xb1,0x37,0x20,0x6f,0x6c,0xdb,0xa1,0x90,0xbe,0x65,0xec,0x03,0x3c,
		0x19,0x1f,0x67,0x6b,0x42,0x8e,0xc1,0x20,0x5d,0xc5,0xe9,0x45,0x82,0x85,0x08,0xd8,
		0x6f,0xd4,0xbc,0x62,0x54,0x02,0xaa,0x68,0x5f,0x60,0x72,0x5e,0xfd,0xb4,0x25,0x96),
	.out = chunk_from_chars(
		0x28,0x10,0x28,0x8a,0x45,0xa5,0x02,0x21,0x1f,0xea,0xde,0x99,0x43,0xb4,0x22,0x4a,
		0x3f,0x54,0xcf,0x97,0xf1,0x91,0xb6,0x21,0xd8,0x0a,0x19,0xf3,0xf1,0xb0,0x36,0x6d,
		0x5e,0xd7,0x3d,0x19,0x5b,0x73,0x2a,0x15,0xb5,0x99,0x05,0x1a,0x13,0xb7,0xc5,0x84,
		0x17,0x55,0x21,0xe3,0xa9,0x7c,0xdb,0x22,0x0c,0x89,0x24,0xb0,0xec,0x23,0x4a,0x94,
		0x2e,0x05,0x79,0x67,0x5f,0x19,0x39,0x26,0xab,0x33,0x29,0x70,0x58,0x55,0x53,0xe2,
		0x30,0xe3,0x42,0x9c,0x6e,0x8b,0xa7,0x1b,0x93,0x74,0xbd,0x4a,0x8d,0xf9,0x69,0x45,
		0x68,0x40,0x52,0x7f,0x2c,0xf6,0x35,0x39,0x2b,0xe9,0xc1,0x7c,0xa8,0x47,0x4c,0xf0,
		0x1b,0x33,0x2b,0x81,0x08,0xb5,0x4c,0x48,0xaa,0xfc,0x86,0xac,0x4c,0xfc,0xe7,0x05,
		0x8e,0xe1,0x14,0x54,0x0e,0x5a,0x1b,0x95,0x22,0x01,0xee,0x3d,0x0a,0xc9,0xd3,0xe3,
		0x3f,0x01,0x80,0x75,0x84,0x33,0x9a,0xc3,0xf4,0x3c,0xcd,0xd3,0x95,0xc5,0x8b,0xb3,
		0xf4,0xa4,0xcc,0x5c,0x57,0x87,0x1e,0xa1,0xd3,0xea,0xa8,0xd0,0x30,0x2b,0xb3,0x08,
		0xd1,0xd2,0x03,0xaf,0x13,0x14,0x7e,0x87,0xab,0x2b,0x91,0x59,0xae,0x36,0x1e,0xc0,
		0x11,0xd6,0x73,0xfe,0x3f,0x16,0x2b,0x39,0x89,0xda,0x34,0x88,0xd8,0xff,0xea,0x2b,
		0x51,0xe7,0x46,0x5f,0xad,0xdc,0xa6,0x7d,0x1d,0x54,0xc8,0x94,0x0a,0x8a,0xb6,0x2e,
		0x8a,0xbe,0x41,0x45,0xcb,0x25,0x15,0x91,0x25,0x95,0x01,0x19,0xe3,0xc0,0x1a,0x81,
		0x35,0x75,0xf0,0xa9,0x59,0xb1,0x43,0xbc,0x31,0xf6,0x2b,0x47,0x34,0xaa,0x45,0xe2),
};

kdf_test_vector_t prf_plus_sha512_1 = {
	.alg = KDF_PRF_PLUS, .arg.prf = PRF_HMAC_SHA2_512,
	.key = chunk_from_chars(
		0xda,0xf9,0xbd,0x6f,0x2f,0x91,0x2d,0xa5,0x53,0x86,0x79,0x66,0xaf,0x38,0x6e,0x67,
		0x90,0x9a,0x8d,0xf0,0xca,0x7e,0x84,0xb8,0x3b,0x35,0x5c,0xb7,0xd7,0xf1,0x02,0x6f,
		0x17,0xd8,0xea,0x34,0xb5,0xd5,0x7f,0xd0,0xd1,0xba,0x38,0x95,0x28,0xfc,0xa1,0xe8,
		0x1d,0x1c,0x8c,0xe5,0x11,0xb2,0x8a,0x24,0x58,0x24,0x11,0x43,0xfe,0xe3,0x0c,0xcc),
	.salt = chunk_from_chars(
		0xdf,0x79,0x31,0xdb,0x9b,0x42,0x9e,0x10,0xb8,0xaa,0x8e,0x4d,0x46,0x04,0x23,0x93,
		0x9b,0xa4,0x0a,0xf9,0x15,0x7f,0x47,0x92,0x53,0x83,0xc7,0xcd,0x3d,0x10,0x46,0xe6),
	.out = chunk_from_chars(
		0x91,0x9a,0x75,0xde,0xae,0xcc,0xbd,0x39,0x3b,0xeb,0xc8,0x96,0x15,0xf8,0xa2,0x15,
		0xed,0xb7,0x07,0x3b,0x23,0x19,0xd8,0x67,0x40,0x22,0x1e,0xee,0x7a,0x2d,0xeb,0x92,
		0x50,0x90,0x44,0xd7,0xe6,0x1f,0xcb,0x1b,0xf9,0x67,0x6b,0x38,0x81,0xc7,0xce,0xd4,
		0x9e,0x6b,0xba,0x15,0x49,0xda,0x31,0x13,0x60,0x16,0x77,0xa2,0x44,0x5d,0x91,0xac,
		0x13,0xa5,0x97,0x33,0x57,0x82,0xdd,0x3a,0x9c,0xf4,0x96,0xd9,0xe1,0x9b,0x93,0x2f,
		0x27,0xdd,0x35,0xe3,0x6f,0x80,0x70,0xd2,0x32,0xd0,0xa3,0xb3,0xf4,0xc4,0xa0,0xb4,
		0xba,0x44,0x70,0x77,0x8d,0xc0,0x90,0xf3,0x1f,0x52,0xab,0x1b,0x6d,0x81,0xe4,0xc7,
		0xdf,0xb3,0x50,0x63,0xad,0x96,0xc3,0x27,0xad,0xcd,0xbd,0xe7,0x14,0xcc,0x9a,0x6b,
		0x5e,0xf0,0x49,0xb8),
};

kdf_test_vector_t prf_plus_sha512_2 = {
	.alg = KDF_PRF_PLUS, .arg.prf = PRF_HMAC_SHA2_512,
	.key = chunk_from_chars(
		0xe5,0x70,0xe7,0x48,0x46,0x34,0x30,0x0d,0x7a,0xdd,0xf5,0xa8,0x52,0x7c,0x13,0x8b,
		0x76,0x96,0xdb,0xc3,0xd8,0xbe,0x09,0x69,0xb7,0x52,0x15,0x31,0x8a,0x11,0xad,0xa2,
		0x13,0x53,0x8f,0x62,0x93,0xb0,0xee,0xe5,0xb3,0x09,0xad,0x8f,0x5d,0x8d,0x94,0xdb,
		0xe5,0x73,0x61,0x27,0xe0,0xd2,0x56,0x0f,0x28,0x1c,0x9c,0x58,0x6b,0xf7,0xc8,0x6f),
	.salt = chunk_from_chars(
		0xd8,0x96,0x84,0xe3,0xcb,0x17,0xf3,0xaa,0xbd,0x85,0x3a,0x78,0xdb,0x3e,0xcd,0x5a,
		0xac,0xc1,0xed,0x71,0xc7,0x0b,0x88,0xa2,0x97,0x56,0xf4,0x6f,0xc7,0x19,0x7c,0x80,
		0x4e,0xc0,0x01,0x54,0x40,0x02,0xa8,0xae,0xa3,0x60,0x68,0x4b,0x18,0x00,0x6d,0xef,
		0x0d,0xbd,0x86,0x33,0xb1,0x01,0x9f,0xbc,0xfa,0x85,0xb9,0x4c,0xac,0x2b,0xb8,0x21,
		0x25,0x84,0xbe,0x62,0xad,0xab,0x0e,0xe4,0xbb,0x8a,0x36,0xae,0xe2,0x52,0x75,0xef,
		0x07,0x13,0x90,0x48,0x0e,0xef,0xa2,0x09,0x2e,0xb3,0x08,0xaa,0x73,0x37,0xc5,0xce,
		0xb6,0x06,0x9a,0xb6,0x90,0xe8,0x96,0x2f,0xbf,0xe0,0x98,0x6e,0x4f,0x5c,0x18,0xf4,
		0x86,0x86,0x1e,0xd3,0xf1,0xdc,0xbe,0xe4,0xc9,0xe4,0xa7,0x66,0x9d,0x74,0x0c,0xa2,
		0xb0,0xe8,0xed,0x40,0x31,0xb0,0xa4,0x99,0xdc,0x31,0x5c,0xed,0xe7,0xef,0x03,0x39,
		0x9c,0xbc,0x33,0xdc,0xd6,0x29,0x70,0x34,0x9f,0x12,0x20,0x88,0x1b,0x55,0x45,0x2e,
		0x0c,0x6c,0x9b,0x52,0xa8,0x8b,0x67,0xf5,0x97,0x58,0x67,0x95,0xb2,0x25,0x70,0x73,
		0x3f,0xd4,0xff,0x2c,0xc2,0xad,0x93,0x1d,0x83,0x30,0x16,0x5a,0x9e,0x45,0x0e,0x38,
		0x88,0x59,0xce,0x62,0x4f,0x01,0xdb,0x17,0xc2,0x50,0x2b,0x4e,0x66,0xad,0xf9,0x65,
		0x27,0x36,0x3d,0x6f,0x90,0x6b,0x20,0x23,0xe8,0xed,0x74,0xd5,0xaf,0x0a,0xa6,0x02,
		0x46,0xb0,0xb9,0x2f,0x49,0xc4,0x93,0x3e,0xf3,0x12,0xf8,0xa2,0x54,0x34,0xee,0x96,
		0x98,0xd6,0xd9,0x20,0x43,0x45,0xbd,0x10,0xbb,0x11,0xaa,0x39,0x86,0x56,0x16,0xd5,
		0xad,0x1b,0x57,0x44,0x70,0x6c,0xfd,0x4e,0xa1,0x40,0x8e,0x20,0xc8,0xfd,0xcf,0x85,
		0x51,0xee,0xe8,0x81,0x4b,0x7b,0x37,0x33,0x0b,0x05,0x26,0xf0,0xbc,0x5e,0xe1,0x5d,
		0x4e,0xcd,0xa7,0xa1,0xbd,0x25,0xaa,0x97,0xf2,0x45,0x84,0xd8,0x5d,0x3f,0x52,0x49,
		0x69,0x4b,0x9f,0x43,0x53,0x9e,0x69,0xea,0x35,0xbf,0xe7,0xfd,0x44,0x07,0xbc,0x8e,
		0x9d,0xca,0x8a,0x9f,0xae,0x4b,0xdc,0x6b,0x7b,0xb3,0x8c,0x6d,0x68,0xf8,0x99,0xe1,
		0xd3,0x2c,0x85,0xbc,0xd6,0x17,0xa5,0x67,0x67,0x8c,0xf8,0x5d,0x22,0x17,0xa4,0xe8,
		0x6a,0x75,0x56,0x24,0xb6,0x40,0x02,0x35,0x4c,0x02,0x68,0x42,0xbc,0x95,0x42,0x49,
		0x1a,0xf1,0xc3,0xd6,0x29,0x09,0x70,0x55,0x9a,0xf1,0x1b,0xdc,0x2c,0x83,0xb5,0x4c,
		0x74,0x14,0x49,0x05,0xc0,0xa3,0x58,0xf3,0x15,0x3d,0xb7,0x67,0xa5,0xda,0x2a,0x86,
		0x27,0xf6,0x96,0x27,0xe1,0xd4,0x1e,0xde,0x9c,0x90,0x7c,0x79,0xb5,0x1f,0xf8,0x15,
		0xe4,0x64,0x5c,0x33,0x75,0xe0,0xf6,0x3f,0x84,0xfc,0xf5,0xd7,0xc3,0x40,0x7a,0x1d,
		0xd6,0x83,0x9e,0x19,0x06,0xa1,0xe3,0x80,0x2c,0xcf,0x5e,0x82,0x30,0xd9,0x5c,0xf4,
		0xb8,0x27,0xb4,0x1c,0x48,0x34,0x25,0xa8,0xa6,0x0b,0xfa,0x51,0x89,0xda,0xc4,0x38,
		0x06,0x0f,0x2f,0x5c,0xd5,0x26,0x66,0x2f,0x29,0x06,0xc1,0xdd,0x64,0xf4,0x84,0x4e,
		0x94,0x2c,0xa8,0x4d,0xae,0xce,0x6d,0xd7,0xbb,0xf7,0x19,0x4d,0x8c,0xe5,0x6b,0xc2,
		0x83,0x10,0x85,0xa7,0xd3,0x10,0xe4,0x94,0x4c,0xfa,0xe7,0x62,0x60,0xaa,0xbf,0x6b,
		0x60,0x9d,0x88,0x78,0x9b,0x8a,0x1a,0xbf,0x50,0x01,0x86,0xc3,0xd9,0x7e,0xd2,0xab),
	.out = chunk_from_chars(
		0x98,0x7f,0xad,0xd8,0x04,0x99,0x45,0x67,0xd5,0x7f,0x98,0x9b,0x6c,0xda,0x66,0xef,
		0xbf,0xfc,0xab,0x28,0xac,0x3e,0x3b,0xc6,0x10,0x78,0x05,0x95,0x4e,0xb1,0xea,0xd0,
		0xce,0xa6,0xfb,0x49,0xaa,0x96,0x1c,0xbe,0x98,0xfd,0xad,0x57,0xdd,0x9c,0x45,0x56,
		0x7c,0xe4,0x96,0x74,0x78,0x62,0xf8,0x8b,0xcd,0x9f,0xc5,0x75,0x59,0xa0,0x7e,0xa9,
		0x27,0xa4,0x20,0x82,0xba,0x87,0xe0,0xf5,0x1b,0x3d,0x5b,0x8c,0xbd,0xea,0xdd,0xed,
		0xa3,0xae,0x4c,0x73,0xb1,0xa5,0x1d,0xd0,0xdc,0xe3,0xcc,0x8e,0xb3,0xb1,0x41,0x9b,
		0x93,0xf2,0x9d,0xc6,0x52,0x2b,0x2f,0x5e,0x93,0x75,0x23,0x82,0x03,0xec,0xa9,0x8e,
		0xd8,0x6b,0xf1,0xe6,0x86,0x8a,0xb4,0x28,0xd8,0x63,0xa5,0xf3,0xb9,0x87,0xd6,0xc0,
		0xf9,0x57,0xaf,0x08,0xa1,0x63,0x1f,0x7c,0xdc,0x00,0x8c,0xbb,0xff,0xc2,0x4b,0x02,
		0x93,0x1a,0x02,0x01,0x94,0xe3,0x68,0xd6,0xb5,0x70,0x2e,0x7f,0x8b,0x96,0x0d,0xbc,
		0x93,0xca,0x56,0xed,0x5a,0x68,0xda,0xe0,0x20,0x69,0x94,0xed,0xf1,0x47,0x6f,0x5f,
		0xa3,0x4e,0xfb,0xda,0xa9,0x73,0xd2,0x7b,0xa3,0x68,0xce,0xdc,0xc6,0x66,0xd7,0x63,
		0xbd,0x6a,0xd5,0x60,0x57,0x38,0x51,0xbf,0xfb,0x70,0x99,0xaf,0x58,0x8f,0x34,0xff,
		0x2e,0x59,0x9b,0x0e,0x87,0xf7,0x0a,0x7f,0x4a,0xa4,0x8e,0x95,0x1d,0x0c,0x5c,0x30,
		0xdd,0xfc,0x1f,0x98,0xc1,0x28,0x63,0x15,0x90,0xcd,0xe0,0x0a,0x7c,0x93,0x15,0x8d,
		0xbf,0x4c,0xaa,0x53,0x7b,0x31,0x59,0x31,0xb7,0x14,0xd7,0x2f,0x4d,0x2a,0x01,0xc9),
};
