#
# Copyright 2002 by Object Craft P/L, Melbourne, Australia.
#
# LICENCE - see LICENCE file distributed with this software for details.
#
#
# Test CGI Request functionality
#
# $Id: request.py 8887 2007-03-16 09:35:33Z andrewm $

import unittest
import cgi
import re
import urllib
import StringIO
import albatross
import albatross.cgiapp


class Context(albatross.NameRecorderMixin, albatross.NamespaceMixin,
              albatross.ExecuteMixin):
    def __init__(self):
        albatross.NameRecorderMixin.__init__(self)
        albatross.NamespaceMixin.__init__(self)
        albatross.ExecuteMixin.__init__(self)


class App(albatross.PickleSignMixin):
    def __init__(self):
        albatross.PickleSignMixin.__init__(self, '<secret>')

    def create_context(self):
        self.ctx = Context()
        self.ctx.app = self
        return self.ctx

    def run(self, req):
        ctx = self.create_context()
        ctx.request = req
        ctx.app = self
        if ctx.request is not None:
            ctx.merge_request()


def build_query(*fields):
    return '&'.join(['%s=%s' % (a, urllib.quote(v)) for a, v in fields])


def make_albform(*inputs):
    # trick albatross into generating an __albform__ value for us
    app = App()
    ctx = app.create_context()
    ctx.push_content_trap()
    ctx.form_open()
    for itype, iname, imulti in inputs:
        ctx.input_add(itype, iname, '', imulti)
    ctx.form_close()
    result = ctx.pop_content_trap()
    match = re.search('name="__albform__" value="([^"]+)"', result)
    if match:
        return match.group(1)
    return ""


class SimpleRequestTest(unittest.TestCase):
    def make_albform(self):
        return make_albform(('text', 'aaa', False), ('text', 'bbb', True))

    def test_request(self, request):
        app = App()
        app.run(request)
        self.failUnless(app.ctx.locals.aaa, 'bah')
        self.failUnless(app.ctx.locals.bbb, ['a','b'])


class CGIGetRequestTestCase(SimpleRequestTest):
    def runTest(self):
        albform = self.make_albform()
        query_string = build_query(('aaa', 'bah'), ('bbb', 'a'), ('bbb', 'b'), 
                                   ('__albform__', albform))
        environ = {
            'REQUEST_METHOD': 'GET',
            'QUERY_STRING': query_string,
        }
        fields = cgi.FieldStorage(environ=environ)
        self.test_request(albatross.cgiapp.Request(fields=fields))


class CGIGetRequestNoAlbformTestCase(SimpleRequestTest):
    def runTest(self):
        query_string = build_query(('aaa', 'bah'), 
                                   ('bbb', 'a'), ('bbb', 'b'),
                                   ('ccc.ddd', 'c'))
        environ = {
            'REQUEST_METHOD': 'GET',
            'QUERY_STRING': query_string,
        }
        fields = cgi.FieldStorage(environ=environ)
        req = albatross.cgiapp.Request(fields=fields)
        app = App()
        app.run(req)
        app.ctx.locals.ccc = albatross.Vars()
        self.failIf(hasattr(app.ctx.locals, 'aaa'))
        self.failIf(hasattr(app.ctx.locals, 'bbb'))
        self.failIf(hasattr(app.ctx.locals.ccc, 'ddd'))
        app.ctx.merge_vars('aa', 'aaaa')
        self.failIf(hasattr(app.ctx.locals, 'aaa'))
        self.failIf(hasattr(app.ctx.locals, 'bbb'))
        self.failIf(hasattr(app.ctx.locals.ccc, 'ddd'))
        app.ctx.merge_vars('aaa', 'bbb')
        self.failUnless(app.ctx.locals.aaa, 'bah')
        self.failUnless(app.ctx.locals.bbb, ['a','b'])
        self.failIf(hasattr(app.ctx.locals.ccc, 'ddd'))
        app.ctx.merge_vars('ccc')
        self.failUnless(app.ctx.locals.ccc.ddd, 'c')

class CGIPostRequestTestCase(SimpleRequestTest):
    def runTest(self):
        albform = self.make_albform()
        query_string = build_query(('aaa', 'bah'), ('bbb', 'a'), ('bbb', 'b'), 
                                   ('__albform__', albform))
        environ = {
            'REQUEST_METHOD': 'POST',
            'CONTENT_TYPE': 'application/x-www-form-urlencoded',
        }
        input_file = StringIO.StringIO(query_string)
        fields = cgi.FieldStorage(fp=input_file, environ=environ)
        self.test_request(albatross.cgiapp.Request(fields=fields))


class CGIFormRequestTestCase(SimpleRequestTest):
    input = """\
----------AXBYCZ
Content-Disposition: form-data; name="aaa"

bah
----------AXBYCZ
Content-Disposition: form-data; name="bbb"

a
----------AXBYCZ
Content-Disposition: form-data; name="bbb"

b
----------AXBYCZ
Content-Disposition: form-data; name="__albform__"

%s
----------AXBYCZ--
"""
    def runTest(self):
        environ = {
            'REQUEST_METHOD': 'POST',
            'CONTENT_TYPE': 'multipart/form-data; boundary=--------AXBYCZ',
        }
        input_file = StringIO.StringIO(self.input % self.make_albform())
        fields = cgi.FieldStorage(fp=input_file, environ=environ)
        self.test_request(albatross.cgiapp.Request(fields=fields))


class FileRequestTest(unittest.TestCase):

    def test_request(self, input_str):
        boundary = '--------AXBYCZ'
        environ = {
            'REQUEST_METHOD': 'POST',
            'CONTENT_TYPE': 'multipart/form-data; boundary=--------AXBYCZ',
        }
        albform = make_albform(('file', 'ccc', False))
        app = App()
        input_file = StringIO.StringIO(input_str % albform)
        fields = cgi.FieldStorage(fp=input_file, environ=environ)
        req = albatross.cgiapp.Request(fields=fields)
        app.run(req)
        return app.ctx.locals.ccc



class CGISingleFileRequestTestCase(FileRequestTest):
    input_str = """\
----------AXBYCZ
Content-Disposition: form-data; name="ccc"; filename="baz"

bah
----------AXBYCZ
Content-Disposition: form-data; name="__albform__"

%s
----------AXBYCZ--
"""
    def runTest(self):
        ccc = self.test_request(self.input_str)
        self.failUnless(len(ccc), 1)
        self.failUnless(ccc[0].file.read(), 'bah')
        self.failUnless(ccc[0].filename, 'baz')


class CGIListFileRequestTestCase(FileRequestTest):
    input_str = """\
----------AXBYCZ
Content-Disposition: form-data; name="ccc"; filename="baz"

bah
----------AXBYCZ
Content-Disposition: form-data; name="ccc"; filename="blot"

bling
----------AXBYCZ
Content-Disposition: form-data; name="__albform__"

%s
----------AXBYCZ--
"""
    def runTest(self):
        ccc = self.test_request(self.input_str)
        self.failUnless(len(ccc), 2)
        self.failUnless(ccc[0].file.read(), 'bah')
        self.failUnless(ccc[0].filename, 'baz')
        self.failUnless(ccc[1].file.read(), 'bling')
        self.failUnless(ccc[1].filename, 'blot')


class CGIMultiFileRequestTestCase(FileRequestTest):
    input_str = """\
----------AXBYCZ
Content-Disposition: form-data; name="ccc"
Content-Type: multipart/mixed; boundary=--------ZCYBXA

----------ZCYBXA
Content-Disposition: attachment; filename="baz"
Content-Type: text/plain

bah
----------ZCYBXA
Content-Disposition: attachment; filename="blot"

bling
----------ZCYBXA--
----------AXBYCZ
Content-Disposition: form-data; name="__albform__"

%s
----------AXBYCZ--
"""
    def runTest(self):
        ccc = self.test_request(self.input_str)
        self.failUnless(len(ccc), 2)
        self.failUnless(ccc[0].file.read(), 'bah')
        self.failUnless(ccc[0].filename, 'baz')
        self.failUnless(ccc[1].file.read(), 'bling')
        self.failUnless(ccc[1].filename, 'blot')


"""
Driver for the above tests
"""

class RequestSuite(unittest.TestSuite):
    def __init__(self):
        unittest.TestSuite.__init__(self)
        self.addTest(CGIGetRequestTestCase())
        self.addTest(CGIGetRequestNoAlbformTestCase())
        self.addTest(CGIPostRequestTestCase())
        self.addTest(CGIFormRequestTestCase())
        self.addTest(CGISingleFileRequestTestCase())
        self.addTest(CGIListFileRequestTestCase())
        self.addTest(CGIMultiFileRequestTestCase())

def suite():
    return RequestSuite()

if __name__ == '__main__':
    unittest.main(defaultTest='suite')
