# -*- coding: utf8 -*-
# Copyright: 2013-2014, Maximiliano Curia <maxy@debian.org>
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 2 of the License, or (at your option)
# any later version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
# FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
# more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc., 51
# Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

import glob
import logging
import os
import re
import subprocess

import util

# Taken from gbp.deb.uscan and modified to suit this program
# Copyright: 2012, Guido Günther <agx@sigxcpu.org>
# License: GPL-2+
'''Interface to uscan'''


class UscanError(Exception):
    pass


class Uscan(object):
    cmd = '/usr/bin/uscan'

    def __init__(self, dir='.', destdir='..'):
        self._uptodate = False
        self._tarball = None
        self._version = None
        self._uversion = None
        self._url = None
        self._dir = os.path.abspath(dir)
        self._destdir = destdir

    @property
    def uptodate(self):
        return self._uptodate

    @property
    def tarball(self):
        return self._tarball

    @property
    def version(self):
        return self._version if self._version else self._uversion

    @property
    def uversion(self):
        return self._uversion

    @property
    def url(self):
        return self._url

    def _parse(self, out, destdir=None):
        r'''
        Parse the uscan output return and update the object's properties

        @param out: uscan output
        @type out: string

        >>> u = Uscan('http://example.com/')
        >>> u._parse('<target>virt-viewer_0.4.0.orig.tar.gz</target>')
        >>> u.tarball
        '../virt-viewer_0.4.0.orig.tar.gz'
        >>> u.uptodate
        False
        >>> u._parse('')
        Traceback (most recent call last):
        ...
        UscanError: Couldn't find 'upstream-url' in uscan output
        '''
        ext = None
        source = None
        self._uptodate = False
        d = {}
        if not destdir:
            destdir = self._destdir

        # Check if uscan downloaded something
        for row in out.split("\n"):
            # uscan >= 2.10.70 has a target element:
            for n in (
                    'package',
                    'debian-uversion',
                    'upstream-version',
                    'upstream-url',
                    'target',
                    'messages',
                    'status',
                    'warnings'):
                m = re.match("<%s>(.*)</%s>" % (n, n), row)
                if m:
                    d[n] = m.group(1)
        if not d:
            raise UscanError('No watch file found.')
        if 'warnings' in d:
            raise UscanError(d['warnings'])
        if 'target' in d:
            source = d['target']
        if not source and 'messages' in d:
            m = re.match(r'.*symlinked ([^\s]+) to it', d['messages'])
            if m:
                source = m.group(1)
            else:
                m = re.match(r'Successfully downloaded updated package '
                             '(.+)', d['messages'])
                if m:
                    source = m.group(1)
        if 'status' in d:
            self._uptodate = (d['status'] == 'up to date')
        if 'upstream-version' in d:
            self._uversion = d['upstream-version']
        if 'debian-uversion' in d:
            self._version = d['debian-uversion']
        if 'upstream-url' in d:
            self._url = d['upstream-url']
            ext = os.path.splitext(self._url)[1]

        if (not source) and ('package' in d) and self.version and ext:
            source = '%s_%s.orig.tar%s' % (d['package'], self.version, ext)
        if source:
            source = os.path.join(destdir, source)
            if os.path.exists(source):
                self._tarball = source
        if (not self._tarball) and ('package' in d) and self.version:
            source = '%s_%s.orig.tar.*' % (d['package'], self.version)
            wild = os.path.join(destdir, source)
            files = glob.glob(wild)
            if len(files):
                self._tarball = files[0]
        if (not self._tarball) and self._url:
            source = self._url.rsplit('/', 1)[1]
            source = os.path.join(destdir, source)
            if os.path.exists(source):
                self._tarball = source

    def _raise_error(self, out):
        r'''
        Parse the uscan output for errors and warnings and raise
        a L{UscanError} exception based on this. If no error detail
        is found a generic error message is used.

        @param out: uscan output
        @type out: string
        @raises UscanError: exception raised

        >>> u = Uscan('http://example.com/')
        >>> u._raise_error("<warnings>uscan warning: "
        ... "In watchfile debian/watch, reading webpage\n"
        ... "http://a.b/ failed: 500 Cant connect "
        ... "to example.com:80 (Bad hostname)</warnings>")
        Traceback (most recent call last):
        ...
        UscanError: Uscan failed: uscan warning: In watchfile debian/watch, reading webpage
        http://a.b/ failed: 500 Cant connect to example.com:80 (Bad hostname)
        >>> u._raise_error("<errors>uscan: Can't use --verbose if "
        ... "you're using --dehs!</errors>")
        Traceback (most recent call last):
        ...
        UscanError: Uscan failed: uscan: Can't use --verbose if you're using --dehs!
        >>> u = u._raise_error('')
        Traceback (most recent call last):
        ...
        UscanError: Uscan failed - debug by running 'uscan --verbose'
        '''
        msg = None

        for n in ('errors', 'warnings'):
            m = re.search("<%s>(.*)</%s>" % (n, n), out, re.DOTALL)
            if m:
                msg = "Uscan failed: %s" % m.group(1)
                break

        if not msg:
            msg = "Uscan failed - debug by running 'uscan --verbose'"
        raise UscanError(msg)

    def scan(self, destdir=None, download=True, force_download=False,
             version=None):
        '''Invoke uscan to fetch a new upstream version'''
        if not destdir:
            destdir = self._destdir
        util.ensure_path(destdir)
        cmd = [self.cmd, '--symlink', '--destdir=%s' % destdir, '--dehs']
        if not download:
            cmd.append('--report')
        if download and force_download or download and version:
            cmd.append('--force-download')
        if version:
            cmd.append('--download-version=%s' % version)

        logging.debug('Calling uscan: %s', cmd)
        p = subprocess.Popen(cmd, cwd=self._dir, stdout=subprocess.PIPE)
        out = p.communicate()[0]
        # uscan exits with 1 in case of uptodate and when an error occured.
        # Don't fail in the uptodate case:
        self._parse(out, destdir)
        if not self.uptodate and p.returncode:
            self._raise_error(out)
        if download and not self._tarball:
            raise UscanError("Couldn't find tarball")


# vi:expandtab:softtabstop=4:shiftwidth=4:smarttab
