/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#ifndef CAMITKCORE_H
#define CAMITKCORE_H

// The following content describes the index.html page of the API documentation.
// It uses CMake variable (surrounded by the '@' pattern), replaced by their content using the CONFIGURE_FILE CMake command
// This file is configured in camitk/cmake/modules/CamiTKApiDoc.cmake into the CamiTKVersion.h file.

/** @mainpage CamiTK API Documentation
 *
 * @section introduction Introduction
 * Welcome to the Computer Assisted Medical Intervention Tool kit (CamiTK) API Documentation.
 * CamiTK is an open-source framework that helps researchers and clinicians to easily and rapidly collaborate in order to prototype CAMI applications, that feature medical images, surgical navigation and biomechanical simulations. CamiTK uses famous tools such as Qt, ITK and VTK.
 * This API documentation would introduce you to the architecture of the framework
 *
 * @section architecture Core Architecture
 * CamiTK follows the Components Based Software Engineering (CBSE) principles. Each new functionality to the framework is a new module represented as a dynamic library (.dll on windows, .so on Linux systems). These principles guarantee CamiTK to be higly modular and easier to maintain.
 * CamiTK is provided with its core and basic modules (to handle 2D and 3D images using VTK for instance). You can easily add new modules to the framework or create new camitk-application.
 *
 * @subsection design-pattern Design Pattern
 * CamiTK core architecture follows the Model-View-Presenter (MVP) design-pattern. It helps dividing the logic in two parts : domain logic (data structures and algorithms) and application logic (the Model, GUI aspects to present and interact with the domain logic).
 *
 * \image html camitk_service_layer_1.png "The service layer" width=10cm
 *
 * To do so, 'viewers' (application logic modules) are used to present and interact with the domain logic : the 'components' (the data logic, the data structures) and the 'actions' (the processing logic, the algorithms to be used on these components).
 * In order to easily glue together all this modules CamiTK uses a Service Layer Pattern. This service layer can be seen as a contract from the viewers consumers of services provided by the 'components' and 'actions'. The viewers interact with the components without knowing exactly which kind of data it represents.
 *
 * @subsection components Components
 *
 * \image html camitk_service_layer_2.png "The four interfaces implemented by the Component class" width=10cm
 *
 * CamiTK components describe data structure and therefore are used to represent the data logic. The @ref camitk::Component "Component" class implements four interfaces, which gather the different possibilities of a component in order to provide the service layer. See the @ref camitk::Component "Component" class to have more details.
 *
 * @subsection actions Actions
 * The @ref camitk::Action "Action" class generalizes the notion of component processing. An action applies a given algorithm to one or several listed components.
 * Generally viewers trigger actions on the currently selected components.
 *
 * @subsection viewers Viewers
 * The @ref camitk::Viewer "Viewer" class handles all the presentation and interaction logic (the View and Presenter part of the MVP). The CamiTK viewers are using either Qt widget or VTK renderer but are not limited to that. Viewers manage the user interactions and map them into calls to the interface methods of the components.
 *
 * @section getting-started Getting started
 *
 * @subsection how-to-extend-camitk How to add a new action or component ?
 * To add a new action or component to CamiTK, you will need to create a new dynamic link library that extends the framework with your desired data structure and / or algorithms. You may want to use the CamiTK wizard to automatically generate a code skeleton for your extension. Please check the wiki documentation about <a href="https://forge.imag.fr/plugins/mediawiki/wiki/camitk/index.php/Create_your_own_custom_action_for_CamiTK">adding a new component / action</a>.
 *
 * @subsection how-to-create-new-application How to create a new application ?
 * You may inherits the @ref camitk::MainWindow "MainWindow" class. The MainWindow class is a kind of super presenter : it instanciates and organizes all the viewers in an application GUI. Check the wiki tutorial : <a href="https://forge.imag.fr/plugins/mediawiki/wiki/camitk/index.php/Create_your_own_application_for_CamiTK">create a new applicaiton</a>.
 *
 * @subsection tutorials Tutorials
 * Many tutorials are proposed in the <a href="https://forge.imag.fr/plugins/mediawiki/wiki/camitk/index.php/Main_Page">CamiTK wiki</a> (you may want to edit it and share your knowledge).
 *
 * @section HomePage Home Page
 * The home page of the project can be found at : <a href="http://camitk.imag.fr">camitk.imag.fr</a>
 *
 * @section copyright (c) UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525
 */

// Here are the related pages of the documention
/**
 * @page home-page CamiTK website
 * The home page of the project can be found at : <a href="http://camitk.imag.fr">camitk.imag.fr</a>
 *
 * @page wiki CamiTK wiki
 * Installation instructions, many tips and tutorials are proposed in the <a href="https://forge.imag.fr/plugins/mediawiki/wiki/camitk/index.php/Main_Page">CamiTK wiki</a> (you may want to edit it and share your knowledge).
 *
 * \namespace CMake
 *
 *  \brief A pseudo-namespace used to group CMake macros and functions.
 *
 *  This is not a regular C++ namespace. It is just used to group the CMake
 *  macros and function of the CamiTK build system.
 *
 * @page CamiTKBuildSystem The CamiTK Build System (based on CMake Macros)
 *
 * \defgroup CamiTKMacros CamiTK CMake Macros
 *    This module contains CMake macros and functions which can be used by CamiTK developers.
 *
 **/

// -- Core stuff
#include "CamiTKAPI.h"
#include "HistoryItem.h"

// -- Qt stuff
#include <QStack>

namespace camitk {
/**
 * Core class specifies the basic static information for the CamiTK API.
 *
 * Extensions can only be in three different places:
 * - CAMITK_DIR (where the libcamitkcore is)
 * - user config directory
 * - current wording directory
 *
 */
class CAMITK_API Core {
public:

    /// @name runtime directory information
    ///@{
    /** get all the component directories. It gets all the valid directories
     *  where components are installed, and insert them in this particular order:
     *  1. current working directory (build install tree)
     *  2. user config directory (user install tree)
     *  3. CamiTK SDK installation directory (global install tree)
     */
    static const QStringList getComponentDirectories();

    /** get all the action directories. It gets all the valid directories
     *  where components are installed, and insert them in this particular order:
     *  1. current working directory (build install tree)
     *  2. user config directory (user install tree)
     *  3. CamiTK SDK installation directory (global install tree)
     */
    static const QStringList getActionDirectories();

    /** Get a valid camitk test data directory name. It returns the first valid
     *  test data directory that is found. Checking is done in this particular order:
     *  1. current working directory (build install tree)
     *  2. user config directory (user install tree)
     *  3. CamiTK SDK installation directory (global install tree)
     *
     *  If none of this three directories is valid, return user's home directory.
     *
     *  A test data directory is valid if it exists and contains a least one file.
     *
     *  @return a null QString, checkable with isNull(), if no test data directory found
     */
    static const QString getTestDataDir();

    /** get all installation directories, suffixed by the given word.
     *  All the returned Strings are unique valid directories, sorted in this particular order:
     *  1. current working directory (build install tree)
     *  2. user config directory (user install tree)
     *  3. CamiTK SDK installation directory (global install tree)
     */
    static const QStringList getInstallDirectories(QString suffix);
    
    /// get more information about installation, etc...
    static const QString getConfig();

    /// get all important paths
    static const QString getPaths();

    /// get the CAMITK_DIR (where camitk-config is installed)
    static const QString getGlobalInstallDir();

    /// get the user config directory (this is the local installation directory)
    static const QString getUserInstallDir();

    /// get the current working directory (during dev, this should be the build directory, which is an installation directory)
    static const QString getCurrentWorkingDir();

    ///@}

    /// @name runtime build information
    ///@{
    /// get the type of build we are running, Debug or Release
    /// @return true if running in Debug, elsewhere false for Release build.
    static const bool isDebugBuild();
    ///@}


    /// @name miscallaneous utility method or variables
    ///@{
    /// complete version string
    static const char *version;
    /// short version string (all in lower case, without the patch number)
    static const char *shortVersion;
    /// version used for so name
    static const char *soVersion;
    /// debug postfix used on MSVC to distinguished between release and debug version
    static const char *debugPostfix;
    ///@}

    ///@cond
    // TODO CAMITK_OBSOLETE. This section list all the methods marked as obsolete. They are to be removed in CamiTK 4.0
    /** Obsolete.
      * get the camitk component extension directory name
      * @return a null QString, checkable with isNull(), if no component dir found
      */
    static const QString getComponentDir();

    /** Obsolete.
      * get the camitk action directory name
      * @return a null QString, checkable with isNull(), if no action dir found
      */
    static const QString getActionDir();

    /** Obsolete.
      * get the camitk module extension directory name
      * @param subpath name of the module type (and subdirectory) to get (this simplifies getting the subdir name on plateform with hasBuildTypeDirectory()
      * @return a null QString, checkable with isNull(), if no module dir found
      */
    static const QString getModuleDir(const QString & subpath = QString());

    /** Obsolete.
      * get the camitk core lib directory name (where the camitkcore dll/shared/dynamic lib is)
      */
    static const QString getCoreLibDir();
    ///@endcond

};

}

#endif // CAMITKCORE_H

