/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

// -- Core stuff
#include "Application.h"
#include "PropertyExplorer.h"

// -- QT stuff
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QPushButton>
#include <QTabWidget>
#include <QSettings>

namespace camitk
{ 
  
// ---------------- constructor ----------------
PropertyExplorer::PropertyExplorer() : Viewer("Property Explorer") {
  theController = NULL;
  tabWidget = NULL;
  revertButton = NULL;
  applyButton = NULL;
  currentComponent = NULL;
}

// ---------------- destructor ----------------
PropertyExplorer::~PropertyExplorer() {
  clear();
}

//----------------------- getInstance ------------------------
PropertyExplorer* PropertyExplorer::getInstance() {
    // static instanciation, static method variable
    static PropertyExplorer *propExplorer = NULL;
    if (!propExplorer)
        propExplorer = new PropertyExplorer();

    return propExplorer;
}

// ---------------- numberOfViewedComponent ----------------
unsigned int PropertyExplorer::numberOfViewedComponent() {
  if (currentComponent)
    return 1;
  else
    return 0;
}

// ---------------- refresh ----------------
void PropertyExplorer::refresh(Viewer *whoIsAsking) {
  // refresh the explorer if there is one selected Component
  if (!Application::getSelectedComponents().isEmpty()) {
    // last selected is different from currently selected
    if (currentComponent != Application::getSelectedComponents().last()) {
      //-- clear
      clear();
      //-- update currentComponent
      currentComponent = Application::getSelectedComponents().last();
      currentComponent->setVisibility(this, true);
      //-- get the property widget
      QWidget * customWidget = currentComponent->getPropertyWidget();
      if (customWidget) {
        tabWidget->addTab(customWidget, QString(currentComponent->getName() + " Properties"));
      }
    }
    //--  update the property editor
    // first clear all (otherwise if a new dynamic properties was added, it won't show)
    theController->setObject(NULL);
    // rebuild all GUI
    theController->setObject(currentComponent->getPropertyObject());
  }
  else {
    // clean everything only if there is no selection and cleaning was not already done
    if (currentComponent != NULL) {
      // clear tabs
      clear();
      theController->setObject(NULL);
    }
  }

  //-- update button states
  revertButton->setEnabled(currentComponent != NULL && currentComponent->getPropertyObject() != NULL);
  applyButton->setEnabled(currentComponent != NULL && currentComponent->getPropertyObject() != NULL);
  
  
}

// ---------------- getWidget ----------------
QWidget * PropertyExplorer::getWidget(QWidget * parent) {
  if (tabWidget == NULL) {
    tabWidget = new QTabWidget(parent);
    tabWidget->setWindowTitle(tr("Property Tabs"));

    // create the property editor tab itself
    QFrame *propertyFrame = new QFrame;
    QVBoxLayout * propertyTabLayout = new QVBoxLayout;
    propertyTabLayout->setSpacing(2);
    propertyTabLayout->setMargin(2);

    // get user pref for view mode
    QSettings & settings = Application::getSettings();
    settings.beginGroup("PropertyExplorer");

    // the background color
    ObjectController::ViewMode mode = (ObjectController::ViewMode) settings.value("viewMode", ObjectController::BUTTON).toInt();
    settings.endGroup();      
    
    // first the property editor
    theController = new ObjectController(propertyFrame,mode);
    propertyTabLayout->addWidget(theController);

    // then the buttons
    QFrame *buttonFrame = new QFrame(parent);
    QHBoxLayout *buttonLayout = new QHBoxLayout;

    applyButton = new QPushButton("Apply");
    buttonLayout->addWidget(applyButton);
    applyButton->show();
    applyButton->setEnabled(false);

    revertButton = new QPushButton("Revert");
    buttonLayout->addWidget(revertButton);
    revertButton->show();
    revertButton->setEnabled(false);

    buttonFrame->setLayout(buttonLayout);
    propertyTabLayout->addWidget(buttonFrame);

    // connect the buttons
    QObject::connect(applyButton, SIGNAL(clicked()), theController, SLOT(apply()));
    QObject::connect(revertButton, SIGNAL(clicked()), theController, SLOT(revert()));

    propertyFrame->setLayout(propertyTabLayout);
    
    tabWidget->addTab(propertyFrame, tr("P&roperties"));
       
  }

  return tabWidget;
}

// ---------------- getPropertyObject ----------------
QObject * PropertyExplorer::getPropertyObject() {
  return this;
}


// ---------------- getViewMode ----------------
const ObjectController::ViewMode PropertyExplorer::getViewMode() {
  return theController->getViewMode();
}
    
// ---------------- setViewMode ----------------
void PropertyExplorer::setViewMode(ObjectController::ViewMode mode) {
  theController->setViewMode(mode);
  QSettings & settings = Application::getSettings();
  settings.beginGroup("PropertyExplorer");
  settings.setValue("viewMode", mode);
  settings.endGroup();
}
    
// ---------------- clear ----------------
void PropertyExplorer::clear() {
  // page #0 is the property widget (not be counted)
  for (int i = tabWidget->count() - 1; i > 0; i--) {
    tabWidget->removeTab(i);
  }

  if (currentComponent && Application::isAlive(currentComponent))
    currentComponent->setVisibility(this, false);
  currentComponent = NULL;
}


}
