/*
	dsp/polynomials.cc
	
	Copyright 2011-12 Tim Goetze <tim@quitte.de>
	
	http://quitte.de/dsp/

	Roughly approximated atan() by IEEE float exponent-indexed table 
	lookup of Chebyshev polynomial coefficients.
	
	Plus other clipping	and waveshaping functions.

*/
/*
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 3
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
	02111-1307, USA or point your web browser to http://www.gnu.org.
*/

#include "../basics.h"

namespace DSP {

namespace Polynomial {

/*
	if (fabs (x) > 1)
	{
		float f = .7238095403;
		int32 i = *((int*) &x);
		i &= 0x80000000;
		i |= *((int*) &f);
		return *((float*) &i);
	}
*/

/* 
 shortened MacLaurin series for atan()
*/
float _power_clip_7 (sample_t x)
{
	if (x < -1) return -.7238095403;
	if (x > 1) return .7238095403;

	float a = x, x2 = x*x;

	x *= x2;
	a -= x * .3333333333333333333333333333333333333333333333333333333334;
	x *= x2;
	a += x * .2;
	x *= x2;
	a -= x * .142857142857143;
	return a;
}

float power_clip_7 (float x)
{
	union { float f; uint32 i; } u;
	u.f = x;
	float a = x, x2 = x*x;

	x *= x2;
	u.i &= 0xFF800000;
	a -= x * .3333333333333333333333333333333333333333333333333333333334;
	u.i += 0x800000;
	x *= x2;
	u.i >>= 30;
	a += x * .2;
	x *= x2;
	a -= x * .142857142857143;
	
	float aa[4] = {a, .7238095403, a, -.7238095403};
	return aa[u.i];
}


float power_clip_11 (sample_t x)
{
	if (x < -1) return -.74401154401154412;
	if (x > 1) return .74401154401154412;

	float a = x, x2 = x*x;

	x *= x2;
	a -= x * .3333333333333333333333333333333333333333333333333333333334;
	x *= x2;
	a += x * .2;
	x *= x2;
	a -= x * .142857142857143;
	x *= x2;
	a += x * .1111111111111111111;
	x *= x2;
	a -= x * .090909090909090912;
	return a;
}


typedef float ptbl_t[256][4];

ptbl_t atantbl = {
	/*   0 */ {0.000000000000e+00,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   1 */ {-2.564183643745e-50,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   2 */ {-5.128367287489e-50,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   3 */ {-1.025673457498e-49,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   4 */ {-2.051346914996e-49,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   5 */ {-4.102693829992e-49,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   6 */ {-8.205387659983e-49,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   7 */ {-1.641077531997e-48,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   8 */ {-3.282155063993e-48,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   9 */ {-6.564310127986e-48,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  10 */ {-1.312862025597e-47,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  11 */ {-2.625724051195e-47,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  12 */ {-5.251448102389e-47,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  13 */ {-1.050289620478e-46,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  14 */ {-2.100579240956e-46,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  15 */ {-4.201158481911e-46,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  16 */ {-8.402316963823e-46,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  17 */ {-1.680463392765e-45,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  18 */ {-3.360926785529e-45,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  19 */ {-6.721853571058e-45,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  20 */ {-1.344370714212e-44,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  21 */ {-2.688741428423e-44,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  22 */ {-5.377482856846e-44,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  23 */ {-1.075496571369e-43,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  24 */ {-2.150993142739e-43,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  25 */ {-4.301986285477e-43,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  26 */ {-8.603972570954e-43,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  27 */ {-1.720794514191e-42,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  28 */ {-3.441589028382e-42,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  29 */ {-6.883178056764e-42,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  30 */ {-1.376635611353e-41,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  31 */ {-2.753271222705e-41,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  32 */ {-5.506542445411e-41,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  33 */ {-1.101308489082e-40,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  34 */ {-2.202616978164e-40,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  35 */ {-4.405233956329e-40,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  36 */ {-8.810467912657e-40,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  37 */ {-1.762093582531e-39,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  38 */ {-3.524187165063e-39,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  39 */ {-7.048374330126e-39,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  40 */ {-1.409674866025e-38,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  41 */ {-2.819349732050e-38,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  42 */ {-5.638699464101e-38,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  43 */ {-1.127739892820e-37,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  44 */ {-2.255479785640e-37,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  45 */ {-4.510959571281e-37,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  46 */ {-9.021919142561e-37,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  47 */ {-1.804383828512e-36,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  48 */ {-3.608767657024e-36,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  49 */ {-7.217535314049e-36,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  50 */ {-1.443507062810e-35,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  51 */ {-2.887014125620e-35,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  52 */ {-5.774028251239e-35,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  53 */ {-1.154805650248e-34,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  54 */ {-2.309611300496e-34,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  55 */ {-4.619222600991e-34,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  56 */ {-9.238445201983e-34,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  57 */ {-1.847689040397e-33,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  58 */ {-3.695378080793e-33,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  59 */ {-7.390756161586e-33,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  60 */ {-1.478151232317e-32,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  61 */ {-2.956302464634e-32,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  62 */ {-5.912604929269e-32,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  63 */ {-1.182520985854e-31,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  64 */ {-2.365041971708e-31,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  65 */ {-4.730083943415e-31,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  66 */ {-9.460167886830e-31,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  67 */ {-1.892033577366e-30,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  68 */ {-3.784067154732e-30,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  69 */ {-7.568134309464e-30,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  70 */ {-1.513626861893e-29,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  71 */ {-3.027253723786e-29,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  72 */ {-6.054507447571e-29,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  73 */ {-1.210901489514e-28,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  74 */ {-2.421309940963e-28,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  75 */ {-4.841140767728e-28,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  76 */ {-9.683267611588e-28,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  77 */ {-1.935864661412e-27,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  78 */ {-3.868968309656e-27,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  79 */ {-7.722948262114e-27,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  80 */ {-1.539225398267e-26,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  81 */ {-3.056993779912e-26,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  82 */ {-6.029421670785e-26,9.999999999999e-01,1.000000000000e+00,1.000000000000e+00},
	/*  83 */ {-1.171553107562e-25,9.999999999998e-01,1.000000000000e+00,1.000000000000e+00},
	/*  84 */ {-2.207295921681e-25,9.999999999997e-01,1.000000000000e+00,1.000000000000e+00},
	/*  85 */ {-3.868826314694e-25,9.999999999993e-01,1.000000000000e+00,1.000000000000e+00},
	/*  86 */ {-5.555600256673e-25,9.999999999986e-01,1.000000000000e+00,1.000000000000e+00},
	/*  87 */ {-2.382991022492e-25,9.999999999973e-01,1.000000000000e+00,1.000000000000e+00},
	/*  88 */ {3.015897282194e-24,9.999999999945e-01,1.000000000000e+00,1.000000000000e+00},
	/*  89 */ {1.999692974976e-23,9.999999999891e-01,1.000000000000e+00,1.000000000000e+00},
	/*  90 */ {9.586005479595e-23,9.999999999782e-01,1.000000000000e+00,1.000000000000e+00},
	/*  91 */ {4.151800440163e-22,9.999999999563e-01,1.000000000000e+00,1.000000000000e+00},
	/*  92 */ {1.724215981601e-21,9.999999999127e-01,1.000000000000e+00,1.000000000000e+00},
	/*  93 */ {7.023829688084e-21,9.999999998254e-01,1.000000000000e+00,1.000000000000e+00},
	/*  94 */ {2.834936659796e-20,9.999999996508e-01,1.000000000000e+00,1.000000000000e+00},
	/*  95 */ {1.139052777398e-19,9.999999993015e-01,1.000000000000e+00,1.000000000000e+00},
	/*  96 */ {4.566368370526e-19,9.999999986030e-01,1.000000000000e+00,1.000000000000e+00},
	/*  97 */ {1.828578903795e-18,9.999999972060e-01,1.000000000000e+00,1.000000000000e+00},
	/*  98 */ {7.432177271939e-18,9.999999943222e-01,1.009896141271e+00,1.000000000000e+00},
	/*  99 */ {2.928164154132e-17,9.999999888241e-01,1.000000016680e+00,1.000000000000e+00},
	/* 100 */ {1.171428252274e-16,9.999999776483e-01,1.000000056671e+00,1.000000000000e+00},
	/* 101 */ {4.356731235734e-16,9.999999586075e-01,9.075074573233e-01,1.000000000000e+00},
	/* 102 */ {1.874381867965e-15,9.999999105980e-01,9.999443907987e-01,1.000000000000e+00},
	/* 103 */ {4.854186256361e-15,9.999998852577e-01,6.171872522779e-01,1.000000000000e+00},
	/* 104 */ {1.108070882572e-14,9.999998580677e-01,4.246496686073e-01,1.000000000000e+00},
	/* 105 */ {-1.448327391394e-15,1.000000015155e+00,-3.457472742332e-02,1.000000000000e+00},
	/* 106 */ {3.769892614819e-14,9.999998562130e-01,1.123076320369e-01,1.000000000000e+00},
	/* 107 */ {-3.935377859261e-15,1.000000014097e+00,-6.976040960729e-03,1.000000000000e+00},
	/* 108 */ {-6.770201874049e-14,1.000000051768e+00,-9.579325111442e-03,1.000000000000e+00},
	/* 109 */ {3.615471290625e-13,9.999998748294e-01,1.058616300009e-02,1.000000000000e+00},
	/* 110 */ {3.219841340437e-13,9.999999411191e-01,2.483019068927e-03,1.000000000000e+00},
	/* 111 */ {-8.542590732813e-13,1.000000080926e+00,-1.819186652638e-03,1.000000000000e+00},
	/* 112 */ {4.158220957910e-12,9.999997813176e-01,2.444633516584e-03,1.000000000000e+00},
	/* 113 */ {1.630817492232e-12,9.999999682121e-01,1.072601513879e-04,1.017138095910e-01},
	/* 114 */ {2.740163581455e-13,1.000000008747e+00,-1.276950707060e-04,3.406998403566e-02},
	/* 115 */ {4.189786635209e-11,9.999996331179e-01,1.030510646005e-03,-1.265787975345e+00},
	/* 116 */ {-1.863615830749e-10,1.000000829896e+00,-1.184637978262e-03,2.100674507060e-01},
	/* 117 */ {-7.459089621359e-12,9.999999637525e-01,6.209819152986e-05,-3.542968433448e-01},
	/* 118 */ {3.025500117000e-10,9.999996671929e-01,1.175524480870e-04,-3.467038411220e-01},
	/* 119 */ {2.437427997639e-10,9.999998794276e-01,1.880569041886e-05,-3.342793825541e-01},
	/* 120 */ {-1.529244666401e-10,1.000000048042e+00,-5.593091292174e-06,-3.330839180179e-01},
	/* 121 */ {-2.294960085386e-11,1.000000186337e+00,-2.198122773483e-05,-3.326249526064e-01},
	/* 122 */ {-1.575790942523e-07,1.000013540749e+00,-4.040679692533e-04,-3.289101567172e-01},
	/* 123 */ {-4.921565755961e-06,1.000211760733e+00,-3.166329188757e-03,-3.159450037735e-01},
	/* 124 */ {-1.431384786725e-04,1.003097376591e+00,-2.335631752073e-02,-2.683374526128e-01},
	/* 125 */ {-3.195712473472e-03,1.035409093494e+00,-1.382478276512e-01,-1.303941468800e-01},
	/* 126 */ {-3.055383324902e-02,1.188628337043e+00,-4.282254697776e-01,5.554912938085e-02},
	/* 127 */ {1.715789781963e-03,1.127203167447e+00,-3.997982359428e-01,5.627744211112e-02},
	/* 128 */ {3.997001586869e-01,5.512163902419e-01,-1.175703571892e-01,9.412150922504e-03},
	/* 129 */ {8.952508976254e-01,1.717077804521e-01,-1.918192245518e-02,7.913500549566e-04},
	/* 130 */ {1.219303789715e+00,4.556670098358e-02,-2.574902476695e-03,5.351111947757e-05},
	/* 131 */ {1.393240685302e+00,1.156669436058e-02,-3.277576315476e-04,3.412030127825e-06},
	/* 132 */ {1.481789220543e+00,2.902778153619e-03,-4.115688490054e-05,2.143256239033e-07},
	/* 133 */ {1.526264014320e+00,7.263912027912e-04,-5.150483547873e-06,1.341219897570e-08},
	/* 134 */ {1.548526572545e+00,1.816413834510e-04,-6.439941572256e-07,8.385259534536e-10},
	/* 135 */ {1.559660999821e+00,4.541307043014e-05,-8.050501215189e-08,5.241199063005e-11},
	/* 136 */ {1.565228607074e+00,1.135343790294e-05,-1.006330598317e-08,3.275813770691e-12},
	/* 137 */ {1.568012459905e+00,2.838370119314e-06,-1.257918856213e-09,2.047393662459e-13},
	/* 138 */ {1.569404392472e+00,7.095931946746e-07,-1.572400321536e-10,1.279622608739e-14},
	/* 139 */ {1.570100359521e+00,1.773983416747e-07,-1.965500973081e-11,7.997643883667e-16},
	/* 140 */ {1.570448343145e+00,4.434958769296e-08,-2.456876360892e-12,4.998527740717e-17},
	/* 141 */ {1.570622334968e+00,1.108739723103e-08,-3.071095575944e-13,3.124080007755e-18},
	/* 142 */ {1.570709330882e+00,2.771849237985e-09,-3.838869306130e-14,1.952549884081e-19},
	/* 143 */ {1.570752828838e+00,6.929623235104e-10,-4.798586814137e-15,1.220343747048e-20},
	/* 144 */ {1.570774577832e+00,1.732403521315e-10,-5.998222476462e-16,7.627131343290e-22},
	/* 145 */ {1.570785452307e+00,4.331017320763e-11,-7.497809915652e-17,4.766991646684e-23},
	/* 146 */ {1.570790889562e+00,1.082748281416e-11,-9.372164490031e-18,2.979320783692e-24},
	/* 147 */ {1.570793608104e+00,2.707061644554e-12,-1.171665584084e-18,1.862422737595e-25},
	/* 148 */ {1.570794966112e+00,6.781167460662e-13,-1.468978251540e-19,1.168644789669e-26},
	/* 149 */ {1.570795893601e+00,4.836613600056e-14,7.971251730297e-22,-2.525797709443e-28},
	/* 150 */ {1.570795915333e+00,6.081780937024e-14,-3.826711761510e-21,8.656563086505e-29},
	/* 151 */ {1.570796203037e+00,4.583347194266e-15,-3.956995288082e-23,-3.904581274631e-31},
	/* 152 */ {1.570796392883e+00,-7.154536286907e-15,1.678440849470e-22,-1.183257952574e-30},
	/* 153 */ {1.570796498009e+00,-6.064275171985e-15,6.349752467441e-23,-2.105703095008e-31},
	/* 154 */ {1.570796163999e+00,2.446297602360e-15,-1.235177518394e-23,2.032590312080e-32},
	/* 155 */ {1.570796383193e+00,-5.396328773814e-16,1.495739368785e-24,-1.287753928877e-33},
	/* 156 */ {1.570796335215e+00,-6.415300312279e-18,-5.840081537359e-26,5.968146952547e-35},
	/* 157 */ {1.570796417850e+00,-1.819863470228e-16,1.159795588961e-25,-2.388918379390e-35},
	/* 158 */ {1.570796209219e+00,1.140945167633e-16,-3.548563614810e-26,3.560870999532e-36},
	/* 159 */ {1.570796468336e+00,-7.104365991227e-17,1.142969970107e-26,-5.921168929555e-37},
	/* 160 */ {1.570796126054e+00,5.046102289535e-17,-4.082900641911e-27,1.065918338272e-37},
	/* 161 */ {1.570796048756e+00,3.492178664949e-17,-1.408120313472e-27,1.828433063876e-38},
	/* 162 */ {1.570796122874e+00,1.254864181959e-17,-2.474990152270e-28,1.572674681937e-39},
	/* 163 */ {1.570796237392e+00,2.788018473862e-18,-2.800638043840e-29,9.050459465081e-41},
	/* 164 */ {1.570796350187e+00,-3.582052125572e-19,1.762027427251e-30,-2.806190101853e-42},
	/* 165 */ {1.570796232401e+00,7.447608472502e-19,-1.883547051448e-30,1.530147901913e-42},
	/* 166 */ {1.570796507552e+00,-7.104255554347e-19,8.986148678559e-31,-3.665693023479e-43},
	/* 167 */ {1.570796371908e+00,-8.822098192552e-20,5.548597957779e-32,-1.124037012863e-44},
	/* 168 */ {1.570796370795e+00,1.262044692810e-21,-1.223041284043e-32,2.268966708018e-45},
	/* 169 */ {1.570796409838e+00,-4.020529625553e-20,6.267259187727e-33,-3.162705153309e-46},
	/* 170 */ {1.570796501880e+00,-4.246372005273e-20,3.317987174734e-33,-8.374516592522e-47},
	/* 171 */ {1.570796469011e+00,-1.751524672325e-20,6.889999604760e-34,-8.705172067046e-48},
	/* 172 */ {1.570796164831e+00,9.787377595822e-21,-1.901686373382e-34,1.192592882179e-48},
	/* 173 */ {1.570796082308e+00,7.445615877089e-21,-7.259155862175e-35,2.278911937158e-49},
	/* 174 */ {1.570796373808e+00,-7.239074525965e-22,3.584232372700e-36,-5.712736790949e-51},
	/* 175 */ {1.570796313162e+00,4.476785637068e-23,5.882272264729e-38,-2.368070369737e-52},
	/* 176 */ {1.570796137959e+00,7.200322641025e-22,-8.805442896447e-37,3.463527118249e-52},
	/* 177 */ {1.570796235945e+00,1.649272133073e-22,-9.587373450955e-38,1.800819388555e-53},
	/* 178 */ {1.570796277389e+00,4.957420101132e-23,-1.590847461846e-38,1.630652253475e-54},
	/* 179 */ {1.570796304348e+00,1.089198346630e-23,-1.676209934470e-39,8.245203121528e-56},
	/* 180 */ {1.570796212669e+00,2.742079796389e-23,-2.116961007880e-39,5.268685728305e-56},
	/* 181 */ {1.570796683300e+00,-4.190754505617e-23,1.581586874814e-39,-1.927124493081e-56},
	/* 182 */ {1.570796846580e+00,-3.095375339645e-23,5.913766592876e-40,-3.638958292482e-57},
	/* 183 */ {1.570796278220e+00,1.331451560692e-24,-1.196723007177e-41,3.515932667231e-59},
	/* 184 */ {1.570796027718e+00,4.420496691456e-24,-2.098295213629e-41,3.211397136796e-59},
	/* 185 */ {1.570796218111e+00,8.240357415279e-25,-2.013429590447e-42,1.583618680501e-60},
	/* 186 */ {1.570796485840e+00,-5.918186850814e-25,7.044633775167e-43,-2.693667754146e-61},
	/* 187 */ {1.570796510500e+00,-3.442843336209e-25,2.068335418975e-43,-3.993691103376e-62},
	/* 188 */ {1.570796237964e+00,8.203019144785e-26,-2.449201392194e-44,2.362142850477e-63},
	/* 189 */ {1.570796245048e+00,3.919400020373e-26,-6.043309302647e-45,2.991378348156e-64},
	/* 190 */ {1.570796402359e+00,-1.109896314378e-26,3.655651695041e-46,1.176858130147e-66},
	/* 191 */ {1.570796485961e+00,-1.831791210647e-26,6.774047711355e-46,-8.086832773861e-66},
	/* 192 */ {1.570796246560e+00,4.953301777631e-27,-9.869909699628e-47,6.317844896890e-67},
	/* 193 */ {1.570796226275e+00,2.959969543297e-27,-2.821097940210e-47,8.689011430367e-68},
	/* 194 */ {1.570796602327e+00,-4.004005041333e-27,1.867470433230e-47,-2.806470349672e-68},
	/* 195 */ {1.570796476409e+00,-1.100083289818e-27,2.576403275931e-48,-1.933091861347e-69},
	/* 196 */ {1.570796232163e+00,3.460886769845e-28,-4.065289106866e-49,1.536783463077e-70},
	/* 197 */ {1.570796495033e+00,-3.108481323934e-28,1.840987970856e-49,-3.504877053761e-71},
	/* 198 */ {1.570796438444e+00,-1.018205346630e-28,2.972065242130e-50,-2.796115858283e-72},
	/* 199 */ {1.570796757810e+00,-1.955492995290e-28,2.846119591114e-50,-1.333885125894e-72},
	/* 200 */ {1.570796336413e+00,-1.779581664997e-30,9.444825192810e-53,-1.303309472534e-75},
	/* 201 */ {1.570796454518e+00,-1.320595417375e-29,4.369413879579e-52,-4.642666828905e-75},
	/* 202 */ {1.570796565230e+00,-1.340606887980e-29,2.420043379268e-52,-1.409658086833e-75},
	/* 203 */ {1.570796613453e+00,-8.072991869680e-30,7.305706355482e-53,-2.132556190538e-76},
	/* 204 */ {1.570796145334e+00,2.574511797285e-30,-1.171263213427e-53,1.716066361488e-77},
	/* 205 */ {1.570796405850e+00,-5.525559393614e-31,1.246891537420e-54,-9.092463007706e-79},
	/* 206 */ {1.570796245051e+00,2.910371600806e-31,-3.308870351153e-55,1.210760379412e-79},
	/* 207 */ {1.570796412235e+00,-1.578440656883e-31,9.291683698648e-56,-1.755113143465e-80},
	/* 208 */ {1.570796164590e+00,1.414014454185e-31,-3.961545830039e-56,3.583322648563e-81},
	/* 209 */ {1.570796395462e+00,-3.127688247913e-32,4.555718523648e-57,-2.116979654090e-82},
	/* 210 */ {1.570796360985e+00,-7.650370319657e-33,5.468626585116e-58,-1.246643346801e-83},
	/* 211 */ {1.570796835058e+00,-5.646355234553e-32,2.013975179361e-57,-2.314527198839e-83},
	/* 212 */ {1.570796247830e+00,4.453164430421e-33,-8.095915030730e-59,4.726954623046e-85},
	/* 213 */ {1.570797386688e+00,-2.949191444514e-32,2.634330158415e-58,-7.578299379734e-85},
	/* 214 */ {1.570796386432e+00,-8.021563631645e-34,3.437426666393e-60,-4.731340305477e-87},
	/* 215 */ {1.570796499287e+00,-1.197652515029e-33,2.670208537753e-60,-1.919489276191e-87},
	/* 216 */ {1.570796480833e+00,-5.290178706658e-34,5.821451023725e-61,-2.066062201844e-88},
	/* 217 */ {1.570796262963e+00,1.113199113524e-34,-6.208374531306e-62,1.106943883952e-89},
	/* 218 */ {1.570796538612e+00,-1.833477041901e-34,5.096349979621e-62,-4.564139938221e-90},
	/* 219 */ {1.570796251887e+00,3.220554690858e-35,-4.444879847449e-63,1.973504494478e-91},
	/* 220 */ {1.570796301530e+00,3.614557294270e-36,-1.204759847342e-64,-5.011057483876e-95},
	/* 221 */ {1.570796164081e+00,1.749404238250e-35,-6.031624985112e-64,6.703016520132e-93},
	/* 222 */ {1.570796205703e+00,6.555313637757e-36,-1.133849809408e-64,6.310294364704e-94},
	/* 223 */ {1.570796149449e+00,4.752808542167e-36,-4.101244302082e-65,1.142291945661e-94},
	/* 224 */ {1.570796406532e+00,-1.042308911202e-36,4.392514102392e-66,-5.995529702771e-96},
	/* 225 */ {1.570796746827e+00,-2.850711856649e-36,6.205799620318e-66,-4.350123119888e-96},
	/* 226 */ {1.570796399376e+00,-2.504380843121e-37,2.782345924930e-67,-9.917520219480e-98},
	/* 227 */ {1.570796474758e+00,-2.546151876142e-37,1.403871627841e-67,-2.489021686299e-98},
	/* 228 */ {1.570796310852e+00,6.991582874463e-39,-2.590335064169e-71,-1.527500330209e-100},
	/* 229 */ {1.570796374000e+00,-1.607576361815e-38,1.824500165061e-69,-7.046071877544e-101},
	/* 230 */ {1.570796293352e+00,7.128755982982e-39,-4.742686210352e-70,1.000386232767e-101},
	/* 231 */ {1.570796412598e+00,-9.276532843364e-39,3.212574968942e-70,-3.586263057370e-102},
	/* 232 */ {1.570796142251e+00,9.761416605545e-39,-1.651937929371e-70,8.996220613334e-103},
	/* 233 */ {1.570796489849e+00,-4.284291826758e-39,3.604782156099e-71,-9.763175398987e-104},
	/* 234 */ {1.570796138897e+00,2.459551298977e-39,-1.033460672938e-71,1.401844439853e-104},
	/* 235 */ {1.570796516751e+00,-1.261638138759e-39,2.690576892021e-72,-1.847521018040e-105},
	/* 236 */ {1.570796104076e+00,7.375348307454e-40,-7.845994091571e-73,2.687807106972e-106},
	/* 237 */ {1.570796293715e+00,5.160884701437e-41,-2.601935706499e-74,4.275014434992e-108},
	/* 238 */ {1.570796109811e+00,1.780205509236e-40,-4.683501456967e-74,3.972332121037e-108},
	/* 239 */ {1.570796352918e+00,-5.481192834015e-42,6.797801847716e-77,2.375874299153e-110},
	/* 240 */ {1.570796452242e+00,-2.598533407433e-41,1.726327442398e-75,-3.692122634268e-110},
	/* 241 */ {1.570796280968e+00,4.801328384971e-42,-1.611616193854e-76,1.743829513909e-111},
	/* 242 */ {1.570796177211e+00,7.614112413159e-42,-1.242898012889e-76,6.539343735174e-112},
	/* 243 */ {1.570796396302e+00,-1.687578598127e-42,1.324159705569e-77,-3.369435570632e-113},
	/* 244 */ {1.570796217031e+00,1.435138458228e-42,-5.997511345888e-78,8.046760462342e-114},
	/* 245 */ {1.570796511339e+00,-1.190815916312e-42,2.463649477261e-78,-1.642343386109e-114},
	/* 246 */ {1.570796225724e+00,3.193028685611e-43,-3.214801743340e-79,1.037835813129e-115},
	/* 247 */ {1.570796103137e+00,3.630298458684e-43,-1.890995007054e-79,3.172324351042e-116},
	/* 248 */ {1.570796418707e+00,-7.544640270868e-44,1.969773391281e-80,-1.645873162525e-117},
	/* 249 */ {1.570796342809e+00,0,0,0}, /* truncated beyond 5.32e+36 */
	/* 250 */ {1.570796342809e+00,0,0,0},
	/* 251 */ {1.570796342809e+00,0,0,0},
	/* 252 */ {1.570796342809e+00,0,0,0},
	/* 253 */ {1.570796342809e+00,0,0,0},
	/* 254 */ {1.570796342809e+00,0,0,0},
	/* 255 */ {1.570796342809e+00,0,0,0},
};

float atan (float x)
{
	union { float f; uint32 i; } u;
	u.f = x;
	int e = (u.i & 0x7F800000) >> 23;
	float *p = atantbl[e];
	x = fabs (x);
	x = p[0]+x*(p[1]+x*(p[2]+x*p[3]));
	u.i &= 0x80000000;
	u.i |= 127 << 23; /* 1.0 - use 126 for .5 instead, etc */
	return x * u.f;
}

/* atan(x)/2 */
float atan1 (float x)
{
	union { float f; uint32 i; } u;
	u.f = x;
	int e = (u.i & 0x7F800000) >> 23;
	float *p = atantbl[e];
	x = fabs (x);
	x = p[0]+x*(p[1]+x*(p[2]+x*p[3]));
	u.i &= 0x80000000;
	u.i |= 126 << 23; /* 1.0 - use 126 for .5 instead, etc */
	return x * u.f;
}

/* not used */
#if 0
ptbl_t normatantbl = {
	/*   0 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   1 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   2 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   3 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   4 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   5 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   6 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   7 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   8 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   9 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  10 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  11 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  12 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  13 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  14 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  15 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  16 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  17 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  18 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  19 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  20 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  21 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  22 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  23 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  24 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  25 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  26 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  27 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  28 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  29 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  30 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  31 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  32 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  33 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  34 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  35 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  36 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  37 */ {4.757904109314e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  38 */ {4.757500212530e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  39 */ {4.756692418963e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  40 */ {4.754672935046e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  41 */ {4.751037863995e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  42 */ {4.744171618676e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  43 */ {4.730035231254e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  44 */ {4.701762456411e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  45 */ {4.645216906725e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  46 */ {4.532529704136e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  47 */ {4.306751402174e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  48 */ {3.855194798251e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  49 */ {2.951677693622e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  50 */ {1.145047381146e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  51 */ {-2.468213243804e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  52 */ {-9.693926700139e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  53 */ {-2.414656530316e-23,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  54 */ {-5.305143861242e-23,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  55 */ {-1.108619930245e-22,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  56 */ {-2.264814862615e-22,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  57 */ {-4.577216844259e-22,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  58 */ {-9.202016768579e-22,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  59 */ {8.962252346710e-30,6.359999979895e-01,1.000000000000e+00,1.000000000000e+00},
	/*  60 */ {6.052778287327e-30,6.360000000000e-01,1.000000000000e+00,1.000000000000e+00},
	/*  61 */ {-2.229678136030e-29,6.360000010330e-01,1.000000000000e+00,1.000000000000e+00},
	/*  62 */ {1.094485802488e-29,6.359999997788e-01,1.000000000000e+00,1.000000000000e+00},
	/*  63 */ {3.974389299996e-28,6.359999948026e-01,1.000000000000e+00,1.000000000000e+00},
	/*  64 */ {-4.039310769266e-28,6.360000000000e-01,1.000000000000e+00,1.000000000000e+00},
	/*  65 */ {3.564606734231e-28,6.359999990299e-01,1.000000000000e+00,1.000000000000e+00},
	/*  66 */ {4.112210296001e-28,6.359999994046e-01,1.000000000000e+00,1.000000000000e+00},
	/*  67 */ {-1.000153469770e-26,6.360000089263e-01,1.000000000000e+00,1.000000000000e+00},
	/*  68 */ {-2.207044695194e-26,6.360000089392e-01,1.000000000000e+00,1.000000000000e+00},
	/*  69 */ {-5.260098282741e-27,6.360000009520e-01,1.000000000000e+00,1.000000000000e+00},
	/*  70 */ {8.351956485422e-28,6.359999999244e-01,1.000000000000e+00,1.000000000000e+00},
	/*  71 */ {1.092112405843e-25,6.359999942085e-01,1.000000000000e+00,1.000000000000e+00},
	/*  72 */ {9.015979112389e-26,6.359999979603e-01,1.000000000000e+00,1.000000000000e+00},
	/*  73 */ {-1.881902119996e-25,6.360000000000e-01,1.000000000000e+00,1.000000000000e+00},
	/*  74 */ {1.464473929354e-25,6.360000000000e-01,1.000000000000e+00,1.000000000000e+00},
	/*  75 */ {-1.794517578374e-24,6.360000060467e-01,1.000000000000e+00,1.000000000000e+00},
	/*  76 */ {-7.335715821756e-25,6.360000010372e-01,1.000000000000e+00,1.000000000000e+00},
	/*  77 */ {-4.325309392205e-25,6.360000003058e-01,1.000000000000e+00,1.000000000000e+00},
	/*  78 */ {5.011363468281e-24,6.359999982286e-01,1.000000000000e+00,1.000000000000e+00},
	/*  79 */ {5.238271658486e-24,6.360000000000e-01,1.000000000000e+00,1.000000000000e+00},
	/*  80 */ {-9.461870061573e-24,6.360000008361e-01,1.000000000000e+00,1.000000000000e+00},
	/*  81 */ {3.586553493565e-23,6.360000000000e-01,1.000000000000e+00,1.000000000000e+00},
	/*  82 */ {-9.499609956099e-24,6.360000002098e-01,1.000000000000e+00,1.000000000000e+00},
	/*  83 */ {1.483231027914e-22,6.359999983615e-01,1.000000000000e+00,1.000000000000e+00},
	/*  84 */ {1.361292013597e-21,6.359999917218e-01,1.000000000000e+00,1.000000000000e+00},
	/*  85 */ {3.703770164784e-22,6.359999999993e-01,1.000000000000e+00,1.000000000000e+00},
	/*  86 */ {-1.990711921854e-21,6.360000034849e-01,1.000000000000e+00,1.000000000000e+00},
	/*  87 */ {2.967459152642e-21,6.359999979498e-01,1.000000000000e+00,1.000000000000e+00},
	/*  88 */ {3.960405871378e-21,6.359999986298e-01,1.000000000000e+00,1.000000000000e+00},
	/*  89 */ {-1.936216556485e-21,6.360000003281e-01,1.000000000000e+00,1.000000000000e+00},
	/*  90 */ {-3.882662399300e-21,6.360000003677e-01,1.000000000000e+00,1.000000000000e+00},
	/*  91 */ {-2.320984142058e-20,6.360000009771e-01,1.000000000000e+00,1.000000000000e+00},
	/*  92 */ {6.979721239444e-20,6.359999981248e-01,1.000000000000e+00,1.000000000000e+00},
	/*  93 */ {1.335146733726e-19,6.359999984623e-01,1.000000000000e+00,1.000000000000e+00},
	/*  94 */ {-1.606775333923e-19,6.360000006717e-01,1.000000000000e+00,1.000000000000e+00},
	/*  95 */ {-1.156148887697e-20,6.359999996412e-01,1.000000000000e+00,1.000000000000e+00},
	/*  96 */ {2.062750231932e-19,6.359999989363e-01,1.000000000000e+00,1.000000000000e+00},
	/*  97 */ {3.167564078680e-19,6.359999982353e-01,9.947072875822e-01,1.000000000000e+00},
	/*  98 */ {4.845849021462e-18,6.359999956969e-01,8.559422706821e-01,1.000000000000e+00},
	/*  99 */ {2.332972904185e-17,6.359999903420e-01,9.179193652688e-01,1.000000000000e+00},
	/* 100 */ {1.532918908139e-16,6.359999707754e-01,1.306843579449e+00,1.000000000000e+00},
	/* 101 */ {4.758815889824e-16,6.359999551774e-01,1.000000000000e+00,1.000000000000e+00},
	/* 102 */ {6.468510096343e-16,6.359999691565e-01,3.449737738439e-01,1.000000000000e+00},
	/* 103 */ {2.833267913672e-15,6.359999324512e-01,3.777601711423e-01,1.000000000000e+00},
	/* 104 */ {3.290815348383e-15,6.359999607714e-01,1.096909728726e-01,1.000000000000e+00},
	/* 105 */ {4.996158272672e-15,6.359999700923e-01,4.197314096141e-02,1.000000000000e+00},
	/* 106 */ {2.483092446301e-14,6.359999260004e-01,5.172814517104e-02,1.000000000000e+00},
	/* 107 */ {-3.083230451878e-14,6.360000455146e-01,-1.591118305514e-02,1.000000000000e+00},
	/* 108 */ {-2.582626067268e-14,6.360000192519e-01,-3.369973819651e-03,1.000000000000e+00},
	/* 109 */ {4.390437562161e-14,6.359999747283e-01,2.938150919891e-03,1.000000000000e+00},
	/* 110 */ {-2.884586110836e-13,6.360000538880e-01,-2.375988592383e-03,9.999988660301e-01},
	/* 111 */ {-5.107894032853e-14,6.360000065817e-01,-2.075509735813e-04,1.000199133053e+00},
	/* 112 */ {8.934938640636e-13,6.359999605926e-01,3.450718206298e-04,9.999998757740e-01},
	/* 113 */ {9.520719386034e-13,6.359999907170e-01,-1.743972439045e-04,1.406683444066e+00},
	/* 114 */ {-2.145906890361e-11,6.360002998671e-01,-1.111246655000e-03,7.306667059300e-01},
	/* 115 */ {-2.480277302591e-11,6.360002183516e-01,-6.154512919751e-04,3.456999449764e-01},
	/* 116 */ {-2.019331473312e-11,6.360000965131e-01,-1.465411038958e-04,-1.413484253327e-01},
	/* 117 */ {5.221504213571e-11,6.359998787037e-01,9.078833097823e-05,-2.338260550078e-01},
	/* 118 */ {-2.074614384868e-10,6.360002308034e-01,-8.236136646657e-05,-2.025458910765e-01},
	/* 119 */ {3.462578142383e-10,6.359998056102e-01,3.482529993106e-05,-2.139947217092e-01},
	/* 120 */ {-2.004956743633e-09,6.360005576663e-01,-5.018865511942e-05,-2.105152501075e-01},
	/* 121 */ {-3.744826849909e-10,6.360001605501e-01,-1.558260451977e-05,-2.115299297955e-01},
	/* 122 */ {-1.002717872461e-07,6.360086141431e-01,-2.570017386577e-04,-2.091869870712e-01},
	/* 123 */ {-3.131711561227e-06,6.361347143217e-01,-2.013898369036e-03,-2.009415090658e-01},
	/* 124 */ {-9.108356975611e-05,6.379704631485e-01,-1.485567788587e-02,-1.706638464527e-01},
	/* 125 */ {-2.033588457575e-03,6.585270452750e-01,-8.793556017816e-02,-8.292931848708e-02},
	/* 126 */ {-1.944167444918e-02,7.560017139957e-01,-2.723876181674e-01,3.534081054169e-02},
	/* 127 */ {1.117474139645e-03,7.168597521881e-01,-2.542553903143e-01,3.579139590728e-02},
	/* 128 */ {2.543149916167e-01,3.504789178164e-01,-7.474995706686e-02,5.984198191882e-03},
	/* 129 */ {5.694697225950e-01,1.091650506538e-01,-1.219415138160e-02,5.030707966931e-04},
	/* 130 */ {7.755289683542e-01,2.896857636864e-02,-1.636832212420e-03,3.401634687773e-05},
	/* 131 */ {8.861278744722e-01,7.353347956757e-03,-2.083492670521e-04,2.168962712181e-06},
	/* 132 */ {9.424314613696e-01,1.845392552458e-03,-2.616258150849e-05,1.362423765975e-07},
	/* 133 */ {9.707106864745e-01,4.617907802470e-04,-3.274053976563e-06,8.525852648383e-09},
	/* 134 */ {9.848662886768e-01,1.154753863741e-04,-4.093734669752e-07,5.330332172376e-10},
	/* 135 */ {9.919460903876e-01,2.887057770996e-05,-5.117533178501e-08,3.331719280387e-11},
	/* 136 */ {9.954862413787e-01,7.217752628660e-06,-6.397030493189e-09,2.082365439264e-12},
	/* 137 */ {9.972563481433e-01,1.804444919342e-06,-7.996323744475e-10,1.301484787623e-13},
	/* 138 */ {9.981414054340e-01,4.511116525776e-07,-9.995415818778e-11,8.134289909680e-15},
	/* 139 */ {9.985839345681e-01,1.127779394797e-07,-1.249427324037e-11,5.083932746992e-16},
	/* 140 */ {9.988051991963e-01,2.819448648112e-08,-1.561784260450e-12,3.177458201657e-17},
	/* 141 */ {9.989158315233e-01,7.048620841205e-09,-1.952229924085e-13,1.985910741613e-18},
	/* 142 */ {9.989711476775e-01,1.762155577990e-09,-2.440288261851e-14,1.241194846330e-19},
	/* 143 */ {9.989988057601e-01,4.405388737659e-10,-3.050360083972e-15,7.757466887139e-21},
	/* 144 */ {9.990126346502e-01,1.101371454235e-10,-3.813076358306e-16,4.848627744709e-22},
	/* 145 */ {9.990195493189e-01,2.753369541891e-11,-4.766191091832e-17,3.030262910845e-23},
	/* 146 */ {9.990230064824e-01,6.883811403231e-12,-5.958244813872e-18,1.894134162303e-24},
	/* 147 */ {9.990247341450e-01,1.723022483697e-12,-7.461216253976e-19,1.186611374114e-25},
	/* 148 */ {9.990257848423e-01,2.442676128814e-13,-3.322330778099e-20,1.189224384603e-27},
	/* 149 */ {9.990261524499e-01,4.239607243043e-14,-2.103934748098e-22,-1.852140680401e-28},
	/* 150 */ {9.990263874370e-01,-8.329726215948e-15,1.392665466874e-21,-4.537510199500e-29},
	/* 151 */ {9.990263532414e-01,7.028744746170e-15,-1.937946643318e-22,1.956680607471e-30},
	/* 152 */ {9.990263740841e-01,3.950154711427e-15,-6.891332982600e-23,4.157198906318e-31},
	/* 153 */ {9.990265538255e-01,-3.339718926322e-15,3.598871333015e-23,-1.221689406119e-31},
	/* 154 */ {9.990263999961e-01,9.000529810812e-16,-4.395811862854e-24,7.079132772483e-33},
	/* 155 */ {9.990265027487e-01,-3.313792163522e-16,8.537394180445e-25,-7.035506426742e-34},
	/* 156 */ {9.990263745120e-01,3.357086714496e-16,-4.063935220603e-25,1.583301406374e-34},
	/* 157 */ {9.990265223415e-01,-1.152566937969e-16,7.209868286738e-26,-1.450499849240e-35},
	/* 158 */ {9.990265280032e-01,-6.258952282492e-17,1.950063130343e-26,-1.953132618420e-36},
	/* 159 */ {9.990264724457e-01,-6.464789195709e-18,1.372046622364e-27,-8.736212074226e-38},
	/* 160 */ {9.990265629395e-01,-2.484270299512e-17,1.996542587813e-27,-5.164945463413e-38},
	/* 161 */ {9.990263694625e-01,1.191308546769e-17,-4.816907439879e-28,6.267122086347e-39},
	/* 162 */ {9.990264978944e-01,-2.094007117524e-18,4.113888032622e-29,-2.599372812936e-40},
	/* 163 */ {9.990262655838e-01,6.135374211248e-18,-6.086967488998e-29,1.946689057268e-40},
	/* 164 */ {9.990263935739e-01,1.096735764668e-18,-5.481521832022e-30,8.817200565562e-42},
	/* 165 */ {9.990264489644e-01,1.079668728638e-19,-2.529746348158e-31,1.923775594324e-43},
	/* 166 */ {9.990265176013e-01,-2.063443650227e-19,2.560897714969e-31,-1.028376050610e-43},
	/* 167 */ {9.990264170171e-01,8.968697115490e-20,-5.427805338911e-32,1.055633187335e-44},
	/* 168 */ {9.990264481573e-01,4.479408684491e-21,1.432298182189e-33,-3.298477079637e-46},
	/* 169 */ {9.990266051555e-01,-6.918644989222e-20,1.086522513524e-32,-5.494536465430e-46},
	/* 170 */ {9.990264726758e-01,2.345675690643e-23,-1.385958288604e-34,6.110524421003e-48},
	/* 171 */ {9.990265241158e-01,-7.558943943423e-21,3.027307576123e-34,-3.892901598578e-48},
	/* 172 */ {9.990266156695e-01,-9.217645012740e-21,1.794443297041e-34,-1.125627254767e-48},
	/* 173 */ {9.990264665436e-01,7.692727968455e-23,-2.512516052752e-36,1.167056291149e-50},
	/* 174 */ {9.990265137327e-01,-7.501559518179e-22,3.656299292790e-36,-5.773239015581e-51},
	/* 175 */ {9.990265037844e-01,-2.980107612504e-22,7.145838477272e-37,-5.538004920986e-52},
	/* 176 */ {9.990263762057e-01,3.356727000731e-22,-4.117322126036e-37,1.625902085501e-52},
	/* 177 */ {9.990263655568e-01,1.885503331983e-22,-1.160848313130e-37,2.300816477320e-53},
	/* 178 */ {9.990265203066e-01,-5.136417399718e-23,1.506333543011e-38,-1.422526953243e-54},
	/* 179 */ {9.990264641319e-01,-1.918157853355e-24,5.970516460325e-40,-4.554819266655e-56},
	/* 180 */ {9.990264006940e-01,1.515996541260e-23,-1.169727177827e-39,2.909508278050e-56},
	/* 181 */ {9.990263710154e-01,1.128914576578e-23,-4.415313904788e-40,5.551348521468e-57},
	/* 182 */ {9.990263859012e-01,4.668240760645e-24,-8.937295335756e-41,5.493457901271e-58},
	/* 183 */ {9.990264310337e-01,1.051222615898e-24,-1.053892928671e-41,3.336568927500e-59},
	/* 184 */ {9.990264258914e-01,6.052833445425e-25,-3.095140027715e-42,5.069879858219e-60},
	/* 185 */ {9.990264686413e-01,1.517583573997e-28,-1.013998595680e-43,1.502267978500e-61},
	/* 186 */ {9.990264512206e-01,3.174926356763e-26,-1.835768765287e-44,-2.427581684235e-64},
	/* 187 */ {9.990265357026e-01,-1.333175390006e-25,7.973387857354e-44,-1.540708507660e-62},
	/* 188 */ {9.990263641091e-01,9.293429839422e-26,-2.773063528301e-44,2.665150369277e-63},
	/* 189 */ {9.990264244921e-01,1.750068963174e-26,-2.469120516784e-45,1.118026470693e-64},
	/* 190 */ {9.990265374168e-01,-1.686405484224e-26,1.242986291427e-45,-2.961090848815e-65},
	/* 191 */ {9.990265948981e-01,-1.517320925081e-26,5.647797542661e-46,-6.779810618080e-66},
	/* 192 */ {9.990266404652e-01,-1.027756417648e-26,1.914891018940e-46,-1.148178599078e-66},
	/* 193 */ {9.990264042606e-01,1.749492127666e-27,-1.636001402980e-47,4.906098016595e-68},
	/* 194 */ {9.990263310484e-01,1.921285448822e-27,-8.926722551405e-48,1.337459466602e-68},
	/* 195 */ {9.990263046770e-01,1.151893731825e-27,-2.676441597076e-48,2.004561167028e-69},
	/* 196 */ {9.990264199246e-01,1.548124399622e-28,-1.767624896425e-49,6.564075374197e-71},
	/* 197 */ {9.990263529911e-01,2.036761867145e-28,-1.200792314596e-49,2.277631143563e-71},
	/* 198 */ {9.990263338725e-01,1.165740412679e-28,-3.359813846560e-50,3.124437226089e-72},
	/* 199 */ {9.990264817950e-01,-8.972200011828e-30,1.391235217220e-51,-6.833944541788e-74},
	/* 200 */ {9.990264705157e-01,-1.697972722156e-30,1.204102970887e-52,-2.689315699364e-75},
	/* 201 */ {9.990265506476e-01,-9.838724341872e-30,3.572629378166e-52,-4.176566613793e-75},
	/* 202 */ {9.990265510169e-01,-4.951840637632e-30,8.999245630233e-53,-5.258715602094e-76},
	/* 203 */ {9.990265518755e-01,-2.499522816119e-30,2.270523482635e-53,-6.635338560387e-77},
	/* 204 */ {9.990264604515e-01,-3.081804959879e-33,2.577550669458e-55,-7.473238410648e-79},
	/* 205 */ {9.990264273134e-01,2.892271889831e-31,-7.178455201723e-55,5.596147534919e-79},
	/* 206 */ {9.990264115716e-01,7.192098114375e-32,2.618642372138e-56,-4.157599902060e-80},
	/* 207 */ {9.990263828734e-01,1.411052850695e-31,-7.901802913582e-56,1.427510949944e-80},
	/* 208 */ {9.990264046240e-01,5.136795271884e-32,-1.440256141513e-56,1.306280447796e-81},
	/* 209 */ {9.990266520848e-01,-8.335145908068e-32,1.183726945535e-56,-5.415140975763e-82},
	/* 210 */ {9.990264056144e-01,1.306764556308e-32,-9.378840305768e-58,2.164801725643e-83},
	/* 211 */ {9.990265464764e-01,-9.137141422481e-33,3.242552677302e-58,-3.706415339011e-84},
	/* 212 */ {9.990264778518e-01,-3.185315364526e-34,6.938673718897e-61,1.013780763787e-86},
	/* 213 */ {9.990264699565e-01,1.363459805871e-35,-1.402514984040e-60,6.574497949561e-87},
	/* 214 */ {9.990265251464e-01,-8.543519157163e-34,3.814134588459e-60,-5.479325484127e-87},
	/* 215 */ {9.990265539872e-01,-6.277061714118e-34,1.401959925115e-60,-1.009313368599e-87},
	/* 216 */ {9.990265203197e-01,-1.916570212333e-34,2.081568272828e-61,-7.290012294916e-89},
	/* 217 */ {9.990265047692e-01,-7.444228114119e-35,4.381456646308e-62,-8.290750031865e-90},
	/* 218 */ {9.990263965103e-01,5.771804801341e-35,-1.595214952867e-62,1.424404391077e-90},
	/* 219 */ {9.990264287875e-01,1.524790903448e-35,-2.146952449858e-63,9.777811482565e-92},
	/* 220 */ {9.990263532797e-01,2.386352524630e-35,-1.655338689664e-63,3.702151038654e-92},
	/* 221 */ {9.990264484442e-01,1.182488831263e-36,-2.053369292590e-65,-7.832365530890e-96},
	/* 222 */ {9.990265162794e-01,-1.992804956969e-36,1.740858308865e-65,-9.320332820748e-97},
	/* 223 */ {9.990264165433e-01,1.243311142370e-36,-1.045327099204e-65,2.837868509599e-95},
	/* 224 */ {9.990266178538e-01,-2.065920220979e-36,8.898130210848e-66,-1.236549699774e-95},
	/* 225 */ {9.990263757547e-01,6.072206328464e-37,-1.337941582221e-66,9.458151591577e-97},
	/* 226 */ {9.990263111651e-01,5.109698072568e-37,-5.496689086767e-67,1.906722541213e-97},
	/* 227 */ {9.990263994674e-01,1.080461021285e-37,-5.811281531597e-68,1.008694658176e-98},
	/* 228 */ {9.990264723709e-01,-4.588138725662e-39,5.423239828731e-70,1.155905404156e-101},
	/* 229 */ {9.990264668327e-01,-9.205520584796e-40,5.769920777385e-71,8.871163544763e-103},
	/* 230 */ {9.990263671156e-01,2.042997401010e-38,-1.387138495939e-69,3.038030541285e-101},
	/* 231 */ {9.990264992186e-01,-3.613741478693e-39,1.192803936405e-70,-1.277050557487e-102},
	/* 232 */ {9.990263949953e-01,3.597802892136e-39,-6.031386632353e-71,3.257797002948e-103},
	/* 233 */ {9.990265127522e-01,-1.294188436482e-39,1.102441905265e-71,-3.023448463794e-104},
	/* 234 */ {9.990263773048e-01,1.129135760955e-39,-4.719244707620e-72,6.350335440077e-105},
	/* 235 */ {9.990263887732e-01,4.873979709497e-40,-1.010086076054e-72,6.744310986466e-106},
	/* 236 */ {9.990264207150e-01,1.433434060491e-40,-1.529391929758e-73,5.239935200045e-107},
	/* 237 */ {9.990264019849e-01,1.005766198642e-40,-5.258797130155e-74,8.897613436037e-108},
	/* 238 */ {9.990265884325e-01,-1.036124055797e-40,2.763966769092e-74,-2.373865526869e-108},
	/* 239 */ {9.990265404942e-01,-3.183967046321e-41,4.247312355172e-75,-1.823373961882e-109},
	/* 240 */ {9.990265368952e-01,-1.529693403128e-41,1.023044823662e-75,-2.201112690469e-110},
	/* 241 */ {9.990264206734e-01,4.731659341487e-42,-1.632351696288e-76,1.790270454410e-111},
	/* 242 */ {9.990264212902e-01,2.333424227084e-42,-4.103467716216e-77,2.307808063190e-112},
	/* 243 */ {9.990265714368e-01,-2.791457984144e-42,2.323121974259e-77,-6.226802733853e-113},
	/* 244 */ {9.990265668649e-01,-1.321505291231e-42,5.440954216183e-78,-7.228148592084e-114},
	/* 245 */ {9.990263862879e-01,5.006270207993e-43,-1.039841743339e-78,6.968957230602e-115},
	/* 246 */ {9.990264066115e-01,1.827014603038e-43,-1.882591436377e-79,6.296864622660e-116},
	/* 247 */ {9.990264392226e-01,2.887580733731e-44,-7.658371107590e-81,-2.014493654098e-119},
	/* 248 */ {9.990264012927e-01,5.233853185663e-44,-1.400334221965e-80,1.202628784683e-117},
	/* 249 */ {9.990267369535e-01,-1.095425595425e-43,1.412255301791e-80,-5.872396753517e-118},
};
#endif

ptbl_t tanhtbl = {
	/*   0 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   1 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   2 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   3 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   4 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   5 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   6 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   7 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   8 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   9 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  10 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  11 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  12 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  13 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  14 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  15 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  16 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  17 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  18 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  19 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  20 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  21 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  22 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  23 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  24 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  25 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  26 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  27 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  28 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  29 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  30 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  31 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  32 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  33 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  34 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  35 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  36 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  37 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  38 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  39 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  40 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  41 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  42 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  43 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  44 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  45 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  46 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  47 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  48 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  49 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  50 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  51 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  52 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  53 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  54 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  55 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  56 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  57 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  58 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  59 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  60 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  61 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  62 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  63 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  64 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  65 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  66 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  67 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  68 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  69 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  70 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  71 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  72 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  73 */ {-1.218354324229e-20,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  74 */ {-4.058148085185e-21,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  75 */ {-4.058148085185e-21,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  76 */ {-4.058148085185e-21,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  77 */ {-4.058148085185e-21,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  78 */ {-4.058148085185e-21,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  79 */ {-4.058148085185e-21,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  80 */ {-4.058148085185e-21,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  81 */ {-4.058148085185e-21,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  82 */ {-4.058148085185e-21,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  83 */ {-4.058148085185e-21,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  84 */ {-4.058148085185e-21,9.999999999995e-01,1.000000000000e+00,1.000000000000e+00},
	/*  85 */ {-4.057941290032e-21,9.999999999995e-01,1.000000000000e+00,1.000000000000e+00},
	/*  86 */ {-4.057527699726e-21,9.999999999985e-01,1.000000000000e+00,1.000000000000e+00},
	/*  87 */ {-4.056493723960e-21,9.999999999973e-01,1.000000000000e+00,1.000000000000e+00},
	/*  88 */ {-4.051117049978e-21,9.999999999945e-01,1.000000000000e+00,1.000000000000e+00},
	/*  89 */ {-4.030437534665e-21,9.999999999891e-01,1.000000000000e+00,1.000000000000e+00},
	/*  90 */ {-3.946478702490e-21,9.999999999782e-01,1.000000000000e+00,1.000000000000e+00},
	/*  91 */ {-3.611056964100e-21,9.999999999563e-01,1.000000000000e+00,1.000000000000e+00},
	/*  92 */ {-2.269783600846e-21,9.999999999127e-01,1.000000000000e+00,1.000000000000e+00},
	/*  93 */ {3.094482671561e-21,9.999999998254e-01,1.000000000000e+00,1.000000000000e+00},
	/*  94 */ {2.455278853211e-20,9.999999996508e-01,1.000000000000e+00,1.000000000000e+00},
	/*  95 */ {1.103851847937e-19,9.999999993015e-01,1.000000000000e+00,1.000000000000e+00},
	/*  96 */ {4.537160106108e-19,9.999999986030e-01,1.000000000000e+00,1.000000000000e+00},
	/*  97 */ {1.827039107084e-18,9.999999972060e-01,1.000000000000e+00,1.000000000000e+00},
	/*  98 */ {7.434533785508e-18,9.999999943440e-01,1.009920641977e+00,1.000000000000e+00},
	/*  99 */ {2.528672012728e-17,9.999999897046e-01,9.621623771491e-01,1.000000000000e+00},
	/* 100 */ {1.195658394873e-16,9.999999788101e-01,8.955530742261e-01,1.000000000000e+00},
	/* 101 */ {-3.289499645717e-16,1.000000028648e+00,-5.913189799518e-01,1.000000000000e+00},
	/* 102 */ {1.212434105646e-15,9.999999403954e-01,6.825396077968e-01,1.000000000000e+00},
	/* 103 */ {2.810512472775e-15,9.999999347222e-01,3.593488385985e-01,1.000000000000e+00},
	/* 104 */ {-2.430366242999e-15,1.000000029863e+00,-8.427095111749e-02,1.000000000000e+00},
	/* 105 */ {2.647073954260e-15,9.999999840086e-01,2.252039200811e-02,1.000000000000e+00},
	/* 106 */ {1.677160707678e-14,9.999999509475e-01,3.396832781982e-02,1.000000000000e+00},
	/* 107 */ {2.772142515310e-16,9.999999972677e-01,1.594542894087e-03,1.000000000000e+00},
	/* 108 */ {-3.916579969120e-14,1.000000031269e+00,-5.754784907908e-03,1.000000000000e+00},
	/* 109 */ {1.849626193360e-14,9.999999881694e-01,1.414652577079e-03,1.000000000000e+00},
	/* 110 */ {2.354212796318e-13,9.999999558924e-01,1.903430297684e-03,1.000000000000e+00},
	/* 111 */ {-1.323074639534e-15,9.999999926026e-01,2.564152650366e-04,1.000000000128e+00},
	/* 112 */ {5.336651211114e-12,9.999997449804e-01,2.745753923820e-03,1.000000000000e+00},
	/* 113 */ {5.022601768338e-13,9.999999755950e-01,3.811058141025e-04,-2.113947329083e+00},
	/* 114 */ {-3.411531641814e-12,1.000000087133e+00,-6.265143004667e-04,9.935304098068e-01},
	/* 115 */ {-5.927774533478e-13,1.000000040389e+00,-2.095775763927e-04,-6.413887477786e-02},
	/* 116 */ {-4.648095817510e-11,1.000000215890e+00,-3.365144219043e-04,-1.593625221322e-01},
	/* 117 */ {-8.966972719379e-10,1.000002026654e+00,-1.458772765482e-03,2.683485287765e-03},
	/* 118 */ {-9.395855163148e-11,1.000000103704e+00,-3.700708334132e-05,-3.290306537490e-01},
	/* 119 */ {6.246855903759e-12,1.000000013744e+00,-5.458495752819e-06,-3.328471732639e-01},
	/* 120 */ {-1.448624383413e-09,1.000000405018e+00,-3.680710247778e-05,-3.322234030371e-01},
	/* 121 */ {-9.703759037238e-10,1.000000247086e+00,-1.970419565415e-05,-3.327941012377e-01},
	/* 122 */ {-1.054850286912e-07,1.000009059947e+00,-2.702144067610e-04,-3.303771710485e-01},
	/* 123 */ {-3.328965058480e-06,1.000143079940e+00,-2.136320880641e-03,-3.216252119771e-01},
	/* 124 */ {-1.008357620845e-04,1.002174158399e+00,-1.631066713398e-02,-2.882959832432e-01},
	/* 125 */ {-2.603067940138e-03,1.028449524066e+00,-1.088826741925e-01,-1.782709462771e-01},
	/* 126 */ {-3.740650643930e-02,1.217822344852e+00,-4.562783873558e-01,3.745670489915e-02},
	/* 127 */ {-4.122154803031e-02,1.276613094027e+00,-5.606005150950e-01,8.680312505401e-02},
	/* 128 */ {6.281731930282e-01,3.065122100044e-01,-8.515422064893e-02,7.930856204314e-03},
	/* 129 */ {9.885694049101e-01,5.166919331324e-03,-7.712074271839e-04,3.799275529049e-05},
	/* 130 */ {9.999955473199e-01,1.028162688042e-06,-7.806098149471e-08,1.949630804794e-09},
	/* 131 */ {1.000000112346e+00,-1.511824152070e-08,6.524238944124e-10,-9.065951457265e-12},
	/* 132 */ {1.000000069587e+00,-4.720072491067e-09,1.030790509665e-10,-7.243691034290e-13},
	/* 133 */ {9.999999324539e-01,2.198578839964e-09,-2.284519319978e-11,7.630889432470e-14},
	/* 134 */ {9.999998277910e-01,2.857356274269e-09,-1.519841343102e-11,2.605849945510e-14},
	/* 135 */ {9.999998544118e-01,1.230530988297e-09,-3.335753266778e-12,2.911457106272e-15},
	/* 136 */ {9.999998703926e-01,5.380816718343e-10,-7.158038190358e-13,3.063418548340e-16},
	/* 137 */ {9.999999322365e-01,1.407624043552e-10,-9.381596139353e-14,2.019842937195e-17},
	/* 138 */ {1.000000072595e+00,-7.155026354447e-11,2.284300584057e-14,-2.382083581760e-18},
	/* 139 */ {1.000000224244e+00,-1.173344713178e-10,1.972349324095e-14,-1.068950775210e-18},
	/* 140 */ {9.999999243002e-01,1.914451920920e-11,-1.545560239119e-15,4.000075234813e-20},
	/* 141 */ {1.000000119732e+00,-1.584039661931e-11,6.719685232883e-16,-9.168924074923e-21},
	/* 142 */ {1.000000010590e+00,-2.356461077357e-13,-5.366878990479e-18,1.061985310001e-22},
	/* 143 */ {1.000000101231e+00,-3.358023900072e-12,3.578399950070e-17,-1.228145179424e-22},
	/* 144 */ {9.999998867430e-01,1.850740978496e-12,-9.697165913489e-18,1.638193903749e-23},
	/* 145 */ {1.000000132697e+00,-1.083106542073e-12,2.830069576593e-18,-2.380364700083e-24},
	/* 146 */ {9.999999125972e-01,3.556491075171e-13,-4.618567755443e-19,1.922456134396e-25},
	/* 147 */ {1.000000000061e+00,-2.951085523368e-15,1.361181513922e-21,-4.076889729614e-28},
	/* 148 */ {1.000000076882e+00,-8.010717903743e-14,2.676768129432e-20,-2.876274392283e-27},
	/* 149 */ {1.000000095853e+00,-4.797935252548e-14,7.668981180237e-21,-3.945859720009e-28},
	/* 150 */ {9.999998967185e-01,2.630060990583e-14,-2.142706418435e-21,5.618043858533e-29},
	/* 151 */ {9.999999278577e-01,9.084480189355e-15,-3.689560570269e-22,4.837109174782e-30},
	/* 152 */ {1.000000143249e+00,-9.242860173680e-15,1.914368566106e-22,-1.275550322420e-30},
	/* 153 */ {9.999997865343e-01,6.787014142199e-15,-6.915208529931e-23,2.269026147243e-31},
	/* 154 */ {9.999998883099e-01,1.782717279343e-15,-9.124459490332e-24,1.504115333334e-32},
	/* 155 */ {1.000000067111e+00,-5.436405530794e-16,1.441506617448e-24,-1.245521037555e-33},
	/* 156 */ {1.000000161842e+00,-6.450736589978e-16,8.249789497421e-25,-3.397136212992e-34},
	/* 157 */ {9.999999572586e-01,8.527830862930e-17,-5.428353479022e-26,1.111122030936e-35},
	/* 158 */ {1.000000118048e+00,-1.195303549020e-16,3.872361087929e-26,-4.033385963197e-36},
	/* 159 */ {1.000000037939e+00,-1.985179002383e-17,3.295922848237e-27,-1.741114607518e-37},
	/* 160 */ {9.999999896454e-01,2.475017314657e-18,-1.866097509346e-28,4.496920989429e-39},
	/* 161 */ {1.000000013351e+00,-1.712308463301e-18,6.805278279452e-29,-8.503143884388e-40},
	/* 162 */ {1.000000007481e+00,-8.408089318045e-21,-9.908945434453e-30,1.258231568861e-40},
	/* 163 */ {1.000000018253e+00,-6.966872094175e-19,8.074539972800e-30,-2.880409629430e-41},
	/* 164 */ {9.999999905302e-01,1.765526141440e-19,-9.859033646688e-31,1.717114709316e-42},
	/* 165 */ {1.000000114140e+00,-8.845247358305e-19,2.197985266074e-30,-1.757316976911e-42},
	/* 166 */ {1.000000183719e+00,-7.132919128185e-19,8.882172089587e-31,-3.565913037698e-43},
	/* 167 */ {9.999999108574e-01,1.696495803180e-19,-1.037999720279e-31,2.051337235032e-44},
	/* 168 */ {9.999999039695e-01,9.211016982948e-20,-2.838556441092e-32,2.824670989561e-45},
	/* 169 */ {1.000000086570e+00,-4.175634602160e-20,6.499287753394e-33,-3.266311329467e-46},
	/* 170 */ {9.999999032680e-01,2.373769191393e-20,-1.869130288205e-33,4.736862494247e-47},
	/* 171 */ {9.999999804926e-01,2.215852802313e-21,-8.104547094829e-35,9.597918743803e-49},
	/* 172 */ {1.000000043277e+00,-2.232615191968e-21,3.675858001015e-35,-1.935821319538e-49},
	/* 173 */ {9.999998379495e-01,4.903182262472e-21,-4.760451993792e-35,1.490717800728e-49},
	/* 174 */ {1.000000238301e+00,-3.620381542899e-21,1.766727222543e-35,-2.778791402060e-50},
	/* 175 */ {9.999998293851e-01,1.303487173021e-21,-3.200551666108e-36,2.534415477358e-51},
	/* 176 */ {9.999999595895e-01,1.408211482820e-22,-1.578775003009e-37,5.767928544476e-53},
	/* 177 */ {1.000000161388e+00,-3.064445333792e-22,1.866406376265e-37,-3.663273378420e-53},
	/* 178 */ {1.000000074412e+00,-7.036405049848e-23,2.130752649339e-38,-2.074539295067e-54},
	/* 179 */ {9.999998836625e-01,5.484297769375e-23,-8.278272658757e-39,4.025962552905e-55},
	/* 180 */ {9.999999917724e-01,1.442739134010e-24,-5.806719124540e-41,-3.781803913518e-59},
	/* 181 */ {1.000000141749e+00,-1.693145939996e-23,6.484096850542e-40,-7.993896033254e-57},
	/* 182 */ {9.999998357884e-01,9.765856910772e-24,-1.861413364999e-40,1.142775556954e-57},
	/* 183 */ {1.000000057138e+00,-1.683241633463e-24,1.580454863448e-41,-4.734314341302e-59},
	/* 184 */ {1.000000065953e+00,-9.996751784851e-25,4.853713247014e-42,-7.572578309191e-60},
	/* 185 */ {1.000000055223e+00,-4.133048615446e-25,9.933479374871e-43,-7.707190671946e-61},
	/* 186 */ {9.999998348828e-01,6.158348492355e-25,-7.355768678463e-43,2.830006109312e-61},
	/* 187 */ {1.000000079020e+00,-1.482713433291e-25,8.956690238398e-44,-1.747418247667e-62},
	/* 188 */ {9.999998269733e-01,1.585785077564e-25,-4.667332109796e-44,4.430901645723e-63},
	/* 189 */ {9.999999726277e-01,1.190003049610e-26,-1.673794687226e-45,7.667306119210e-65},
	/* 190 */ {9.999998912477e-01,2.552326096385e-26,-1.913184734223e-45,4.609950910530e-65},
	/* 191 */ {9.999998738202e-01,1.457545456813e-26,-5.394783453705e-46,6.442566749465e-66},
	/* 192 */ {9.999999280352e-01,4.137113939605e-27,-7.599036740323e-47,4.493731540685e-67},
	/* 193 */ {9.999999110236e-01,2.523656047423e-27,-2.294318199147e-47,6.717283864274e-68},
	/* 194 */ {9.999997443597e-01,3.698597465688e-27,-1.716968673750e-47,2.568851715358e-68},
	/* 195 */ {1.000000075889e+00,-5.608875810188e-28,1.324756971345e-48,-1.002116603496e-69},
	/* 196 */ {1.000000011145e+00,-4.822881637519e-29,6.655082138392e-50,-2.856856190467e-71},
	/* 197 */ {1.000000170703e+00,-3.084736542236e-28,1.792064029653e-49,-3.357657787017e-71},
	/* 198 */ {9.999999961500e-01,5.333458711138e-30,-2.034274806433e-51,2.231561673060e-73},
	/* 199 */ {9.999999155722e-01,3.757327326036e-29,-5.377466995971e-51,2.487439754572e-73},
	/* 200 */ {1.000000078594e+00,-1.762133515562e-29,1.270420984138e-51,-2.953435836177e-74},
	/* 201 */ {9.999996360571e-01,4.112814580472e-29,-1.491114830435e-51,1.742106390002e-74},
	/* 202 */ {9.999998517067e-01,8.412746302150e-30,-1.531318551863e-52,8.978306223308e-76},
	/* 203 */ {1.000000117394e+00,-3.326967922774e-30,3.019824882531e-53,-8.818961722772e-77},
	/* 204 */ {1.000000081867e+00,-1.149525368782e-30,5.192492406518e-54,-7.565751936824e-78},
	/* 205 */ {1.000000029531e+00,-2.041704866793e-31,4.533813330439e-55,-3.231024629124e-79},
	/* 206 */ {1.000000185442e+00,-6.631519647689e-31,7.607228014122e-55,-2.808621476411e-79},
	/* 207 */ {1.000000062990e+00,-1.144544407878e-31,6.661857905424e-56,-1.243982343275e-80},
	/* 208 */ {1.000000102190e+00,-9.197471639490e-32,2.653262445123e-56,-2.460017978867e-81},
	/* 209 */ {1.000000087022e+00,-3.807320229394e-32,5.364619815939e-57,-2.437829443503e-82},
	/* 210 */ {1.000000115991e+00,-2.536921025785e-32,1.781023171222e-57,-4.035428522653e-83},
	/* 211 */ {9.999999958354e-01,9.157617302079e-34,-4.686702780752e-59,6.807103654424e-85},
	/* 212 */ {9.999998938664e-01,5.815873038410e-33,-1.018596195759e-58,5.741423559967e-85},
	/* 213 */ {9.999999157900e-01,2.257281890112e-33,-1.951043244251e-59,5.454580256124e-86},
	/* 214 */ {9.999999069206e-01,1.270264795048e-33,-5.557448833245e-60,7.836344101423e-87},
	/* 215 */ {9.999998385627e-01,1.110476291648e-33,-2.448341656945e-60,1.737872351923e-87},
	/* 216 */ {9.999999868845e-01,3.127873967183e-35,-1.256997334065e-62,-3.622391440887e-90},
	/* 217 */ {9.999998607297e-01,2.410089356671e-34,-1.337957219686e-61,2.392770147670e-89},
	/* 218 */ {9.999998528181e-01,1.275505372857e-34,-3.544665559974e-62,3.170912834694e-90},
	/* 219 */ {9.999999239838e-01,3.233723972509e-35,-4.397547565138e-63,1.925907869558e-91},
	/* 220 */ {9.999999035902e-01,2.073481403354e-35,-1.433708440694e-63,3.198678612525e-92},
	/* 221 */ {9.999999546079e-01,4.853895680861e-36,-1.666454895062e-64,1.848865444210e-93},
	/* 222 */ {1.000000072590e+00,-3.862168038072e-36,6.636961049939e-65,-3.690761308951e-94},
	/* 223 */ {9.999998359220e-01,4.405600794072e-36,-3.798301351327e-65,1.056684555842e-94},
	/* 224 */ {1.000000087269e+00,-1.160682802436e-36,4.981163300382e-66,-6.920188218309e-96},
	/* 225 */ {9.999999331039e-01,4.479359296347e-37,-9.618135677475e-67,6.672527097978e-97},
	/* 226 */ {1.000000124853e+00,-4.239099486916e-37,4.611278517919e-67,-1.611951354626e-97},
	/* 227 */ {9.999999487502e-01,7.812716313893e-38,-3.808991791291e-68,6.053114697443e-99},
	/* 228 */ {9.999999242034e-01,6.335452467872e-38,-1.703867478248e-68,1.482837428891e-99},
	/* 229 */ {1.000000024157e+00,-1.038237453205e-38,1.405066819046e-69,-6.056931032564e-101},
	/* 230 */ {1.000000081239e+00,-1.732075997780e-38,1.184774540036e-69,-2.607903748216e-101},
	/* 231 */ {1.000000206468e+00,-2.164875726930e-38,7.298109883073e-70,-7.937861069442e-102},
	/* 232 */ {9.999998190400e-01,9.551085882192e-39,-1.620565910235e-70,8.862965964274e-103},
	/* 233 */ {9.999999286321e-01,1.948798284698e-39,-1.692204900421e-71,4.710452322140e-104},
	/* 234 */ {9.999999904544e-01,4.392374732602e-42,4.331812211215e-73,-1.137331429533e-105},
	/* 235 */ {1.000000008913e+00,-3.798630914335e-41,4.001563973018e-74,-2.855494004748e-108},
	/* 236 */ {9.999998494791e-01,4.970288268047e-40,-5.251584279346e-73,1.786142057392e-106},
	/* 237 */ {1.000000029772e+00,-4.576400938828e-41,2.213564896335e-74,-3.435610283177e-108},
	/* 238 */ {1.000000051531e+00,-4.196135608052e-41,1.084779217182e-74,-8.993205806732e-109},
	/* 239 */ {1.000000043059e+00,-1.900496243898e-41,2.690644820064e-75,-1.216829574919e-109},
	/* 240 */ {9.999999603644e-01,8.153509584335e-42,-5.393677465286e-76,1.154972972671e-110},
	/* 241 */ {1.000000030124e+00,-3.260111957658e-42,1.136719538963e-76,-1.267728199750e-111},
	/* 242 */ {1.000000065375e+00,-3.402115087521e-42,5.680458530977e-77,-3.039329917369e-112},
	/* 243 */ {1.000000052859e+00,-1.325323413398e-42,1.069262431252e-77,-2.778287038082e-113},
	/* 244 */ {9.999998771403e-01,1.568319482396e-42,-6.415649261176e-78,8.466656728526e-114},
	/* 245 */ {1.000000179439e+00,-1.149185957943e-42,2.364907874986e-78,-1.568719683025e-114},
	/* 246 */ {1.000000060365e+00,-1.972736196328e-43,2.073990410141e-79,-7.022327553282e-116},
	/* 247 */ {1.000000183002e+00,-2.946237450698e-43,1.521800031389e-79,-2.532481282975e-116},
	/* 248 */ {9.999998410927e-01,1.269621869587e-43,-3.250638001021e-80,2.682873626680e-117},
	/* 249 */ {9.999998537795e-01,5.939842335078e-44,-7.729900533604e-81,3.239187106063e-118},
};

float tanh (float x)
{
	union { float f; uint32 i; } u;
	u.f = x;
	int e = (u.i & 0x7F800000) >> 23;
	float *p = tanhtbl[e];
	x = fabs (x);
	x = p[0]+x*(p[1]+x*(p[2]+x*p[3]));
	u.i &= 0x80000000;
	u.i |= 127 << 23; /* 1.0 - use 126 for .5 instead, etc */
	return x * u.f;
}

float one5tbl[][4] = {
	/*   0 */ {-6.406079902339e-48,1.087756662521e-09,1.000000000000e+00,1.000000000000e+00},
	/*   1 */ {-1.793958785972e-46,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*   2 */ {-3.587917571945e-46,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*   3 */ {-7.175835143890e-46,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*   4 */ {-1.435167028778e-45,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*   5 */ {-2.870334057556e-45,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*   6 */ {-5.740668115112e-45,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*   7 */ {-1.148133623022e-44,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*   8 */ {-2.296267246045e-44,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*   9 */ {-4.592534492090e-44,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  10 */ {-9.185068984179e-44,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  11 */ {-1.837013796836e-43,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  12 */ {-3.674027593672e-43,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  13 */ {-7.348055187343e-43,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  14 */ {-1.469611037469e-42,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  15 */ {-2.939222074937e-42,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  16 */ {-5.878444149875e-42,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  17 */ {-1.175688829975e-41,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  18 */ {-2.351377659950e-41,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  19 */ {-4.702755319900e-41,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  20 */ {-9.405510639799e-41,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  21 */ {-1.881102127960e-40,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  22 */ {-3.762204255920e-40,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  23 */ {-7.524408511839e-40,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  24 */ {-1.504881702368e-39,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  25 */ {-3.009763404736e-39,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  26 */ {-6.019526809472e-39,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  27 */ {-1.203905361894e-38,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  28 */ {-2.407810723789e-38,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  29 */ {-4.815621447577e-38,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  30 */ {-9.631242895155e-38,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  31 */ {-1.926248579031e-37,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  32 */ {-3.852497158062e-37,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  33 */ {-7.704994316124e-37,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  34 */ {-1.540998863225e-36,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  35 */ {-3.081997726449e-36,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  36 */ {-6.163995452899e-36,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  37 */ {-1.232799090580e-35,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  38 */ {-2.465598181160e-35,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  39 */ {-4.931196362319e-35,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  40 */ {-9.862392724638e-35,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  41 */ {-1.972478544928e-34,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  42 */ {-3.944957089855e-34,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  43 */ {-7.889914179711e-34,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  44 */ {-1.577982835942e-33,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  45 */ {-3.155965671884e-33,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  46 */ {-6.311931343768e-33,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  47 */ {-1.262386268754e-32,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  48 */ {-2.524772537507e-32,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  49 */ {-5.049545075015e-32,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  50 */ {-1.009909015003e-31,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  51 */ {-2.019818030006e-31,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  52 */ {-4.039636060012e-31,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  53 */ {-8.079272120024e-31,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  54 */ {-1.615854424005e-30,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  55 */ {-3.231708848009e-30,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  56 */ {-6.463417696019e-30,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  57 */ {-1.292683539204e-29,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  58 */ {-2.585367078408e-29,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  59 */ {-5.170734156815e-29,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  60 */ {-1.034146831363e-28,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  61 */ {-2.068293662726e-28,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  62 */ {-4.136587325452e-28,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  63 */ {-8.273174650904e-28,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  64 */ {-1.654634930181e-27,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  65 */ {-3.309269860362e-27,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  66 */ {-6.618539720723e-27,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  67 */ {-1.323707944145e-26,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  68 */ {-2.647415888289e-26,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  69 */ {-5.294831776579e-26,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  70 */ {-1.058966355316e-25,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  71 */ {-2.117932710631e-25,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  72 */ {-4.235865421263e-25,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  73 */ {-8.471730842526e-25,1.017057338348e-08,1.000000000000e+00,1.000000000000e+00},
	/*  74 */ {1.320206241812e-16,-1.792757350753e+00,1.000000000000e+00,1.000000000000e+00},
	/*  75 */ {3.560356719263e-16,-2.402294729386e+00,1.000000000000e+00,1.000000000000e+00},
	/*  76 */ {-2.910703221197e-16,-8.963786019276e-01,1.000000000000e+00,1.000000000000e+00},
	/*  77 */ {-1.570792639010e-16,-1.048762980944e+00,1.000000000000e+00,1.000000000000e+00},
	/*  78 */ {-4.271512410088e-16,-8.605235043353e-01,1.000000000000e+00,1.000000000000e+00},
	/*  79 */ {-1.001749502526e-16,-9.916188388130e-01,1.000000000000e+00,1.000000000000e+00},
	/*  80 */ {-4.333005841344e-17,-1.001142881005e+00,1.000000000000e+00,1.000000000000e+00},
	/*  81 */ {-8.666011682666e-17,-1.001142881005e+00,1.000000000000e+00,1.000000000000e+00},
	/*  82 */ {-1.082951744618e-16,-9.987618749354e-01,1.000000000000e+00,1.000000000000e+00},
	/*  83 */ {9.744631731857e-17,-1.001142862502e+00,1.000000000000e+00,1.000000000000e+00},
	/*  84 */ {7.790402708405e-17,-1.001107865779e+00,1.000000000000e+00,1.000000000000e+00},
	/*  85 */ {-1.272705839326e-16,-9.999523595411e-01,1.000000000000e+00,1.000000000000e+00},
	/*  86 */ {-2.545299330200e-16,-9.999523958797e-01,1.000000000000e+00,1.000000000000e+00},
	/*  87 */ {-5.760170517504e-16,-9.996591651839e-01,1.000000000000e+00,1.000000000000e+00},
	/*  88 */ {7.315210301605e-17,-1.000026820472e+00,1.000000000000e+00,1.000000000000e+00},
	/*  89 */ {-5.680235886028e-17,-9.999895809352e-01,1.000000000000e+00,1.000000000000e+00},
	/*  90 */ {3.321483283975e-16,-1.000035519693e+00,1.000000000000e+00,1.000000000000e+00},
	/*  91 */ {-5.358904394133e-16,-9.999805355516e-01,1.000000000000e+00,1.000000000000e+00},
	/*  92 */ {-4.838378200173e-17,-9.999988818153e-01,1.000000000000e+00,1.000000000000e+00},
	/*  93 */ {-1.319406312557e-17,-1.000001120549e+00,1.000000000000e+00,1.000000000000e+00},
	/*  94 */ {-3.445621280512e-16,-9.999983349712e-01,1.000000000000e+00,1.000000000000e+00},
	/*  95 */ {7.081024003279e-17,-1.000000372008e+00,1.000000000000e+00,1.000000000000e+00},
	/*  96 */ {1.047515087426e-16,-1.000000309661e+00,1.000000000000e+00,1.000000000000e+00},
	/*  97 */ {-1.606744058225e-16,-1.000000065288e+00,1.000000000000e+00,1.000000000000e+00},
	/*  98 */ {-5.043194881009e-16,-9.999998742443e-01,3.211778193609e+00,1.000000000000e+00},
	/*  99 */ {-5.914717197419e-17,-9.999999653865e-01,-6.020094833141e+00,1.000000000000e+00},
	/* 100 */ {7.656414821489e-16,-1.000000148192e+00,5.283895405413e+00,1.000000000000e+00},
	/* 101 */ {6.745397168041e-17,-1.000000036227e+00,1.290802921139e+00,1.000000000000e+00},
	/* 102 */ {1.068553175417e-15,-1.000000071339e+00,1.091775259267e+00,1.000000000000e+00},
	/* 103 */ {2.155175718092e-14,-1.000000467839e+00,2.515132880623e+00,1.000000000000e+00},
	/* 104 */ {-2.107173038090e-15,-9.999999701670e-01,-8.198412100678e-03,1.000000000000e+00},
	/* 105 */ {2.499901148950e-14,-1.000000156703e+00,3.479972857276e-01,1.000000000000e+00},
	/* 106 */ {-2.301160748148e-14,-9.999999354679e-01,8.145633571008e-02,1.000000000000e+00},
	/* 107 */ {-2.748536870946e-13,-9.999995892377e-01,-2.073083264581e-02,1.000000000000e+00},
	/* 108 */ {1.570682364171e-13,-1.000000111241e+00,1.426049116120e-01,1.000000000000e+00},
	/* 109 */ {-7.514739347048e-13,-9.999996906112e-01,9.703110180785e-02,1.000000000000e+00},
	/* 110 */ {9.473497603852e-14,-1.000000016890e+00,1.256691908176e-01,1.000000000000e+00},
	/* 111 */ {1.781022286743e-12,-1.000000173018e+00,1.287658209287e-01,1.000000000000e+00},
	/* 112 */ {-8.719191185764e-14,-9.999999939048e-01,1.248631626665e-01,1.000396640271e+00},
	/* 113 */ {-4.910368041369e-12,-9.999998310334e-01,1.231347131030e-01,6.681281920614e+00},
	/* 114 */ {7.754842842296e-12,-1.000000133897e+00,1.257433863388e-01,-1.319837582319e+00},
	/* 115 */ {-3.921798375125e-11,-9.999996534356e-01,1.240167077290e-01,9.071896774085e-01},
	/* 116 */ {-3.806082978180e-11,-9.999998357403e-01,1.247728315489e-01,1.117816109636e-01},
	/* 117 */ {-4.149796177252e-11,-9.999998978911e-01,1.249215081565e-01,2.944511018893e-02},
	/* 118 */ {-2.781803387383e-10,-9.999996943784e-01,1.248922599624e-01,2.266664434768e-02},
	/* 119 */ {-9.909017836368e-11,-9.999999389518e-01,1.249876206841e-01,1.122493745681e-02},
	/* 120 */ {8.631974963866e-10,-1.000000238942e+00,1.250209529797e-01,9.860580125074e-03},
	/* 121 */ {-5.300352271629e-10,-9.999999034733e-01,1.249935675570e-01,1.060254678516e-02},
	/* 122 */ {2.657402990452e-09,-9.999999874263e-01,1.249908389219e-01,1.067300462220e-02},
	/* 123 */ {-1.457217488422e-07,-9.999933906198e-01,1.248906522889e-01,1.119511752297e-02},
	/* 124 */ {-2.578025667976e-06,-9.999418726660e-01,1.245239208849e-01,1.207715920581e-02},
	/* 125 */ {-5.125015277684e-05,-9.994293134520e-01,1.227115464087e-01,1.424072582164e-02},
	/* 126 */ {-1.379830566792e-03,-9.925329264495e-01,1.107229694671e-01,2.126097500684e-02},
	/* 127 */ {-1.864743370686e-01,-5.410928126904e-01,-2.525549828458e-01,1.181933200624e-01},
	/* 128 */ {-1.333333250335e+00,-8.565483047383e-08,2.824978502689e-08,-3.012237206711e-09},
	/* 129 */ {-1.333333540321e+00,1.111525017127e-07,-1.920410480904e-08,1.069861901114e-09},
	/* 130 */ {-1.333333221675e+00,-2.961633120628e-08,2.526754221249e-09,-6.942890497331e-11},
	/* 131 */ {-1.333333178783e+00,-2.076060857398e-08,8.926610057869e-10,-1.236120478060e-11},
	/* 132 */ {-1.333332964414e+00,-2.464513068630e-08,5.285258121644e-10,-3.653796655975e-12},
	/* 133 */ {-1.333333194209e+00,-4.631876236273e-09,4.931526478562e-11,-1.690869185959e-13},
	/* 134 */ {-1.333333178110e+00,-2.578657066978e-09,1.380502130945e-11,-2.384624466307e-14},
	/* 135 */ {-1.333333185228e+00,-1.242888897080e-09,3.337931583350e-12,-2.881897498592e-15},
	/* 136 */ {-1.333333579696e+00,1.025409303919e-09,-1.371794867687e-12,5.920247755778e-16},
	/* 137 */ {-1.333333161485e+00,-3.597820161867e-10,2.416390802897e-13,-5.224677798352e-17},
	/* 138 */ {-1.333333246360e+00,-9.881157884375e-11,3.549404193340e-14,-4.057583183548e-18},
	/* 139 */ {-1.333333269323e+00,-1.840348366522e-11,2.585826943029e-16,1.369743437665e-19},
	/* 140 */ {-1.333333248702e+00,-2.193347873298e-11,1.838592924749e-15,-4.992324871011e-20},
	/* 141 */ {-1.333333275266e+00,-7.371550857208e-12,3.045034090572e-16,-4.072865612961e-21},
	/* 142 */ {-1.333333346489e+00,5.272756690752e-13,-1.749029739154e-18,-5.770278284456e-23},
	/* 143 */ {-1.333333012783e+00,-1.051405392059e-11,1.105760126058e-16,-3.742014752457e-22},
	/* 144 */ {-1.333333073000e+00,-4.295090894217e-12,2.274254877935e-17,-3.878518481833e-23},
	/* 145 */ {-1.333333067559e+00,-2.175431407385e-12,5.718295828262e-18,-4.849357209383e-24},
	/* 146 */ {-1.333333494621e+00,6.524851151307e-13,-8.491610527547e-19,3.569930800568e-25},
	/* 147 */ {-1.333333486844e+00,3.154680722372e-13,-2.086057488680e-19,4.453185858003e-26},
	/* 148 */ {-1.333333369799e+00,1.085767578313e-14,3.471122025255e-21,-8.487990053450e-28},
	/* 149 */ {-1.333333364790e+00,1.716224770088e-14,-2.804461085609e-21,1.433071168328e-28},
	/* 150 */ {-1.333333632432e+00,7.698987738752e-14,-6.355915366934e-21,1.688527984608e-28},
	/* 151 */ {-1.333333293202e+00,-5.177734386698e-15,2.146750019752e-22,-2.830710781164e-30},
	/* 152 */ {-1.333333390721e+00,3.677858024510e-15,-7.625476062881e-23,5.098628074604e-31},
	/* 153 */ {-1.333333339788e+00,1.062797082465e-16,1.625227317215e-24,-1.869721718576e-32},
	/* 154 */ {-1.333333518560e+00,2.959493692538e-15,-1.516808165446e-23,2.505304810835e-32},
	/* 155 */ {-1.333333277163e+00,-4.825043823439e-16,1.320024312550e-24,-1.148750546884e-33},
	/* 156 */ {-1.333333126147e+00,-8.304243672564e-16,1.070882046646e-24,-4.450707317208e-34},
	/* 157 */ {-1.333333513595e+00,3.642769802863e-16,-2.368399123798e-25,4.963745743040e-35},
	/* 158 */ {-1.333333279624e+00,-5.491807238956e-17,1.801821983223e-26,-1.904365939054e-36},
	/* 159 */ {-1.333333163576e+00,-8.410464774398e-17,1.339697945699e-26,-6.882302496649e-37},
	/* 160 */ {-1.333333573927e+00,6.048826793261e-17,-4.872550040688e-27,1.263934890016e-37},
	/* 161 */ {-1.333333513310e+00,2.259478053470e-17,-9.114155499169e-28,1.184050219516e-38},
	/* 162 */ {-1.333333463726e+00,7.954360600539e-18,-1.557762201875e-28,9.855508511897e-40},
	/* 163 */ {-1.333333081894e+00,-7.782166151735e-18,7.735877932811e-29,-2.480984483132e-40},
	/* 164 */ {-1.333333362749e+00,3.133662987015e-19,-1.086482502490e-30,1.127971334651e-42},
	/* 165 */ {-1.333333181646e+00,-1.188443988607e-18,2.980115128575e-30,-2.405786069724e-42},
	/* 166 */ {-1.333333469173e+00,5.212344483575e-19,-6.417412556427e-31,2.547545767967e-43},
	/* 167 */ {-1.333333441437e+00,2.126300287424e-19,-1.336852548084e-31,2.705487971627e-44},
	/* 168 */ {-1.333333223779e+00,-1.051688538371e-19,3.237218330476e-32,-3.208267118141e-45},
	/* 169 */ {-1.333333426626e+00,4.489430127676e-20,-6.949398647029e-33,3.477443371349e-46},
	/* 170 */ {-1.333333445908e+00,2.751010810477e-20,-2.163090936539e-33,5.490816035931e-47},
	/* 171 */ {-1.333333328495e+00,-1.115325216755e-21,6.488595525187e-35,-1.018695499426e-48},
	/* 172 */ {-1.333333468192e+00,8.271651229993e-21,-1.636652610423e-34,1.044910729711e-48},
	/* 173 */ {-1.333333476287e+00,4.279922146670e-21,-4.116279459785e-35,1.278808223189e-49},
	/* 174 */ {-1.333333354231e+00,3.473845998397e-22,-1.856785180070e-36,3.192414722843e-51},
	/* 175 */ {-1.333333159805e+00,-1.321857285772e-21,3.241732393336e-36,-2.566596018474e-51},
	/* 176 */ {-1.333333406952e+00,2.936851967571e-22,-3.743020372784e-37,1.520553068232e-52},
	/* 177 */ {-1.333333569399e+00,4.522192564992e-22,-2.782954612095e-37,5.518547628464e-53},
	/* 178 */ {-1.333333302313e+00,-2.740503644050e-23,7.705591264764e-39,-7.079708452719e-55},
	/* 179 */ {-1.333333315224e+00,-3.971164528404e-24,-2.065457825099e-40,5.489136627352e-56},
	/* 180 */ {-1.333332906432e+00,-1.015557537341e-22,7.756093322742e-39,-1.908015729601e-55},
	/* 181 */ {-1.333333081343e+00,-3.018552968401e-23,1.162858960916e-39,-1.444352347945e-56},
	/* 182 */ {-1.333333463227e+00,7.692685065883e-24,-1.461439628281e-40,8.951375479841e-58},
	/* 183 */ {-1.333333405674e+00,2.123193041042e-24,-2.007724788594e-41,6.146728572377e-59},
	/* 184 */ {-1.333333356006e+00,6.554176163511e-26,2.941905420493e-43,-8.402271478600e-61},
	/* 185 */ {-1.333333460200e+00,9.404335673608e-25,-2.246356912986e-42,1.734685563731e-60},
	/* 186 */ {-1.333333414982e+00,3.014727460472e-25,-3.595635971699e-43,1.389012792231e-61},
	/* 187 */ {-1.333333167729e+00,-3.076697974714e-25,1.832451083604e-43,-3.513227106871e-62},
	/* 188 */ {-1.333332916851e+00,-3.871571125672e-25,1.157318816049e-43,-1.115564348455e-62},
	/* 189 */ {-1.333333300373e+00,-1.541782634316e-26,2.271090651741e-45,-1.091393517610e-64},
	/* 190 */ {-1.333333682258e+00,8.078564979792e-26,-6.011391030083e-45,1.442803916087e-64},
	/* 191 */ {-1.333333383336e+00,4.107459933020e-27,-8.142259967927e-47,1.614568941322e-67},
	/* 192 */ {-1.333333174547e+00,-9.236785854343e-27,1.720395942807e-46,-1.033239025252e-66},
	/* 193 */ {-1.333333158697e+00,-5.109386168397e-27,4.795024728031e-47,-1.448654336068e-67},
	/* 194 */ {-1.333333232266e+00,-1.445824503918e-27,6.635684826750e-48,-9.811404861057e-69},
	/* 195 */ {-1.333333220124e+00,-8.126746265684e-28,1.876529685275e-48,-1.399040491864e-69},
	/* 196 */ {-1.333333548652e+00,7.780883207615e-28,-9.025396956015e-49,3.375524076659e-70},
	/* 197 */ {-1.333333539111e+00,3.761657294365e-28,-2.206558286879e-49,4.167997033166e-71},
	/* 198 */ {-1.333333309058e+00,-1.774769508328e-29,3.881986439151e-51,-2.301838497525e-73},
	/* 199 */ {-1.333333314200e+00,-2.341040495519e-30,-4.416213441928e-52,5.352668524855e-74},
	/* 200 */ {-1.333333280783e+00,-1.072201976691e-29,6.824690665701e-52,-1.373826150447e-74},
	/* 201 */ {-1.333333513696e+00,2.073387023997e-29,-7.666057473491e-52,9.136859723976e-75},
	/* 202 */ {-1.333333495026e+00,9.317777255826e-30,-1.726025920772e-52,1.029199302590e-75},
	/* 203 */ {-1.333333581117e+00,7.031971986589e-30,-6.406945981455e-53,1.880714530854e-76},
	/* 204 */ {-1.333333401823e+00,9.964767173825e-31,-4.635238529828e-54,6.935131446495e-78},
	/* 205 */ {-1.333333885666e+00,3.904522189524e-30,-8.864756252001e-54,6.489135316917e-78},
	/* 206 */ {-1.333333395061e+00,1.254667019346e-31,-4.695480278120e-56,-1.401291531088e-80},
	/* 207 */ {-1.333333030847e+00,-5.375956263849e-31,3.069466789112e-55,-5.643691431013e-80},
	/* 208 */ {-1.333333327166e+00,1.919835509847e-33,-2.472520948105e-57,3.880917228092e-82},
	/* 209 */ {-1.333333294021e+00,-1.658985332084e-32,2.230998904875e-57,-9.415070576108e-83},
	/* 210 */ {-1.333333361679e+00,6.358095729257e-33,-4.570269710415e-58,1.055780541129e-83},
	/* 211 */ {-1.333333500412e+00,1.845864462817e-32,-6.543426944569e-58,7.473526031092e-84},
	/* 212 */ {-1.333333286627e+00,-2.834868219146e-33,5.500216762870e-59,-3.404841129697e-85},
	/* 213 */ {-1.333333545730e+00,5.946596755806e-33,-5.343449166573e-59,1.544846313888e-85},
	/* 214 */ {-1.333333545150e+00,2.899623025988e-33,-1.274981363419e-59,1.808621015228e-86},
	/* 215 */ {-1.333333605636e+00,1.897318502316e-33,-4.239919649531e-60,3.052223672120e-87},
	/* 216 */ {-1.333333523495e+00,6.529697929759e-34,-7.185656200683e-61,2.546695192938e-88},
	/* 217 */ {-1.333333477348e+00,2.450668124232e-34,-1.341684078631e-61,2.370855346389e-89},
	/* 218 */ {-1.333333193052e+00,-1.217309352390e-34,3.402577770924e-62,-3.072692016640e-90},
	/* 219 */ {-1.333333339392e+00,7.908045653855e-36,-1.838949931058e-63,1.133340902040e-91},
	/* 220 */ {-1.333333163813e+00,-3.656572613654e-35,2.540160208825e-63,-5.697415173470e-92},
	/* 221 */ {-1.333333325895e+00,-8.461523994936e-37,2.460003106521e-65,-1.828509092437e-94},
	/* 222 */ {-1.333333308247e+00,-1.021327791009e-36,1.243643259428e-65,-5.108189895746e-95},
	/* 223 */ {-1.333333605387e+00,7.318244746107e-36,-6.309078864111e-65,1.751858971001e-94},
	/* 224 */ {-1.333333195422e+00,-1.852406740141e-36,7.950562249551e-66,-1.096366968338e-95},
	/* 225 */ {-1.333333294841e+00,-2.610347469364e-37,5.966345057630e-67,-4.498314354059e-97},
	/* 226 */ {-1.333333524875e+00,6.515416617171e-37,-7.110263355564e-67,2.499155694999e-97},
	/* 227 */ {-1.333333482063e+00,2.474651700275e-37,-1.323415374505e-67,2.287458779407e-98},
	/* 228 */ {-1.333333577470e+00,2.062150338207e-37,-5.591850689061e-68,4.887095378501e-99},
	/* 229 */ {-1.333333184886e+00,-6.323456781108e-38,8.640159727147e-69,-3.801349760595e-100},
	/* 230 */ {-1.333333395134e+00,1.253471590393e-38,-8.090815159581e-70,1.670722975841e-101},
	/* 231 */ {-1.333333207156e+00,-1.331252491037e-38,4.505631127676e-70,-4.914453594789e-102},
	/* 232 */ {-1.333333245652e+00,-4.563875926943e-39,7.595363459349e-71,-4.062772598731e-103},
	/* 233 */ {-1.333333302055e+00,-8.877934778420e-40,8.099944141362e-72,-2.371084759745e-104},
	/* 234 */ {-1.333333189701e+00,-1.914983812459e-39,8.191087891426e-72,-1.127996844709e-104},
	/* 235 */ {-1.333333237420e+00,-6.308954602504e-40,1.338934233779e-72,-9.169914290398e-106},
	/* 236 */ {-1.333333523342e+00,6.312185287525e-40,-6.731726131512e-73,2.311247545053e-106},
	/* 237 */ {-1.333333202754e+00,-2.128634999570e-40,1.117869911204e-73,-1.896033352413e-107},
	/* 238 */ {-1.333333272774e+00,-4.917537468076e-41,1.298433284411e-74,-1.112714216448e-108},
	/* 239 */ {-1.333333211901e+00,-5.024674207066e-41,6.682770959075e-75,-2.864381967640e-109},
	/* 240 */ {-1.333333143734e+00,-3.887408861238e-41,2.556959799296e-75,-5.419421025130e-110},
	/* 241 */ {-1.333333305617e+00,-2.849685061098e-42,9.479868688356e-77,-1.011386606835e-111},
	/* 242 */ {-1.333333251860e+00,-3.953692862608e-42,6.333410196389e-77,-3.315329272212e-112},
	/* 243 */ {-1.333332977860e+00,-9.214200536708e-42,7.650452832164e-77,-2.042817958937e-112},
	/* 244 */ {-1.333333292332e+00,-2.609821050270e-43,1.733794685545e-79,7.478955100301e-115},
	/* 245 */ {-1.333333590248e+00,1.654327092587e-42,-3.415797502847e-78,2.272572466733e-114},
	/* 246 */ {-1.333333312543e+00,-6.361628807155e-45,-3.670836706573e-80,2.216616377711e-116},
	/* 247 */ {-1.333333356401e+00,2.347061971336e-44,-5.302200364329e-81,-1.355743215953e-118},
	/* 248 */ {-1.333332987450e+00,-2.764122616025e-43,7.082154648612e-80,-5.850171201763e-117},
	/* 249 */ {-1.333333518196e+00,7.455251794245e-44,-9.655955167332e-81,4.031772620559e-118},
	/* 250 */ {-1.333333521104e+00,3.748559811449e-44,-2.406769566015e-81,4.986725054466e-119},
	/* 251 */ {-1.333333160733e+00,-1.714856148542e-44,5.480067496428e-82,-5.655412448161e-120},
	/* 252 */ {-1.333333408861e+00,3.716869205815e-45,-6.022777911842e-83,3.193295929403e-121},
	/* 253 */ {-1.333333315591e+00,-1.824608898865e-46,-3.703588286582e-85,4.562105304730e-123},
	/* 254 */ {1.570796326795e+00,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/* 255 */ {1.570796326795e+00,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
};

float atan15 (float x)
{
	static ptbl_t * tbl[2] = {&atantbl, &one5tbl};

	union { float f; uint32 i; } u;
	u.f = x;
	int e = (u.i & 0x7F800000) >> 23;
	ptbl_t *pt = tbl[u.i >> 31];
	float *p = (*pt)[e];
	x = fabs (x);
	x = p[0]+x*(p[1]+x*(p[2]+x*p[3]));
	return x;
}


float one5 (float x)
{
	union { float f; uint32 i; } u;
	u.f = x;
	int e = (u.i & 0x7F800000) >> 23;
	float *p = one5tbl[e];
	x = fabs (x);
	x = p[0]+x*(p[1]+x*(p[2]+x*p[3]));
	u.i &= 0x80000000;
	u.i |= 127 << 23; /* 1.0 - use 126 for .5 instead, etc */
	return x * u.f;
}

float sintbl[][4] = {
	/*   0 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   1 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   2 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   3 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   4 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   5 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   6 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   7 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   8 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   9 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  10 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  11 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  12 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  13 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  14 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  15 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  16 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  17 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  18 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  19 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  20 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  21 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  22 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  23 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  24 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  25 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  26 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  27 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  28 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  29 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  30 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  31 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  32 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  33 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  34 */ {4.758711902881e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  35 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  36 */ {4.758711902881e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  37 */ {4.759115799664e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  38 */ {4.759519696448e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  39 */ {4.761135283582e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  40 */ {4.763962561066e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  41 */ {4.769617116035e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  42 */ {4.780522329188e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  43 */ {4.802736652279e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  44 */ {4.846761401678e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  45 */ {4.935214797259e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  46 */ {5.112525485203e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  47 */ {5.466742964309e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  48 */ {6.174774025738e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  49 */ {7.591240045378e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  50 */ {1.042417208466e-23,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  51 */ {1.609003616322e-23,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  52 */ {2.742176432034e-23,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  53 */ {5.008481673781e-23,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  54 */ {9.541132546951e-23,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  55 */ {1.860643429329e-22,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  56 */ {3.673707817565e-22,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  57 */ {7.299832555070e-22,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  58 */ {1.455208203008e-21,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  59 */ {-1.563910088637e-29,1.570796329975e+00,1.000000000000e+00,1.000000000000e+00},
	/*  60 */ {-5.707218298877e-29,1.570796334697e+00,1.000000000000e+00,1.000000000000e+00},
	/*  61 */ {-1.293237432142e-29,1.570796327394e+00,1.000000000000e+00,1.000000000000e+00},
	/*  62 */ {3.407308104185e-28,1.570796320617e+00,1.000000000000e+00,1.000000000000e+00},
	/*  63 */ {2.041716292920e-27,1.570796304854e+00,1.000000000000e+00,1.000000000000e+00},
	/*  64 */ {7.278991505894e-28,1.570796321567e+00,1.000000000000e+00,1.000000000000e+00},
	/*  65 */ {-2.700063504065e-32,1.570796326795e+00,1.000000000000e+00,1.000000000000e+00},
	/*  66 */ {-1.229961912617e-27,1.570796328576e+00,1.000000000000e+00,1.000000000000e+00},
	/*  67 */ {-8.825012408449e-27,1.570796331015e+00,1.000000000000e+00,1.000000000000e+00},
	/*  68 */ {3.690684881417e-27,1.570796325459e+00,1.000000000000e+00,1.000000000000e+00},
	/*  69 */ {-1.171304235795e-26,1.570796328915e+00,1.000000000000e+00,1.000000000000e+00},
	/*  70 */ {-7.692079887003e-26,1.570796331629e+00,1.000000000000e+00,1.000000000000e+00},
	/*  71 */ {-2.141841575883e-26,1.570796327764e+00,1.000000000000e+00,1.000000000000e+00},
	/*  72 */ {3.489068347284e-25,1.570796320617e+00,1.000000000000e+00,1.000000000000e+00},
	/*  73 */ {-2.280393991830e-25,1.570796329374e+00,1.000000000000e+00,1.000000000000e+00},
	/*  74 */ {-1.743744921150e-25,1.570796327781e+00,1.000000000000e+00,1.000000000000e+00},
	/*  75 */ {-2.811648342704e-24,1.570796340206e+00,1.000000000000e+00,1.000000000000e+00},
	/*  76 */ {8.459685135478e-25,1.570796325271e+00,1.000000000000e+00,1.000000000000e+00},
	/*  77 */ {6.732563310105e-24,1.570796321080e+00,1.000000000000e+00,1.000000000000e+00},
	/*  78 */ {-1.978705836396e-23,1.570796331669e+00,1.000000000000e+00,1.000000000000e+00},
	/*  79 */ {-3.127115916058e-23,1.570796335864e+00,1.000000000000e+00,1.000000000000e+00},
	/*  80 */ {1.243501226559e-23,1.570796325696e+00,1.000000000000e+00,1.000000000000e+00},
	/*  81 */ {3.719993719899e-23,1.570796325151e+00,1.000000000000e+00,1.000000000000e+00},
	/*  82 */ {2.376329732946e-22,1.570796323586e+00,1.000000000000e+00,1.000000000000e+00},
	/*  83 */ {5.776941160887e-22,1.570796322582e+00,1.000000000000e+00,1.000000000000e+00},
	/*  84 */ {1.621995197153e-21,1.570796317836e+00,1.000000000000e+00,1.000000000000e+00},
	/*  85 */ {9.440915986436e-22,1.570796323994e+00,1.000000000000e+00,1.000000000000e+00},
	/*  86 */ {1.551492753323e-21,1.570796324652e+00,1.000000000000e+00,1.000000000000e+00},
	/*  87 */ {-1.542339644416e-21,1.570796327858e+00,1.000000000000e+00,1.000000000000e+00},
	/*  88 */ {4.649785888114e-21,1.570796325187e+00,1.000000000000e+00,1.000000000000e+00},
	/*  89 */ {-1.806978540849e-20,1.570796330271e+00,1.000000000000e+00,1.000000000000e+00},
	/*  90 */ {6.291516352033e-21,1.570796326240e+00,1.000000000000e+00,1.000000000000e+00},
	/*  91 */ {-1.354446356118e-20,1.570796327355e+00,1.000000000000e+00,1.000000000000e+00},
	/*  92 */ {1.973752071543e-19,1.570796321452e+00,1.000000000000e+00,1.000000000000e+00},
	/*  93 */ {-9.196857067950e-20,1.570796327690e+00,1.000000000000e+00,1.000000000000e+00},
	/*  94 */ {-1.701682819794e-19,1.570796327531e+00,1.000000000000e+00,1.000000000000e+00},
	/*  95 */ {1.698291650065e-18,1.570796321825e+00,1.000000000000e+00,1.000000000000e+00},
	/*  96 */ {2.923584459106e-18,1.570796321428e+00,1.000000000000e+00,1.000000000000e+00},
	/*  97 */ {-2.529901162371e-18,1.570796326997e+00,1.000000000000e+00,1.000000000000e+00},
	/*  98 */ {7.338942370259e-18,1.570796321055e+00,1.043122527316e+00,1.000000000000e+00},
	/*  99 */ {2.666315986850e-17,1.570796321288e+00,1.590911046606e-01,1.000000000000e+00},
	/* 100 */ {1.171934963772e-16,1.570796304443e+00,1.000000053837e+00,1.000000000000e+00},
	/* 101 */ {2.737724169374e-16,1.570796300179e+00,6.180225603937e-01,1.000000000000e+00},
	/* 102 */ {4.589778554308e-16,1.570796301893e+00,2.805434494752e-01,1.000000000000e+00},
	/* 103 */ {7.333699773295e-15,1.570796153658e+00,9.682530490813e-01,1.000000000000e+00},
	/* 104 */ {9.796638580358e-15,1.570796210897e+00,3.240093903940e-01,1.000000000000e+00},
	/* 105 */ {-2.136524793955e-15,1.570796336916e+00,-8.013391361560e-03,1.000000000000e+00},
	/* 106 */ {-1.792665655513e-14,1.570796380957e+00,-3.833921888963e-02,1.000000000000e+00},
	/* 107 */ {4.585271732686e-14,1.570796259353e+00,2.356907759556e-02,1.000000000000e+00},
	/* 108 */ {-9.316646985782e-14,1.570796398132e+00,-1.279287227247e-02,1.000000000000e+00},
	/* 109 */ {4.347160896000e-13,1.570796164873e+00,1.413834734420e-02,1.000000000000e+00},
	/* 110 */ {3.073513133864e-13,1.570796270491e+00,2.435665403308e-03,1.000000000000e+00},
	/* 111 */ {-5.817034508482e-13,1.570796380811e+00,-1.223756765608e-03,1.000000000000e+00},
	/* 112 */ {-2.082564090431e-12,1.570796428118e+00,-1.205145101669e-03,1.801097822025e-02},
	/* 113 */ {-2.587117105632e-12,1.570796392635e+00,-5.736216415477e-04,5.147527326370e-01},
	/* 114 */ {-1.066794831791e-11,1.570796519823e+00,-1.110760374836e-03,1.401277894217e+00},
	/* 115 */ {-1.441209219481e-10,1.570797593144e+00,-3.569101929873e-03,2.594425029761e+00},
	/* 116 */ {-2.220809492385e-10,1.570797305936e+00,-1.387348943101e-03,-1.259714626877e-02},
	/* 117 */ {-2.892373811480e-10,1.570796962387e+00,-4.481483482810e-04,-5.441656466535e-01},
	/* 118 */ {1.061746210843e-09,1.570795172571e+00,4.029311575389e-04,-6.913127628055e-01},
	/* 119 */ {9.421095055502e-10,1.570795811227e+00,9.058407335105e-05,-6.510813453009e-01},
	/* 120 */ {2.823863607161e-09,1.570795556673e+00,6.702618674769e-05,-6.478246610518e-01},
	/* 121 */ {-3.000526538381e-09,1.570796808200e+00,-2.658450169992e-05,-6.454302976274e-01},
	/* 122 */ {-6.329481758134e-08,1.570801761845e+00,-1.620463519343e-04,-6.441922448135e-01},
	/* 123 */ {-2.021268766714e-06,1.570883120348e+00,-1.294120239636e-03,-6.388869784623e-01},
	/* 124 */ {-6.416202795031e-05,1.572174548920e+00,-1.028213968552e-02,-6.178181828219e-01},
	/* 125 */ {-1.987720249735e-03,1.592186442677e+00,-8.001977813081e-02,-5.359502029548e-01},
	/* 126 */ {-5.567780016233e-02,1.872915648625e+00,-5.721480952473e-01,-2.450897532157e-01},
	/* 127 */ {1.000000095682e+00,-2.085920724763e-07,1.462415963824e-07,-3.298948742713e-08},
	/* 128 */ {9.999998414954e-01,1.704867192286e-07,-5.879433680231e-08,6.527213702327e-09},
	/* 129 */ {9.999999465195e-01,2.882678144132e-08,-4.946723915021e-09,2.726387258461e-10},
	/* 130 */ {9.999999550595e-01,1.275250531829e-08,-1.168540616645e-09,3.453109727956e-11},
	/* 131 */ {1.000000108439e+00,-1.434744484993e-08,6.100927401123e-10,-8.366281170084e-12},
	/* 132 */ {1.000000051187e+00,-3.359184349398e-09,7.089076660832e-11,-4.812208960150e-13},
	/* 133 */ {9.999999540160e-01,1.527039088564e-09,-1.620138119628e-11,5.546019357137e-14},
	/* 134 */ {9.999998425154e-01,2.647193018433e-09,-1.428122112754e-11,2.484023281132e-14},
	/* 135 */ {9.999997588466e-01,2.004506984040e-09,-5.355592315085e-12,4.618677334033e-15},
	/* 136 */ {9.999998703926e-01,5.380816718342e-10,-7.158038190356e-13,3.063418548339e-16},
	/* 137 */ {9.999999322365e-01,1.407624043552e-10,-9.381596139353e-14,2.019842937195e-17},
	/* 138 */ {1.000000104817e+00,-1.115334091805e-10,3.800024588958e-14,-4.158984339950e-18},
	/* 139 */ {1.000000149830e+00,-7.838040298066e-11,1.317204113087e-14,-7.131425448685e-19},
	/* 140 */ {9.999999243002e-01,1.914451920920e-11,-1.545560239119e-15,4.000075234813e-20},
	/* 141 */ {1.000000119732e+00,-1.584039661931e-11,6.719685232883e-16,-9.168924074923e-21},
	/* 142 */ {1.000000010590e+00,-2.356461077357e-13,-5.366878990479e-18,1.061985310001e-22},
	/* 143 */ {1.000000101231e+00,-3.358023900072e-12,3.578399950070e-17,-1.228145179424e-22},
	/* 144 */ {9.999998867430e-01,1.850740978496e-12,-9.697165913489e-18,1.638193903749e-23},
	/* 145 */ {1.000000132697e+00,-1.083106542073e-12,2.830069576593e-18,-2.380364700083e-24},
	/* 146 */ {9.999999125972e-01,3.556491075171e-13,-4.618567755443e-19,1.922456134396e-25},
	/* 147 */ {1.000000000061e+00,-2.951085523368e-15,1.361181513922e-21,-4.076889729614e-28},
	/* 148 */ {1.000000076882e+00,-8.010717903743e-14,2.676768129432e-20,-2.876274392283e-27},
	/* 149 */ {1.000000095853e+00,-4.797935252548e-14,7.668981180237e-21,-3.945859720009e-28},
	/* 150 */ {9.999998458240e-01,3.951461383815e-14,-3.249551765188e-21,8.609234350037e-29},
	/* 151 */ {9.999999278577e-01,9.084480189355e-15,-3.689560570269e-22,4.837109174782e-30},
	/* 152 */ {1.000000143249e+00,-9.242860173679e-15,1.914368566106e-22,-1.275550322420e-30},
	/* 153 */ {9.999997865343e-01,6.787014142199e-15,-6.915208529931e-23,2.269026147243e-31},
	/* 154 */ {9.999998883099e-01,1.782717279343e-15,-9.124459490332e-24,1.504115333334e-32},
	/* 155 */ {1.000000067111e+00,-5.436405530794e-16,1.441506617448e-24,-1.245521037555e-33},
	/* 156 */ {1.000000161842e+00,-6.450736589978e-16,8.249789497421e-25,-3.397136212992e-34},
	/* 157 */ {9.999999572586e-01,8.527830862930e-17,-5.428353479022e-26,1.111122030936e-35},
	/* 158 */ {1.000000117366e+00,-1.187782241028e-16,3.861092144532e-26,-4.038741459196e-36},
	/* 159 */ {1.000000037939e+00,-1.985179002383e-17,3.295922848237e-27,-1.741114607518e-37},
	/* 160 */ {9.999999896454e-01,2.475017314656e-18,-1.866097509345e-28,4.496920989427e-39},
	/* 161 */ {9.999998014609e-01,2.458038311656e-17,-9.761266072237e-28,1.249552980487e-38},
	/* 162 */ {1.000000054392e+00,-3.396470808080e-18,6.807375293267e-29,-4.396543509624e-40},
	/* 163 */ {1.000000018253e+00,-6.966872094171e-19,8.074539972796e-30,-2.880409629428e-41},
	/* 164 */ {9.999999905302e-01,1.765526141440e-19,-9.859033646684e-31,1.717114709313e-42},
	/* 165 */ {1.000000114140e+00,-8.845247358301e-19,2.197985266073e-30,-1.757316976910e-42},
	/* 166 */ {1.000000213321e+00,-8.339061815931e-19,1.047402352079e-30,-4.239457424654e-43},
	/* 167 */ {1.000000006518e+00,-1.980750637671e-20,1.690368610192e-32,-4.286082107818e-45},
	/* 168 */ {9.999999285595e-01,6.911315299970e-20,-2.138710680698e-32,2.131249037154e-45},
	/* 169 */ {1.000000125192e+00,-6.121178796577e-20,9.617668455593e-33,-4.871037857530e-46},
	/* 170 */ {9.999999056122e-01,2.320004389134e-20,-1.828613201106e-33,4.641531149487e-47},
	/* 171 */ {9.999999804926e-01,2.215852802313e-21,-8.104547094829e-35,9.597918743804e-49},
	/* 172 */ {1.000000043277e+00,-2.232615191972e-21,3.675858001022e-35,-1.935821319545e-49},
	/* 173 */ {9.999998379495e-01,4.903182262480e-21,-4.760451993802e-35,1.490717800731e-49},
	/* 174 */ {1.000000113640e+00,-1.724254943830e-21,8.377391887291e-36,-1.311460758967e-50},
	/* 175 */ {9.999998258058e-01,1.319058820319e-21,-3.200092629014e-36,2.500573391753e-51},
	/* 176 */ {9.999999508260e-01,1.850256975813e-22,-2.233027634605e-37,8.699059850511e-53},
	/* 177 */ {1.000000135364e+00,-2.605503245704e-22,1.610072820301e-37,-3.199126124546e-53},
	/* 178 */ {1.000000074412e+00,-7.036405049848e-23,2.130752649339e-38,-2.074539295067e-54},
	/* 179 */ {9.999998992778e-01,4.855097578515e-23,-7.503336923122e-39,3.734825889942e-55},
	/* 180 */ {9.999999917724e-01,1.442739134010e-24,-5.806719124540e-41,-3.781803913518e-59},
	/* 181 */ {9.999998831770e-01,1.382115080098e-23,-5.245876941414e-40,6.414204306753e-57},
	/* 182 */ {9.999997729614e-01,1.345336025384e-23,-2.557214357754e-40,1.565854913741e-57},
	/* 183 */ {1.000000057138e+00,-1.683241633463e-24,1.580454863448e-41,-4.734314341302e-59},
	/* 184 */ {1.000000065953e+00,-9.996751784855e-25,4.853713247015e-42,-7.572578309191e-60},
	/* 185 */ {1.000000050696e+00,-3.741324320814e-25,8.893357157927e-43,-6.815715674325e-61},
	/* 186 */ {9.999998424798e-01,5.837470062265e-25,-6.943194261073e-43,2.661352534011e-61},
	/* 187 */ {1.000000079020e+00,-1.482713433291e-25,8.956690238398e-44,-1.747418247667e-62},
	/* 188 */ {9.999999766948e-01,2.326943865410e-26,-7.101520437953e-45,6.859449075325e-64},
	/* 189 */ {1.000000077593e+00,-3.415683033757e-26,4.866250743329e-45,-2.247215762902e-64},
	/* 190 */ {9.999998908052e-01,2.536017164692e-26,-1.896516176117e-45,4.574986132627e-65},
	/* 191 */ {9.999999321465e-01,8.216846544075e-27,-3.165588987332e-46,3.905182670884e-66},
	/* 192 */ {9.999999280352e-01,4.137113939608e-27,-7.599036740329e-47,4.493731540689e-67},
	/* 193 */ {9.999999674187e-01,9.091083446181e-28,-8.027325363436e-48,2.260896935255e-68},
	/* 194 */ {9.999997443597e-01,3.698597465688e-27,-1.716968673750e-47,2.568851715358e-68},
	/* 195 */ {1.000000092805e+00,-6.580218773578e-28,1.504601594149e-48,-1.108362173634e-69},
	/* 196 */ {1.000000049030e+00,-1.798219238164e-28,2.102207731361e-49,-7.890301397391e-71},
	/* 197 */ {1.000000170703e+00,-3.084736542236e-28,1.792064029653e-49,-3.357657787017e-71},
	/* 198 */ {9.999998285442e-01,1.560408495912e-28,-4.557923090849e-50,4.289688601356e-72},
	/* 199 */ {1.000000194726e+00,-8.815156872694e-29,1.281572925316e-50,-6.005787768649e-73},
	/* 200 */ {1.000000078594e+00,-1.762133515562e-29,1.270420984138e-51,-2.953435836177e-74},
	/* 201 */ {9.999996360571e-01,4.112814580462e-29,-1.491114830432e-51,1.742106389998e-74},
	/* 202 */ {9.999998517067e-01,8.412746302150e-30,-1.531318551863e-52,8.978306223308e-76},
	/* 203 */ {1.000000117394e+00,-3.326967922774e-30,3.019824882531e-53,-8.818961722772e-77},
	/* 204 */ {9.999996394824e-01,5.098646445195e-30,-2.317447007715e-53,3.395651082255e-77},
	/* 205 */ {1.000000029531e+00,-2.041704866793e-31,4.533813330439e-55,-3.231024629124e-79},
	/* 206 */ {1.000000152655e+00,-5.393777697551e-31,6.110188657097e-55,-2.230317294137e-79},
	/* 207 */ {9.999999179505e-01,1.446097504969e-31,-8.142127769280e-56,1.472907829760e-80},
	/* 208 */ {1.000000102190e+00,-9.197471639490e-32,2.653262445123e-56,-2.460017978867e-81},
	/* 209 */ {1.000000057494e+00,-2.591933677676e-32,3.764513776930e-57,-1.764642419651e-82},
	/* 210 */ {1.000000115991e+00,-2.536921025788e-32,1.781023171223e-57,-4.035428522656e-83},
	/* 211 */ {9.999999958354e-01,9.157617302051e-34,-4.686702780738e-59,6.807103654408e-85},
	/* 212 */ {9.999998938664e-01,5.815873038410e-33,-1.018596195759e-58,5.741423559967e-85},
	/* 213 */ {9.999999050071e-01,2.618789028206e-33,-2.324654480479e-59,6.680457257886e-86},
	/* 214 */ {9.999999069206e-01,1.270264795048e-33,-5.557448833245e-60,7.836344101423e-87},
	/* 215 */ {9.999998385627e-01,1.110476291648e-33,-2.448341656945e-60,1.737872351923e-87},
	/* 216 */ {1.000000101811e+00,-3.519065922491e-34,3.916699858925e-61,-1.406364382855e-88},
	/* 217 */ {9.999998607297e-01,2.410089356671e-34,-1.337957219686e-61,2.392770147670e-89},
	/* 218 */ {9.999998528181e-01,1.275505372857e-34,-3.544665559974e-62,3.170912834694e-90},
	/* 219 */ {9.999999239838e-01,3.233723972509e-35,-4.397547565138e-63,1.925907869558e-91},
	/* 220 */ {9.999996118971e-01,8.411281157375e-35,-5.848446347542e-63,1.309774394240e-91},
	/* 221 */ {9.999999605271e-01,4.141519890754e-36,-1.396094428160e-64,1.521729581197e-93},
	/* 222 */ {1.000000072590e+00,-3.862168038072e-36,6.636961049939e-65,-3.690761308951e-94},
	/* 223 */ {1.000000146083e+00,-3.921518551525e-36,3.385252907905e-65,-9.431665222219e-95},
	/* 224 */ {1.000000087269e+00,-1.160682802436e-36,4.981163300382e-66,-6.920188218309e-96},
	/* 225 */ {9.999999214892e-01,5.408438281848e-37,-1.181400175870e-66,8.267703272589e-97},
	/* 226 */ {1.000000124853e+00,-4.239099486916e-37,4.611278517919e-67,-1.611951354626e-97},
	/* 227 */ {9.999995882150e-01,6.963139070614e-37,-3.776724553831e-67,6.598844912505e-98},
	/* 228 */ {9.999998732078e-01,1.069649664032e-37,-2.887963102910e-68,2.508290312775e-99},
	/* 229 */ {1.000000024157e+00,-1.038237453211e-38,1.405066819055e-69,-6.056931032608e-101},
	/* 230 */ {9.999999348467e-01,1.373980743256e-38,-9.271256580808e-70,2.012780046963e-101},
	/* 231 */ {1.000000305594e+00,-3.240110361875e-38,1.102789746590e-69,-1.209070774866e-101},
	/* 232 */ {9.999998809930e-01,6.211990743146e-39,-1.043020781466e-70,5.661208606020e-103},
	/* 233 */ {9.999999286321e-01,1.948798284698e-39,-1.692204900421e-71,4.710452322140e-104},
	/* 234 */ {1.000000080744e+00,-1.094066256495e-39,4.748854612732e-72,-6.615065097876e-105},
	/* 235 */ {1.000000008913e+00,-3.798630914335e-41,4.001563973018e-74,-2.855494004748e-108},
	/* 236 */ {9.999998494791e-01,4.970288268048e-40,-5.251584279346e-73,1.786142057392e-106},
	/* 237 */ {1.000000029772e+00,-4.576400938828e-41,2.213564896335e-74,-3.435610283177e-108},
	/* 238 */ {1.000000310188e+00,-2.544669953312e-40,6.696405447292e-74,-5.677766704986e-108},
	/* 239 */ {1.000000170774e+00,-7.019588359451e-41,9.258212582303e-75,-3.936043785877e-109},
	/* 240 */ {9.999999752089e-01,5.277141296816e-42,-3.539545456101e-76,7.557372978604e-111},
	/* 241 */ {1.000000030124e+00,-3.260111957658e-42,1.136719538963e-76,-1.267728199750e-111},
	/* 242 */ {1.000000065375e+00,-3.402115087521e-42,5.680458530977e-77,-3.039329917369e-112},
	/* 243 */ {1.000000032932e+00,-8.742542345987e-43,7.420341710824e-78,-2.006984069535e-113},
	/* 244 */ {1.000000063193e+00,-7.863191748751e-43,3.142737510953e-78,-4.043591433283e-114},
	/* 245 */ {9.999998737472e-01,8.221443583807e-43,-1.720355117727e-78,1.160771254620e-114},
	/* 246 */ {1.000000058226e+00,-1.861684218988e-43,1.900852677889e-79,-6.233034741038e-116},
	/* 247 */ {1.000000183002e+00,-2.946237450692e-43,1.521800031387e-79,-2.532481282972e-116},
	/* 248 */ {9.999998186643e-01,1.464788377915e-43,-3.793002223441e-80,3.161791286655e-117},
	/* 249 */ {1.000000152235e+00,-6.151206918663e-44,7.977009067619e-81,-3.330676668599e-118},
};

float sin1 (float x)
{
	union { float f; uint32 i; } u;
	u.f = -x;
	int e = (u.i & 0x7F800000) >> 23;
	float *p = sintbl[e];
	x = fabs (x);
	x = p[0]+x*(p[1]+x*(p[2]+x*p[3]));
	u.i &= 0x80000000;
	u.i |= 127 << 23; /* 1.0 - use 126 for .5 instead, etc */
	return x * u.f;
}

float clip3tbl[][4] = {
	/*   0 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   1 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   2 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   3 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   4 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   5 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   6 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   7 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   8 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   9 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  10 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  11 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  12 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  13 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  14 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  15 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  16 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  17 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  18 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  19 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  20 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  21 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  22 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  23 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  24 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  25 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  26 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  27 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  28 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  29 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  30 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  31 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  32 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  33 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  34 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  35 */ {4.757904109314e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  36 */ {4.757500212530e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  37 */ {4.757096315747e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  38 */ {4.755884625396e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  39 */ {4.753461244696e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  40 */ {4.748614483294e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  41 */ {4.738920960491e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  42 */ {4.719533914884e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  43 */ {4.680759823670e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  44 */ {4.603211641244e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  45 */ {4.448115276390e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  46 */ {4.137922546683e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  47 */ {3.517537087268e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  48 */ {2.276766168438e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  49 */ {-2.047756692209e-25,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  50 */ {-5.167859344539e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  51 */ {-1.509402669518e-23,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  52 */ {-3.494636139645e-23,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  53 */ {-7.465103079899e-23,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  54 */ {-1.540603696041e-22,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  55 */ {-3.128790472143e-22,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  56 */ {-6.305164024346e-22,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  57 */ {-1.265791112875e-21,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  58 */ {-2.536340533757e-21,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  59 */ {-7.034381313986e-39,1.377425589194e-18,1.000000000000e+00,1.000000000000e+00},
	/*  60 */ {9.465525899283e-41,-2.033122502002e-20,1.000000000000e+00,1.000000000000e+00},
	/*  61 */ {5.530942284335e-40,-5.798412628578e-20,1.425925932565e+00,1.000000000000e+00},
	/*  62 */ {-4.312598204865e-59,-6.014750754107e-36,2.764743250389e-17,1.000000000000e+00},
	/*  63 */ {7.376799804131e-39,-1.680240410675e-19,1.000000000000e+00,1.000000000000e+00},
	/*  64 */ {1.271828537329e-32,-9.338226169131e-14,1.000000000000e+00,1.000000000000e+00},
	/*  65 */ {9.952817631163e-38,-6.517453240027e-19,1.000000000000e+00,1.000000000000e+00},
	/*  66 */ {3.970559283422e-37,-1.300867671208e-18,9.999999998929e-01,1.000000000000e+00},
	/*  67 */ {1.625839198223e-36,-2.558784641806e-18,1.000000000000e+00,1.000000000000e+00},
	/*  68 */ {6.503354736646e-36,-5.225890685089e-18,1.000000000000e+00,1.000000000000e+00},
	/*  69 */ {2.529872861266e-35,-1.038664407722e-17,1.000000000000e+00,1.000000000000e+00},
	/*  70 */ {1.016392504791e-34,-2.081600372914e-17,1.000000000000e+00,1.000000000000e+00},
	/*  71 */ {4.065852103997e-34,-4.163336044713e-17,1.000000000000e+00,1.000000000000e+00},
	/*  72 */ {1.626340735517e-33,-8.326672168953e-17,1.000000000000e+00,1.000000000000e+00},
	/*  73 */ {-1.998401485960e-16,5.400000049174e+00,1.000000000000e+00,1.000000000000e+00},
	/*  74 */ {-9.992007519363e-17,3.600000017882e+00,1.000000000000e+00,1.000000000000e+00},
	/*  75 */ {9.992007420266e-17,2.699999994040e+00,1.000000000000e+00,1.000000000000e+00},
	/*  76 */ {-9.992003427844e-17,3.149999946358e+00,1.000000000000e+00,1.000000000000e+00},
	/*  77 */ {9.992011471154e-17,2.924999969762e+00,1.000000000000e+00,1.000000000000e+00},
	/*  78 */ {-9.992022307197e-17,3.037500053308e+00,1.000000000000e+00,1.000000000000e+00},
	/*  79 */ {9.991992234114e-17,2.981250026466e+00,1.000000000000e+00,1.000000000000e+00},
	/*  80 */ {-9.991946690949e-17,3.009374946520e+00,1.000000000000e+00,1.000000000000e+00},
	/*  81 */ {9.992067637945e-17,2.995312473315e+00,1.000000000000e+00,1.000000000000e+00},
	/*  82 */ {-9.992007664693e-17,3.002343750105e+00,1.000000000000e+00,1.000000000000e+00},
	/*  83 */ {9.993221615007e-17,2.998827990854e+00,1.000000000000e+00,1.000000000000e+00},
	/*  84 */ {-9.992978324655e-17,3.000585991137e+00,1.000000000000e+00,1.000000000000e+00},
	/*  85 */ {9.992978937786e-17,2.999707004412e+00,1.000000000000e+00,1.000000000000e+00},
	/*  86 */ {-9.990381649282e-17,3.000146460550e+00,1.000000000000e+00,1.000000000000e+00},
	/*  87 */ {9.992006710982e-17,2.999926757815e+00,1.000000000000e+00,1.000000000000e+00},
	/*  88 */ {-9.992007054716e-17,3.000036621089e+00,1.000000000000e+00,1.000000000000e+00},
	/*  89 */ {1.000501949089e-16,2.999981665603e+00,1.000000000000e+00,1.000000000000e+00},
	/*  90 */ {-9.983326133578e-17,3.000009167176e+00,1.000000000000e+00,1.000000000000e+00},
	/*  91 */ {9.992051924641e-17,2.999995422320e+00,1.000000000000e+00,1.000000000000e+00},
	/*  92 */ {-9.887764509714e-17,3.000002264894e+00,1.000000000000e+00,1.000000000000e+00},
	/*  93 */ {1.020085266754e-16,2.999998831579e+00,1.000000000000e+00,1.000000000000e+00},
	/*  94 */ {-9.981892599159e-17,3.000000571855e+00,1.000000000000e+00,1.000000000000e+00},
	/*  95 */ {1.002341692529e-16,2.999999713199e+00,1.000000000000e+00,1.000000000000e+00},
	/*  96 */ {-8.618124242249e-17,3.000000123130e+00,1.000000000000e+00,1.000000000000e+00},
	/*  97 */ {3.513169196232e-17,2.999999973369e+00,1.000000000000e+00,1.000000000000e+00},
	/*  98 */ {9.229798358184e-18,2.999999985411e+00,1.000000000000e+00,1.000000000000e+00},
	/*  99 */ {3.039299249045e-16,2.999999924942e+00,3.160431357339e-01,1.000000000000e+00},
	/* 100 */ {-1.869500482808e-16,2.999999984350e+00,1.203706612070e-01,1.000000000000e+00},
	/* 101 */ {1.820386624052e-15,2.999999828537e+00,7.614131056293e-01,1.000000000000e+00},
	/* 102 */ {5.636545748983e-15,2.999999733053e+00,-5.158733158656e-02,1.000000000000e+00},
	/* 103 */ {2.326460983366e-14,2.999999444027e+00,1.121695638409e-01,1.000000000000e+00},
	/* 104 */ {7.553998377602e-15,2.999999909102e+00,-2.744270044120e+00,1.000000000000e+00},
	/* 105 */ {-2.894453663133e-15,3.000000016901e+00,-3.023404505909e+00,1.000000000000e+00},
	/* 106 */ {1.698965995225e-14,2.999999949328e+00,-2.964965741450e+00,1.000000000000e+00},
	/* 107 */ {7.841240497332e-15,2.999999988648e+00,-2.996136121572e+00,1.000000000000e+00},
	/* 108 */ {-2.263236642401e-14,3.000000016393e+00,-3.002801270253e+00,1.000000000000e+00},
	/* 109 */ {-7.112210081078e-14,3.000000025836e+00,-3.002211877172e+00,1.000000000000e+00},
	/* 110 */ {1.719843941285e-13,2.999999968773e+00,-2.998663449274e+00,1.000000000000e+00},
	/* 111 */ {-3.314510960674e-13,3.000000030088e+00,-3.000643816386e+00,1.000000000000e+00},
	/* 112 */ {-3.132279061635e-13,3.000000007378e+00,-2.999691108342e+00,1.000000000000e+00},
	/* 113 */ {3.042518786493e-12,2.999999889070e+00,-2.998703229083e+00,-3.844415908371e+00},
	/* 114 */ {1.796016634512e-11,2.999999660032e+00,-2.997953447234e+00,-2.921472790512e+00},
	/* 115 */ {-1.352526755914e-12,3.000000014272e+00,-3.000046690970e+00,1.047696181513e+00},
	/* 116 */ {1.356938769201e-10,2.999999393490e+00,-2.999128489701e+00,5.962105266738e-01},
	/* 117 */ {2.763784304116e-09,2.999993902570e+00,-2.995660459221e+00,-5.525262153133e-05},
	/* 118 */ {-1.507470718555e-09,3.000001648314e+00,-3.000578174309e+00,1.065310057612e+00},
	/* 119 */ {-6.696653062772e-12,3.000000003655e+00,-3.000000640367e+00,1.000036162751e+00},
	/* 120 */ {-9.907636716702e-12,3.000000002706e+00,-3.000000237252e+00,1.000006702886e+00},
	/* 121 */ {-2.422200996422e-09,3.000000335108e+00,-3.000014767711e+00,1.000207730591e+00},
	/* 122 */ {3.447618999311e-11,2.999999997649e+00,-2.999999948528e+00,9.999996367863e-01},
	/* 123 */ {-3.460818652918e-10,3.000000011801e+00,-3.000000129173e+00,1.000000455757e+00},
	/* 124 */ {-4.421881262699e-11,3.000000000754e+00,-3.000000004126e+00,1.000000007279e+00},
	/* 125 */ {2.295977171600e-11,2.999999999804e+00,-2.999999999464e+00,9.999999995276e-01},
	/* 126 */ {-1.967513474802e-12,3.000000000008e+00,-3.000000000011e+00,1.000000000005e+00},
	/* 127 */ {1.000000095682e+00,-2.085920724763e-07,1.462415963824e-07,-3.298948742713e-08},
	/* 128 */ {9.999998414954e-01,1.704867192286e-07,-5.879433680231e-08,6.527213702327e-09},
	/* 129 */ {9.999999465195e-01,2.882678144132e-08,-4.946723915021e-09,2.726387258461e-10},
	/* 130 */ {9.999999550595e-01,1.275250531829e-08,-1.168540616645e-09,3.453109727956e-11},
	/* 131 */ {1.000000108439e+00,-1.434744484993e-08,6.100927401123e-10,-8.366281170084e-12},
	/* 132 */ {1.000000051187e+00,-3.359184349398e-09,7.089076660832e-11,-4.812208960150e-13},
	/* 133 */ {9.999999540160e-01,1.527039088564e-09,-1.620138119628e-11,5.546019357137e-14},
	/* 134 */ {9.999998425154e-01,2.647193018433e-09,-1.428122112754e-11,2.484023281132e-14},
	/* 135 */ {9.999997588466e-01,2.004506984040e-09,-5.355592315085e-12,4.618677334033e-15},
	/* 136 */ {9.999998703926e-01,5.380816718342e-10,-7.158038190356e-13,3.063418548339e-16},
	/* 137 */ {9.999999322365e-01,1.407624043552e-10,-9.381596139353e-14,2.019842937195e-17},
	/* 138 */ {1.000000104817e+00,-1.115334091805e-10,3.800024588958e-14,-4.158984339950e-18},
	/* 139 */ {1.000000149830e+00,-7.838040298066e-11,1.317204113087e-14,-7.131425448685e-19},
	/* 140 */ {9.999999243002e-01,1.914451920920e-11,-1.545560239119e-15,4.000075234813e-20},
	/* 141 */ {1.000000119732e+00,-1.584039661931e-11,6.719685232883e-16,-9.168924074923e-21},
	/* 142 */ {1.000000010590e+00,-2.356461077357e-13,-5.366878990479e-18,1.061985310001e-22},
	/* 143 */ {1.000000101231e+00,-3.358023900072e-12,3.578399950070e-17,-1.228145179424e-22},
	/* 144 */ {9.999998867430e-01,1.850740978496e-12,-9.697165913489e-18,1.638193903749e-23},
	/* 145 */ {1.000000132697e+00,-1.083106542073e-12,2.830069576593e-18,-2.380364700083e-24},
	/* 146 */ {9.999999125972e-01,3.556491075171e-13,-4.618567755443e-19,1.922456134396e-25},
	/* 147 */ {1.000000000061e+00,-2.951085523368e-15,1.361181513922e-21,-4.076889729614e-28},
	/* 148 */ {1.000000076882e+00,-8.010717903743e-14,2.676768129432e-20,-2.876274392283e-27},
	/* 149 */ {1.000000095853e+00,-4.797935252548e-14,7.668981180237e-21,-3.945859720009e-28},
	/* 150 */ {9.999998458240e-01,3.951461383815e-14,-3.249551765188e-21,8.609234350037e-29},
	/* 151 */ {9.999999278577e-01,9.084480189355e-15,-3.689560570269e-22,4.837109174782e-30},
	/* 152 */ {1.000000143249e+00,-9.242860173679e-15,1.914368566106e-22,-1.275550322420e-30},
	/* 153 */ {9.999997865343e-01,6.787014142199e-15,-6.915208529931e-23,2.269026147243e-31},
	/* 154 */ {9.999998883099e-01,1.782717279343e-15,-9.124459490332e-24,1.504115333334e-32},
	/* 155 */ {1.000000067111e+00,-5.436405530794e-16,1.441506617448e-24,-1.245521037555e-33},
	/* 156 */ {1.000000161842e+00,-6.450736589978e-16,8.249789497421e-25,-3.397136212992e-34},
	/* 157 */ {9.999999572586e-01,8.527830862930e-17,-5.428353479022e-26,1.111122030936e-35},
	/* 158 */ {1.000000117366e+00,-1.187782241028e-16,3.861092144532e-26,-4.038741459196e-36},
	/* 159 */ {1.000000037939e+00,-1.985179002383e-17,3.295922848237e-27,-1.741114607518e-37},
	/* 160 */ {9.999999896454e-01,2.475017314656e-18,-1.866097509345e-28,4.496920989427e-39},
	/* 161 */ {9.999998014609e-01,2.458038311656e-17,-9.761266072237e-28,1.249552980487e-38},
	/* 162 */ {1.000000054392e+00,-3.396470808080e-18,6.807375293267e-29,-4.396543509624e-40},
	/* 163 */ {1.000000018253e+00,-6.966872094171e-19,8.074539972796e-30,-2.880409629428e-41},
	/* 164 */ {9.999999905302e-01,1.765526141440e-19,-9.859033646684e-31,1.717114709313e-42},
	/* 165 */ {1.000000114140e+00,-8.845247358301e-19,2.197985266073e-30,-1.757316976910e-42},
	/* 166 */ {1.000000213321e+00,-8.339061815931e-19,1.047402352079e-30,-4.239457424654e-43},
	/* 167 */ {1.000000006518e+00,-1.980750637671e-20,1.690368610192e-32,-4.286082107818e-45},
	/* 168 */ {9.999999285595e-01,6.911315299970e-20,-2.138710680698e-32,2.131249037154e-45},
	/* 169 */ {1.000000125192e+00,-6.121178796577e-20,9.617668455593e-33,-4.871037857530e-46},
	/* 170 */ {9.999999056122e-01,2.320004389134e-20,-1.828613201106e-33,4.641531149487e-47},
	/* 171 */ {9.999999804926e-01,2.215852802313e-21,-8.104547094829e-35,9.597918743804e-49},
	/* 172 */ {1.000000043277e+00,-2.232615191972e-21,3.675858001022e-35,-1.935821319545e-49},
	/* 173 */ {9.999998379495e-01,4.903182262480e-21,-4.760451993802e-35,1.490717800731e-49},
	/* 174 */ {1.000000113640e+00,-1.724254943830e-21,8.377391887291e-36,-1.311460758967e-50},
	/* 175 */ {9.999998258058e-01,1.319058820319e-21,-3.200092629014e-36,2.500573391753e-51},
	/* 176 */ {9.999999508260e-01,1.850256975813e-22,-2.233027634605e-37,8.699059850511e-53},
	/* 177 */ {1.000000135364e+00,-2.605503245704e-22,1.610072820301e-37,-3.199126124546e-53},
	/* 178 */ {1.000000074412e+00,-7.036405049848e-23,2.130752649339e-38,-2.074539295067e-54},
	/* 179 */ {9.999998992778e-01,4.855097578515e-23,-7.503336923122e-39,3.734825889942e-55},
	/* 180 */ {9.999999917724e-01,1.442739134010e-24,-5.806719124540e-41,-3.781803913518e-59},
	/* 181 */ {9.999998831770e-01,1.382115080098e-23,-5.245876941414e-40,6.414204306753e-57},
	/* 182 */ {9.999997729614e-01,1.345336025384e-23,-2.557214357754e-40,1.565854913741e-57},
	/* 183 */ {1.000000057138e+00,-1.683241633463e-24,1.580454863448e-41,-4.734314341302e-59},
	/* 184 */ {1.000000065953e+00,-9.996751784855e-25,4.853713247015e-42,-7.572578309191e-60},
	/* 185 */ {1.000000050696e+00,-3.741324320814e-25,8.893357157927e-43,-6.815715674325e-61},
	/* 186 */ {9.999998424798e-01,5.837470062265e-25,-6.943194261073e-43,2.661352534011e-61},
	/* 187 */ {1.000000079020e+00,-1.482713433291e-25,8.956690238398e-44,-1.747418247667e-62},
	/* 188 */ {9.999999766948e-01,2.326943865410e-26,-7.101520437953e-45,6.859449075325e-64},
	/* 189 */ {1.000000077593e+00,-3.415683033757e-26,4.866250743329e-45,-2.247215762902e-64},
	/* 190 */ {9.999998908052e-01,2.536017164692e-26,-1.896516176117e-45,4.574986132627e-65},
	/* 191 */ {9.999999321465e-01,8.216846544075e-27,-3.165588987332e-46,3.905182670884e-66},
	/* 192 */ {9.999999280352e-01,4.137113939608e-27,-7.599036740329e-47,4.493731540689e-67},
	/* 193 */ {9.999999674187e-01,9.091083446181e-28,-8.027325363436e-48,2.260896935255e-68},
	/* 194 */ {9.999997443597e-01,3.698597465688e-27,-1.716968673750e-47,2.568851715358e-68},
	/* 195 */ {1.000000092805e+00,-6.580218773578e-28,1.504601594149e-48,-1.108362173634e-69},
	/* 196 */ {1.000000049030e+00,-1.798219238164e-28,2.102207731361e-49,-7.890301397391e-71},
	/* 197 */ {1.000000170703e+00,-3.084736542236e-28,1.792064029653e-49,-3.357657787017e-71},
	/* 198 */ {9.999998285442e-01,1.560408495912e-28,-4.557923090849e-50,4.289688601356e-72},
	/* 199 */ {1.000000194726e+00,-8.815156872694e-29,1.281572925316e-50,-6.005787768649e-73},
	/* 200 */ {1.000000078594e+00,-1.762133515562e-29,1.270420984138e-51,-2.953435836177e-74},
	/* 201 */ {9.999996360571e-01,4.112814580462e-29,-1.491114830432e-51,1.742106389998e-74},
	/* 202 */ {9.999998517067e-01,8.412746302150e-30,-1.531318551863e-52,8.978306223308e-76},
	/* 203 */ {1.000000117394e+00,-3.326967922774e-30,3.019824882531e-53,-8.818961722772e-77},
	/* 204 */ {9.999996394824e-01,5.098646445195e-30,-2.317447007715e-53,3.395651082255e-77},
	/* 205 */ {1.000000029531e+00,-2.041704866793e-31,4.533813330439e-55,-3.231024629124e-79},
	/* 206 */ {1.000000152655e+00,-5.393777697551e-31,6.110188657097e-55,-2.230317294137e-79},
	/* 207 */ {9.999999179505e-01,1.446097504969e-31,-8.142127769280e-56,1.472907829760e-80},
	/* 208 */ {1.000000102190e+00,-9.197471639490e-32,2.653262445123e-56,-2.460017978867e-81},
	/* 209 */ {1.000000057494e+00,-2.591933677676e-32,3.764513776930e-57,-1.764642419651e-82},
	/* 210 */ {1.000000115991e+00,-2.536921025788e-32,1.781023171223e-57,-4.035428522656e-83},
	/* 211 */ {9.999999958354e-01,9.157617302051e-34,-4.686702780738e-59,6.807103654408e-85},
	/* 212 */ {9.999998938664e-01,5.815873038410e-33,-1.018596195759e-58,5.741423559967e-85},
	/* 213 */ {9.999999050071e-01,2.618789028206e-33,-2.324654480479e-59,6.680457257886e-86},
	/* 214 */ {9.999999069206e-01,1.270264795048e-33,-5.557448833245e-60,7.836344101423e-87},
	/* 215 */ {9.999998385627e-01,1.110476291648e-33,-2.448341656945e-60,1.737872351923e-87},
	/* 216 */ {1.000000101811e+00,-3.519065922491e-34,3.916699858925e-61,-1.406364382855e-88},
	/* 217 */ {9.999998607297e-01,2.410089356671e-34,-1.337957219686e-61,2.392770147670e-89},
	/* 218 */ {9.999998528181e-01,1.275505372857e-34,-3.544665559974e-62,3.170912834694e-90},
	/* 219 */ {9.999999239838e-01,3.233723972509e-35,-4.397547565138e-63,1.925907869558e-91},
	/* 220 */ {9.999996118971e-01,8.411281157375e-35,-5.848446347542e-63,1.309774394240e-91},
	/* 221 */ {9.999999605271e-01,4.141519890754e-36,-1.396094428160e-64,1.521729581197e-93},
	/* 222 */ {1.000000072590e+00,-3.862168038072e-36,6.636961049939e-65,-3.690761308951e-94},
	/* 223 */ {1.000000146083e+00,-3.921518551525e-36,3.385252907905e-65,-9.431665222219e-95},
	/* 224 */ {1.000000087269e+00,-1.160682802436e-36,4.981163300382e-66,-6.920188218309e-96},
	/* 225 */ {9.999999214892e-01,5.408438281848e-37,-1.181400175870e-66,8.267703272589e-97},
	/* 226 */ {1.000000124853e+00,-4.239099486916e-37,4.611278517919e-67,-1.611951354626e-97},
	/* 227 */ {9.999995882150e-01,6.963139070614e-37,-3.776724553831e-67,6.598844912505e-98},
	/* 228 */ {9.999998732078e-01,1.069649664032e-37,-2.887963102910e-68,2.508290312775e-99},
	/* 229 */ {1.000000024157e+00,-1.038237453211e-38,1.405066819055e-69,-6.056931032608e-101},
	/* 230 */ {9.999999348467e-01,1.373980743256e-38,-9.271256580808e-70,2.012780046963e-101},
	/* 231 */ {1.000000305594e+00,-3.240110361875e-38,1.102789746590e-69,-1.209070774866e-101},
	/* 232 */ {9.999998809930e-01,6.211990743146e-39,-1.043020781466e-70,5.661208606020e-103},
	/* 233 */ {9.999999286321e-01,1.948798284698e-39,-1.692204900421e-71,4.710452322140e-104},
	/* 234 */ {1.000000080744e+00,-1.094066256495e-39,4.748854612732e-72,-6.615065097876e-105},
	/* 235 */ {1.000000008913e+00,-3.798630914335e-41,4.001563973018e-74,-2.855494004748e-108},
	/* 236 */ {9.999998494791e-01,4.970288268048e-40,-5.251584279346e-73,1.786142057392e-106},
	/* 237 */ {1.000000029772e+00,-4.576400938828e-41,2.213564896335e-74,-3.435610283177e-108},
	/* 238 */ {1.000000310188e+00,-2.544669953312e-40,6.696405447292e-74,-5.677766704986e-108},
	/* 239 */ {1.000000170774e+00,-7.019588359451e-41,9.258212582303e-75,-3.936043785877e-109},
	/* 240 */ {9.999999752089e-01,5.277141296816e-42,-3.539545456101e-76,7.557372978604e-111},
	/* 241 */ {1.000000030124e+00,-3.260111957658e-42,1.136719538963e-76,-1.267728199750e-111},
	/* 242 */ {1.000000065375e+00,-3.402115087521e-42,5.680458530977e-77,-3.039329917369e-112},
	/* 243 */ {1.000000032932e+00,-8.742542345987e-43,7.420341710824e-78,-2.006984069535e-113},
	/* 244 */ {1.000000063193e+00,-7.863191748751e-43,3.142737510953e-78,-4.043591433283e-114},
	/* 245 */ {9.999998737472e-01,8.221443583807e-43,-1.720355117727e-78,1.160771254620e-114},
	/* 246 */ {1.000000058226e+00,-1.861684218988e-43,1.900852677889e-79,-6.233034741038e-116},
	/* 247 */ {1.000000183002e+00,-2.946237450692e-43,1.521800031387e-79,-2.532481282972e-116},
	/* 248 */ {9.999998186643e-01,1.464788377915e-43,-3.793002223441e-80,3.161791286655e-117},
	/* 249 */ {1.000000152235e+00,-6.151206918663e-44,7.977009067619e-81,-3.330676668599e-118},
};

float clip3 (float x)
{
	union { float f; uint32 i; } u;
	u.f = -x;
	int e = (u.i & 0x7F800000) >> 23;
	float *p = clip3tbl[e];
	x = fabs (x);
	x = p[0]+x*(p[1]+x*(p[2]+x*p[3]));
	u.i &= 0x80000000;
	u.i |= 127 << 23; /* 1.0 - use 126 for .5 instead, etc */
	return x * u.f;
}

float clip9tbl[][4] = {
	/*   0 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   1 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   2 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   3 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   4 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   5 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   6 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   7 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   8 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*   9 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  10 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  11 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  12 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  13 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  14 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  15 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  16 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  17 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  18 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  19 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  20 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  21 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  22 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  23 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  24 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  25 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  26 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  27 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  28 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  29 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  30 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  31 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  32 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  33 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  34 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  35 */ {4.757904109314e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  36 */ {4.757500212530e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  37 */ {4.757096315747e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  38 */ {4.755884625396e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  39 */ {4.753461244696e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  40 */ {4.748614483294e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  41 */ {4.738920960491e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  42 */ {4.719533914884e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  43 */ {4.680759823670e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  44 */ {4.603211641244e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  45 */ {4.448115276390e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  46 */ {4.137922546683e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  47 */ {3.517537087268e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  48 */ {2.276766168438e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  49 */ {-2.047756692209e-25,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  50 */ {-5.167859344539e-24,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  51 */ {-1.509402669518e-23,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  52 */ {-3.494636139645e-23,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  53 */ {-7.465103079899e-23,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  54 */ {-1.540603696041e-22,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  55 */ {-3.128790472143e-22,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  56 */ {-6.305164024346e-22,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  57 */ {-1.265791112875e-21,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  58 */ {-2.536340533757e-21,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  59 */ {-7.034381313986e-39,1.377425589194e-18,1.000000000000e+00,1.000000000000e+00},
	/*  60 */ {9.465525899283e-41,-2.033122502002e-20,1.000000000000e+00,1.000000000000e+00},
	/*  61 */ {5.530942284335e-40,-5.798412628578e-20,1.425925932565e+00,1.000000000000e+00},
	/*  62 */ {-4.312598204865e-59,-6.014750754107e-36,2.764743250389e-17,1.000000000000e+00},
	/*  63 */ {7.376799804131e-39,-1.680240410675e-19,1.000000000000e+00,1.000000000000e+00},
	/*  64 */ {1.271828537329e-32,-9.338226169131e-14,1.000000000000e+00,1.000000000000e+00},
	/*  65 */ {9.952817631163e-38,-6.517453240027e-19,1.000000000000e+00,1.000000000000e+00},
	/*  66 */ {3.970559283422e-37,-1.300867671208e-18,9.999999998929e-01,1.000000000000e+00},
	/*  67 */ {1.625839198223e-36,-2.558784641806e-18,1.000000000000e+00,1.000000000000e+00},
	/*  68 */ {6.503354736646e-36,-5.225890685089e-18,1.000000000000e+00,1.000000000000e+00},
	/*  69 */ {2.529872861266e-35,-1.038664407722e-17,1.000000000000e+00,1.000000000000e+00},
	/*  70 */ {1.016392504791e-34,-2.081600372914e-17,1.000000000000e+00,1.000000000000e+00},
	/*  71 */ {4.065852103997e-34,-4.163336044713e-17,1.000000000000e+00,1.000000000000e+00},
	/*  72 */ {1.626340735517e-33,-8.326672168953e-17,1.000000000000e+00,1.000000000000e+00},
	/*  73 */ {6.661288301847e-16,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  74 */ {1.332267634308e-15,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  75 */ {2.664535263857e-15,1.000000000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  76 */ {-2.997602030358e-16,9.449999979564e+00,1.000000000000e+00,1.000000000000e+00},
	/*  77 */ {2.997601694023e-16,8.775000033632e+00,1.000000000000e+00,1.000000000000e+00},
	/*  78 */ {-2.997602400141e-16,9.112500020847e+00,1.000000000000e+00,1.000000000000e+00},
	/*  79 */ {2.997602166489e-16,8.943750000000e+00,1.000000000000e+00,1.000000000000e+00},
	/*  80 */ {-2.997607016080e-16,9.028125082368e+00,1.000000000000e+00,1.000000000000e+00},
	/*  81 */ {2.997622780808e-16,8.985937435484e+00,1.000000000000e+00,1.000000000000e+00},
	/*  82 */ {-2.997598780017e-16,9.007031222610e+00,1.000000000000e+00,1.000000000000e+00},
	/*  83 */ {2.997583726036e-16,8.996484426906e+00,1.000000000000e+00,1.000000000000e+00},
	/*  84 */ {-2.997796391777e-16,9.001757919770e+00,1.000000000000e+00,1.000000000000e+00},
	/*  85 */ {2.997213860445e-16,8.999121200980e+00,1.000000000000e+00,1.000000000000e+00},
	/*  86 */ {-2.997137170821e-16,9.000439384361e+00,1.000000000000e+00,1.000000000000e+00},
	/*  87 */ {2.996047669969e-16,8.999780380760e+00,1.000000000000e+00,1.000000000000e+00},
	/*  88 */ {-2.997602388928e-16,9.000109863273e+00,1.000000000000e+00,1.000000000000e+00},
	/*  89 */ {2.997603299766e-16,8.999945068350e+00,1.000000000000e+00,1.000000000000e+00},
	/*  90 */ {-2.989931138035e-16,9.000027403280e+00,1.000000000000e+00,1.000000000000e+00},
	/*  91 */ {2.973110109842e-16,8.999986372805e+00,1.000000000000e+00,1.000000000000e+00},
	/*  92 */ {-2.997598777010e-16,9.000006866368e+00,1.000000000000e+00,1.000000000000e+00},
	/*  93 */ {2.998150649618e-16,8.999996566598e+00,1.000000000000e+00,1.000000000000e+00},
	/*  94 */ {-3.000866226909e-16,9.000001716264e+00,1.000000000000e+00,1.000000000000e+00},
	/*  95 */ {2.944878870761e-16,8.999999169615e+00,1.000000000000e+00,1.000000000000e+00},
	/*  96 */ {-1.940396190519e-16,9.000000286748e+00,1.000000000000e+00,1.000000000000e+00},
	/*  97 */ {3.291213035565e-16,8.999999718387e+00,1.000000000000e+00,1.000000000000e+00},
	/*  98 */ {4.732675606252e-16,8.999999724237e+00,1.000000000000e+00,1.000000000000e+00},
	/*  99 */ {1.465090975179e-15,8.999999523908e+00,1.000000000000e+00,1.000000000000e+00},
	/* 100 */ {2.444072022703e-15,8.999999349668e+00,1.000000000000e+00,1.000000000000e+00},
	/* 101 */ {1.718109591442e-14,8.999998384328e+00,3.162157577162e-01,1.000000000000e+00},
	/* 102 */ {6.303097964364e-14,8.999996986954e+00,-2.224901042364e+00,1.000000000000e+00},
	/* 103 */ {2.669782457904e-13,8.999993663972e+00,-7.034565074715e-01,1.000000000000e+00},
	/* 104 */ {-9.008397868654e-15,9.000000102968e+00,-3.627765454282e+01,1.000000000000e+00},
	/* 105 */ {2.088816652977e-14,8.999999861783e+00,-3.579192574391e+01,1.000000000000e+00},
	/* 106 */ {1.160757114562e-14,8.999999965650e+00,-3.597603042948e+01,1.000000000000e+00},
	/* 107 */ {-4.234540986731e-13,9.000000665445e+00,-3.624308670131e+01,1.000000000000e+00},
	/* 108 */ {-3.112215648774e-14,9.000000021743e+00,-3.600332491945e+01,1.000000000000e+00},
	/* 109 */ {-2.343871224792e-13,9.000000082356e+00,-3.600630226115e+01,1.000000000000e+00},
	/* 110 */ {6.643601215826e-13,8.999999857738e+00,-3.599203533014e+01,1.000000000000e+00},
	/* 111 */ {2.559099571039e-12,8.999999718185e+00,-3.599089834295e+01,1.000000000000e+00},
	/* 112 */ {-3.848865244836e-12,9.000000178758e+00,-3.599714589886e+01,1.000000000000e+00},
	/* 113 */ {1.007468193466e-10,8.999996695214e+00,-3.596879746404e+01,1.000000000000e+00},
	/* 114 */ {4.796162168734e-10,8.999991654325e+00,-3.595357778432e+01,1.004121334205e+00},
	/* 115 */ {3.112446061511e-10,8.999997272073e+00,-3.599230997796e+01,7.697033686113e+01},
	/* 116 */ {4.311435168954e-10,8.999998074845e+00,-3.599716842710e+01,8.253189413701e+01},
	/* 117 */ {5.058490161532e-10,8.999998521849e+00,-3.599841902596e+01,8.326445979535e+01},
	/* 118 */ {8.051445784528e-09,8.999988242703e+00,-3.599370829530e+01,8.253450535975e+01},
	/* 119 */ {1.273590932700e-07,8.999906950409e+00,-3.597507509173e+01,8.109032038478e+01},
	/* 120 */ {1.990287480500e-06,8.999271921338e+00,-3.590224969935e+01,7.826614245849e+01},
	/* 121 */ {3.037058272677e-05,8.994429060871e+00,-3.562417755691e+01,7.286615769792e+01},
	/* 122 */ {4.415224079389e-04,8.959262542549e+00,-3.461169788595e+01,6.300469998014e+01},
	/* 123 */ {5.807455283203e-03,8.728564355667e+00,-3.127053265771e+01,4.662593111296e+01},
	/* 124 */ {6.171622639330e-02,7.511892920823e+00,-2.233777599570e+01,2.440555883860e+01},
	/* 125 */ {4.068420158983e-01,3.630231357569e+00,-7.567862051505e+00,5.344437545546e+00},
	/* 126 */ {9.632201646091e-01,1.349451303155e-01,-1.630015432099e-01,6.483624828532e-02},
	/* 127 */ {1.000000095682e+00,-2.085920724763e-07,1.462415963824e-07,-3.298948742713e-08},
	/* 128 */ {9.999998414954e-01,1.704867192286e-07,-5.879433680231e-08,6.527213702327e-09},
	/* 129 */ {9.999999465195e-01,2.882678144132e-08,-4.946723915021e-09,2.726387258461e-10},
	/* 130 */ {9.999999550595e-01,1.275250531829e-08,-1.168540616645e-09,3.453109727956e-11},
	/* 131 */ {1.000000108439e+00,-1.434744484993e-08,6.100927401123e-10,-8.366281170084e-12},
	/* 132 */ {1.000000051187e+00,-3.359184349398e-09,7.089076660832e-11,-4.812208960150e-13},
	/* 133 */ {9.999999540160e-01,1.527039088564e-09,-1.620138119628e-11,5.546019357137e-14},
	/* 134 */ {9.999998425154e-01,2.647193018433e-09,-1.428122112754e-11,2.484023281132e-14},
	/* 135 */ {9.999997588466e-01,2.004506984040e-09,-5.355592315085e-12,4.618677334033e-15},
	/* 136 */ {9.999998703926e-01,5.380816718342e-10,-7.158038190356e-13,3.063418548339e-16},
	/* 137 */ {9.999999322365e-01,1.407624043552e-10,-9.381596139353e-14,2.019842937195e-17},
	/* 138 */ {1.000000104817e+00,-1.115334091805e-10,3.800024588958e-14,-4.158984339950e-18},
	/* 139 */ {1.000000149830e+00,-7.838040298066e-11,1.317204113087e-14,-7.131425448685e-19},
	/* 140 */ {9.999999243002e-01,1.914451920920e-11,-1.545560239119e-15,4.000075234813e-20},
	/* 141 */ {1.000000119732e+00,-1.584039661931e-11,6.719685232883e-16,-9.168924074923e-21},
	/* 142 */ {1.000000010590e+00,-2.356461077357e-13,-5.366878990479e-18,1.061985310001e-22},
	/* 143 */ {1.000000101231e+00,-3.358023900072e-12,3.578399950070e-17,-1.228145179424e-22},
	/* 144 */ {9.999998867430e-01,1.850740978496e-12,-9.697165913489e-18,1.638193903749e-23},
	/* 145 */ {1.000000132697e+00,-1.083106542073e-12,2.830069576593e-18,-2.380364700083e-24},
	/* 146 */ {9.999999125972e-01,3.556491075171e-13,-4.618567755443e-19,1.922456134396e-25},
	/* 147 */ {1.000000000061e+00,-2.951085523368e-15,1.361181513922e-21,-4.076889729614e-28},
	/* 148 */ {1.000000076882e+00,-8.010717903743e-14,2.676768129432e-20,-2.876274392283e-27},
	/* 149 */ {1.000000095853e+00,-4.797935252548e-14,7.668981180237e-21,-3.945859720009e-28},
	/* 150 */ {9.999998458240e-01,3.951461383815e-14,-3.249551765188e-21,8.609234350037e-29},
	/* 151 */ {9.999999278577e-01,9.084480189355e-15,-3.689560570269e-22,4.837109174782e-30},
	/* 152 */ {1.000000143249e+00,-9.242860173679e-15,1.914368566106e-22,-1.275550322420e-30},
	/* 153 */ {9.999997865343e-01,6.787014142199e-15,-6.915208529931e-23,2.269026147243e-31},
	/* 154 */ {9.999998883099e-01,1.782717279343e-15,-9.124459490332e-24,1.504115333334e-32},
	/* 155 */ {1.000000067111e+00,-5.436405530794e-16,1.441506617448e-24,-1.245521037555e-33},
	/* 156 */ {1.000000161842e+00,-6.450736589978e-16,8.249789497421e-25,-3.397136212992e-34},
	/* 157 */ {9.999999572586e-01,8.527830862930e-17,-5.428353479022e-26,1.111122030936e-35},
	/* 158 */ {1.000000117366e+00,-1.187782241028e-16,3.861092144532e-26,-4.038741459196e-36},
	/* 159 */ {1.000000037939e+00,-1.985179002383e-17,3.295922848237e-27,-1.741114607518e-37},
	/* 160 */ {9.999999896454e-01,2.475017314656e-18,-1.866097509345e-28,4.496920989427e-39},
	/* 161 */ {9.999998014609e-01,2.458038311656e-17,-9.761266072237e-28,1.249552980487e-38},
	/* 162 */ {1.000000054392e+00,-3.396470808080e-18,6.807375293267e-29,-4.396543509624e-40},
	/* 163 */ {1.000000018253e+00,-6.966872094171e-19,8.074539972796e-30,-2.880409629428e-41},
	/* 164 */ {9.999999905302e-01,1.765526141440e-19,-9.859033646684e-31,1.717114709313e-42},
	/* 165 */ {1.000000114140e+00,-8.845247358301e-19,2.197985266073e-30,-1.757316976910e-42},
	/* 166 */ {1.000000213321e+00,-8.339061815931e-19,1.047402352079e-30,-4.239457424654e-43},
	/* 167 */ {1.000000006518e+00,-1.980750637671e-20,1.690368610192e-32,-4.286082107818e-45},
	/* 168 */ {9.999999285595e-01,6.911315299970e-20,-2.138710680698e-32,2.131249037154e-45},
	/* 169 */ {1.000000125192e+00,-6.121178796577e-20,9.617668455593e-33,-4.871037857530e-46},
	/* 170 */ {9.999999056122e-01,2.320004389134e-20,-1.828613201106e-33,4.641531149487e-47},
	/* 171 */ {9.999999804926e-01,2.215852802313e-21,-8.104547094829e-35,9.597918743804e-49},
	/* 172 */ {1.000000043277e+00,-2.232615191972e-21,3.675858001022e-35,-1.935821319545e-49},
	/* 173 */ {9.999998379495e-01,4.903182262480e-21,-4.760451993802e-35,1.490717800731e-49},
	/* 174 */ {1.000000113640e+00,-1.724254943830e-21,8.377391887291e-36,-1.311460758967e-50},
	/* 175 */ {9.999998258058e-01,1.319058820319e-21,-3.200092629014e-36,2.500573391753e-51},
	/* 176 */ {9.999999508260e-01,1.850256975813e-22,-2.233027634605e-37,8.699059850511e-53},
	/* 177 */ {1.000000135364e+00,-2.605503245704e-22,1.610072820301e-37,-3.199126124546e-53},
	/* 178 */ {1.000000074412e+00,-7.036405049848e-23,2.130752649339e-38,-2.074539295067e-54},
	/* 179 */ {9.999998992778e-01,4.855097578515e-23,-7.503336923122e-39,3.734825889942e-55},
	/* 180 */ {9.999999917724e-01,1.442739134010e-24,-5.806719124540e-41,-3.781803913518e-59},
	/* 181 */ {9.999998831770e-01,1.382115080098e-23,-5.245876941414e-40,6.414204306753e-57},
	/* 182 */ {9.999997729614e-01,1.345336025384e-23,-2.557214357754e-40,1.565854913741e-57},
	/* 183 */ {1.000000057138e+00,-1.683241633463e-24,1.580454863448e-41,-4.734314341302e-59},
	/* 184 */ {1.000000065953e+00,-9.996751784855e-25,4.853713247015e-42,-7.572578309191e-60},
	/* 185 */ {1.000000050696e+00,-3.741324320814e-25,8.893357157927e-43,-6.815715674325e-61},
	/* 186 */ {9.999998424798e-01,5.837470062265e-25,-6.943194261073e-43,2.661352534011e-61},
	/* 187 */ {1.000000079020e+00,-1.482713433291e-25,8.956690238398e-44,-1.747418247667e-62},
	/* 188 */ {9.999999766948e-01,2.326943865410e-26,-7.101520437953e-45,6.859449075325e-64},
	/* 189 */ {1.000000077593e+00,-3.415683033757e-26,4.866250743329e-45,-2.247215762902e-64},
	/* 190 */ {9.999998908052e-01,2.536017164692e-26,-1.896516176117e-45,4.574986132627e-65},
	/* 191 */ {9.999999321465e-01,8.216846544075e-27,-3.165588987332e-46,3.905182670884e-66},
	/* 192 */ {9.999999280352e-01,4.137113939608e-27,-7.599036740329e-47,4.493731540689e-67},
	/* 193 */ {9.999999674187e-01,9.091083446181e-28,-8.027325363436e-48,2.260896935255e-68},
	/* 194 */ {9.999997443597e-01,3.698597465688e-27,-1.716968673750e-47,2.568851715358e-68},
	/* 195 */ {1.000000092805e+00,-6.580218773578e-28,1.504601594149e-48,-1.108362173634e-69},
	/* 196 */ {1.000000049030e+00,-1.798219238164e-28,2.102207731361e-49,-7.890301397391e-71},
	/* 197 */ {1.000000170703e+00,-3.084736542236e-28,1.792064029653e-49,-3.357657787017e-71},
	/* 198 */ {9.999998285442e-01,1.560408495912e-28,-4.557923090849e-50,4.289688601356e-72},
	/* 199 */ {1.000000194726e+00,-8.815156872694e-29,1.281572925316e-50,-6.005787768649e-73},
	/* 200 */ {1.000000078594e+00,-1.762133515562e-29,1.270420984138e-51,-2.953435836177e-74},
	/* 201 */ {9.999996360571e-01,4.112814580462e-29,-1.491114830432e-51,1.742106389998e-74},
	/* 202 */ {9.999998517067e-01,8.412746302150e-30,-1.531318551863e-52,8.978306223308e-76},
	/* 203 */ {1.000000117394e+00,-3.326967922774e-30,3.019824882531e-53,-8.818961722772e-77},
	/* 204 */ {9.999996394824e-01,5.098646445195e-30,-2.317447007715e-53,3.395651082255e-77},
	/* 205 */ {1.000000029531e+00,-2.041704866793e-31,4.533813330439e-55,-3.231024629124e-79},
	/* 206 */ {1.000000152655e+00,-5.393777697551e-31,6.110188657097e-55,-2.230317294137e-79},
	/* 207 */ {9.999999179505e-01,1.446097504969e-31,-8.142127769280e-56,1.472907829760e-80},
	/* 208 */ {1.000000102190e+00,-9.197471639490e-32,2.653262445123e-56,-2.460017978867e-81},
	/* 209 */ {1.000000057494e+00,-2.591933677676e-32,3.764513776930e-57,-1.764642419651e-82},
	/* 210 */ {1.000000115991e+00,-2.536921025788e-32,1.781023171223e-57,-4.035428522656e-83},
	/* 211 */ {9.999999958354e-01,9.157617302051e-34,-4.686702780738e-59,6.807103654408e-85},
	/* 212 */ {9.999998938664e-01,5.815873038410e-33,-1.018596195759e-58,5.741423559967e-85},
	/* 213 */ {9.999999050071e-01,2.618789028206e-33,-2.324654480479e-59,6.680457257886e-86},
	/* 214 */ {9.999999069206e-01,1.270264795048e-33,-5.557448833245e-60,7.836344101423e-87},
	/* 215 */ {9.999998385627e-01,1.110476291648e-33,-2.448341656945e-60,1.737872351923e-87},
	/* 216 */ {1.000000101811e+00,-3.519065922491e-34,3.916699858925e-61,-1.406364382855e-88},
	/* 217 */ {9.999998607297e-01,2.410089356671e-34,-1.337957219686e-61,2.392770147670e-89},
	/* 218 */ {9.999998528181e-01,1.275505372857e-34,-3.544665559974e-62,3.170912834694e-90},
	/* 219 */ {9.999999239838e-01,3.233723972509e-35,-4.397547565138e-63,1.925907869558e-91},
	/* 220 */ {9.999996118971e-01,8.411281157375e-35,-5.848446347542e-63,1.309774394240e-91},
	/* 221 */ {9.999999605271e-01,4.141519890754e-36,-1.396094428160e-64,1.521729581197e-93},
	/* 222 */ {1.000000072590e+00,-3.862168038072e-36,6.636961049939e-65,-3.690761308951e-94},
	/* 223 */ {1.000000146083e+00,-3.921518551525e-36,3.385252907905e-65,-9.431665222219e-95},
	/* 224 */ {1.000000087269e+00,-1.160682802436e-36,4.981163300382e-66,-6.920188218309e-96},
	/* 225 */ {9.999999214892e-01,5.408438281848e-37,-1.181400175870e-66,8.267703272589e-97},
	/* 226 */ {1.000000124853e+00,-4.239099486916e-37,4.611278517919e-67,-1.611951354626e-97},
	/* 227 */ {9.999995882150e-01,6.963139070614e-37,-3.776724553831e-67,6.598844912505e-98},
	/* 228 */ {9.999998732078e-01,1.069649664032e-37,-2.887963102910e-68,2.508290312775e-99},
	/* 229 */ {1.000000024157e+00,-1.038237453211e-38,1.405066819055e-69,-6.056931032608e-101},
	/* 230 */ {9.999999348467e-01,1.373980743256e-38,-9.271256580808e-70,2.012780046963e-101},
	/* 231 */ {1.000000305594e+00,-3.240110361875e-38,1.102789746590e-69,-1.209070774866e-101},
	/* 232 */ {9.999998809930e-01,6.211990743146e-39,-1.043020781466e-70,5.661208606020e-103},
	/* 233 */ {9.999999286321e-01,1.948798284698e-39,-1.692204900421e-71,4.710452322140e-104},
	/* 234 */ {1.000000080744e+00,-1.094066256495e-39,4.748854612732e-72,-6.615065097876e-105},
	/* 235 */ {1.000000008913e+00,-3.798630914335e-41,4.001563973018e-74,-2.855494004748e-108},
	/* 236 */ {9.999998494791e-01,4.970288268048e-40,-5.251584279346e-73,1.786142057392e-106},
	/* 237 */ {1.000000029772e+00,-4.576400938828e-41,2.213564896335e-74,-3.435610283177e-108},
	/* 238 */ {1.000000310188e+00,-2.544669953312e-40,6.696405447292e-74,-5.677766704986e-108},
	/* 239 */ {1.000000170774e+00,-7.019588359451e-41,9.258212582303e-75,-3.936043785877e-109},
	/* 240 */ {9.999999752089e-01,5.277141296816e-42,-3.539545456101e-76,7.557372978604e-111},
	/* 241 */ {1.000000030124e+00,-3.260111957658e-42,1.136719538963e-76,-1.267728199750e-111},
	/* 242 */ {1.000000065375e+00,-3.402115087521e-42,5.680458530977e-77,-3.039329917369e-112},
	/* 243 */ {1.000000032932e+00,-8.742542345987e-43,7.420341710824e-78,-2.006984069535e-113},
	/* 244 */ {1.000000063193e+00,-7.863191748751e-43,3.142737510953e-78,-4.043591433283e-114},
	/* 245 */ {9.999998737472e-01,8.221443583807e-43,-1.720355117727e-78,1.160771254620e-114},
	/* 246 */ {1.000000058226e+00,-1.861684218988e-43,1.900852677889e-79,-6.233034741038e-116},
	/* 247 */ {1.000000183002e+00,-2.946237450692e-43,1.521800031387e-79,-2.532481282972e-116},
	/* 248 */ {9.999998186643e-01,1.464788377915e-43,-3.793002223441e-80,3.161791286655e-117},
	/* 249 */ {1.000000152235e+00,-6.151206918663e-44,7.977009067619e-81,-3.330676668599e-118},
};

float clip9 (float x)
{
	union { float f; uint32 i; } u;
	u.f = -x;
	int e = (u.i & 0x7F800000) >> 23;
	float *p = clip9tbl[e];
	x = fabs (x);
	x = p[0]+x*(p[1]+x*(p[2]+x*p[3]));
	u.i &= 0x80000000;
	u.i |= 127 << 23; /* 1.0 - use 126 for .5 instead, etc */
	return x * u.f;
}


float one5tbl3[][3] = {
	/*   0 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*   1 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*   2 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*   3 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*   4 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*   5 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*   6 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*   7 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*   8 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*   9 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*  10 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*  11 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*  12 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*  13 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*  14 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*  15 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*  16 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*  17 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*  18 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*  19 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*  20 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*  21 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*  22 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*  23 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*  24 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*  25 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*  26 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*  27 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*  28 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*  29 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*  30 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*  31 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*  32 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*  33 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*  34 */ {4.758308006097e-24,1.000000000000e+00,1.000000000000e+00},
	/*  35 */ {4.757904109314e-24,1.000000000000e+00,1.000000000000e+00},
	/*  36 */ {4.757096315747e-24,1.000000000000e+00,1.000000000000e+00},
	/*  37 */ {4.756692418963e-24,1.000000000000e+00,1.000000000000e+00},
	/*  38 */ {4.755884625396e-24,1.000000000000e+00,1.000000000000e+00},
	/*  39 */ {4.753461244696e-24,1.000000000000e+00,1.000000000000e+00},
	/*  40 */ {4.748210586510e-24,1.000000000000e+00,1.000000000000e+00},
	/*  41 */ {4.738517063707e-24,1.000000000000e+00,1.000000000000e+00},
	/*  42 */ {4.719533914884e-24,1.000000000000e+00,1.000000000000e+00},
	/*  43 */ {4.680759823670e-24,1.000000000000e+00,1.000000000000e+00},
	/*  44 */ {4.602807744460e-24,1.000000000000e+00,1.000000000000e+00},
	/*  45 */ {4.448115276390e-24,1.000000000000e+00,1.000000000000e+00},
	/*  46 */ {4.137922546683e-24,1.000000000000e+00,1.000000000000e+00},
	/*  47 */ {3.517537087268e-24,1.000000000000e+00,1.000000000000e+00},
	/*  48 */ {2.276766168438e-24,1.000000000000e+00,1.000000000000e+00},
	/*  49 */ {-2.051795660044e-25,1.000000000000e+00,1.000000000000e+00},
	/*  50 */ {-5.168263241323e-24,1.000000000000e+00,1.000000000000e+00},
	/*  51 */ {-1.509402669518e-23,1.000000000000e+00,1.000000000000e+00},
	/*  52 */ {-3.494676529323e-23,1.000000000000e+00,1.000000000000e+00},
	/*  53 */ {-7.465143469578e-23,1.000000000000e+00,1.000000000000e+00},
	/*  54 */ {-1.540603696041e-22,1.000000000000e+00,1.000000000000e+00},
	/*  55 */ {-3.128790472143e-22,1.000000000000e+00,1.000000000000e+00},
	/*  56 */ {-6.305164024346e-22,1.000000000000e+00,1.000000000000e+00},
	/*  57 */ {-1.265791112875e-21,1.000000000000e+00,1.000000000000e+00},
	/*  58 */ {-2.536340937654e-21,1.000000000000e+00,1.000000000000e+00},
	/*  59 */ {-5.077439779416e-21,1.000000000000e+00,1.000000000000e+00},
	/*  60 */ {-1.015963705905e-20,1.000000000000e+00,1.000000000000e+00},
	/*  61 */ {3.458796973524e-40,-3.458524859145e-20,1.000000000000e+00},
	/*  62 */ {1.567265451843e-39,-8.336987957799e-20,1.000000000000e+00},
	/*  63 */ {-4.109723676411e-32,5.008262841818e-13,1.000000000000e+00},
	/*  64 */ {8.202645130444e-33,-2.462057654703e-14,1.000000000000e+00},
	/*  65 */ {-4.930950992797e-32,1.185601566988e-13,1.000000000000e+00},
	/*  66 */ {-4.930726106034e-32,8.447547628526e-14,1.000000000000e+00},
	/*  67 */ {-8.810330159413e-36,6.547577295953e-15,1.000000000000e+00},
	/*  68 */ {-8.225836899302e-33,2.185454668937e-16,1.000000000000e+00},
	/*  69 */ {-5.750345052274e-32,1.009217565049e-14,1.000000000000e+00},
	/*  70 */ {8.302625333085e-33,-1.313942022183e-16,1.000000000000e+00},
	/*  71 */ {-4.069529116452e-32,2.289771828106e-15,1.000000000000e+00},
	/*  72 */ {1.594564699046e-33,-1.942918279522e-16,1.000000000000e+00},
	/*  73 */ {-3.467945931721e-32,3.885976790945e-16,1.000000000000e+00},
	/*  74 */ {1.850371724252e-16,-2.000000009934e+00,1.000000000000e+00},
	/*  75 */ {1.480297349623e-16,-1.999999995033e+00,1.000000000000e+00},
	/*  76 */ {-2.220446049250e-16,-1.000000000000e+00,1.000000000000e+00},
	/*  77 */ {-2.220446049250e-16,-1.000000000000e+00,1.000000000000e+00},
	/*  78 */ {-4.070817873265e-16,-8.749999959361e-01,1.000000000000e+00},
	/*  79 */ {-7.401486352674e-17,-1.000000000004e+00,1.000000000000e+00},
	/*  80 */ {4.803948342630e-24,-1.000000000420e+00,1.000000000000e+00},
	/*  81 */ {4.851204266296e-24,-1.000000000212e+00,1.000000000000e+00},
	/*  82 */ {4.945716113629e-24,-1.000000000108e+00,1.000000000000e+00},
	/*  83 */ {-7.401486317212e-17,-1.000000000005e+00,1.000000000000e+00},
	/*  84 */ {2.590520446088e-16,-1.001953125005e+00,1.000000000000e+00},
	/*  85 */ {-2.220446048173e-16,-1.000000000001e+00,1.000000000000e+00},
	/*  86 */ {-2.220446044942e-16,-1.000000000001e+00,1.000000000000e+00},
	/*  87 */ {-4.070817739727e-16,-9.997558593777e-01,1.000000000000e+00},
	/*  88 */ {-7.401484475524e-17,-1.000000000010e+00,1.000000000000e+00},
	/*  89 */ {5.613680614137e-23,-1.000000000016e+00,1.000000000000e+00},
	/*  90 */ {2.590522017376e-16,-1.000030517604e+00,1.000000000000e+00},
	/*  91 */ {-4.810961028644e-16,-9.999847412590e-01,1.000000000000e+00},
	/*  92 */ {-7.401290845467e-17,-1.000000000092e+00,1.000000000000e+00},
	/*  93 */ {2.590594833601e-16,-1.000003814876e+00,1.000000000000e+00},
	/*  94 */ {-4.810676430175e-16,-9.999980930050e-01,1.000000000000e+00},
	/*  95 */ {-7.390040226246e-17,-1.000000000703e+00,1.000000000000e+00},
	/*  96 */ {1.854919736279e-16,-1.000000478239e+00,1.000000000000e+00},
	/*  97 */ {-1.462166335039e-16,-1.000000002798e+00,1.000000000000e+00},
	/*  98 */ {-3.998541112982e-16,-9.999998863786e-01,1.000000000000e+00},
	/*  99 */ {5.959625446781e-17,-1.000000019575e+00,1.328224488321e-01},
	/* 100 */ {1.714048024590e-16,-1.000000027329e+00,1.512159649189e-01},
	/* 101 */ {1.621274000604e-20,-1.000000014529e+00,4.903944097690e-01},
	/* 102 */ {2.013745816574e-16,-1.000000016543e+00,3.286051863283e-01},
	/* 103 */ {3.503106511757e-16,-1.000000006757e+00,1.406268363229e-01},
	/* 104 */ {-1.652756128688e-14,-9.999998041232e-01,-4.164899992457e-01},
	/* 105 */ {1.308002149388e-14,-1.000000077012e+00,2.299889124943e-01},
	/* 106 */ {1.004638728409e-14,-1.000000031609e+00,1.479794786255e-01},
	/* 107 */ {-3.100267427351e-15,-9.999999956163e-01,1.235169852291e-01},
	/* 108 */ {-2.606061978980e-14,-9.999999812131e-01,1.221427970291e-01},
	/* 109 */ {6.897871460500e-14,-1.000000022206e+00,1.266634813281e-01},
	/* 110 */ {-1.543259337164e-14,-9.999999971831e-01,1.248796427907e-01},
	/* 111 */ {-1.598784302997e-14,-9.999999985407e-01,1.249690957119e-01},
	/* 112 */ {-3.157557304758e-13,-9.999999855516e-01,1.248429135325e-01},
	/* 113 */ {-7.799717324925e-13,-9.999999838400e-01,1.249259722881e-01},
	/* 114 */ {2.254784574031e-13,-1.000000002949e+00,1.250109159836e-01},
	/* 115 */ {-1.148847196731e-12,-9.999999952637e-01,1.250011797005e-01},
	/* 116 */ {1.460276344289e-12,-1.000000009884e+00,1.250187020590e-01},
	/* 117 */ {2.896743743285e-11,-1.000000064621e+00,1.250458024437e-01},
	/* 118 */ {2.331979610732e-10,-1.000000258669e+00,1.250916552967e-01},
	/* 119 */ {1.868808217419e-09,-1.000001036212e+00,1.251835163277e-01},
	/* 120 */ {1.499997394248e-08,-1.000004157175e+00,1.253678584107e-01},
	/* 121 */ {1.207998777732e-07,-1.000016728166e+00,1.257390454666e-01},
	/* 122 */ {9.794348095576e-07,-1.000067722200e+00,1.264916123790e-01},
	/* 123 */ {8.051887022244e-06,-1.000277595079e+00,1.280390306807e-01},
	/* 124 */ {6.815131388215e-05,-1.001168035966e+00,1.313161944527e-01},
	/* 125 */ {6.155413690708e-04,-1.005210153311e+00,1.387264229497e-01},
	/* 126 */ {6.569884799094e-03,-1.027012560051e+00,1.585138627092e-01},
	/* 127 */ {1.617604580795e-01,-1.299831645537e+00,2.761423749154e-01},
	/* 128 */ {-1.333333333306e+00,-2.192246384425e-12,-2.179367797339e-12},
	/* 129 */ {-1.333333333236e+00,-2.208011551375e-12,-2.179145752734e-12},
	/* 130 */ {-1.333333332969e+00,-2.233324636336e-12,-2.179367797339e-12},
	/* 131 */ {-1.333333331930e+00,-2.283950806259e-12,-2.179367797339e-12},
	/* 132 */ {-1.333333329612e+00,-2.385425190710e-12,-1.395747939969e-12},
	/* 133 */ {-1.333333337016e+00,-2.587485781191e-12,4.317522796918e-13},
	/* 134 */ {-1.333333324373e+00,-2.992495140575e-12,-2.132487632469e-13},
	/* 135 */ {-1.333333335147e+00,-3.803179993156e-12,2.139521044688e-14},
	/* 136 */ {-1.333333328507e+00,-1.512199787296e-11,1.040779754155e-14},
	/* 137 */ {-1.333333327295e+00,-4.178725028461e-12,3.989099860473e-16},
	/* 138 */ {-1.333333333333e+00,8.170530616499e-12,-1.996570656494e-15},
	/* 139 */ {-1.333333342078e+00,1.257235193506e-12,-2.376334509575e-17},
	/* 140 */ {-1.333333324896e+00,-2.200506347397e-12,1.069326262877e-16},
	/* 141 */ {-1.333333337888e+00,4.279297543673e-13,-6.609176699882e-18},
	/* 142 */ {-1.333333336601e+00,6.660909887893e-14,-2.166884171107e-21},
	/* 143 */ {-1.333333306056e+00,-5.792986671114e-13,2.975852537137e-18},
	/* 144 */ {-1.333333338253e+00,2.821889411730e-14,1.313874873592e-22},
	/* 145 */ {-1.333333342143e+00,1.735989759173e-14,-2.920173744511e-23},
	/* 146 */ {-1.333333358899e+00,7.389128713195e-14,-4.739824689375e-20},
	/* 147 */ {-1.333333338877e+00,3.281325604376e-15,2.632397286286e-24},
	/* 148 */ {-1.333333311755e+00,-1.581175057675e-14,2.595013524022e-21},
	/* 149 */ {-1.333333321022e+00,-1.988605999125e-15,4.620579202933e-24},
	/* 150 */ {-1.333333333333e+00,2.185287327367e-16,-4.582225399172e-24},
	/* 151 */ {-1.333333348235e+00,6.154580132264e-16,1.262167297644e-24},
	/* 152 */ {-1.333333372490e+00,1.441282562803e-15,-1.347820587954e-23},
	/* 153 */ {-1.333333308836e+00,-5.290555683694e-16,2.642126576376e-24},
	/* 154 */ {-1.333333320723e+00,-1.617916714853e-16,4.165877422063e-25},
	/* 155 */ {-1.333333272412e+00,-3.308493711540e-16,4.165348444411e-25},
	/* 156 */ {-1.333333328917e+00,1.311608817649e-17,-1.819336035114e-26},
	/* 157 */ {-1.333333333338e+00,6.664594195340e-21,-2.029737568390e-30},
	/* 158 */ {-1.333333333338e+00,3.332503892823e-21,-5.076269850919e-31},
	/* 159 */ {-1.333333333338e+00,1.664597585186e-21,-1.269308203973e-31},
	/* 160 */ {-1.333333333338e+00,8.360728041380e-22,-3.195238148360e-32},
	/* 161 */ {-1.333333333338e+00,4.196907632941e-22,-8.058813111045e-33},
	/* 162 */ {-1.333333333338e+00,2.122235259081e-22,-2.061346893602e-33},
	/* 163 */ {-1.333333333338e+00,1.087419388080e-22,-5.436896471423e-34},
	/* 164 */ {-1.333333333338e+00,5.678788775633e-23,-1.535313171609e-34},
	/* 165 */ {-1.333333333337e+00,3.075270109365e-23,-5.099294493867e-35},
	/* 166 */ {-1.333333333331e+00,1.770521940033e-23,-2.293389478454e-35},
	/* 167 */ {-1.333333333309e+00,1.122994616769e-23,-1.473188295168e-35},
	/* 168 */ {-1.333333333218e+00,7.932532827413e-24,-1.206571482722e-35},
	/* 169 */ {-1.333333332856e+00,6.245052066062e-24,-1.110107477558e-35},
	/* 170 */ {-1.333333331403e+00,5.411005208190e-24,-1.070801885202e-35},
	/* 171 */ {-1.333333328141e+00,4.713879359915e-24,-5.104515380172e-36},
	/* 172 */ {-1.333333337335e+00,4.369759300396e-24,1.378636900000e-36},
	/* 173 */ {-1.333333330682e+00,3.111216923094e-24,-3.487741013312e-37},
	/* 174 */ {-1.333333337476e+00,-3.473916234653e-24,1.054656648163e-37},
	/* 175 */ {-1.333333329505e+00,-2.182658217889e-24,-1.589908896129e-38},
	/* 176 */ {-1.333333336057e+00,-1.128414128032e-23,1.009368407500e-38},
	/* 177 */ {-1.333333313241e+00,-2.586506067176e-23,7.925364322632e-39},
	/* 178 */ {-1.333333297774e+00,-2.160550737870e-23,2.939568402767e-39},
	/* 179 */ {-1.333333313149e+00,-6.191097450815e-24,4.586245210540e-40},
	/* 180 */ {-1.333333321693e+00,-6.835768718886e-25,2.188121385044e-43},
	/* 181 */ {-1.333333333272e+00,-2.065980820640e-27,-7.165233163286e-45},
	/* 182 */ {-1.333333318437e+00,-7.597835478790e-25,8.002476791036e-42},
	/* 183 */ {-1.333333318442e+00,-1.104171278579e-25,-8.049589977799e-44},
	/* 184 */ {-1.333333339200e+00,4.695044567077e-26,-9.189141418759e-44},
	/* 185 */ {-1.333333336404e+00,1.652249274305e-26,-2.055201291347e-44},
	/* 186 */ {-1.333333372920e+00,9.820658321120e-26,-5.694262164122e-44},
	/* 187 */ {-1.333333353423e+00,2.596832255683e-26,-7.736749642269e-45},
	/* 188 */ {-1.333333352884e+00,1.186366033233e-26,-1.707362674604e-45},
	/* 189 */ {-1.333333330206e+00,-4.165197883231e-28,-1.748897318955e-47},
	/* 190 */ {-1.333333360567e+00,4.546206738511e-27,-1.770641776211e-46},
	/* 191 */ {-1.333333356844e+00,1.712025313420e-27,-3.054757864880e-47},
	/* 192 */ {-1.333333355280e+00,8.971838611066e-28,-8.119680402948e-48},
	/* 193 */ {-1.333333322133e+00,-1.945370099908e-28,8.286234633322e-49},
	/* 194 */ {-1.333333315179e+00,-1.655930770905e-28,3.618190126479e-49},
	/* 195 */ {-1.333333333328e+00,-2.426047828920e-32,2.684955323805e-53},
	/* 196 */ {-1.333333333338e+00,1.208746735018e-32,-6.689711553227e-54},
	/* 197 */ {-1.333333333338e+00,6.028311189209e-33,-1.666692757427e-54},
	/* 198 */ {-1.333333333338e+00,3.008889379913e-33,-4.154783704233e-55},
	/* 199 */ {-1.333333333338e+00,1.499272514813e-33,-1.032721831392e-55},
	/* 200 */ {-1.333333333338e+00,7.462508336760e-34,-2.561144167757e-56},
	/* 201 */ {-1.333333333338e+00,3.679062219204e-34,-6.285245430640e-57},
	/* 202 */ {-1.333333333338e+00,1.786516314380e-34,-1.530861758354e-57},
	/* 203 */ {-1.333333333338e+00,8.419478287765e-35,-3.813541549964e-58},
	/* 204 */ {-1.333333333335e+00,3.679591191661e-35,-1.128942682201e-58},
	/* 205 */ {-1.333333333327e+00,1.325002538567e-35,-5.568140154480e-59},
	/* 206 */ {-1.333333333282e+00,1.476163765244e-36,-4.627841907063e-59},
	/* 207 */ {-1.333333333162e+00,-4.441899278170e-36,-4.618157978469e-59},
	/* 208 */ {-1.333333332636e+00,-7.734752828411e-36,-4.735325687997e-59},
	/* 209 */ {-1.333333330540e+00,-8.631801954882e-36,-4.832074128645e-59},
	/* 210 */ {-1.333333328680e+00,-8.671474889222e-36,-1.681281734694e-59},
	/* 211 */ {-1.333333330118e+00,-1.057724510549e-35,-3.533942924845e-60},
	/* 212 */ {-1.333333331264e+00,-1.543644487820e-35,-4.570614819802e-61},
	/* 213 */ {-1.333333339247e+00,-5.930368999898e-36,4.563261674152e-61},
	/* 214 */ {-1.333333332211e+00,-6.557054425681e-36,6.914035081726e-63},
	/* 215 */ {-1.333333339170e+00,7.428756955212e-36,1.130142039451e-62},
	/* 216 */ {-1.333333333007e+00,-3.454392187983e-37,-6.055858604458e-66},
	/* 217 */ {-1.333333316243e+00,-1.722863570819e-35,3.970043753736e-63},
	/* 218 */ {-1.333333304732e+00,-1.688601116004e-35,2.308114698866e-63},
	/* 219 */ {-1.333333407869e+00,2.188547117156e-35,-1.456791044223e-63},
	/* 220 */ {-1.333333291847e+00,-6.094303302897e-36,2.060729881218e-64},
	/* 221 */ {-1.333333343033e+00,6.188805488257e-37,-1.117680214791e-65},
	/* 222 */ {-1.333333344742e+00,4.059014872168e-37,-3.384544667779e-66},
	/* 223 */ {-1.333333329376e+00,-3.028446932393e-38,-1.112933442231e-67},
	/* 224 */ {-1.333333338006e+00,3.685806753457e-38,-6.646975454624e-68},
	/* 225 */ {-1.333333374600e+00,1.966010607575e-37,-2.162814923735e-67},
	/* 226 */ {-1.333333333328e+00,-1.209451716694e-41,6.357462322273e-72},
	/* 227 */ {-1.333333310981e+00,-2.644873123318e-38,6.954818566705e-69},
	/* 228 */ {-1.333333333347e+00,7.754465549736e-42,-1.019583172580e-72},
	/* 229 */ {-1.333333334114e+00,-4.114184675166e-40,6.265679940333e-71},
	/* 230 */ {-1.333333310982e+00,-3.306079033824e-39,1.086682267237e-70},
	/* 231 */ {-1.333333318435e+00,-1.102068323106e-39,1.811135798298e-71},
	/* 232 */ {-1.333333355685e+00,8.265194579127e-40,-6.791759272636e-72},
	/* 233 */ {-1.333333377104e+00,8.390817455865e-40,-3.560818241256e-72},
	/* 234 */ {-1.333333333328e+00,-4.415909484475e-44,8.896278365786e-77},
	/* 235 */ {-1.333333333338e+00,2.204171324695e-44,-2.221662794556e-77},
	/* 236 */ {-1.333333333338e+00,1.102564621784e-44,-5.561009415054e-78},
	/* 237 */ {-1.333333333338e+00,5.527534005883e-45,-1.395787007684e-78},
	/* 238 */ {-1.333333333338e+00,2.773517248604e-45,-3.514834683010e-79},
	/* 239 */ {-1.333333333338e+00,1.398989195615e-45,-8.930394710816e-80},
	/* 240 */ {-1.333333333338e+00,7.090310222933e-46,-2.305899897778e-80},
	/* 241 */ {-1.333333333338e+00,3.667033182246e-46,-6.259944784552e-81},
	/* 242 */ {-1.333333333338e+00,1.946627993654e-46,-1.894172503984e-81},
	/* 243 */ {-1.333333333334e+00,1.085570114651e-46,-7.260596791226e-82},
	/* 244 */ {-1.333333333323e+00,6.532236951465e-47,-3.953967673410e-82},
	/* 245 */ {-1.333333333277e+00,4.383868677493e-47,-2.941668301206e-82},
	/* 246 */ {-1.333333333090e+00,3.344163205267e-47,-2.594851608616e-82},
	/* 247 */ {-1.333333332338e+00,2.719003539577e-47,-2.457476165447e-82},
	/* 248 */ {-1.333333329328e+00,2.369005000790e-47,-2.400370448439e-82},
	/* 249 */ {-1.333333329916e+00,2.497564983346e-47,-5.110848108394e-83},
};

float one53 (float x)
{
	union { float f; uint32 i; } u;
	u.f = x;
	int e = (u.i & 0x7F800000) >> 23;
	float *p = one5tbl3[e];
	x = fabs (x);
	x = p[0]+x*(p[1]+x*p[2]);
	u.i &= 0x80000000;
	u.i |= 127 << 23; /* 1.0 - use 126 for .5 instead, etc */
	return x * u.f;
}


#if 0
template <int N>
class ATan
{
	public:
		float v[N], over_N;

		ATan()
			{
				over_N = 1. / N;
				float x = 0;
				float dx = 50. * over_N;
				for (int i = 0; i < N; ++i, x += dx)
					v[i] = atanf (x);
			}

		float get (float x)
			{
				int i;
				x *= N * .02;
				float s = sgn (x);
				x = fabs (x);
				fistp (x, i);
				x -= i;
				x = (1-x) * v[i] + x * v[i+1];
				return s*x;
			}
};
#endif

}; /* namespace Polynomial */
}; /* namespace DSP */

