#include <stdio.h>
#include <inttypes.h>
#include <stdlib.h>

#include <prscfl.h>


void 
dumpStructName(FILE *fh, ParamDef *def, char *delim) {
	if (def && def->parent) {
		ParamDef	*p = def->parent;

		if (p->name == NULL)
			p = p->parent;

		if (p != NULL) {
			dumpStructName(fh, p, delim);
			fputs(delim, fh);
			fputs(p->name, fh);
		}
	}
}

static void
dumpComment(FILE *fh, ParamDef *def, int istab) {
	if (def->comment) {
		ParamDef	*i = def->comment;

		if (i->next) {
			/* multiline comment */
			fprintf(fh, "\n%s/*\n", istab ? "\t" : "");
			while(i) {
				fprintf(fh, "%s * %s\n", istab ? "\t" : "", i->paramValue.commentval);
				i = i->next;
			}
			fprintf(fh, "%s */\n", istab ? "\t" : "");
		} else {
			/* single line comment */
			fprintf(fh, "\n%s/* %s */\n", istab ? "\t" : "", i->paramValue.commentval);
		}
	}
}

static void
dumpParamDef(FILE *fh, char* name, ParamDef *def) {
	
	dumpComment(fh, def, 1);
	
	switch(def->paramType) {
		case	int32Type:
			fprintf(fh, "\tint32_t\t%s;\n", def->name);
			break;
		case	uint32Type:
			fprintf(fh, "\tu_int32_t\t%s;\n", def->name);
			break;
		case	int64Type:
			fprintf(fh, "\tint64_t\t%s;\n", def->name);
			break;
		case	uint64Type:
			fprintf(fh, "\tu_int64_t\t%s;\n", def->name);
			break;
		case	doubleType:
			fprintf(fh, "\tdouble\t%s;\n", def->name);
			break;
		case	stringType:
			fprintf(fh, "\tchar*\t%s;\n", def->name);
			break;
		case	boolType:
			fprintf(fh, "\tconfetti_bool_t\t%s;\n", def->name);
			break;
		case	commentType:
			fprintf(stderr, "Unexpected comment"); 
			break;
		case	structType:
			fprintf(fh, "\t%s", name);
			dumpStructName(fh, def->paramValue.structval, "_");
			fprintf(fh, "*\t%s;\n", def->name);
			break;
		case	arrayType:
			fprintf(fh, "\t%s", name);
			dumpStructName(fh, def->paramValue.arrayval->paramValue.structval, "_");
			fprintf(fh, "**\t%s;\n", def->name);
			break;
		case 	builtinType:
			break;
		default:
			fprintf(stderr,"Unknown paramType (%d)\n", def->paramType);
			exit(1);
	}
}

static void
dumpParamList(FILE *fh, char* name, ParamDef *def) {
	while(def) {
		dumpParamDef(fh, name, def);
		def = def->next;
	}
}

static void
dumpStruct(FILE *fh, char* name, ParamDef *def) {
	ParamDef *list = NULL;

	switch(def->paramType) {
		case structType:
			list = def->paramValue.structval;
			break;
		case arrayType:
			list = def->paramValue.arrayval->paramValue.structval;
			break;
		default:
			fprintf(stderr,"Non-struct paramType (%d)\n", def->paramType);
			exit(1);
			break;
	}

	fprintf(fh, "typedef struct %s", name);
	dumpStructName(fh, list, "_");
	fputs(" {\n", fh);
	fputs("\tunsigned char __confetti_flags;\n\n", fh);
	dumpParamList(fh, name, list);
	fprintf(fh, "} %s", name);
	dumpStructName(fh, list, "_");
	fputs(";\n\n", fh);
}

static void
dumpRecursive(FILE *fh, char* name, ParamDef *def) {

	while(def) {
		switch(def->paramType) {
			case structType:
				dumpRecursive(fh, name, def->paramValue.structval);
				dumpStruct(fh, name, def);
				break;
			case arrayType:
				dumpComment(fh, def->paramValue.arrayval, 0);
				dumpRecursive(fh, name, def->paramValue.arrayval->paramValue.structval);
				dumpStruct(fh, name, def);
				break;
			default:
				break;
		}

		def = def->next;
	}
}

void 
hDump(FILE *fh, char* name, ParamDef *def) {
	ParamDef	root;

	root.paramType = structType;
	root.paramValue.structval = def;
	root.name = NULL;
	root.parent = NULL;
	root.next = NULL;
	def->parent = &root;

	fprintf(fh, "#ifndef %s_CFG_H\n", name);
	fprintf(fh, "#define %s_CFG_H\n\n", name);
	
	fputs(
		"#include <stdio.h>\n"
		"#include <stdbool.h>\n"
		"#include <sys/types.h>\n\n"
		"#ifndef confetti_bool_t\n"
		"#define confetti_bool_t char\n"
		"#endif\n\n"
		"/*\n"
		" * Autogenerated file, do not edit it!\n"
		" */\n\n",
		fh
	);

	dumpRecursive(fh, name, &root);

	fprintf(fh, 
		"#ifndef CNF_FLAG_STRUCT_NEW\n"
		"#define CNF_FLAG_STRUCT_NEW\t0x01\n"
		"#endif\n"
		"#ifndef CNF_FLAG_STRUCT_NOTSET\n"
		"#define CNF_FLAG_STRUCT_NOTSET\t0x02\n"
		"#endif\n"
		"#ifndef CNF_STRUCT_DEFINED\n"
		"#define CNF_STRUCT_DEFINED(s) ((s) != NULL && ((s)->__confetti_flags & CNF_FLAG_STRUCT_NOTSET) == 0)\n"
		"#endif\n\n"
	);

	fprintf(fh, "void init_%s(%s *c);\n\n", name, name);
	fprintf(fh, "int fill_default_%s(%s *c);\n\n", name, name);
	fprintf(fh, "void swap_%s(struct %s *c1, struct %s *c2);\n\n",
		name, name, name);
	fprintf(fh, "void parse_cfg_file_%s(%s *c, FILE *fh, int check_rdonly, int *n_accepted, int *n_skipped, int *n_optional);\n\n", name, name);
	fprintf(fh, "void parse_cfg_buffer_%s(%s *c, char *buffer, int check_rdonly, int *n_accepted, int *n_skipped, int *n_optional);\n\n", name, name);
	fprintf(fh, "int check_cfg_%s(%s *c);\n\n", name, name);
	fprintf(fh, "int dup_%s(%s *dst, %s *src);\n\n", name, name, name);
	fprintf(fh, "void destroy_%s(%s *c);\n\n", name, name);
	fprintf(fh, "char *cmp_%s(%s* c1, %s* c2, int only_check_rdonly);\n\n", name, name, name);
	fprintf(fh, "typedef struct %s_iterator_t %s_iterator_t;\n", name, name);
	fprintf(fh, "%s_iterator_t* %s_iterator_init();\n", name, name);
	fprintf(fh, "char* %s_iterator_next(%s_iterator_t* i, %s *c, char **v);\n\n", name, name, name);

	fputs("#endif\n", fh);

	def->parent = NULL;
}
