/* $Id: model.c,v 1.99 2003/06/10 11:56:37 paul Exp $ */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>

#include <SDL.h>
#include <SDL_thread.h>

#include "global.h"
#include "model.h"
#include "object.h"
#include "food.h"
#include "player.h"
#include "collision.h"

#define MDEBUG(...)      DEBUG(DMODEL, "Model", __VA_ARGS__)

/* String mappings for enums */
char *event_type_string[] = {
  "throw", "jump", "deliver", "pickup", "hit", "powerup",
};

char *object_type_string[] = {
  "banana", "pepper", "pumpkin", "cucumber", "orange", "cherries",
  "powerup", "player", "home", "box", "#types"
};

char food_type_short_string[] = {
  'B', 'P', 'p', 'C', 'O', 'c'
};

char *player_state_string[] = {
  "normal", "unconscious", "dead", "powerup"
};

char *food_state_string[] = {
  "normal", "powerup", "foodsalad"
};

char *graphics_state_string[] = {
  "normal", "invisible", "deleted"
};

char *look_dir_string[] = {
  "left", "right"
};


/*****************************************************************************/
/* Simple helper functions                                                   */
/*****************************************************************************/

/* Search Listfunc to find certain element in a List using the user data
 * field. Index is zero-based (e.g. 0 is the first node of the list */
static int
find_with_index(void *node, void *num) {
  return !(*((int *)num))--;
}

/* Find the object that is not dead. */
static int
find_alive(void *object, void *data) {
  Model_object *o = (Model_object *)object;
  
  return !IS_UNCONSCIOUS(o) || !IS_DEAD(o);
}

/* Dumb random number generator using the random seed of model, returns
 * integer with maximum value \c maxnum - 1. */
static int
random_number(Model *m, int maxnum) {
  int r;

  /* Manipulate current seed, do some calculations, add framecounter for
   * randomness and save new value. */
  r = m->random_seed = ((m->random_seed + m->framecounter) * 10) / 7;
  m->random_seed %= 1000;
  r %= maxnum;
  
  return r;
}

/* Returns a random spawn location (vector) for food or NULL if there
 * aren't defined any. */
static Vector *
random_foodplace(Model *m) {
  int len, r;

  len = list_length(m->foodplaces);
  if (len == 0)
    return NULL;
  else {
    r = random_number(m, len);

    return (Vector *)list_search(m->foodplaces, find_with_index, &r);
  }
}

/* Returns the one player that survived all others, return NULL
 * if not found. */
static Model_object *
find_surv_player(Model *m) {
  Model_object *p;

  if ((p = list_search(m->act_players,  find_alive, NULL)) ||
      (p = list_search(m->pas_players,  find_alive, NULL)) ||
      (p = list_search(m->act_uplayers, find_alive, NULL)) ||
      (p = list_search(m->pas_uplayers, find_alive, NULL)))
    return p;

  return NULL;
}

/* Tries to spawn all requested food (m->spawn_food) and returns number
 * of food objects succesfully spawned. */
static int
spawn_food(Model *m) {
  Object_type t;
  int32_t i, sf, nf;
  Vector *size, *pos;
  int32_t spawned = 0;
  
  /* Calculate amount of food in level and spawn requested food if
   * possible/allowed. */
  nf = list_length(m->act_food) + list_length(m->pas_food);

  /* Calculate requested food. */
  sf = m->spawn_food = MAX(MIN(m->max_food - nf, m->spawn_food), 0);

  /* Spawn requested food */
  for (i = 0; i < sf; i++) {
    t = random_number(m, (m->settings->spawn_powerups ? OT_POWERUP 
						      : OT_NRFOOD) + 1); 
    pos = random_foodplace(m);
    size = m->reg_types[t];

    /* Check if food can be spawned, if not.. skip that place. */
    if (pos != NULL && food_generate(m, t, NULL, pos, size) != NULL)
	spawned++;
  }

  return spawned;
}


/*****************************************************************************/
/* Model settings ADT                                                        */
/*****************************************************************************/

Model_settings *
new_model_settings(void) {
  Model_settings *ms;

  ms = malloc(sizeof(Model_settings));
  assert(ms != NULL);

  /* Default game settings. */
  ms->spawn_powerups         = TRUE;
  ms->spawn_constant         = FALSE;
  ms->spawn_deliver          = TRUE;
  ms->spawn_hit              = TRUE;
  ms->spawn_pickup_powerup   = TRUE;
  ms->min_food               = MODEL_MIN_FOOD;
  ms->max_food               = MODEL_MAX_FOOD;
  ms->init_food_spawn        = FD_INIT_SPAWN;
  ms->spawn_delay            = MODEL_SPAWN_DELAY;
  ms->spawn_constant_delay   = MODEL_CSPAWN_DELAY;
  ms->starvation_delay       = MODEL_STARV_DELAY;
  ms->starvation_amount      = MODEL_STARV_AMNT;

  /* Default global model settings. */
  ms->random_seed            = 0;
  ms->gravity                = new_vector(0, OBJ_Y_GRAV_ACCEL);
  ms->bounciness             = new_vector(OBJ_X_BOUNCINESS, OBJ_Y_BOUNCINESS);

  /* Default player settings. */
  ms->player_max_life        = PL_MAX_LIFE;
  ms->player_ground_friction = PL_X_FRICT_GROUND;
  ms->player_air_friction    = PL_X_FRICT_AIR;
  ms->player_ground_accel    = PL_X_ACCEL_GROUND;
  ms->player_air_accel       = PL_X_ACCEL_AIR;
  ms->player_jump_speed      = new_vector(PL_X_SPEED_JUMP, PL_Y_SPEED_JUMP);
  ms->player_max_speed       = new_vector(PL_X_MAX_SPEED, OBJ_Y_MAX_SPEED);
  ms->foodstack_size         = PL_MAX_FOOD;
  ms->powerup_length         = PL_MAX_POWEREDUP;
  ms->unconc_length          = PL_MAX_UNCONC;

  /* Default food settings. */
  ms->food_ground_friction   = FD_X_FRICT_GROUND;
  ms->food_air_friction      = FD_X_FRICT_AIR;
  ms->food_max_speed         = new_vector(FD_X_MAX_SPEED, OBJ_Y_MAX_SPEED);
  ms->food_throw_speed       = new_vector(FD_X_SPEED_THROW, FD_Y_SPEED_THROW);
  ms->food_deliver_life      = FD_INC_LIFE;

  return ms;
}

Model_settings *
copy_model_settings(Model_settings *ms) {
  Model_settings *new_ms;

  assert(ms != NULL);

  new_ms = malloc(sizeof(Model_settings));
  assert(new_ms != NULL);

  /* Copy all entire Model_settings data, but create new
   * vectors. */
  memcpy(new_ms, ms, sizeof(Model_settings));
  new_ms->gravity           = copy_vector(ms->gravity);
  new_ms->bounciness        = copy_vector(ms->bounciness);
  new_ms->player_jump_speed = copy_vector(ms->player_jump_speed);
  new_ms->player_max_speed  = copy_vector(ms->player_max_speed);
  new_ms->food_max_speed    = copy_vector(ms->food_max_speed);
  new_ms->food_throw_speed  = copy_vector(ms->food_throw_speed);

  return new_ms;
}

void
del_model_settings(Model_settings *ms) {
  /* Free the vectors. */
  del_vector(ms->gravity);
  del_vector(ms->bounciness);
  del_vector(ms->player_jump_speed);
  del_vector(ms->player_max_speed);
  del_vector(ms->food_max_speed);
  del_vector(ms->food_throw_speed);
  
  free(ms);
}


/*****************************************************************************/
/* Model API for main                                                        */
/*****************************************************************************/

Model *
new_model(int32_t width, int32_t height, Model_settings *ms) {
  int i;
  Model *m;
  
  m = malloc(sizeof(Model));
  assert(m != NULL);

  /* Initialize with default values. */
  /* Public */
  m->objects      = new_list();
  m->width        = width;
  m->height       = height;
  m->framecounter = 0;
  /* Private */
  m->alive        = 0;
  m->settings     = copy_model_settings(ms);
  m->lock         = SDL_CreateMutex();
  m->changed      = SDL_CreateMutex();
  m->homes        = new_list();        
  m->act_uplayers = new_list();        
  m->pas_uplayers = new_list();        
  m->act_food     = new_list();        
  m->pas_food     = new_list();        
  m->boxes        = new_list();        
  m->act_players  = new_list();        
  m->pas_players  = new_list();        
  m->foodplaces   = new_list();
  m->min_food     = 0;
  m->max_food     = 0;
  m->starvation   = -1;
  m->spawn_food   = 0;
  m->random_seed  = ms->random_seed;
  
  /* Register default size for all types. */
  for (i = 0; i < OT_NRTYPES; i++)
    m->reg_types[i] = new_vector(OBJ_X_SIZE, OBJ_Y_SIZE);
    
  /* No reading until first tick. */
  if (SDL_mutexP(m->changed) == -1)
    WARN("Couldn't block model reads until first tick!");

  return m;
}

void
del_model(Model *m) {
  int i;
  Vector *v;
  Model_object *o;
  
  /* Remove all objects (without graphics data) from model. */
  while ((o = (Model_object *)list_pop(m->objects)))
    if (o->gfx_data == NULL) {
      /* GFX data is cleared out, object needs to be deleted, force it. */
      o->gfx_state = GFX_DELETED;
      del_object(m, o);
    }
  del_list(m->objects);

  /* Remove default settings. */
  del_model_settings(m->settings);

  /* Destory locks. */
  SDL_DestroyMutex(m->lock);
  SDL_DestroyMutex(m->changed);

  /* Now all objects are removed, remove the sublist. */
  del_list(m->homes);
  del_list(m->act_uplayers);
  del_list(m->pas_uplayers);
  del_list(m->act_food);
  del_list(m->pas_food);
  del_list(m->boxes);
  del_list(m->act_players);
  del_list(m->pas_players);

  /* Clear out food spawn places. */
  while ((v = (Vector *)list_pop(m->foodplaces)))
    del_vector(v);
  del_list(m->foodplaces);

  /* Remove registered types (size vectors). */
  for (i = 0; i < OT_NRTYPES; i++) 
    del_vector(m->reg_types[i]);

  free(m);
}

void
reset_model(Model *m) {
  Vector *v;
  
  /* Mark all objects as deleted. */
  list_foreach(m->objects, object_set_deleted, NULL);

  /* Clear out food spawn places. */
  while ((v = (Vector *)list_pop(m->foodplaces)))
    del_vector(v);

  m->framecounter = 0;
}

int 
model_register_type(Model *m, Object_type type, 
                    int32_t width, int32_t height) {
  /* Make sure type is not out of bound. */
  assert(type < OT_NRTYPES);

  /* Update size vector. */
  (m->reg_types[type])->x = width;
  (m->reg_types[type])->y = height;

  return TRUE;
}

Model_object * 
model_add_object(Model *m, Object_type type, Model_object *owner, 
                 int32_t x, int32_t y, int32_t width, int32_t height) {
  Vector *v;
  Model_object *o;

  /* Create new object. */
  o = new_object(m, type, owner, x, y, width, height);
  
  if (IS_FOOD(o)) {
    /* If object type is food, at create location to foodplaces
     * for spawning random food. */
    v = new_vector(x, y);
    list_append(m->foodplaces, v);
  }
  else if (IS_PLAYER(o)) {
    /* If object type is player, request food objects to be spawned
     * at start of game and increase the maximum food. */
    m->min_food   += m->settings->min_food;
    m->max_food   += m->settings->max_food;
    m->spawn_food += m->settings->init_food_spawn;
    m->starvation += m->settings->starvation_amount;
  }

  return o;
}


/*****************************************************************************/
/* Model API for IO */
/*****************************************************************************/

/* Handles one tick. Calls object tick function for all objects and sets
 * all new positions afterwards when all new desired positions are known
 * and collisionless. */
int
model_tick(Model *m) {
  Model_object *winner;
  
  MDEBUG("Frame %d:", m->framecounter);

  /* Settings re set. */
  assert(m->settings != NULL);

  /* Lock the model. */
  if (SDL_mutexP(m->lock) == -1)
    WARN("Couldn't lock model!");

  /* Model is dead if all players are not alive which is
     updated in player_tick. */
  m->alive = 0;

  /* Look for objects that need to be cleaned up. */
  list_foreach(m->objects,      object_check_deleted, m);

  /* Update all non-scenery objects, determine desired new position.
   * In case of player, update score, life an consciousness. */
  list_foreach(m->pas_uplayers, object_tick, m);
  list_foreach(m->act_uplayers, object_tick, m);
  list_foreach(m->pas_players,  object_tick, m);
  list_foreach(m->act_players,  object_tick, m);
  list_foreach(m->pas_food,     object_tick, m); 
  list_foreach(m->act_food,     object_tick, m); 

  /* Update all passive non-scenery objects and check if they
   * can become active. */
  list_foreach(m->pas_players,  object_passive_tick, m);
  list_foreach(m->pas_uplayers, object_passive_tick, m);
  list_foreach(m->pas_food,     object_passive_tick, m); 

  /* Do collision resolving */
  resolve_collisions(m);

  /* Update all active non-scenery objects. */
  list_foreach(m->act_players,  object_active_tick, m);
  list_foreach(m->act_uplayers, object_active_tick, m);
  list_foreach(m->act_food,     object_active_tick, m); 

  /* Decrement maximum food if starvation delay has expired. */
  if (m->framecounter % m->settings->starvation_delay == 0) {
    m->max_food = MAX(m->max_food - m->starvation, 
                      m->min_food);
  }

  /* Spawn food constantly every spawn_constant_delay frames (if enabled). */
  if (m->settings->spawn_constant && 
      m->framecounter % m->settings->spawn_constant_delay == 0)
    m->spawn_food++;
  
  if (m->framecounter % m->settings->spawn_delay == 0)
    m->spawn_food -= spawn_food(m);

  /* Update frame counter */
  m->framecounter++;
  
  /* Signify model has changed/been revised */
  if (SDL_mutexV(m->changed) == -1)
    WARN("Couldn't mark model as changed!");

  /* Unlock the model */
  if (SDL_mutexV(m->lock) == -1)
    WARN("Couldn't unlock model!");

  /* Model is alive if at least 2 players are alive. */
  if (m->alive <= 1) {
    MDEBUG("Game model is dead, only 1 or no players alive.");

    if (m->alive == 1) {
      /* One player has survived the rest, update score. */
      winner = find_surv_player(m);
      assert(winner != NULL);

      winner->score += SC_WIN;
    }
  }

  return m->alive > 1;
}

void
model_object_move_x(Model_object *o, int x_move) {
  /* Invariant: o->left || o->right || (!o->left && !o->right) */
  assert(o->left || o->right || (!o->left && !o->right));

  /* Set flags for left/right movement. */
  if (x_move < 0) {
    /* Moving to the left. */
    o->left = TRUE;
    o->right = FALSE;
  }
  else if (x_move > 0) {
    /* Moving to the right. */
    o->left = FALSE;
    o->right = TRUE;
  }
  else 
    /* Stop! */
    o->left = o->right = FALSE;

  if (o->left || o->right)
    MDEBUG("Player moves to the %s.", o->left ? "left" : "right");
}

void
model_object_move_y(Model_object *o, int y_move) {
  /* Set flag for jumping. */
  if (y_move > 0) o->jump = TRUE;
  else            o->jump = FALSE;
  
  if (o->jump)
    MDEBUG("Player jumps.");
}

void
model_object_throw(Model_object *o) {
  /* Set flag for action. This flag is automatically set to FALSE
   * when action is handled. */
  o->action = TRUE;

  MDEBUG("Player does action.");
}


/*****************************************************************************/
/* Model API for GFX                                                         */
/*****************************************************************************/

void
model_object_set_gfx_data(Model_object *o, void *data) {
  o->gfx_data = data;
}

void *
model_object_get_gfx_data(Model_object *o) {
  return o->gfx_data;
}

void
model_foreach_object(Model *m, ListFunc func, void *user_data) {
  /* Require that model has been changed. */
  if (SDL_mutexP(m->changed) == -1)
    WARN("Couldn't detect change in foreach_object!");

  /* Lock the model. */
  if (SDL_mutexP(m->lock) == -1)
    WARN("Couldn't lock model in foreach_object!");
  
  /* Iterate sublists in logical order, by doing all lists this
   * is the same as iterating all objects. */
  list_foreach(m->homes, func, user_data);
  list_foreach(m->act_uplayers, func, user_data);
  list_foreach(m->pas_uplayers, func, user_data);
  list_foreach(m->act_food, func, user_data);
  list_foreach(m->pas_food, func, user_data);
  list_foreach(m->boxes, func, user_data);
  list_foreach(m->act_players, func, user_data);
  list_foreach(m->pas_players, func, user_data);

  /* Unlock the model. */
  if (SDL_mutexV(m->lock) == -1)
    WARN("Couldn't unlock model in foreach_object!");
}
