/******************************************************************************
 * This file is part of MPlayer Audio Visualization.                          *
 *                                                                            *
 *                                                                            *
 *  MPlayer Audio Visualization is free software; you can redistribute it     *
 *  and/or modify it under the terms of the GNU General Public License as     *
 *  published by the Free Software Foundation; either version 2 of the        *
 *  License, or (at your option) any later version.                           *
 *                                                                            *
 *  MPlayer Audio Visualization is distributed in the hope that it will be    *
 *  useful, but WITHOUT ANY WARRANTY; without even the implied warranty of    *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General *
 *  Public License for more details.                                          *
 *                                                                            * 
 *  You should have received a copy of the GNU General Public License         *
 *  along with MPlayer Audio Visualization; if not, write to the Free         *
 *  Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA         *
 *  02111-1307  USA                                                           *
 ******************************************************************************/
#define _C_SLAVE_
#include <slave.h>

int slave_init()
{
  mpav_fmsg( stderr, 
	     "# Welcome to MPlayer Audio Visualization.\n"
	     "# This is GNU GPL software.\n"
	     "# This is free software; see the source for copying conditions.\n"
	     "# There is NO warranty; not even for MERCHANTABILITY or "
	     "FITNESS FOR A PARTICULAR PURPOSE.\n"
	     "# Type \"help\" if you need some...\n" );
  if ( ( slave_thread = SDL_CreateThread( (void*)__slave_thread_func, NULL ) ) == NULL )
    return( -1 );

  mpav_debug( "Slave thread started...\n" );

  return( 0 );
}

void slave_quit()
{
  SDL_WaitThread( slave_thread, NULL );
  mpav_debug( "Slave thread finished!\n" );
}

void __slave_thread_func()
{
  char *line = NULL;
  char *s = NULL;

  __slave_initialize_readline();

  while ( ! the_end )
    {
      mpav_debug( "Slave loop...\n" );
      if ( ( line = readline( NULL ) ) == NULL )
	{
	  __slave_com_quit( NULL ); // do the right thing to terminate process
	  break;
	}

      s = __slave_stripwhite( line );

      if ( *s )
	{
	  add_history( s );
	  __slave_execute_line( s );
	}

      free( line );
    }
}


char *__slave_get_word( char *s )
{
  char *w = NULL;
  char is_quote=0;
  int i=0;

  // find word (skip white spaces)
  while ( ( s[ i ] ) && ( __slave_whitespace( s[ i ] ) ) )
    i++;

  // there is a word?
  if ( ! s[ i ] )
    return( NULL );

  // is it a quote?
  if ( ( ! is_quote ) && ( ( s[ i ] == '\'' ) || ( s[ i ] == '"' ) ) )
    is_quote = s[ i++ ];

  w = s + i; // mark word's start

  // find the word's end
  while ( 
	 ( ( is_quote ) || ( ! __slave_whitespace( s[ i ] ) ) ) && 
	 ( is_quote != s[ i ] ) 
	 )
    i++;

  // mark word's end
  s[ i ] = 0;

  return( w );
}


/* Code from GNU Readline Library Documentation */
int  __slave_execute_line( char *line )
{
  register int i;
  command_t *command;
  char *word;

  /* Isolate the command word. */
  i = 0;
  while ( ( line[ i ] ) && ( __slave_whitespace( line[ i ] ) ) )
    i++;
  word = line + i;

  while ( ( line[i] ) && ( ! __slave_whitespace( line[ i ] ) ) )
    i++;

  if ( line[ i ] )
    line[ i++ ] = '\0';

  command = __slave_find_command( word );

  if ( ! command )
    {
      mpav_fmsg( stderr, "%s: No such command.\n", word );
      return( -1 );
    }

  /* Get argument to command, if any. */
  while ( __slave_whitespace( line[ i ] ) )
    i++;

  word = line + i;

  /* Call the function. */
  return( ( *( command->func ) )( word ) );
}

command_t *__slave_find_command( char *name )
{
  register int i;

  for ( i = 0; commands[ i ].name; i ++ )
    if ( strcmp( name, commands[ i ].name ) == 0 )
      return( &commands[ i ] );

  return( ( command_t * ) NULL );
}

char *__slave_stripwhite( char *string )
{
  register char *s, *t;

  for ( s = string; __slave_whitespace( *s ); s ++ );
    
  if ( ! *s )
    return( s );

  t = s + strlen( s ) - 1;
  while ( ( t > s ) && ( __slave_whitespace( *t ) ) )
    t --;
  *( ++ t ) = 0;

  return s;
}





/* **************************************************************** */
/*                                                                  */
/*                  Interface to Readline Completion                */
/*                                                                  */
/* **************************************************************** */


void __slave_initialize_readline()
{
  /* Allow conditional parsing of the ~/.inputrc file. */
  rl_readline_name = "mpav";

  /* Tell the completer that we want a crack first. */
  rl_attempted_completion_function = __slave_completion;
}

char **__slave_completion( const char *text, int start, int end )
{
  char **matches = NULL;

  /* If this word is at the start of the line, then it is a command
     to complete.  Otherwise it is the name of a file in the current
     directory. */
  if ( start == 0 )
    matches = rl_completion_matches( text, __slave_command_generator );

  return( matches );
}


char *__slave_command_generator( const char *text, int state )
{
  static int list_index=0, len;
  char *name;
  
  /* If this is a new word to complete, initialize now.  This
     includes saving the length of TEXT for efficiency, and
     initializing the index variable to 0. */
  if ( ! state )
    {
      list_index = 0;
      len = strlen( text );
    }

  /* Return the next name which partially matches from the
     command list. */
  while ( ( name = commands[ list_index ].name ) != NULL )
    {
      list_index ++;

      if ( strncmp( name, text, len ) == 0 )
        return( strdup( name ) );
    }

  /* If no names matched, then return NULL. */
  return( ( char * ) NULL );
}


int __slave_com_help( char *argument )
{
  register int i;
  int printed = 0;

  for ( i = 0; commands[ i ].name; i ++ )
    {
      if ( ( ! *argument ) || ( strcmp( argument, commands[ i ].name ) == 0 ) )
        {
          printf( "%s\t\t%s.\n", commands[ i ].name, commands[ i ].doc );
          printed ++;
        }
    }

  if ( ! printed )
    {
      printf( "No commands match `%s'.  Possibilties are:\n", argument );

      for ( i = 0; commands[ i ].name; i ++ )
        {
          /* Print in six columns. */
          if ( printed == 6 )
            {
              printed = 0;
              printf( "\n" );
            }

          printf( "%s\t", commands[ i ].name );
          printed ++;
        }

      if ( printed )
        printf( "\n" );
    }
  return( 0 );
}

int __slave_com_ir( char *argument )
{
  __slave_post_event( EVENT_INCREASE_RESOLUTION );
  return( 0 );
}

int __slave_com_dr( char *argument )
{
  __slave_post_event( EVENT_DECREASE_RESOLUTION );
  return( 0 );
}

int __slave_com_sr( char *argument )
{
  int w, h, n;
  // cmdline: <width> <height>
  if ( ! argument ) 
    {
      __slave_command_error_noargs( "cm" );
      return( -1 );
    }
  
  n = sscanf( argument, "%d %d", &w, &h );
  
  if ( n < 1 )
    {
      __slave_command_error_noargs( "sr" );
      return( -1 );
    }
  else if ( n == 1 )
    {
      __slave_command_error_misarg( "sr", "height" );
      return( -1 );
    }
  
  if ( w <= 0 )
    {
      mpav_fmsg( stderr, "ERROR: argument 'width' for 'sr' must be greater than zero! ('%d' was provided)\n", w );
      return( -1 );
    }

  if ( h <= 0 )
    {
      mpav_fmsg( stderr, "ERROR: argument 'height' for 'sr' must be greater than zero! ('%d' was provided)\n", h );
      return( -1 );
    }

  return screen_set_resolution( w, h );
}


int __slave_com_fs( char *argument )
{
  __slave_post_event( EVENT_TOGGLE_FULLSCREEN );
  return( 0 );
}

int __slave_com_m( char *argument )
{
  __slave_post_event( EVENT_TOGGLE_MESSAGES );
  return( 0 );
}

int __slave_com_quit( char *argument )
{
  __slave_post_event( EVENT_QUIT );
  the_end = 1; // be sure we don't enter the readline again
  return( 0 );
}

int __slave_com_cm( char *argument )
{
  msgnode_t *node = NULL;
  // cmdline: <fontfile> <ptsize> <x> <y> <fgcolor> <msg>
  char *font = NULL;
  char *msg  = NULL;
  char *w    = NULL;
  int ptsize = 0;
  int x = 0;
  int y = 9;
  Uint32 fgcolor = 0;

  if ( ! argument )
    {
      __slave_command_error_noargs( "cm" );
      return( -1 );
    }

  // font
  font = __slave_get_word( argument );
  if ( ! font )
    {
      __slave_command_error_misarg( "cm", "fontfile" );
      return( -1 );
    }

  // ptsize
  w = __slave_get_next_word( font );
  if ( ! w )
    {
      __slave_command_error_misarg( "cm", "ptsize" );
      return( -1 );
    }
  ptsize = ( int ) strtol( w, NULL, 10 );
  if ( ptsize <= 0 )
    {
      mpav_fmsg( stderr, "ERROR: argument 'ptsize' for 'cm' must be greater than zero! ('%d' was provided)\n", ptsize );
      return( -1 );
    }
  

  // x
  w = __slave_get_next_word( w );
  if ( ! w )
    {
      __slave_command_error_misarg( "cm", "x" );
      return( -1 );
    }
  x = ( int ) strtol( w, NULL, 10 );
  if ( x <= 0 )
    {
      mpav_fmsg( stderr, "ERROR: argument 'x' for 'cm' must be greater than zero! ('%d' was provided)\n", x );
      return( -1 );
    }
  
  // y
  w = __slave_get_next_word( w );
  if ( ! w )
    {
      __slave_command_error_misarg( "cm", "y" );
      return( -1 );
    }
  y = ( int ) strtol( w, NULL, 10 );
  if ( y <= 0 )
    {
      mpav_fmsg( stderr, "ERROR: argument 'y' for 'cm' must be greater than zero! ('%d' was provided)\n", y );
      return( -1 );
    }

  // fgcolor
  w = __slave_get_next_word( w );
  if ( ! w )
    {
      __slave_command_error_misarg( "cm", "fgcolor" );
      return( -1 );
    }
  fgcolor = ( Uint32 ) strtol( w, NULL, 16 );
  if ( fgcolor <= 0 )
    {
      mpav_fmsg( stderr, "ERROR: argument 'fgcolor' for 'cm' must be in hexa RGB (ie: 0xff0000 is red)! ('0x%x' was provided)\n", fgcolor );
      return( -1 );
    }

  // msg
  msg = w + 1 + strlen( w );
  if ( ! msg )
    {
      __slave_command_error_misarg( "cm", "msg" );
      return( -1 );
    }
  
  int l = strlen( msg ) - 1;
  if ( ( ( msg[ 0 ] == '"' ) && ( msg[ l ] == '"' ) ) ||
       ( ( msg[ 0 ] == '\'' ) && ( msg[ l ] == '\'' ) ) )
    { 
      *(msg + l )= 0;
      msg++;
    }
  

  node = msglist_createnode_nsh( msg, font, ptsize, x, y, fgcolor );
  if ( node )
    {
      int i = 0;
      SDL_mutexP( messages_mutex );
      i = msglist_addnode( &message_list, node );
      SDL_mutexV( messages_mutex );
      if ( i == 0 )
	mpav_fmsg( stdout, "Message ID: %p\n", node );
      else
	mpav_fmsg( stderr, "ERROR: Could not create message!\n" );
    }
  else
    {
      // no msg. it was printed out by msglist_createnode_nsh
      return( -1 );
    }

  return( 0 );
}

int __slave_com_cms( char *argument )
{
  msgnode_t *node = NULL;
  // cmdline: <fontfile> <ptsize> <x> <y> <fgcolor> <shx> <shy> <shcolor> <msg>
  char *font = NULL;
  char *msg  = NULL;
  char *w    = NULL;
  int ptsize = 0;
  int x = 0;
  int y = 0;
  int shx = 0;
  int shy = 0;  
  Uint32 fgcolor = 0;
  Uint32 shcolor = 0;

  if ( ! argument )
    {
      __slave_command_error_noargs( "cms" );
      return( -1 );
    }

  // font
  font = __slave_get_word( argument );
  if ( ! font )
    {
      __slave_command_error_misarg( "cms", "fontfile" );
      return( -1 );
    }

  // ptsize
  w = __slave_get_next_word( font );
  if ( ! w )
    {
      __slave_command_error_misarg( "cms", "ptsize" );
      return( -1 );
    }
  ptsize = ( int ) strtol( w, NULL, 10 );
  if ( ptsize <= 0 )
    {
      mpav_fmsg( stderr, "ERROR: argument 'ptsize' for 'cms' must be greater than zero! ('%d' was provided)\n", ptsize );
      return( -1 );
    }
  
  // x
  w = __slave_get_next_word( w );
  if ( ! w )
    {
      __slave_command_error_misarg( "cms", "x" );
      return( -1 );
    }
  x = ( int ) strtol( w, NULL, 10 );
  if ( x <= 0 )
    {
      mpav_fmsg( stderr, "ERROR: argument 'x' for 'cms' must be greater than zero! ('%d' was provided)\n", x );
      return( -1 );
    }
  
  // y
  w = __slave_get_next_word( w );
  if ( ! w )
    {
      __slave_command_error_misarg( "cms", "y" );
      return( -1 );
    }
  y = ( int ) strtol( w, NULL, 10 );
  if ( y <= 0 )
    {
      mpav_fmsg( stderr, "ERROR: argument 'y' for 'cms' must be greater than zero! ('%d' was provided)\n", y );
      return( -1 );
    }

  // fgcolor
  w = __slave_get_next_word( w );
  if ( ! w )
    {
      __slave_command_error_misarg( "cms", "fgcolor" );
      return( -1 );
    }
  fgcolor = ( Uint32 ) strtol( w, NULL, 16 );
  if ( fgcolor < 0 )
    {
      mpav_fmsg( stderr, "ERROR: argument 'fgcolor' for 'cms' must be in hexa RGB (ie: 0xff0000 is red)! ('0x%x' was provided)\n", fgcolor );
      return( -1 );
    }

  // shx
  w = __slave_get_next_word( w );
  if ( ! w )
    {
      __slave_command_error_misarg( "cms", "shx" );
      return( -1 );
    }
  shx = ( int ) strtol( w, NULL, 10 );
  if ( shx <= 0 )
    {
      mpav_fmsg( stderr, "ERROR: argument 'shx' for 'cms' must be greater than zero! ('%d' was provided)\n", shx );
      return( -1 );
    }
  
  // shy
  w = __slave_get_next_word( w );
  if ( ! w )
    {
      __slave_command_error_misarg( "cms", "shy" );
      return( -1 );
    }
  shy = ( int ) strtol( w, NULL, 10 );
  if ( shy < 0 )
    {
      mpav_fmsg( stderr, "ERROR: argument 'shy' for 'cms' must be greater than zero! ('%d' was provided)\n", shy );
      return( -1 );
    }

  // shcolor
  w = __slave_get_next_word( w );
  if ( ! w )
    {
      __slave_command_error_misarg( "cms", "shcolor" );
      return( -1 );
    }
  shcolor = ( Uint32 ) strtol( w, NULL, 16 );
  if ( shcolor <  0 )
    {
      mpav_fmsg( stderr, "ERROR: argument 'shcolor' for 'cms' must be in hexa RGB (ie: 0xff0000 is red)! ('0x%x' was provided)\n", shcolor );
      return( -1 );
    }

  // msg
  msg = w + 1 + strlen( w );
  if ( ! msg )
    {
      __slave_command_error_misarg( "cms", "msg" );
      return( -1 );
    }
  
  int l = strlen( msg ) - 1;
  if ( ( ( msg[ 0 ] == '"' ) && ( msg[ l ] == '"' ) ) ||
       ( ( msg[ 0 ] == '\'' ) && ( msg[ l ] == '\'' ) ) )
    { 
      *(msg + l) = 0;
      msg++;
    }  

  node = msglist_createnode( msg, font, ptsize, x, y, fgcolor, 1, shx, shy, shcolor );
  if ( node )
    {
      int i = 0;
      SDL_mutexP( messages_mutex );
      i = msglist_addnode( &message_list, node );
      SDL_mutexV( messages_mutex );
      if ( i == 0 )
	mpav_fmsg( stdout, "Message ID: %p\n", node );
      else
	mpav_fmsg( stderr, "ERROR: Could not create message!\n" );
    }
  else
    {
      // no msg. it was printed out by msglist_createnode
      return( -1 );
    }

  return( 0 );
}

int __slave_com_dm( char *argument )
{
  msglist_t *id = NULL;
  msglist_t *node = NULL;

  if ( ! argument )
    {
      __slave_command_error_misarg( "dm", "ID" );
      return( -1 );
    }
  sscanf( argument, "%p", &id );
  SDL_mutexP( messages_mutex );
  node = msglist_delnode( &message_list, id );
  SDL_mutexV( messages_mutex );

  if ( ! node )
    {
      mpav_fmsg( stderr, "ERROR: argument 'ID' for 'dm' must be the message id ('%p' was provided)\n", id );
      return( -1 );
    }
  
  msglist_destroynode( &node );

  return( 0 );
}

int __slave_com_lm( char *argument )
{
  SDL_mutexP( messages_mutex );
  msglist_print( message_list );
  SDL_mutexV( messages_mutex );
  return( 0 );
}


