/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   lcas_utils.c
    \brief  the utilities for the LCAS
    \author Martijn Steenbakkers for the EU DataGrid.
*/

/*****************************************************************************
                            Include header files
******************************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <stdarg.h>
#include "lcas_defines.h"
#include "lcas_types.h"
#include <openssl/x509.h>
#ifndef NOGLOBUS
    #include <gssapi.h>
    #include "_lcas_gsi_utils.h"
#endif /* NOGLOBUS */

#include "_lcas_log.h"


/******************************************************************************
                             Define constants
******************************************************************************/
#if 0
#define LCAS_MOD_HOME LCAS_LIB_HOME"/lcas" /* default directory for the LCAS
                                              plugins/modules */
#endif

/******************************************************************************
                          Module specific prototypes
******************************************************************************/
static int fexist(char *);
int lcas_fill_cred(
        char * this_dn,
#ifndef NOGLOBUS
        gss_cred_id_t cred,
#else
        void * cred,
#endif /* NOGLOBUS */
        lcas_cred_id_t * plcas_cred);

int lcas_release_cred(lcas_cred_id_t * plcas_cred);
char * lcas_get_dn( lcas_cred_id_t lcas_credential );
#ifndef NOGLOBUS
    gss_cred_id_t lcas_get_gss_cred( lcas_cred_id_t lcas_credential );
#endif /* NOGLOBUS */

#ifdef LCAS_INTERFACE_EXTENDED
STACK_OF(X509) * lcas_get_cert_stack( lcas_cred_id_t lcas_credential );
X509 * lcas_get_user_certificate( lcas_cred_id_t lcas_credential );

X509 * lcas_get_first_delegation( lcas_cred_id_t lcas_credential );
X509 * lcas_get_leaf_proxy( lcas_cred_id_t lcas_credential );
#endif /* LCAS_INTERFACE_EXTENDED */

char * lcas_genfilename( char * prefixp, char * pathp, char * suffixp );

char * lcas_getfexist( int n, ... );

char * lcas_findfile( char * name );
char * lcas_finddbfile( char * name );
char * lcas_findplugin( char * name );
int lcas_tokenize( const char * command,  char ** args, int * n, char * sep );



/******************************************************************************
Function:       lcas_fill_cred()
Description:    Fill cedential from dn and globus credential
Parameters:
                dn: distinguished name
                cred: globus credential
                plcas_cred: pointer to lcas_credential
Returns:        0: succes
                1: failure
******************************************************************************/
int lcas_fill_cred(
        char * this_dn,
#ifndef NOGLOBUS
        gss_cred_id_t cred,
#else
        void * cred,
#endif /* NOGLOBUS */
        lcas_cred_id_t * plcas_cred)
{
#ifndef NOGLOBUS
    /* Copy the credential */
    plcas_cred->cred = cred;

    if (cred == GSS_C_NO_CREDENTIAL) /* Empty credential: Fill DN with passed dn */
    {
        plcas_cred->dn = strdup(this_dn);
    }
    else
    {
        plcas_cred->dn = lcas_gss_cred_to_dn (cred);
    }
#else
    if (this_dn)
        plcas_cred->dn = strdup(this_dn);
    else
        plcas_cred->dn = NULL;
#endif /* NOGLOBUS */


    if (plcas_cred->dn == NULL)
        return 1; /* Cannot find user dn */

    return 0;
}


/******************************************************************************
Function:       lcas_release_cred()
Description:    release lcas credential
Parameters:
                plcas_cred: pointer to lcas_credential
Returns:        0: succes
                1: failure
******************************************************************************/
int lcas_release_cred(
        lcas_cred_id_t * plcas_cred
)
{
    if (plcas_cred == NULL)
        return 0;

    if (plcas_cred->dn != NULL)
        free(plcas_cred->dn);
    /* Don't release globus credential (not copied) */

    return 0;
}


/******************************************************************************
Function:       lcas_get_dn()
Description:    returns user dn
Parameters:
                lcas_credential: lcas_credential
Returns:        user dn
******************************************************************************/
char * lcas_get_dn(
        lcas_cred_id_t lcas_credential
)
{
    return (lcas_credential.dn);
}


/******************************************************************************
Function:       lcas_get_gss_cred()
Description:    returns globus gss credential
Parameters:
                lcas_credential: lcas_credential
Returns:        globus gss credential
******************************************************************************/
#ifndef NOGLOBUS
gss_cred_id_t lcas_get_gss_cred(
        lcas_cred_id_t lcas_credential
)
{
    return (lcas_credential.cred);
}
#endif /* NOGLOBUS */


#ifdef LCAS_INTERFACE_EXTENDED

/******************************************************************************
Function:       lcas_get_cert_stack()
Description:    returns certificate stack
Parameters:
                lcas_credential: lcas_credential
Returns:        user certificate stack
******************************************************************************/
STACK_OF(X509) * lcas_get_cert_stack(
        lcas_cred_id_t lcas_credential
)
{
    return (lcas_credential.cert_stack);
}

/******************************************************************************
Function:       lcas_get_user_cert()
Description:    returns user cert
Parameters:
                lcas_credential: lcas_credential
Returns:        user certificate
******************************************************************************/
X509 * lcas_get_user_certificate(
        lcas_cred_id_t lcas_credential
)
{
    return (lcas_credential.user_cert);
}

/******************************************************************************
Function:       lcas_get_first_delegation()
Description:    returns user dn
Parameters:
                lcas_credential: lcas_credential
Returns:        first delegation
******************************************************************************/
X509 * lcas_get_first_delegation(
        lcas_cred_id_t lcas_credential
)
{
    return (lcas_credential.first_delegation);
}

/******************************************************************************
Function:       lcas_get_leaf_proxy()
Description:    returns leaf proxy
Parameters:
                lcas_credential: lcas_credential
Returns:        leaf proxy
******************************************************************************/
X509 * lcas_get_leaf_proxy(
        lcas_cred_id_t lcas_credential
)
{
    return (lcas_credential.leaf_proxy);
}
#endif /* LCAS_INTERFACE_EXTENDED */


/******************************************************************************
Function:       lcas_genfilename() (copied from GLOBUS gatekeeper.c)
Description:    generate an absolute file name given a starting prefix,
                a relative or absolute path, and a suffix
                Only use prefix if path is relative.
Parameters:
Returns:        a pointer to a string which could be freeded.
******************************************************************************/
char * lcas_genfilename(
        char * prefixp,
        char * pathp,
        char * suffixp
)
{
    char * newfilename;
    int    prefixl, pathl, suffixl;
    char * prefix,  * path, * suffix;

    prefix = (prefixp) ? prefixp : "";
    path   = (pathp) ? pathp : "";
    suffix  = (suffixp) ? suffixp : "";

    prefixl = strlen(prefix);
    pathl   =  strlen(path);
    suffixl  =  strlen(suffix); 

    newfilename = (char *) calloc(1, (prefixl + pathl + suffixl + 3));
    if (newfilename) 
    {
        if (*path != '/')
        {
            strcat(newfilename, prefix);
            if ((prefixl != 0) && (prefix[prefixl-1] != '/'))
            {
                strcat(newfilename, "/");
            }
        }
        strcat(newfilename, path);
        if ((pathl  != 0) &&
            (suffixl != 0) && 
            (path[pathl-1] != '/') && 
             suffix[0] != '/')
        {
            strcat(newfilename, "/");
        }
        strcat(newfilename, suffix);
    }
    return newfilename;
}

/******************************************************************************
Function:       fexist()
Description:    check the existence of file corresponding to <path>
Parameters:     path
Returns:        1, if file exists
******************************************************************************/
/*!
    \fn fexist(
        char * path
        )
    \brief check the existence of file corresponding to \<path\>
    \param path absolute filename to be checked.
    \retval 1 file exists.
    \retval 0 failure.
*/
static int fexist(
        char * path
)
{
  struct stat sbuf;
  int res;
  
  if(!path || !*path) return 0;

  res=stat(path,&sbuf);
  if (res)
  {
      if (errno==ENOENT)
      {
          return 0;
      }
      else
      {
          return -1;
      }
  }
  return 1;
}

/******************************************************************************
Function:       lcas_getfexist()
Description:    picks the first existing file in argument list
Parameters:     n   : number of paths,
                ... : list of paths
Returns:        returns filename found or NULL
******************************************************************************/
char * lcas_getfexist(
        int n,
        ...
)
{
  va_list pvar;
  int i;
  char *cfilenm=NULL;

  va_start(pvar, n);

  for(i=0;i<n;i++) {
    cfilenm=va_arg(pvar,char*);
    if(*cfilenm) if(fexist(cfilenm)) return cfilenm;
  }
  va_end(pvar);
  return NULL;
}

/******************************************************************************
Function:       lcas_findplugin()
Description:    Checks for file in standard directories
Parameters:     name
Returns:        returns filename found (should be freeed) or NULL
******************************************************************************/
char * lcas_findplugin(
        char * name
)
{
    char *tmpname=NULL;
    char *lcas_modules_dir=NULL;

    /* New format for db: either path is set in db file and plugins have an
     * absolute path with respect to that (when path is relative, the eval
     * manager prefixes with $libdir) or there isn't a path set and the plugin
     * name is to be searched in either lcas_MODULES_DIR if set, or in  */

    lcas_modules_dir=getenv("LCAS_MODULES_DIR");
    if (lcas_modules_dir) {
        /* test in and in lcas_MODULES_DIR and build-time default
         * lcas_MOD_HOME */
        tmpname=lcas_genfilename(lcas_modules_dir,name,NULL);
        if (!tmpname)    { /* calloc error */
            lcas_log_time(0,"%s: Cannot calloc\n", __func__);
            return NULL;
        }
        if (fexist(tmpname)) return tmpname;
        free(tmpname);
    }
    /* No name so far: test default build-in location */
    tmpname=lcas_genfilename(LCAS_MOD_HOME,name,NULL);
    if (!tmpname)    { /* calloc error */
        lcas_log_time(0,"%s: Cannot calloc\n", __func__);
        return NULL;
    }
    if (fexist(tmpname)) return tmpname;
    free(tmpname);

    return NULL;
}

/******************************************************************************
Function:       lcas_finddbfile()
Description:    Checks for db files in LCAS_ETC_HOME directories
Parameters:     name
Returns:        returns filename found (should be freeed) or NULL
******************************************************************************/
char * lcas_finddbfile(
        char * name
)
{
    char *tmpname=NULL;

    /* Is absolute: test if it exists */
    if (name[0]=='/')	{
	if (fexist(name))   {
	    if ( (tmpname=strdup(name)) == NULL)    {
		lcas_log_time(0,"%s: Cannot calloc\n", __func__);
		return NULL;
	    }
	    /* Absolute existing */
	    return tmpname;
	}
	/* Absolute but non-existing */
	return NULL;
    }

    /* test default built-in location */
    tmpname=lcas_genfilename(LCAS_ETC_HOME,name,NULL);
    if (!tmpname)    { /* calloc error */
        lcas_log_time(0,"%s: Cannot calloc\n", __func__);
        return NULL;
    }
    /* If exists: return it */
    if (fexist(tmpname)) return tmpname;
    free(tmpname);

    return NULL;
}

/******************************************************************************
Function:       lcas_findfile()
Description:    Checks for db files in LCAS_ETC_HOME directories
Parameters:     name
Returns:        returns filename found (should be freeed) or NULL
******************************************************************************/
char * lcas_findfile(
        char * name
)
{
    return lcas_finddbfile(name);
}

/******************************************************************************
Function:   lcas_tokenize() (in modified form from globus_gatekeeper_utils.c)

Description:
    Breakup the command in to args, pointing the args array
    at the tokens. Replace white space at the end of each
    token with a null. A token maybe in quotes. 

Parameters:
    command: The command line to be parsed.
    args:    A pointer to an array of pointers to be filled it
    n:       Size of the array, on input, and set to size used on output. 
    sep:     string of seperating characters

Returns:
    0 on success. 
    -1 on to malloc
    -2 on to many args
    -3 on quote not matched
******************************************************************************/
int lcas_tokenize(
        const char * command, 
        char ** args,
        int * n,
        char * sep
    )
{
    int maxargs;
    int i;
    const char * cp;
    const char * pp;
    const char * qp;
    char ** arg;

    arg = args;
/*    i = *n - 1; */
    i = 0;
    maxargs = *n;

    cp = command;
    while (*cp)
    {
    /* skip leading sep characters */
        while (*cp && strchr(sep, *cp))
        {
            cp++;
        }
        pp = NULL;
        if (*cp == '\"')
        {
            cp++;
            pp = cp;
            if ((qp = strchr(cp,'\"')) == NULL)
            {
                *n = i;
                return -3;
            }
            cp = qp + 1;
        }
        else if (*cp)
        {
            pp = cp;
            if ((qp = strpbrk(cp,sep)) == NULL)
            {
                qp = strchr(cp,'\0');
            }
            cp = qp;
        }
        else
        {
            continue;
        }
        if (pp)
        {
            /*
             * fill at most maxargs-1 arguments; let the last one point to NULL
             */
            i++;
            if (i >= maxargs)
            {
                i--;
                *n = i;
                return(-2); /* too many args */
            }
            *arg = (char*)malloc((qp - pp) + 1);
            if (*arg == NULL)
            {
                i--;
                *n = i;
                return -1;
            }
            memcpy(*arg,pp,qp - pp);
            *(*arg + (qp - pp)) = '\0';
            arg++;
        }
    }
    *arg = NULL;
    *n = i;
    return(0);
}

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcas/src/lcas_utils.c,v $
    $Date: 2010-05-03 10:42:49 $
    $Revision: 2.20 $
    $Author: okoeroo $
******************************************************************************/
