#include <assert.h>
#include <errno.h>
#include <dlfcn.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <time.h>
#include <unistd.h>

#include <set>
#include <string>

#include "compat.h"
#include "lib/util.h"
#include "socketwatch.h"

using std::set;
using std::string;


#define LIB_NAME "libsocketwatch"

#define DEBUG 0
#if DEBUG
# define Dprintf(x...) printf(x)
#else
# define Dprintf(x...) do { } while(0)
#endif


class watchdog_client
{
public:
	watchdog_client()
	{
		char *shared_fd_cstr;
		int shared_fd;

		Dprintf("Starting " LIB_NAME " for %d\n", getpid());

		//
		// Set up shared heartbeat page

		shared_fd_cstr = getenv(SOCKETWATCH_SHARED_FD);
		die_if(!shared_fd_cstr, "getenv(%s): does not exist\n",
		       SOCKETWATCH_SHARED_FD);

		shared_fd = atoi(shared_fd_cstr);

		heartbeat = (time_t*) mmap(NULL, sysconf(_SC_PAGE_SIZE),
		                           PROT_READ | PROT_WRITE, MAP_SHARED,
		                           shared_fd, 0);
		die_if(heartbeat == MAP_FAILED, "mmap(shared_fd)");

		// leave shared_fd open so that it can be re-mapped after an exec()
	}

	~watchdog_client()
	{
		int r = munmap(heartbeat, sysconf(_SC_PAGE_SIZE));
		die_if(r < 0, "munmap");
	}

	void insert_fd(int fd)
	{
		fds.insert(fd);
		Dprintf(LIB_NAME ": Starting to watch fd %d\n", fd);
	}

	void erase_fd(int fd)
	{
		set<int>::size_type r = fds.erase(fd);
		if (r)
			Dprintf(LIB_NAME ": Stopping watch of fd %d\n", fd);
	}

	void use_fd(int fd)
	{
		if (fds.find(fd) != fds.end())
		{
			*heartbeat = time(NULL);
			Dprintf(LIB_NAME ": Emitting heartbeat at %ld\n", *heartbeat);
		}
	}

private:
	time_t *heartbeat;
	set<int> fds;
};

static watchdog_client watchdog;


//
// Overridden functions.
// These call into the original functions and notify the watchdog of activity.

ssize_t read(int fd, void *buf, size_t count)
{
	ssize_t (*orig)(int, void*, size_t);
	char *error;
	int saved_errno;
	int r;

	*(void**) (&orig) = dlsym(RTLD_NEXT, __FUNCTION__);
	if ((error = dlerror()))
	{
		fprintf(stderr, "%s: dlsym(%s): %s\n", LIB_NAME, __FUNCTION__, error);
		exit(EXIT_FAILURE);
	}

	r = orig(fd, buf, count);
	saved_errno = errno;

	if (r > 0)
		watchdog.use_fd(fd);

	errno = saved_errno;
	return r;
}

ssize_t write(int fd, const void *buf, size_t count)
{
	ssize_t (*orig)(int, const void*, size_t);
	char *error;
	int saved_errno;
	int r;

	*(void**) (&orig) = dlsym(RTLD_NEXT, __FUNCTION__);
	if ((error = dlerror()))
	{
		fprintf(stderr, "%s: dlsym(%s): %s\n", LIB_NAME, __FUNCTION__, error);
		exit(EXIT_FAILURE);
	}

	r = orig(fd, buf, count);
	saved_errno = errno;

	if (r > 0)
		watchdog.use_fd(fd);

	errno = saved_errno;
	return r;
}

int socket(int domain, int type, int protocol)
{
	int (*orig)(int, int, int);
	char *error;
	int saved_errno;
	int r;

	*(void**) (&orig) = dlsym(RTLD_NEXT, __FUNCTION__);
	if ((error = dlerror()))
	{
		fprintf(stderr, "%s: dlsym(%s): %s\n", LIB_NAME, __FUNCTION__, error);
		exit(EXIT_FAILURE);
	}

	r = orig(domain, type, protocol);
	saved_errno = errno;

	if (r >= 0)
		watchdog.insert_fd(r);

	errno = saved_errno;
	return r;
}

int close(int fd)
{
	int (*orig)(int);
	char *error;
	int saved_errno;
	int r;

	*(void**) (&orig) = dlsym(RTLD_NEXT, __FUNCTION__);
	if ((error = dlerror()))
	{
		fprintf(stderr, "%s: dlsym(%s): %s\n", LIB_NAME, __FUNCTION__, error);
		exit(EXIT_FAILURE);
	}

	r = orig(fd);
	saved_errno = errno;

	if (r >= 0)
		watchdog.erase_fd(fd);

	errno = saved_errno;
	return r;
}
