--  music123 by David Starner <dvdeug@debian.org>

with Ada.Command_Line; use Ada.Command_Line;
with Ada.Text_IO;

with Support_Routines; use Support_Routines;
with Intl;
with Commands;

procedure Music123 is
   Seeing_Minus_Options : Boolean := True;
   Arg_Num : Positive;
   Option_Task : Boolean := False;
   Option_Quiet : Boolean := False;
   Option_Recurse : Boolean := False;
   Option_Random : Boolean := False;
   Option_Loop : Boolean := False;
   Option_Eternal_Random : Boolean := False;
   Option_Ignore_Extension_Case : Boolean := False;
   Option_List_Files_Only : Boolean := False;
   Delay_Length : Duration := 0.5;
   File_List : UString_List.Vector;
   Program_List : Tool_List.Vector;
   Command_Task : Commands.Handler_Task;

begin
   Intl.Set_Locale;
   Intl.Text_Domain ("music123");
   Intl.Bind_Text_Domain ("music123", "/usr/share/locale");

   --  Import conffile first
   Import_Conffile (Program_List);

   --  Read command-line arguments
   if Argument_Count = 0 then
      Error (N ("No arguments found."));
      raise Noted_Error;
   end if;

   Arg_Num := 1;
   while Arg_Num <= Argument_Count loop
      if Seeing_Minus_Options
        and then Argument (Arg_Num)'Length = 2
        and then Argument (Arg_Num) (1) = '-'
      then
         case Argument (Arg_Num) (2) is
            when 'h' =>
               Error ("");
               Set_Exit_Status (Success);
               return;
            when 'q' => Option_Quiet := True;
            when 'z' => Option_Random := True;
            when 'Z' => Option_Eternal_Random := True;
            when 'l' => Option_Loop := True;
            when 'r' => Option_Recurse := True;
            when 'i' => Option_Ignore_Extension_Case := True;
            when 'v' =>
               Ada.Text_IO.Put_Line (Version);
               Set_Exit_Status (Success);
               return;
            when 'D' =>
               Delay_Length := 0.0;
            when 'd' =>
               if Arg_Num >= Argument_Count then
                  Error (N ("Missing argument for -d."));
                  raise Noted_Error;
               end if;
               Arg_Num := Arg_Num + 1;
               begin
                  Delay_Length := Duration'Value (Argument (Arg_Num));
               exception
                  when Constraint_Error =>
                     Error (N ("Bad argument for -d."));
                     raise Noted_Error;
               end;
            when 'L' => Option_List_Files_Only := True;
            when 'T' => Option_Task := True;
            when '-' => Seeing_Minus_Options := False;
            when '@' =>
               if Arg_Num >= Argument_Count then
                  Error (N ("Missing argument for -@."));
                  raise Noted_Error;
               end if;
               Arg_Num := Arg_Num + 1;
               Read_Playlist (Argument (Arg_Num), File_List);
            when others =>
               Error (N ("Unknown argument found."));
               raise Noted_Error;
         end case;
      elsif Check_Filename (Argument (Arg_Num),
                            Program_List,
                            Option_Ignore_Extension_Case)
      then
         File_List.Append (Argument (Arg_Num));
      else
         Ada.Text_IO.Put_Line ("Ignoring file """ & Argument (Arg_Num) & """");
      end if;
      Arg_Num := Arg_Num + 1;
   end loop;

   Expand_And_Check_Filenames (File_List,
                               Option_Recurse,
                               Program_List,
                               Option_Ignore_Extension_Case);

   if Option_List_Files_Only then
      Display_Songs (File_List);
      Set_Exit_Status (Success);
      return;
   end if;

   if Option_Task then
      Command_Task.Start;
   end if;

   Play_Songs
     (File_List,
      Program_List,
      Delay_Length => Delay_Length,
      Option_Quiet => Option_Quiet,
      Option_Loop => Option_Loop,
      Option_Random => Option_Random,
      Option_Eternal_Random => Option_Eternal_Random,
      Option_Ignore_Extension_Case => Option_Ignore_Extension_Case);

   if Option_Task then
      Command_Task.Stop;
   end if;

   Set_Exit_Status (Success);
exception
   when Noted_Error =>
      Set_Exit_Status (Failure);
end Music123;
