PROGRAM gen_error_vector
!
!****   gen_error_vector
!
!       Purpose:
!       --------
!       This program compares norm and angles of an interpolated vector field
!       with norm and angles of the initial vector field mapped of the target grid
!
!    
!       History:
!       --------
!        Version   Programmer      Date        Description
!        -------   ----------      ----        ------------
!         1.0      E. Rapaport    2004/04       Creation
!*---------------------------------------------------------------------------------

  IMPLICIT NONE

  INCLUDE 'netcdf.inc'

!*---------------------------------------------------
  REAL(KIND=8), PARAMETER :: pi=3.14159265359
  REAL(kind=8), PARAMETER :: pi2=2.0*pi
  REAL(KIND=8), PARAMETER :: deg2rad = pi/180
  REAL(KIND=8), PARAMETER :: rad2deg = 180/pi
  REAL(KIND=8), PARAMETER :: two=2.
  REAL(KIND=8), PARAMETER :: length = 0.6*pi2

  CHARACTER(len=1) :: &
       namefonc_I, namefonc_J

  CHARACTER(len=4) :: &
       namegrid,  &
       namedim1, namedim2, &
       namegrid_src

  CHARACTER(len=8) :: &
       output_I, output_J, varIout, varJout, &
       namelon, namelat, namemask, &
       name_angl

  CHARACTER(len=12) :: &
       angle_ana, angle_interp
  
  CHARACTER(len=14) :: &
       name_norm_error, name_angl_error

 !netcdf id:
  INTEGER :: &
       nc_output_I, nc_output_J, &
       id_varIin, id_varJin, id_dimin(2),&
       nc_grdid, angl_id, nc_angl_interp, ig_file_id

  INTEGER ::dimin1, dimin2

  INTEGER :: lonid, latid, &
       dimid(2), dim1, dim2,id_varIout,id_varJout, &
       norm_errorid, angl_errorid, &
       nc_maskid, maskid, nc_anaid, func_anaid, &
       angl_anaid, angl_interpid

  INTEGER :: il_unit, nbflds, i, k, stat, j, ii, il_log

  REAL(KIND=8), DIMENSION(:,:), ALLOCATABLE :: &
       rla_lon, rla_lat, comput_valI, comput_valJ, &
       ana_valI, ana_valJ, ana_val_tmp, &
       angl_error, norm_error, &
       norm_interp, norm_ana, &
       val_angl, angl_ana, angl_interp, &
       ana_valI_new, ana_valJ_new, &
       new_comput_valI, new_comput_valJ

  INTEGER, DIMENSION(:,:), ALLOCATABLE :: mask

  LOGICAL :: l_dst_spher

  INTEGER :: size, ecart_anglid, ecart_normid, &
       moy_normid, moy_anglid, max_normid, max_anglid, &
       var_normid, var_anglid, comput_valIid, comput_valJid

  REAL(KIND=8) :: &
       temp1, temp2, moy_norm, moy_angl, &
       var_norm, var_angl, max_norm, min_norm, &
       max_angl, min_angl
!*------------------------------------------------------
  il_log = 88
  OPEN(unit=il_log, file='error_log', form='formatted')
  WRITE(il_log,*) 'Start gen_error_vector'
!
!* Get informations about grid, function and composant
!
  il_unit = 6
  OPEN(unit=il_unit, file='compute_error', status='old')
  READ(il_unit, '(I1)') nbflds

!
!*loop on the total number of fields 
!
  DO ii=1, nbflds   

     READ(il_unit, '(A4)') namegrid
     READ(il_unit, '(A1)') namefonc_I
     READ(il_unit, '(A1)') namefonc_J

!  
!* Get output files info
!
     output_I = namefonc_I//namegrid(1:1)//'Iou.nc'
     output_J = namefonc_J//namegrid(1:1)//'Jou.nc'

     varIout = namefonc_I//'_'//namegrid(1:1)//'_Iout' 
     varJout = namefonc_J//'_'//namegrid(1:1)//'_Jout'

     namelon = namegrid(1:4)//'.lon' 
     namelat = namegrid(1:4)//'.lat'
     namemask = namegrid(1:4)//'.msk'   

     name_norm_error = namefonc_I//'_'//namefonc_J//'_'// &
          namegrid(1:1)//'_norm_err'
     name_angl_error = namefonc_I//'_'//namefonc_J//'_'// &
          namegrid(1:1)//'_angl_err'

     angle_ana = 'angl_ana_deg'
     angle_interp = 'angl_int_deg'

     WRITE(IL_LOG,*)'output files : ', output_I, ' ', output_J 
     WRITE(IL_LOG,*)'variables =', varIout, ' ', varJout 

!
!* Test if files really exists; if not, the program will stop.
!
     stat = NF_OPEN(output_I,NF_WRITE, nc_output_I)
     IF(stat .ne. NF_NOERR) THEN
        write(il_log,*)'WARNING: output file', output_I, 'does not exist'
        write(il_log,*)'STOP'
        STOP
     END IF

     stat = NF_OPEN(output_J,NF_NOWRITE, nc_output_J)
     IF(stat .ne. NF_NOERR) THEN
        write(il_log,*)'WARNING: output file', output_J, 'does not exist'
        write(il_log,*)'STOP'
        STOP
     END IF     
!
!*Get information in the output files
!
     CALL hdlerr(NF_INQ_VARID &
          (nc_output_I, namelon, lonid))
         
     CALL hdlerr(NF_INQ_VARID &
          (nc_output_I, namelat, latid))
     
     CALL hdlerr(NF_INQ_VARDIMID &
          (nc_output_I, lonid, dimid))
     
     CALL hdlerr(NF_INQ_DIMLEN &
          (nc_output_I, dimid(1), dim1))
     
     CALL hdlerr(NF_INQ_DIMNAME &
          (nc_output_I, dimid(1), namedim1)) 
        
     CALL hdlerr(NF_INQ_DIMLEN &
          (nc_output_I, dimid(2), dim2))
     
     CALL hdlerr(NF_INQ_DIMNAME &
          (nc_output_I, dimid(2), namedim2))   
     
     CALL hdlerr(NF_INQ_VARID &
          (nc_output_I, varIout, id_varIout)) 
     
     CALL hdlerr(NF_INQ_VARID &
          (nc_output_J, varJout, id_varJout)) 
    
     ALLOCATE(rla_lon(dim1,dim2), &
          rla_lat(dim1,dim2), &
          comput_valI(dim1, dim2), &
          comput_valJ(dim1, dim2), &
          ana_valI(dim1, dim2), &
          ana_valJ(dim1, dim2), &
          norm_error(dim1, dim2), &
          norm_interp(dim1, dim2), &
          norm_ana(dim1, dim2), &
          angl_interp(dim1, dim2), &
          angl_ana(dim1, dim2), &
          angl_error(dim1, dim2), &
          mask(dim1, dim2))
           
     CALL hdlerr(NF_GET_VAR_DOUBLE &
          (nc_output_I, lonid, rla_lon))  
     
     CALL hdlerr(NF_GET_VAR_DOUBLE &
          (nc_output_I, latid, rla_lat)) 
     
     CALL hdlerr(NF_GET_VAR_DOUBLE &
          (nc_output_I, id_varIout, comput_valI))
     
     CALL hdlerr(NF_GET_VAR_DOUBLE &
          (nc_output_J, id_varJout, comput_valJ))
     
!-- Convert into radians
     rla_lat(:,:) = rla_lat(:,:) * deg2rad
     rla_lon(:,:) = rla_lon(:,:) * deg2rad

!
!* Compute analytic value of the 1 comp
!  
     SELECT CASE(namefonc_I)
     CASE('a')
        ALLOCATE(ana_val_tmp(dim1, dim2))
        ana_valI = COS(rla_lat)* COS(rla_lon)
        ana_val_tmp = ACOS(-ana_valI)/length
        ana_valI = 1.5 + ana_val_tmp
        DEALLOCATE(ana_val_tmp)                    

     CASE('c')
        ana_valI = two + COS(rla_lat)**2 * &
             COS(two*rla_lon)  

     CASE('s')
        ana_valI = two + SIN(two*rla_lat)**16 * &
                         COS(16.*rla_lon)   

     CASE('u')
       ana_valI = 1.

     CASE('z')
        ana_valI=0.

     CASE('p')  !champ variant bcp au pole nord
!        ana_valI = 1 + COS(rla_lon) * SIN(rla_lon)   
        ana_valI = COS(rla_lat) * SIN(rla_lat)

     CASE('I')
        WHERE(sin(rla_lat) .ne. 0.) 
           ana_valI = - sin(rla_lon) - sin(rla_lat)* cos(rla_lon) * 0.01+ &
                cos(rla_lat) * cos(rla_lon) * (-0.01* cos(rla_lat)/sin(rla_lat))
        ELSEWHERE
           ana_valI = - sin(rla_lon)        
        END WHERE
 
     CASE DEFAULT
        write(il_log,*)'the function does not exist'
          write(il_log,*)'STOP'
          STOP
       END SELECT  
!
!* Compute analytic value of the 2 comp
!  
     SELECT CASE(namefonc_J)
     CASE('a')
        ALLOCATE(ana_val_tmp(dim1, dim2))
        ana_valJ = COS(rla_lat) * COS(rla_lon)
        ana_val_tmp = ACOS(-ana_valJ)/length
        ana_valJ = 1.5 + ana_val_tmp
        DEALLOCATE(ana_val_tmp)                    

     CASE('c')
        ana_valJ = two + COS(rla_lat)**2 * &
             COS(two*rla_lon)  

     CASE('s')
        ana_valJ = two + SIN(two*rla_lat)**16 * &
                         COS(16.*rla_lon)   

     CASE('u')
       ana_valJ = 1.

     CASE('z')
        ana_valJ=0.

     CASE('p')  !champ variant bcp au pole nord
!        ana_valJ = 1 + COS(rla_lon) * SIN(rla_lon)
        ana_valJ = COS(rla_lat) * SIN(rla_lat)

     CASE('J')
         WHERE(sin(rla_lat) .ne. 0.) 
            ana_valJ =  cos(rla_lon) - sin(rla_lat) * sin(rla_lon) *0.01+ &
                cos(rla_lat) * sin(rla_lon) * (- 0.01*cos(rla_lat)/sin(rla_lat))
        ELSEWHERE
           ana_valJ =  cos(rla_lon)
        END WHERE

     CASE DEFAULT
        write(il_log,*)'the function does not exist'
          write(il_log,*)'STOP'
          STOP
       END SELECT
!
!* Compute norm 2 and angle with the j dir of the spheric referential
!  -------------------------------------------------------------------
!
!*Norm 2:
!
       norm_interp(:,:) = SQRT(comput_valI(:,:)**2 + comput_valJ(:,:)**2)
       
       norm_ana(:,:) = SQRT(ana_valI(:,:)**2 + ana_valJ(:,:)**2)
!
!*Angle:
!
!If target referential is not spheric, add angle between spheric and
!local refenrential to have info in the spheric referential
!  
     CALL hdlerr(NF_OPEN &
          ('grids.nc',NF_NOWRITE,nc_grdid))
     
     name_angl = namegrid(1:4)//'.ang'

     stat = NF_INQ_VARID &
          (nc_grdid, name_angl, angl_id)

     IF(stat == NF_NOERR) THEN
        
        ALLOCATE(val_angl(dim1,dim2))
       
        CALL hdlerr(NF_GET_VAR_DOUBLE &
             (nc_grdid,angl_id,val_angl))
        
        l_dst_spher = .true.

     ELSE
        l_dst_spher = .false.
        WRITE(IL_LOG,*)'WARNING'
        WRITE(IL_LOG,*)'*******'
        WRITE(IL_LOG,*)'grids.nc contains no angle info'
        WRITE(IL_LOG,*)'grid ',namegrid, ' will be considered spheric'
     END IF

     write(il_log,*)'l_dst_spher', l_dst_spher

     CALL hdlerr(NF_CLOSE(nc_grdid))

     IF(l_dst_spher) THEN
        ALLOCATE(new_comput_valI(dim1,dim2))
        ALLOCATE(new_comput_valJ(dim1,dim2))
        DO i=1, dim1
           DO j=1, dim2 
              new_comput_valI(i,j) = comput_valI(i,j) * cos(val_angl(i,j)*deg2rad) - &
                   comput_valJ(i,j) * sin(val_angl(i,j)*deg2rad)
              new_comput_valJ(i,j) = comput_valI(i,j) * sin(val_angl(i,j)*deg2rad) + &
                   comput_valJ(i,j) * cos(val_angl(i,j)*deg2rad)
           END DO
        END DO
     END IF

     DO i=1, dim1
        DO j=1, dim2
           IF(ABS(comput_valJ(i,j)) <= 0.99 .or. &
                ABS(ana_valJ(i,j)) <= 0.99) THEN         
              
              IF(l_dst_spher) THEN
                 angl_interp(i,j) = &
                      ATAN2(new_comput_valI(i,j),new_comput_valJ(i,j))
                 angl_interp(i,j) = angl_interp(i,j) * rad2deg        
              ELSE
                 angl_interp(i,j) = &
                      ATAN2(comput_valI(i,j),comput_valJ(i,j))
                 angl_interp(i,j) = angl_interp(i,j) * rad2deg
              END IF

              angl_ana(i,j) = ATAN2(ana_valI(i,j),ana_valJ(i,j))
              angl_ana(i,j) = angl_ana(i,j) * rad2deg

           ELSE

              IF(l_dst_spher) THEN
                 angl_interp(i,j) = &
                      ATAN2(new_comput_valJ(i,j),new_comput_valI(i,j))
                 angl_interp(i,j) = angl_interp(i,j) * rad2deg 
              ELSE
                 angl_interp(i,j) = &
                      ATAN2(comput_valJ(i,j),comput_valI(i,j))
                 angl_interp(i,j) = angl_interp(i,j) * rad2deg
              END IF

              angl_ana(i,j) = ATAN2(ana_valJ(i,j),ana_valI(i,j))
              angl_ana(i,j) = angl_ana(i,j) * rad2deg

           END IF
        END DO
     END DO
           
!!==> End of calculation of norm 2 and angle

!
!* Get mask of target grid
!
       CALL hdlerr(NF_OPEN('masks.nc', NF_NOWRITE, nc_maskid))

       CALL hdlerr(NF_INQ_VARID &
            (nc_maskid, namemask, maskid))

       CALL hdlerr(NF_GET_VAR_INT &
            (nc_maskid, maskid, mask)) 

!
!* Compute error
! 
       WHERE(norm_ana <= 0.000001)          
          norm_error = ABS(norm_interp)
       ELSEWHERE
          norm_error = ABS((norm_ana - norm_interp) / norm_ana)
       END WHERE
       WHERE(ABS(angl_ana) <= 0.000001)
            angl_error = ABS(angl_interp)
       ELSEWHERE         
          angl_error = ABS((angl_ana - angl_interp) / angl_ana)
       END WHERE

       WHERE(mask .ne. 0)
          norm_error = 0.
          angl_error = 0.
          angl_ana = 0.
          angl_interp = 0.
          ana_valI = 0.
          ana_valJ = 0.
       ENDWHERE
!
!*Si erreurs masques
!
       WHERE( comput_valI .eq. 0 .and. comput_valJ .eq. 0)
          norm_error = 0.
          angl_error = 0.
          angl_ana = 0.
          angl_interp = 0.
          ana_valI = 0.
          ana_valJ = 0.
       ENDWHERE  

!
!* Compute some error values
!Moyenne
       size = dim1 * dim2
      
       moy_norm = SUM(norm_error)/size
 
       moy_angl = SUM(angl_error)/size
 
!Ecart type
        temp1 = 0.
        temp2 = 0.
        DO i=1,dim1
           DO j=1,dim2
              temp1 = temp1 +&
                   (norm_error(i,j) - moy_norm)**2
              temp2 = temp2 +&
                   (angl_error(i,j) - moy_angl)**2
           END DO
        END DO
        var_norm = temp1/size
        var_angl = temp2/size

!max et min
        max_norm = MAXVAL(norm_error)
        max_angl = MAXVAL(angl_error)

        WRITE(IL_LOG,*)'moy norme:',moy_norm
        WRITE(IL_LOG,*)'moy angle:',moy_angl
        WRITE(IL_LOG,*)'ecart type norme:',var_norm
        WRITE(IL_LOG,*)'ecart type angle:',var_angl
        WRITE(IL_LOG,*)'max norme:', max_norm
        WRITE(IL_LOG,*)'max angle:', max_angl

       DO i=1,dim1
          DO j=1,dim2

                IF(comput_valI(i,j) .eq. 0. .and. &
                     ana_valI(i,j) .ne. 0.) THEN          
                   write(il_log,*)'mask error at point:',i,'',j
                END IF
          END DO
       END DO

       CALL hdlerr(NF_CLOSE(nc_output_I))
       CALL hdlerr(NF_CLOSE(nc_output_J))
       CALL hdlerr(NF_CLOSE(nc_maskid))  

       CALL hdlerr(NF_CREATE('error.nc', 0, ig_file_id))       
       CALL hdlerr(NF_DEF_DIM(ig_file_id, namedim1, dim1, dimid(1)))
       CALL hdlerr(NF_DEF_DIM(ig_file_id, namedim2, dim2, dimid(2)))

       CALL hdlerr(NF_DEF_VAR &
            (ig_file_id, varIout, NF_DOUBLE, 2, &
            dimid, id_varIout))

       CALL hdlerr(NF_DEF_VAR &
            (ig_file_id, varJout, NF_DOUBLE, 2, &
            dimid, id_varJout))

       CALL hdlerr(NF_DEF_VAR &
            (ig_file_id, name_norm_error, NF_DOUBLE, 2, &
            dimid, norm_errorid))

       CALL hdlerr(NF_DEF_VAR &
            (ig_file_id, name_angl_error, NF_DOUBLE, 2, &
            dimid, angl_errorid))

       CALL hdlerr(NF_ENDDEF(ig_file_id))

       CALL hdlerr(NF_PUT_VAR_DOUBLE &
            (ig_file_id, id_varIout, comput_valI)) 

       CALL hdlerr(NF_PUT_VAR_DOUBLE &
            (ig_file_id, id_varJout, comput_valJ)) 

       CALL hdlerr(NF_PUT_VAR_DOUBLE &
            (ig_file_id, norm_errorid, norm_error))
       CALL hdlerr(NF_PUT_VAR_DOUBLE &
            (ig_file_id, angl_errorid, angl_error))

       CALL hdlerr(NF_CLOSE(ig_file_id))
  
       DEALLOCATE(rla_lon, rla_lat, &
            comput_valI, comput_valJ, &
            ana_valI, ana_valJ, &
            norm_ana, norm_interp,&
            angl_ana, angl_interp,&
            norm_error, angl_error, &
            mask)
       IF(l_dst_spher) DEALLOCATE(new_comput_valI, &
            new_comput_valJ)

    END DO

    CLOSE(il_unit)

    WRITE(IL_LOG,*)'END OF ERROR COMPUTATION'

END PROGRAM gen_error_vector


SUBROUTINE hdlerr(istatus)
  IMPLICIT NONE 
  include 'netcdf.inc'
  INTEGER :: istatus
  
  if (istatus .ne. NF_NOERR) then
     print *, NF_STRERROR(istatus)
     stop 'stopped'
  endif
END SUBROUTINE hdlerr









