/* OGMRip - A library for DVD ripping and encoding
 * Copyright (C) 2004-2009 Olivier Rolland <billl@users.sourceforge.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <ogmrip.h>
#include <ogmjob.h>
#include <ogmrip-mplayer.h>

#include <glib/gstdio.h>
#include <glib/gi18n-lib.h>

#define PROGRAM "aften"

typedef struct _OGMRipAc3 OGMRipAc3;
typedef struct _OGMRipAc3Class OGMRipAc3Class;

struct _OGMRipAc3
{
  OGMRipAudioCodec parent_instance;
};

struct _OGMRipAc3Class
{
  OGMRipAudioCodecClass parent_class;
};

static gint ogmrip_ac3_run (OGMJobSpawn *spawn);

static const guint16 a52_bitratetab[] =
{
   32,  40,  48,  56,  64,  80,  96, 112, 128, 160,
  192, 224, 256, 320, 384, 448, 512, 576, 640,   0
};


gchar **
ogmrip_aften_command (OGMRipAudioCodec *audio, const gchar *input)
{
  GPtrArray *argv;
  OGMDvdAudioStream *astream;
  OGMDvdTitle *title;

  const gchar *output;
  gint i, bitrate;

  g_return_val_if_fail (OGMRIP_IS_AUDIO_CODEC (audio), NULL);
  g_return_val_if_fail (input != NULL, NULL);

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup (PROGRAM));
/*
  g_ptr_array_add (argv, g_strdup ("-threads"));
  g_ptr_array_add (argv, g_strdup_printf ("%u"), 1);
*/
  title = ogmrip_codec_get_input (OGMRIP_CODEC (audio));
  astream = ogmrip_audio_codec_get_dvd_audio_stream (audio);

  bitrate = ogmdvd_audio_stream_get_bitrate (astream) / 1000;
  bitrate = 128 + ((bitrate - 128) * ogmrip_audio_codec_get_quality (audio) / 10);

  for (i = 0; a52_bitratetab[i]; i++)
    if (a52_bitratetab[i] > bitrate)
      break;

  bitrate = a52_bitratetab[i] ? a52_bitratetab[i] : 640;

  g_ptr_array_add (argv, g_strdup ("-b"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", bitrate));

  g_ptr_array_add (argv, g_strdup (input));

  output = ogmrip_codec_get_output (OGMRIP_CODEC (audio));
  g_ptr_array_add (argv, g_strdup (output));

  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

static gchar **
ogmrip_wav_command (OGMRipAudioCodec *audio, gboolean header, const gchar *output)
{
  GPtrArray *argv;

  argv = ogmrip_mplayer_wav_command (audio, TRUE, output);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

static void
ogmrip_ac3_class_init (OGMRipAc3Class *klass)
{
  OGMJobSpawnClass *spawn_class;

  spawn_class = OGMJOB_SPAWN_CLASS (klass);

  spawn_class->run = ogmrip_ac3_run;
}

static void
ogmrip_ac3_init (OGMRipAc3 *ac3)
{
}

G_DEFINE_TYPE (OGMRipAc3, ogmrip_ac3, OGMRIP_TYPE_AUDIO_CODEC)

static gint
ogmrip_ac3_run (OGMJobSpawn *spawn)
{
  GError *error = NULL;
  OGMJobSpawn *pipeline;
  OGMJobSpawn *child;
  gchar **argv, *fifo;
  gint result;

  result = OGMJOB_RESULT_ERROR;

  fifo = ogmrip_fs_mkftemp ("fifo.XXXXXX", &error);
  if (!fifo)
  {
    ogmjob_spawn_propagate_error (spawn, error);
    return OGMJOB_RESULT_ERROR;
  }

  pipeline = ogmjob_pipeline_new ();
  ogmjob_container_add (OGMJOB_CONTAINER (spawn), pipeline);
  g_object_unref (pipeline);

  argv = ogmrip_wav_command (OGMRIP_AUDIO_CODEC (spawn), FALSE, fifo);
  if (argv)
  {
    child = ogmjob_exec_newv (argv);
    ogmjob_exec_add_watch_full (OGMJOB_EXEC (child), (OGMJobWatch) ogmrip_mplayer_wav_watch, spawn, TRUE, FALSE, FALSE);
    ogmjob_container_add (OGMJOB_CONTAINER (pipeline), child);
    g_object_unref (child);

    argv = ogmrip_aften_command (OGMRIP_AUDIO_CODEC (spawn), fifo);
    if (argv)
    {
      child = ogmjob_exec_newv (argv);
      ogmjob_container_add (OGMJOB_CONTAINER (pipeline), child);
      g_object_unref (child);

      result = OGMJOB_SPAWN_CLASS (ogmrip_ac3_parent_class)->run (spawn);
    }
  }

  ogmjob_container_remove (OGMJOB_CONTAINER (spawn), pipeline);

  if (g_file_test (fifo, G_FILE_TEST_EXISTS))
    g_unlink (fifo);
  g_free (fifo);

  return result;
}

static OGMRipAudioPlugin ac3_plugin =
{
  NULL,
  G_TYPE_NONE,
  "ac3",
  "AC-3",
  OGMRIP_FORMAT_AC3
};

OGMRipAudioPlugin *
ogmrip_init_plugin (GError **error)
{
  gboolean have_mplayer, have_aften;
  gchar *fullname;

  g_return_val_if_fail (error == NULL || *error == NULL, NULL);

#ifdef ENABLE_NLS
  bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);
#endif /* ENABLE_NLS */

  have_mplayer = ogmrip_check_mplayer ();

  fullname = g_find_program_in_path (PROGRAM);
  have_aften = fullname != NULL;
  g_free (fullname);

  ac3_plugin.type = ogmrip_ac3_get_type ();

  if (have_mplayer && have_aften)
    return &ac3_plugin;

  if (!have_mplayer && !have_aften)
    g_set_error (error, OGMRIP_PLUGIN_ERROR, OGMRIP_PLUGIN_ERROR_REQ, _("MPlayer and Aften are missing"));
  else if (!have_mplayer)
    g_set_error (error, OGMRIP_PLUGIN_ERROR, OGMRIP_PLUGIN_ERROR_REQ, _("MPlayer is missing"));
  else if (!have_aften)
    g_set_error (error, OGMRIP_PLUGIN_ERROR, OGMRIP_PLUGIN_ERROR_REQ, _("Aften is missing"));

  return NULL;
}

