/*
 *  Copyright (c) 2011 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "ColorConverters.h"

#include "Macros_p.h"
#include "PixelDescription.h"
#include "Type.h"

#include "RgbColorConverter_p.h"
#include "GrayColorConverter_p.h"
#include "AbstractColorConverter_p.h"


using namespace GTLCore;

struct ColorConverters::Private
{
  Private() : sgrayaU8ColorConverterInstance(0), sgrayU8ColorConverterInstance(0),
              sgrayaU16ColorConverterInstance(0), sgrayU16ColorConverterInstance(0),
              sgrayaF32ColorConverterInstance(0), sgrayF32ColorConverterInstance(0),
              srgbaU8ColorConverterInstance(0), srgbU8ColorConverterInstance(0),
              srgbaU16ColorConverterInstance(0), srgbU16ColorConverterInstance(0),
              srgbaF32ColorConverterInstance(0), srgbF32ColorConverterInstance(0)
  {
  }
  ~Private()
  {
    delete sgrayaU8ColorConverterInstance;
    delete sgrayU8ColorConverterInstance;
    delete sgrayaU16ColorConverterInstance;
    delete sgrayU16ColorConverterInstance;
    delete sgrayaF32ColorConverterInstance;
    delete sgrayF32ColorConverterInstance;
    
    delete srgbaU8ColorConverterInstance;
    delete srgbU8ColorConverterInstance;
    delete srgbaU16ColorConverterInstance;
    delete srgbU16ColorConverterInstance;
    delete srgbaF32ColorConverterInstance;
    delete srgbF32ColorConverterInstance;
  }
  
  GrayColorConverter<gtl_uint8, true>* sgrayaU8ColorConverterInstance;
  GrayColorConverter<gtl_uint8, false>* sgrayU8ColorConverterInstance;
  GrayColorConverter<gtl_uint16, true>* sgrayaU16ColorConverterInstance;
  GrayColorConverter<gtl_uint16, false>* sgrayU16ColorConverterInstance;
  GrayColorConverter<float, true>* sgrayaF32ColorConverterInstance;
  GrayColorConverter<float, false>* sgrayF32ColorConverterInstance;

  RgbColorConverter<gtl_uint8, true>* srgbaU8ColorConverterInstance;
  RgbColorConverter<gtl_uint8, false>* srgbU8ColorConverterInstance;
  RgbColorConverter<gtl_uint16, true>* srgbaU16ColorConverterInstance;
  RgbColorConverter<gtl_uint16, false>* srgbU16ColorConverterInstance;
  RgbColorConverter<float, true>* srgbaF32ColorConverterInstance;
  RgbColorConverter<float, false>* srgbF32ColorConverterInstance;

};

STATIC_POINTER(ColorConverters, s_instance);

ColorConverters::ColorConverters() : d(new Private)
{

}

ColorConverters::~ColorConverters()
{
  delete d;
}


const ColorConverters* ColorConverters::instance()
{
  if( not s_instance )
  {
    s_instance = new ColorConverters;
  }
  return s_instance;
}

const AbstractColorConverter* ColorConverters::guess(const GTLCore::PixelDescription& _pixelDescription) const
{
  if(_pixelDescription.hasSameTypeChannels())
  {
    return guess(_pixelDescription.channelTypes()[0], _pixelDescription.channels());
  } else {
    return 0;
  }
}

const AbstractColorConverter* ColorConverters::guess(const GTLCore::Type* _type, int _channels) const
{
  GTLCore::Type::DataType dt = _type->dataType();
  switch(_channels)
  {
    case 1:
    {
      switch(dt)
      {
        case Type::UNSIGNED_INTEGER8:
          return sGrayU8();
        case Type::UNSIGNED_INTEGER16:
          return sGrayU16();
        case Type::FLOAT32:
          return sGrayF32();
        default:
          break;
      }
    }
    case 2:
    {
      switch(dt)
      {
        case Type::UNSIGNED_INTEGER8:
          return sGrayaU8();
        case Type::UNSIGNED_INTEGER16:
          return sGrayaU16();
        case Type::FLOAT32:
          return sGrayaF32();
        default:
          break;
      }
    }
    case 3:
    {
      switch(dt)
      {
        case Type::UNSIGNED_INTEGER8:
          return sRgbU8();
        case Type::UNSIGNED_INTEGER16:
          return sRgbU16();
        case Type::FLOAT32:
          return sRgbF32();
        default:
          break;
      }
    }
    case 4:
    {
      switch(dt)
      {
        case Type::UNSIGNED_INTEGER8:
          return sRgbaU8();
        case Type::UNSIGNED_INTEGER16:
          return sRgbaU16();
        case Type::FLOAT32:
          return sRgbaF32();
        default:
          break;
      }
    }
    default:
      break;
  }
  return 0;
}

const AbstractColorConverter* ColorConverters::sGrayaU8() const
{
  if(d->sgrayaU8ColorConverterInstance == 0)
  {
    d->sgrayaU8ColorConverterInstance = new GrayColorConverter<gtl_uint8, true>(2.2, PixelDescription(Type::UnsignedInteger8, 2, 1) );
    d->sgrayaU8ColorConverterInstance->d->deletable = false;
  }
  return d->sgrayaU8ColorConverterInstance;
}

const AbstractColorConverter* ColorConverters::sGrayU8() const
{
  if(d->sgrayU8ColorConverterInstance == 0)
  {
    d->sgrayU8ColorConverterInstance = new GrayColorConverter<gtl_uint8, false>(2.2, PixelDescription(Type::UnsignedInteger8, 1) );
    d->sgrayU8ColorConverterInstance->d->deletable = false;
  }
  return d->sgrayU8ColorConverterInstance;
}

const AbstractColorConverter* ColorConverters::sGrayaU16() const
{
  if(d->sgrayaU16ColorConverterInstance == 0)
  {
    d->sgrayaU16ColorConverterInstance = new GrayColorConverter<gtl_uint16, true>(2.2, PixelDescription(Type::UnsignedInteger16, 2, 1) );
    d->sgrayaU16ColorConverterInstance->d->deletable = false;
  }
  return d->sgrayaU16ColorConverterInstance;
}

const AbstractColorConverter* ColorConverters::sGrayU16() const
{
  if(d->sgrayU16ColorConverterInstance == 0)
  {
    d->sgrayU16ColorConverterInstance = new GrayColorConverter<gtl_uint16, false>(2.2, PixelDescription(Type::UnsignedInteger16, 1) );
    d->sgrayU16ColorConverterInstance->d->deletable = false;
  }
  return d->sgrayU16ColorConverterInstance;
}

const AbstractColorConverter* ColorConverters::sGrayaF32() const
{
  if(d->sgrayaF32ColorConverterInstance == 0)
  {
    d->sgrayaF32ColorConverterInstance = new GrayColorConverter<float, true>(2.2, PixelDescription(Type::UnsignedInteger16, 2, 1) );
    d->sgrayaF32ColorConverterInstance->d->deletable = false;
  }
  return d->sgrayaF32ColorConverterInstance;
}

const AbstractColorConverter* ColorConverters::sGrayF32() const
{
  if(d->sgrayF32ColorConverterInstance == 0)
  {
    d->sgrayF32ColorConverterInstance = new GrayColorConverter<float, false>(2.2, PixelDescription(Type::UnsignedInteger16, 1) );
    d->sgrayF32ColorConverterInstance->d->deletable = false;
  }
  return d->sgrayF32ColorConverterInstance;
}

const AbstractColorConverter* ColorConverters::sRgbaU8() const
{
  if(d->srgbaU8ColorConverterInstance == 0)
  {
    d->srgbaU8ColorConverterInstance = new RgbColorConverter<gtl_uint8, true>(2.2, PixelDescription(Type::UnsignedInteger8, 4, 3) );
    d->srgbaU8ColorConverterInstance->d->deletable = false;
  }
  return d->srgbaU8ColorConverterInstance;
}

const AbstractColorConverter* ColorConverters::sRgbU8() const
{
  if(d->srgbU8ColorConverterInstance == 0)
  {
    d->srgbU8ColorConverterInstance = new RgbColorConverter<gtl_uint8, false>(2.2, PixelDescription(Type::UnsignedInteger8, 3) );
    d->srgbU8ColorConverterInstance->d->deletable = false;
  }
  return d->srgbU8ColorConverterInstance;
}

const AbstractColorConverter* ColorConverters::sRgbaU16() const
{
  if(d->srgbaU16ColorConverterInstance == 0)
  {
    d->srgbaU16ColorConverterInstance = new RgbColorConverter<gtl_uint16, true>(2.2, PixelDescription(Type::UnsignedInteger16, 4, 3) );
    d->srgbaU16ColorConverterInstance->d->deletable = false;
  }
  return d->srgbaU16ColorConverterInstance;
}

const AbstractColorConverter* ColorConverters::sRgbU16() const
{
  if(d->srgbU16ColorConverterInstance == 0)
  {
    d->srgbU16ColorConverterInstance = new RgbColorConverter<gtl_uint16, false>(2.2, PixelDescription(Type::UnsignedInteger16, 3) );
    d->srgbU16ColorConverterInstance->d->deletable = false;
  }
  return d->srgbU16ColorConverterInstance;
}

const AbstractColorConverter* ColorConverters::sRgbaF32() const
{
  if(d->srgbaF32ColorConverterInstance == 0)
  {
    d->srgbaF32ColorConverterInstance = new RgbColorConverter<float, true>(2.2, PixelDescription(Type::UnsignedInteger16, 4, 3) );
    d->srgbaF32ColorConverterInstance->d->deletable = false;
  }
  return d->srgbaF32ColorConverterInstance;
}

const AbstractColorConverter* ColorConverters::sRgbF32() const
{
  if(d->srgbF32ColorConverterInstance == 0)
  {
    d->srgbF32ColorConverterInstance = new RgbColorConverter<float, false>(2.2, PixelDescription(Type::UnsignedInteger16, 3) );
    d->srgbF32ColorConverterInstance->d->deletable = false;
  }
  return d->srgbF32ColorConverterInstance;
}


