/*
 *  Copyright (c) 2007 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _GTLCORE_VALUE_H_
#define _GTLCORE_VALUE_H_

#include <vector>

#include <GTLCore/Export.h>
#include <GTLCore/StdTypes.h>

namespace GTLCore {
  class Color;
  class Type;
  /**
   * Value is a class use to hold a primitive value, like a float, an integer or a boolean.
   * @ingroup GTLCore
   */
  class GTLCORE_EXPORT Value {
    public:
      Value();
      Value(float v);
      Value(bool v);
      Value(gtl_int32 v);
      Value(gtl_uint32 v);
      Value(gtl_uint64 v);
      Value(const std::vector< Value >& v, const GTLCore::Type* _type = 0);
      Value(const Color& c);
      Value(const Value& v);
      Value& operator=(const Value& rhs);
      ~Value();
    public:
      bool operator==(const Value& rhs) const;
    public:
      /**
       * @return true if it is valid
       */
      bool isValid() const;
      /**
       * @return the value as a float
       */
      float asFloat32() const;
      /**
       * Set the value to be a float.
       */
      void setFloat32( float _v );
      /**
       * @return the value as a boolean
       */
      bool asBoolean() const;
      /**
       * Set the value to be a boolean.
       */
      void setBoolean( bool _v);
      /**
       * @return the value as an integer
       */
      gtl_int32 asInt32() const;
      /**
       * Set the value to be an integer
       */
      void setInt32(gtl_int32 v);
      /**
       * @return the value as an integer
       */
      gtl_int64 asInt64() const;
      /**
       * Set the value to be an integer
       */
      void setInt64(gtl_int64 v);
      /**
       * @return the value as an unsigned integer
       */
      gtl_uint32 asUnsignedInt32() const;
      /**
       * @return the value as an array, or 0 if not an array
       */
      const std::vector< Value >* asArray() const;
      /**
       * @return the value as an array, or 0 if not an array
       */
      void setArray( const std::vector< Value >& _array, const GTLCore::Type* _type );
      /**
       *
       */
      Color asColor() const;
      void setColor(const Color& c);
      /**
       * @return the type of the value.
       */
      const Type* type() const;
    private:
      void deref();
      struct Private;
      Private* d;
  };
}

#endif
