/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "GTLFragment/MetadataParser_p.h"
#include "GTLFragment/Metadata.h"

#include "GTLCore/Color.h"
#include "GTLCore/CompilationMessage.h"
#include "GTLCore/Debug.h"
#include "GTLCore/Type.h"
#include "GTLCore/TypesManager.h"
#include "GTLCore/Value.h"
#include "GTLCore/Metadata/TextEntry.h"
#include "GTLCore/Metadata/Group.h"
#include "GTLCore/Metadata/ValueEntry.h"
#include "GTLCore/Metadata/ParameterEntry.h"
#include <GTLCore/CompilationMessages.h>

#define TESTMEDATAPARSER_TEST_TEXT_ENTRY( entry, txt ) \
  GTLTEST_CHECK_NOT_EQUAL( entry, 0 ) \
  GTLTEST_CHECK_EQUAL( entry->asTextEntry(), entry ) \
  GTLTEST_CHECK_EQUAL( entry->asTextEntry()->text(), txt )

#define TESTMEDATAPARSER_TEST_VALUE_ENTRY( entry, val ) \
  GTLTEST_CHECK_NOT_EQUAL( entry, 0 ) \
  GTLTEST_CHECK_EQUAL( entry->asValueEntry(), entry ) \
  GTLTEST_CHECK_EQUAL( entry->asValueEntry()->value(), GTLCore::Value( val ) )

class TestMetadataParser : public GTLTest::Case {
  public:
    TestMetadataParser() : GTLTest::Case("MetadataParser") {}
    virtual void runTest()
    {
      std::istringstream iss(" \
< \
  version: 0; \
  info: < \
    author: \"Joe Doe; Joe Doe Jr\"; \
    vendor: < \
      name: \"DoeGraphics\"; \
      address: \"1242 Main Street\"; \
    >; \
    license: \"LGPLv2+\"; \
  >; \
  parameters: < \
    param1: < \
      type: int; \
      minValue:-10; \
      maxValue: 100; \
      defaultValue: 50; \
      description: \"This is the first parameter\"; \
    >; \
    categorie2: < \
      description: \"This is a categorie of parameters\"; \
      param2: < \
        type: curve; \
        defaultValue: {{0.25,0},{1,0.5}}; \
      >; \
      param3: < \
          type: color; \
          defaultValue: {1,0,0}; \
        >; \
     >; \
  >; \
>");
      GTLFragment::MetadataLexer* lng = new GTLFragment::MetadataLexer(&iss);
      GTLFragment::MetadataParser parser( lng, "");
      GTLFragment::Metadata* km = parser.parse();
      GTLTEST_CHECK_EQUAL( parser.errorMessages().errors().size(), 0 );
      GTLTEST_CHECK_EQUAL( km->version(), 0 );
      
      // Test info
      TESTMEDATAPARSER_TEST_TEXT_ENTRY( km->informationRecords()->entry( "author" ), "Joe Doe; Joe Doe Jr");
      TESTMEDATAPARSER_TEST_TEXT_ENTRY( km->informationRecords()->entry( "license" ), "LGPLv2+");
      GTLTEST_CHECK_NOT_EQUAL( km->informationRecords()->entry( "vendor" ), 0 );
      const GTLCore::Metadata::Group* vendorGroup = km->informationRecords()->entry( "vendor" )->asGroup();
      GTLTEST_CHECK_NOT_EQUAL( vendorGroup, 0 );
      TESTMEDATAPARSER_TEST_TEXT_ENTRY( vendorGroup->entry( "name" ), "DoeGraphics");
      TESTMEDATAPARSER_TEST_TEXT_ENTRY( vendorGroup->entry( "address" ), "1242 Main Street");
      
      // Test parameters
      GTLTEST_CHECK_NOT_EQUAL( km->parameters()->entry( "param1" ), 0 );
      const GTLCore::Metadata::ParameterEntry* param1 = km->parameters()->entry( "param1" )->asParameterEntry();
      GTLTEST_CHECK_NOT_EQUAL( param1, 0 );
      TESTMEDATAPARSER_TEST_TEXT_ENTRY( param1->entry( "type" ), "int");
      TESTMEDATAPARSER_TEST_TEXT_ENTRY( param1->entry( "description" ), "This is the first parameter");
      GTLTEST_CHECK_EQUAL( param1->description(), "This is the first parameter");
      TESTMEDATAPARSER_TEST_VALUE_ENTRY( param1->entry( "minValue" ), -10);
      GTLTEST_CHECK_EQUAL( param1->minimumValue(), -10);
      TESTMEDATAPARSER_TEST_VALUE_ENTRY( param1->entry( "defaultValue" ), 50);
      GTLTEST_CHECK_EQUAL( param1->defaultValue(), 50);
      TESTMEDATAPARSER_TEST_VALUE_ENTRY( param1->entry( "maxValue" ), 100);
      GTLTEST_CHECK_EQUAL( param1->maximumValue(), 100);
      
      // Test categorie parameter
      GTLTEST_CHECK_NOT_EQUAL( km->parameters()->entry( "categorie2" ), 0 );
      const GTLCore::Metadata::Group* categorie2 = km->parameters()->entry( "categorie2" )->asGroup();
      GTLTEST_CHECK_NOT_EQUAL( categorie2, 0 );
      TESTMEDATAPARSER_TEST_TEXT_ENTRY( categorie2->entry( "description" ), "This is a categorie of parameters");
      
      // Test parameter2
      GTLTEST_CHECK_NOT_EQUAL( categorie2->entry("param2"), 0 );
      const GTLCore::Metadata::ParameterEntry* param2 = categorie2->entry( "param2" )->asParameterEntry();
      GTLTEST_CHECK_NOT_EQUAL( param2, 0 );
      TESTMEDATAPARSER_TEST_TEXT_ENTRY( param2->entry( "type" ), "curve");
      const GTLCore::Type* typeVector2 = GTLCore::TypesManager::getVector( GTLCore::Type::Float32, 2 );
      std::vector< GTLCore::Value > v1; v1.push_back( 0.25f ); v1.push_back( 0.0f );
      std::vector< GTLCore::Value > v2; v2.push_back( 1.0f ); v2.push_back( 0.5f );
      std::vector< GTLCore::Value > v3; v3.push_back( GTLCore::Value( v1, typeVector2 ) ); v3.push_back( GTLCore::Value( v2, typeVector2 ) );
      GTLCore::Value param2value( v3, GTLCore::TypesManager::getArray( typeVector2 ) );
      TESTMEDATAPARSER_TEST_VALUE_ENTRY( param2->entry( "defaultValue" ), param2value);
      GTLTEST_CHECK_EQUAL( param2->defaultValue(), param2value);
      
      // Test parameter3
      GTLTEST_CHECK_NOT_EQUAL( categorie2->entry("param3"), 0 );
      const GTLCore::Metadata::ParameterEntry* param3 = categorie2->entry( "param3" )->asParameterEntry();
      GTLTEST_CHECK_NOT_EQUAL( param3, 0 );
      TESTMEDATAPARSER_TEST_TEXT_ENTRY( param3->entry( "type" ), "color");
      GTLCore::Color param3value(1.0, 0.0, 0.0);
      TESTMEDATAPARSER_TEST_VALUE_ENTRY( param3->entry( "defaultValue" ), param3value);
      GTLTEST_CHECK_EQUAL( param3->defaultValue(), param3value);
      GTLTEST_CHECK_EQUAL( param3->defaultValue().type(), param3->type());
    }
};
