/*
 *  Copyright (c) 2008, 2010 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "LibrariesManager.h"

#include <map>

// LLVM
#ifdef LLVM_27_OR_28
#include <llvm/System/Mutex.h>
#include <llvm/Support/MutexGuard.h>
#include <llvm/System/Path.h>
#else
#include <llvm/Support/Mutex.h>
#include <llvm/Support/MutexGuard.h>
#include <llvm/Support/Path.h>
#endif

#include "GTLCore/Debug.h"
#include "Library.h"

#include <GTLCore/Macros_p.h>
#include <GTLFragment/LibrariesManager.h>

using namespace OpenRijn;

struct LibrariesManager::Private {
  std::map<GTLCore::String, Library*> libraries;
  std::list<GTLCore::String> directories;
  llvm::sys::Mutex mutex;
};

STATIC_POINTER(LibrariesManager, s_instance);

LibrariesManager::LibrariesManager() : d(new Private)
{
  addDirectory( "." );
  addDirectory( _OPENRIJN_RIJN_STD_LIB_SRC_DIR_ );
  addDirectory( _OPENRIJN_RIJN_LIBRARIES_DIR_ );
  GTLFragment::LibrariesManager::instance()->registerLibrariesManager(this);
}

LibrariesManager::~LibrariesManager()
{
  for(std::map<GTLCore::String, Library*>::iterator it = d->libraries.begin();
      it != d->libraries.end(); ++it)
  {
    delete it->second;
  }
  delete d;
}

void LibrariesManager::addDirectory(const GTLCore::String& directory)
{
  d->directories.push_back( directory );
}

GTLFragment::Library* LibrariesManager::library(const GTLCore::String& name, int _channelsNb)
{
  GTL_ASSERT(_channelsNb == -1);
  std::map<GTLCore::String, Library*>::iterator it = d->libraries.find(name);
  if( it == d->libraries.end())
  {
    return 0;
  }
  return it->second;
}


GTLFragment::Library* LibrariesManager::loadLibrary( const GTLCore::String& name, int _channelsNb)
{
  llvm::MutexGuard mg(d->mutex);
  GTL_ASSERT(_channelsNb == -1);
  GTLFragment::Library* l = library( name, _channelsNb );
  if(not l )
  {
    GTLCore::String sourceName = name + ".rijn";
    for( std::list<GTLCore::String>::iterator it = d->directories.begin();
        it != d->directories.end(); ++it )
    {
      llvm::sys::Path path( (const std::string&)*it );
      path.appendComponent( (const std::string&)sourceName);
      GTL_DEBUG("try " << path.c_str() );
      if(path.exists() and path.canRead())
      {
        Library* osl = new Library( Library::RIJN_LIBRARY );
        registerLibrary( name, osl );
        osl->loadFromFile( path.c_str());
        l = osl;
        break;
      }
    }
  }
  return l;
}

void LibrariesManager::registerLibrary(const GTLCore::String& name, Library* l)
{
  llvm::MutexGuard mg(d->mutex);
  d->libraries[ name ] = l;
}

LibrariesManager* LibrariesManager::instance()
{
  if( not s_instance )
  {
    s_instance = new LibrariesManager;
  }
  return s_instance;
}

Library::Type LibrariesManager::libraryType() const
{
  return Library::RIJN_LIBRARY;
}
