/*
 *  Copyright (c) 2009,2010 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "AbstractColorConverter.h"
#include "StdTypes.h"
#include "FloatHalfConverter_p.h"
#include "Utils_p.h"
#include "PixelDescription.h"
#include "Type.h"
#include "Debug.h"
#include "Macros_p.h"

#include "AbstractColorConverter_p.h"

using namespace GTLCore;

class GTLCore::Channel
{
public:
    virtual ~Channel() { }
    virtual float scaleToF32(const char* srcPixel) const = 0;
    virtual void scaleFromF32(char* dstPixel, float value) const = 0;
};

template<typename _ChannelType_>
class ChannelImpl : public Channel
{
public:
    ChannelImpl(std::size_t _pos) : m_pos(_pos) { }
    virtual ~ChannelImpl() {}
    virtual float scaleToF32(const char* srcPixel) const;
    virtual void scaleFromF32(char* dstPixel, float value) const;
private:
    inline const _ChannelType_* channel(const char* pixel) const {
        return reinterpret_cast<const _ChannelType_*>(pixel + m_pos);
    }
    inline _ChannelType_* channel(char* pixel) const {
        return reinterpret_cast<_ChannelType_*>(pixel + m_pos);
    }
private:
    std::size_t m_pos;
};

template<>
float ChannelImpl<gtl_uint8>::scaleToF32(const char* srcPixel) const {
  return gtl_uint8(*channel(srcPixel) / 255.0);
}

template<>
void ChannelImpl<gtl_uint8>::scaleFromF32(char* dstPixel, float value) const {
  *channel(dstPixel) = gtl_uint8(value * 255);
}

template<>
float ChannelImpl<gtl_uint16>::scaleToF32(const char* srcPixel) const {
  return gtl_uint16(*channel(srcPixel) / float(UINT16_MAX));
}

template<>
void ChannelImpl<gtl_uint16>::scaleFromF32(char* dstPixel, float value) const {
  *channel(dstPixel) = gtl_uint16(value * UINT16_MAX);
}

template<>
float ChannelImpl<float>::scaleToF32(const char* srcPixel) const {
  return *channel(srcPixel);
}

template<>
void ChannelImpl<float>::scaleFromF32(char* dstPixel, float value) const {
  *channel(dstPixel) = value;
}


class ChannelHalf : public Channel
{
public:
    ChannelHalf(std::size_t _pos) : m_pos(_pos) { }
    virtual ~ChannelHalf() {}
    virtual float scaleToF32(const char* srcPixel) const { return half2float(*channel(srcPixel)); }
    virtual void scaleFromF32(char* dstPixel, float value) const { *channel(dstPixel) = float2half(value); }
private:
    inline const half* channel(const char* pixel) const {
        return reinterpret_cast<const half*>(pixel + m_pos);
    }
    inline half* channel(char* pixel) const {
        return reinterpret_cast<half*>(pixel + m_pos);
    }
private:
    std::size_t m_pos;
};

AbstractColorConverter::AbstractColorConverter(const GTLCore::PixelDescription& _pixelDescription) : d(new Private(_pixelDescription))
{
  int currentPos = 0;
  for(std::size_t i = 0; i < _pixelDescription.channels(); ++i)
  {
    switch(_pixelDescription.channelTypes()[i]->dataType())
    {
      case Type::UNSIGNED_INTEGER8:
        d->channels.push_back(new ChannelImpl<gtl_uint8>(currentPos));
        currentPos += 1;
        break;
      case Type::UNSIGNED_INTEGER16:
        d->channels.push_back(new ChannelImpl<gtl_uint16>(currentPos));
        currentPos += 2;
        break;
      case Type::FLOAT16:
        d->channels.push_back(new ChannelHalf(currentPos));
        currentPos += 2;
        break;
      case Type::FLOAT32:
        d->channels.push_back(new ChannelImpl<float>(currentPos));
        currentPos += 4;
        break;
      default:
        GTL_ABORT("Unimplemnted");
    }
  }
}

AbstractColorConverter::~AbstractColorConverter()
{
  deleteAll(d->channels);
  delete d;
}

void AbstractColorConverter::release(const GTLCore::AbstractColorConverter* _acc)
{
  if(_acc and _acc->d->deletable)
  {
    delete _acc;
  }
}

void AbstractColorConverter::vectorToRgba(const float* _data, RgbaF* _rgba) const
{
  char* buffer = new char[d->pixelDescription.bitsSize() / 8];
  for(std::size_t i = 0; i < d->pixelDescription.channels(); ++i)
  {
    d->channels[i]->scaleFromF32(buffer, _data[i]);
  }
  pixelToRgba(buffer, _rgba);
}

void AbstractColorConverter::rgbaToVector(const RgbaF* _rgba, float* _data) const
{
  char* buffer = new char[d->pixelDescription.bitsSize() / 8];
  rgbaToPixel(_rgba, buffer);
  for(std::size_t i = 0; i < d->pixelDescription.channels(); ++i)
  {
    _data[i] = d->channels[i]->scaleToF32(buffer);
  }
}
