/*
 *  Copyright (c) 2010 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _GTL_CORE_TIME_MACHINE_VALUE_P_H_
#define _GTL_CORE_TIME_MACHINE_VALUE_P_H_

#include "Value.h"
#include "TimeController.h"

namespace GTLCore {
  /**
    * @internal
    * @ingroup GTLCore_TimeMachine
    *
    * This class control a @ref Value when the time changed.
    */
  class TimeValue {
    public:
      virtual ~TimeValue();
      virtual Value value(double t) const = 0;
  };
  template<typename _T_>
  class TimeNumericValue : public TimeValue {
    public:
      TimeNumericValue(_T_ min, _T_ max, const TimeController* timeController) : m_min(min), m_max(max), m_timeController(timeController) {}
      virtual ~TimeNumericValue() {}
      virtual Value value(double t) const
      {
        t = m_timeController->valueAt(t);
        return _T_(t * m_max + (1-t) * m_min);
      }
    private:
      _T_ m_min, m_max;
      const TimeController* m_timeController;
  };
  class TimeVectorArrayValue : public TimeValue {
    public:
      TimeVectorArrayValue( const Type* type, std::vector<TimeValue*> timeValues );
      virtual ~TimeVectorArrayValue();
      virtual Value value(double t) const;
    private:
      const Type* m_type;
      std::vector<TimeValue*> m_timeValues;
  };
  class TimeColorValue : public TimeValue {
    public:
      TimeColorValue( const TimeController* redController, const TimeController* greenController, const TimeController* blueController, const TimeController* alphaController );
      virtual ~TimeColorValue();
      virtual Value value(double t) const;
    private:
      const TimeController* m_red;
      const TimeController* m_green;
      const TimeController* m_blue;
      const TimeController* m_alpha;
  };
}

#endif
