/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _GTLCORE_UTILS_P_H_
#define _GTLCORE_UTILS_P_H_

#include <config-endian.h>
#include "StdTypes.h"

namespace GTLCore {

  /**
   * @internal
   * Delete all pointers of a list or a vector.
   * @ingroup GTLCore
   */
  template<class _Type_>
  inline void deleteAll( _Type_& list )
  {
    for( typename _Type_::iterator it = list.begin(); it != list.end(); ++it)
    {
      delete *it;
    }
  }
  /**
   * @internal
   * Delete all second members of a map
   * @ingroup GTLCore
   */
  template<class _Type_>
  inline void deleteAllSecondMember( _Type_& map )
  {
    for( typename _Type_::iterator it = map.begin(); it != map.end(); ++it)
    {
      delete it->second;
    }
  }
  
  template<typename _T_>
  inline _T_ min( _T_ t1, _T_ t2 )
  {
    return t1 < t2 ? t1 : t2;
  }
  
  template<typename _T_>
  inline _T_ max( _T_ t1, _T_ t2 )
  {
    return t1 > t2 ? t1 : t2;
  }
  
  template<typename _T_>
  inline _T_ bound( _T_ min, _T_ val, _T_ max )
  {
    if( val < min ) return min;
    if( val > max ) return max;
    return val;
  }
  
template <typename T> inline void toUnaligned(const T src, unsigned char *dest)
{
    memcpy(dest, &src, sizeof(T));
}

#define MAKE_UINT64(c) c ## LL

template <typename T> T bswap(T source);
template <> inline gtl_uint64 bswap<gtl_uint64>(gtl_uint64 source)
{
    return 0
        | ((source & MAKE_UINT64(0x00000000000000ff)) << 56)
        | ((source & MAKE_UINT64(0x000000000000ff00)) << 40)
        | ((source & MAKE_UINT64(0x0000000000ff0000)) << 24)
        | ((source & MAKE_UINT64(0x00000000ff000000)) << 8)
        | ((source & MAKE_UINT64(0x000000ff00000000)) >> 8)
        | ((source & MAKE_UINT64(0x0000ff0000000000)) >> 24)
        | ((source & MAKE_UINT64(0x00ff000000000000)) >> 40)
        | ((source & MAKE_UINT64(0xff00000000000000)) >> 56);
}

template <> inline gtl_uint32 bswap<gtl_uint32>(gtl_uint32 source)
{
    return 0
        | ((source & 0x000000ff) << 24)
        | ((source & 0x0000ff00) << 8)
        | ((source & 0x00ff0000) >> 8)
        | ((source & 0xff000000) >> 24);
}

template <> inline gtl_uint16 bswap<gtl_uint16>(gtl_uint16 source)
{
    return 0
        | ((source & 0x00ff) << 8)
        | ((source & 0xff00) >> 8);
}

// signed specializations
template <> inline gtl_int64 bswap<gtl_int64>(gtl_int64 source)
{
    return bswap<gtl_uint64>(gtl_uint64(source));
}

template <> inline gtl_int32 bswap<gtl_int32>(gtl_int32 source)
{
    return bswap<gtl_uint32>(gtl_uint32(source));
}

template <> inline gtl_int16 bswap<gtl_int16>(gtl_int16 source)
{
    return bswap<gtl_uint16>(gtl_uint16(source));
}

#ifdef WORDS_BIGENDIAN

template <typename T> inline T toBigEndian(T source)
{ return source; }
template <typename T> inline T fromBigEndian(T source)
{ return source; }
template <typename T> inline T toLittleEndian(T source)
{ return bswap<T>(source); }
template <typename T> inline T fromLittleEndian(T source)
{ return bswap<T>(source); }
template <typename T> inline void toBigEndian(T src, uchar *dest)
{ toUnaligned<T>(src, dest); }
template <typename T> inline void toLittleEndian(T src, uchar *dest)
{ bswap<T>(src, dest); }
#else // WORDS_BIGENDIAN

template <typename T> inline T toBigEndian(T source)
{ return bswap<T>(source); }
/**
 * Convert a value from big endian, when the cpu is little-endian (like x86).
 */
template <typename T> inline T fromBigEndian(T source)
{ return bswap<T>(source); }
template <typename T> inline T toLittleEndian(T source)
{ return source; }
template <typename T> inline T fromLittleEndian(T source)
{ return source; }
template <typename T> inline void toBigEndian(T src, unsigned char *dest)
{ bswap<T>(src, dest); }
template <typename T> inline void toLittleEndian(T src, unsigned char *dest)
{ toUnaligned<T>(src, dest); }

#endif // WORDS_BIGENDIAN  

}

#endif
