/*
 *  Copyright (c) 2010 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _OPENRIJN_ABSTRACT_BRUSH_H_
#define _OPENRIJN_ABSTRACT_BRUSH_H_

#include <OpenRijn/Export.h>

namespace GTLCore {
  class Color;
}

namespace OpenRijn {
  class AbstractCanvas;
  struct DrawingPoint;
  struct DistanceInformation;
  /**
   * @ingroup OpenRijn
   *
   * Represent a brush that move around lines and curves to draw.
   */
  class OPENRIJN_EXPORT AbstractBrush {
      GTL_NO_COPY(AbstractBrush)
    public:
      AbstractBrush();
      virtual ~AbstractBrush();
      /**
       * @return true if this brush can paint a single point
       */
      bool supportDrawPoint() const;
      /**
       * @return paint a single point, and return the spacing to the next point on a line.
       */
      virtual float drawPoint( AbstractCanvas*, const DrawingPoint&) = 0;
      /**
       * Paint a line, the default implementation will make multiple call to @p paintAt seperated
       * by the spacing return by @p paintAt
       */
      virtual DistanceInformation drawLine( OpenRijn::AbstractCanvas* canvas, const OpenRijn::DrawingPoint& pt1, const OpenRijn::DrawingPoint& pt2, const OpenRijn::DistanceInformation& information);
      /**
       * Paint following a bezier curve, the default implementation will make multiple call
       * to @p paintAt seperated by the spacing return by @p paintAt
       */
      virtual DistanceInformation drawCurve(  OpenRijn::AbstractCanvas* canvas, const OpenRijn::DrawingPoint& pt1, float x1, float y1, float x2, float y2, const OpenRijn::DrawingPoint& pt2, const OpenRijn::DistanceInformation& information);
    public:
      /**
       * Set the color for this brush
       */
      void setPaintColor(const GTLCore::Color& color);
      /**
       * @return the color used for this brush
       */
      GTLCore::Color paintColor() const;
    protected:
      /**
       * Call this with the @p value set to false if this brush does not work with to draw a single
       * point.
       */
      void setSupportDrawPoint(bool value);
    private:
      struct Private;
      Private* const d;
  };
}

#endif
