/******************************************************************************
 * Copied from "gdal" project. See licence below.
 *
 * Project:  ISO 8211 Access
 * Purpose:  Implements the DDFFieldDefn class.
 * Author:   Frank Warmerdam, warmerdam@pobox.com
 *
 ******************************************************************************
 * Copyright (c) 1999, Frank Warmerdam
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 ******************************************************************************
 * $Id: ossimDdffielddefn.cpp 15833 2009-10-29 01:41:53Z eshirschorn $
 */

#include <cstring>
#include <ossim/imaging/ossimIso8211.h>
#include <ossim/base/ossimNotifyContext.h>
#include <ossim/base/ossimCplUtil.h>
#include <ossim/base/ossimTrace.h>
#include <ctype.h>

// Static trace for debugging
static ossimTrace traceDebug("ossimDDFFieldDefn:debug");


/************************************************************************/
/*                            DDFFieldDefn()                            */
/************************************************************************/

ossimDDFFieldDefn::ossimDDFFieldDefn()

{
    poModule = NULL;
    pszTag = NULL;
    _fieldName = NULL;
    _arrayDescr = NULL;
    _formatControls = NULL;
    nSubfieldCount = 0;
    papoSubfields = NULL;
    bRepeatingSubfields = false;
    nFixedWidth = 0;
}

/************************************************************************/
/*                           ~DDFFieldDefn()                            */
/************************************************************************/

ossimDDFFieldDefn::~ossimDDFFieldDefn()

{
    int   i;

    free( pszTag );
    free( _fieldName );
    free( _arrayDescr );
    free( _formatControls );

    for( i = 0; i < nSubfieldCount; i++ )
        delete papoSubfields[i];
    free( papoSubfields );
}

/************************************************************************/
/*                            AddSubfield()                             */
/************************************************************************/

void ossimDDFFieldDefn::AddSubfield( const char *pszName, 
                                const char *pszFormat )

{
    ossimDDFSubfieldDefn *poSFDefn = new ossimDDFSubfieldDefn;

    poSFDefn->SetName( pszName );
    poSFDefn->SetFormat( pszFormat );
    AddSubfield( poSFDefn );
}

/************************************************************************/
/*                            AddSubfield()                             */
/************************************************************************/

void ossimDDFFieldDefn::AddSubfield( ossimDDFSubfieldDefn *poNewSFDefn,
                                int bDontAddToFormat )

{
    nSubfieldCount++;
    papoSubfields = (ossimDDFSubfieldDefn ** )
        realloc( papoSubfields, sizeof(void*) * nSubfieldCount );
    papoSubfields[nSubfieldCount-1] = poNewSFDefn;
    
    if( bDontAddToFormat )
        return;

/* -------------------------------------------------------------------- */
/*      Add this format to the format list.  We don't bother            */
/*      aggregating formats here.                                       */
/* -------------------------------------------------------------------- */
    if( _formatControls == NULL || strlen(_formatControls) == 0 )
    {
        free( _formatControls );
        _formatControls = strdup( "()" );
    }
    
    int nOldLen = (int)strlen(_formatControls);
    
    char *pszNewFormatControls = (char *) 
        malloc(nOldLen+3+strlen(poNewSFDefn->GetFormat()));
    
    strcpy( pszNewFormatControls, _formatControls );
    pszNewFormatControls[nOldLen-1] = '\0';
    if( pszNewFormatControls[nOldLen-2] != '(' )
        strcat( pszNewFormatControls, "," );
    
    strcat( pszNewFormatControls, poNewSFDefn->GetFormat() );
    strcat( pszNewFormatControls, ")" );
    
    free( _formatControls );
    _formatControls = pszNewFormatControls;

/* -------------------------------------------------------------------- */
/*      Add the subfield name to the list.                              */
/* -------------------------------------------------------------------- */
    if( _arrayDescr == NULL )
        _arrayDescr = strdup("");

    _arrayDescr = (char *) 
        realloc(_arrayDescr, 
                   strlen(_arrayDescr)+strlen(poNewSFDefn->GetName())+2);
    if( strlen(_arrayDescr) > 0 )
        strcat( _arrayDescr, "!" );
    strcat( _arrayDescr, poNewSFDefn->GetName() );
}

/************************************************************************/
/*                               Create()                               */
/*                                                                      */
/*      Initialize a new field defn from application input, instead     */
/*      of from an existing file.                                       */
/************************************************************************/

int ossimDDFFieldDefn::Create( const char *pszTag, const char *pszFieldName, 
                          const char *pszDescription, 
                          DDF_data_struct_code eDataStructCode,
                          DDF_data_type_code   eDataTypeCode,
                          const char *pszFormat )

{
   // assert( this->pszTag == NULL );
   poModule = NULL;
   this->pszTag = strdup( pszTag );
   _fieldName = strdup( pszFieldName );
   _arrayDescr = strdup( pszDescription );
   _formatControls = strdup( "" );
   
   _data_struct_code = eDataStructCode;
   _data_type_code = eDataTypeCode;
   
   if( pszFormat != NULL )
      _formatControls = strdup( pszFormat );
   
   if( pszDescription != NULL && *pszDescription == '*' )
      bRepeatingSubfields = true;
   
   return true;
}

/************************************************************************/
/*                          GenerateDDREntry()                          */
/************************************************************************/

int ossimDDFFieldDefn::GenerateDDREntry( char **ppachData, 
                                    int *pnLength )

{
    *pnLength = 9 + (int)strlen(_fieldName) + 1 
        + (int)strlen(_arrayDescr) + 1
        + (int)strlen(_formatControls) + 1;

    if( strlen(_formatControls) == 0 )
        *pnLength -= 1;

    if( ppachData == NULL )
        return true;

    *ppachData = (char *) malloc( *pnLength+1 );
    
    if( _data_struct_code == dsc_elementary )
        (*ppachData)[0] = '0';
    else if( _data_struct_code == dsc_vector )
        (*ppachData)[0] = '1';
    else if( _data_struct_code == dsc_array )
        (*ppachData)[0] = '2';
    else if( _data_struct_code == dsc_concatenated )
        (*ppachData)[0] = '3';
    
    if( _data_type_code == dtc_char_string )
        (*ppachData)[1] = '0';
    else if( _data_type_code == dtc_implicit_point )
        (*ppachData)[1] = '1';
    else if( _data_type_code == dtc_explicit_point )
        (*ppachData)[1] = '2';
    else if( _data_type_code == dtc_explicit_point_scaled )
        (*ppachData)[1] = '3';
    else if( _data_type_code == dtc_char_bit_string )
        (*ppachData)[1] = '4';
    else if( _data_type_code == dtc_bit_string )
        (*ppachData)[1] = '5';
    else if( _data_type_code == dtc_mixed_data_type )
        (*ppachData)[1] = '6';

    (*ppachData)[2] = '0';
    (*ppachData)[3] = '0';
    (*ppachData)[4] = ';';
    (*ppachData)[5] = '&';
    (*ppachData)[6] = ' ';
    (*ppachData)[7] = ' ';
    (*ppachData)[8] = ' ';
    sprintf( *ppachData + 9, "%s%c%s", 
             _fieldName, OSSIM_DDF_UNIT_TERMINATOR, _arrayDescr );

    if( strlen(_formatControls) > 0 )
        sprintf( *ppachData + strlen(*ppachData), "%c%s",
                 OSSIM_DDF_UNIT_TERMINATOR, _formatControls );
    sprintf( *ppachData + strlen(*ppachData), "%c", OSSIM_DDF_FIELD_TERMINATOR );

    return true;
}

/************************************************************************/
/*                             Initialize()                             */
/*                                                                      */
/*      Initialize the field definition from the information in the     */
/*      DDR record.  This is called by DDFModule::Open().               */
/************************************************************************/

int ossimDDFFieldDefn::Initialize( ossimDDFModule * poModuleIn,
                              const char * pszTagIn, 
                              int nFieldEntrySize,
                              const char * pachFieldArea )

{
    int         iFDOffset = poModuleIn->GetFieldControlLength();
    int         nCharsConsumed;

    poModule = poModuleIn;
    
    pszTag = strdup( pszTagIn );

/* -------------------------------------------------------------------- */
/*      Set the data struct and type codes.                             */
/* -------------------------------------------------------------------- */
    switch( pachFieldArea[0] )
    {
      case '0':
        _data_struct_code = dsc_elementary;
        break;

      case '1':
        _data_struct_code = dsc_vector;
        break;

      case '2':
        _data_struct_code = dsc_array;
        break;

      case '3':
        _data_struct_code = dsc_concatenated;
        break;

      default:
         if (traceDebug())
         {
            ossimNotify(ossimNotifyLevel_WARN)
               << "Unrecognised data_struct_code value %c.\n"
               << "Field %s initialization incorrect.\n"
               << pachFieldArea[0]
               << pszTag
               << std::endl;
         }
        _data_struct_code = dsc_elementary;
    }

    switch( pachFieldArea[1] )
    {
      case '0':
        _data_type_code = dtc_char_string;
        break;
        
      case '1':
        _data_type_code = dtc_implicit_point;
        break;
        
      case '2':
        _data_type_code = dtc_explicit_point;
        break;
        
      case '3':
        _data_type_code = dtc_explicit_point_scaled;
        break;
        
      case '4':
        _data_type_code = dtc_char_bit_string;
        break;
        
      case '5':
        _data_type_code = dtc_bit_string;
        break;
        
      case '6':
        _data_type_code = dtc_mixed_data_type;
        break;

      default:
         if (traceDebug())
         {
            ossimNotify(ossimNotifyLevel_WARN)
               << "Unrecognised data_type_code value %c.\n"
               << "Field %s initialization incorrect.\n"
               << pachFieldArea[1] << pszTag << std::endl;
         }
        _data_type_code = dtc_char_string;
    }
    
/* -------------------------------------------------------------------- */
/*      Capture the field name, description (sub field names), and      */
/*      format statements.                                              */
/* -------------------------------------------------------------------- */

    _fieldName =
        ossimDDFFetchVariable( pachFieldArea + iFDOffset,
                          nFieldEntrySize - iFDOffset,
                          OSSIM_DDF_UNIT_TERMINATOR,
                          OSSIM_DDF_FIELD_TERMINATOR,
                          &nCharsConsumed );
    iFDOffset += nCharsConsumed;
    
    _arrayDescr =
        ossimDDFFetchVariable( pachFieldArea + iFDOffset,
                               nFieldEntrySize - iFDOffset,
                               OSSIM_DDF_UNIT_TERMINATOR,
                               OSSIM_DDF_FIELD_TERMINATOR, 
                               &nCharsConsumed );
    iFDOffset += nCharsConsumed;
    
    _formatControls =
        ossimDDFFetchVariable( pachFieldArea + iFDOffset,
                               nFieldEntrySize - iFDOffset,
                               OSSIM_DDF_UNIT_TERMINATOR,
                               OSSIM_DDF_FIELD_TERMINATOR, 
                               &nCharsConsumed );
    
/* -------------------------------------------------------------------- */
/*      Parse the subfield info.                                        */
/* -------------------------------------------------------------------- */
    if( _data_struct_code != dsc_elementary )
    {
        if( !BuildSubfields() )
            return false;

        if( !ApplyFormats() )
            return false;
    }
    
    return true;
}

/************************************************************************/
/*                                Dump()                                */
/************************************************************************/

/**
 * Write out field definition info to debugging file.
 *
 * A variety of information about this field definition, and all it's
 * subfields is written to the give debugging file handle.
 *
 * @param fp The standard io file handle to write to.  ie. stderr
 */

void ossimDDFFieldDefn::Dump( FILE * fp )

{
    const char  *pszValue = "";
    
    fprintf( fp, "  DDFFieldDefn:\n" );
    fprintf( fp, "      Tag = `%s'\n", pszTag );
    fprintf( fp, "      _fieldName = `%s'\n", _fieldName );
    fprintf( fp, "      _arrayDescr = `%s'\n", _arrayDescr );
    fprintf( fp, "      _formatControls = `%s'\n", _formatControls );

    switch( _data_struct_code )
    {
      case dsc_elementary:
        pszValue = "elementary";
        break;
        
      case dsc_vector:
        pszValue = "vector";
        break;
        
      case dsc_array:
        pszValue = "array";
        break;
        
      case dsc_concatenated:
        pszValue = "concatenated";
        break;
        
      default:
         // assert( false );
        pszValue = "(unknown)";
    }

    fprintf( fp, "      _data_struct_code = %s\n", pszValue );

    switch( _data_type_code )
    {
      case dtc_char_string:
        pszValue = "char_string";
        break;
        
      case dtc_implicit_point:
        pszValue = "implicit_point";
        break;
        
      case dtc_explicit_point:
        pszValue = "explicit_point";
        break;
        
      case dtc_explicit_point_scaled:
        pszValue = "explicit_point_scaled";
        break;
        
      case dtc_char_bit_string:
        pszValue = "char_bit_string";
        break;
        
      case dtc_bit_string:
        pszValue = "bit_string";
        break;
        
      case dtc_mixed_data_type:
        pszValue = "mixed_data_type";
        break;

      default:
         // assert( false );
        pszValue = "(unknown)";
        break;
    }
    
    fprintf( fp, "      _data_type_code = %s\n", pszValue );

    for( int i = 0; i < nSubfieldCount; i++ )
        papoSubfields[i]->Dump( fp );
}

/************************************************************************/
/*                           BuildSubfields()                           */
/*                                                                      */
/*      Based on the _arrayDescr build a set of subfields.              */
/************************************************************************/

int ossimDDFFieldDefn::BuildSubfields()

{
    char        **papszSubfieldNames;
    const char  *pszSublist = _arrayDescr;

    if( pszSublist[0] == '*' )
    {
        bRepeatingSubfields = true;
        pszSublist++;
    }

    papszSubfieldNames = ossimCSLTokenizeStringComplex( pszSublist, "!",
                                                   false, false );

    int nSFCount = ossimCSLCount( papszSubfieldNames );
    for( int iSF = 0; iSF < nSFCount; iSF++ )
    {
        ossimDDFSubfieldDefn *poSFDefn = new ossimDDFSubfieldDefn;
        
        poSFDefn->SetName( papszSubfieldNames[iSF] );
        AddSubfield( poSFDefn, true );
    }

    ossimCSLDestroy( papszSubfieldNames );

    return true;
}

/************************************************************************/
/*                          ExtractSubstring()                          */
/*                                                                      */
/*      Extract a substring terminated by a comma (or end of            */
/*      string).  Commas in brackets are ignored as terminated with     */
/*      bracket nesting understood gracefully.  If the returned         */
/*      string would being and end with a bracket then strip off the    */
/*      brackets.                                                       */
/*                                                                      */
/*      Given a string like "(A,3(B,C),D),X,Y)" return "A,3(B,C),D".    */
/*      Give a string like "3A,2C" return "3A".                         */
/************************************************************************/

char *ossimDDFFieldDefn::ExtractSubstring( const char * pszSrc )

{
    int         nBracket=0, i;
    char        *pszReturn;

    for( i = 0;
         pszSrc[i] != '\0' && (nBracket > 0 || pszSrc[i] != ',');
         i++ )
    {
        if( pszSrc[i] == '(' )
            nBracket++;
        else if( pszSrc[i] == ')' )
            nBracket--;
    }

    if( pszSrc[0] == '(' )
    {
        pszReturn = strdup( pszSrc + 1 );
        pszReturn[i-2] = '\0';
    }
    else
    {
        pszReturn = strdup( pszSrc  );
        pszReturn[i] = '\0';
    }

    return pszReturn;
}

/************************************************************************/
/*                            ExpandFormat()                            */
/************************************************************************/

char *ossimDDFFieldDefn::ExpandFormat( const char * pszSrc )

{
    int         nDestMax = 32;
    char       *pszDest = (char *) malloc(nDestMax+1);
    int         iSrc, iDst;
    int         nRepeat = 0;

    iSrc = 0;
    iDst = 0;
    pszDest[0] = '\0';

    while( pszSrc[iSrc] != '\0' )
    {
        /* This is presumably an extra level of brackets around some
           binary stuff related to rescaning which we don't care to do
           (see 6.4.3.3 of the standard.  We just strip off the extra
           layer of brackets */
        if( (iSrc == 0 || pszSrc[iSrc-1] == ',') && pszSrc[iSrc] == '(' )
        {
            char       *pszContents = ExtractSubstring( pszSrc+iSrc );
            char       *pszExpandedContents = ExpandFormat( pszContents );

            if( (int) (strlen(pszExpandedContents) + strlen(pszDest) + 1)
                > nDestMax )
            {
                nDestMax = 2 * ((int)strlen(pszExpandedContents) + (int)strlen(pszDest));
                pszDest = (char *) realloc(pszDest,nDestMax+1);
            }

            strcat( pszDest, pszExpandedContents );
            iDst = (int)strlen(pszDest);
            
            iSrc = iSrc + (int)strlen(pszContents) + 2;

            free( pszContents );
            free( pszExpandedContents );
        }

        /* this is a repeated subclause */
        else if( (iSrc == 0 || pszSrc[iSrc-1] == ',')
                 && isdigit(pszSrc[iSrc]) )
        {
            const char *pszNext;
            nRepeat = atoi(pszSrc+iSrc);
            
            // skip over repeat count.
            for( pszNext = pszSrc+iSrc; isdigit(*pszNext); pszNext++ )
                iSrc++;

            char       *pszContents = ExtractSubstring( pszNext );
            char       *pszExpandedContents = ExpandFormat( pszContents );
                
            for( int i = 0; i < nRepeat; i++ )
            {
                if( (int) (strlen(pszExpandedContents) + strlen(pszDest) + 1)
                    > nDestMax )
                {
                    nDestMax = 
                        2 * ((int)strlen(pszExpandedContents) + (int)strlen(pszDest));
                    pszDest = (char *) realloc(pszDest,nDestMax+1);
                }

                strcat( pszDest, pszExpandedContents );
                if( i < nRepeat-1 )
                    strcat( pszDest, "," );
            }

            iDst = (int)strlen(pszDest);
            
            if( pszNext[0] == '(' )
                iSrc = iSrc + (int)strlen(pszContents) + 2;
            else
                iSrc = iSrc + (int)strlen(pszContents);

            free( pszContents );
            free( pszExpandedContents );
        }
        else
        {
            if( iDst+1 >= nDestMax )
            {
                nDestMax = 2 * iDst;
                pszDest = (char *) realloc(pszDest,nDestMax);
            }

            pszDest[iDst++] = pszSrc[iSrc++];
            pszDest[iDst] = '\0';
        }
    }

    return pszDest;
}
                                 
/************************************************************************/
/*                            ApplyFormats()                            */
/*                                                                      */
/*      This method parses the format string partially, and then        */
/*      applies a subfield format string to each subfield object.       */
/*      It in turn does final parsing of the subfield formats.          */
/************************************************************************/

int ossimDDFFieldDefn::ApplyFormats()

{
    char        *pszFormatList;
    char        **papszFormatItems;
    
/* -------------------------------------------------------------------- */
/*      Verify that the format string is contained within brackets.     */
/* -------------------------------------------------------------------- */
    if( strlen(_formatControls) < 2
        || _formatControls[0] != '('
        || _formatControls[strlen(_formatControls)-1] != ')' )
    {
       ossimNotify(ossimNotifyLevel_WARN)
          << "Format controls for `%s' field missing brackets:%s\n"
          << pszTag << _formatControls << std::endl;
        
        return false;
    }

/* -------------------------------------------------------------------- */
/*      Duplicate the string, and strip off the brackets.               */
/* -------------------------------------------------------------------- */

    pszFormatList = ExpandFormat( _formatControls );

/* -------------------------------------------------------------------- */
/*      Tokenize based on commas.                                       */
/* -------------------------------------------------------------------- */
    papszFormatItems =
        ossimCSLTokenizeStringComplex(pszFormatList, ",", false, false );

    free( pszFormatList );

/* -------------------------------------------------------------------- */
/*      Apply the format items to subfields.                            */
/* -------------------------------------------------------------------- */
    int iFormatItem;
    
    for( iFormatItem = 0;
         papszFormatItems[iFormatItem] != NULL;
         iFormatItem++ )
    {
        const char      *pszPastPrefix;

        pszPastPrefix = papszFormatItems[iFormatItem];
        while( *pszPastPrefix >= '0' && *pszPastPrefix <= '9' )
            pszPastPrefix++;

        ///////////////////////////////////////////////////////////////
        // Did we get too many formats for the subfields created
        // by names?  This may be legal by the 8211 specification, but
        // isn't encountered in any formats we care about so we just
        // blow.
        
        if( iFormatItem >= nSubfieldCount )
        {
           ossimNotify(ossimNotifyLevel_WARN)
              << "Got more formats than subfields for field `%s'.\n"
              << pszTag << std::endl;
            break;
        }
        
        if( !papoSubfields[iFormatItem]->SetFormat(pszPastPrefix) )
            return false;
    }

/* -------------------------------------------------------------------- */
/*      Verify that we got enough formats, cleanup and return.          */
/* -------------------------------------------------------------------- */
    ossimCSLDestroy( papszFormatItems );

    if( iFormatItem < nSubfieldCount )
    {
       ossimNotify(ossimNotifyLevel_DEBUG)
          << "Got less formats than subfields for field `%s',\n"
          << pszTag << std::endl;
        return false;
    }

/* -------------------------------------------------------------------- */
/*      If all the fields are fixed width, then we are fixed width      */
/*      too.  This is important for repeating fields.                   */
/* -------------------------------------------------------------------- */
    nFixedWidth = 0;
    for( int i = 0; i < nSubfieldCount; i++ )
    {
        if( papoSubfields[i]->GetWidth() == 0 )
        {
            nFixedWidth = 0;
            break;
        }
        else
            nFixedWidth += papoSubfields[i]->GetWidth();
    }

    return true;
}

/************************************************************************/
/*                          FindSubfieldDefn()                          */
/************************************************************************/

/**
 * Find a subfield definition by it's mnemonic tag.  
 *
 * @param pszMnemonic The name of the field.
 *
 * @return The subfield pointer, or NULL if there isn't any such subfield.
 */
 

ossimDDFSubfieldDefn *ossimDDFFieldDefn::FindSubfieldDefn( const char * pszMnemonic )

{
    for( int i = 0; i < nSubfieldCount; i++ )
    {
       ossimString s = papoSubfields[i]->GetName();
       if( s == pszMnemonic )
            return papoSubfields[i];
    }

    return NULL;
}

/************************************************************************/
/*                            GetSubfield()                             */
/*                                                                      */
/*      Fetch a subfield by it's index.                                 */
/************************************************************************/

/**
 * Fetch a subfield by index.
 *
 * @param i The index subfield index. (Between 0 and GetSubfieldCount()-1)
 *
 * @return The subfield pointer, or NULL if the index is out of range.
 */

ossimDDFSubfieldDefn *ossimDDFFieldDefn::GetSubfield( int i )

{
    if( i < 0 || i >= nSubfieldCount )
    {
       // assert( false );
       return NULL;
    }
             
    return papoSubfields[i];
}

/************************************************************************/
/*                          GetDefaultValue()                           */
/************************************************************************/

/**
 * Return default data for field instance.
 */

char *ossimDDFFieldDefn::GetDefaultValue( int *pnSize )

{                                                                       
/* -------------------------------------------------------------------- */
/*      Loop once collecting the sum of the subfield lengths.           */
/* -------------------------------------------------------------------- */
    int iSubfield;
    int nTotalSize = 0;

    for( iSubfield = 0; iSubfield < nSubfieldCount; iSubfield++ )
    {
        int nSubfieldSize;

        if( !papoSubfields[iSubfield]->GetDefaultValue( NULL, 0, 
                                                        &nSubfieldSize ) )
            return NULL;
        nTotalSize += nSubfieldSize;
    }

/* -------------------------------------------------------------------- */
/*      Allocate buffer.                                                */
/* -------------------------------------------------------------------- */
    char *pachData = (char *) malloc( nTotalSize );

    if( pnSize != NULL )
        *pnSize = nTotalSize;

/* -------------------------------------------------------------------- */
/*      Loop again, collecting actual default values.                   */
/* -------------------------------------------------------------------- */
    int nOffset = 0;
    for( iSubfield = 0; iSubfield < nSubfieldCount; iSubfield++ )
    {
        int nSubfieldSize;

        if( !papoSubfields[iSubfield]->GetDefaultValue( 
                pachData + nOffset, nTotalSize - nOffset, &nSubfieldSize ) )
        {
           // assert( false );
           return NULL;
        }

        nOffset += nSubfieldSize;
    }

    // assert( nOffset == nTotalSize );

    return pachData;
}
