///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/ViewportManager.h>
#include <core/scene/animation/AnimManager.h>
#include <core/gui/properties/BooleanPropertyUI.h>
#include <core/gui/properties/SubObjectParameterUI.h>
#include <core/gui/mainwnd/MainFrame.h>
#include <boost/iterator/counting_iterator.hpp>

#include "CNAModifier.h"

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(CommonNeighborAnalysisModifier, AtomsObjectAnalyzerBase)
DEFINE_VECTOR_REFERENCE_FIELD(CommonNeighborAnalysisModifier, AtomType, "AtomTypes", _atomTypesList)
DEFINE_REFERENCE_FIELD(CommonNeighborAnalysisModifier, AtomTypeDataChannel, "CNATypeChannel", _cnaChannel)
SET_PROPERTY_FIELD_LABEL(CommonNeighborAnalysisModifier, _atomTypesList, "Atom Types")

/******************************************************************************
* Constructs the modifier object.
******************************************************************************/
CommonNeighborAnalysisModifier::CommonNeighborAnalysisModifier(bool isLoading)
	: AtomsObjectAnalyzerBase(isLoading)
{
	INIT_PROPERTY_FIELD(CommonNeighborAnalysisModifier, _atomTypesList);
	INIT_PROPERTY_FIELD(CommonNeighborAnalysisModifier, _cnaChannel);

	if(!isLoading) {

		// Create the internal atom types.
		intrusive_ptr<AtomType> fccType(new AtomType());
		fccType->setName(tr("FCC"));
		fccType->setColor(Color(0.4f, 1.0f, 0.4f));
		_atomTypesList.push_back(fccType);

		intrusive_ptr<AtomType> hcpType(new AtomType());
		hcpType->setName(tr("HCP"));
		hcpType->setColor(Color(1.0f, 0.4f, 0.4f));
		_atomTypesList.push_back(hcpType);

		intrusive_ptr<AtomType> bccType(new AtomType());
		bccType->setName(tr("BCC"));
		bccType->setColor(Color(0.4f, 0.4f, 1.0f));
		_atomTypesList.push_back(bccType);

		intrusive_ptr<AtomType> icosahedralType(new AtomType());
		icosahedralType->setName(tr("Icosahedral"));
		icosahedralType->setColor(Color(0.2f, 1.0f, 1.0f));
		_atomTypesList.push_back(icosahedralType);

		intrusive_ptr<AtomType> noneType(new AtomType());
		noneType->setName(tr("Other"));
		noneType->setColor(Color(1.0f, 1.0f, 1.0f));
		_atomTypesList.push_back(noneType);

		intrusive_ptr<AtomType> cgnType(new AtomType());
		cgnType->setName(tr("cg-N"));
		cgnType->setColor(Color(1.0f, 1.0f, 0.0f));
		_atomTypesList.push_back(cgnType);

		intrusive_ptr<AtomType> diamondType(new AtomType());
		diamondType->setName(tr("Diamond"));
		diamondType->setColor(Color(1.0f, 0.4f, 0.0f));
		_atomTypesList.push_back(diamondType);

		intrusive_ptr<AtomType> hexDiamondType(new AtomType());
		hexDiamondType->setName(tr("Hex-Diamond"));
		hexDiamondType->setColor(Color(0.7f, 0.0f, 1.0f));
		_atomTypesList.push_back(hexDiamondType);

		intrusive_ptr<AtomType> diamondStackingFaultType(new AtomType());
		diamondStackingFaultType->setName(tr("Diamond stacking fault"));
		diamondStackingFaultType->setColor(Color(0.2f, 1.0f, 1.0f));
		_atomTypesList.push_back(diamondStackingFaultType);

		intrusive_ptr<AtomType> bccTwinType(new AtomType());
		bccTwinType->setName(tr("BCC Twin"));
		bccTwinType->setColor(Color(1.0f, 0.4f, 0.4f));
		_atomTypesList.push_back(bccTwinType);

		// Create data channel for the computation results.
		_cnaChannel = new AtomTypeDataChannel(DataChannel::CNATypeChannel);
	}
}

/******************************************************************************
* Applies the previously calculated analysis results to the atoms object.
******************************************************************************/
EvaluationStatus CommonNeighborAnalysisModifier::applyResult(TimeTicks time, TimeInterval& validityInterval)
{
	// Check if it is still valid.
	if(input()->atomsCount() != _cnaChannel->size())
		throw Exception(tr("Number of atoms of input object has changed. Analysis results became invalid."));

	// Create a copy of the CNA atom type channel that stores the calculated atom types.
	CloneHelper cloneHelper;
	AtomTypeDataChannel::SmartPtr cnaAtomTypeChannel = cloneHelper.cloneObject(_cnaChannel, true);

	// Remove old atoms types.
	while(!cnaAtomTypeChannel->atomTypes().empty())
		cnaAtomTypeChannel->removeAtomType(cnaAtomTypeChannel->atomTypes().front());

	// Add analysis atom types.
	Vector3 atomTypeColors[NumCNAAtomTypes];
	int index = 0;
	Q_FOREACH(AtomType* atype, atomTypes()) {
		// Add only a copy of the atom type to the AtomsObject and keep the original.
		cnaAtomTypeChannel->insertAtomType(cloneHelper.cloneObject(atype, true).get());
		atype->colorController()->getValue(time, atomTypeColors[index++], validityInterval);
	}

	// Put the new channel into the output object.
	output()->insertDataChannel(cnaAtomTypeChannel);

	int typeCounters[NumCNAAtomTypes];
	for(size_t i = 0; i < NumCNAAtomTypes; i++)
		typeCounters[i] = 0;
	const int* in = _cnaChannel->constDataInt();
	Vector3* colorIter = outputStandardChannel(DataChannel::ColorChannel)->dataVector3();
	for(size_t i = _cnaChannel->size(); i != 0; --i, ++in) {
		OVITO_ASSERT(*in < NumCNAAtomTypes && *in >= 0);
		*colorIter++ = atomTypeColors[*in];
		typeCounters[*in]++;
	}


	QString statusMessage;
	statusMessage += tr("%n FCC atoms\n", 0, typeCounters[FCC]);
	statusMessage += tr("%n HCP atoms\n", 0, typeCounters[HCP]);
	statusMessage += tr("%n BCC atoms\n", 0, typeCounters[BCC]);
	statusMessage += tr("%n diamond atoms\n", 0, typeCounters[DIAMOND]);
	statusMessage += tr("%n other atoms", 0, typeCounters[OTHER]);
	return EvaluationStatus(EvaluationStatus::EVALUATION_SUCCESS, QString(), statusMessage);
}

/******************************************************************************
* This is the actual analysis method.
******************************************************************************/
EvaluationStatus CommonNeighborAnalysisModifier::doAnalysis(TimeTicks time, bool suppressDialogs)
{
	if(calculate(input(), suppressDialogs))
		return EvaluationStatus();
	else
		return EvaluationStatus(EvaluationStatus::EVALUATION_ERROR, tr("Calculation has been canceled by the user."));
}

/******************************************************************************
* Performs the analysis.
* Throws an exception on error.
* Returns false when the operation has been canceled by the user.
******************************************************************************/
bool CommonNeighborAnalysisModifier::calculate(AtomsObject* atomsObject, bool suppressDialogs)
{
	ProgressIndicator progress(tr("Performing common neighbor analysis (on %n processor(s))", NULL, QThread::idealThreadCount()), atomsObject->atomsCount(), suppressDialogs);

	// Prepare the neighbor list.
	OnTheFlyNeighborList neighborList(nearestNeighborList()->nearestNeighborCutoff());
	if(!neighborList.prepare(atomsObject, suppressDialogs)) {
		_cnaChannel->setSize(0);
		return false;
	}

	// Reserve output array memory.
	_cnaChannel->setSize(atomsObject->atomsCount());

	// Measure analysis time.
	QTime timer;
	timer.start();

	// Execute CNA analysis code for each atom in a parallel fashion.
	Kernel kernel(neighborList, _cnaChannel);
	boost::counting_iterator<int> firstAtom(0);
	boost::counting_iterator<int> lastAtom(atomsObject->atomsCount());
	QFuture<void> future = QtConcurrent::map(firstAtom, lastAtom, kernel);
	progress.waitForFuture(future);

	// Throw away results obtained so far if the user cancels the calculation.
	if(future.isCanceled()) {
		_cnaChannel->setSize(0);
		return false;
	}
	else {
		VerboseLogger() << "Common neighbor analysis took" << (timer.elapsed()/1000) << "sec." << endl;
	}

	return true;
}

#define CNA_MAX_PATTERN_NEIGHBORS 20

/// Pair of neighbor atoms that form a bond (bit-wise storage).
typedef unsigned int CNAPairBond;

/**
 * A bit-flag array indicating which pairs of neighbors are bonded
 * and which are not.
 */
struct NeighborBondArray
{
	/// Default constructor.
	NeighborBondArray() {
		memset(neighborArray, 0, sizeof(neighborArray));
	}

	/// Two-dimensional bit array that stores the bonds between neighbors.
	unsigned int neighborArray[CNA_MAX_PATTERN_NEIGHBORS];

	/// Returns whether two nearest neighbors have a bond between them.
	inline bool neighborBond(int neighborIndex1, int neighborIndex2) const {
		OVITO_ASSERT(neighborIndex1 < CNA_MAX_PATTERN_NEIGHBORS);
		OVITO_ASSERT(neighborIndex2 < CNA_MAX_PATTERN_NEIGHBORS);
		return (neighborArray[neighborIndex1] & (1<<neighborIndex2));
	}

	/// Sets whether two nearest neighbors have a bond between them.
	void setNeighborBond(int neighborIndex1, int neighborIndex2, bool bonded) {
		OVITO_ASSERT(neighborIndex1 < CNA_MAX_PATTERN_NEIGHBORS);
		OVITO_ASSERT(neighborIndex2 < CNA_MAX_PATTERN_NEIGHBORS);
		if(bonded) {
			neighborArray[neighborIndex1] |= (1<<neighborIndex2);
			neighborArray[neighborIndex2] |= (1<<neighborIndex1);
		}
		else {
			neighborArray[neighborIndex1] &= ~(1<<neighborIndex2);
			neighborArray[neighborIndex2] &= ~(1<<neighborIndex1);
		}
	}
};

/******************************************************************************
* Find all atoms that are nearest neighbors of the given pair of atoms.
******************************************************************************/
static int findCommonNeighbors(const NeighborBondArray& neighborArray, int neighborIndex, unsigned int& commonNeighbors, int numNeighbors)
{
	commonNeighbors = neighborArray.neighborArray[neighborIndex];
#ifndef Q_CC_MSVC
	// Count the number of bits set in neighbor bit field.
	return __builtin_popcount(commonNeighbors);
#else
	// Count the number of bits set in neighbor bit field.
	unsigned int v = commonNeighbors - ((commonNeighbors >> 1) & 0x55555555);
	v = (v & 0x33333333) + ((v >> 2) & 0x33333333);
	return ((v + (v >> 4) & 0xF0F0F0F) * 0x1010101) >> 24;
#endif
}

/******************************************************************************
* Finds all bonds between common nearest neighbors.
******************************************************************************/
static int findNeighborBonds(const NeighborBondArray& neighborArray, unsigned int commonNeighbors, int numNeighbors, CNAPairBond* neighborBonds)
{
	int numBonds = 0;

	unsigned int nib[CNA_MAX_PATTERN_NEIGHBORS];
	int nibn = 0;
	unsigned int ni1b = 1;
	for(int ni1 = 0; ni1 < numNeighbors; ni1++, ni1b <<= 1) {
		if(commonNeighbors & ni1b) {
			unsigned int b = commonNeighbors & neighborArray.neighborArray[ni1];
			for(int n = 0; n < nibn; n++) {
				if(b & nib[n]) {
					OVITO_ASSERT(numBonds < CNA_MAX_PATTERN_NEIGHBORS*CNA_MAX_PATTERN_NEIGHBORS);
					neighborBonds[numBonds++] = ni1b | nib[n];
				}
			}

			nib[nibn++] = ni1b;
		}
	}
	return numBonds;
}

/******************************************************************************
* Find all chains of bonds.
******************************************************************************/
static int getAdjacentBonds(unsigned int atom, CNAPairBond* bondsToProcess, int& numBonds, unsigned int& atomsToProcess, unsigned int& atomsProcessed)
{
    int adjacentBonds = 0;
	for(int b = numBonds - 1; b >= 0; b--) {
		if(atom & *bondsToProcess) {
            ++adjacentBonds;
   			atomsToProcess |= *bondsToProcess & (~atomsProcessed);
   			memmove(bondsToProcess, bondsToProcess + 1, sizeof(CNAPairBond) * b);
   			numBonds--;
		}
		else ++bondsToProcess;
	}
	return adjacentBonds;
}

/******************************************************************************
* Find all chains of bonds between common neighbors and determine the length
* of the longest continuous chain.
******************************************************************************/
static int calcMaxChainLength(CNAPairBond* neighborBonds, int numBonds)
{
    // Group the common bonds into clusters.
	int maxChainLength = 0;
	while(numBonds) {
        // Make a new cluster starting with the first remaining bond to be processed.
		numBonds--;
        unsigned int atomsToProcess = neighborBonds[numBonds];
        unsigned int atomsProcessed = 0;
		int clusterSize = 1;
        do {
#ifndef Q_CC_MSVC
        	// Determine the number of trailing 0-bits in atomsToProcess, starting at the least significant bit position.
			int nextAtomIndex = __builtin_ctz(atomsToProcess);
#else
			unsigned long nextAtomIndex;
			_BitScanForward(&nextAtomIndex, atomsToProcess);
			OVITO_ASSERT(nextAtomIndex >= 0 && nextAtomIndex < 32);
#endif
			unsigned int nextAtom = 1 << nextAtomIndex;
        	atomsProcessed |= nextAtom;
			atomsToProcess &= ~nextAtom;
			clusterSize += getAdjacentBonds(nextAtom, neighborBonds, numBonds, atomsToProcess, atomsProcessed);
		}
        while(atomsToProcess);
        if(clusterSize > maxChainLength)
        	maxChainLength = clusterSize;
	}
	return maxChainLength;
}

/******************************************************************************
* Determines the crystal structure type of a single atom.
******************************************************************************/
void CommonNeighborAnalysisModifier::Kernel::operator()(int atomIndex)
{
	int& result = atomTypeIndices.dataInt()[atomIndex];
	result = OTHER; // Start with assuming that the CNA type is unknown.

	// Store neighbor vectors in a local array.
	int numNeighbors = 0;
	Vector3 neighborVectors[CNA_MAX_PATTERN_NEIGHBORS];
	for(OnTheFlyNeighborList::iterator neighborIter(nnlist, atomIndex); !neighborIter.atEnd(); neighborIter.next()) {
		if(numNeighbors == CNA_MAX_PATTERN_NEIGHBORS) return;
		neighborVectors[numNeighbors] = neighborIter.delta();
		numNeighbors++;
	}

	if(numNeighbors == 12) { // Detect FCC and HCP atoms each having 12 NN.

		// Compute bond bit-flag array.
		NeighborBondArray neighborArray;
		for(int ni1 = 0; ni1 < 12; ni1++) {
			neighborArray.setNeighborBond(ni1, ni1, false);
			for(int ni2 = ni1+1; ni2 < 12; ni2++)
				neighborArray.setNeighborBond(ni1, ni2, LengthSquared(neighborVectors[ni1] - neighborVectors[ni2]) <= nnlist.cutoffRadiusSquared());
		}

		int n421 = 0;
		int n422 = 0;
		int n555 = 0;
		for(int ni = 0; ni < 12; ni++) {

			// Determine number of neighbors the two atoms have in common.
			unsigned int commonNeighbors;
			int numCommonNeighbors = findCommonNeighbors(neighborArray, ni, commonNeighbors, 12);
			if(numCommonNeighbors != 4 && numCommonNeighbors != 5)
				return;

			// Determine the number of bonds among the common neighbors.
			CNAPairBond neighborBonds[CNA_MAX_PATTERN_NEIGHBORS*CNA_MAX_PATTERN_NEIGHBORS];
			int numNeighborBonds = findNeighborBonds(neighborArray, commonNeighbors, 12, neighborBonds);
			if(numNeighborBonds != 2 && numNeighborBonds != 5)
				break;

			// Determine the number of bonds in the longest continuous chain.
			int maxChainLength = calcMaxChainLength(neighborBonds, numNeighborBonds);
			if(numCommonNeighbors == 4 && numNeighborBonds == 2) {
				if(maxChainLength == 1) n421++;
				else if(maxChainLength == 2) n422++;
				else return;
			}
			else if(numCommonNeighbors == 5 && numNeighborBonds == 5 && maxChainLength == 5) n555++;
			else return;
		}
		if(n421 == 12) result = FCC;
		else if(n421 == 6 && n422 == 6) result = HCP;
		else if(n555 == 12) result = ICOSAHEDRAL;
	}
	else if(numNeighbors == 14) { // Detect BCC atoms having 14 NN (in 1st and 2nd shell).

		// Compute bond bit-flag array.
		NeighborBondArray neighborArray;
		for(int ni1 = 0; ni1 < 14; ni1++) {
			neighborArray.setNeighborBond(ni1, ni1, false);
			for(int ni2 = ni1+1; ni2 < 14; ni2++)
				neighborArray.setNeighborBond(ni1, ni2, LengthSquared(neighborVectors[ni1] - neighborVectors[ni2]) <= nnlist.cutoffRadiusSquared());
		}

		int n444 = 0;
		int n555 = 0;
		int n666 = 0;
		for(int ni = 0; ni < 14; ni++) {

			// Determine number of neighbors the two atoms have in common.
			unsigned int commonNeighbors;
			int numCommonNeighbors = findCommonNeighbors(neighborArray, ni, commonNeighbors, 14);
			if(numCommonNeighbors != 4 && numCommonNeighbors != 5 && numCommonNeighbors != 6)
				return;

			// Determine the number of bonds among the common neighbors.
			CNAPairBond neighborBonds[CNA_MAX_PATTERN_NEIGHBORS*CNA_MAX_PATTERN_NEIGHBORS];
			int numNeighborBonds = findNeighborBonds(neighborArray, commonNeighbors, 14, neighborBonds);
			if(numNeighborBonds != 4 && numNeighborBonds != 5 && numNeighborBonds != 6)
				break;

			// Determine the number of bonds in the longest continuous chain.
			int maxChainLength = calcMaxChainLength(neighborBonds, numNeighborBonds);
			if(numCommonNeighbors == 4 && numNeighborBonds == 4 && maxChainLength == 4) n444++;
			else if(numCommonNeighbors == 6 && numNeighborBonds == 6 && maxChainLength == 6) n666++;
			else if(numCommonNeighbors == 5 && numNeighborBonds == 5 && maxChainLength == 5) n555++;
			else return;
		}
		if(n666 == 8 && n444 == 6) result = BCC;
		else if(n666 == 6 && n444 == 4 && n555 == 4) result = BCC_TWIN;
	}
	else if(numNeighbors == 9) { // Detect cg-N atoms having 9 NN.

		// Compute bond bit-flag array.
		NeighborBondArray neighborArray;
		for(int ni1 = 0; ni1 < 9; ni1++) {
			neighborArray.setNeighborBond(ni1, ni1, false);
			for(int ni2 = ni1+1; ni2 < 9; ni2++)
				neighborArray.setNeighborBond(ni1, ni2, LengthSquared(neighborVectors[ni1] - neighborVectors[ni2]) <= nnlist.cutoffRadiusSquared());
		}

		int n211 = 0;
		int n421 = 0;
		for(int ni = 0; ni < 9; ni++) {

			// Determine number of neighbors the two atoms have in common.
			unsigned int commonNeighbors;
			int numCommonNeighbors = findCommonNeighbors(neighborArray, ni, commonNeighbors, 9);
			if(numCommonNeighbors != 2 && numCommonNeighbors != 4)
				return;

			// Determine the number of bonds among the common neighbors.
			CNAPairBond neighborBonds[CNA_MAX_PATTERN_NEIGHBORS*CNA_MAX_PATTERN_NEIGHBORS];
			int numNeighborBonds = findNeighborBonds(neighborArray, commonNeighbors, 9, neighborBonds);
			if(numNeighborBonds != 1 && numNeighborBonds != 2)
				break;

			// Determine the number of bonds in the longest continuous chain.
			int maxChainLength = calcMaxChainLength(neighborBonds, numNeighborBonds);
			if(numCommonNeighbors == 2 && numNeighborBonds == 1 && maxChainLength == 1) n211++;
			else if(numCommonNeighbors == 4 && numNeighborBonds == 2 && maxChainLength == 1) n421++;
			else return;
		}
		if(n211 == 6 && n421 == 3) result = CG_N;
	}
	else if(numNeighbors == 16) { // Detect diamond atoms having 16 NN.

		// Compute bond bit-flag array.
		NeighborBondArray neighborArray;
		for(int ni1 = 0; ni1 < 16; ni1++) {
			neighborArray.setNeighborBond(ni1, ni1, false);
			for(int ni2 = ni1+1; ni2 < 16; ni2++)
				neighborArray.setNeighborBond(ni1, ni2, LengthSquared(neighborVectors[ni1] - neighborVectors[ni2]) <= nnlist.cutoffRadiusSquared());
		}
		int n543 = 0;
		int n555 = 0;
		int n663 = 0;
		int n677 = 0;
		int n699 = 0;
		int n799 = 0;
		for(int ni = 0; ni < 16; ni++) {

			// Determine number of neighbors the two atoms have in common.
			unsigned int commonNeighbors;
			int numCommonNeighbors = findCommonNeighbors(neighborArray, ni, commonNeighbors, 16);
			if(numCommonNeighbors < 5 || numCommonNeighbors > 7) return;

			// Determine the number of bonds among the common neighbors.
			CNAPairBond neighborBonds[CNA_MAX_PATTERN_NEIGHBORS*CNA_MAX_PATTERN_NEIGHBORS];
			int numNeighborBonds = findNeighborBonds(neighborArray, commonNeighbors, 16, neighborBonds);
			if(numNeighborBonds < 4 || numNeighborBonds > 9) break;

			// Determine the number of bonds in the longest continuous chain.
			int maxChainLength = calcMaxChainLength(neighborBonds, numNeighborBonds);
			if(numCommonNeighbors == 5 && numNeighborBonds == 4 && maxChainLength == 3) n543++;
			else if(numCommonNeighbors == 6 && numNeighborBonds == 6 && maxChainLength == 3) n663++;
			else if(numCommonNeighbors == 6 && numNeighborBonds == 7 && maxChainLength == 7) n677++;
			else if(numCommonNeighbors == 5 && numNeighborBonds == 5 && maxChainLength == 5) n555++;
			else if(numCommonNeighbors == 6 && numNeighborBonds == 9 && maxChainLength == 9) n699++;
			else if(numCommonNeighbors == 7 && numNeighborBonds == 9 && maxChainLength == 9) n799++;
			else return;
		}
		if(n543 == 12 && n663 == 4) result = DIAMOND;
		else if(n555 == 6 && n677 == 3 && n699 == 1 && n543 == 3 && n799 == 3) result = DIAMOND_STACKINGFAULT;
	}
	else if(numNeighbors == 17) { // Detect hexagonal diamond atoms having 17 NN.
		// Compute bond bit-flag array.
		NeighborBondArray neighborArray;
		for(int ni1 = 0; ni1 < 17; ni1++) {
			neighborArray.setNeighborBond(ni1, ni1, false);
			for(int ni2 = ni1+1; ni2 < 17; ni2++)
				neighborArray.setNeighborBond(ni1, ni2, LengthSquared(neighborVectors[ni1] - neighborVectors[ni2]) <= nnlist.cutoffRadiusSquared());
		}
		int n677 = 0;
		int n555 = 0;
		int n81212 = 0;
		int n699 = 0;
		int n543 = 0;
		int n799 = 0;
		int n663 = 0;
		for(int ni = 0; ni < 17; ni++) {

			// Determine number of neighbors the two atoms have in common.
			unsigned int commonNeighbors;
			int numCommonNeighbors = findCommonNeighbors(neighborArray, ni, commonNeighbors, 17);
			if(numCommonNeighbors < 5 || numCommonNeighbors > 8) return;

			// Determine the number of bonds among the common neighbors.
			CNAPairBond neighborBonds[CNA_MAX_PATTERN_NEIGHBORS*CNA_MAX_PATTERN_NEIGHBORS];
			int numNeighborBonds = findNeighborBonds(neighborArray, commonNeighbors, 17, neighborBonds);
			if(numNeighborBonds < 4 || numNeighborBonds > 12) break;

			// Determine the number of bonds in the longest continuous chain.
			int maxChainLength = calcMaxChainLength(neighborBonds, numNeighborBonds);
			if(numCommonNeighbors == 5 && numNeighborBonds == 4 && maxChainLength == 3) n543++;
			else if(numCommonNeighbors == 6 && numNeighborBonds == 7 && maxChainLength == 7) n677++;
			else if(numCommonNeighbors == 6 && numNeighborBonds == 6 && maxChainLength == 3) n663++;
			else if(numCommonNeighbors == 5 && numNeighborBonds == 5 && maxChainLength == 5) n555++;
			else if(numCommonNeighbors == 8 && numNeighborBonds == 12 && maxChainLength == 12) n81212++;
			else if(numCommonNeighbors == 6 && numNeighborBonds == 9 && maxChainLength == 9) n699++;
			else if(numCommonNeighbors == 7 && numNeighborBonds == 9 && maxChainLength == 9) n799++;
			else return;
		}
		if(n677 == 6 && n555 == 6 && n81212 == 3 && n699 == 2) result = HEX_DIAMOND;
		else if(n699 == 1 && n677 == 3 && n543 == 9 && n799 == 3 && n663 == 1) result = DIAMOND_STACKINGFAULT;
	}
}

IMPLEMENT_PLUGIN_CLASS(CommonNeighborAnalysisModifierEditor, AtomsObjectModifierEditorBase)

/******************************************************************************
* Sets up the UI widgets of the editor.
******************************************************************************/
void CommonNeighborAnalysisModifierEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create a rollout.
	QWidget* rollout = createRollout(tr("Common neighbor analysis"), rolloutParams, "atomviz.modifiers.common_neighbor_analysis");

    // Create the rollout contents.
	QVBoxLayout* layout1 = new QVBoxLayout(rollout);
	layout1->setContentsMargins(4,4,4,4);
#ifndef Q_WS_MAC
	layout1->setSpacing(0);
#endif

	BooleanPropertyUI* autoUpdateUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(AtomsObjectAnalyzerBase, _autoUpdateOnTimeChange));
	layout1->addWidget(autoUpdateUI->checkBox());

	BooleanPropertyUI* saveResultsUI = new BooleanPropertyUI(this, "storeResultsWithScene", tr("Save results in scene file"));
	layout1->addWidget(saveResultsUI->checkBox());

	QPushButton* recalcButton = new QPushButton(tr("Calculate"), rollout);
	layout1->addSpacing(6);
	layout1->addWidget(recalcButton);
	connect(recalcButton, SIGNAL(clicked(bool)), this, SLOT(onRecalculate()));

	// Status label.
	layout1->addSpacing(10);
	layout1->addWidget(statusLabel());

	// Derive a custom class from the list parameter UI to
	// give the items a color.
	class CustomRefTargetListParameterUI : public RefTargetListParameterUI {
	public:
		CustomRefTargetListParameterUI(PropertiesEditor* parentEditor, const PropertyFieldDescriptor& refField)
			: RefTargetListParameterUI(parentEditor, refField, RolloutInsertionParameters(), NULL, 260) {}
	protected:
		virtual QVariant getItemData(RefTarget* target, const QModelIndex& index, int role) {
			if(role == Qt::DecorationRole && target != NULL) {
				return (QColor)static_object_cast<AtomType>(target)->color();
			}
			else return RefTargetListParameterUI::getItemData(target, index, role);
		}
		/// Do not open sub-editor for selected atom type.
		virtual void openSubEditor() {}
	};

	atomsTypesPUI = new CustomRefTargetListParameterUI(this, PROPERTY_FIELD_DESCRIPTOR(CommonNeighborAnalysisModifier, _atomTypesList));
	layout1->addSpacing(10);
	layout1->addWidget(new QLabel(tr("Type colors:")));
	layout1->addWidget(atomsTypesPUI->listWidget());
	connect(atomsTypesPUI->listWidget(), SIGNAL(doubleClicked(const QModelIndex&)), this, SLOT(onDoubleClickAtomType(const QModelIndex&)));

	// Open a sub-editor for the NearestNeighborList sub-object.
	SubObjectParameterUI* subEditorUI = new SubObjectParameterUI(this, PROPERTY_FIELD_DESCRIPTOR(AtomsObjectAnalyzerBase, _nearestNeighborList), rolloutParams.before(rollout));
}

/******************************************************************************
* Is called when the user presses the Recalculate button.
******************************************************************************/
void CommonNeighborAnalysisModifierEditor::onRecalculate()
{
	if(!editObject()) return;
	CommonNeighborAnalysisModifier* modifier = static_object_cast<CommonNeighborAnalysisModifier>(editObject());
	try {
		modifier->performAnalysis(ANIM_MANAGER.time());
	}
	catch(Exception& ex) {
		ex.prependGeneralMessage(tr("Failed to perform common neighbor analysis (CNA)."));
		ex.showError();
	}
}

/******************************************************************************
* Is called when the user has double-clicked on one of the CNA
* atom types in the list widget.
******************************************************************************/
void CommonNeighborAnalysisModifierEditor::onDoubleClickAtomType(const QModelIndex& index)
{
	// Let the user select a color for the atom type.
	AtomType* atype = static_object_cast<AtomType>(atomsTypesPUI->selectedObject());
	if(!atype || !atype->colorController()) return;

	QColor oldColor = Color(atype->colorController()->getCurrentValue());
	QColor newColor = QColorDialog::getColor(oldColor, container());
	if(!newColor.isValid() || newColor == oldColor) return;

	UNDO_MANAGER.beginCompoundOperation(tr("Change color"));
	atype->colorController()->setCurrentValue(Color(newColor));
	UNDO_MANAGER.endCompoundOperation();
}

};	// End of namespace AtomViz
