///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/mainwnd/MainFrame.h>
#include <core/gui/ApplicationManager.h>
#include <core/actions/ActionManager.h>
#include <core/scene/animation/AnimManager.h>
#include <core/gui/panels/animation/AnimationSettingsDialog.h>
#include <core/viewport/ViewportManager.h>
#include "StdActions.h"

namespace Core {

/// Constructs the action objects and registers them with the ActionManager.
AnimationActionsHandler::AnimationActionsHandler()
{
	animationPlaybackMode = new AnimationPlaybackViewportMode();

	connect(addCommandAction(ACTION_GOTO_START_OF_ANIMATION, tr("Goto Start of Animation"), ":/core/animation/goto_animation_start.png"), SIGNAL(triggered(bool)), this, SLOT(onGotoStartOfAnimation()));
	connect(addCommandAction(ACTION_GOTO_END_OF_ANIMATION, tr("Goto End of Animation"), ":/core/animation/goto_animation_end.png"), SIGNAL(triggered(bool)), this, SLOT(onGotoEndOfAnimation()));
	connect(addCommandAction(ACTION_GOTO_PREVIOUS_FRAME, tr("Goto Previous Frame"), ":/core/animation/goto_previous_frame.png"), SIGNAL(triggered(bool)), this, SLOT(onGotoPreviousFrame()));
	connect(addCommandAction(ACTION_GOTO_NEXT_FRAME, tr("Goto Next Frame"), ":/core/animation/goto_next_frame.png"), SIGNAL(triggered(bool)), this, SLOT(onGotoNextFrame()));
	connect(addCommandAction(ACTION_START_ANIMATION_PLAYBACK, tr("Start Animation Playback"), ":/core/animation/play_animation.png"), SIGNAL(triggered(bool)), this, SLOT(onStartPlayback()));
	connect(addCommandAction(ACTION_STOP_ANIMATION_PLAYBACK, tr("Stop Animation Playback"), ":/core/animation/stop_animation.png"), SIGNAL(triggered(bool)), this, SLOT(onStopPlayback()));
	connect(addCommandAction(ACTION_ANIMATION_SETTINGS, tr("Animation Settings"), ":/core/animation/animation_settings.png"), SIGNAL(triggered(bool)), this, SLOT(onAnimationSettings()));
	ACTION_MANAGER.addAction(new ViewportModeAction(ACTION_TOGGLE_ANIMATION_PLAYBACK, animationPlaybackMode), tr("Play Animation"), ":/core/animation/play_animation.png");

	ActionProxy* animModeAction = addCommandAction(ACTION_ANIMATION_MODE_TOGGLE, tr("Animation Mode"), ":/core/animation/animation_mode.png");
	animModeAction->setCheckable(true);
	animModeAction->setChecked(ANIM_MANAGER.animationMode());
	connect(animModeAction, SIGNAL(toggled(bool)), &ANIM_MANAGER, SLOT(setAnimationMode(bool)));
	connect(&ANIM_MANAGER, SIGNAL(animationModeChanged(bool)), animModeAction, SLOT(setChecked(bool)));
}

/// Creates and registers a single command action.
ActionProxy* AnimationActionsHandler::addCommandAction(const QString& id, const QString& title, const char* iconPath)
{
	ActionProxy* proxy = ACTION_MANAGER.addAction(new Action(id), title, iconPath);
	return proxy;
}

/// Handles ACTION_GOTO_START_OF_ANIMATION command.
void AnimationActionsHandler::onGotoStartOfAnimation()
{
	ANIM_MANAGER.setTime(ANIM_MANAGER.animationInterval().start());
}

/// Handles ACTION_GOTO_END_OF_ANIMATION command.
void AnimationActionsHandler::onGotoEndOfAnimation()
{
	ANIM_MANAGER.setTime(ANIM_MANAGER.animationInterval().end());
}

/// Handles ACTION_GOTO_PREVIOUS_FRAME command.
void AnimationActionsHandler::onGotoPreviousFrame()
{
	// Subtract one frame from current time.
	TimeTicks newTime = ANIM_MANAGER.frameToTime(ANIM_MANAGER.timeToFrame(ANIM_MANAGER.time()) - 1);
	// Clamp new time
	newTime = max(newTime, ANIM_MANAGER.animationInterval().start());
	// Set new time.
	ANIM_MANAGER.setTime(newTime);
}

/// Handles ACTION_GOTO_NEXT_FRAME command.
void AnimationActionsHandler::onGotoNextFrame()
{
	// Add one frame to current time.
	TimeTicks newTime = ANIM_MANAGER.frameToTime(ANIM_MANAGER.timeToFrame(ANIM_MANAGER.time()) + 1);
	// Clamp new time
	newTime = min(newTime, ANIM_MANAGER.animationInterval().end());
	// Set new time.
	ANIM_MANAGER.setTime(newTime);
}

/// Handles ACTION_START_ANIMATION_PLAYBACK command.
void AnimationActionsHandler::onStartPlayback()
{
	VIEWPORT_INPUT_MANAGER.pushInputHandler(animationPlaybackMode);
}

/// Handles ACTION_STOP_ANIMATION_PLAYBACK command.
void AnimationActionsHandler::onStopPlayback()
{
	VIEWPORT_INPUT_MANAGER.removeInputHandler(animationPlaybackMode.get());
}

/// Handles ACTION_ANIMATION_SETTINGS command.
void AnimationActionsHandler::onAnimationSettings()
{
	if(APPLICATION_MANAGER.guiMode())
		AnimationSettingsDialog(MAIN_FRAME).exec();
}

/// Is called periodically by the timer.
void AnimationPlaybackViewportMode::onTimer()
{
	if(VIEWPORT_INPUT_MANAGER.currentHandler() != this)
		return;

	// Add one frame to current time
	TimeTicks newTime = ANIM_MANAGER.frameToTime(ANIM_MANAGER.timeToFrame(ANIM_MANAGER.time()) + 1);
	// Loop
	if(newTime > ANIM_MANAGER.animationInterval().end())
		newTime = ANIM_MANAGER.animationInterval().start();
	// Set new time
	ANIM_MANAGER.setTime(newTime);
	// Redraw viewports
	VIEWPORT_MANAGER.processViewportUpdates();

	// Process user input events.
	QCoreApplication::processEvents();

	if(VIEWPORT_INPUT_MANAGER.currentHandler() == this) {
		int timerSpeed = 1000;
		if(ANIM_MANAGER.playbackSpeed() > 1) timerSpeed /= ANIM_MANAGER.playbackSpeed();
		else if(ANIM_MANAGER.playbackSpeed() < -1) timerSpeed *= -ANIM_MANAGER.playbackSpeed();
		QTimer::singleShot(timerSpeed / ANIM_MANAGER.framesPerSecond(), this, SLOT(onTimer()));
	}
}

};
