///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/actions/ActionManager.h>
#include <core/gui/mainwnd/MainFrame.h>
#include <core/viewport/Window3DContainer.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/ViewportManager.h>
#include <core/utilities/ProgressIndicator.h>
#include <core/data/DataSetManager.h>
#include <core/rendering/RenderSettings.h>
#include <core/rendering/PluginRenderer.h>
#include <core/rendering/FrameBuffer.h>
#include <core/rendering/FrameBufferWindow.h>
#include "StdActions.h"

namespace Core {

/// Constructs the action objects and registers them with the ActionManager.
RenderingActionsHandler::RenderingActionsHandler()
{
	connect(addCommandAction(ACTION_RENDER_ACTIVE_VIEWPORT, tr("Render Active Viewport"), ":/core/rendering/render_active_viewport.png"), SIGNAL(triggered(bool)), this, SLOT(onRenderActiveViewport()));
	connect(addCommandAction(ACTION_SELECT_RENDERER_DIALOG, tr("Select Renderer")), SIGNAL(triggered(bool)), this, SLOT(onSelectRendererDialog()));
	connect(addCommandAction(ACTION_SHOW_FRAME_BUFFER, tr("Show render buffer")), SIGNAL(triggered(bool)), this, SLOT(onShowFrameBuffer()));
}

/// Creates and registers a single command action.
ActionProxy* RenderingActionsHandler::addCommandAction(const QString& id, const QString& title, const char* iconPath, const QKeySequence& shortcut)
{
	ActionProxy* proxy = ACTION_MANAGER.addAction(new Action(id), title, iconPath);
	if(!shortcut.isEmpty()) proxy->setShortcut(shortcut);
	return proxy;
}

/// Handles ACTION_RENDER_ACTIVE_VIEWPORT command.
void RenderingActionsHandler::onRenderActiveViewport()
{
	try {
		// Get the renderer instance.
		RenderSettings* settings = DATASET_MANAGER.currentSet()->renderSettings();
		if(!settings || !settings->renderer()) throw Exception(tr("No renderer has been selected."));
		PluginRenderer* renderer = settings->renderer();

		// Open progress dialog.
		ProgressIndicator progress(tr("Rendering ..."), 1);
		progress.isCanceled();

		// Allocate the frame buffer.
		boost::shared_ptr<FrameBuffer> frameBuffer(new FrameBuffer(settings->outputImageWidth(), settings->outputImageHeight()));

		try {
			// Initialize the renderer.
			if(renderer->startRender(DATASET_MANAGER.currentSet())) {

				if(settings->renderingRangeType() == RenderSettings::CURRENT_FRAME) {
					// Render a single frame.
					TimeTicks renderTime = ANIM_MANAGER.time();
					int frameNumber = ANIM_MANAGER.timeToFrame(renderTime);
					if(renderFrame(renderTime, frameNumber, settings, renderer, frameBuffer.get(), progress)) {
						// Open a display window for the rendered frame.
						if(APPLICATION_MANAGER.guiMode()) {
							FrameBufferWindow* display = MAIN_FRAME->frameBufferWindow();
							display->setFrameBuffer(frameBuffer);
							display->setWindowTitle(tr("Frame %1").arg(frameNumber));
							display->show();
							display->activateWindow();
							display->updateFrame();
						}
					}
				}
				else if(settings->renderingRangeType() == RenderSettings::ANIMATION_INTERVAL || settings->renderingRangeType() == RenderSettings::CUSTOM_INTERVAL) {
					// Render an animation interval.
					TimeTicks renderTime;
					int firstFrameNumber, numberOfFrames;
					if(settings->renderingRangeType() == RenderSettings::ANIMATION_INTERVAL) {
						renderTime = ANIM_MANAGER.animationInterval().start();
						firstFrameNumber = ANIM_MANAGER.timeToFrame(ANIM_MANAGER.animationInterval().start());
						numberOfFrames = (ANIM_MANAGER.timeToFrame(ANIM_MANAGER.animationInterval().end()) - firstFrameNumber + 1);
					}
					else {
						firstFrameNumber = settings->customRangeStart();
						renderTime = ANIM_MANAGER.frameToTime(firstFrameNumber);
						numberOfFrames = (settings->customRangeEnd() - firstFrameNumber + 1);
					}
					numberOfFrames = (numberOfFrames + settings->everyNthFrame()-1) / settings->everyNthFrame();
					if(numberOfFrames < 1)
						throw Exception(tr("Invalid rendering range: Frame %1 to %2").arg(settings->customRangeStart()).arg(settings->customRangeEnd()));
					progress.setMaximum(numberOfFrames);

					// Render frames, one by one.
					for(int frameIndex = 0; frameIndex < numberOfFrames; frameIndex++) {
						if(progress.isCanceled()) break;
						progress.setLabelText(tr("Rendering frame %1").arg(ANIM_MANAGER.timeToFrame(renderTime)));
						progress.setValue(frameIndex);
						int frameNumber = firstFrameNumber + frameIndex * settings->everyNthFrame() + settings->fileNumberBase();
						if(renderFrame(renderTime, frameNumber, settings, renderer, frameBuffer.get(), progress)) {
							// Open a display window for the rendered frame.
							if(APPLICATION_MANAGER.guiMode()) {
								FrameBufferWindow* display = MAIN_FRAME->frameBufferWindow();
								display->setWindowTitle(tr("Frame %1").arg(ANIM_MANAGER.timeToFrame(renderTime)));
								if(frameIndex == 0) {
									display->setFrameBuffer(frameBuffer);
									display->show();
								}
								display->updateFrame();
							}
						}

						// Goto next animation frame.
						renderTime += ANIM_MANAGER.ticksPerFrame() * settings->everyNthFrame();
					}
				}
			}
		}
		catch(...) {
			renderer->endRender();
			throw;
		}
		// Release renderer.
		renderer->endRender();
	}
	catch(const Exception& ex) {
		ex.showError();
	}

	// Redraw viewports.
	VIEWPORT_MANAGER.updateViewports();
}

/// Renders a single frame and saves the output file.
bool RenderingActionsHandler::renderFrame(TimeTicks renderTime, int frameNumber, RenderSettings* settings, PluginRenderer* renderer, FrameBuffer* frameBuffer, ProgressIndicator& progress)
{
	// Generate output filename.
	QString imageFilename;
	if(settings->saveToFile()) {
		imageFilename = settings->imageFilename();
		if(imageFilename.isEmpty())
			throw Exception(tr("Cannot save rendered image to file. No output filename specified."));

		if(settings->renderingRangeType() != RenderSettings::CURRENT_FRAME) {
			// Append frame number to file name if rendering an animation.
			QFileInfo fileInfo(imageFilename);
			imageFilename = fileInfo.path() + QChar('/') + fileInfo.baseName() + QString("%1.").arg(frameNumber, 4, 10, QChar('0')) + fileInfo.completeSuffix();

			// Check for existing image file and skip.
			if(settings->skipExistingImages() && QFileInfo(imageFilename).isFile())
				return false;
		}
	}

	// Do not update the viewports while rendering.
	ViewportSuspender noVPUpdates;

	// Setup camera view.
	ViewportRecord::SmartPtr viewportSettings;
	if(APPLICATION_MANAGER.guiMode()) {
		Viewport* vp = VIEWPORT_MANAGER.activeViewport();
		if(vp) viewportSettings = vp->settings();
	}
	else {
		viewportSettings = DATASET_MANAGER.currentSet()->viewportConfig()->activeViewportSettings();
	}
	if(!viewportSettings)
		throw Exception(tr("There is no active viewport."));

	Box3 sceneBoundingBox;
    SceneRenderer* viewportRenderer = SceneRenderer::activeRenderer();
	if(viewportRenderer)
		sceneBoundingBox = viewportRenderer->sceneExtents(viewportSettings.get(), renderTime, SceneRenderer::RENDERABLE_OBJECTS);
	CameraViewDescription view = viewportSettings->getViewDescription(renderTime, settings->outputImageAspectRatio(), sceneBoundingBox);

	// Render one frame.
	frameBuffer->clear();
	if(!renderer->renderFrame(renderTime, view, frameBuffer)) {
		progress.setCanceled(true);
		return false;
	}

	// Save rendered image to disk.
	if(settings->saveToFile()) {
		if(!frameBuffer->image().save(imageFilename, settings->imageInfo().format()))
			throw Exception(tr("Failed to save rendered image to image file '%1'.").arg(imageFilename));
	}

	return true;
}


/// Handles ACTION_SELECT_RENDERER_DIALOG command.
void RenderingActionsHandler::onSelectRendererDialog()
{
	if(!DATASET_MANAGER.currentSet()) return;
	RenderSettings* renderSettings = DATASET_MANAGER.currentSet()->renderSettings();
	if(!renderSettings) return;

	QList<PluginClassDescriptor*> rendererClasses = PluginRenderer::availableRendererClasses();
	QStringList itemList;
	Q_FOREACH(PluginClassDescriptor* clazz, rendererClasses)
		itemList << clazz->schematicTitle();

	int currentIndex = 0;
	if(renderSettings->rendererClass())
		currentIndex = itemList.indexOf(renderSettings->rendererClass()->schematicTitle());

	bool ok;
	QString selectedClass = QInputDialog::getItem(NULL, tr("Choose renderer"), tr("Select the renderer implementation:"), itemList, currentIndex, false, &ok);
	if(!ok) return;

	UNDO_MANAGER.beginCompoundOperation(tr("Change renderer"));
	try {
		int newIndex = itemList.indexOf(selectedClass);
		if(newIndex != currentIndex)
			renderSettings->setRendererClass(rendererClasses[newIndex]);
	}
	catch(const Exception& ex) {
		ex.showError();
	}
	UNDO_MANAGER.endCompoundOperation();
}

/// Handles ACTION_SHOW_FRAME_BUFFER command.
void RenderingActionsHandler::onShowFrameBuffer()
{
	if(APPLICATION_MANAGER.guiMode()) {
		FrameBufferWindow* display = MAIN_FRAME->frameBufferWindow();
		display->show();
		display->adjustSize();
		display->activateWindow();
	}
}

};
