///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/properties/FloatControllerUI.h>
#include <core/scene/animation/controller/Controller.h>
#include <core/scene/animation/AnimManager.h>
#include <core/data/units/ParameterUnit.h>

namespace Core {

// Gives the class run-time type information.
IMPLEMENT_ABSTRACT_PLUGIN_CLASS(FloatControllerUI, NumericalControllerUI)

/******************************************************************************
* The constructor.
******************************************************************************/
FloatControllerUI::FloatControllerUI(PropertiesEditor* parentEditor, const PropertyFieldDescriptor& refField) : 
	NumericalControllerUI(parentEditor, refField, UNITS_MANAGER.floatIdentity())
{
}

/******************************************************************************
* Takes the value entered by the user and stores it in the parameter object
* this parameter UI is bound to.
******************************************************************************/
void FloatControllerUI::updateParameterValue()
{
	FloatController* ctrl = dynamic_object_cast<FloatController>(parameterObject());
	if(ctrl != NULL && spinner())
		ctrl->setCurrentValue(spinner()->floatValue());
}

/******************************************************************************
* This method updates the displayed value of the parameter UI.
******************************************************************************/
void FloatControllerUI::updateUI()
{
	FloatController* ctrl = dynamic_object_cast<FloatController>(parameterObject());
	if(ctrl != NULL && spinner() && !spinner()->isDragging()) {
		spinner()->setFloatValue(ctrl->getCurrentValue());
	}
}

/******************************************************************************
* Gets the minimum value to be entered.
* This value is in native controller units.
******************************************************************************/
FloatType FloatControllerUI::minValue() const
{
	return (spinner() ? spinner()->minValue() : -FLOATTYPE_MAX);
}

/******************************************************************************
* Sets the minimum value to be entered.
* This value must be specified in native controller units.
******************************************************************************/
void FloatControllerUI::setMinValue(FloatType minValue)
{
	if(spinner()) spinner()->setMinValue(minValue);
}

/******************************************************************************
* Gets the maximum value to be entered.
* This value is in native controller units.
******************************************************************************/
FloatType FloatControllerUI::maxValue() const
{
	return (spinner() ? spinner()->maxValue() : FLOATTYPE_MAX);
}

/******************************************************************************
* Sets the maximum value to be entered.
* This value must be specified in native controller units.
******************************************************************************/
void FloatControllerUI::setMaxValue(FloatType maxValue)
{
	if(spinner()) spinner()->setMaxValue(maxValue);
}

};

