///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file NativePluginClassDescriptor.h
 * \brief Contains the definition of the Core::NativePluginClassDescriptor and the Core::NativePluginClassInfo class.
 */

#ifndef __OVITO_NATIVE_PLUGINCLASS_DESCRIPTOR_H
#define __OVITO_NATIVE_PLUGINCLASS_DESCRIPTOR_H

#include <core/Core.h>
#include "PluginClassDescriptor.h"

namespace Core {

class PluginClass;					// defined below
class PropertyFieldDescriptor;		// defined in PropertyFieldDescriptor.h
class ObjectSaveStream;			// defined in ObjectSaveStream.h
class ObjectLoadStream;			// defined in ObjectLoadStream.h
class Plugin;						// defined in Plugin.h

typedef PluginClass* (*PluginClassConstructorFn)();
typedef PluginClass* (*PluginClassSerConstructorFn)(bool);

/**
 * \brief Every C++ class derived from PluginClass owns an instance of this class.
 *
 * This is the runtime part of the plugin class system. Every C++ plugin class derived from
 * PluginClass has an instance of NativePluginClassInfo as a static member. It is used to identify
 * the plugin class at runtime.
 *
 * Normaly a plugin developer does not have to deal with a NativePluginClassInfo. More important
 * is the PluginClassDescriptor also associated with each PluginClass derived class.
 * It can be accessed through the NativePluginClassInfo::descriptor(), the PLUGINCLASSINFO macro or
 * the static \c pluginClassDescriptor() method of each plugin class.
 *
 * \author Alexander Stukowski
 * \sa PluginClassDescriptor
 * \sa NativePluginClassDescriptor
 */
class CORE_DLLEXPORT NativePluginClassInfo
{
public:
	/// \brief Constructs the plugin class info object.
	/// \note This is an internal constructor that is not for public use.
	NativePluginClassInfo(const NativePluginClassInfo* _baseInfo, const QMetaObject* _classInfo, PluginClassConstructorFn _ctor, PluginClassSerConstructorFn _serctor, PluginClass* _singletonInstance)
		: baseInfo(_baseInfo), classInfo(_classInfo), next(_firstInfo), _descriptor(NULL), ctor(_ctor), serctor(_serctor), singletonInstance(_singletonInstance), pureClassName(NULL),
		_firstNativePropertyField(NULL)
	{
		_firstInfo = this;
	}

	/// \brief Returns the descriptor associated with the C++ plugin class.
	/// \return The class descriptor or \c NULL if the plugin that contains the
	///         class has not been completely loaded yet.
	inline PluginClassDescriptor* descriptor() {
		CHECK_POINTER(_descriptor);
		return _descriptor;
	};

	/// \brief Returns the name of this class.
	/// \return A pointer to the class name string (without namespace qualifier).
	const char* className() {
		if(pureClassName) return pureClassName;

		// Remove namespace qualifier from Qt's class name.
		pureClassName = classInfo->className();
		for(const char* p = pureClassName; *p != '\0'; p++) {
			if(p[0] == ':' && p[1] == ':') {
				p++;
				pureClassName = p+1;
			}
		}

		return pureClassName;
	}

	/// \brief Returns the first element of the linked list of property fields defined for this class if it is a RefMaker derived class.
	/// \note This is an internal method not meant for public use.
	const PropertyFieldDescriptor* firstNativePropertyField() const { return _firstNativePropertyField; }

	/// If this is the descriptor of a RefMaker-derived class then this method will return
	/// the property field with the given identifier that has been defined in the RefMaker-derived
	/// class.
	/// \return If no such field is defined by that class then NULL is returned.
	/// \note This method will not return property fields that have been defined in super-classes.
	/// \note This is an internal method not meant for public use.
	const PropertyFieldDescriptor* findNativePropertyField(const char* identifier) const;

private:

	/// The runtime-type information provided by Qt.
	const QMetaObject* classInfo;
	/// The base class info.
	const NativePluginClassInfo* baseInfo;
	/// The name of the class.
	const char* pureClassName;
	/// The linked list of property fields if this is a RefMaker class.
	PropertyFieldDescriptor* _firstNativePropertyField;
	/// All instances of this class are connected as a linked list.
	NativePluginClassInfo* next;
    /// The class descriptor from the manifest file. This will be filled in after the plugin containing
	/// this class has been loaded.
	PluginClassDescriptor* _descriptor;
	/// The constructor function that creates an instance of this class.
	PluginClassConstructorFn ctor;
	/// The deserialization constructor function that creates an instance of this class.
	PluginClassSerConstructorFn serctor;
	/// The instance of the class if this is a singleton class; otherwise NULL.
	PluginClass* singletonInstance;

	/// Calls the default constructor of this class.
	PluginClass* createInstance() { return ctor ? (*ctor)() : NULL; }
	/// Calls the deserialization constructor of this class.
	PluginClass* createInstance(bool isLoading) { return serctor ? (*serctor)(isLoading) : NULL; }

	/// The first element of the linked list.
	static NativePluginClassInfo* _firstInfo;

	friend class NativePlugin;
	friend class NativePluginClassDescriptor;
	friend class NativePropertyFieldDescriptor;
};

//////////////////////////// Macros ////////////////////////////////

/// This is for internal use only.
#define _DECLARE_PLUGIN_CLASS_COMMON(name)								\
	public:																\
		friend class Plugin;											\
		static NativePluginClassInfo __pluginClassInfo;						\
		virtual PluginClassDescriptor* pluginClassDescriptor() const; 	\
		typedef intrusive_ptr<name> SmartPtr;

#define DECLARE_PLUGIN_CLASS(name)			\
	_DECLARE_PLUGIN_CLASS_COMMON(name)		\
    static PluginClass* createInstance();

#define DECLARE_SERIALIZABLE_PLUGIN_CLASS(name)	\
	DECLARE_PLUGIN_CLASS(name)				\
	static PluginClass* createSerializedInstance(bool isLoading=false);

#define DECLARE_ABSTRACT_PLUGIN_CLASS(name) \
	_DECLARE_PLUGIN_CLASS_COMMON(name)

#define DECLARE_SINGLETON_PLUGIN_CLASS(name)			\
    private:                                                    \
		static intrusive_ptr<name> __singleton_instance;						\
		_DECLARE_PLUGIN_CLASS_COMMON(name)				\
	public:														\
		inline static name& getInstance() { return *__singleton_instance.get(); }

/// This is for internal use only.
#define __IMPLEMENT_PLUGIN_CLASS_COMMON(name, basename, ctor, serctor, instance)			\
	NativePluginClassInfo name::__pluginClassInfo(&basename::__pluginClassInfo, &name::staticMetaObject, (PluginClassConstructorFn)ctor, (PluginClassSerConstructorFn)serctor, instance);	\
	PluginClassDescriptor* name::pluginClassDescriptor() const	\
		{ return name::__pluginClassInfo.descriptor(); }

#define IMPLEMENT_DYNAMIC_CLASS(name, basename)		\
    PluginClass* name::createInstance()           \
        { return new name(); }

#define IMPLEMENT_ABSTRACT_CLASS(name, basename)

#define IMPLEMENT_PLUGIN_CLASS(name, basename)					\
	IMPLEMENT_DYNAMIC_CLASS(name, basename)						\
	__IMPLEMENT_PLUGIN_CLASS_COMMON(name, basename, name::createInstance, NULL, NULL)

#define IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(name, basename)		\
	IMPLEMENT_DYNAMIC_CLASS(name, basename)						\
	__IMPLEMENT_PLUGIN_CLASS_COMMON(name, basename, name::createInstance, name::createSerializedInstance, NULL) \
    PluginClass* name::createSerializedInstance(bool isLoading)           \
        { return new name(isLoading); }

#define IMPLEMENT_ABSTRACT_PLUGIN_CLASS(name, basename)			\
	IMPLEMENT_ABSTRACT_CLASS(name, basename)					\
	__IMPLEMENT_PLUGIN_CLASS_COMMON(name, basename, NULL, NULL, NULL)

#define IMPLEMENT_SINGLETON_PLUGIN_CLASS(name, basename)		\
	IMPLEMENT_ABSTRACT_CLASS(name, basename)					\
	intrusive_ptr<name> name::__singleton_instance(new name());							\
	__IMPLEMENT_PLUGIN_CLASS_COMMON(name, basename, NULL, NULL, name::__singleton_instance.get())

#define PLUGINCLASSINFO(name)		(name::__pluginClassInfo.descriptor())

/**
 * \brief Describes a class defined by a native plugin.
 *
 * Each public C++ class provided by a plug-in is decribed by such a descriptor structure
 * that is loaded from the plug-in's manifest file.
 *
 * \author Alexander Stukowski
 * \sa Plugin, PluginClass, NativePluginClassInfo
 */
class CORE_DLLEXPORT NativePluginClassDescriptor : public PluginClassDescriptor
{
public:

	/// The descriptor of the root class of all plugin classes: PluginClass.
	static NativePluginClassDescriptor nativeRootClass;

protected:

	/// \brief Creates an instance of the class described by this descriptor.
	/// \param isLoading This specifies whether the object is being loaded from a file or just
	///                  created from scratch at runtime.
	/// \return The new instance of the class. The pointer can safely be cast to the appropriate C++ class type.
	/// \throw Exception if the instance could not be created.
	virtual intrusive_ptr<PluginClass> createInstanceImpl(bool isLoading);

private:

	/// \brief Associates this plugin class descriptor with the corresponding
	///        class info record after a native plugin has been loaded.
	void associateWithClassInfo(NativePluginClassInfo* classInfo);

	/// The descriptor of the root class of all plugin classes: PluginClass.
	static NativePluginClassDescriptor rootClass;

	/// The class info for this native plugin class.
	/// This is only available when the plugin has been loaded.
	NativePluginClassInfo* classInfo;

	/// Constructor for the root element.
	NativePluginClassDescriptor();

	/// Constructor.
	NativePluginClassDescriptor(const QString& name, PluginClassDescriptor* baseClass, Plugin* plugin, const QDomElement& classNode, bool isAbstract, bool serializable);

	friend class NativePlugin;
	friend class PluginManager;
};

};

#endif // __OVITO_NATIVE_PLUGINCLASS_DESCRIPTOR_H
