///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/plugins/branding/Branding.h>
#include <core/plugins/branding/UIEventFilter.h>
#include <core/plugins/PluginManager.h>

namespace Core {

IMPLEMENT_PLUGIN_CLASS(Branding, PluginClass)
IMPLEMENT_ABSTRACT_PLUGIN_CLASS(UIEventFilter, PluginClass)

/******************************************************************************
* Returns the internal name of the application.
******************************************************************************/
QString Branding::applicationName() 
{ 
	return tr("ovito"); 
}

/******************************************************************************
* Returns the display name of the application.
******************************************************************************/
QString Branding::productName() 
{ 
	return tr("Ovito (Open Visualization Tool)"); 
}

/******************************************************************************
* Returns the version of the application as a string.
******************************************************************************/
QString Branding::productVersion() 
{ 
	return tr("Version %1.%2.%3 (r%4)")
		.arg(OVITO_PROJ_VERSION_MAJOR)
		.arg(OVITO_PROJ_VERSION_MINOR)
		.arg(OVITO_PROJ_VERSION_REVISION)
		.arg(OVITO_PROJ_REPOSITORY_REVISION); 
}

/******************************************************************************
* Returns the name of the organization that wrote this application.
******************************************************************************/
QString Branding::organizationName()
{
	return tr("ovito");
}

/******************************************************************************
* Returns a string that describes the application.
******************************************************************************/
QString Branding::productDescription()
{ 
	return tr("Visualization and analysis software for atomistic simulation data."); 
}

/******************************************************************************
* Returns a string that contains the copyright information
******************************************************************************/
QString Branding::copyrightString()
{ 
	return tr("Copyright 2009 Alexander Stukowski\n"
    		  "This program comes with ABSOLUTELY NO WARRANTY.\n"
    		  "This is free software, and you are welcome to redistribute\n"
    		  "it under certain conditions. See the source for copying conditions."); 
}

/******************************************************************************
* Returns the URL of the product's website.
******************************************************************************/
QString Branding::productWebSite()
{ 
	return tr("http://www.ovito.org/"); 
}

/******************************************************************************
* Returns the default file extension to use for scene files.
******************************************************************************/
QString Branding::sceneFileExtension()
{
	return tr("ovito");
}

/******************************************************************************
* Returns the string that is used as description in the filter box
* of the File open and FIle save dialog boxes.
******************************************************************************/
QString Branding::sceneFileFilter()
{
	return tr("Scene Files (*.ovito);;All Files (*)");
}

/// Stores the list of installed brandings.
QVector<Branding::SmartPtr> BrandingManager::_brandings;

/******************************************************************************
* Returns all registered brandings.
******************************************************************************/
const QVector<Branding::SmartPtr>& BrandingManager::brandings() 
{
	// Check if list of brandings must be generated.
	if(_brandings.empty()) {

        // Add the default branding.
		_brandings.push_back(new Branding());

		// Find all installed branding plugins.
		Q_FOREACH(PluginClassDescriptor* clazz, PLUGIN_MANAGER.listClasses(PLUGINCLASSINFO(Branding))) {
			if(clazz == PLUGINCLASSINFO(Branding)) continue;	// Skip the default branding class. We already created an instance of this.

			try {
				// Create an instance of the branding class.
				Branding::SmartPtr branding = static_object_cast<Branding>(clazz->createInstance());
				_brandings.push_back(branding);
			}
			catch(const Exception& ex) {
				// Loading a branding plugin may fail. Continue after showing the error message.
				ex.showError();
			}
		}
	}
	return _brandings;
}

/******************************************************************************
* Releases all installed brandings.
******************************************************************************/
void BrandingManager::shutdown()
{
	_brandings.clear();
}

};
