///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include "BezierCurve.h"

namespace Core {

/******************************************************************************
* Computes the length of a segment of the curve.
******************************************************************************/
FloatType BezierCurve::segmentLength(int segment) const
{
	OVITO_ASSERT_MSG(segment >= 0 && segment < segmentCount(), "BezierCurve::segmentLength()", "Segment index out of range.");
	const BezierVertex& v1 = _vertices[segment];
	const BezierVertex& v2 = _vertices[(segment == vertexCount()-1) ? 0 : segment+1];
	if(v1.segmentType() == BezierVertex::SegmentLine ||
		v1.vertexType() == BezierVertex::VertexCorner ||
		v2.vertexType() == BezierVertex::VertexCorner) {
		return Distance(v1.pos(), v2.pos());
	}
	else {
		return segmentLengthImpl(v1.pos(), v1.outPoint(), v2.inPoint(), v2.pos());
	}
}

/******************************************************************************
* Computes an interpolation point on a curve segment.
******************************************************************************/
Point3 BezierCurve::interpolateSegment(int segment, FloatType t) const
{
	OVITO_ASSERT_MSG(t>=0.0 && t <= 1.0, "BezierCurve::interpolateSegment()", "Interpolation parameter t is out of range.");
	OVITO_ASSERT_MSG(segment >= 0 && segment < segmentCount(), "BezierCurve::interpolateSegment()", "Segment index out of range.");
	const BezierVertex& v1 = _vertices[segment];
	const BezierVertex& v2 = _vertices[(segment == vertexCount()-1) ? 0 : segment+1];
	if(v1.segmentType() == BezierVertex::SegmentLine ||
		v1.vertexType() == BezierVertex::VertexCorner ||
		v2.vertexType() == BezierVertex::VertexCorner) {

		return v1.pos() + t * (v2.pos() - v1.pos());
	}
	else {
		FloatType U2 = square(t);
		FloatType U3 = U2 * t;
		FloatType Ti = 1.0 - t;
		FloatType T2 = square(Ti);
		FloatType T3 = T2 * Ti;
		return v1.pos() * T3 + v1.outPoint() * (3.0 * t * T2) +
				v2.inPoint() * (3.0 * U2 * Ti) + v2.pos() * U3;
	}
}

/******************************************************************************
* Computes the derivative at a point of a given curve segment.
******************************************************************************/
Vector3 BezierCurve::derivativeSegment(int segment, FloatType t) const
{
	OVITO_ASSERT_MSG(t>=0.0 && t <= 1.0, "BezierCurve::derivativeSegment()", "Interpolation parameter t is out of range.");
	OVITO_ASSERT_MSG(segment >= 0 && segment < segmentCount(), "BezierCurve::derivativeSegment()", "Segment index out of range.");
	const BezierVertex& v1 = _vertices[segment];
	const BezierVertex& v2 = _vertices[(segment == vertexCount()-1) ? 0 : segment+1];
	if(v1.segmentType() == BezierVertex::SegmentLine ||
		v1.vertexType() == BezierVertex::VertexCorner ||
		v2.vertexType() == BezierVertex::VertexCorner) {

		return v2.pos() - v1.pos();
	}
	else {
		FloatType ti = 1.0 - t;
		Point3 c0 = v1.pos()*ti + v1.outPoint()*t;
		Point3 c1 = v2.inPoint()*t + v1.outPoint()*ti;
		Point3 c2 = v2.pos()*t + v2.inPoint()*ti;
		Point3 d0 = c0*ti + c1*t;
		Point3 d1 = c2*t + c1*ti;
		return (d1-d0) * 3.0;
	}
}

/******************************************************************************
* Computes the second derivative at a point of a given curve segment.
******************************************************************************/
Vector3 BezierCurve::derivativeSegment2(int segment, FloatType t) const
{
	OVITO_ASSERT_MSG(t>=0.0 && t <= 1.0, "BezierCurve::derivativeSegment()", "Interpolation parameter t is out of range.");
	OVITO_ASSERT_MSG(segment >= 0 && segment < segmentCount(), "BezierCurve::derivativeSegment()", "Segment index out of range.");
	const BezierVertex& v1 = _vertices[segment];
	const BezierVertex& v2 = _vertices[(segment == vertexCount()-1) ? 0 : segment+1];
	if(v1.segmentType() == BezierVertex::SegmentLine ||
		v1.vertexType() == BezierVertex::VertexCorner ||
		v2.vertexType() == BezierVertex::VertexCorner) {

		return NULL_VECTOR;
	}
	else {
		FloatType ti = 1.0 - t;
		Point3 c0 = v1.pos()*ti + v1.outPoint()*t;
		Point3 c1 = v2.inPoint()*t + v1.outPoint()*ti;
		Point3 c2 = v2.pos()*t + v2.inPoint()*ti;
		Point3 d0 = c0*ti + c1*t;
		Point3 d1 = c2*t + c1*ti;
		return Vector3((c2.X - c1.X*2.0 + c0.X) * 6.0,
						(c2.Y - c1.Y*2.0 + c0.Y) * 6.0,
						(c2.Z - c1.Z*2.0 + c0.Z) * 6.0);
	}
}

/******************************************************************************
* Computes an interpolation point on the curve.
******************************************************************************/
Point3 BezierCurve::interpolateCurve(FloatType t) const
{
	OVITO_ASSERT_MSG(t>=0.0 && t <= 1.0, "BezierCurve::interpolateCurve()", "Interpolation parameter t is out of range.");
	if(vertexCount() < 2) return ORIGIN;
	if(t == 1.0 && isClosed()) t = 0.0;
	FloatType segCount = segmentCount();
	int segment = (int)(t * segCount);
	return interpolateSegment(segment, (t - (FloatType)segment / segCount) * segCount);
}

/******************************************************************************
* Rebuilds the cached state data of the curve.
******************************************************************************/
void BezierCurve::validate()
{
	if(_isValid) return;

	_boundingBox.setEmpty();
	for(QVector<BezierVertex>::const_iterator vertex = vertices().begin(); vertex != vertices().end(); ++vertex) {
		_boundingBox += vertex->pos();
		if(vertex->vertexType() != BezierVertex::VertexCorner) {
			_boundingBox += vertex->inPoint();
			_boundingBox += vertex->outPoint();
		}
	}
	_polygon.setCurve(*this, 8);

	_isValid = true;
}

/******************************************************************************
* Saves the curve to the given stream.
******************************************************************************/
void BezierCurve::saveToStream(SaveStream& stream) const
{
	stream.beginChunk(0x01);

	stream << isClosed();
	stream.writeSizeT(vertexCount());
	for(QVector<BezierVertex>::const_iterator vertex = vertices().begin(); vertex != vertices().end(); ++vertex) {
		stream.writeEnum(vertex->vertexType());
		stream.writeEnum(vertex->segmentType());
		stream << vertex->pos();
		stream << vertex->inPoint();
		stream << vertex->outPoint();
	}

	stream.endChunk();
}

/******************************************************************************
* Loads the curve from the given stream.
******************************************************************************/
void BezierCurve::loadFromStream(LoadStream& stream)
{
	stream.expectChunk(0x01);

	stream >> _isClosed;
	size_t nvertices;
	stream.readSizeT(nvertices);
	_vertices.resize(nvertices);
	for(QVector<BezierVertex>::iterator vertex = vertices().begin(); vertex != vertices().end(); ++vertex) {
		stream.readEnum(vertex->_vertexType);
		stream.readEnum(vertex->_segmentType);
		stream >> vertex->_pos;
		stream >> vertex->_in;
		stream >> vertex->_out;
	}

	stream.closeChunk();
	invalidate();
}

};	// End of namespace Core
