///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file ViewportManager.h
 * \brief Contains the definition of the Core::ViewportManager class.
 */

#ifndef __OVITO_VIEWPORT_MANAGER_H
#define __OVITO_VIEWPORT_MANAGER_H

#include <core/Core.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/ViewportConfiguration.h>

namespace Core {

/// \def VIEWPORT_MANAGER
/// \brief The predefined instance of the Core::ViewportManager class.
///
/// Always use this macro to access the Core::ViewportManager class instance.
#define VIEWPORT_MANAGER		(*ViewportManager::getSingletonInstance())

/**
 * \brief Manages the Viewport windows.
 *
 * This is a singleton class with only one predefined instance of this class.
 * You can access the instance of this class using the VIEWPORT_MANAGER macro.
 *
 * \author Alexander Stukowski
 * \sa Viewport
 */
class CORE_DLLEXPORT ViewportManager : public QObject
{
	Q_OBJECT
public:

	/// \brief Returns the one and only instance of this class.
	/// \return The predefined instance of the ViewportManager singleton class.
	/// \note You should use the VIEWPORT_MANAGER macro to access the ViewportManager instance instead
	///       of this method.
	inline static ViewportManager* getSingletonInstance() {
		OVITO_ASSERT_MSG(_singletonInstance != NULL, "ViewportManager::getSingletonInstance", "ViewportManager class is not initialized yet.");
		return _singletonInstance;
	}

	/// \brief Returns the active viewport.
	/// \return The active Viewport or \c NULL if no viewport is currently active.
	/// \sa setActiveViewport()
	Viewport* activeViewport() { return _activeViewport; }

	/// \brief Sets the active viewport.
	/// \param vp The viewport to be made active.
	/// \note Calling this method will redraw all viewports.
	/// \sa activeViewport()
	void setActiveViewport(Viewport* vp);

	/// \brief Returns the maximized viewport.
	/// \return The maximized viewport or \c NULL if no one is currently maximized.
	/// \sa setMaximizedViewport()
	Viewport* maximizedViewport() { return _maximizedViewport; }

	/// \brief Maximizes a viewport.
	/// \param vp The viewport to be maximized or \c NULL to restore the currently maximized viewport to
	///           its original state.
	/// \sa maximizedViewport()
	void setMaximizedViewport(Viewport* vp);

	/// \brief This will flag all viewports for redrawing.
	/// \param forceCompleteUpdate If set to \c true a full update of all viewports will be performed.
	///                            If set to \c false a fast update will be done on the viewports and only
	///                            those parts of the scene will be redrawn that have changed.
	///
	/// This function does not cause an immediate repaint of the viewports; instead it schedules a
	/// paint event for processing when Qt returns to the main event loop. You can call this method as often
	/// as you want; it will return immediately and will cause only one viewport repaint when Qt returns to the
	/// main event loop.
	///
	/// To update only a single viewport Viewport::updateViewport() should be used.
	///
	/// To redraw all viewports immediately without waiting for the paint event to be processed,
	/// call processViewportUpdates() subsequently.
	///
	/// \sa Viewport::updateViewport(), processViewportUpdates()
	void updateViewports(bool forceCompleteUpdate = false);

	/// \brief Immediately repaints all viewport that have been flagged for
	///        an update using updateViewports().
	/// \sa updateViewports()
	void processViewportUpdates();

	/// \brief A call to this method suspends redrawing of the viewports.
	///
	/// To resume redrawing of viewports call resumeViewportUpdates().
	///
	/// Calling updateViewports() while redrawing is suspended will update the
	/// viewports as soon as redrawing is resumed.
	///
	/// Normally you should use the ViewportSuspender helper class to suspend viewport update.
	/// It has the advantage of being exception-safe.
	///
	/// \sa resumeViewportUpdates(), isSuspended(), ViewportSuspender
	void suspendViewportUpdates() { viewportSuspendCount++; }

	/// \brief This will resume redrawing of the viewports after a call to suspendViewportUpdates().
	/// \sa suspendViewportUpdates(), isSuspended()
	void resumeViewportUpdates();

	/// \brief Returns whether viewport updates are suspended.
	/// \return \c true if suspendViewportUpdates() has been called to suspend any viewport updates.
	/// \sa suspendViewportUpdates(), resumeViewportUpdates()
	bool isSuspended() const { return viewportSuspendCount > 0; }

	/// \brief Returns whether any of the viewports in the main viewport panel
	///        is currently being updated.
	/// \return \c true if there is currently a rendering operation going on.
	///
	/// No new windows or dialogs should be shown during this phase
	/// to prevent an infinite update loop.
	///
	/// \sa Window3DContainer::isRendering()
	bool isRendering() const;

	/// \brief Returns all viewports of the main viewport panel.
	/// \return The list of viewports.
	const QVector<Viewport*>& viewports() const;

	/// \brief Returns the default ViewportConfiguration used for new scene files.
	/// \return Pointer to the default viewport configuration stored by the ViewportManager.
	/// \note You may change the returned ViewportConfiguration object to modify the
	///       default configuration used for new scene files.
	/// \note Instead of using the returned object in a scene file you should make a copy of it first
	///       and leave the original object intact.
	ViewportConfiguration::SmartPtr defaultConfiguration();

private:

	/// The active viewport or NULL.
	Viewport* _activeViewport;

	/// The maximized viewport or NULL.
	Viewport* _maximizedViewport;

	/// This counter is for suspending the viewport updates.
	int viewportSuspendCount;

	/// Indicates that the viewports have been invalidated.
	bool viewportsNeedUpdate;

	/// Indicates that a full viewport update should be done.
	bool forceCompleteUpdate;

	/// This holds the default configuration to use for new scenes.
	ViewportConfiguration::SmartPtr defaultConfig;

private:

	/// Private constructor.
	/// This is a singleton class; no public instances are allowed.
	ViewportManager();

	/// Initializes the ViewportManager class.
	/// This is called at program startup.
	static void initialize() {
		OVITO_ASSERT(_singletonInstance == NULL);
		_singletonInstance = new ViewportManager();
	}

	/// Releases the ViewportManager.
	/// This is called at program shutdown.
	static void shutdown() {
		delete _singletonInstance;
		_singletonInstance = NULL;
	}

	static bool isInitialized() { return _singletonInstance != NULL; }

	/// The singleton instance of this class.
	static ViewportManager* _singletonInstance;

	friend class ApplicationManager;
};

/**
 * \brief Small helper class that suspends viewport redrawing while it
 * exists. It can be used to make your code exception-safe.
 *
 * The constructor of this class calls ViewportManager::suspendViewportUpdates() and
 * the destructor calls ViewportManager::resumeViewportUpdates().
 *
 * Just create an instance of this class on the stack to suspend viewport updates
 * during the lifetime of the class instance.
 *
 * \author Alexander Stukowski
 * \sa ViewportManager
 */
class ViewportSuspender {
public:
	ViewportSuspender() { VIEWPORT_MANAGER.suspendViewportUpdates(); }
	~ViewportSuspender() { VIEWPORT_MANAGER.resumeViewportUpdates(); }
};

};

#endif // __OVITO_VIEWPORT_MANAGER_H
