///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file CalculateExtrinsicStrainModifier.h
 * \brief Contains the definition of the CrystalAnalysis::CalculateExtrinsicStrainModifier class.
 */

#ifndef __CALCULATE_EXTRINSIC_STRAIN_MODIFIER_H
#define __CALCULATE_EXTRINSIC_STRAIN_MODIFIER_H

#include <core/Core.h>
#include <core/gui/properties/PropertiesEditor.h>

#include "../CrystalAnalysis.h"
#include <atomviz/modifier/AtomsObjectAnalyzerBase.h>
#include <atomviz/atoms/datachannels/DeformationGradientDataChannel.h>

namespace CrystalAnalysis {

/**
 * \brief This modifier computes the per-atom deformation gradient tensor
 *        based on the displacements of the neighbor atoms.
 *
 * \author Alexander Stukowski
 */
class CRYSTALANALYSIS_DLLEXPORT CalculateExtrinsicStrainModifier : public AtomsObjectAnalyzerBase
{
public:

	/// \brief Constructs a new instance of this class.
	/// \param isLoading Specifies whether the object's data fields will be initialized from the
	///                  data stored in a scene file after the instance has been created.
	CalculateExtrinsicStrainModifier(bool isLoading = false);

	/////////////////////////// specific methods ///////////////////////////////

	/// \brief Calculates the local deformation of each atom.
	/// \param atoms The atoms that should be analyzed. The input AtomsObject must
	///        contain the DataChannel::DisplacementChannel DataChannel since this deformation analysis
	///        relies on a previously performed calculation of the atomic displacement vectors.
	///
	/// The results get stored in internal arrays that can be accessed via the
	/// deformationGradients() and strainTensors() methods.
	///
	/// \throws Exception on error.
	/// \return \c false when the operation has been canceled by the user; \c true on success.
	bool calculate(AtomsObject* atoms, bool suppressDialogs = false);

	/// Returns the data channel that stores the computed deformation gradient for each atom.
	DeformationGradientDataChannel* deformationGradients() const { return deformationGradientChannel; }

	/// Returns the data channel that stores the computed strain tensor for each atom.
	DataChannel* strainTensors() const { return strainTensorChannel; }

	/// Returns the data channel that stores the computed hydrostatic strain for each atom.
	DataChannel* hydrostaticStrains() const { return hydrostaticStrainChannel; }

	/// Returns the data channel that stores the computed von Mises shear strain for each atom.
	DataChannel* shearStrains() const { return shearStrainChannel; }

	/// \brief Returns whether the analysis results are saved along with the scene.
	/// \return \c true if data is stored in the scene file; \c false if the data needs to be recomputed after loading the scene file.
	bool storeResultsWithScene() const { return deformationGradients() ? deformationGradients()->serializeData() : false; }

	/// \brief Returns whether analysis results are saved along with the scene.
	/// \param on \c true if data should be stored in the scene file; \c false if the data needs to be recomputed after loading the scene file.
	/// \undoable
	void setStoreResultsWithScene(bool on) { if(deformationGradients()) deformationGradients()->setSerializeData(on); }

public:

	Q_PROPERTY(bool storeResultsWithScene READ storeResultsWithScene WRITE setStoreResultsWithScene)

protected:

	/// This is the actual analysis method.
	/// It is responsible for storing the analysis results in the results container object that
	/// will be stored along with the modifier application object.
	virtual EvaluationStatus doAnalysis(TimeTicks time, bool suppressDialogs);

	/// Applies the previously calculated analysis results to the atoms object.
	virtual EvaluationStatus applyResult(TimeTicks time, TimeInterval& validityInterval);

	/// Analyzes the local strain of a single atom.
	/// Returns the local deformation gradient in the reference parameter F.
	void analyzeAtom(const NearestNeighborList::NeighborListAtom& atom, Tensor2& F, FloatType fccLatticeConstant);

	/// Adjust the distance vector between two atoms to the closest perdiodic image.
	void adjustVector(Vector3& r) const;

	/// This data channel is used to store the calculated deformation gradient
	/// for each atom in the system.
	ReferenceField<DeformationGradientDataChannel> deformationGradientChannel;

	/// This data channel is used to store the calculated strain tensor
	/// for each atom in the system.
	ReferenceField<DataChannel> strainTensorChannel;

	/// This data channel is used to store the calculated hydrostatic strain
	/// for each atom in the system.
	ReferenceField<DataChannel> hydrostaticStrainChannel;

	/// This data channel is used to store the calculated von Mises shear strain
	/// for each atom in the system.
	ReferenceField<DataChannel> shearStrainChannel;

private:

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(CalculateExtrinsicStrainModifier)
	DECLARE_REFERENCE_FIELD(deformationGradientChannel)
	DECLARE_REFERENCE_FIELD(strainTensorChannel)
	DECLARE_REFERENCE_FIELD(hydrostaticStrainChannel)
	DECLARE_REFERENCE_FIELD(shearStrainChannel)
};

/**
 * \brief A properties editor for the CalculateExtrinsicStrainModifier class.
 *
 * This editor class creates and manages the user interface through which the
 * user can alter the modifier's parameters.
 *
 * \author Alexander Stukowski
 */
class CRYSTALANALYSIS_DLLEXPORT CalculateExtrinsicStrainModifierEditor : public AtomsObjectModifierEditorBase
{
protected:

	/// Creates the user interface controls for the editor.
	virtual void createUI(const RolloutInsertionParameters& rolloutParams);

protected Q_SLOTS:

	/// Is called when the user presses the "Calculate" button.
	void onCalculate();

private:

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(CalculateExtrinsicStrainModifierEditor)
};

};	// End of namespace CrystalAnalysis

#endif // __CALCULATE_EXTRINSIC_STRAIN_MODIFIER_H
