///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __TACHYON_EXPORT_INTERFACE_H
#define __TACHYON_EXPORT_INTERFACE_H

#include <core/Core.h>
#include <core/scene/animation/TimeInterval.h>
#include <core/viewport/Viewport.h>
#include "../TachyonPlugin.h"

#include "../tachyonlib/tachyon.h"

namespace Core {

class ObjectNode;		// defined in ObjectNode.h
class SceneObject;		// defined in SceneObject.h
class DataSet;			// defined in DataSet.h

};	// End of namespace Core

namespace TachyonPlugin {

class TachyonWriter; 	// defined below
class TachyonRenderer;	// defined in TachyonRenderer.h

/******************************************************************************
* Plugins can implement this abstract interface class to export
* their custom object types to the Tachyon scene description format.
******************************************************************************/
class TACHYON_DLLEXPORT TachyonExportInterface : public PluginClass
{
public:
	/// The default constructor.
	TachyonExportInterface() : PluginClass() {}

	/// Exports the given scene object in the Tachyon scene description format and returns true.
	/// Return false if the scene object type is not supported by this interface class.
	/// Throws an exception if an error occurred.
	virtual bool exportSceneObject(SceneObject* sceneObj, TachyonWriter& writer, ObjectNode* contextNode, const AffineTransformation& objToWorldTM) = 0;

private:

	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(TachyonExportInterface)
};

/******************************************************************************
* This interface class sends primitives to the Tachyon renderer.
******************************************************************************/
class TACHYON_DLLEXPORT TachyonWriter
{
public:

	/// Constructor.
	TachyonWriter(DataSet* dataset, TimeTicks time, const CameraViewDescription& view, TachyonRenderer* renderer, SceneHandle rtscene) :
		_dataset(dataset), _time(time), _view(view), _renderer(renderer), _rtscene(rtscene), _currentTM(IDENTITY) {}

	/// Returns the camera settings.
	const CameraViewDescription& view() const { return _view; }

	/// Returns the scene that is being exported.
	DataSet* dataSet() const { return _dataset; }

	/// Returns the scene time that is being exported.
	TimeTicks time() const { return _time; }

	/// Returns the renderer object.
	TachyonRenderer* renderer() const { return _renderer; }

	/// Sets the transformation to be applied to all primitives before sending them to Tachyon.
	void setTransformation(const AffineTransformation& tm) { _currentTM = tm; }

	/// Returns the transformation being applied to all primitives before sending them to Tachyon.
	const AffineTransformation& transformation() const { return _currentTM; }

	/// Renders a list of spheres.
	void sphereArray(int numberOfSpheres, Point3 const* centers, FloatType const* radii, Color const* colors);

	/// Renders a cylinder (without caps).
	void cylinder(const Point3& from, const Point3& to, FloatType radius, const Color& color, bool topCap = false, bool bottomCap = false);

	/// Renders a sphere.
	void sphere(const Point3& center, FloatType radius, const Color& color);

private:

	/// Creates a texture with the given color.
	void* getTexture(const Color& color);

private:

	/// Specifies the viewing parameters and camera positions.
	CameraViewDescription _view;

	/// The scene that is being exported.
	DataSet* _dataset;

	/// The scene time that is being exported.
	TimeTicks _time;

	/// The active TachyonRenderer.
	TachyonRenderer* _renderer;

	/// The Tachyon internal scene handle.
	SceneHandle _rtscene;

	/// Transformation applied to all primitives before sending them to Tachyon.
	AffineTransformation _currentTM;
};

};

#endif // __TACHYON_EXPORT_INTERFACE_H
