///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/scene/animation/controller/StandardControllers.h>
#include <core/gui/properties/FloatControllerUI.h>
#include <core/gui/properties/ColorControllerUI.h>
#include <core/gui/properties/BooleanPropertyUI.h>
#include <core/undo/UndoManager.h>

#if !defined(Q_WS_MAC)
	#include <GL/glu.h>
#endif

#include "DisplacementDataChannel.h"
#include <atomviz/atoms/AtomsObject.h>

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(DisplacementDataChannel, DataChannel)
DEFINE_FLAGS_REFERENCE_FIELD(DisplacementDataChannel, VectorController, "ArrowColor", PROPERTY_FIELD_ALWAYS_DEEP_COPY, _arrowColor)
DEFINE_FLAGS_REFERENCE_FIELD(DisplacementDataChannel, FloatController, "ArrowWidth", PROPERTY_FIELD_ALWAYS_DEEP_COPY, _arrowWidth)
DEFINE_FLAGS_REFERENCE_FIELD(DisplacementDataChannel, FloatController, "ScalingFactor", PROPERTY_FIELD_ALWAYS_DEEP_COPY, _scalingFactor)
DEFINE_PROPERTY_FIELD(DisplacementDataChannel, "SolidArrows", _solidArrows)
DEFINE_PROPERTY_FIELD(DisplacementDataChannel, "ReverseArrowDirection", _reverseArrowDirection)
DEFINE_PROPERTY_FIELD(DisplacementDataChannel, "FlipDisplacementVectors", _flipDisplacementVectors)
SET_PROPERTY_FIELD_LABEL(DisplacementDataChannel, _arrowColor, "Arrow color")
SET_PROPERTY_FIELD_LABEL(DisplacementDataChannel, _arrowWidth, "Arrow width")
SET_PROPERTY_FIELD_LABEL(DisplacementDataChannel, _scalingFactor, "Scaling factor")
SET_PROPERTY_FIELD_LABEL(DisplacementDataChannel, _solidArrows, "Solid arrows")
SET_PROPERTY_FIELD_LABEL(DisplacementDataChannel, _reverseArrowDirection, "Reverse arrow direction")
SET_PROPERTY_FIELD_LABEL(DisplacementDataChannel, _flipDisplacementVectors, "Flip displacement vectors")
SET_PROPERTY_FIELD_UNITS(DisplacementDataChannel, _arrowWidth, WorldParameterUnit)

/******************************************************************************
* Serialization constructor.
******************************************************************************/
DisplacementDataChannel::DisplacementDataChannel(bool isLoading) : DataChannel(isLoading),
	displacementLinesValidity(TimeNever), _solidArrows(false), _reverseArrowDirection(false), _flipDisplacementVectors(false)
{
	init(isLoading);
}

/******************************************************************************
* Special constructor to create a standard data channel.
******************************************************************************/
DisplacementDataChannel::DisplacementDataChannel(DataChannelIdentifier which) : DataChannel(which),
	displacementLinesValidity(TimeNever), _solidArrows(false), _reverseArrowDirection(false), _flipDisplacementVectors(false)
{
	init(false);
}

/******************************************************************************
* Initializes the object. This is called from the constructors.
******************************************************************************/
void DisplacementDataChannel::init(bool isLoading)
{
	INIT_PROPERTY_FIELD(DisplacementDataChannel, _arrowColor);
	INIT_PROPERTY_FIELD(DisplacementDataChannel, _arrowWidth);
	INIT_PROPERTY_FIELD(DisplacementDataChannel, _solidArrows);
	INIT_PROPERTY_FIELD(DisplacementDataChannel, _scalingFactor);
	INIT_PROPERTY_FIELD(DisplacementDataChannel, _reverseArrowDirection);
	INIT_PROPERTY_FIELD(DisplacementDataChannel, _flipDisplacementVectors);
	if(!isLoading) {
		_arrowColor = CONTROLLER_MANAGER.createDefaultController<VectorController>();
		_arrowWidth = CONTROLLER_MANAGER.createDefaultController<FloatController>();
		_scalingFactor = CONTROLLER_MANAGER.createDefaultController<FloatController>();

		_arrowColor->setCurrentValue(Color(1.0, 1.0, 0.0));
		_arrowWidth->setCurrentValue(0.1f);
		_scalingFactor->setCurrentValue(1.0);
	}
}

/******************************************************************************
* Render the displacement vectors into the viewport.
******************************************************************************/
void DisplacementDataChannel::render(TimeTicks time, Viewport* vp, AtomsObject* atoms, ObjectNode* contextNode)
{
	AffineTransformation inverseTM = vp->inverseWorldMatrix() * vp->inverseViewMatrix();
	Vector3 viewDir = Normalize(inverseTM * Vector3(0,0,-1));
	Point3 viewPoint = inverseTM * Point3(0,0,0);
	OVITO_ASSERT(viewDir != NULL_VECTOR);
	renderDisplacements(time, atoms, vp->isPerspectiveProjection(), viewDir, viewPoint, vp);
}

/******************************************************************************
* Renders the channel's contents in high-quality mode to an offscreen buffer.
******************************************************************************/
void DisplacementDataChannel::renderHQ(TimeTicks time, AtomsObject* atoms, const CameraViewDescription& view, ObjectNode* contextNode, int imageWidth, int imageHeight, Window3D* glcontext)
{
	Matrix4 tm;
	glGetFloatTypev(GL_MODELVIEW_MATRIX, tm.data());
	Matrix4 inverseTM = tm.inverse();
	Vector3 viewDir = Normalize(inverseTM * Vector3(0,0,-1));
	Point3 viewPoint = inverseTM * Point3(0,0,0);

	renderDisplacements(time, atoms, view.isPerspective, viewDir, viewPoint, NULL);
}

/******************************************************************************
* Renders the displacement vectors.
******************************************************************************/
void DisplacementDataChannel::renderDisplacements(TimeTicks time, AtomsObject* atoms, bool isPerspective, Vector3 viewDir, const Point3& viewPoint, Viewport* vp)
{
	// Get the position channel.
	DataChannel* posChannel = atoms->getStandardDataChannel(DataChannel::PositionChannel);
	if(!posChannel) return;

	// Setup material
	glPushAttrib(GL_LIGHTING_BIT);
	Color color(0,0,0);
	if(_arrowColor) {
		color = _arrowColor->getValueAtTime(time);
		color.clampMinMax();
	}
	Window3DMaterial material;
	if(vp) {
		material.diffuse = color;
		vp->setMaterialCount(1);
		vp->setMaterial(0, &material);
		if(_solidArrows) {
			glEnable(GL_LIGHTING);
			vp->realizeMaterial(0);
		}
		else {
			vp->setRenderingColor(color);
			glDisable(GL_LIGHTING);
		}
	}
	else {
		if(_solidArrows) {
			glEnable(GL_LIGHTING);
			float ambient[4] = { 0.2f, 0.2f, 0.2f, 1.0f };
			glMaterialfv(GL_FRONT, GL_AMBIENT, ambient);
			float diffuse[4] = { color.r, color.g, color.b, 1 };
			glMaterialfv(GL_FRONT, GL_DIFFUSE, diffuse);
			float specular[4] = { 0.1f, 0.1f, 0.1f, 1.0f };
			glMaterialfv(GL_FRONT, GL_SPECULAR, specular);
			float emission[4] = { 0, 0, 0, 1 };
			glMaterialfv(GL_FRONT, GL_EMISSION, emission);
			glMaterialf(GL_FRONT, GL_SHININESS, 0);
			glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER, 0);
		}
		else {
			glDisable(GL_LIGHTING);
			glColor3v(color.constData());
		}
	}

	FloatType scalingFactor = 1;
	if(_scalingFactor) scalingFactor = _scalingFactor->getValueAtTime(time);

	const Point3* p = posChannel->constDataPoint3();
	const Vector3* d = this->constDataVector3();
	FloatType arrowLineWidth = 0.2f;
	if(_arrowWidth) arrowLineWidth = _arrowWidth->getValueAtTime(time) * 0.5;
	FloatType arrowHeadRadius = arrowLineWidth * 2.0;
	FloatType arrowHeadLength = arrowHeadRadius * 1.5;
	if(_solidArrows) {
		GLUquadricObj* cylinderQuadric = gluNewQuadric();
		GLUquadricObj* diskQuadric = gluNewQuadric();
		gluQuadricNormals(cylinderQuadric, GLU_SMOOTH);
		for(size_t i = size(); i != 0; --i, ++p, ++d) {
			if(*d == NULL_VECTOR) continue;
			glPushMatrix();
			AffineTransformation tm;
			Vector3 scaledd = (*d) * scalingFactor;
			if((bool)_flipDisplacementVectors ^ (bool)_reverseArrowDirection)
				tm.column(2) = -scaledd;
			else
				tm.column(2) = scaledd;
			tm.column(3) = *p - ORIGIN;
			if(_reverseArrowDirection) {
				tm.column(3) += tm.column(2);
			}
			tm.column(1) = CrossProduct(tm.column(2), Vector3(0,0,1));
			if(tm.column(1) == NULL_VECTOR)
				tm.column(1) = CrossProduct(tm.column(2), Vector3(1,0,0));
			tm.column(0) = CrossProduct(tm.column(2), tm.column(1));
			tm.column(0) = Normalize(tm.column(0));
			tm.column(1) = Normalize(tm.column(1));
			tm.column(2) = Normalize(tm.column(2));
			FloatType len = Length(scaledd);
			glMultMatrix(Matrix4(tm).constData());
			glScalef(-1,-1,-1);
			if(len > arrowHeadLength) {
				gluCylinder(cylinderQuadric, 1e-2, arrowHeadRadius, arrowHeadLength, 16, 1);
				glTranslatef(0,0,arrowHeadLength);
				gluDisk(diskQuadric, 0, arrowHeadRadius, 16, 1);
				gluCylinder(cylinderQuadric, arrowLineWidth, arrowLineWidth, len - arrowHeadLength, 16, 1);
				glTranslatef(0,0,len - arrowHeadLength);
				gluDisk(diskQuadric, 0, arrowLineWidth, 16, 1);
			}
			else {
				gluCylinder(cylinderQuadric, 0, arrowHeadRadius * len / arrowHeadLength, len, 16, 1);
				glTranslatef(0,0,len);
				gluDisk(diskQuadric, 0, arrowHeadRadius * len / arrowHeadLength, 16, 1);
			}
			glPopMatrix();
		}
		gluDeleteQuadric(cylinderQuadric);
		gluDeleteQuadric(diskQuadric);
	}
	else {
		arrowHeadLength = -arrowHeadLength;
		for(size_t i = size(); i != 0; --i, ++p, ++d) {
			if(*d == NULL_VECTOR) continue;
			glPushMatrix();
			AffineTransformation tm;
			if(isPerspective)
				viewDir = *p - viewPoint;
			Vector3 scaledd = (*d) * scalingFactor;
			if((bool)_flipDisplacementVectors ^ (bool)_reverseArrowDirection)
				tm.column(2) = -scaledd;
			else
				tm.column(2) = scaledd;
			tm.column(3) = *p - ORIGIN;
			if(_reverseArrowDirection) {
				tm.column(3) += tm.column(2);
			}
			tm.column(1) = CrossProduct(tm.column(2), viewDir);
			if(tm.column(1) == NULL_VECTOR) {
				glPopMatrix();
				continue;
			}
			tm.column(0) = CrossProduct(tm.column(2), tm.column(1));
			tm.column(0) = Normalize(tm.column(0));
			tm.column(1) = Normalize(tm.column(1));
			tm.column(2) = Normalize(tm.column(2));
			FloatType len = -Length(scaledd);
			glMultMatrix(Matrix4(tm).constData());
			if(len < arrowHeadLength) {
				glBegin(GL_QUADS);
				glVertex3(0, -arrowLineWidth, arrowHeadLength);
				glVertex3(0, arrowLineWidth, arrowHeadLength);
				glVertex3(0, arrowLineWidth, len);
				glVertex3(0, -arrowLineWidth, len);
				glEnd();
				glBegin(GL_TRIANGLES);
				glVertex3(0, 0, 0);
				glVertex3(0, arrowHeadRadius, arrowHeadLength);
				glVertex3(0, -arrowHeadRadius, arrowHeadLength);
				glEnd();
			}
			else {
				glBegin(GL_TRIANGLES);
				glVertex3(0, 0, 0);
				glVertex3(0, arrowHeadRadius * len / arrowHeadLength, len);
				glVertex3(0, -arrowHeadRadius * len / arrowHeadLength, len);
				glEnd();
			}
			glPopMatrix();
		}
	}
	glPopAttrib();
	if(vp)
		vp->setMaterialCount(0);
}

/******************************************************************************
* Returns the bounding box of the channel's geometry when rendered in the viewports.
******************************************************************************/
Box3 DisplacementDataChannel::boundingBox(TimeTicks time, AtomsObject* atoms, ObjectNode* contextNode, TimeInterval& validityInterval)
{
	Box3 bb;
	if(size() == 0) return bb;

	// Get the position channel.
	DataChannel* posChannel = atoms->getStandardDataChannel(DataChannel::PositionChannel);
	if(!posChannel) return bb;

	FloatType scalingFactor = 1;
	if(_scalingFactor) _scalingFactor->getValue(time, scalingFactor, validityInterval);

	const Point3* p = posChannel->constDataPoint3();
	const Vector3* d = this->constDataVector3();
	bool flipVectors = (bool)_flipDisplacementVectors ^ (bool)_reverseArrowDirection;
	for(size_t i = size(); i != 0; --i, ++p, ++d) {
		bb.addPoint(*p);
		if(flipVectors)
			bb.addPoint(*p - *d * scalingFactor);
		else
			bb.addPoint(*p + *d * scalingFactor);
	}

	FloatType arrowLineWidth = 0.2f;
	if(_arrowWidth) _arrowWidth->getValue(time, arrowLineWidth, validityInterval);
	arrowLineWidth *= 0.5;
	FloatType arrowHeadRadius = arrowLineWidth * 2.0;

	return bb.padBox(max(arrowHeadRadius, arrowLineWidth));
}

/******************************************************************************
* Lets the channel clear all its internal caches.
******************************************************************************/
void DisplacementDataChannel::clearCaches()
{
	DataChannel::clearCaches();
	displacementLinesValidity.setEmpty();
}

IMPLEMENT_PLUGIN_CLASS(DisplacementDataChannelEditor, PropertiesEditor)

/******************************************************************************
* Sets up the UI widgets of the editor.
******************************************************************************/
void DisplacementDataChannelEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create a rollout.
	QWidget* rollout = createRollout(tr("Displacement Vectors"), rolloutParams, "atomviz.data_channels.displacement.parameters");

	QGridLayout* layout = new QGridLayout(rollout);
	layout->setContentsMargins(4,4,4,4);
#ifndef Q_WS_MAC
	layout->setSpacing(0);
#endif
	layout->setColumnStretch(1, 1);

	BooleanPropertyUI* showDisplacementsUI = new BooleanPropertyUI(this, "isVisible", tr("Show displacement vectors"));
	layout->addWidget(showDisplacementsUI->checkBox(), 0, 0, 1, 2);

	BooleanPropertyUI* solidArrowsUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(DisplacementDataChannel, _solidArrows));
	layout->addWidget(solidArrowsUI->checkBox(), 1, 0, 1, 2);

	BooleanPropertyUI* reverseArrowDirectionUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(DisplacementDataChannel, _reverseArrowDirection));
	layout->addWidget(reverseArrowDirectionUI->checkBox(), 2, 0, 1, 2);

	BooleanPropertyUI* flipDisplacementVectorsUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(DisplacementDataChannel, _flipDisplacementVectors));
	layout->addWidget(flipDisplacementVectorsUI->checkBox(), 3, 0, 1, 2);

	FloatControllerUI* scalingFactorUI = new FloatControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(DisplacementDataChannel, _scalingFactor));
	layout->addWidget(scalingFactorUI->label(), 4, 0);
	layout->addLayout(scalingFactorUI->createFieldLayout(), 4, 1);
	scalingFactorUI->setMinValue(0);

	FloatControllerUI* arrowWidthUI = new FloatControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(DisplacementDataChannel, _arrowWidth));
	layout->addWidget(arrowWidthUI->label(), 5, 0);
	layout->addLayout(arrowWidthUI->createFieldLayout(), 5, 1);
	arrowWidthUI->setMinValue(0);

	ColorControllerUI* arrowColorUI = new ColorControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(DisplacementDataChannel, _arrowColor));
	layout->addWidget(arrowColorUI->label(), 6, 0);
	layout->addWidget(arrowColorUI->colorPicker(), 6, 1);
}

};	// End of namespace AtomViz
