///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/utilities/ProgressIndicator.h>
#include "IMDAtomFileWriter.h"
#include <atomviz/atoms/AtomsObject.h>
#include <atomviz/atoms/datachannels/AtomTypeDataChannel.h>

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(IMDAtomFileWriter, MultiFileWriter)

/******************************************************************************
* Writes the output file.
******************************************************************************/
bool IMDAtomFileWriter::writeAtomsFile(const QString& filepath, DataSet* dataset, const QVector<TimeTicks>& exportFrames, bool suppressDialogs)
{
	if(exportFrames.size() != 1)
		MsgLogger() << "Warning: The IMD atom file writer can only write single frames." << endl;

	TimeTicks time = exportFrames[0];
	int frame = time / dataset->animationSettings()->ticksPerFrame();

	// Extract the atoms to be exported from the scene.
	PipelineFlowState flowState = retrieveAtoms(dataset, time);
	AtomsObject* atoms = dynamic_object_cast<AtomsObject>(flowState.result());
	if(atoms == NULL)
		throw Exception(tr("The scene does not contain any atoms that could be exported (at animation frame %1).").arg(frame));

	MsgLogger() << "Opening IMD atom file" << filepath << "for writing." << endl;

	QFile stream(filepath);
	if(!stream.open(QIODevice::WriteOnly|QIODevice::Text))
		throw Exception(tr("Failed to open the file %1 for writing: %2").arg(filepath, stream.errorString()));
	QTextStream textStream(&stream);
	textStream.setRealNumberPrecision(12);

	int numAtoms = atoms->atomsCount();
	ProgressIndicator progress(tr("Writing IMD atom file (frame %1)").arg(frame), numAtoms, suppressDialogs);

	DataChannel* posChannel = atoms->getStandardDataChannel(DataChannel::PositionChannel);
	AtomTypeDataChannel* atomTypeChannel = static_object_cast<AtomTypeDataChannel>(atoms->getStandardDataChannel(DataChannel::AtomTypeChannel));
	DataChannel* atomIndexChannel = atoms->getStandardDataChannel(DataChannel::AtomIndexChannel);
	if(atomIndexChannel == NULL) atomIndexChannel = atoms->findDataChannelByName("number");
	DataChannel* velocityChannel = atoms->getStandardDataChannel(DataChannel::VelocityChannel);
	DataChannel* massChannel = atoms->getStandardDataChannel(DataChannel::MassChannel);
	if(atomTypeChannel && atomTypeChannel->atomTypes().empty()) atomTypeChannel = NULL;

	ChannelColumnMapping channelMapping;
	QVector<QString> columnNames;
	textStream << "#F A ";
	if(atomIndexChannel) {
		textStream << "1 ";
		channelMapping.insertColumn(channelMapping.columnCount(), atomIndexChannel->id(), atomIndexChannel->name());
		columnNames.push_back("number");
	}
	else textStream << "0 ";
	if(atomTypeChannel) {
		textStream << "1 ";
		channelMapping.insertColumn(channelMapping.columnCount(), DataChannel::AtomTypeChannel, atomTypeChannel->name());
		columnNames.push_back("type");
	}
	else textStream << "0 ";
	if(massChannel) {
		textStream << "1 ";
		channelMapping.insertColumn(channelMapping.columnCount(), DataChannel::MassChannel, massChannel->name());
		columnNames.push_back("mass");
	}
	else textStream << "0 ";
	if(posChannel) {
		textStream << "3 ";
		channelMapping.insertColumn(channelMapping.columnCount(), DataChannel::PositionChannel, posChannel->name(), 0);
		channelMapping.insertColumn(channelMapping.columnCount(), DataChannel::PositionChannel, posChannel->name(), 1);
		channelMapping.insertColumn(channelMapping.columnCount(), DataChannel::PositionChannel, posChannel->name(), 2);
		columnNames.push_back("x");
		columnNames.push_back("y");
		columnNames.push_back("z");
	}
	else textStream << "0 ";
	if(velocityChannel) {
		textStream << "3 ";
		channelMapping.insertColumn(channelMapping.columnCount(), DataChannel::VelocityChannel, velocityChannel->name(), 0);
		channelMapping.insertColumn(channelMapping.columnCount(), DataChannel::VelocityChannel, velocityChannel->name(), 1);
		channelMapping.insertColumn(channelMapping.columnCount(), DataChannel::VelocityChannel, velocityChannel->name(), 2);
		columnNames.push_back("vx");
		columnNames.push_back("vy");
		columnNames.push_back("vz");
	}
	else textStream << "0 ";

	int otherColumnsCount = 0;
	Q_FOREACH(DataChannel* channel, atoms->dataChannels()) {
		if(channel == posChannel || channel == atomTypeChannel || channel == atomIndexChannel || channel == massChannel || channel == velocityChannel)
			continue;
		if(channel->id() == DataChannel::ColorChannel || channel->id() == DataChannel::SelectionChannel)
			continue;
		for(size_t component = 0; component < channel->componentCount(); component++) {
			channelMapping.insertColumn(channelMapping.columnCount(), channel->id(), channel->name(), component);
			otherColumnsCount += 1;
			QString columnName = channel->name();
			columnName.remove(QRegExp("[^A-Za-z\\d_]"));
			if(channel->componentNames().empty())
				columnNames.push_back(columnName);
			else {
				QString componentName = channel->componentNames()[component];
				componentName.remove(QRegExp("[^A-Za-z\\d_]"));
				columnNames.push_back(columnName + componentName);
			}
		}
	}
	textStream << otherColumnsCount << endl;

	textStream << "#C";
	Q_FOREACH(QString cname, columnNames) textStream << " " << cname;
	textStream << endl;

	AffineTransformation simCell = atoms->simulationCell()->cellMatrix();
	textStream << "#X " << simCell.column(0)[0] << " " << simCell.column(0)[1] << " " << simCell.column(0)[2] << endl;
	textStream << "#Y " << simCell.column(1)[0] << " " << simCell.column(1)[1] << " " << simCell.column(1)[2] << endl;
	textStream << "#Z " << simCell.column(2)[0] << " " << simCell.column(2)[1] << " " << simCell.column(2)[2] << endl;

	textStream << "## Generated on " << QDateTime::currentDateTime().toString() << endl;
	textStream << "## IMD file written by " << QCoreApplication::applicationName() << endl;
	textStream << "#E" << endl;

	DataRecordWriterHelper helper(&channelMapping, atoms);

	// Write atomic positions.
	for(int i=0; i<numAtoms; i++) {

		// Update progress indicator.
		if((i % 1000) == 0) {
			progress.setValue(i * 100 / numAtoms);
			progress.isCanceled();
		}

		// Shift atomic position by corner of simulation cell.
		Point3 oldPos;
		if(posChannel) {
			oldPos = posChannel->getPoint3(i);
			posChannel->setPoint3(i, oldPos - simCell.getTranslation());
		}

		helper.writeAtom(i, stream);
		stream.putChar('\n');

		// Restore old position.
		if(posChannel) {
			posChannel->setPoint3(i, oldPos);
		}
	}
	return true;
}

};	// End of namespace AtomViz
