///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <base/Base.h>
#include <base/io/SaveStream.h>
#include <base/utilities/Exception.h>

namespace Base {

/******************************************************************************
* Opens the stream for writing.
******************************************************************************/
SaveStream::SaveStream(QDataStream& destination) : os(destination), _isOpen(false)
{
	OVITO_ASSERT_MSG(!os.device()->isSequential(), "SaveStream constructor", "SaveStream class requires a seekable output stream.");
	if(os.device()->isSequential())
		throw Exception("SaveStream class requires a seekable output stream.");

	_isOpen = true;

	// Write file header.
	
	// This is used to recognize the application file format.
	*this << (quint32)0x0FACC5AA;	// The first magic file code.
	*this << (quint32)0x0AFCCA5A;	// The second magic file code.
	
	// This is the overall version of the application file format.
	*this << (quint32)11;		// Version 1.1
	os.setVersion(QDataStream::Qt_4_3);

	// Write number of floating-point bytes used.
	// This is the floating-point precision used throughout the file. 
	*this << (quint32)sizeof(FloatType);

	// Write application name.
	*this << QCoreApplication::applicationName();

	// Write application version.
	*this << (quint32)OVITO_PROJ_VERSION_MAJOR;
	*this << (quint32)OVITO_PROJ_VERSION_MINOR;
	*this << (quint32)OVITO_PROJ_VERSION_REVISION;
}

/******************************************************************************
* Closes the stream.
******************************************************************************/
void SaveStream::close()
{
	if(_isOpen) {		
		_isOpen = false;
	}
}

/******************************************************************************
* Writes an array of bytes to the ouput stream.
******************************************************************************/
void SaveStream::write(const void* buffer, size_t numBytes)
{
	if(os.writeRawData((const char*)buffer, numBytes) != numBytes)
		throw Exception(tr("Failed to write data to output file."));
}

/******************************************************************************
* Start a new chunk with the given id.
******************************************************************************/
void SaveStream::beginChunk(quint32 chunkId)
{
    *this << chunkId;
	*this << (quint32)0;	// This will be backpatched in endChunk()

	chunks.push(os.device()->pos());
}
    
/******************************************************************************
* Closes the current chunck.
******************************************************************************/
void SaveStream::endChunk()
{
	OVITO_ASSERT(!chunks.empty());
	qint64 chunkStart = chunks.top();
	chunks.pop();

	qint64 chunkSize = os.device()->pos() - chunkStart;
	OVITO_ASSERT(chunkSize >= 0 && chunkSize <= 0xFFFFFFFF);

	// Write chunk end code.
	*this << (quint32)0x0FFFFFFF;

	// Seek to chunk size field.
	if(!os.device()->seek(chunkStart - sizeof(unsigned int)) ) 
		throw Exception(tr("Failed to close chunk in output file."));

    // Patch chunk size field.
	*this << (quint32)chunkSize;

	// Jump back to end of file.
	if(!os.device()->seek(os.device()->size()))
		throw Exception(tr("Failed to close chunk in output file."));
	OVITO_ASSERT(os.device()->pos() == chunkStart + chunkSize + sizeof(unsigned int));
}

/******************************************************************************
* Writes a pointer to the stream.
* This method generates a unique ID for the given pointer that
* is written to the stream instead of the pointer itself. 
******************************************************************************/
void SaveStream::writePointer(void* pointer) 
{
	if(pointer == NULL) *this << (quint64)0;
	else {
		quint64& id = pointerMap[pointer];
		if(id == 0) id = (quint64)pointerMap.size();
		*this << id;
	}	
}

/******************************************************************************
* Returns the ID for a pointer that was used to write the pointer to the stream.
* Return 0 if the given pointer hasn't been written to the stream yet.
******************************************************************************/
quint64 SaveStream::pointerID(void* pointer) const 
{
	map<void*, quint64>::const_iterator iter = pointerMap.find(pointer);
	if(iter == pointerMap.end()) return 0;
	return iter->second;
}

};	// End of namespace Base
