///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/mainwnd/MainFrame.h>
#include <core/gui/panels/CommandPanel.h>
#include <core/gui/SpinnerWidget.h>
#include <core/utilities/PathManager.h>
#include <core/viewport/ViewportPanel.h>
#include <core/actions/ActionManager.h>
#include <core/data/DataSetManager.h>
#include "MainMenu.h"
#include "../panels/animation/AnimationTimeSlider.h"

namespace Core {

/// The global instance of this window class.
MainFrame* MainFrame::_singletonInstance = NULL;

/******************************************************************************
* The constructor of the main window class.
******************************************************************************/
MainFrame::MainFrame(const QString& title) :
	QMainWindow(), _viewportPanel(NULL), _commandPanel(NULL), _qtAssistant(NULL)
{
	OVITO_ASSERT_MSG(_singletonInstance == NULL, "MainFrame constructor", "Only one main window should be created.");
	_singletonInstance = this;

	setWindowTitle(title);
	setAttribute(Qt::WA_DeleteOnClose);

	// Create the main menu
	setMenuBar(new MainMenu(this));

	// Create the main toolbar.
	createMainToolbar();

	// Setup the layout of docking widgets.
	setCorner(Qt::BottomLeftCorner, Qt::LeftDockWidgetArea);
	setCorner(Qt::BottomRightCorner, Qt::RightDockWidgetArea);

	// Create the viewport panel.
	_viewportPanel = new ViewportPanel(this);
	setCentralWidget(_viewportPanel);

	// Create the command panel.
	QDockWidget* commandPanelDockWidget = new QDockWidget(tr("Command Panel"), this);
	commandPanelDockWidget->setObjectName("CommandPanel");
	commandPanelDockWidget->setAllowedAreas(Qt::DockWidgetAreas(Qt::LeftDockWidgetArea | Qt::RightDockWidgetArea));
	commandPanelDockWidget->setTitleBarWidget(new QWidget());
	_commandPanel = new CommandPanel(commandPanelDockWidget);
	commandPanelDockWidget->setWidget(_commandPanel);
	addDockWidget(Qt::RightDockWidgetArea, commandPanelDockWidget);

	// Create the animation panel below the viewports.
	QWidget* animationPanel = new QWidget();
	QVBoxLayout* animationPanelLayout = new QVBoxLayout(animationPanel);
	animationPanelLayout->setSpacing(0);
	animationPanelLayout->setContentsMargins(0, 2, 0, 0);
	animationPanel->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Preferred);

	// Create animation time slider
	AnimationTimeSlider* timeSlider = new AnimationTimeSlider();
	timeSlider->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Preferred);
	animationPanelLayout->addWidget(timeSlider);
	animationPanelLayout->addStretch(1);

	// Create status bar.
	_statusBar = new QStatusBar(animationPanel);
	_statusBar->setSizeGripEnabled(false);
	_statusBar->setSizePolicy(QSizePolicy::Ignored, QSizePolicy::Preferred);
	setStatusBar(_statusBar);
	animationPanelLayout->addWidget(_statusBar);

	QDockWidget* animationPanelDockWidget = new QDockWidget(tr("Animation Panel"), this);
	animationPanelDockWidget->setObjectName("AnimationPanel");
	animationPanelDockWidget->setAllowedAreas(Qt::DockWidgetAreas(Qt::LeftDockWidgetArea | Qt::RightDockWidgetArea));
	animationPanelDockWidget->setFeatures(QDockWidget::DockWidgetClosable);
	animationPanelDockWidget->setWidget(animationPanel);
	animationPanelDockWidget->setTitleBarWidget(new QWidget());
	addDockWidget(Qt::BottomDockWidgetArea, animationPanelDockWidget);

    // Create the animation control toolbar.
	QToolBar* animationControlBar1 = new QToolBar();
	animationControlBar1->addAction(ACTION_MANAGER.findActionProxy(ACTION_GOTO_START_OF_ANIMATION));
	animationControlBar1->addSeparator();
	animationControlBar1->addAction(ACTION_MANAGER.findActionProxy(ACTION_GOTO_PREVIOUS_FRAME));
	animationControlBar1->addAction(ACTION_MANAGER.findActionProxy(ACTION_TOGGLE_ANIMATION_PLAYBACK));
	animationControlBar1->addAction(ACTION_MANAGER.findActionProxy(ACTION_GOTO_NEXT_FRAME));
	animationControlBar1->addSeparator();
	animationControlBar1->addAction(ACTION_MANAGER.findActionProxy(ACTION_GOTO_END_OF_ANIMATION));
	animationControlBar1->setStyleSheet("QToolBar { padding: 0px; margin: 0px; border: 0px none black; } QToolButton { padding: 0px; margin: 0px }");
	QToolBar* animationControlBar2 = new QToolBar();
	animationControlBar2->addAction(ACTION_MANAGER.findActionProxy(ACTION_ANIMATION_MODE_TOGGLE));
	class TimeEditBox : public QLineEdit {
	public:
		virtual QSize sizeHint() const { return minimumSizeHint(); }
	};
	QLineEdit* timeEditBox = new TimeEditBox();
	timeEditBox->setToolTip(tr("Current Animation Time"));
	SpinnerWidget* currentTimeSpinner = ANIM_MANAGER.createCurrentTimeSpinner();
	currentTimeSpinner->setTextBox(timeEditBox);
	animationControlBar2->addWidget(timeEditBox);
	animationControlBar2->addWidget(currentTimeSpinner);
	animationControlBar2->addAction(ACTION_MANAGER.findActionProxy(ACTION_ANIMATION_SETTINGS));
	animationControlBar2->addWidget(new QWidget());
	animationControlBar2->setStyleSheet("QToolBar { padding: 0px; margin: 0px; border: 0px none black; } QToolButton { padding: 0px; margin: 0px }");

	QWidget* animationControlPanel = new QWidget();
	QVBoxLayout* animationControlPanelLayout = new QVBoxLayout(animationControlPanel);
	animationControlPanelLayout->setSpacing(0);
	animationControlPanelLayout->setContentsMargins(0, 1, 0, 0);
	animationControlPanelLayout->addWidget(animationControlBar1);
	animationControlPanelLayout->addWidget(animationControlBar2);
	animationControlPanelLayout->addStretch(1);
	animationControlPanel->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Preferred);

	QDockWidget* animationControlDockWidget = new QDockWidget(tr("Animation Control"), this);
	animationControlDockWidget->setObjectName("AnimationControlPanel");
	animationControlDockWidget->setAllowedAreas(Qt::AllDockWidgetAreas);
	animationControlDockWidget->setFeatures(QDockWidget::DockWidgetClosable);
	animationControlDockWidget->setWidget(animationControlPanel);
	animationControlDockWidget->setTitleBarWidget(new QWidget());
	addDockWidget(Qt::BottomDockWidgetArea, animationControlDockWidget);

    // Create the viewport control toolbar.
	QToolBar* viewportControlBar1 = new QToolBar();
	viewportControlBar1->addAction(ACTION_MANAGER.findActionProxy(ACTION_VIEWPORT_ZOOM));
	viewportControlBar1->addAction(ACTION_MANAGER.findActionProxy(ACTION_VIEWPORT_PAN));
	viewportControlBar1->addAction(ACTION_MANAGER.findActionProxy(ACTION_VIEWPORT_ORBIT));
	viewportControlBar1->addAction(ACTION_MANAGER.findActionProxy(ACTION_VIEWPORT_PICK_ORBIT_CENTER));
	viewportControlBar1->setStyleSheet("QToolBar { padding: 0px; margin: 0px; border: 0px none black; } QToolButton { padding: 0px; margin: 0px }");
	QToolBar* viewportControlBar2 = new QToolBar();
	viewportControlBar2->addAction(ACTION_MANAGER.findActionProxy(ACTION_VIEWPORT_ZOOM_SCENE_EXTENTS));
	viewportControlBar2->addAction(ACTION_MANAGER.findActionProxy(ACTION_VIEWPORT_ZOOM_SELECTION_EXTENTS));
	viewportControlBar2->addAction(ACTION_MANAGER.findActionProxy(ACTION_VIEWPORT_FOV));
	viewportControlBar2->addAction(ACTION_MANAGER.findActionProxy(ACTION_VIEWPORT_MAXIMIZE));
	viewportControlBar2->setStyleSheet("QToolBar { padding: 0px; margin: 0px; border: 0px none black; } QToolButton { padding: 0px; margin: 0px }");
	QWidget* viewportControlPanel = new QWidget();
	QVBoxLayout* viewportControlPanelLayout = new QVBoxLayout(viewportControlPanel);
	viewportControlPanelLayout->setSpacing(0);
	viewportControlPanelLayout->setContentsMargins(0, 1, 0, 0);
	viewportControlPanelLayout->addWidget(viewportControlBar1);
	viewportControlPanelLayout->addWidget(viewportControlBar2);
	viewportControlPanelLayout->addStretch(1);
	viewportControlPanel->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Preferred);

	QDockWidget* viewportControlDockWidget = new QDockWidget(tr("Viewport Control"), this);
	viewportControlDockWidget->setObjectName("ViewportControlPanel");
	viewportControlDockWidget->setAllowedAreas(Qt::AllDockWidgetAreas);
	viewportControlDockWidget->setFeatures(QDockWidget::DockWidgetClosable);
	viewportControlDockWidget->setWidget(viewportControlPanel);
	viewportControlDockWidget->setTitleBarWidget(new QWidget());
	addDockWidget(Qt::BottomDockWidgetArea, viewportControlDockWidget);

	if(APPLICATION_MANAGER.experimentalMode()) {
		// Create the modifier toolbar.
		QToolBar* modifierToolBar = new QToolBar(tr("Modifiers"));
		modifierToolBar->setObjectName("ModifiersToolbar");
		Q_FOREACH(ActionProxy* proxy, ACTION_MANAGER.getActionProxies()) {
			// Add all actions that start with "Modifier.Apply." and that have an icon to the toolbar.
			if(proxy->objectName().startsWith("Modifier.Apply.") && proxy->icon().isNull() == false)
				modifierToolBar->addAction(proxy);
		}
		addToolBar(Qt::TopToolBarArea, modifierToolBar);
	}

	// Create the frame buffer window.
	_frameBufferWindow = new FrameBufferWindow(this);

	// Restore window layout.
	QSettings settings;
	settings.beginGroup("app/mainwindow");
	QVariant state = settings.value("state");
	if(state.canConvert<QByteArray>())
		restoreState(state.toByteArray());
}

/******************************************************************************
* Creates and fills the main toolbar.
******************************************************************************/
void MainFrame::createMainToolbar()
{
	_mainToolbar = addToolBar(tr("Main Toolbar"));
	_mainToolbar->setObjectName("MainToolbar");

	_mainToolbar->addAction(ACTION_MANAGER.findActionProxy(ACTION_FILE_OPEN));
	_mainToolbar->addAction(ACTION_MANAGER.findActionProxy(ACTION_FILE_SAVE));

	_mainToolbar->addSeparator();

	_mainToolbar->addAction(ACTION_MANAGER.findActionProxy(ACTION_FILE_IMPORT));
	_mainToolbar->addAction(ACTION_MANAGER.findActionProxy(ACTION_FILE_EXPORT));

	_mainToolbar->addSeparator();

	_mainToolbar->addAction(ACTION_MANAGER.findActionProxy(ACTION_EDIT_UNDO));
	_mainToolbar->addAction(ACTION_MANAGER.findActionProxy(ACTION_EDIT_REDO));

	if(APPLICATION_MANAGER.experimentalMode()) {
		_mainToolbar->addSeparator();

		_mainToolbar->addAction(ACTION_MANAGER.findActionProxy(ACTION_SELECTION_MODE));
		_mainToolbar->addAction(ACTION_MANAGER.findActionProxy(ACTION_MOVE_MODE));
		_mainToolbar->addAction(ACTION_MANAGER.findActionProxy(ACTION_ROTATION_MODE));
		_mainToolbar->addAction(ACTION_MANAGER.findActionProxy(ACTION_SCALING_MODE));

		//_mainToolbar->addSeparator();
		//_mainToolbar->addAction(ACTION_MANAGER.findActionProxy(ACTION_XFORM_SYSTEM));

		_mainToolbar->addSeparator();
		_mainToolbar->addAction(ACTION_MANAGER.findActionProxy(ACTION_SNAPPING_OBJECT));
		_mainToolbar->addAction(ACTION_MANAGER.findActionProxy(ACTION_SNAPPING_ANGLE));
		_mainToolbar->addAction(ACTION_MANAGER.findActionProxy(ACTION_SNAPPING_PERCENT));
	}

	_mainToolbar->addSeparator();

	_mainToolbar->addAction(ACTION_MANAGER.findActionProxy(ACTION_RENDER_ACTIVE_VIEWPORT));
}

/******************************************************************************
* Is called when the window receives an event.
******************************************************************************/
bool MainFrame::event(QEvent* event)
{
	if(event->type() == QEvent::StatusTip) {
		statusBar()->showMessage(static_cast<QStatusTipEvent*>(event)->tip());
		return true;
	}
	return QMainWindow::event(event);
}

/******************************************************************************
* Is called when the user closes the window.
******************************************************************************/
void MainFrame::closeEvent(QCloseEvent* event)
{
	// Save changes.
	if(!DATASET_MANAGER.askForSaveChanges()) {
		event->ignore();
		return;
	}

	// Close current scene file.
	DATASET_MANAGER.setCurrentSet(new DataSet());

	// Save window layout.
	QSettings settings;
	settings.beginGroup("app/mainwindow");
	settings.setValue("state", saveState());

	// Destroy main window.
	event->accept();
}

/******************************************************************************
* Opens the external Qt Assistant program to display online help.
******************************************************************************/
QProcess* MainFrame::showQtAssistant(const char* topic_identifier)
{
	if(_qtAssistant != NULL) {
		_qtAssistant->terminate();
		_qtAssistant->waitForFinished();
		delete _qtAssistant;
	}

	QFileInfo helpFile(PATH_MANAGER.helpDirectory() + QString("/manual/assistant/documentation.qhc"));
	if(!helpFile.exists()) {
		if(QMessageBox::critical(this, tr("Help file missing"), tr("The documention file is not available on your system. This might be because your version of OVITO has been built without the documention option enabled. Do you want to open the online manual in your web browser instead? (Requires internet connection.)"), QMessageBox::Yes|QMessageBox::Cancel, QMessageBox::Yes)
			== QMessageBox::Yes) {
			try {
				// Use the web brwoser to show the online help instead.
				if(!QDesktopServices::openUrl(QUrl("http://www.ovito.org/manual/")))
					throw Exception(tr("Could not lauch the web browser to display the online manual. The URL is http://ovito.org/manual/"));
			}
			catch(const Exception& ex) {
				ex.showError();
			}
			return NULL;
		}
	}

#ifdef Q_OS_WIN
	QString qtAssistantExecutable = QLatin1String("assistant.exe");
#elif defined(Q_OS_MAC)
	QString qtAssistantExecutable = QLatin1String("Assistant.app/Contents/MacOS/Assistant");
#else
	QString qtAssistantExecutable = QLatin1String("assistant");
#endif
	
	QString qtAssistantApp = QLibraryInfo::location(QLibraryInfo::BinariesPath) + QDir::separator() + qtAssistantExecutable;
	if(!QFileInfo(qtAssistantApp).isExecutable()) {
		VerboseLogger() << "Did not find Qt assistant in this location: " << qtAssistantApp << endl;
		QString qtLocalAssistantApp = PATH_MANAGER.executableDirectory() + QDir::separator() + qtAssistantExecutable;
		if(!QFileInfo(qtLocalAssistantApp).isExecutable()) {
			VerboseLogger() << "Did not find Qt assistant in the secondary location: " << qtLocalAssistantApp << endl;
			if(QMessageBox::critical(this, tr("Help viewer missing"), tr("The Qt help file viewer (Qt Assistant) is not available on your system to display the context help. Do you want to open the online manual in your web browser instead? (Requires internet connection)"), QMessageBox::Yes|QMessageBox::Cancel, QMessageBox::Yes)
				== QMessageBox::Yes) {
				try {
					// Use the web browser to display the online help instead.
					if(!QDesktopServices::openUrl(QUrl("http://www.ovito.org/manual/")))
						throw Exception(tr("Could not lauch the web browser to display the online manual. The URL is http://ovito.org/manual/"));
				}
				catch(const Exception& ex) {
					ex.showError();
				}
				return NULL;
			}
		}
		else {
			qtAssistantApp = qtLocalAssistantApp;
		}
	}

	QStringList arguments;
	arguments << "-collectionFile" << QDir::toNativeSeparators(helpFile.absoluteFilePath());
	arguments << "-enableRemoteControl";

	_qtAssistant = new QProcess(this);

	QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
	try {
		VerboseLogger() << logdate << "Starting Qt Assistant: " << qtAssistantApp << endl;
		VerboseLogger() << logdate << "Qt Assistant parameters: " << arguments << endl;
		_qtAssistant->start(qtAssistantApp, arguments);
		if(!_qtAssistant->waitForStarted()) {
			throw Exception(tr("Could not start the external Qt Assistant to display online help. Path: %1.").arg(qtAssistantApp));
		}
		QApplication::restoreOverrideCursor();
	}
	catch(...) {
		QApplication::restoreOverrideCursor();
		throw;
	}

	if(topic_identifier) {
		VerboseLogger() << logdate << "Activating help topic (id=" << topic_identifier << ")" << endl;
		QTextStream str(_qtAssistant);
		str << QLatin1String("activateIdentifier ");
		str << topic_identifier;
		str << QLatin1String("\0") << endl;
		str << QLatin1String("syncContents") << endl;
	}

	return _qtAssistant;
}

};
