///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/properties/IntegerRadioButtonPropertyUI.h>
#include <core/undo/UndoManager.h>

namespace Core {

// Gives the class run-time type information.
IMPLEMENT_ABSTRACT_PLUGIN_CLASS(IntegerRadioButtonPropertyUI, PropertyParameterUI)

/******************************************************************************
* The constructor.
******************************************************************************/
IntegerRadioButtonPropertyUI::IntegerRadioButtonPropertyUI(PropertiesEditor* parentEditor, const char* propertyName) : 
	PropertyParameterUI(parentEditor, propertyName)
{
	_buttonGroup = new QButtonGroup(this);
	connect(_buttonGroup, SIGNAL(buttonClicked(int)), this, SLOT(updatePropertyValue()));	
}

/******************************************************************************
* Constructor for a PropertyField property.
******************************************************************************/
IntegerRadioButtonPropertyUI::IntegerRadioButtonPropertyUI(PropertiesEditor* parentEditor, const PropertyFieldDescriptor& propField) :
	PropertyParameterUI(parentEditor, propField)
{
	_buttonGroup = new QButtonGroup(this);
	connect(_buttonGroup, SIGNAL(buttonClicked(int)), this, SLOT(updatePropertyValue()));
}

/******************************************************************************
* Creates a new radio button widget that can be selected by the user
* to set the property value to the given value. 
******************************************************************************/
QRadioButton* IntegerRadioButtonPropertyUI::addRadioButton(int value, const QString& caption)
{
	QRadioButton* button = new QRadioButton(caption);
	if(buttonGroup()) {
		button->setEnabled(editObject() != NULL && isEnabled());
		buttonGroup()->addButton(button, value);
	}
	return button;
}

/******************************************************************************
* This method is called when a new editable object has been assigned to the properties owner this
* parameter UI belongs to. 
******************************************************************************/
void IntegerRadioButtonPropertyUI::resetUI()
{
	PropertyParameterUI::resetUI();	
	
	if(buttonGroup()) {
		Q_FOREACH(QAbstractButton* button, buttonGroup()->buttons()) 
			button->setEnabled(editObject() != NULL && isEnabled());
	}
}

/******************************************************************************
* This method is called when a new editable object has been assigned to the properties owner this
* parameter UI belongs to. 
******************************************************************************/
void IntegerRadioButtonPropertyUI::updateUI()
{
	PropertyParameterUI::updateUI();	
	
	if(buttonGroup() && editObject()) {
		QVariant val;
		if(propertyName()) {
			val = editObject()->property(propertyName());
			OVITO_ASSERT_MSG(val.isValid() && val.canConvert(QVariant::Int), "IntegerRadioButtonPropertyUI::updateUI()", QString("The object class %1 does not define a property with the name %2 that can be cast to integer type.").arg(editObject()->metaObject()->className(), QString(propertyName())).toLocal8Bit().constData());
			if(!val.isValid() || !val.canConvert(QVariant::Int)) {
				throw Exception(tr("The object class %1 does not define a property with the name %2 that can be cast to integer type.").arg(editObject()->metaObject()->className(), QString(propertyName())));
			}
		}
		else if(propertyField()) {
			val = editObject()->getPropertyFieldValue(*propertyField());
			OVITO_ASSERT(val.isValid());
		}
		int id = val.toInt();
		QAbstractButton* btn = buttonGroup()->button(id);
		if(btn != NULL) 
			btn->setChecked(true);
		else {
			btn = buttonGroup()->checkedButton();
			if(btn) btn->setChecked(false);
		}
	}
}

/******************************************************************************
* Sets the enabled state of the UI.
******************************************************************************/
void IntegerRadioButtonPropertyUI::setEnabled(bool enabled)
{
	if(enabled == isEnabled()) return;
	PropertyParameterUI::setEnabled(enabled);
	if(buttonGroup()) {
		Q_FOREACH(QAbstractButton* button, buttonGroup()->buttons()) 
			button->setEnabled(editObject() != NULL && isEnabled());
	}
}

/******************************************************************************
* Takes the value entered by the user and stores it in the property field 
* this property UI is bound to.
******************************************************************************/
void IntegerRadioButtonPropertyUI::updatePropertyValue()
{
	if(buttonGroup() && editObject()) {
		
		UNDO_MANAGER.beginCompoundOperation(tr("Change parameter"));

		if(propertyName()) {
			int id = buttonGroup()->checkedId();
			if(id != -1 && !editObject()->setProperty(propertyName(), id)) {
				OVITO_ASSERT_MSG(false, "IntegerRadioButtonPropertyUI::updatePropertyValue()", QString("The value of property %1 of object class %2 could not be set.").arg(QString(propertyName()), editObject()->metaObject()->className()).toLocal8Bit().constData());
			}
		}
		else if(propertyField()) {
			int id = buttonGroup()->checkedId();
			if(id != -1)
				editObject()->setPropertyFieldValue(*propertyField(), id);
		}
		
		UNDO_MANAGER.endCompoundOperation();
	}
}

};

