///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file BezierCurve.h
 * \brief Contains definition of Core::BezierCurve class.
 */

#ifndef __OVITO_BEZIER_CURVE_H
#define __OVITO_BEZIER_CURVE_H

#include <core/Core.h>
#include "BezierVertex.h"
#include "BezierPolygon.h"

namespace Core {

/**
 * \brief A bezier curve is made up of a sequence of BezierVertex objects.
 *
 * \author Alexander Stukowski
 * \sa BezierVertex
 */
class CORE_DLLEXPORT BezierCurve
{
public:

	/// \brief Creates an empty bezier curve with no vertices.
	BezierCurve() : _isClosed(false), _isValid(false) {}

	/// \brief Copy constructor.
	BezierCurve(const BezierCurve& curve) : _vertices(curve._vertices), _isClosed(curve._isClosed),
		_isValid(curve._isValid), _boundingBox(curve._boundingBox), _polygon(curve._polygon) {}

	/// \brief Returns the sequence of vertices for this curve.
	/// \return A read-only list of vertices.
	const QVector<BezierVertex>& vertices() const { return _vertices; }

	/// \brief Returns the vector of vertices for this curve.
	/// \return A mutable list of vertices.
	QVector<BezierVertex>& vertices() { return _vertices; }

	/// \brief Returns the number of vertices in this curve.
	/// \return The number of vertices. Can be any non-negative number.
	/// \sa setVertexCount()
	int vertexCount() const { return _vertices.size(); }

	/// \brief Sets the number of vertices in this curve.
	/// \param n The new number of vertices. Can be any non-negative number.
	///
	/// Existing vertices are preserved by this method.
	///
	/// \sa vertexCount()
	void setVertexCount(int n) { OVITO_ASSERT(n >= 0); _vertices.resize(n); invalidate(); }

	/// \brief Returns the number of line segments in this curve.
	/// \return The number of segments. This depends on the number vertices and whether the curve is closed or open.
	/// \sa vertexCount()
	int segmentCount() const { return isClosed() ? vertexCount() : max(0,vertexCount()-1); }

	/// \brief Indicates whether the curve is closed or not.
	/// \return \c true if the bezier curve is closed; \c false if it is open.
	/// \sa setClosed()
	bool isClosed() const { return _isClosed; }

	/// \brief Sets whether the curve is closed.
	/// \param closed The closed state flag.
	/// \sa isClosed()
	void setClosed(bool closed) { _isClosed = closed; }

	/// \brief Returns whether a segment of the bezier curve is a round curve or a straight line.
	/// \param segment The index of the segement whose type should be returned. This must be an integer between
	///              0 (inclusive) and segmentCount() (exclusive).
	/// \return \c true if the segment is curved; \c false if it is a straight line connecting the two vertices.
	bool isCurveSegment(int segment) const {
		OVITO_ASSERT_MSG(segment >= 0 && segment < segmentCount(), "BezierCurve::isCurveSegment()", "Segment index out of range.");
		const BezierVertex& v1 = _vertices[segment];
		const BezierVertex& v2 = _vertices[(segment == vertexCount()-1) ? 0 : segment+1];
		return v1.segmentType() != BezierVertex::SegmentLine &&
			v1.vertexType() != BezierVertex::VertexCorner &&
			v2.vertexType() != BezierVertex::VertexCorner;
	}

	/// \brief Computes the length of the curve.
	/// \return The integrated length of the curve in world units.
	/// \sa segmentLength()
	FloatType length() const {
		FloatType len = 0.0;
		for(int i=segmentCount()-1; i>=0; i--)
			len += segmentLength(i);
		return len;
	}

	/// \brief Returns the bounding box of the curve.
	/// \return The bounding box of the curve.
	///
	/// \note The bounding box is only computed once and then cached until
	///       it becomes invalid. The cached bounding box becomes invalid when the curve has been
	///       altered in some way and invalidate() was called.
	const Box3& boundingBox() const {
		const_cast<BezierCurve&>(*this).validate();
		return _boundingBox;
	}

	/// \brief Returns the approximation polygon of the curve.
	/// \return A polygon that apprxomiates the curve with linear segments.
	///
	/// \note The polygon is cached by the BezierCurve class until the curve is
	///       altered in some way and invalidate() is called.
	const BezierPolygon& polygon() const {
		const_cast<BezierCurve&>(*this).validate();
		return _polygon;
	}

	/// \brief Computes the length of a segment of the curve.
	/// \param index The index of the segement whose length should be computed. This must be an integer between
	///              0 (inclusive) and segmentCount() (exclusive).
	/// \return The integrated length of the curve segment in world units.
	/// \sa length()
	/// \sa isClosed()
	FloatType segmentLength(int index) const;

	/// \brief Computes an interpolation point on the curve.
	/// \param t A value between 0.0 and 1.0 that specifies the position on the curve. The value 0.0 is mapped
	///          to the first curve vertex and 1.0 is mapped to the last vertex if the curve is open or the first
	///          vertex if the curve is closed.
	/// \return The computed point on the curve.
	///
	/// \note The parameter space is not normalized to the segment length. That is, each segment
	///       occupies the same length in the [0,1] interval of the t parameter.
	Point3 interpolateCurve(FloatType t) const;

	/// \brief Computes an interpolation point on a curve segment.
	/// \param segmentIndex The index of the segment. This must be an integer between
	///        0 (inclusive) and segmentCount() (exclusive).
	/// \param t A value between 0.0 and 1.0 that specifies the position on the segment. The value 0.0 is mapped
	///          to the first vertex of the segment and 1.0 is mapped to the second vertex of the selected segment.
	/// \return The computed point on the segment.
	Point3 interpolateSegment(int segmentIndex, FloatType t) const;

	/// \brief Computes the derivative at a point of a given curve segment.
	/// \param segmentIndex The index of the segment. This must be an integer between
	///        0 (inclusive) and segmentCount() (exclusive).
	/// \param t A value between 0.0 and 1.0 that specifies the position on the segment. The value 0.0 is mapped
	///          to the first vertex of the segment and 1.0 is mapped to the second vertex of the selected segment.
	/// \return The computed derivative vector at the given point on the segment.
	Vector3 derivativeSegment(int segmentIndex, FloatType t) const;

	/// \brief Computes the second derivative at a point of a given curve segment.
	/// \param segmentIndex The index of the segment. This must be an integer between
	///        0 (inclusive) and segmentCount() (exclusive).
	/// \param t A value between 0.0 and 1.0 that specifies the position on the segment. The value 0.0 is mapped
	///          to the first vertex of the segment and 1.0 is mapped to the second vertex of the selected segment.
	/// \return The computed second derivative vector at the given point on the segment.
	Vector3 derivativeSegment2(int segmentIndex, FloatType t) const;

	/// \brief Saves the curve to the given output stream.
    /// \param stream The destination stream.
	/// \sa loadFromStream()
	void saveToStream(SaveStream& stream) const;

	/// \brief Loads the curve from the given input stream.
    /// \param stream The source stream.
	/// \sa saveToStream()
	void loadFromStream(LoadStream& stream);

	/// \brief Invalidates the cached state data of the curve.
	///
	/// This method must be called each time the vertices of the curve have been changed.
	void invalidate() { _isValid =  false; }

private:

	/// The list of vertices that make up the curve.
	QVector<BezierVertex> _vertices;

	/// Indicates whether this curve is closed.
	bool _isClosed;

	/// The bounding box of the curve.
	Box3 _boundingBox;

	/// Indicates whether the automatic vertex tangents and the cached
	/// bounding box are up-to-date.
	bool _isValid;

	/// The approximation of the curve used for rendering.
	BezierPolygon _polygon;

private:

	/// \brief Rebuilds the cached state data of the curve.
	void validate();

	/// \brief Return the length of the control polygon of a curve segment.
	static inline FloatType segmentHullLength(const Point3& b0, const Point3& b1, const Point3& b2, const Point3& b3) {
		return Distance(b1, b0) + Distance(b2, b1) + Distance(b2, b3);
	}

	/// \brief Recursively computes the length of one Bezier segment.
	static inline FloatType segmentLengthImpl(const Point3& b0, const Point3& b1, const Point3& b2, const Point3& b3) {
		FloatType l1 = segmentHullLength(b0, b1, b2, b3);
		Point3 c0 = (b0+b1)*0.5;
		Point3 c1 = (b1+b2)*0.5;
		Point3 c2 = (b2+b3)*0.5;
		Point3 d0 = (c0+c1)*0.5;
		Point3 d1 = (c1+c2)*0.5;
		Point3 e0 = (d0+d1)*0.5;
		FloatType bl = segmentHullLength(b0,c0,d0,e0);
		FloatType cl = segmentHullLength(e0,d1,c2,b3);
		FloatType l2 = bl + cl;
		if((l1 - l2) <= 1e-6)
			return l2;
		else
			return segmentLengthImpl(b0,c0,d0,e0) + segmentLengthImpl(e0,d1,c2,b3);
	}
};


};	// End of namespace Core

#endif // __OVITO_BEZIER_CURVE_H
