///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/** 
 * \file PathManager.h 
 * \brief Contains the definition of the Core::PathManager class. 
 */

#ifndef __OVITO_PATH_MANAGER_H
#define __OVITO_PATH_MANAGER_H

#include <core/Core.h>

namespace Core {

/// \def PATH_MANAGER
/// \brief The predefined instance of the Core::PathManager class.
/// 
/// Always use this macro to access the Core::PathManager class instance.
#define PATH_MANAGER		(*PathManager::getSingletonInstance())

/**
 * \brief Provides absolute path names for various application directories.
 * 
 * This is a singleton class with only one predefined instance of this class. 
 * You can access the instance of this class using the PATH_MANAGER macro.
 * 
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT PathManager : public QObject
{
	Q_OBJECT
	
public:

	/// \brief Returns the one and only instance of this class.
	/// \return The predefined instance of the PathManager singleton class.
	/// \note You should use the PATH_MANAGER macro to access the PathManager instance instead
	///       of this method.
	inline static PathManager* getSingletonInstance() {
		OVITO_ASSERT_MSG(_singletonInstance != NULL, "PathManager::getSingletonInstance", "PathManager class is not initialized yet.");
		return _singletonInstance;
	}

	/// \brief Returns the prefix directory where the application has been installed.
	/// \return Absolute path to the installation directory.
	const QString& prefixDirectory() const { return _prefixDirectory; }

	/// \brief Returns the directory where the executable file is located.
	/// \return Absolute path to the directory that contains the main executable
	///         of this application.
	const QString& executableDirectory() const { return _executableDirectory; }

	/// \brief Returns the directory where the core library files are located.
	/// \return Absolute path to the directory that contains the shared library files of this application.
	const QString& libraryDirectory() const { return _libraryDirectory; }

	/// \brief Returns the directory where the plugins are located.
	/// \return Absolute path to the directory from which the application plugins are loaded.
	const QString& pluginsDirectory() const { return _pluginsDirectory; }

	/// \brief Returns the directory where the user's scene files are located by default.
	/// \return Absolute path the scene file sirectory. 
	const QString& scenesDirectory() const { return _scenesDirectory; }

	/// \brief Returns the directory where the script files are located by default.
	/// \return Absolute path to the scripts directory.
	const QString& scriptsDirectory() const { return _scriptsDirectory; }

	/// \brief Returns the directory where the online help files are located.
	/// \return Absolute path to the help files directory.
	const QString& helpDirectory() const { return _helpDirectory; }

private:

	/// The installation (or prefix) directory.
	QString _prefixDirectory;

	/// The directory where the executable is located.
	QString _executableDirectory;

	/// Directory containing the shared libraries.
	QString _libraryDirectory;

	/// The directory where the plugins are located.
	QString _pluginsDirectory;

	/// The directory where the scene files are located by default.
	QString _scenesDirectory;

	/// The directory where the script files are located by default.
	QString _scriptsDirectory;

	/// The directory where the help files are located.
	QString _helpDirectory;

private:
	
	/// This is a singleton class. No public instances allowed.
	PathManager();

	/// Initializes the PathManager.
	static void initialize() { 
		OVITO_ASSERT(_singletonInstance == NULL);
		_singletonInstance = new PathManager();
	}
	
	/// PathManager shutdown.
	static void shutdown() {
		delete _singletonInstance;
		_singletonInstance = NULL;
	}
	
	/// The singleton instance of this class.
	static PathManager* _singletonInstance;

	friend class ApplicationManager;
};

};

#endif // __OVITO_PATH_MANAGER_H
